/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set sw=2 ts=8 et tw=80 : */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef nsDNSService2_h_
#define nsDNSService2_h_

#include "DNSServiceBase.h"
#include "nsClassHashtable.h"
#include "nsPIDNSService.h"
#include "nsIMemoryReporter.h"
#include "nsIObserver.h"
#include "nsHostResolver.h"
#include "nsString.h"
#include "nsTHashSet.h"
#include "nsHashKeys.h"
#include "mozilla/Atomics.h"
#include "mozilla/Mutex.h"
#include "TRRService.h"

class nsAuthSSPI;

class DNSServiceWrapper final : public nsPIDNSService {
 public:
  NS_DECL_THREADSAFE_ISUPPORTS
  NS_FORWARD_NSPIDNSSERVICE(PIDNSService()->)
  NS_FORWARD_NSIDNSSERVICE(DNSService()->)

  DNSServiceWrapper() = default;

  static already_AddRefed<nsIDNSService> GetSingleton();
  static void SwitchToBackupDNSService();

 private:
  ~DNSServiceWrapper() = default;
  nsIDNSService* DNSService();
  nsPIDNSService* PIDNSService();

  mozilla::Mutex mLock{"DNSServiceWrapper.mLock"};
  nsCOMPtr<nsIDNSService> mDNSServiceInUse MOZ_GUARDED_BY(mLock);
  nsCOMPtr<nsIDNSService> mBackupDNSService;
};

class nsDNSService final : public mozilla::net::DNSServiceBase,
                           public nsPIDNSService,
                           public nsIMemoryReporter {
 public:
  NS_DECL_ISUPPORTS_INHERITED
  NS_DECL_NSPIDNSSERVICE
  NS_DECL_NSIDNSSERVICE
  NS_DECL_NSIOBSERVER
  NS_DECL_NSIMEMORYREPORTER

  nsDNSService() = default;

  static already_AddRefed<nsIDNSService> GetXPCOMSingleton();

  size_t SizeOfIncludingThis(mozilla::MallocSizeOf mallocSizeOf);

  bool GetOffline() const;

 protected:
  friend class nsAuthSSPI;
  friend class DNSServiceWrapper;

  nsresult DeprecatedSyncResolve(
      const nsACString& aHostname, nsIDNSService::DNSFlags flags,
      const mozilla::OriginAttributes& aOriginAttributes,
      nsIDNSRecord** result);

 private:
  ~nsDNSService() = default;

  void ReadPrefs(const char* name) override;
  static already_AddRefed<nsDNSService> GetSingleton();

  uint16_t GetAFForLookup(const nsACString& host,
                          nsIDNSService::DNSFlags flags);

  nsresult PreprocessHostname(bool aLocalDomain, const nsACString& aInput,
                              nsACString& aACE);

  bool IsLocalDomain(const nsACString& aHostname) const MOZ_REQUIRES(mLock);

  nsresult AsyncResolveInternal(
      const nsACString& aHostname, uint16_t type, nsIDNSService::DNSFlags flags,
      nsIDNSAdditionalInfo* aInfo, nsIDNSListener* aListener,
      nsIEventTarget* target_,
      const mozilla::OriginAttributes& aOriginAttributes,
      nsICancelable** result);

  nsresult CancelAsyncResolveInternal(
      const nsACString& aHostname, uint16_t aType,
      nsIDNSService::DNSFlags aFlags, nsIDNSAdditionalInfo* aInfo,
      nsIDNSListener* aListener, nsresult aReason,
      const mozilla::OriginAttributes& aOriginAttributes);

  nsresult ResolveInternal(const nsACString& aHostname,
                           nsIDNSService::DNSFlags flags,
                           const mozilla::OriginAttributes& aOriginAttributes,
                           nsIDNSRecord** result);

  // Locks the mutex and returns an addreffed resolver. May return null.
  already_AddRefed<nsHostResolver> GetResolverLocked();

  RefPtr<nsHostResolver> mResolver MOZ_GUARDED_BY(mLock);

  // mLock protects access to mResolver, mLocalDomains, mIPv4OnlyDomains,
  // mFailedSVCDomainNames, and mMockHTTPSRRDomain.
  mozilla::Mutex mLock{"nsDNSServer.mLock"};

  // mIPv4OnlyDomains is a comma-separated list of domains for which only
  // IPv4 DNS lookups are performed. This allows the user to disable IPv6 on
  // a per-domain basis and work around broken DNS servers. See bug 68796.
  nsCString mIPv4OnlyDomains MOZ_GUARDED_BY(mLock);
  nsCString mForceResolve;
  nsCString mMockHTTPSRRDomain MOZ_GUARDED_BY(mLock);
  mozilla::Atomic<bool, mozilla::Relaxed> mHasMockHTTPSRRDomainSet{false};
  bool mNotifyResolution = false;
  bool mForceResolveOn = false;
  nsTHashSet<nsCString> mLocalDomains MOZ_GUARDED_BY(mLock);
  RefPtr<mozilla::net::TRRService> mTrrService;

  nsClassHashtable<nsCStringHashKey, nsTArray<nsCString>> mFailedSVCDomainNames
      MOZ_GUARDED_BY(mLock);
  ;
};

already_AddRefed<nsIDNSService> GetOrInitDNSService();

#endif  // nsDNSService2_h_
