// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F366

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_16_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x08827ea8u, 0x4bc8db48u, 0, 109 }, // 08827ea84bc8db48 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x2797f9d6u, 0x6d4a9c95u, 0, 126 }, // 2797f9d66d4a9c95 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__16_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xfe763d51u, 0x67000586u, 0, 6 }, // fe763d5167000586 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x9a779578u, 0xc71ae6f3u, 0, 143 }, // 9a779578c71ae6f3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xb5dad42au, 0x7d413b52u, 0, 29 }, // b5dad42a7d413b52 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x52581388u, 0x5a219aa1u, 0, 86 }, // 525813885a219aa1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0xc197bb6du, 0x6467a55fu, 23, 109 }, // c197bb6d6467a55f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xd82064e6u, 0x92a7ecf0u, 23, 6 }, // d82064e692a7ecf0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x1d0ac6d7u, 0xe465928cu, 23, 143 }, // 1d0ac6d7e465928c = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x1d61957fu, 0x1ee9ac5eu, 23, 86 }, // 1d61957f1ee9ac5e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__32_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x8cf4d907u, 0x53262167u, 63, 143 }, // 8cf4d90753262167 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_16_3_T_T_0___gfx950__P__32_32__CO__wave3_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{2,5,5,3,5,5,5},
 {7,7,8,7,7,8,5},
 {7,8,7,7,7,7,7},
 {7,7,7,6,7,7,7},
 {7,7,7,7,7,7,7},
 {8,7,7,7,7,7,7},
 {7,7,7,8,7,7,7}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F366 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 16
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float32' AND inputs$BLOCK_DMODEL = 16 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

