/* s7, a Scheme interpreter
 *
 *   derived from TinyScheme 1.39, but not a single byte of that code remains
 *   SPDX-License-Identifier: 0BSD
 *
 * Bill Schottstaedt, bil@ccrma.stanford.edu
 *
 * Mike Scholz provided the FreeBSD support (complex trig funcs, etc)
 * Rick Taube, Andrew Burnson, Donny Ward, Greg Santucci, and Christos Vagias provided the MS Visual C++ support
 * Kjetil Matheussen provided the mingw support
 *
 * Documentation is in s7.h, s7.html, s7-ffi.html, and s7-scm.html.
 * s7test.scm is a regression test.
 * repl.scm is a vt100-based listener.
 * nrepl.scm is a notcurses-based listener.
 * cload.scm and lib*.scm tie in various C libraries.
 * lint.scm checks Scheme code for infelicities.
 * r7rs.scm implements some of r7rs (small).
 * write.scm currrently has pretty-print.
 * mockery.scm has the mock-data definitions.
 * reactive.scm has reactive-set and friends.
 * stuff.scm has some stuff.
 * profile.scm has code to display profile data.
 * debug.scm has debugging aids.
 * case.scm has case*, an extension of case to pattern matching.
 * timing tests are in the s7 tools directory
 *
 * s7.c is organized as follows:
 *    structs and type flags
 *    internal debugging stuff
 *    constants
 *    GC
 *    stacks
 *    symbols and keywords
 *    lets
 *    continuations
 *    numbers
 *    characters
 *    strings
 *    ports
 *    format
 *    lists
 *    vectors
 *    hash-tables
 *    c-objects
 *    functions
 *    equal?
 *    generic length, copy, reverse, fill!, append
 *    error handlers
 *    sundry leftovers
 *    the optimizers
 *    multiple-values, quasiquote
 *    eval
 *    *s7*
 *    initialization and free
 *    repl
 *
 * naming conventions: s7_* usually are C accessible (s7.h), g_* are scheme accessible,
 *   H_* are documentation strings, Q_* are procedure signatures, scheme "?" corresponds to C "is_", scheme "->" to C "_to_",
 *   *_1 are ancillary functions, big_* refer to gmp, *_nr means no return, Inline means always-inline.
 *   In variables, i, j, and k are ints, p is a pair (usually), e is a let (environment), x and y are numbers (usually), o is opt_info*.
 *   Variable and function names use lower case, as do macros that take parameters.  Named constants (via enum or macros) and labels are
 *   in some facsimile of camel-case, as are macro parameters and the type-check macros.  Op names and basic types are in upper case.
 *
 * ---------------- compile time switches ----------------
 */

#if defined __has_include
#  if __has_include ("mus-config.h")
#    include "mus-config.h"
#  endif
#else
#    include "mus-config.h"
#endif

/*
 * Your config file goes here, or just replace that #include line with the defines you need.
 * The compile-time switches involve booleans, complex numbers, and multiprecision arithmetic.
 * Currently we assume we have setjmp.h (used by the error handlers).
 *
 * Complex number support, which is problematic in C++, Solaris, and netBSD
 *   is on the HAVE_COMPLEX_NUMBERS switch. In OSX or Linux, if you're not using C++,
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 1
 *
 *   In g++ I use:
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 0
 *
 *   In Windows and tcc both are 0.
 *
 *   Some systems (FreeBSD) have complex.h, but some random subset of the trig funcs, so
 *   HAVE_COMPLEX_NUMBERS means we can find
 *      cimag creal cabs csqrt carg conj
 *   and HAVE_COMPLEX_TRIG means we have
 *      cacos cacosh casin casinh catan catanh ccos ccosh cexp clog cpow csin csinh ctan ctanh
 *
 * When HAVE_COMPLEX_NUMBERS is 0, the complex functions are stubs that simply return their
 *   argument -- this will be very confusing for the s7 user because, for example, (sqrt -2)
 *   will return something bogus (it might not signal an error).
 *
 * so the incoming (non-s7-specific) compile-time switches are
 *     HAVE_COMPLEX_NUMBERS, HAVE_COMPLEX_TRIG, SIZEOF_VOID_P
 * if SIZEOF_VOID_P is not defined, we look for __SIZEOF_POINTER__ instead,
 *   the default is to assume that we're running on a 64-bit machine.
 *
 * To get multiprecision arithmetic, set WITH_GMP to 1.
 *   You'll also need libgmp, libmpfr, and libmpc (version 0.8.0 or later)
 *
 * and we use these predefined macros: __cplusplus, _MSC_VER, __GNUC__, __clang__, __ANDROID__
 *
 * if WITH_SYSTEM_EXTRAS is 1 (default is 1 unless _MSC_VER), various OS and file related functions are included.
 * if you want this file to compile into a stand-alone interpreter, define WITH_MAIN,
 *   to use nrepl also define WITH_NOTCURSES
 *
 * -O3 is often slower than -O2 (at least according to callgrind)
 * -march=native seems to improve tree-vectorization which is important in Snd
 * -ffast-math makes a mess of NaNs, and does not appear to be faster
 *   -fno-math-errno -fno-signed-zeros are slower
 *   I also tried -fno-signaling-nans -fno-trapping-math -fassociative-math, but at least one of them is much slower
 * this code doesn't compile anymore in gcc 4.3
 */

#if (defined(__GNUC__) || defined(__clang__) || defined(__TINYC__)) /* s7 uses PRId64 so (for example) g++ 4.4 is too old. clang defines __GNUC__ */
  #define WITH_GCC 1
#else
  #define WITH_GCC 0
#endif
#if (defined(__clang__) && __cplusplus) /* pointless -- this is a moving target */
  #define WITH_CLANG_PP 1
#else
  #define WITH_CLANG_PP 0
#endif


/* ---------------- initial sizes ---------------- */

#ifndef INITIAL_HEAP_SIZE
  #define INITIAL_HEAP_SIZE 64000         /* 29-Jul-21 -- seems faster */
#endif
#define Initial_Heap_Size INITIAL_HEAP_SIZE
/* the heap grows as needed, this is its initial size. If the initial heap is small, s7 can run in about 2.5 Mbytes of memory.
 * There are many cases where a bigger heap is faster (but hardware cache size probably matters more).
 * The heap size must be a multiple of 32.  Each object takes 48 bytes.  s7 is fine with the initial heap size set to 800.
 */

#ifndef SYMBOL_TABLE_SIZE
  #define SYMBOL_TABLE_SIZE 32749
#endif
#define Symbol_Table_Size SYMBOL_TABLE_SIZE
/* names are hashed into the symbol table (a vector) and collisions are chained as lists.
 *   4129: tlet +530 [symbol_p_pp], thash +565 [make_symbol], max-bin: (3 5),  tlet: (258 3)
 *  16381: tlet +80  [symbol_p_pp], thash +80  [make_symbol], max-bin: (2 25), tlet: (85 1)
 *  24001: tlet +33  [symbol_p_pp], thash +50  [make_symbol], max-bin: (2 19), tlet: (56 7)
 *  32749: (677 symbols if exit.scm)                          max-bin: (2 13), tlet: (40 4)
 *  72101: tlet -40  [symbol_p_pp], thash -40  [make_symbol], max-bin: (2 11), tlet: (30 5)
 */

#ifndef INITIAL_STACK_SIZE
  #define INITIAL_STACK_SIZE 4096  /* was 2048 17-Mar-21 */
#endif
#define Initial_Stack_Size INITIAL_STACK_SIZE
/* the stack grows as needed, each frame takes 4 entries, this is its initial size. (*s7* 'stack-top) divides size by 4 */

#define Stack_Resize_Trigger 256   /* was Initial_Stack_Size/2 which seems excessive */

#ifndef GC_TEMPS_SIZE
  #define GC_TEMPS_SIZE 256
#endif
#define Gc_Temps_Size GC_TEMPS_SIZE
/* the number of recent objects that are temporarily gc-protected; 8 works for s7test and snd-test.
 *    For the FFI, this sets the lag between a call on s7_cons and the first moment when its result
 *    might be vulnerable to the GC.
 */

#ifndef INITIAL_PROTECTED_OBJECTS_SIZE
  #define INITIAL_PROTECTED_OBJECTS_SIZE 16
#endif
#define Initial_Protected_Objects_Size INITIAL_PROTECTED_OBJECTS_SIZE
/* a vector of objects that are (semi-permanently) protected from the GC, grows as needed */


/* ---------------- scheme choices ---------------- */

#ifndef DISABLE_DEPRECATED
  #define DISABLE_DEPRECATED 0
#endif
#define Disable_Deprecated DISABLE_DEPRECATED

#ifndef DISABLE_AUTOLOAD
  #define DISABLE_AUTOLOAD 0
#endif
#define Disable_Autoload DISABLE_AUTOLOAD

#ifndef WITH_GMP
  #define WITH_GMP 0
  /* this includes multiprecision arithmetic for all numeric types and functions, using gmp, mpfr, and mpc
   * WITH_GMP adds the following functions: bignum and bignum?, and (*s7* 'bignum-precision)
   */
#endif
#define With_Gmp WITH_GMP

#ifndef DEFAULT_BIGNUM_PRECISION
  #define DEFAULT_BIGNUM_PRECISION 128 /* (*s7* 'bignum-precision) initial value, must be >= 2 */
#endif
#define Default_Bignum_Precision DEFAULT_BIGNUM_PRECISION

#ifndef WITH_PURE_S7
  #define WITH_PURE_S7 0
#endif
#define With_Pure_s7 WITH_PURE_S7

#if With_Pure_s7
  #define WITH_EXTRA_EXPONENT_MARKERS 0
  #define WITH_IMMUTABLE_UNQUOTE 1
  /* also omitted: *-ci* functions, char-ready?, cond-expand, multiple-values-bind|set!, call-with-values
   *   and a lot more (inexact/exact, integer-length, etc) -- see s7.html.
   */
#endif

#ifndef WITH_R7RS
  #define WITH_R7RS !With_Pure_s7
  /* this also requires (set! (*s7* 'scheme-version) 'r7rs) */
#endif
#define With_r7rs WITH_R7RS

#ifndef WITH_EXTRA_EXPONENT_MARKERS
  #define WITH_EXTRA_EXPONENT_MARKERS 0
#endif
/* if 1, s7 recognizes "d", "f", "l", and "s" as exponent markers, in addition to "e" (also "D", "F", "L", "S") */
#define With_Extra_Exponent_Markers WITH_EXTRA_EXPONENT_MARKERS

#ifdef _MSC_VER
  #define MS_Windows 1
#else
  #define MS_Windows 0
#endif

#ifndef WITH_SYSTEM_EXTRAS
  #define WITH_SYSTEM_EXTRAS (!MS_Windows)
  /* this adds several functions that access file info, directories, times, etc */
#endif
#define With_System_Extras WITH_SYSTEM_EXTRAS

#ifndef WITH_IMMUTABLE_UNQUOTE
  #define WITH_IMMUTABLE_UNQUOTE 0  /* this removes the name "unquote" */
#endif
#define With_Immutable_Unquote WITH_IMMUTABLE_UNQUOTE

#ifndef WITH_C_LOADER
  #if WITH_GCC && (!__MINGW32__) && (!__CYGWIN__)
    #define WITH_C_LOADER 1
  /* (load file.so [e]) looks for ([e] 'init_func) and if found, calls it as the shared object init function.
   * If With_System_Extras is 0, the caller needs to supply system and delete-file so that cload.scm works.
   */
  #else
    #define WITH_C_LOADER 0
    /* I think dlopen et al are available in MS C, but I have no way to test them; see load_shared_object below */
  #endif
#endif
#define With_C_Loader WITH_C_LOADER

#ifndef WITH_NOTCURSES
  #define WITH_NOTCURSES 0
#endif
#define With_Notcurses WITH_NOTCURSES

#ifndef WITH_HISTORY
  #define WITH_HISTORY 0
  /* this includes a circular buffer of previous evaluations for debugging, ((owlet) 'error-history) and (*s7* 'history-size) */
#endif
#define With_History WITH_HISTORY

#ifndef DEFAULT_HISTORY_SIZE
  #define DEFAULT_HISTORY_SIZE 8
  /* this is the default length of the eval history buffer */
#endif
#define Default_History_Size DEFAULT_HISTORY_SIZE
#if With_History
  #define Max_History_Size 1048576
#endif

#ifndef DEFAULT_PRINT_LENGTH
  #define DEFAULT_PRINT_LENGTH 40 /* (*s7* 'print-length) initial value, was 32 but that's too small 26-May-24 */
#endif
#define Default_Print_Length DEFAULT_PRINT_LENGTH

#ifndef WITH_NUMBER_SEPARATOR
  #define WITH_NUMBER_SEPARATOR 0
#endif
#define With_Number_Separator WITH_NUMBER_SEPARATOR

/* in case mus-config.h forgets these */
#if MS_Windows
  #ifndef HAVE_COMPLEX_NUMBERS
    #define HAVE_COMPLEX_NUMBERS 0
    /* Da Shen adds that you'll need the compiler flag /fp:precise if you're using github actions */
  #endif
  #ifndef HAVE_COMPLEX_TRIG
    #define HAVE_COMPLEX_TRIG 0
  #endif
#else
  #ifndef HAVE_COMPLEX_NUMBERS
    #if __TINYC__ || (__clang__ && __cplusplus) /* clang++ is hopeless */
      #define HAVE_COMPLEX_NUMBERS 0
    #else
      #define HAVE_COMPLEX_NUMBERS 1
    #endif
  #endif
  #if __cplusplus || __TINYC__
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 0
    #endif
  #else
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 1
    #endif
  #endif
#endif
#define Have_Complex_Numbers HAVE_COMPLEX_NUMBERS
#define Have_Complex_Trig HAVE_COMPLEX_TRIG

#ifndef WITH_MULTITHREAD_CHECKS
  #define WITH_MULTITHREAD_CHECKS 0
  /* debugging aid if using s7 in a multithreaded program -- this code courtesy of Kjetil Matheussen */
#endif
#define With_Multithread_Checks WITH_MULTITHREAD_CHECKS

#ifndef WITH_WARNINGS
  #define WITH_WARNINGS 0
  /* int+int overflows to real, etc: this adds warnings which are expensive even though they are never called (procedure overhead) */
#endif
#define With_Warnings WITH_WARNINGS

#ifndef S7_DEBUGGING
  #define S7_DEBUGGING 0
#endif
#define s7_Debugging S7_DEBUGGING

#ifndef SHOW_EVAL_OPS        /* print info about eval ops */
  #define SHOW_EVAL_OPS 0
#endif
#define Show_Eval_Ops SHOW_EVAL_OPS
#if Show_Eval_Ops && !s7_Debugging
  #undef s7_Debugging
  #define s7_Debugging 1
#endif

#ifndef OPT_PRINT           /* print info about opt_* optimizations */
  #define OPT_PRINT 0
#endif
#define Opt_Print OPT_PRINT
#if Opt_Print && !s7_Debugging
  #undef s7_Debugging
  #define s7_Debugging 1
#endif

#ifndef DO_PRINT            /* print info about do_* optimizations */
  #define DO_PRINT 0
#endif
#define Do_Print DO_PRINT
#if Do_Print && !s7_Debugging
  #undef s7_Debugging
  #define s7_Debugging 1
#endif

#ifndef _GNU_SOURCE
  #define _GNU_SOURCE  /* for qsort_r, grumble... */
#endif

#if !MS_Windows
  #include <unistd.h>
  #include <sys/param.h>
  #include <strings.h>
  #include <errno.h>
  #include <locale.h>
#else
  /* in Snd these are in mus-config.h */
  #ifndef MUS_CONFIG_H_LOADED
    #if _MSC_VER < 1900
      #define snprintf _snprintf
    #endif
    #if _MSC_VER > 1200
      #define _CRT_SECURE_NO_DEPRECATE 1
      #define _CRT_NONSTDC_NO_DEPRECATE 1
      #define _CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES 1
    #endif
  #endif
  #include <io.h>
  #pragma warning(disable: 4244) /* conversion might cause loss of data warning */
#endif

#if WITH_GCC && (!s7_Debugging)
  #define Inline inline __attribute__((__always_inline__))
#else
  #if MS_Windows
    #define Inline __forceinline
  #else
    #define Inline inline
  #endif
#endif

#ifndef WITH_VECTORIZE
  #define WITH_VECTORIZE 1
#endif
#define With_Vectorize WITH_VECTORIZE

#if (With_Vectorize) && (defined(__GNUC__) && (__GNUC__ >= 5)) /* is this included -in -O2 now? */
  #define Vectorized __attribute__((optimize("tree-vectorize")))
#else
  #define Vectorized
#endif

#if WITH_GCC
  #define Sentinel __attribute__((sentinel))
#else
  #define Sentinel
#endif

#if MS_Windows
  #define no_return _Noreturn /* deprecated in C23 */
#else
  #define no_return __attribute__((noreturn))
  /* this is ok in gcc/g++/clang and tcc; clang++ complains about "noreturn", hence "no_return" */
  /* pure attribute is rarely applicable here, and does not seem to be helpful (maybe safe_strlen) */
#endif

#ifndef S7_ALIGNED
  #define S7_ALIGNED 0
  /* memclr, local_memset, local_strncmp */
#endif
#define s7_Aligned S7_ALIGNED

#include <stdio.h>
#include <limits.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
/* #include <time.h> */
#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <inttypes.h>
#include <setjmp.h>

#if MS_Windows || defined(__MINGW32__)
  #define Jmp_Buf       jmp_buf
  #define SetJmp(A, B)  setjmp(A)
  #define LongJmp(A, B) longjmp(A, B)
#else
  #define Jmp_Buf       sigjmp_buf
  #define SetJmp(A, B)  sigsetjmp(A, B)
  #define LongJmp(A, B) siglongjmp(A, B)
  /* we need sigsetjmp, not setjmp for nrepl's interrupt (something to do with signal masks??)
   *   unfortunately sigsetjmp is slower than setjmp. In one case, the sigsetjmp version runs
   *   in 24 seconds, but the setjmp version takes 10 seconds, yet callgrind says there is almost no difference?
   */
#endif

#if !MS_Windows
  #include <pthread.h>
#endif

#if __cplusplus
  #include <cmath>
#else
  #include <math.h>
#endif

#include "s7.h"

/* there is also apparently __STDC_NO_COMPLEX__ */
#if WITH_CLANG_PP
  #define CMPLX(x, y) __builtin_complex ((double) (x), (double) (y))
#endif
#if Have_Complex_Numbers
  #if __cplusplus
    #include <complex>
    using namespace std;  /* the code has to work in C as well as C++, so we can't scatter std:: all over the place */
    /* moved the typedef to s7.h. */
  #else
    #include <complex.h>
    /* typedef double complex s7_complex; */
    #if defined(__sun) && defined(__SVR4)
      #undef _Complex_I
      #define _Complex_I 1.0i
    #endif
  #endif
  #ifndef CMPLX
    #if (!(defined(__cplusplus))) && (__GNUC__ > 4 || ((__GNUC__ == 4) && (__GNUC_MINOR__ >= 7))) && !defined(__INTEL_COMPILER)
      #define CMPLX(x, y) __builtin_complex ((double) (x), (double) (y))
    #else
      #define CMPLX(r, i) ((r) + ((i) * (s7_complex)_Complex_I))
    #endif
  #endif
#endif

#if WITH_CLANG_PP
  #define s7_complex_i ((double)1.0i)
#else
#if (defined(__GNUC__))
  #define s7_complex_i 1.0i
#else
  #define s7_complex_i (s7_complex)_Complex_I /* a float, but we want a double */
#endif
#endif

#ifndef M_PI
  #define M_PI 3.1415926535897932384626433832795029L
#endif

#ifndef INFINITY
  #ifndef HUGE_VAL
    #define INFINITY (1.0/0.0) /* -log(0.0) is triggering dumb complaints from cppcheck */
    /* there is sometimes a function, infinity(), MSC apparently uses HUGE_VALF, gcc has __builtin_huge_val() */
  #else
    #define INFINITY HUGE_VAL
  #endif
#endif
#define Infinity INFINITY

#ifndef NAN /* deprecated in C23? */
  #define NAN (Infinity / Infinity) /* apparently ieee754 suggests 0.0/0.0 */
#endif
#define Nan NAN

#if ((!__NetBSD__) && (MS_Windows || (!defined(__STC__)) || (defined(__STDC_VERSION__) && (__STDC_VERSION__ < 199901L))))
  #define __func__ __FUNCTION__
#endif

#ifndef POINTER_32 /* for testing */
#if (((defined(SIZEOF_VOID_P)) && (SIZEOF_VOID_P == 4)) || ((defined(__SIZEOF_POINTER__)) && (__SIZEOF_POINTER__ == 4)) || (!defined(__LP64__)))
  #define POINTER_32 true
#else
  #define POINTER_32 false
#endif
#endif
#define Pointer_32 POINTER_32

#define Write_Real_Precision 16
#ifdef __TINYC__
  typedef double long_double; /* (- .1 1) -> 0.9! and others similarly: (- double long_double) is broken */
#else
  typedef long double long_double;
#endif
typedef uint64_t s7_uint;

#define ld64 PRId64
/* #define lu64 PRIu64 */
#define p64 PRIdPTR

#define Max_Float_Format_Precision 128 /* does this make any sense? 53 bits in mantissa: 16 digits, are the extra digits just garbage? */

/* types */
enum {T_FREE = 0,
      T_PAIR, T_NIL, T_UNUSED, T_UNDEFINED, T_UNSPECIFIED, T_EOF, T_BOOLEAN, T_CHARACTER, T_SYNTAX, T_SYMBOL,
      T_INTEGER, T_RATIO, T_REAL, T_COMPLEX, T_BIG_INTEGER, T_BIG_RATIO, T_BIG_REAL, T_BIG_COMPLEX,
      T_STRING, T_C_OBJECT, T_VECTOR, T_INT_VECTOR, T_FLOAT_VECTOR, T_BYTE_VECTOR, T_COMPLEX_VECTOR,
      T_CATCH, T_DYNAMIC_WIND, T_HASH_TABLE, T_LET, T_ITERATOR,
      T_STACK, T_COUNTER, T_SLOT, T_C_POINTER, T_OUTPUT_PORT, T_INPUT_PORT, T_RANDOM_STATE, T_CONTINUATION, T_GOTO,
      T_CLOSURE, T_CLOSURE_STAR, T_MACRO, T_MACRO_STAR, T_BACRO, T_BACRO_STAR,
      T_C_MACRO, T_C_FUNCTION_STAR, T_C_FUNCTION, T_C_RST_NO_REQ_FUNCTION,
      Num_Types};
/* T_UNUSED, T_STACK, T_SLOT, T_DYNAMIC_WIND, T_CATCH, and T_COUNTER are internal */

static const char *s7_type_names[] =
  {"free", "pair", "nil", "unused", "undefined", "unspecified", "eof_object", "boolean", "character", "syntax", "symbol",
   "integer", "ratio", "real", "complex", "big_integer", "big_ratio", "big_real", "big_complex",
   "string", "c_object", "vector", "int_vector", "float_vector", "byte_vector", "complex_vector",
   "catch", "dynamic_wind", "hash_table", "let", "iterator",
   "stack", "counter", "slot", "c_pointer", "output_port", "input_port", "random_state", "continuation", "goto",
   "closure", "closure*", "macro", "macro*", "bacro", "bacro*",
   "c_macro", "c_function*", "c_function", "c_rst_no_req_function",
   };

/* 1:pair, 2:nil, 3:unused, 4:undefined, 5:unspecified, 6:eof, 7:boolean, 8:character, 9:syntax, 10:symbol,
   11:integer, 12:ratio, 13:real, 14:complex, 15:big_integer, 16:big_ratio, 17:big_real, 18:big_complex,
   19:string, 20:c_object, 21:vector, 22:int_vector, 23:float_vector, 24:byte_vector, 25:complex_vector,
   26:catch, 27:dynamic_wind, 28:hash_table, 29:let, 30:iterator,
   31:stack, 32:counter, 33:slot, 34:c_pointer, 35:output_port, 36:input_port, 37:random_state, 38:continuation, 39:goto,
   40:closure, 41:closure_star, 42:macro, 43:macro_star, 44:bacro, 45:bacro_star,
   46:c_macro, 47:c_function_star, 48:c_function, 49:c_rst_no_req_function,
   50:num_types
*/

typedef struct block_t {
  union {
    void *data;
    s7_pointer d_ptr;
    s7_int *i_ptr;
    s7_int tag;
  } dx;
  int32_t index;
  union {
    bool needs_free;
    uint32_t iter_or_size;
  } ln;
  union {
    s7_int size;
    s7_uint usize;
  } sz;
  union {
    struct block_t *next;
    char *documentation;
    s7_pointer ksym;
    s7_uint nx_uint;
    s7_int *ix_ptr;
    struct {
      uint32_t i1, i2;
    } ix;
  } nx;
  union {
    s7_pointer ex_ptr;
    void *ex_info;
    s7_int ckey;
  } ex;
} block_t;

#define Num_Block_Lists 18
#define Top_Block_List 17
#define Block_List 0

#define block_data(p)                    p->dx.data
#define block_index(p)                   p->index
#define block_set_index(p, Index)        p->index = Index
#define block_size(p)                    p->sz.size
#define block_set_size(p, Size)          p->sz.size = Size
#define block_next(p)                    p->nx.next
#define block_info(p)                    p->ex.ex_info

typedef block_t hash_entry_t; /* I think this means we waste 8 bytes per entry but can use the mallocate functions */
#define hash_entry_key(p)                p->dx.d_ptr
#define hash_entry_value(p)              (p)->ex.ex_ptr
#define hash_entry_set_value(p, Val)     p->ex.ex_ptr = Val
#define hash_entry_next(p)               block_next(p)
#define hash_entry_raw_hash(p)           p->sz.usize         /* block_size(p) */
#define hash_entry_set_raw_hash(p, Hash) p->sz.usize = Hash  /* block_set_size(p, Hash) */

typedef block_t vdims_t;
#define vdims_rank(p)                    p->sz.size
#define vector_elements_should_be_freed(p) p->ln.needs_free
#define vdims_dims(p)                    p->dx.i_ptr
#define vdims_offsets(p)                 p->nx.ix_ptr
#define vdims_original(p)                p->ex.ex_ptr


typedef enum {Token_Eof, Token_Left_Paren, Token_Right_Paren, Token_Dot, Token_Atom, Token_Quote, Token_Double_Quote,
	      Token_Back_Quote, Token_Comma, Token_At_Mark, Token_Sharp_Const,
              Token_Vector, Token_Byte_Vector, Token_Int_Vector, Token_Float_Vector, Token_Complex_Vector} token_t;

typedef enum {No_Article, Indefinite_Article} article_t;
typedef enum {Dwind_Init, Dwind_Body, Dwind_Finish} dwind_t;
enum {No_Safety = 0, Immutable_Vector_Safety, More_Safety_Warnings};  /* (*s7* 'safety) settings, if typedef'd becomes uint32_t (but we want -1) */

typedef enum {File_Port, String_Port, Function_Port} port_type_t;

typedef struct {
  int32_t (*read_character)(s7_scheme *sc, s7_pointer port);             /* function to read a character, int32_t for EOF */
  void (*write_character)(s7_scheme *sc, uint8_t c, s7_pointer port);    /* function to write a character */
  void (*write_string)(s7_scheme *sc, const char *str, s7_int len, s7_pointer port); /* function to write a string of known length */
  token_t (*read_semicolon)(s7_scheme *sc, s7_pointer port);             /* internal skip-to-semicolon reader */
  int32_t (*read_white_space)(s7_scheme *sc, s7_pointer port);           /* internal skip white space reader */
  s7_pointer (*read_name)(s7_scheme *sc, s7_pointer port);               /* internal get-next-name reader */
  s7_pointer (*read_sharp)(s7_scheme *sc, s7_pointer port);              /* internal get-next-sharp-constant reader */
  s7_pointer (*read_line)(s7_scheme *sc, s7_pointer port, bool eol_case);/* function to read a string up to \n */
  void (*displayer)(s7_scheme *sc, const char *s, s7_pointer port);      /* (display s pt) -- port_write_string without strlen?? */
  void (*close_port)(s7_scheme *sc, s7_pointer port);                    /* close-in|output-port */
} port_functions_t;

typedef struct {
  bool needs_free, is_closed;
  port_type_t ptype;
  FILE *file;
  char *filename;
  block_t *filename_block;
  uint32_t line_number, file_number;
  s7_int filename_length;
  block_t *block;
  s7_pointer orig_str;    /* GC protection for string port string or function port function */
  const port_functions_t *pf;
  s7_pointer (*input_function)(s7_scheme *sc, s7_read_t read_choice, s7_pointer port);
  void (*output_function)(s7_scheme *sc, uint8_t c, s7_pointer port);
} port_t;

typedef enum {Oft_d_v, Oft_d_vd, Oft_d_vdd, Oft_d_vid, Oft_d_id, Oft_d_7pi, Oft_d_7pii, Oft_d_7piid,
	      Oft_d_ip, Oft_d_pd, Oft_d_7p, Oft_d_7pid, Oft_d, Oft_d_d, Oft_d_dd, Oft_d_7dd, Oft_d_ddd, Oft_d_dddd,
	      Oft_i_i, Oft_i_7i, Oft_i_ii, Oft_i_7ii, Oft_i_iii, Oft_i_7pi, Oft_i_7pii, Oft_i_7piii, Oft_d_p,
	      Oft_b_p, Oft_b_7p, Oft_b_pp, Oft_b_7pp, Oft_b_pp_unchecked, Oft_b_pi, Oft_b_ii, Oft_b_7ii, Oft_b_dd,
	      Oft_p, Oft_p_p, Oft_p_ii, Oft_p_d, Oft_p_dd, Oft_i_7d, Oft_i_7p, Oft_d_7d, Oft_p_pp, Oft_p_ppp, Oft_p_pi, Oft_p_pi_unchecked,
	      Oft_p_ppi, Oft_p_i, Oft_p_pii, Oft_p_pip, Oft_p_pip_unchecked, Oft_p_piip, Oft_b_i, Oft_b_d} opt_func_t;

typedef struct opt_funcs_t {
  opt_func_t typ;
  void *func;
  struct opt_funcs_t *next;
} opt_funcs_t;

typedef struct {
  const char *name;
  int32_t name_length;
  uint32_t class_id;     /* can't use "class" -- confuses g++ */
  const char *doc;
  opt_funcs_t *opt_data; /* vunion-functions (see below) */
  s7_pointer generic_ff, setter, signature, pars, let;
  s7_pointer (*chooser)(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr);
  /* arg_defaults|names call_args only T_C_FUNCTION_STAR -- call args for GC protection */
  union {
    s7_pointer *arg_defaults;
    s7_pointer bool_setter;
  } dam;
  union {
    s7_pointer *arg_names;
    s7_pointer c_sym;
  } sam;
  union {
    s7_pointer call_args;
    void (*marker)(s7_pointer p, s7_int len);
  } cam;
} c_proc_t; /* 104 = sizeof(c_proc_t) */


typedef struct {
  s7_int type, outer_type;
  s7_pointer scheme_name, getter, setter;
  void (*mark)(void *val);
  void (*free)(void *value);
  bool (*eql)(void *val1, void *val2);
#if !Disable_Deprecated
  char *(*print)(s7_scheme *sc, void *value);
#endif
  s7_pointer (*equal)      (s7_scheme *sc, s7_pointer args);
  s7_pointer (*equivalent) (s7_scheme *sc, s7_pointer args);
  s7_pointer (*ref)        (s7_scheme *sc, s7_pointer args);
  s7_pointer (*set)        (s7_scheme *sc, s7_pointer args);
  s7_pointer (*length)     (s7_scheme *sc, s7_pointer args);
  s7_pointer (*reverse)    (s7_scheme *sc, s7_pointer args);
  s7_pointer (*copy)       (s7_scheme *sc, s7_pointer args);
  s7_pointer (*fill)       (s7_scheme *sc, s7_pointer args);
  s7_pointer (*to_list)    (s7_scheme *sc, s7_pointer args);
  s7_pointer (*to_string)  (s7_scheme *sc, s7_pointer args);
  s7_pointer (*gc_mark)    (s7_scheme *sc, s7_pointer args);
  s7_pointer (*gc_free)    (s7_scheme *sc, s7_pointer args);
} c_object_t;


typedef s7_uint (*hash_map_t)(s7_scheme *sc, s7_pointer table, s7_pointer key);         /* hash-table object->location mapper */
typedef hash_entry_t *(*hash_check_t)(s7_scheme *sc, s7_pointer table, s7_pointer key); /* hash-table object equality function */
static hash_map_t default_hash_map[Num_Types];

typedef s7_int (*s7_i_7pi_t)(s7_scheme *sc, s7_pointer p, s7_int i1);
typedef s7_int (*s7_i_7pii_t)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2);
typedef s7_int (*s7_i_7piii_t)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2, s7_int i3);
typedef s7_int (*s7_i_iii_t)(s7_int i1, s7_int i2, s7_int i3);
typedef s7_int (*s7_i_7i_t)(s7_scheme *sc, s7_int i1);
typedef s7_int (*s7_i_7ii_t)(s7_scheme *sc, s7_int i1, s7_int i2);
typedef bool (*s7_b_pp_t)(s7_pointer p1, s7_pointer p2);
typedef bool (*s7_b_7pp_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
typedef bool (*s7_b_7p_t)(s7_scheme *sc, s7_pointer p1);
typedef bool (*s7_b_pi_t)(s7_scheme *sc, s7_pointer p1, s7_int i2);
typedef bool (*s7_b_d_t)(s7_double p1);
typedef bool (*s7_b_i_t)(s7_int p1);
typedef bool (*s7_b_ii_t)(s7_int p1, s7_int p2);
typedef bool (*s7_b_7ii_t)(s7_scheme *sc, s7_int p1, s7_int p2);
typedef bool (*s7_b_dd_t)(s7_double p1, s7_double p2);
typedef s7_pointer (*s7_p_t)(s7_scheme *sc);
typedef s7_pointer (*s7_p_ppi_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int i1);
typedef s7_pointer (*s7_p_pi_t)(s7_scheme *sc, s7_pointer p1, s7_int i1);
typedef s7_pointer (*s7_p_pii_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2);
typedef s7_pointer (*s7_p_pip_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2);
typedef s7_pointer (*s7_p_piip_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_pointer p3);
typedef s7_pointer (*s7_p_i_t)(s7_scheme *sc, s7_int i);
typedef s7_pointer (*s7_p_ii_t)(s7_scheme *sc, s7_int i1, s7_int i2);
typedef s7_pointer (*s7_p_dd_t)(s7_scheme *sc, s7_double x1, s7_double x2);
typedef s7_double (*s7_d_7d_t)(s7_scheme *sc, s7_double p1);
typedef s7_double (*s7_d_7dd_t)(s7_scheme *sc, s7_double p1, s7_double p2);
typedef s7_double (*s7_d_7p_t)(s7_scheme *sc, s7_pointer p1);
typedef s7_double (*s7_d_7pii_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2);
typedef s7_double (*s7_d_7piid_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_double x1);

typedef struct opt_info opt_info;

typedef union {
  s7_int i;
  s7_double x;
  s7_pointer p;
  void *gen;
  opt_info *o1;
  s7_function call;
  s7_double (*d_f)(void);
  s7_double (*d_d_f)(s7_double x);
  s7_double (*d_7d_f)(s7_scheme *sc, s7_double x);
  s7_double (*d_dd_f)(s7_double x1, s7_double x2);
  s7_double (*d_7dd_f)(s7_scheme *sc, s7_double x1, s7_double x2);
  s7_double (*d_ddd_f)(s7_double x1, s7_double x2, s7_double x3);
  s7_double (*d_dddd_f)(s7_double x1, s7_double x2, s7_double x3, s7_double x4);
  s7_double (*d_v_f)(void *obj);
  s7_double (*d_vd_f)(void *obj, s7_double fm);
  s7_double (*d_vdd_f)(void *obj, s7_double x1, s7_double x2);
  s7_double (*d_vid_f)(void *obj, s7_int i, s7_double fm);
  s7_double (*d_id_f)(s7_int i, s7_double fm);
  s7_double (*d_7pi_f)(s7_scheme *sc, s7_pointer obj, s7_int i1);
  s7_double (*d_7pid_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_double x);
  s7_double (*d_7pii_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_int i2);
  s7_double (*d_7piid_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_int i2, s7_double x);
  s7_double (*d_ip_f)(s7_int i1, s7_pointer p);
  s7_double (*d_pd_f)(s7_pointer obj, s7_double x);
  s7_double (*d_p_f)(s7_pointer p);
  s7_double (*d_7p_f)(s7_scheme *sc, s7_pointer p);
  s7_int (*i_7d_f)(s7_scheme *sc, s7_double i1);
  s7_int (*i_7p_f)(s7_scheme *sc, s7_pointer i1);
  s7_int (*i_i_f)(s7_int i1);
  s7_int (*i_7i_f)(s7_scheme *sc, s7_int i1);
  s7_int (*i_ii_f)(s7_int i1, s7_int i2);
  s7_int (*i_7ii_f)(s7_scheme *sc, s7_int i1, s7_int i2);
  s7_int (*i_iii_f)(s7_int i1, s7_int i2, s7_int i3);
  s7_int (*i_7pi_f)(s7_scheme *sc, s7_pointer p, s7_int i1);
  s7_int (*i_7pii_f)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2);
  s7_int (*i_7piii_f)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2, s7_int i3);
  bool (*b_i_f)(s7_int p);
  bool (*b_d_f)(s7_double p);
  bool (*b_p_f)(s7_pointer p);
  bool (*b_pp_f)(s7_pointer p1, s7_pointer p2);
  bool (*b_7pp_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
  bool (*b_7p_f)(s7_scheme *sc, s7_pointer p1);
  bool (*b_pi_f)(s7_scheme *sc, s7_pointer p1, s7_int i2);
  bool (*b_ii_f)(s7_int i1, s7_int i2);
  bool (*b_7ii_f)(s7_scheme *sc, s7_int i1, s7_int i2);
  bool (*b_dd_f)(s7_double x1, s7_double x2);
  s7_pointer (*p_f)(s7_scheme *sc);
  s7_pointer (*p_p_f)(s7_scheme *sc, s7_pointer p);
  s7_pointer (*p_pp_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
  s7_pointer (*p_ppp_f)(s7_scheme *sc, s7_pointer p, s7_pointer p2, s7_pointer p3);
  s7_pointer (*p_pi_f)(s7_scheme *sc, s7_pointer p1, s7_int i1);
  s7_pointer (*p_pii_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2);
  s7_pointer (*p_ppi_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int i1);
  s7_pointer (*p_pip_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2);
  s7_pointer (*p_piip_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_int i2, s7_pointer p3);
  s7_pointer (*p_i_f)(s7_scheme *sc, s7_int i);
  s7_pointer (*p_ii_f)(s7_scheme *sc, s7_int x1, s7_int x2);
  s7_pointer (*p_d_f)(s7_scheme *sc, s7_double x);
  s7_pointer (*p_dd_f)(s7_scheme *sc, s7_double x1, s7_double x2);
  s7_double (*fd)(opt_info *o);
  s7_int (*fi)(opt_info *o);
  bool (*fb)(opt_info *o);
  s7_pointer (*fp)(opt_info *o);
} vunion;
/* libgsl 15 d_i */

#define Num_Vunions 15
struct opt_info {
  vunion v[Num_Vunions];
  s7_scheme *sc;
};

#define q_temp(o) o->v[Num_Vunions - 1]

#if With_Gmp
#define Mpfr_Rndn MPFR_RNDN

typedef struct bigint {mpz_t n; struct bigint *nxt;} bigint;
typedef struct bigrat {mpq_t q; struct bigrat *nxt;} bigrat;
typedef struct bigflt {mpfr_t x; struct bigflt *nxt;} bigflt;
typedef struct bigcmp {mpc_t z; struct bigcmp *nxt;} bigcmp;

typedef struct {
  mpfr_t error, ux, x0, x1;
  mpz_t i, i0, i1, n;
  mpz_t p0, q0, r, r1, p1, q1, old_p1, old_q1;
  mpfr_t val, e0, e1, e0p, e1p, old_e0, old_e1, old_e0p;
  mpq_t q;
} rat_locals_t;
#endif

typedef intptr_t opcode_t;

typedef struct unlet_entry_t {
  s7_pointer symbol;
  struct unlet_entry_t *next;
} unlet_entry_t;


/* -------------------------------- cell structure -------------------------------- */

typedef struct s7_cell {
  union {
    s7_uint u64_type;             /* type info */
    s7_int s64_type;
    uint8_t type_field;
    struct {
      uint16_t low_bits;          /* 8 bits for type (type_field above, pair?/string? etc, 6 bits in use), 8 flag bits */
      uint16_t mid_bits;          /* 16 more flag bits */
      uint16_t opt_bits;          /* 16 bits for opcode (eval choice), 10 in use) */
      uint16_t high_bits;         /* 16 more flag bits */
    } bits;
  } tf;
  union {

    union {
      s7_int integer_value;       /* integers */
      s7_double real_value;       /* floats */

      struct {                    /* ratios */
	s7_int numerator;
	s7_int denominator;
      } fraction_value;

      union {
#if !WITH_CLANG_PP
	s7_complex z;
#endif
	struct {                  /* complex numbers */
	  s7_double rl;
	  s7_double im;
	} complex_value;
      } cz;

#if With_Gmp
      bigint *bgi;                /* bignums (integer) */
      bigrat *bgr;                /*         (ratio) */
      bigflt *bgf;                /*         (float) */
      bigcmp *bgc;                /*         (complex) */
#endif
    } number;

    struct {                      /* ports */
      port_t *port;
      uint8_t *data;
      s7_int size, point;
      block_t *block;
    } prt;

    struct{                       /* characters */
      uint8_t c, up_c;
      int32_t length;
      bool alpha_c, digit_c, space_c, upper_c, lower_c;
      char c_name[12];
    } chr;

    struct {                      /* c-pointers */
      void *c_pointer;
      s7_pointer c_type, info, weak1, weak2;
    } cptr;

    struct {                      /* vectors */
      s7_int length;
      union {
	s7_pointer *objects;
	s7_int *ints;
	s7_double *floats;
	s7_complex *complexes;
	uint8_t *bytes;
      } elements;
      block_t *block;
      s7_pointer (*vget)(s7_scheme *sc, s7_pointer vec, s7_int loc);
      union {
	s7_pointer (*vset)(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
	s7_pointer fset;
      } setv;
    } vector;

    struct {                        /* stacks (internal) struct must match vector above for length/objects/block */
      s7_int length;                /* unused via stk, but punned to vector so not actually unused */
      s7_pointer *objects;
      block_t *block;               /* same as length */
      s7_int top, flags;
    } stk;

    struct {                        /* hash-tables */
      s7_uint mask;
      hash_entry_t **elements;      /* a pointer into block below: takes up a field in object.hasher but is faster (50 in thash) */
      hash_check_t hash_func;
      hash_map_t *loc;
      block_t *block;
    } hasher;

    struct {                        /* iterators */
      s7_pointer seq, cur;
      union {
	s7_int loc;
	s7_pointer slot;            /* let iterator current slow */
      } lc;
      union {
	s7_int len;
	s7_pointer slow;            /* pair iterator cycle check */
	hash_entry_t *entry;        /* hash-table iterator current entry */
      } lw;
      s7_pointer (*next)(s7_scheme *sc, s7_pointer iterator);
    } iter;

    struct {
      c_proc_t *c_proc;             /* C functions, macros */
      s7_function ff;
      s7_int required_args, optional_args, all_args; /* these could be uint32_t */
    } fnc;

    struct {                        /* pairs */
      s7_pointer car, cdr, opt1;
      union
      {
	s7_pointer opt2;
	s7_int n;
      } o2;
      union {
	s7_pointer opt3;
	s7_int n;
	uint8_t opt_type;
      } o3;
    } cons;

    struct {                       /* special purpose pairs (symbol-table etc) */
      s7_pointer unused_car, unused_cdr;
      s7_uint hash;
      const char *fstr;
      s7_uint location;            /* line/file/position, also used in symbol_table as raw_len */
    } sym_cons;

    struct {                       /* scheme functions */
      s7_pointer args, body, let, setter; /* args can be a symbol, as well as a list, setter can be #f as well as a procedure/closure */
      int32_t arity;
    } func;

    struct {                       /* strings */
      s7_int length;
      char *svalue;
      s7_uint hash;                /* string hash-index */
      block_t *block;
      block_t *gensym_block;
    } string;

    struct {                       /* symbols */
      s7_pointer name, global_slot, local_slot;
      s7_int id;                   /* which let last bound the symbol -- for faster symbol lookup */
      uint32_t ctr;                /* how many times has symbol been bound */
      uint32_t small_symbol_tag;   /* symbol as member of a (small) set (tree-set-memq etc), assumed to be uint32_t in clear_small_symbol_set */
    } sym;

    struct {                       /* syntax */
      s7_pointer symbol;
      opcode_t op;
      int32_t min_args, max_args;
      const char *documentation;
      /* 1 unused */
    } syn;

    struct {                       /* slots (bindings) */
      s7_pointer sym, val, nxt, pending_value, expr;  /* pending_value is also the setter field which works by a whisker */
    } slt;

    struct {                       /* lets (environments) */
      s7_pointer slots, nxt;
      s7_int id;                   /* id of rootlet is -1 */
      union {
	struct {
	  s7_pointer function;     /* *function* (symbol) if this is a funclet */
	  uint32_t line, file;     /* *function* location if it is known */
	} efnc;
	struct {
	  s7_pointer dox1, dox2;   /* do loop variables */
	} dox;
	s7_int key;                /* sc->baffle_ctr type */
      } edat;
    } let;

    struct {                       /* special stuff like #<unspecified> */
      s7_pointer car, cdr;         /* unique_car|cdr, for sc->nil these are sc->unspecified for faster assoc etc */
      s7_int unused_let_id;
      const char *name;
      s7_int len;
    } unq;

    struct {                        /* #<...> */
      char *name;                   /* not const because the GC frees it */
      s7_int len;
      /* 3 unused */
    } undef;

    struct {                        /* #<eof> */
      const char *name;
      s7_int len;
      /* 3 unused */
    } eof;

    struct {                        /* counter (internal) */
      s7_pointer result, list, let, slots; /* let = counter_let (curlet after map/for-each let created) */
      s7_uint cap;                 /* sc->capture_let_counter for let reuse */
    } ctr;

    struct {                        /* random-state */
#if With_Gmp
      gmp_randstate_t state;
#else
      s7_uint seed, carry;
      /* for 64-bit floats we probably need 4 state fields */
#endif
    } rng;

    struct {                        /* additional object types (C) */
      s7_int type;
      void *value;                  /* the value the caller associates with the c_object */
      s7_pointer let;               /* the method list, if any (openlet) */
      s7_scheme *sc;
      /* 1 unused */
    } c_obj;

    struct {                        /* continuations */
      block_t *block;
      s7_pointer stack, op_stack;
      s7_pointer *stack_start, *stack_end;
    } cwcc;

    struct {                        /* call-with-exit */
      s7_uint goto_loc, op_stack_loc;
      bool active;
      s7_pointer name;
      /* 1 unused */
    } rexit;

    struct {                        /* catch */
      s7_uint goto_loc, op_stack_loc;
      s7_pointer tag;
      s7_pointer handler;
      Jmp_Buf *cstack;
    } rcatch; /* C++ reserves "catch" I guess */

    struct {                       /* dynamic-wind */
      s7_pointer in, out, body;
      dwind_t state;
      /* 1 unused */
    } winder;
  } object;

#if s7_Debugging
  int32_t alloc_line, uses, explicit_free_line, gc_line, holders, carrier_line;
  s7_int alloc_type, debugger_bits;
  const char *alloc_func, *gc_func, *root;
  s7_pointer holder;
#endif
} s7_cell;


typedef struct s7_big_cell {
  s7_cell cell;
  s7_int big_hloc;
} s7_big_cell;
typedef struct s7_big_cell *s7_big_pointer;

typedef struct heap_block_t {
  intptr_t start, end;
  s7_int offset;
  struct heap_block_t *next;
} heap_block_t;

typedef struct {
  s7_pointer *objs;
  int32_t size, top, ref, size2;
  bool has_hits;
  int32_t *refs;
  s7_pointer cycle_port, init_port;
  s7_int cycle_loc, init_loc, ctr;
  bool *defined;
} shared_info_t;

typedef struct {
  s7_int loc, curly_len, ctr;
  char *curly_str;
  s7_pointer args, orig_str, curly_arg, port, strport;
} format_data_t;

typedef struct gc_obj_t {
  s7_pointer p;
  struct gc_obj_t *nxt;
} gc_obj_t;

typedef struct {
  s7_pointer *list;
  s7_int size, loc;
} gc_list_t;

typedef struct {
  s7_int size, top, excl_size, excl_top;
  s7_pointer *funcs, *let_names, *files;
  s7_int *timing_data, *excl, *lines;
} profile_data_t;

typedef enum {No_Jump, Call_With_Exit_Jump, Throw_Jump, Catch_Jump, Error_Jump, Error_Quit_Jump} jump_loc_t;
typedef enum {No_Set_Jump, Read_Set_Jump, Load_Set_Jump, Dynamic_Wind_Set_Jump, s7_Call_Set_Jump, Eval_Set_Jump} setjmp_loc_t;
static const char *jump_string[6] = {"No_Jump", "Call_With_Exit_Jump", "Throw_Jump", "Catch_Jump", "Error_Jump", "Error_Quit_Jump"};


/* -------------------------------- s7_scheme struct -------------------------------- */
struct s7_scheme {
  s7_pointer code;    /* layout of first 4 entries should match stack frame layout */
  s7_pointer curlet;
  s7_pointer args;
  opcode_t cur_op;

  s7_pointer value, cur_code;
  s7_pointer nil;                     /* empty list */
  s7_pointer T;                       /* #t */
  s7_pointer F;                       /* #f */
  s7_pointer undefined;               /* #<undefined> */
  s7_pointer unspecified;             /* #<unspecified> */
  s7_pointer no_value;                /* the (values) value */
  s7_pointer unused;                  /* a marker for an unoccupied slot in sc->protected_objects (and other similar stuff) */

  s7_pointer stack;                   /* stack is a vector */
  uint32_t stack_size;
  s7_pointer *stack_start, *stack_end, *stack_resize_trigger;

  s7_pointer *op_stack, *op_stack_now, *op_stack_end;
  uint32_t op_stack_size, max_stack_size;

  s7_cell **heap, **free_heap, **free_heap_top, **free_heap_trigger, **previous_free_heap_top;
  s7_int heap_size, gc_freed, gc_total_freed, max_heap_size, gc_temps_size;
  s7_double gc_resize_heap_fraction, gc_resize_heap_by_4_fraction;
  s7_int gc_calls, gc_total_time, gc_start, gc_end, gc_true_calls, gc_true_total_time;
  heap_block_t *heap_blocks;

#if With_History
  s7_pointer eval_history1, eval_history2, error_history, history_sink, history_pairs, old_cur_code;
  bool using_history1;
#endif

#if With_Multithread_Checks
  int32_t lock_count;
  pthread_mutex_t lock;
#endif

  gc_obj_t *semipermanent_objects, *semipermanent_lets;
  s7_pointer protected_objects, protected_setters, protected_setter_symbols;  /* vectors of gc-protected objects */
  s7_int *protected_objects_free_list;    /* to avoid a linear search for a place to store an object in sc->protected_objects */
  s7_int protected_objects_size, protected_setters_size, protected_setters_loc;
  s7_int protected_objects_free_list_loc;

  s7_pointer symbol_table;
  s7_pointer rootlet, rootlet_slots, shadow_rootlet;
  unlet_entry_t *unlet_entries;       /* original bindings of predefined functions */

  s7_pointer input_port;              /* current-input-port */
  s7_pointer *input_port_stack;       /*   input port stack (load and read internally) */
  uint32_t input_port_stack_size, input_port_stack_loc;

  s7_pointer output_port;             /* current-output-port */
  s7_pointer error_port;              /* current-error-port */
  s7_pointer owlet;                   /* owlet */
  s7_pointer error_type, error_data, error_code, error_line, error_file, error_position; /* owlet slots */
  s7_pointer standard_input, standard_output, standard_error;

  s7_pointer sharp_readers;           /* the binding pair for the global *#readers* list */
  s7_pointer load_hook;               /* *load-hook* hook object */
  s7_pointer autoload_hook;           /* *autoload-hook* hook object */
  s7_pointer unbound_variable_hook;   /* *unbound-variable-hook* hook object */
  s7_pointer missing_close_paren_hook, rootlet_redefinition_hook;
  s7_pointer error_hook, read_error_hook; /* *error-hook* hook object, and *read-error-hook* */
  token_t tok;
  bool gc_off, gc_in_progress;        /* gc_off: if true, the GC won't run */
  uint32_t gc_stats, gensym_counter, f_class, add_class, multiply_class, subtract_class, num_eq_class;
  int32_t format_column, error_argnum;
  s7_uint capture_let_counter;
  bool short_print, is_autoloading, in_with_let, object_out_locked, has_openlets, is_expanding, accept_all_keyword_arguments;
  bool got_tc, got_rec, not_tc, muffle_warnings, symbol_quote, reset_error_hook;
  s7_int rec_tc_args;
  s7_int let_number;
  unsigned char number_separator;
  s7_double default_rationalize_error, equivalent_float_epsilon, hash_table_float_epsilon;
  s7_int default_hash_table_length, initial_string_port_length, print_length, objstr_max_len, history_size, true_history_size, output_file_port_length;
  s7_int max_vector_length, max_string_length, max_list_length, max_vector_dimensions, max_string_port_length, rec_loc, rec_len, max_show_stack_frames;
  s7_pointer stacktrace_defaults, symbol_printer, do_body_p, iterator_at_end_value, scheme_version;

  s7_pointer rec_stack, rec_testp, rec_f1p, rec_f2p, rec_f3p, rec_f4p, rec_f5p, rec_f6p, rec_f7p, rec_f8p;
  s7_pointer rec_resp, rec_slot1, rec_slot2, rec_slot3, rec_p1, rec_p2;
  s7_pointer *rec_els;
  s7_function rec_testf, rec_f1f, rec_f2f, rec_f3f, rec_f4f, rec_f5f, rec_f6f, rec_f7f, rec_f8f, rec_resf, rec_fn;
  s7_int (*rec_fi1)(opt_info *o);
  s7_int (*rec_fi2)(opt_info *o);
  s7_int (*rec_fi3)(opt_info *o);
  s7_int (*rec_fi4)(opt_info *o);
  s7_int (*rec_fi5)(opt_info *o);
  s7_int (*rec_fi6)(opt_info *o);
  bool (*rec_fb1)(opt_info *o);
  bool (*rec_fb2)(opt_info *o);

  opt_info *rec_test_o, *rec_result_o, *rec_a1_o, *rec_a2_o, *rec_a3_o, *rec_a4_o, *rec_a5_o, *rec_a6_o;
  s7_i_ii_t rec_i_ii_f;
  s7_d_dd_t rec_d_dd_f;
  s7_pointer rec_val1, rec_val2;
  bool rec_bool;

  int32_t float_format_precision;
  vdims_t *wrap_only;

  char *typnam;
  int32_t typnam_len, print_width;
  s7_pointer *singletons;
  block_t *unentry;                   /* hash-table lookup failure indicator */

  #define Initial_File_Names_Size 8
  s7_pointer *file_names;
  int32_t file_names_size, file_names_top;

  #define Initial_Strbuf_Size 1024
  s7_int strbuf_size;
  char *strbuf;

  char *read_line_buf;
  s7_int read_line_buf_size;

  s7_pointer v, w, x, y, z;
  s7_pointer temp1, temp2, temp3, temp4, temp5, temp6, temp7, temp8, temp9, read_dims;
  s7_pointer t1_1, t2_1, t2_2, t3_1, t3_2, t3_3, t4_1, u1_1;
  s7_pointer elist_1, elist_2, elist_3, elist_4, elist_5, elist_6, elist_7;
  s7_pointer plist_1, plist_2, plist_2_2, plist_3, plist_4;
  s7_pointer qlist_2, qlist_3, clist_1, clist_2, dlist_1, mlist_1, mlist_2; /* dlist|clist and ulist must not overlap */

  Jmp_Buf *Goto_Start;
  bool longjmp_ok;
  setjmp_loc_t setjmp_loc;

  void (*begin_hook)(s7_scheme *sc, bool *val);
  opcode_t begin_op;

  bool debug_or_profile, profiling_gensyms;
  s7_int current_line, s7_call_line, debug, profile, profile_position;
  s7_pointer profile_prefix;
  profile_data_t *profile_data;
  const char *current_file, *s7_call_file, *s7_call_name;

  shared_info_t *circle_info;
  format_data_t **fdats;
  int32_t num_fdats, safety;
  gc_list_t *strings, *vectors, *input_ports, *output_ports, *input_string_ports, *continuations, *c_objects, *hash_tables;
  gc_list_t *gensyms, *undefineds, *multivectors, *weak_refs, *weak_hash_iterators, *opt1_funcs;
#if With_Gmp
  gc_list_t *big_integers, *big_ratios, *big_reals, *big_complexes, *big_random_states;
  mpz_t mpz_1, mpz_2, mpz_3, mpz_4;
  mpq_t mpq_1, mpq_2, mpq_3;
  mpfr_t mpfr_1, mpfr_2, mpfr_3;
  mpc_t mpc_1, mpc_2;
  rat_locals_t *ratloc;
  bigint *bigints;
  bigrat *bigrats;
  bigflt *bigflts;
  bigcmp *bigcmps;
#endif
  s7_pointer *setters;
  s7_int setters_size, setters_loc;
  s7_pointer *tree_pointers;
  int32_t tree_pointers_size, tree_pointers_top, semipermanent_cells, num_to_str_size;
  s7_pointer format_ports;
  uint32_t alloc_pointer_k, alloc_function_k, alloc_symbol_k, alloc_big_pointer_k;
  s7_cell *alloc_pointer_cells;
  c_proc_t *alloc_function_cells;
  s7_big_cell *alloc_big_pointer_cells;
  s7_pointer string_wrappers, integer_wrappers, real_wrappers, complex_wrappers, c_pointer_wrappers, let_wrappers, slot_wrappers;
  uint8_t *alloc_symbol_cells;
  char *num_to_str;

  block_t *block_lists[Num_Block_Lists];
  size_t alloc_string_k;
  char *alloc_string_cells;

  c_object_t **c_object_types;
  int32_t c_object_types_size, num_c_object_types;
  s7_pointer type_to_typers[Num_Types];

  s7_int big_symbol_tag;
  uint32_t small_symbol_tag;
#if s7_Debugging
  int32_t big_symbol_set_line, small_symbol_set_line, big_symbol_set_state, small_symbol_set_state, y_line, v_line, x_line, t_line;
  const char *big_symbol_set_func, *small_symbol_set_func;
#endif
  int32_t bignum_precision;
  s7_int baffle_ctr, map_call_ctr;
  s7_pointer default_random_state;

  s7_pointer sort_body, sort_begin, sort_v1, sort_v2;
  opcode_t sort_op;
  s7_int sort_body_len;
  s7_b_7pp_t sort_f;
  opt_info *sort_o;
  bool (*sort_fb)(opt_info *o);

  #define Int_To_Str_Size 32
  char int_to_str1[Int_To_Str_Size], int_to_str2[Int_To_Str_Size], int_to_str3[Int_To_Str_Size], int_to_str4[Int_To_Str_Size], int_to_str5[Int_To_Str_Size];

  s7_pointer abs_symbol, acos_symbol, acosh_symbol, add_symbol, angle_symbol, append_symbol, apply_symbol, apply_values_symbol, arity_symbol,
             ash_symbol, asin_symbol, asinh_symbol, assoc_symbol, assq_symbol, assv_symbol, atan_symbol, atanh_symbol, autoload_symbol, autoloader_symbol,
             bacro_symbol, bacro_star_symbol, bignum_symbol, byte_vector_symbol, byte_vector_ref_symbol, byte_vector_set_symbol, byte_vector_to_string_symbol,
             c_pointer_symbol, c_pointer_info_symbol, c_pointer_to_list_symbol, c_pointer_type_symbol, c_pointer_weak1_symbol, c_pointer_weak2_symbol,
             caaaar_symbol, caaadr_symbol, caaar_symbol, caadar_symbol, caaddr_symbol, caadr_symbol,
             caar_symbol, cadaar_symbol, cadadr_symbol, cadar_symbol, caddar_symbol, cadddr_symbol, caddr_symbol, cadr_symbol,
             call_cc_symbol, call_with_current_continuation_symbol, call_with_exit_symbol, call_with_input_file_symbol,
             call_with_input_string_symbol, call_with_output_file_symbol, call_with_output_string_symbol, car_symbol,
             catch_symbol, cdaaar_symbol, cdaadr_symbol, cdaar_symbol, cdadar_symbol, cdaddr_symbol, cdadr_symbol, cdar_symbol,
             cddaar_symbol, cddadr_symbol, cddar_symbol, cdddar_symbol, cddddr_symbol, cdddr_symbol, cddr_symbol, cdr_symbol,
             ceiling_symbol, char_downcase_symbol, char_eq_symbol, char_geq_symbol, char_gt_symbol, char_leq_symbol, char_lt_symbol,
             char_position_symbol, char_to_integer_symbol, char_upcase_symbol, cload_directory_symbol, close_input_port_symbol,
             close_output_port_symbol, complex_symbol, complex_vector_ref_symbol, complex_vector_set_symbol, complex_vector_symbol,
             cond_expand_symbol, cons_symbol, copy_symbol, cos_symbol, cosh_symbol, coverlet_symbol,
             curlet_symbol, current_error_port_symbol, current_input_port_symbol, current_output_port_symbol, cutlet_symbol, cyclic_sequences_symbol,
             denominator_symbol, dilambda_symbol, display_symbol, divide_symbol, documentation_symbol, dynamic_wind_symbol, dynamic_unwind_symbol,
             num_eq_symbol, error_symbol, eval_string_symbol, eval_symbol, exact_to_inexact_symbol, exit_symbol, exp_symbol, expt_symbol,
             features_symbol, file__symbol, fill_symbol, float_vector_ref_symbol, float_vector_set_symbol, float_vector_symbol, floor_symbol,
             flush_output_port_symbol, for_each_symbol, format_symbol, funclet_symbol, _function__symbol, procedure_arglist_symbol,
             gc_symbol, gcd_symbol, gensym_symbol, geq_symbol, get_output_string_symbol, gt_symbol,
             hash_table_entries_symbol, hash_table_key_typer_symbol, hash_table_ref_symbol, hash_table_set_symbol, hash_table_symbol,
             hash_table_value_typer_symbol, help_symbol, hook_functions_symbol,
             imag_part_symbol, immutable_symbol, inexact_to_exact_symbol, inlet_symbol, int_vector_ref_symbol, int_vector_set_symbol, int_vector_symbol,
             integer_decode_float_symbol, integer_to_char_symbol,
             is_aritable_symbol, is_bignum_symbol, is_boolean_symbol, is_byte_symbol, is_byte_vector_symbol,
             is_c_object_symbol, c_object_let_symbol, c_object_type_symbol, is_c_pointer_symbol,
             is_char_alphabetic_symbol, is_char_lower_case_symbol, is_char_numeric_symbol, is_char_symbol, is_char_upper_case_symbol, is_char_whitespace_symbol,
             is_complex_symbol, is_complex_vector_symbol, is_constant_symbol,
             is_continuation_symbol, is_defined_symbol, is_dilambda_symbol, is_eof_object_symbol, is_eq_symbol, is_equal_symbol,
             is_eqv_symbol, is_even_symbol, is_exact_symbol, is_float_vector_symbol, is_funclet_symbol,
             is_gensym_symbol, is_goto_symbol, is_hash_table_symbol, is_immutable_symbol,
             is_inexact_symbol, is_infinite_symbol, is_input_port_symbol, is_int_vector_symbol, is_integer_symbol, is_iterator_symbol,
             is_keyword_symbol, is_let_symbol, is_list_symbol, is_macro_symbol, is_equivalent_symbol, is_nan_symbol, is_negative_symbol,
             is_null_symbol, is_number_symbol, is_odd_symbol, is_openlet_symbol, is_output_port_symbol, is_pair_symbol,
             is_port_closed_symbol, is_positive_symbol, is_procedure_symbol, is_proper_list_symbol, is_provided_symbol,
             is_random_state_symbol, is_rational_symbol, is_real_symbol, is_sequence_symbol, is_string_symbol, is_subvector_symbol,
             is_symbol_symbol, is_syntax_symbol, is_vector_symbol, is_weak_hash_table_symbol, is_zero_symbol,
             is_float_symbol, is_integer_or_real_at_end_symbol, is_integer_or_any_at_end_symbol, is_integer_or_number_at_end_symbol,
             is_unspecified_symbol, is_undefined_symbol,
             iterate_symbol, iterator_is_at_end_symbol, iterator_sequence_symbol,
             keyword_to_symbol_symbol,
             lcm_symbol, length_symbol, leq_symbol, let_ref_fallback_symbol, let_ref_symbol, let_set_fallback_symbol,
             let_set_symbol, let_temporarily_symbol, libraries_symbol, list_ref_symbol, list_set_symbol, list_symbol, list_tail_symbol, list_values_symbol,
             load_path_symbol, load_symbol, log_symbol, logand_symbol, logbit_symbol, logior_symbol, lognot_symbol, logxor_symbol, lt_symbol,
             local_documentation_symbol, local_signature_symbol, local_setter_symbol, local_iterator_symbol,
             macro_symbol, macro_star_symbol, magnitude_symbol,
             make_byte_vector_symbol, make_complex_vector_symbol, make_float_vector_symbol, make_hash_table_symbol,
             make_weak_hash_table_symbol, make_int_vector_symbol, make_iterator_symbol, make_list_symbol, make_string_symbol,
             make_vector_symbol, map_symbol, max_symbol, member_symbol, memq_symbol, memv_symbol, min_symbol, modulo_symbol, multiply_symbol,
             name_symbol, nan_symbol, nan_payload_symbol, newline_symbol, not_symbol, number_to_string_symbol, numerator_symbol,
             object_to_string_symbol, object_to_let_symbol, open_input_file_symbol, open_input_function_symbol, open_input_string_symbol,
             open_output_file_symbol, open_output_function_symbol, open_output_string_symbol, openlet_symbol, outlet_symbol, owlet_symbol,
             pair_filename_symbol, pair_line_number_symbol, peek_char_symbol, pi_symbol, port_filename_symbol, port_line_number_symbol,
             port_file_symbol, port_position_symbol, port_string_symbol, procedure_source_symbol, provide_symbol,
             qq_append_symbol, quotient_symbol,
             random_state_symbol, random_state_to_list_symbol, random_symbol, rationalize_symbol, read_byte_symbol,
             read_char_symbol, read_line_symbol, read_string_symbol, read_symbol, reader_cond_symbol, real_part_symbol, remainder_symbol,
             require_symbol, reverse_symbol, reverseb_symbol, rootlet_symbol, round_symbol,
             setter_symbol, set_car_symbol, set_cdr_symbol,
             set_current_error_port_symbol, set_current_input_port_symbol, set_current_output_port_symbol,
             signature_symbol, sin_symbol, sinh_symbol, sort_symbol, sqrt_symbol,
             stacktrace_symbol, string_append_symbol, string_copy_symbol, string_downcase_symbol, string_eq_symbol, string_fill_symbol,
             string_geq_symbol, string_gt_symbol, string_leq_symbol, string_lt_symbol, string_position_symbol, string_ref_symbol,
             string_set_symbol, string_symbol, string_to_keyword_symbol, string_to_number_symbol, string_to_symbol_symbol, string_upcase_symbol,
             sublet_symbol, substring_symbol, substring_uncopied_symbol, subtract_symbol, subvector_symbol, subvector_position_symbol, subvector_vector_symbol,
             symbol_symbol, symbol_to_dynamic_value_symbol, symbol_initial_value_symbol,
             symbol_to_keyword_symbol, symbol_to_string_symbol, symbol_to_value_symbol,
             tan_symbol, tanh_symbol, throw_symbol, string_to_byte_vector_symbol,
             tree_count_symbol, tree_leaves_symbol, tree_memq_symbol, tree_set_memq_symbol, tree_is_cyclic_symbol, truncate_symbol, type_of_symbol,
             unlet_symbol,
             values_symbol, varlet_symbol, vector_append_symbol, vector_dimension_symbol, vector_dimensions_symbol, vector_fill_symbol,
             vector_rank_symbol, vector_ref_symbol, vector_set_symbol, vector_symbol, vector_typer_symbol,
             weak_hash_table_symbol, with_input_from_file_symbol, with_input_from_string_symbol, with_output_to_file_symbol, with_output_to_string_symbol,
             write_byte_symbol, write_char_symbol, write_string_symbol, write_symbol;
  s7_pointer hash_code_symbol, dummy_equal_hash_table, features_setter;
#if !With_Pure_s7
  s7_pointer char_ci_eq_symbol, char_ci_geq_symbol, char_ci_gt_symbol, char_ci_leq_symbol, char_ci_lt_symbol, integer_length_symbol,
             is_char_ready_symbol, let_to_list_symbol, list_to_string_symbol, list_to_vector_symbol, make_polar_symbol, string_ci_eq_symbol,
             string_ci_geq_symbol, string_ci_gt_symbol, string_ci_leq_symbol, string_ci_lt_symbol, string_length_symbol,
             string_to_list_symbol, vector_length_symbol, vector_to_list_symbol;
#endif
#if With_r7rs
  s7_pointer unlink_symbol, access_symbol, time_symbol, clock_gettime_symbol, getenvs_symbol, uname_symbol;
#endif
  bool r7rs_inited;
  s7_pointer s7_symbol, r5rs_symbol, r7rs_symbol, global_is_eq, initial_is_eq, global_memq, initial_memq, global_assq, initial_assq,
             global_string_copy, initial_string_copy, global_delay, initial_delay, delay_symbol, saved_sharp_readers;

  /* syntax symbols et al */
  s7_pointer allow_other_keys_keyword, and_symbol, anon_symbol, autoload_error_symbol, bad_result_symbol, baffled_symbol, begin_symbol, body_symbol, case_symbol,
             class_name_symbol, cond_symbol, define_bacro_star_symbol, define_bacro_symbol, define_constant_symbol, define_expansion_star_symbol,
             define_expansion_symbol, define_macro_star_symbol, define_macro_symbol, define_star_symbol, define_symbol, display_keyword,
             division_by_zero_symbol, do_symbol, else_symbol, feed_to_symbol, format_error_symbol, if_keyword, if_symbol, immutable_error_symbol,
             invalid_exit_function_symbol, io_error_symbol, lambda_star_symbol, lambda_symbol, let_star_symbol, let_symbol,
             letrec_star_symbol, letrec_symbol, macroexpand_symbol, missing_method_symbol, no_setter_symbol, number_to_real_symbol, or_symbol,
             out_of_memory_symbol, out_of_range_symbol, profile_in_symbol, quasiquote_function, quasiquote_symbol, quote_function, quote_symbol,
             read_error_symbol, readable_keyword, rest_keyword, set_symbol, string_read_error_symbol, symbol_table_symbol,
             syntax_error_symbol, trace_in_symbol, type_symbol, unbound_variable_symbol, unless_symbol,
             unquote_symbol, value_symbol, when_symbol, with_baffle_symbol, with_let_symbol, write_keyword,
             wrong_number_of_args_symbol, wrong_type_arg_symbol;

  /* signatures of sequences used as applicable objects: ("hi" 1) */
  s7_pointer  byte_vector_signature, c_object_signature, float_vector_signature, hash_table_signature, int_vector_signature,
             let_signature, pair_signature, string_signature, vector_signature, complex_vector_signature;
  /* common signatures */
  s7_pointer pcl_bc, pcl_bs, pcl_bt, pcl_c, pcl_f, pcl_i, pcl_n, pcl_r, pcl_s, pcl_v, pl_bc, pl_bn, pl_bt, pl_p, pl_sf, pl_tl, pl_nn;

  /* optimizer s7_functions */
  s7_pointer add_1x, add_2, add_2_ints, add_3, add_4, add_i_random, add_x1, append_2, ash_ic, ash_ii, bv_ref_2, bv_ref_3, bv_set_3,
             cdr_let_ref, cdr_let_set, char_equal_2, char_greater_2, char_less_2, char_position_csi, complex_wrapped, curlet_ref, cv_ref_2, cv_set_3,
             display_2, display_f, dynamic_wind_body, dynamic_wind_init, dynamic_wind_unchecked,
             format_as_objstr, format_f, format_just_control_string, format_no_column, fv_ref_2, fv_ref_3, fv_set_3, fv_set_unchecked, geq_2,
             get_output_string_uncopied, hash_table_2, hash_table_ref_2, int_log2, is_defined_in_rootlet, is_defined_in_unlet, iv_ref_2, iv_ref_3, iv_set_3,
             list_0, list_1, list_2, list_3, list_4, list_ref_at_0, list_ref_at_1, list_ref_at_2, list_set_i,
             logand_2, logand_ii, logior_ii, logior_2, logxor_2, memq_2, memq_3, memq_4, memq_any, multiply_3,
             outlet_unlet, profile_out, read_char_1, restore_setter, rootlet_ref, simple_char_eq, simple_char_eq1, simple_char_eq2,
             simple_inlet, simple_list_values, starlet_ref, starlet_set,
             string_append_2, string_c1, string_equal_2, string_equal_2c, string_greater_2, string_less_2, sublet_curlet, substring_uncopied, subtract_1,
             subtract_2, subtract_2f, subtract_3, subtract_f2, subtract_x1, sv_unlet_ref, symbol_to_string_uncopied, tree_set_memq_syms,
             unlet_disabled, unlet_ref, unlet_set, values_uncopied, vector_2, vector_3, vector_ref_2, vector_ref_3, vector_set_3, vector_set_4, write_2;

  s7_pointer divide_2, divide_by_2, geq_xf, geq_xi, greater_2, greater_xf, greater_xi, invert_1, invert_x, leq_2, leq_ixx,
             leq_xi, less_2, less_x0, less_xf, less_xi, max_2, max_3, min_2, min_3,
             multiply_2, num_eq_2, num_eq_ix, num_eq_xi, random_1, random_f, random_i;
  s7_pointer seed_symbol, carry_symbol;

  /* object->let symbols */
  s7_pointer active_symbol, alias_symbol, at_end_symbol, c_object_ref_symbol, c_type_symbol, class_symbol, closed_symbol,
             current_value_symbol, data_symbol, dimensions_symbol, entries_symbol, file_info_symbol, file_symbol, function_symbol, info_symbol,
             is_mutable_symbol, line_symbol, open_symbol, original_vector_symbol, pointer_symbol, port_type_symbol, position_symbol,
             sequence_symbol, size_symbol, source_symbol, weak_symbol;

#if With_System_Extras
  s7_pointer is_directory_symbol, file_exists_symbol, delete_file_symbol, system_symbol, directory_to_list_symbol, file_mtime_symbol;
#endif
#if With_System_Extras || With_r7rs
  s7_pointer getenv_symbol;
#endif
  s7_pointer open_input_function_choices[S7_NUM_READ_CHOICES];
  s7_pointer closed_input_function, closed_output_function;
  s7_pointer vector_set_function, string_set_function, list_set_function, hash_table_set_function, let_set_function, c_object_set_function, last_function;
  s7_pointer wrong_type_arg_info, out_of_range_info, sole_arg_wrong_type_info, sole_arg_out_of_range_info;

  #define Num_Safe_Prelists 8
  #define Num_Safe_Lists 32               /* 36 is the biggest normally (lint.scm), 49 in s7test, 57 in snd-test, > 16 doesn't happen much */
  s7_pointer safe_lists[Num_Safe_Lists];
  int32_t current_safe_list;
#if s7_Debugging
  s7_int safe_list_uses[Num_Safe_Lists];
  int32_t *tc_rec_calls;
  bool printing_gc_info;
  s7_int blocks_allocated, format_ports_allocated, c_functions_allocated;
  s7_int blocks_borrowed[Num_Block_Lists], blocks_freed[Num_Block_Lists], blocks_mallocated[Num_Block_Lists];
  s7_int string_wrapper_allocs, integer_wrapper_allocs, real_wrapper_allocs, complex_wrapper_allocs, c_pointer_wrapper_allocs, let_wrapper_allocs, slot_wrapper_allocs;
#endif

  int32_t **current_distance; /* levenshtein data */
  s7_pointer autoload_table, starlet, starlet_symbol, temp_error_hook;
  const char ***autoload_names;
  s7_int *autoload_names_sizes;
  bool **autoloaded_already;
  s7_int autoload_names_loc, autoload_names_top;
  int32_t format_depth;
  bool undefined_identifier_warnings, undefined_constant_warnings;

  opt_funcs_t *alloc_opt_func_cells;
  int32_t alloc_opt_func_k;

  int32_t pc;
  #define Opts_Size 256      /* pqw-vox needs 178 */
  opt_info *opts[Opts_Size]; /* this form is a lot faster than opt_info**! */

  #define Initial_Saved_Pointers_Size 256
  void **saved_pointers;
  s7_int saved_pointers_loc, saved_pointers_size;

  s7_pointer type_names[Num_Types];
  s7_int overall_start_time;
};     /* store all s7_scheme bools in one int? ca 60 bytes saved out of 11440? */

static no_return void error_nr(s7_scheme *sc, s7_pointer type, s7_pointer info);
static s7_pointer wrap_string(s7_scheme *sc, const char *str, s7_int len);

#if s7_Debugging
  static void gdb_break(void) {};
#endif

#ifndef DISABLE_FILE_OUTPUT
  #define DISABLE_FILE_OUTPUT 0
#endif
#define Disable_File_Output DISABLE_FILE_OUTPUT

#if s7_Debugging || Disable_File_Output || Pointer_32
  static s7_scheme *cur_sc = NULL;
#endif
#if s7_Debugging || ((Disable_File_Output || Pointer_32) && (!WITH_GCC))
  static s7_scheme *original_cur_sc = NULL;
#endif

static s7_pointer set_elist_1(s7_scheme *sc, s7_pointer x1);

#if Disable_File_Output
static FILE *old_fopen(const char *pathname, const char *mode) {return(fopen(pathname, mode));}

#if !WITH_GCC
/* I assume that MS C can't handle the ({...}) business (WITH_GCC include clang and tinyc) */
#define fwrite local_fwrite
static size_t local_fwrite(const void *ptr, size_t size, size_t nmemb, FILE *stream)
{
  error_nr(cur_sc, cur_sc->io_error_symbol,
	   set_elist_1(cur_sc, wrap_string(cur_sc, "writing a file is not allowed in this version of s7", 51)));
  return(0);
}
static FILE *local_fopen(const char *pathname, const char *mode)
{
  if ((mode[0] == 'w') || (mode[0] == 'a'))
    error_nr(cur_sc, cur_sc->io_error_symbol,
	     set_elist_1(cur_sc, wrap_string(cur_sc, "opening a file is not allowed in this version of s7", 51)));
  return(old_fopen(pathname, mode));
}
#else
#define fwrite(Ptr, Size, N, Stream) ({error_nr(sc, sc->io_error_symbol, set_elist_1(sc, wrap_string(sc, "writing a file is not allowed in this version of s7", 51))); 0;})
#define fopen(Path, Mode) \
  ({if ((Mode[0] == 'w') || (Mode[0] == 'a')) \
      error_nr(sc, sc->io_error_symbol, set_elist_1(cur_sc, wrap_string(cur_sc, "opening a file is not allowed in this version of s7", 51))); \
    old_fopen(Path, Mode);})
#endif
#endif /* Disable_File_Output */

#if Pointer_32
/* passing in sc here gloms up the 64-bit code intolerably -- 32-bit users will just have to live with cur_sc! */
static void *Malloc(size_t bytes)
{
  void *p = malloc(bytes);
  if (!p) error_nr(cur_sc, cur_sc->out_of_memory_symbol, set_elist_1(cur_sc, wrap_string(cur_sc, "malloc failed", 13)));
  return(p);
}

static void *Calloc(size_t nmemb, size_t size)
{
  void *p = calloc(nmemb, size);
  if (!p) error_nr(cur_sc, cur_sc->out_of_memory_symbol, set_elist_1(cur_sc, wrap_string(cur_sc, "calloc failed", 13)));
  return(p);
}

static void *Realloc(void *ptr, size_t size)
{
  void *p = realloc(ptr, size);
  if (!p) error_nr(cur_sc, cur_sc->out_of_memory_symbol, set_elist_1(cur_sc, wrap_string(cur_sc, "realloc failed", 14)));
  return(p);
}
#else
#define Malloc(Size)       malloc(Size)
#define Calloc(N, Size)    calloc(N, Size)
#define Realloc(Ptr, Size) realloc(Ptr, Size)
#endif


/* -------------------------------- mallocate -------------------------------- */
static void add_saved_pointer(s7_scheme *sc, void *p)
{
  if (sc->saved_pointers_loc == sc->saved_pointers_size)
    {
      sc->saved_pointers_size *= 2;
      sc->saved_pointers = (void **)Realloc(sc->saved_pointers, sc->saved_pointers_size * sizeof(void *));
    }
  sc->saved_pointers[sc->saved_pointers_loc++] = p;
}

#define Intlen_Bits_Size 256
static const int32_t intlen_bits[Intlen_Bits_Size] =
  {0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
   6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
   7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
   7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8};

static void memclr(void *s, size_t n)
{
  uint8_t *s2;
#if s7_Aligned
  s2 = (uint8_t *)s;
#else
#if (defined(__x86_64__) || defined(__i386__))
  if (n >= 8)
    {
      s7_int *s1 = (s7_int *)s;
      size_t n8 = n >> 3;
      do {*s1++ = 0;} while (--n8 > 0); /* Loop_4 here is slower */
      n &= 7;
      s2 = (uint8_t *)s1;
    }
  else s2 = (uint8_t *)s;
#else
  s2 = (uint8_t *)s;
#endif
#endif
  while (n > 0)
    {
      *s2++ = 0;
      n--;
    }
}

#define Loop_4(Code) do {Code; Code; Code; Code;} while (0)
#define Loop_8(Code) do {Code; Code; Code; Code; Code; Code; Code; Code;} while (0)
#define Step_8(Var) (((Var) & 0x7) == 0)
#define Step_64(Var) (((Var) & 0x3f) == 0)

#if Pointer_32
#define memclr64 memclr
#else
static Vectorized void memclr64(void *p, size_t bytes)
{
  size_t n = bytes >> 3;
  s7_int *vals = (s7_int *)p;
  for (size_t i = 0; i < n; )
    Loop_8(vals[i++] = 0);
}
#endif

static void init_block_lists(s7_scheme *sc)
{
  for (int32_t i = 0; i < Num_Block_Lists; i++)
    sc->block_lists[i] = NULL;
#if s7_Debugging
  sc->blocks_allocated = 0;
  for (int32_t i = 0; i < Num_Block_Lists; i++)
    sc->blocks_borrowed[i] = 0;
#endif
}

static inline void liberate(s7_scheme *sc, block_t *blk)
{
#if s7_Debugging
  sc->blocks_freed[block_index(blk)]++;
#endif
  if (block_index(blk) != Top_Block_List)
    {
      block_next(blk) = (struct block_t *)sc->block_lists[block_index(blk)];
      sc->block_lists[block_index(blk)] = blk;
    }
  else /* biggest blocks (allocated according to each particular size) are freed and placed on the 0-th list */
    {
      if (block_data(blk))
	{
	  free(block_data(blk));
	  block_data(blk) = NULL;
	}
      block_next(blk) = (struct block_t *)sc->block_lists[Block_List];
      sc->block_lists[Block_List] = blk;
    }
}

static inline void liberate_block(s7_scheme *sc, block_t *blk)
{
#if s7_Debugging
  sc->blocks_freed[Block_List]++;
#endif
  block_next(blk) = (struct block_t *)sc->block_lists[Block_List]; /* Block_List==0 */
  sc->block_lists[Block_List] = blk;
}

static void fill_block_list(s7_scheme *sc)
{
  #define Block_Malloc_Size 256
  block_t *b = (block_t *)Malloc(Block_Malloc_Size * sizeof(block_t)); /* batch alloc means blocks in this batch can't be freed, only returned to the list */
#if s7_Debugging
  sc->blocks_allocated += Block_Malloc_Size;
#endif
  add_saved_pointer(sc, b);
  sc->block_lists[Block_List] = b;
  for (int32_t i = 0; i < Block_Malloc_Size - 1; b++, i++)
    block_next(b) = (block_t *)(b + 1);
  block_next(b) = NULL;
}

static inline block_t *mallocate_block(s7_scheme *sc)
{
  block_t *p;
  if (!sc->block_lists[Block_List])
    fill_block_list(sc);                /* this is much faster than allocating blocks as needed */
  p = sc->block_lists[Block_List];
  sc->block_lists[Block_List] = (block_t *)(block_next(p));
  block_set_index(p, Block_List);
  return(p);
}

static inline char *permalloc(s7_scheme *sc, size_t len)
{
  #define Alloc_String_Size (65536 * 8) /* going up to 16 made no difference in timings */
  #define Alloc_Max_String (512 * 8)    /* was 256 -- sets max size of block space lost at the end, but smaller = more direct malloc calls */
  size_t next_k;

  len = (len + 7) & (~7);               /* 8-byte aligned -- more than half the time, len is already 8-byte aligned */
  next_k = sc->alloc_string_k + len;
  if (next_k > Alloc_String_Size)
    {
      if (len >= Alloc_Max_String)
	{
	  char *result = (char *)Malloc(len);
	  add_saved_pointer(sc, result);
	  return(result);
	}
      sc->alloc_string_cells = (char *)Malloc(Alloc_String_Size); /* get a new block */
      add_saved_pointer(sc, sc->alloc_string_cells);
      sc->alloc_string_k = 0;
      next_k = len;
    }
  {
    char *result = &(sc->alloc_string_cells[sc->alloc_string_k]);
    sc->alloc_string_k = next_k;
    return(result);
  }
}

static Inline block_t *inline_mallocate(s7_scheme *sc, size_t bytes)
{
  block_t *blk;
  if (bytes > 0)
    {
      int32_t index;
      if (bytes <= 8) /* presetting a version of intlen_bits with 3's gave only a small speed-up */
	index = 3;
      else
	{
	  if (bytes <= Intlen_Bits_Size)
	    index = intlen_bits[bytes - 1];
	  else index = (bytes <= 65536) ? (8 + intlen_bits[(bytes - 1) >> 8]) : Top_Block_List;   /* expansion to (1 << 17) made no difference */
	}
      blk = sc->block_lists[index];
      if (blk)
	{
#if s7_Debugging
	  sc->blocks_mallocated[index]++;
#endif
	  sc->block_lists[index] = (block_t *)block_next(blk);
	}
      else
	{
	  if (index < (Top_Block_List - 1))
	    {
	      blk = sc->block_lists[index + 1];
	      if (blk)
		{
		  /* we are "borrowing" a block from the next larger bin -- this saves space but costs a bit of time.
		   *   in a tauto run repeating every call 1000 times, the old form ends up at 129M and 31.1 secs,
		   *   whereas the borrowing form ends at 116M and 31.5 secs, but most of my tests show a slight
		   *   speed-up, probably because grabbing a block here is faster than making a new one.
		   *   Worst case is tlet: 8 slower in callgrind.
		   */
#if s7_Debugging
		  sc->blocks_mallocated[index + 1]++;
		  sc->blocks_borrowed[index + 1]++;
#endif
		  sc->block_lists[index + 1] = (block_t *)block_next(blk);
		  block_set_size(blk, bytes);
		  return(blk);
		}}
	  blk = mallocate_block(sc);
	  block_data(blk) = (index < Top_Block_List) ? (void *)permalloc(sc, (size_t)(1 << index)) : Malloc(bytes);
	  block_set_index(blk, index);
#if s7_Debugging
	  sc->blocks_mallocated[index]++;
#endif
	}}
  else
    {
#if s7_Debugging
      sc->blocks_mallocated[Block_List]++;
#endif
      blk = mallocate_block(sc);
    }
  block_set_size(blk, bytes);
  return(blk);
}

static block_t *mallocate(s7_scheme *sc, size_t bytes) {return(inline_mallocate(sc, bytes));}

static block_t *callocate(s7_scheme *sc, size_t bytes)
{
  block_t *blk = inline_mallocate(sc, bytes);
  if ((block_data(blk)) && (block_index(blk) != Block_List))
    {
      if ((bytes & (~0x3f)) > 0)
	memclr64((void *)block_data(blk), bytes & (~0x3f));
      if ((bytes & 0x3f) > 0)
	memclr((void *)((uint8_t *)block_data(blk) + (bytes & (~0x3f))), bytes & 0x3f);
    }
  return(blk);
}

static block_t *reallocate(s7_scheme *sc, block_t *old_blk, size_t bytes)
{
  block_t *new_blk = inline_mallocate(sc, bytes);
  if ((s7_Debugging) && (bytes < (size_t)block_size(old_blk))) fprintf(stderr, "reallocate to smaller block?\n");
  if (block_data(old_blk))  /* presumably block_data(new_blk) is not null */
    memcpy((uint8_t *)(block_data(new_blk)), (uint8_t *)(block_data(old_blk)), block_size(old_blk));
  liberate(sc, old_blk);
  return(new_blk);
}

/* we can't export mallocate et al without also exporting block_t or accessors for it
 *   that is, the block_t* pointer returned can't be used as if it were the void* pointer returned by malloc
 * ideally we'd have a way to release excessive mallocate bins, but they are permalloc'd individually
 */


/* -------------------------------------------------------------------------------- */
typedef enum {p_Display, p_Write, p_Readable, p_Key, p_Code} use_write_t;

static s7_pointer too_many_arguments_string, not_enough_arguments_string, cant_bind_immutable_string,
  a_boolean_string, a_byte_vector_string, a_format_port_string, a_let_string, a_list_string, a_non_constant_symbol_string,
  a_non_negative_integer_string, a_normal_procedure_string, a_normal_real_string, a_number_string, a_procedure_string, a_procedure_or_a_macro_string,
  a_proper_list_string, a_random_state_object_string, a_rational_string, a_sequence_string, a_symbol_string, a_thunk_string, a_valid_radix_string,
  an_association_list_string, an_eq_func_string, an_input_file_port_string, an_input_port_string, an_input_string_port_string, an_open_input_port_string,
  an_open_output_port_string, an_output_port_or_f_string, an_output_file_port_string, an_output_port_string, an_output_string_port_string,
  an_unsigned_byte_string, caaar_a_list_string, caadr_a_list_string, caar_a_list_string, cadar_a_list_string, caddr_a_list_string,
  cadr_a_list_string, car_a_list_string, cdaar_a_list_string, cdadr_a_list_string, cdar_a_list_string, cddar_a_list_string,
  cdddr_a_list_string, cddr_a_list_string, cdr_a_list_string, immutable_error_string, it_is_infinite_string, it_is_nan_string,
  it_is_negative_string, it_is_too_large_string, it_is_too_small_string, parameter_set_twice_string, result_is_too_large_string,
  something_applicable_string, too_many_indices_string, intermediate_too_large_string,
  format_string_1, format_string_2, format_string_3, format_string_4, keyword_value_missing_string;

static bool t_number_p[Num_Types], t_small_real_p[Num_Types], t_rational_p[Num_Types], t_real_p[Num_Types], t_big_number_p[Num_Types];
static bool t_simple_p[Num_Types], t_structure_p[Num_Types], t_immutable_p[Num_Types];
static bool t_any_macro_p[Num_Types], t_any_closure_p[Num_Types], t_has_closure_let[Num_Types];
static bool t_mappable_p[Num_Types], t_sequence_p[Num_Types], t_vector_p[Num_Types], t_ignores_immutable_p[Num_Types];
static bool t_procedure_p[Num_Types], t_applicable_p[Num_Types], t_macro_setter_p[Num_Types];
#if s7_Debugging
static bool t_ext_p[Num_Types], t_exs_p[Num_Types];    /* make sure internal types don't leak out */
#endif

static void init_types(void)
{
  for (int32_t i = 0; i < Num_Types; i++)
    {
      t_any_closure_p[i] = false;
      t_any_macro_p[i] = false;
      t_applicable_p[i] = false;
      t_has_closure_let[i] = false;
      t_immutable_p[i] = true;
      t_macro_setter_p[i] = false;
      t_mappable_p[i] = false;
      t_number_p[i] = false;
      t_procedure_p[i] = false;
      t_rational_p[i] = false;
      t_real_p[i] = false;
      t_sequence_p[i] = false;
      t_simple_p[i] = false;
      t_small_real_p[i] = false;
      t_structure_p[i] = false;
      t_vector_p[i] = false;
      t_ignores_immutable_p[i] = false;
#if s7_Debugging
      t_ext_p[i] = false;
      t_exs_p[i] = false;
#endif
    }
  {
    const int32_t nums[8] = {T_INTEGER, T_RATIO, T_REAL, T_COMPLEX, T_BIG_INTEGER, T_BIG_RATIO, T_BIG_REAL, T_BIG_COMPLEX};
    for (int32_t i = 0; i < 8; i++) t_number_p[nums[i]] = true;
  }

  t_rational_p[T_INTEGER] = true;
  t_rational_p[T_RATIO] = true;
  t_rational_p[T_BIG_INTEGER] = true;
  t_rational_p[T_BIG_RATIO] = true;

  t_small_real_p[T_INTEGER] = true;
  t_small_real_p[T_RATIO] = true;
  t_small_real_p[T_REAL] = true;

  t_real_p[T_INTEGER] = true;
  t_real_p[T_RATIO] = true;
  t_real_p[T_REAL] = true;
  t_real_p[T_BIG_INTEGER] = true;
  t_real_p[T_BIG_RATIO] = true;
  t_real_p[T_BIG_REAL] = true;

  t_big_number_p[T_BIG_INTEGER] = true;
  t_big_number_p[T_BIG_RATIO] = true;
  t_big_number_p[T_BIG_REAL] = true;
  t_big_number_p[T_BIG_COMPLEX] = true;

  {
    const int32_t recs[8] = {T_PAIR, T_VECTOR, T_HASH_TABLE, T_SLOT, T_LET, T_ITERATOR, T_C_OBJECT, T_C_POINTER};
    for (int32_t i = 0; i < 8; i++) t_structure_p[recs[i]] = true;
  }
  {
    const int32_t seqs[11] = {
      T_NIL, T_PAIR, T_STRING, T_VECTOR, T_INT_VECTOR, T_FLOAT_VECTOR, T_BYTE_VECTOR, T_COMPLEX_VECTOR, T_HASH_TABLE, T_LET, T_C_OBJECT};
    /* this assumes the object has a length method? */
    for (int32_t i = 0; i < 11; i++) t_sequence_p[seqs[i]] = true;
  }
  {
    const int32_t maps[18] = {
      T_PAIR, T_STRING, T_VECTOR, T_INT_VECTOR, T_FLOAT_VECTOR, T_BYTE_VECTOR, T_COMPLEX_VECTOR, T_HASH_TABLE,
      T_LET, T_C_OBJECT, T_ITERATOR, T_C_MACRO, T_MACRO, T_MACRO_STAR, T_BACRO, T_BACRO_STAR, T_CLOSURE, T_CLOSURE_STAR};
    for (int32_t i = 0; i < 18; i++) t_mappable_p[maps[i]] = true;
  }
  {
    const int32_t appls[24] = {
      T_PAIR, T_STRING, T_VECTOR, T_INT_VECTOR, T_FLOAT_VECTOR, T_BYTE_VECTOR, T_COMPLEX_VECTOR,
      T_HASH_TABLE, T_ITERATOR, T_LET, T_C_OBJECT, T_C_MACRO, T_MACRO, T_MACRO_STAR, T_BACRO, T_BACRO_STAR,
      T_SYNTAX, T_C_FUNCTION, T_C_FUNCTION_STAR, T_C_RST_NO_REQ_FUNCTION,
      T_CLOSURE, T_CLOSURE_STAR, T_GOTO, T_CONTINUATION};
    for (int32_t i = 0; i < 24; i++) t_applicable_p[appls[i]] = true;
  }
  {
    const int32_t immuts[15] = {
      T_PAIR, T_UNDEFINED, T_SYMBOL, T_STRING, T_C_OBJECT, T_C_POINTER, T_VECTOR, T_FLOAT_VECTOR, T_INT_VECTOR,
      T_BYTE_VECTOR, T_COMPLEX_VECTOR, T_HASH_TABLE, T_LET, T_SLOT, T_RANDOM_STATE};
    for (int32_t i = 0; i < 15; i++) t_immutable_p[immuts[i]] = false;
    /* T_ITERATOR, T_INPUT_PORT, T_OUTPUT_PORT ?? */
  }
  {
    const int32_t vecs[5] = {T_VECTOR, T_INT_VECTOR, T_FLOAT_VECTOR, T_BYTE_VECTOR, T_COMPLEX_VECTOR};
    for (int32_t i = 0; i < 5; i++) t_vector_p[vecs[i]] = true;
  }
  {
    const int32_t procs[7] = {T_C_FUNCTION, T_C_FUNCTION_STAR, T_C_RST_NO_REQ_FUNCTION, T_CLOSURE, T_CLOSURE_STAR, T_GOTO, T_CONTINUATION};
    for (int32_t i = 0; i < 7; i++) t_procedure_p[procs[i]] = true;
  }
  for (int32_t i = T_CLOSURE; i < Num_Types; i++) t_macro_setter_p[i] = true;
  t_macro_setter_p[T_SYMBOL] = true; /* (slot setter); apparently T_LET and T_C_OBJECT are not possible here */

  t_any_macro_p[T_C_MACRO] = true;
  t_any_macro_p[T_MACRO] = true;
  t_any_macro_p[T_MACRO_STAR] = true;
  t_any_macro_p[T_BACRO] = true;
  t_any_macro_p[T_BACRO_STAR] = true;

  t_any_closure_p[T_CLOSURE] = true;
  t_any_closure_p[T_CLOSURE_STAR] = true;

  t_has_closure_let[T_MACRO] = true;
  t_has_closure_let[T_MACRO_STAR] = true;
  t_has_closure_let[T_BACRO] = true;
  t_has_closure_let[T_BACRO_STAR] = true;
  t_has_closure_let[T_CLOSURE] = true;
  t_has_closure_let[T_CLOSURE_STAR] = true;

  /* not T_UNDEFINED here: only #<undefined> itself will work with eq? */
  /* T_LET needs let_equal in member et al, 29-Nov-22.  Also not sure about ports. */
  {
    const int32_t simps[12] = {T_NIL, T_EOF, T_BOOLEAN, T_CHARACTER, T_SYMBOL, T_SYNTAX,
			       T_C_MACRO, T_C_FUNCTION, T_C_FUNCTION_STAR, T_C_RST_NO_REQ_FUNCTION, T_INPUT_PORT, T_OUTPUT_PORT};
    for (int32_t i = 0; i < 12; i++) t_simple_p[simps[i]] = true;
  }

  t_ignores_immutable_p[T_ITERATOR] = true;
  for (int32_t i = T_OUTPUT_PORT; i < Num_Types; i++) t_ignores_immutable_p[i] = true;
  /* (define (f x) (+ x 1)) (immutable! f) (set! f 3), now f is 3. (immutable! 'f) to block (set! f 3)
   *   maybe this is a bug, but what about generators?  random-state objects and iterators ignore it...
   */

#if s7_Debugging
  t_ext_p[T_UNUSED] = true;
  t_ext_p[T_SLOT] = true;
  t_ext_p[T_STACK] = true;
  t_ext_p[T_DYNAMIC_WIND] = true;
  t_ext_p[T_CATCH] = true;
  t_ext_p[T_COUNTER] = true;
#if !With_Gmp
  t_ext_p[T_BIG_INTEGER] = true;
  t_ext_p[T_BIG_RATIO] = true;
  t_ext_p[T_BIG_REAL] = true;
  t_ext_p[T_BIG_COMPLEX] = true;
#endif
  /* these cases are errors (null pointer, T_FREE checked by check_nref called by check_ref_exs) */
  t_exs_p[T_STACK] = true;
  t_exs_p[T_DYNAMIC_WIND] = true;
  t_exs_p[T_CATCH] = true;
  t_exs_p[T_COUNTER] = true;
#if !With_Gmp
  t_exs_p[T_BIG_INTEGER] = true;
  t_exs_p[T_BIG_RATIO] = true;
  t_exs_p[T_BIG_REAL] = true;
  t_exs_p[T_BIG_COMPLEX] = true;
#endif
#endif
}

#if With_History
#define current_code(Sc)               car(Sc->cur_code)
#define set_current_code(Sc, Code)     do {Sc->cur_code = cdr(Sc->cur_code); set_car(Sc->cur_code, Code);} while (0)
#define replace_current_code(Sc, Code) set_car(Sc->cur_code, Code)
#define mark_current_code(Sc)          do {int32_t _i_; s7_pointer _p_; for (_p_ = Sc->cur_code, _i_ = 0; _i_ < Sc->history_size; _i_++, _p_ = cdr(_p_)) gc_mark(car(_p_));} while (0)
#else
#define current_code(Sc)               Sc->cur_code
#define set_current_code(Sc, Code)     Sc->cur_code = Code
#define replace_current_code(Sc, Code) Sc->cur_code = Code
#define mark_current_code(Sc)          gc_mark(Sc->cur_code)
#endif

#define full_type(p)  ((p)->tf.u64_type)
#define low_type_bits(p) ((p)->tf.bits.low_bits)
#define Type_Mask    0xff

#if s7_Debugging
  static void print_gc_info(s7_scheme *sc, s7_pointer obj, const char *func, int32_t line);
  static s7_pointer check_ref_one(s7_pointer p, uint8_t expected_type, const char *func, int32_t line, const char *func1, const char *func2);
  static s7_pointer check_ref_num(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref_app(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref_ext(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref_exs(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref_out(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref_sln(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_nref(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_opcode(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_let_ref(s7_pointer p, s7_uint role, const char *func, int32_t line);
  static s7_pointer check_ref_two(s7_pointer p, uint8_t expected_type, int32_t other_type, const char *func, int32_t line, const char *func1, const char *func2); /* for REPORT_ROOTLET_REDEF below */
  #define type_unchecked(p) ((p)->tf.type_field)
#if WITH_GCC
  #define type(p) ({uint8_t _t_; _t_ = (p)->tf.type_field; if ((!cur_sc->printing_gc_info) && ((_t_ == T_FREE) || (_t_ >= Num_Types))) print_gc_info(cur_sc, p, __func__, __LINE__); _t_;})
#else
  #define type(p) (p)->tf.type_field
#endif

  #define set_full_type(p, f) set_type_1(p, f, __func__, __LINE__)
  /* these check most s7_cell field references (and many type bits) for consistency */
  #define T_App(P) check_ref_app(P,                      __func__, __LINE__)                /* applicable or #f */
  #define T_Arg(P) check_ref_arg(P,                      __func__, __LINE__)                /* closure arg (list, symbol) */
  #define T_BVc(P) check_ref_one(P, T_BYTE_VECTOR,       __func__, __LINE__, "sweep", NULL)
  #define T_Bgf(P) check_ref_one(P, T_BIG_RATIO,         __func__, __LINE__, "sweep", "free_big_ratio")
  #define T_Bgi(P) check_ref_one(P, T_BIG_INTEGER,       __func__, __LINE__, "sweep", "free_big_integer")
  #define T_Bgr(P) check_ref_one(P, T_BIG_REAL,          __func__, __LINE__, "sweep", "free_big_real")
  #define T_Bgz(P) check_ref_one(P, T_BIG_COMPLEX,       __func__, __LINE__, "sweep", "free_big_complex")
  #define T_CMac(P) check_ref_one(P, T_C_MACRO,          __func__, __LINE__, NULL, NULL)
  #define T_Cat(P) check_ref_one(P, T_CATCH,             __func__, __LINE__, NULL, NULL)
  #define T_CFn(P) check_ref_cfn(P,                      __func__, __LINE__)                /* c-functions (not c-macro) */
  #define T_Chr(P) check_ref_one(P, T_CHARACTER,         __func__, __LINE__, NULL, NULL)
  #define T_Clo(P) check_ref_clo(P,                      __func__, __LINE__)                /* has closure let */
  #define T_Cmp(P) check_ref_one(P, T_COMPLEX,           __func__, __LINE__, NULL, NULL)
  #define T_Con(P) check_ref_one(P, T_CONTINUATION,      __func__, __LINE__, "sweep", "process_continuation")
  #define T_Ctr(P) check_ref_one(P, T_COUNTER,           __func__, __LINE__, NULL, NULL)
  #define T_Cvc(P) check_ref_one(P, T_COMPLEX_VECTOR,    __func__, __LINE__, "sweep", NULL)
  #define T_Dyn(P) check_ref_one(P, T_DYNAMIC_WIND,      __func__, __LINE__, NULL, NULL)
  #define T_Eof(P) check_ref_one(P, T_EOF,               __func__, __LINE__, "sweep", NULL)
  #define T_Exs(P) check_ref_exs(P,                      __func__, __LINE__)                /* not an internal (user-visible) type, but #<unused> and slot are ok */
  #define T_Ext(P) check_ref_ext(P,                      __func__, __LINE__)                /* not an internal type */
  #define T_Fnc(P) check_ref_fnc(P,                      __func__, __LINE__)                /* any c_function|c_macro */
  #define T_Frc(P) check_ref_two(P, T_RATIO, T_INTEGER,  __func__, __LINE__, NULL, NULL)
  #define T_Fst(P) check_ref_one(P, T_C_FUNCTION_STAR,   __func__, __LINE__, NULL, NULL)
  #define T_Fvc(P) check_ref_one(P, T_FLOAT_VECTOR,      __func__, __LINE__, "sweep", NULL)
  #define T_Got(P) check_ref_one(P, T_GOTO,              __func__, __LINE__, NULL, NULL)
  #define T_Hsh(P) check_ref_one(P, T_HASH_TABLE,        __func__, __LINE__, "sweep", "free_hash_table")
  #define T_Int(P) check_ref_one(P, T_INTEGER,           __func__, __LINE__, NULL, NULL)
  #define T_Itr(P) check_ref_one(P, T_ITERATOR,          __func__, __LINE__, "sweep", "process_iterator")
  #define T_Ivc(P) check_ref_one(P, T_INT_VECTOR,        __func__, __LINE__, "sweep", NULL)
  #define T_Key(P) check_ref_key(P,                      __func__, __LINE__)                /* keyword */
  #define T_Let(P) check_ref_one(P, T_LET,               __func__, __LINE__, NULL, NULL)
  #define T_Lst(P) check_ref_two(P, T_PAIR, T_NIL,       __func__, __LINE__, "gc", NULL)
  #define T_Mac(P) check_ref_mac(P,                      __func__, __LINE__)                /* a non-C macro */
  #define T_Met(P) check_ref_met(P,                      __func__, __LINE__)                /* anything that might contain a method */
  #define T_Muti(P) check_ref_muti(P,                    __func__, __LINE__)                /* a mutable integer */
  #define T_Nmv(P) check_ref_nmv(P,                      __func__, __LINE__)                /* not multiple-value, not free, only affects slot values */
  #define T_Num(P) check_ref_num(P,                      __func__, __LINE__)                /* any number (not bignums) */
  #define T_Nvc(P) check_ref_one(P, T_VECTOR,            __func__, __LINE__, "sweep", NULL)
  #define T_Obj(P) check_ref_one(P, T_C_OBJECT,          __func__, __LINE__, "sweep", "s7_c_object_value")
  #define T_Op(P)  check_opcode(P,                       __func__, __LINE__)
  #define T_Out(P) check_ref_out(P,                      __func__, __LINE__)                /* let or NULL */
  #define T_Pair(P) check_ref_one(P, T_PAIR,             __func__, __LINE__, NULL, NULL)
  #define T_Pcs(P) check_ref_two(P, T_PAIR, T_CLOSURE_STAR, __func__, __LINE__, NULL, NULL)
  #define T_Pos(P) check_nref(P,                         __func__, __LINE__)                /* not free */
  #define T_Prc(P) check_ref_prc(P,                      __func__, __LINE__)                /* any procedure (3-arg setters) or #f|#t */
  #define T_Prf(P) check_ref_prf(P,                      __func__, __LINE__)                /* pair or #f */
  #define T_Pri(P) check_ref_pri(P,                      __func__, __LINE__)                /* input_port or #f */
  #define T_Pro(P) check_ref_pro(P,                      __func__, __LINE__)                /* output_port or #f */
  #define T_Prt(P) check_ref_prt(P,                      __func__, __LINE__)                /* input|output_port */
  #define T_Ptr(P) check_ref_one(P, T_C_POINTER,         __func__, __LINE__, NULL, NULL)
  #define T_Ran(P) check_ref_one(P, T_RANDOM_STATE,      __func__, __LINE__, NULL, NULL)
  #define T_Rel(P) check_ref_one(P, T_REAL,              __func__, __LINE__, NULL, NULL)
  #define T_Seq(P) check_ref_seq(P,                      __func__, __LINE__)                /* any sequence or structure */
  #define T_Sld(P) check_ref_two(P, T_SLOT, T_UNDEFINED, __func__, __LINE__, NULL, NULL)
  #define T_Sln(P) check_ref_sln(P,                      __func__, __LINE__)                /* slot, #<undefined> or end_slot, only for traversing let slot lists */
  #define T_Slt(P) check_ref_one(P, T_SLOT,              __func__, __LINE__, NULL, NULL)
  #define T_Stk(P) check_ref_one(P, T_STACK,             __func__, __LINE__, NULL, NULL)
  #define T_Str(P) check_ref_one(P, T_STRING,            __func__, __LINE__, "sweep", NULL)
  #define T_SVec(P) check_ref_svec(P,                    __func__, __LINE__)                /* subvector */
  #define T_Sym(P) check_ref_one(P, T_SYMBOL,            __func__, __LINE__, "sweep", "remove_gensym_from_symbol_table")
  #define T_Syn(P) check_ref_one(P, T_SYNTAX,            __func__, __LINE__, NULL, NULL)
  #define T_Undf(P) check_ref_one(P, T_UNDEFINED,        __func__, __LINE__, "sweep", NULL)
  #define T_Vec(P) check_ref_vec(P,                      __func__, __LINE__)                /* any vector */
#else
  /* if not debugging, all those checks go away */
  #define T_App(P)  P
  #define T_Arg(P)  P
  #define T_BVc(P)  P
  #define T_Bgf(P)  P
  #define T_Bgi(P)  P
  #define T_Bgr(P)  P
  #define T_Bgz(P)  P
  #define T_CMac(P) P
  #define T_Cat(P)  P
  #define T_CFn(P)  P
  #define T_Chr(P)  P
  #define T_Clo(P)  P
  #define T_Cmp(P)  P
  #define T_Con(P)  P
  #define T_Ctr(P)  P
  #define T_Cvc(P)  P
  #define T_Dyn(P)  P
  #define T_Eof(P)  P
  #define T_Exs(P)  P
  #define T_Ext(P)  P
  #define T_Fnc(P)  P
  #define T_Frc(P)  P
  #define T_Fst(P)  P
  #define T_Fvc(P)  P
  #define T_Got(P)  P
  #define T_Hsh(P)  P
  #define T_Int(P)  P
  #define T_Itr(P)  P
  #define T_Ivc(P)  P
  #define T_Key(P)  P
  #define T_Let(P)  P
  #define T_Lst(P)  P
  #define T_Mac(P)  P
  #define T_Met(P)  P
  #define T_Muti(P) P
  #define T_Nmv(P)  P
  #define T_Num(P)  P
  #define T_Nvc(P)  P
  #define T_Obj(P)  P
  #define T_Op(P)   P
  #define T_Out(P)  P
  #define T_Pair(P) P
  #define T_Pcs(P)  P
  #define T_Pos(P)  P
  #define T_Prc(P)  P
  #define T_Prf(P)  P
  #define T_Pri(P)  P
  #define T_Pro(P)  P
  #define T_Prt(P)  P
  #define T_Ptr(P)  P
  #define T_Ran(P)  P
  #define T_Rel(P)  P
  #define T_Seq(P)  P
  #define T_Sld(P)  P
  #define T_Sln(P)  P
  #define T_Slt(P)  P
  #define T_Stk(P)  P
  #define T_Str(P)  P
  #define T_SVec(P) P
  #define T_Sym(P)  P
  #define T_Syn(P)  P
  #define T_Undf(P) P
  #define T_Vec(P)  P

  #define type_unchecked(p)            ((p)->tf.type_field)
  #define type(p)                      ((p)->tf.type_field)
  #define set_full_type(p, f)          full_type(p) = f
#endif
#define signed_type(p)                 (p)->tf.s64_type
#define clear_type(p)                  full_type(p) = T_FREE

#define is_number(P)                   t_number_p[type(P)]
#define is_small_real(P)               t_small_real_p[type(P)]
#define is_real(P)                     t_real_p[type(P)]
#define is_rational(P)                 t_rational_p[type(P)]
#define is_big_number(p)               t_big_number_p[type(p)]
#define is_t_integer(p)                (type(p) == T_INTEGER)
#define is_t_ratio(p)                  (type(p) == T_RATIO)
#define is_t_real(p)                   (type(p) == T_REAL)
#define is_t_complex(p)                (type(p) == T_COMPLEX)
#define is_t_big_integer(p)            (type(p) == T_BIG_INTEGER)
#define is_t_big_ratio(p)              (type(p) == T_BIG_RATIO)
#define is_t_big_real(p)               (type(p) == T_BIG_REAL)
#define is_t_big_complex(p)            (type(p) == T_BIG_COMPLEX)

#define is_boolean(p)                  (type(p) == T_BOOLEAN)

#define is_free(p)                     (type_unchecked(p) == T_FREE)
#define is_free_and_clear(p)           (full_type(p) == T_FREE) /* protect against new_cell in-between states? full_type is unchecked */
#define is_simple(P)                   t_simple_p[type(P)]      /* eq? */
#define has_structure(P)               ((t_structure_p[type(P)]) && ((!is_t_vector(P)) || (!has_simple_elements(P))))

#define is_any_macro(P)                t_any_macro_p[type(P)]
#define is_any_closure(P)              t_any_closure_p[type(P)]
#define is_any_procedure(P)            (type(P) >= T_CLOSURE)
#define has_closure_let(P)             t_has_closure_let[type(P)]

#define is_simple_sequence(P)          (t_sequence_p[type(P)])
#define is_sequence(P)                 ((t_sequence_p[type(P)]) || (has_methods(P)))
#define is_mutable_sequence(P)         (((t_sequence_p[type(P)]) || (has_methods(P))) && (!is_immutable(P)))
#define is_sequence_or_iterator(P)     ((t_sequence_p[type(P)]) || (is_iterator(P)))
#define is_mappable(P)                 (t_mappable_p[type(P)])
#define is_applicable(P)               (t_applicable_p[type(P)])
/* this misses #() which is not applicable to anything, and "", and inapplicable c-objects like random-state */
#define is_procedure(p)                ((t_procedure_p[type(p)]) || ((is_c_object(p)) && (is_safe_procedure(p))))
#define is_t_procedure(p)              (t_procedure_p[type(p)])

/* the layout of these bits does matter in several cases -- don't shadow SYNTACTIC_PAIR and OPTIMIZED_PAIR */

#define set_type_bit(p, b)             full_type(p) |= (b)
#define clear_type_bit(p, b)           full_type(p) &= (~(b))
#define has_type_bit(p, b)             ((full_type(p) & (b)) != 0)

#define set_low_type_bit(p, b)         low_type_bits(p) |= (b)
#define clear_low_type_bit(p, b)       low_type_bits(p) &= (~(b))
#define has_low_type_bit(p, b)         ((low_type_bits(p) & (b)) != 0)

#define set_mid_type_bit(p, b)         (p)->tf.bits.mid_bits |= (b)
#define clear_mid_type_bit(p, b)       (p)->tf.bits.mid_bits &= (~(b))
#define has_mid_type_bit(p, b)         (((p)->tf.bits.mid_bits & (b)) != 0)

#define set_high_type_bit(p, b)        (p)->tf.bits.high_bits |= (b)
#define clear_high_type_bit(p, b)      (p)->tf.bits.high_bits &= (~(b))
#define has_high_type_bit(p, b)        (((p)->tf.bits.high_bits & (b)) != 0)

/* -------- low type bits -------- */
#define T_Syntactic                    (1 << (8 + 1))
#define is_symbol_and_syntactic(p)     (low_type_bits(T_Ext(p)) == (uint16_t)(T_SYMBOL | T_Syntactic))
#define is_syntactic_symbol(p)         has_low_type_bit(T_Sym(p), T_Syntactic)
#define is_syntactic_pair(p)           has_low_type_bit(T_Pair(p), T_Syntactic)
#define clear_syntactic(p)             clear_low_type_bit(T_Pair(p), T_Syntactic)
#define set_syntactic_pair(p)          full_type(T_Pair(p)) = (T_PAIR | T_Syntactic | (full_type(p) & (0xffffffffffff0000 & ~T_Optimized))) /* used only in pair_set_syntax_op */
/* this marks symbols that represent syntax objects, it should be in the second byte */

#define T_Simple_Arg_Defaults          (1 << (8 + 2))
#define lambda_has_simple_defaults(p)  has_low_type_bit(T_Pair(closure_body(p)), T_Simple_Arg_Defaults)
#define lambda_set_simple_defaults(p)  set_low_type_bit(T_Pair(p), T_Simple_Arg_Defaults)
/* are all lambda* default values simple? This is set on closure_body, so it doesn't mess up closure_is_ok_1 */

#define T_Safe_List_In_Use             T_Simple_Arg_Defaults  /* only on sc->safe_lists */
#define safe_list_is_in_use(p)         has_low_type_bit(T_Pair(p), T_Safe_List_In_Use)
#if s7_Debugging
  #define set_safe_list_in_use(Sc, p)   do {set_low_type_bit(T_Pair(p), T_Safe_List_In_Use); p->carrier_line = __LINE__;} while (0)
  #define clear_safe_list_in_use(Sc, p) do {clear_low_type_bit(T_Pair(p), T_Safe_List_In_Use); p->carrier_line = 0; Sc->current_safe_list = 0;} while (0)
#else
  #define set_safe_list_in_use(Sc, p)   set_low_type_bit(p, T_Safe_List_In_Use)
  #define clear_safe_list_in_use(Sc, p) do {clear_low_type_bit(p, T_Safe_List_In_Use); Sc->current_safe_list = 0;} while (0)
#endif

#define T_One_Form                     T_Simple_Arg_Defaults
#define set_closure_has_one_form(p)    set_low_type_bit(T_Clo(p), T_One_Form)
#define T_Multiform                    (1 << (8 + 0))
#define set_closure_has_multiform(p)   set_low_type_bit(T_Clo(p), T_Multiform)
#define T_One_Form_Fx_Arg              (T_One_Form | T_Multiform)
#define set_closure_one_form_fx_arg(p) set_low_type_bit(T_Clo(p), T_One_Form_Fx_Arg)
/* can't use T_Has_Fx here because closure_is_ok wants to examine low_type_bits */

#define T_Optimized                    (1 << (8 + 3))
#define set_optimized(p)               set_low_type_bit(T_Pair(p), T_Optimized)
#define clear_optimized(p)             clear_low_type_bit(T_Pair(p), T_Optimized | T_Syntactic | T_Has_Fx | T_Has_Fn)
#define is_optimized(p)                (low_type_bits(T_Ext(p)) == (uint16_t)(T_PAIR | T_Optimized))
/* optimizer flag for an expression that has optimization info, it should be in the second byte */

#define T_Scope_Safe                   T_Optimized
#define is_scope_safe(p)               has_low_type_bit(T_Fnc(p), T_Scope_Safe)
#define set_scope_safe(p)              set_low_type_bit(T_Fnc(p), T_Scope_Safe)

#define T_Safe_Closure                 (1 << (8 + 4))
#define is_safe_closure(p)             has_low_type_bit(T_Clo(p), T_Safe_Closure)
#define set_safe_closure(p)            set_low_type_bit(T_Clo(p), T_Safe_Closure)
#define is_safe_closure_body(p)        has_low_type_bit(T_Pair(p), T_Safe_Closure)
#define set_safe_closure_body(p)       set_low_type_bit(T_Pair(p), T_Safe_Closure)
#define clear_safe_closure_body(p)     clear_low_type_bit(T_Pair(p), T_Safe_Closure)

/* optimizer flag for a closure body that is completely simple (every expression is safe)
 *   set_safe_closure happens in define_funchcecked letrec_setup_closures etc, clear only in procedure_source, bits only here
 *   this has to be separate from T_Safe_Procedure, and should be in the second byte (closure_is_ok_1 checks low_type_bits).
 * define -> optimize_lambda sets safe -> define_funchecked -> make_funclet for the let
 *   similarly, named let -> optimize_lambda, then let creates the let if safe
 *   thereafter, optimizer uses OP_Safe_Closure* which calls update_let*
 */

#define T_Dont_Eval_Args               (1 << (8 + 5))
#define dont_eval_args(p)              has_low_type_bit(T_Ext(p), T_Dont_Eval_Args)
/* this marks things that don't evaluate their arguments */

#define T_Expansion                    (1 << (8 + 6))
#define is_expansion(p)                has_low_type_bit(T_Ext(p), T_Expansion)
#define clear_expansion(p)             clear_low_type_bit(T_Sym(p), T_Expansion)
/* this marks the symbol and its run-time macro value, distinguishing it from an ordinary macro */

#define T_Multiple_Value               (1 << (8 + 7))
#define is_multiple_value(p)           has_low_type_bit(T_Exs(p), T_Multiple_Value) /* not T_Ext -- can be a slot */
#if s7_Debugging
  #define set_multiple_value(p)        do {if (!in_heap(p)) {fprintf(stderr, "%s[%d] (from set_multiple_value): arg not in heap\n", __func__, __LINE__); abort();} set_low_type_bit(T_Pair(p), T_Multiple_Value);} while (0)
#else
  #define set_multiple_value(p)        set_low_type_bit(T_Pair(p), T_Multiple_Value)
#endif
#define clear_multiple_value(p)        clear_low_type_bit(T_Pair(p), T_Multiple_Value)
#define multiple_value(p)              p
/* this bit marks a list (from "values") that is waiting for a chance to be spliced into its caller's argument list */

#define T_Matched                      T_Multiple_Value
#define is_matched_pair(p)             has_low_type_bit(T_Pair(p), T_Matched)
#define clear_match_pair(p)            clear_low_type_bit(T_Pair(p), T_Matched)
#define set_match_pair(p)              set_low_type_bit(T_Pair(p), T_Matched)
#define set_match_symbol(p)            set_low_type_bit(T_Sym(p), T_Matched)
#define is_matched_symbol(p)           has_low_type_bit(T_Sym(p), T_Matched)
#define clear_match_symbol(p)          clear_low_type_bit(T_Sym(p), T_Matched)

/* -------- mid type bits -------- */

#define T_Unsafe_Do                    (1 << (16 + 0))
#define T_Mid_Unsafe_Do                (1 << 0)
#define is_unsafe_do(p)                has_mid_type_bit(T_Pair(p), T_Mid_Unsafe_Do)
#define set_unsafe_do(p)               set_mid_type_bit(T_Pair(p), T_Mid_Unsafe_Do)
/* marks do-loops that resist optimization */

#define T_Mid_Dox_Slot1                T_Mid_Unsafe_Do
#define has_dox_slot1(p)               has_mid_type_bit(T_Let(p), T_Mid_Dox_Slot1)
#define set_has_dox_slot1(p)           set_mid_type_bit(T_Let(p), T_Mid_Dox_Slot1)
/* marks a let that includes the dox_slot1 */

#define T_Mid_Even_Args                T_Mid_Unsafe_Do
#define has_even_args(p)               has_mid_type_bit(T_CFn(p), T_Mid_Even_Args)
#define set_has_even_args(p)           set_mid_type_bit(T_CFn(p), T_Mid_Even_Args)

#define T_Mid_Maybe_Shadowed           T_Mid_Unsafe_Do
#define is_maybe_shadowed(p)           has_mid_type_bit(T_Sym(p), T_Mid_Maybe_Shadowed)
#define set_is_maybe_shadowed(p)       set_mid_type_bit(T_Sym(p), T_Mid_Maybe_Shadowed)

#define T_Collected                    (1 << (16 + 1))
#define T_Mid_Collected                (1 << 1)
#define is_collected(p)                has_mid_type_bit(T_Seq(p), T_Mid_Collected)
#define is_collected_unchecked(p)      has_mid_type_bit(p, T_Mid_Collected)
#define set_collected(p)               set_mid_type_bit(T_Seq(p), T_Mid_Collected)
/* #define clear_collected(p)          clear_mid_type_bit(T_Seq(p), T_Mid_Collected) */
/* this is a transient flag used by the printer to catch cycles.  It affects only objects that have structure.
 *   We can't use a low bit (bit 7 for example), because collect_shared_info inspects the object's type.
 */

#define T_Location                     (1 << (16 + 2))
#define T_Mid_Location                 (1 << 2)
#define has_location(p)                has_mid_type_bit(T_Pair(p), T_Mid_Location)
#define set_has_location(p)            set_mid_type_bit(T_Pair(p), T_Mid_Location)
/* pair in question has line/file/position info added during read, or the environment has function placement info
 *   this bit should not be in the first byte -- Syntactic_Pair ignores it.
 */

#define T_Loader_Port                  T_Mid_Location
#define is_loader_port(p)              has_mid_type_bit(T_Pri(p), T_Loader_Port)
#define set_loader_port(p)             set_mid_type_bit(T_Pri(p), T_Loader_Port)
#define clear_loader_port(p)           clear_mid_type_bit(T_Pri(p), T_Loader_Port)
/* this bit marks a port used by the loader so that random load-time reads do not screw up the load process */

#define T_Has_Setter                   T_Mid_Location
#define slot_has_setter(p)             has_mid_type_bit(T_Slt(p), T_Has_Setter)
#define slot_set_has_setter(p)         set_mid_type_bit(T_Slt(p), T_Has_Setter)
/* marks a slot that has a setter or symbol that might have a setter */

#define T_With_Let_Let                 T_Mid_Location
#define is_with_let_let(p)             has_mid_type_bit(T_Let(p), T_With_Let_Let)
#define set_with_let_let(p)            set_mid_type_bit(T_Let(p), T_With_Let_Let)
/* marks a let that is the argument to with-let (but not rootlet in its uses) */

#define T_Simple_Defaults              T_Mid_Location
#define c_func_has_simple_defaults(p)  has_mid_type_bit(T_Fst(p), T_Simple_Defaults)
#define c_func_set_simple_defaults(p)  set_mid_type_bit(T_Fst(p), T_Simple_Defaults)
#define c_func_clear_simple_defaults(p) clear_mid_type_bit(T_Fst(p), T_Simple_Defaults)
/* flag c_func_star arg defaults that need GC protection */

#define T_No_Setter                    T_Mid_Location
#define closure_no_setter(p)           has_mid_type_bit(T_Clo(p), T_No_Setter)
#define closure_set_no_setter(p)       set_mid_type_bit(T_Clo(p), T_No_Setter)

#define T_Shared                       (1 << (16 + 3))
#define T_Mid_Shared                   (1 << 3)
#define is_shared(p)                   has_mid_type_bit(T_Seq(p), T_Mid_Shared)
#define set_shared(p)                  set_mid_type_bit(T_Seq(p), T_Mid_Shared)
#define is_collected_or_shared(p)      has_mid_type_bit(T_Seq(p), T_Mid_Collected | T_Mid_Shared)
#define clear_collected_and_shared(p)  clear_mid_type_bit(T_Seq(p), T_Mid_Collected | T_Mid_Shared) /* this can clear free cells = calloc */

#define T_Low_Count                    (1 << (16 + 4))
#define T_Mid_Low_Count                (1 << 4)
#define has_low_count(p)               has_mid_type_bit(T_Pair(p), T_Low_Count)
#define set_has_low_count(p)           set_mid_type_bit(T_Pair(p), T_Low_Count)

#define T_Tc                           T_Mid_Low_Count
#define has_tc(p)                      has_mid_type_bit(T_Pair(p), T_Tc)
#define set_has_tc(p)                  set_mid_type_bit(T_Pair(p), T_Tc)

#define T_Initial_Value                T_Mid_Low_Count
#define is_initial_value(p)            has_mid_type_bit(p, T_Initial_Value)
#define set_is_initial_value(p)        set_mid_type_bit(p, T_Initial_Value)
#define initial_value_is_defined(Sc, p) (initial_value(T_Sym(p)) != Sc->undefined)

#define T_Safe_Procedure               (1 << (16 + 5))
#define T_Mid_Safe_Procedure           (1 << 5)
#define is_safe_procedure(p)           has_mid_type_bit(T_App(p), T_Mid_Safe_Procedure)
#define is_safe_or_scope_safe_procedure(p) ((full_type(T_Fnc(p)) & (T_Scope_Safe | T_Safe_Procedure)) != 0) /* T_Scope_Safe is a low_type bit */
/* applicable objects that do not return or modify their arg list directly (no :rest arg in particular),
 *    and that can't call themselves either directly or via s7_call, and that don't mess with the stack.
 */

#define T_Checked                      (1 << (16 + 6))
#define T_Mid_Checked                  (1 << 6)
#define set_checked(p)                 set_mid_type_bit(T_Pair(p), T_Mid_Checked)
#define is_checked(p)                  has_mid_type_bit(T_Pair(p), T_Mid_Checked)
#define clear_checked(p)               clear_mid_type_bit(T_Pair(p), T_Mid_Checked)
#define set_checked_slot(p)            set_mid_type_bit(T_Slt(p), T_Mid_Checked)
#define is_checked_slot(p)             has_mid_type_bit(T_Slt(p), T_Mid_Checked)
#define clear_checked_slot(p)          clear_mid_type_bit(T_Slt(p), T_Mid_Checked)

#define T_All_Integer                  T_Mid_Checked
#define is_all_integer(p)              has_mid_type_bit(T_Sym(p), T_All_Integer)
#define set_all_integer(p)             set_mid_type_bit(T_Sym(p), T_All_Integer)

#define T_Unsafe                       (1 << (16 + 7))
#define T_Mid_Unsafe                   (1 << 7)
#define set_unsafe(p)                  set_mid_type_bit(T_Pair(p), T_Mid_Unsafe)
#define set_unsafely_optimized(p)      full_type(T_Pair(p)) = (full_type(p) | T_Unsafe | T_Optimized) /* T_Optimized is a low_type bit */
#define is_unsafe(p)                   has_mid_type_bit(T_Pair(p), T_Mid_Unsafe)
#define clear_unsafe(p)                clear_mid_type_bit(T_Pair(p), T_Mid_Unsafe)
#define is_safely_optimized(p)         ((full_type(T_Pair(p)) & (T_Optimized | T_Unsafe)) == T_Optimized)
/* optimizer flag saying "this expression is not completely self-contained.  It might involve the stack, etc" */
/* see also T_No_Float_Opt below */

#define T_Clean_Symbol                 T_Mid_Unsafe
#define is_clean_symbol(p)             has_mid_type_bit(T_Sym(p), T_Clean_Symbol)
#define set_clean_symbol(p)            set_mid_type_bit(T_Sym(p), T_Clean_Symbol)
/* set if we know the symbol name can be printed without quotes (slashification) */

#define T_Has_Stepper                  T_Mid_Unsafe
#define has_stepper(p)                 has_mid_type_bit(T_Slt(p), T_Has_Stepper)
#define set_has_stepper(p)             set_mid_type_bit(T_Slt(p), T_Has_Stepper)

#define T_Dox_Slot2                    T_Mid_Unsafe
#define has_dox_slot2(p)               has_mid_type_bit(T_Let(p), T_Dox_Slot2)
#define set_has_dox_slot2(p)           set_mid_type_bit(T_Let(p), T_Dox_Slot2)
/* marks a let that includes the dox_slot2 */

#define T_Immutable                    (1 << (16 + 8))
#define T_Mid_Immutable                (1 << 8)
#define is_immutable(p)                has_mid_type_bit(T_Exs(p), T_Mid_Immutable)
#define set_immutable(p)               set_mid_type_bit(T_Exs(p), T_Mid_Immutable) /* can be a slot, so not T_Ext */
#define set_immutable_let(p)           set_mid_type_bit(T_Let(p), T_Mid_Immutable)
#define set_immutable_slot(p)          set_mid_type_bit(T_Slt(p), T_Mid_Immutable)
#define set_immutable_string(p)        set_mid_type_bit(T_Str(p), T_Mid_Immutable)
#define set_immutable_pair(p)          set_mid_type_bit(T_Pair(p), T_Mid_Immutable)
#define is_immutable_port(p)           has_mid_type_bit(T_Prt(p), T_Mid_Immutable)
#define is_immutable_symbol(p)         has_mid_type_bit(T_Sym(p), T_Mid_Immutable)
#define is_immutable_slot(p)           has_mid_type_bit(T_Slt(p), T_Mid_Immutable)
#define is_immutable_pair(p)           has_mid_type_bit(T_Pair(p), T_Mid_Immutable)
#define is_immutable_vector(p)         has_mid_type_bit(T_Vec(p), T_Mid_Immutable)
#define is_immutable_string(p)         has_mid_type_bit(T_Str(p), T_Mid_Immutable)
#define is_immutable_hash_table(p)     has_mid_type_bit(T_Hsh(p), T_Mid_Immutable)
#define is_immutable_let(p)            has_mid_type_bit(T_Let(p), T_Mid_Immutable)
#define ignores_immutable(p)           t_ignores_immutable_p[type(p)]
/* T_Immutable is compatible with T_Mutable -- the latter is an internal bit for locally mutable numbers */

#define T_Full_Allow_Other_Keys        (1 << (16 + 9))
#define T_Allow_Other_Keys             (1 << 9)
#define set_allow_other_keys(p)        set_mid_type_bit(T_Pair(p), T_Allow_Other_Keys)
#define allows_other_keys(p)           has_mid_type_bit(T_Pair(p), T_Allow_Other_Keys)
#define c_function_set_allow_other_keys(p) set_mid_type_bit(T_Fst(p), T_Allow_Other_Keys)
#define c_function_allows_other_keys(p)    has_mid_type_bit(T_Fst(p), T_Allow_Other_Keys)
/* marks arglist (or c_function*) that allows keyword args other than those in the parameter list;
 *   we can't allow (define* (f :allow-other-keys)...) (where there are no args) because there's only one nil, and besides, it does say "other".
 */

#define T_Let_Removed                  T_Allow_Other_Keys
#define let_set_removed(p)             set_mid_type_bit(T_Let(p), T_Let_Removed)
#define let_removed(p)                 has_mid_type_bit(T_Let(p), T_Let_Removed)
/* mark lets that have been removed from the heap or checked for that possibility */

#define T_Has_Expression               T_Allow_Other_Keys
#define slot_set_has_expression(p)     set_mid_type_bit(T_Slt(p), T_Has_Expression)
#define slot_has_expression(p)         has_mid_type_bit(T_Slt(p), T_Has_Expression)

#define T_Mutable                      (1 << (16 + 10))
#define T_Mid_Mutable                  (1 << 10)
#define is_mutable(p)                  has_mid_type_bit(p, T_Mid_Mutable)
#define is_mutable_number(p)           has_mid_type_bit(T_Num(p), T_Mid_Mutable)
#define is_mutable_integer(p)          has_mid_type_bit(T_Int(p), T_Mid_Mutable)
#if s7_Debugging
#define clear_mutable_number(p)        do {check_mutable_bit(p); clear_mid_type_bit(T_Num(p), T_Mid_Mutable);} while (0)
#define clear_mutable_integer(p)       do {check_mutable_bit(p); clear_mid_type_bit(T_Int(p), T_Mid_Mutable);} while (0)
#else
#define clear_mutable_number(p)        clear_mid_type_bit(T_Num(p), T_Mid_Mutable)
#define clear_mutable_integer(p)       clear_mid_type_bit(T_Int(p), T_Mid_Mutable)
#endif
/* used for mutable numbers, can occur with T_Immutable (outside view vs inside) */

#define T_Has_Keyword                  T_Mid_Mutable
#define has_keyword(p)                 has_mid_type_bit(T_Sym(p), T_Has_Keyword)
#define set_has_keyword(p)             set_mid_type_bit(T_Sym(p), T_Has_Keyword)

#define T_Mark_Seq                     T_Mid_Mutable
#define has_carrier(p)                 has_mid_type_bit(T_Itr(p), T_Mark_Seq)
#if s7_Debugging
  #define set_has_carrier(p)           do {set_mid_type_bit(T_Itr(p), T_Mark_Seq); p->carrier_line = __LINE__;} while (0)
#else
  #define set_has_carrier(p)           set_mid_type_bit(T_Itr(p), T_Mark_Seq)
#endif
/* used in iterators for GC mark of sequence */

#define T_Has_Loop_End                 T_Mid_Mutable
#define has_loop_end(p)                has_mid_type_bit(T_Slt(p), T_Has_Loop_End)
#define loop_end_fits(Slot, Len)       ((has_loop_end(Slot)) && (denominator(slot_value(Slot)) <= Len))
#define set_has_loop_end(p)            set_mid_type_bit(T_Slt(p), T_Has_Loop_End)
/* marks a slot that holds a do-loop's step-or-end variable, numerator=current, denominator=end */

#define T_No_Cell_Opt                  T_Mid_Mutable
#define set_no_cell_opt(p)             set_mid_type_bit(T_Pair(p), T_No_Cell_Opt)
#define no_cell_opt(p)                 has_mid_type_bit(T_Pair(p), T_No_Cell_Opt)

#define T_Is_Elist                     T_Mutable
#define T_Mid_Is_Elist                 T_Mid_Mutable
#define set_is_elist(p)                set_mid_type_bit(T_Lst(p), T_Mid_Is_Elist)
#define is_elist(p)                    has_mid_type_bit(T_Lst(p), T_Mid_Is_Elist)

#define T_No_Int_Opt                   T_Allow_Other_Keys
#define set_no_int_opt(p)              set_mid_type_bit(T_Pair(p), T_No_Int_Opt)
#define no_int_opt(p)                  has_mid_type_bit(T_Pair(p), T_No_Int_Opt)

#define T_No_Float_Opt                 T_Mid_Unsafe
#define set_no_float_opt(p)            set_mid_type_bit(T_Pair(p), T_No_Float_Opt)
#define no_float_opt(p)                has_mid_type_bit(T_Pair(p), T_No_Float_Opt)

#define T_Integer_Keys                 T_Allow_Other_Keys
#define set_has_integer_keys(p)        set_mid_type_bit(T_Pair(p), T_Integer_Keys)
#define has_integer_keys(p)            has_mid_type_bit(T_Pair(p), T_Integer_Keys)

#define T_Safe_Stepper                 (1 << (16 + 11))
#define T_Mid_Safe_Stepper             (1 << 11)
#define is_safe_stepper(p)             has_mid_type_bit(T_Slt(p), T_Mid_Safe_Stepper)
#define set_safe_stepper(p)            set_mid_type_bit(T_Slt(p), T_Mid_Safe_Stepper)
#define clear_safe_stepper(p)          clear_mid_type_bit(T_Slt(p), T_Mid_Safe_Stepper)
#define is_safe_stepper_expr(p)        has_mid_type_bit(T_Pair(p), T_Mid_Safe_Stepper)
#define set_safe_stepper_expr(p)       set_mid_type_bit(T_Pair(p), T_Mid_Safe_Stepper)

#define T_Wrapper                      T_Safe_Stepper
#define T_Mid_Wrapper                  T_Mid_Safe_Stepper
#define is_wrapper(p)                  has_mid_type_bit(T_Str(p), T_Mid_Wrapper)
/* string_wrapper only currently */

#define T_No_Bool_Opt                  T_Mid_Safe_Stepper
#define set_no_bool_opt(p)             set_mid_type_bit(T_Pair(p), T_No_Bool_Opt)
#define no_bool_opt(p)                 has_mid_type_bit(T_Pair(p), T_No_Bool_Opt)

#define T_Maybe_Safe                   T_Mid_Safe_Stepper
#define is_maybe_safe(p)               has_mid_type_bit(T_Fnc(p), T_Maybe_Safe)
#define set_maybe_safe(p)              set_mid_type_bit(T_Fnc(p), T_Maybe_Safe)

#define T_Pair_Macro                   T_Mid_Safe_Stepper
#define has_pair_macro(p)              has_mid_type_bit(T_Mac(p), T_Pair_Macro)
#define set_has_pair_macro(p)          set_mid_type_bit(T_Mac(p), T_Pair_Macro)

#define T_Weak_Hash                    T_Mid_Safe_Stepper
#define set_weak_hash_table(p)         set_mid_type_bit(T_Hsh(p), T_Weak_Hash)
#define is_weak_hash_table(p)          has_mid_type_bit(T_Hsh(p), T_Weak_Hash)

#define T_All_Float                    T_Mid_Safe_Stepper
#define is_all_float(p)                has_mid_type_bit(T_Sym(p), T_All_Float)
#define set_all_float(p)               set_mid_type_bit(T_Sym(p), T_All_Float)
#define set_all_integer_and_float(p)   set_mid_type_bit(T_Sym(p), (T_All_Integer | T_All_Float))

#define T_Copy_Args                    (1 << (16 + 12))
#define T_Mid_Copy_Args                (1 << 12)
#define needs_copied_args(p)           has_mid_type_bit(T_Ext(p), T_Mid_Copy_Args) /* set via explicit T_Copy_Args */
#define set_needs_copied_args(p)       set_mid_type_bit(T_Pair(p), T_Mid_Copy_Args)
#define clear_needs_copied_args(p)     clear_mid_type_bit(T_Pair(p), T_Mid_Copy_Args)
/* this marks something that might mess with its argument list, it should not be in the second byte */

#define T_Gensym                       (1 << (16 + 13))
#define T_Mid_Gensym                   (1 << 13)
#define is_gensym(p)                   has_mid_type_bit(T_Sym(p), T_Mid_Gensym)
/* symbol is from gensym (GC-able etc) */

#define T_Funclet                      T_Gensym
#define T_Mid_Funclet                  T_Mid_Gensym
#define is_funclet(p)                  has_mid_type_bit(T_Let(p), T_Mid_Funclet)
#define set_funclet(p)                 set_mid_type_bit(T_Let(p), T_Mid_Funclet)
/* this marks a funclet */

#define T_Hash_Chosen                  T_Mid_Gensym
#define hash_chosen(p)                 has_mid_type_bit(T_Hsh(p), T_Hash_Chosen)
#define hash_set_chosen(p)             set_mid_type_bit(T_Hsh(p), T_Hash_Chosen)
#define hash_clear_chosen(p)           clear_mid_type_bit(T_Hsh(p), T_Hash_Chosen)

#define T_Documented                   T_Mid_Gensym
#define is_documented(p)               has_mid_type_bit(T_Str(p), T_Documented)
#define set_documented(p)              set_mid_type_bit(T_Str(p), T_Documented)
/* this marks a symbol that has documentation (bit is set on name cell) */

#define T_Fx_Treed                     T_Mid_Gensym
#define is_fx_treed(p)                 has_mid_type_bit(T_Pair(p), T_Fx_Treed)
#define set_fx_treed(p)                set_mid_type_bit(T_Pair(p), T_Fx_Treed)

#define T_Subvector                    T_Gensym
#define T_Mid_Subvector                T_Mid_Gensym
#define is_subvector(p)                has_mid_type_bit(T_Vec(p), T_Mid_Subvector)

#define T_Has_Pending_Value            T_Mid_Gensym
#define slot_set_has_pending_value(p)  set_mid_type_bit(T_Slt(p), T_Has_Pending_Value)
#define slot_has_pending_value(p)      has_mid_type_bit(T_Slt(p), T_Has_Pending_Value)
#define slot_clear_has_pending_value(p) do {clear_mid_type_bit(T_Slt(p), T_Has_Pending_Value); slot_set_pending_value(p, sc->F);} while (0)
#define slot_has_setter_or_pending_value(p) has_mid_type_bit(T_Slt(p), T_Has_Setter | T_Has_Pending_Value)

#define T_Has_Methods                  (1 << (16 + 14))
#define T_Mid_Has_Methods              (1 << 14)
#define has_methods(p)                 has_mid_type_bit(T_Exs(p), T_Mid_Has_Methods) /* display slot hits T_Ext here */
#define has_methods_unchecked(p)       has_mid_type_bit(p, T_Mid_Has_Methods)
#define is_openlet(p)                  has_mid_type_bit(T_Let(p), T_Mid_Has_Methods)
#define has_active_methods(Sc, p)      ((has_mid_type_bit(T_Ext(p), T_Mid_Has_Methods)) && (Sc->has_openlets)) /* g_char #<eof> */
#define set_has_methods(p)             set_mid_type_bit(T_Met(p), T_Mid_Has_Methods)
#define clear_has_methods(p)           clear_mid_type_bit(T_Met(p), T_Mid_Has_Methods)
/* this marks a let or closure that is "open" for generic functions etc, don't reuse this bit */

#define mid_type(p)                    (p)->tf.bits.mid_bits
#define T_Has_Let_Set_Fallback         T_Safe_Stepper
#define T_Mid_Has_Let_Set_Fallback     T_Mid_Safe_Stepper
#define T_Has_Let_Ref_Fallback         T_Mutable
#define T_Mid_Has_Let_Ref_Fallback     T_Mid_Mutable
#define has_let_ref_fallback(p)        ((mid_type(T_Let(p)) & (T_Mid_Has_Let_Ref_Fallback | T_Mid_Has_Methods)) == (T_Mid_Has_Let_Ref_Fallback | T_Mid_Has_Methods))
#define has_let_set_fallback(p)        ((mid_type(T_Let(p)) & (T_Mid_Has_Let_Set_Fallback | T_Mid_Has_Methods)) == (T_Mid_Has_Let_Set_Fallback | T_Mid_Has_Methods))
#define set_has_let_ref_fallback(p)    set_mid_type_bit(T_Let(p), T_Mid_Has_Let_Ref_Fallback)
#define set_has_let_set_fallback(p)    set_mid_type_bit(T_Let(p), T_Mid_Has_Let_Set_Fallback)
#define has_let_fallback(p)            has_mid_type_bit(T_Let(p), (T_Mid_Has_Let_Ref_Fallback | T_Mid_Has_Let_Set_Fallback))
#define set_all_methods(p, Let)        mid_type(T_Let(p)) |= (mid_type(Let) & (T_Mid_Has_Methods | T_Mid_Has_Let_Ref_Fallback | T_Mid_Has_Let_Set_Fallback))

#define T_Iter_Ok                      (1LL << (16 + 15))
#define T_Mid_Iter_Ok                  (1 << 15)
#define iter_ok(p)                     has_mid_type_bit(T_Itr(p), T_Mid_Iter_Ok)
#define clear_iter_ok(p)               clear_mid_type_bit(T_Itr(p), T_Mid_Iter_Ok)

#define T_Loop_End_Possible            T_Mid_Iter_Ok
#define loop_end_possible(p)           has_mid_type_bit(T_Pair(p), T_Loop_End_Possible)
#define set_loop_end_possible(p)       set_mid_type_bit(T_Pair(p), T_Loop_End_Possible)

#define T_In_Rootlet                   T_Mid_Iter_Ok
#define in_rootlet(p)                  has_mid_type_bit(T_Slt(p), T_In_Rootlet)
#define set_in_rootlet(p)              set_mid_type_bit(T_Slt(p), T_In_Rootlet)

#define T_Bool_Function                T_Mid_Iter_Ok
#define is_bool_function(p)            has_mid_type_bit(T_Prc(p), T_Bool_Function)
#define set_is_bool_function(p)        set_mid_type_bit(T_CFn(p), T_Bool_Function)

#define T_Symbol_From_Symbol           T_Mid_Iter_Ok
#define is_symbol_from_symbol(p)       has_mid_type_bit(T_Sym(p), T_Symbol_From_Symbol)
#define set_is_symbol_from_symbol(p)   set_mid_type_bit(T_Sym(p), T_Symbol_From_Symbol)
#define clear_symbol_from_symbol(p)    clear_mid_type_bit(T_Sym(p), T_Symbol_From_Symbol) /* was high_type?? 20-Dec-23 */


/* -------- high type bits -------- */
/* it's faster here to use the high_bits bits rather than typeflag bits */
#define T_Full_Symcons                 (1LL << (48 + 0))
#define T_Symcons                      (1 << 0)
#define is_possibly_constant(p)        has_high_type_bit(T_Sym(p), T_Symcons)
#define set_possibly_constant(p)       set_high_type_bit(T_Sym(p), T_Symcons)
#define is_probably_constant(p)        has_type_bit(T_Sym(p), (T_Full_Symcons | T_Immutable))

#define T_Has_Let_Arg                  T_Symcons
#define has_let_arg(p)                 has_high_type_bit(T_Prc(p), T_Has_Let_Arg)
#define set_has_let_arg(p)             set_high_type_bit(T_Prc(p), T_Has_Let_Arg)
/* p is a setter procedure, "let arg" refers to the setter's optional third (let) argument */

#define T_Hash_Value_Type              T_Symcons
#define has_hash_value_type(p)         has_high_type_bit(T_Hsh(p), T_Hash_Value_Type)
#define set_has_hash_value_type(p)     set_high_type_bit(T_Hsh(p), T_Hash_Value_Type)

#define T_Int_Optable                  T_Symcons
#define is_int_optable(p)              has_high_type_bit(T_Pair(p), T_Int_Optable)
#define set_is_int_optable(p)          set_high_type_bit(T_Pair(p), T_Int_Optable)

#define T_Unlet                        T_Symcons
#define is_unlet(p)                    has_high_type_bit(T_Let(p), T_Unlet)
#define set_is_unlet(p)                set_high_type_bit(T_Let(p), T_Unlet)

#define T_Symbol_Table                 T_Symcons
#define is_symbol_table(p)             has_high_type_bit(T_Nvc(p), T_Symbol_Table)
#define set_is_symbol_table(p)         set_high_type_bit(T_Nvc(p), T_Symbol_Table)

#define T_Full_Has_Let_File            (1LL << (48 + 1))
#define T_Has_Let_File                 (1 << 1)
#define has_let_file(p)                has_high_type_bit(T_Let(p), T_Has_Let_File)
#define set_has_let_file(p)            set_high_type_bit(T_Let(p), T_Has_Let_File)
#define clear_has_let_file(p)          clear_high_type_bit(T_Let(p), T_Has_Let_File)

#define T_Typed_Vector                 T_Has_Let_File
#define is_typed_vector(p)             has_high_type_bit(T_Nvc(p), T_Typed_Vector)
#define is_typed_t_vector(p)           ((is_t_vector(p)) && (is_typed_vector(p)))
#define set_typed_vector(p)            set_high_type_bit(T_Nvc(p), T_Typed_Vector)
#define clear_typed_vector(p)          clear_high_type_bit(T_Nvc(p), T_Typed_Vector)

#define T_Typed_Hash_Table             T_Has_Let_File
#define is_typed_hash_table(p)         has_high_type_bit(T_Hsh(p), T_Typed_Hash_Table)
#define set_is_typed_hash_table(p)     set_high_type_bit(T_Hsh(p), T_Typed_Hash_Table)
#define clear_is_typed_hash_table(p)   clear_high_type_bit(T_Hsh(p), T_Typed_Hash_Table)

#define T_Bool_Setter                  T_Has_Let_File
#define c_function_has_bool_setter(p)  has_high_type_bit(T_CFn(p), T_Bool_Setter)
#define c_function_set_has_bool_setter(p) set_high_type_bit(T_CFn(p), T_Bool_Setter)

#define T_Rest_Slot                    T_Has_Let_File
#define is_rest_slot(p)                has_high_type_bit(T_Slt(p), T_Rest_Slot)
#define set_is_rest_slot(p)            set_high_type_bit(T_Slt(p), T_Rest_Slot)

#define T_No_Defaults                  T_Has_Let_File
#define T_Full_No_Defaults             T_Full_Has_Let_File
#define has_no_defaults(p)             has_high_type_bit(T_Pcs(p), T_No_Defaults)
#define set_has_no_defaults(p)         set_high_type_bit(T_Pcs(p), T_No_Defaults)
/* pair=closure* body, transferred to closure* */

#define T_Full_Definer                 (1LL << (48 + 2))
#define T_Definer                      (1 << 2)
#define is_definer(p)                  has_high_type_bit(T_Sym(p), T_Definer)
#define set_is_definer(p)              set_high_type_bit(T_Sym(p), T_Definer)
#define is_func_definer(p)             has_high_type_bit(T_CFn(p), T_Definer)
#define set_func_is_definer(p)         do {set_high_type_bit(T_CFn(initial_value(p)), T_Definer); set_high_type_bit(T_Sym(p), T_Definer);} while (0)
#define is_syntax_definer(p)           has_high_type_bit(T_Syn(p), T_Definer)
#define set_syntax_is_definer(p)       do {set_high_type_bit(T_Syn(initial_value(p)), T_Definer); set_high_type_bit(T_Sym(p), T_Definer);} while (0)
/* this marks "definers" like define and define-macro */

#define T_Maclet                       T_Definer
#define is_maclet(p)                   has_high_type_bit(T_Let(p), T_Maclet)
#define set_maclet(p)                  set_high_type_bit(T_Let(p), T_Maclet)

#define T_Has_Fx                       T_Definer
#define set_has_fx(p)                  set_high_type_bit(T_Pair(p), T_Has_Fx)
#define has_fx(p)                      has_high_type_bit(T_Pair(p), T_Has_Fx)
#define clear_has_fx(p)                clear_high_type_bit(T_Pair(p), T_Has_Fx)

#define T_Slot_Defaults                T_Definer
#define slot_defaults(p)               has_high_type_bit(T_Slt(p), T_Slot_Defaults)
#define set_slot_defaults(p)           set_high_type_bit(T_Slt(p), T_Slot_Defaults)

#define T_Weak_Hash_Iterator           T_Definer
#define is_weak_hash_iterator(p)       has_high_type_bit(T_Itr(p), T_Weak_Hash_Iterator)
#define set_weak_hash_iterator(p)      set_high_type_bit(T_Itr(p), T_Weak_Hash_Iterator)
#define clear_weak_hash_iterator(p)    clear_high_type_bit(T_Itr(p), T_Weak_Hash_Iterator)

#define T_Hash_Key_Type                T_Definer
#define has_hash_key_type(p)           has_high_type_bit(T_Hsh(p), T_Hash_Key_Type)
#define set_has_hash_key_type(p)       set_high_type_bit(T_Hsh(p), T_Hash_Key_Type)

#define T_Full_Binder                  (1LL << (48 + 3))
#define T_Binder                       (1 << 3)
#define set_syntax_is_binder(p)        do {set_high_type_bit(T_Syn(initial_value(p)), T_Binder); set_high_type_bit(T_Sym(p), T_Binder);} while (0)
#define is_definer_or_binder(p)        has_high_type_bit(T_Sym(p), T_Definer | T_Binder)
/* this marks "binders" like let */

#define T_Semisafe                     T_Binder
#define is_semisafe(p)                 has_high_type_bit(T_CFn(p), T_Semisafe)
#define set_is_semisafe(p)             set_high_type_bit(T_CFn(p), T_Semisafe)

/* #define T_Tree_Collected            T_Full_Binder */
#define T_Short_Tree_Collected         T_Binder
#define tree_is_collected(p)           has_high_type_bit(T_Pair(p), T_Short_Tree_Collected)
#define tree_set_collected(p)          set_high_type_bit(T_Pair(p), T_Short_Tree_Collected)
#define tree_clear_collected(p)        clear_high_type_bit(T_Pair(p), T_Short_Tree_Collected)

#define T_Simple_Values                T_Binder
#define has_simple_values(p)           has_high_type_bit(T_Hsh(p), T_Simple_Values)
#define set_has_simple_values(p)       set_high_type_bit(T_Hsh(p), T_Simple_Values)
#define clear_has_simple_values(p)     clear_high_type_bit(T_Hsh(p), T_Simple_Values)

#define T_Very_Safe_Closure            (1LL << (48 + 4))
#define T_Short_Very_Safe_Closure      (1 << 4)
#define is_very_safe_closure(p)        has_high_type_bit(T_Clo(p), T_Short_Very_Safe_Closure)
#define set_very_safe_closure(p)       set_high_type_bit(T_Clo(p), T_Short_Very_Safe_Closure)
#define closure_bits(p)                (full_type(T_Pair(p)) & (T_Safe_Closure | T_Very_Safe_Closure | T_Full_No_Defaults))
#define is_very_safe_closure_body(p)   has_high_type_bit(T_Pair(p), T_Short_Very_Safe_Closure)
#define set_very_safe_closure_body(p)  set_high_type_bit(T_Pair(p), T_Short_Very_Safe_Closure)

#define T_Baffle_Let                   T_Short_Very_Safe_Closure
#define is_baffle_let(p)               has_high_type_bit(T_Let(p), T_Baffle_Let)
#define set_baffle_let(p)              set_high_type_bit(T_Let(p), T_Baffle_Let)

#define T_Cyclic                       (1LL << (48 + 5))
#define T_Short_Cyclic                 (1 << 5)
#define is_cyclic(p)                   has_high_type_bit(T_Seq(p), T_Short_Cyclic)
#define set_cyclic(p)                  set_high_type_bit(T_Seq(p), T_Short_Cyclic)

#define T_Cyclic_Set                   (1LL << (48 + 6))
#define T_Short_Cyclic_Set             (1 << 6)
#define is_cyclic_set(p)               has_high_type_bit(T_Seq(p), T_Short_Cyclic_Set)
#define set_cyclic_set(p)              set_high_type_bit(T_Seq(p), T_Short_Cyclic_Set)
#define clear_cyclic_bits(p)           clear_type_bit(p, T_Collected | T_Shared | T_Cyclic | T_Cyclic_Set) /* not T_Seq, p can be free(!) */

#define T_Keyword                      (1LL << (48 + 7))
#define T_Short_Keyword                (1 << 7)
#define is_keyword(p)                  has_high_type_bit(T_Sym(p), T_Short_Keyword)
#define is_symbol_and_keyword(p)       ((is_symbol(p)) && (is_keyword(p)))
/* this bit distinguishes a symbol from a symbol that is also a keyword */

#define T_Fx_Treeable                  T_Short_Keyword
#define is_fx_treeable(p)              has_high_type_bit(T_Pair(p), T_Fx_Treeable)
#define set_is_fx_treeable(p)          set_high_type_bit(T_Pair(p), T_Fx_Treeable)

#define T_Full_Simple_Elements         (1LL << (48 + 8))
#define T_Simple_Elements              (1 << 8)
#define has_simple_elements(p)         has_high_type_bit(T_Nvc(p), T_Simple_Elements)
#define set_has_simple_elements(p)     set_high_type_bit(T_Nvc(p), T_Simple_Elements)
#define clear_has_simple_elements(p)   clear_high_type_bit(T_Nvc(p), T_Simple_Elements)
#define c_function_has_simple_elements(p)     has_high_type_bit(T_CFn(p), T_Simple_Elements)
#define c_function_set_has_simple_elements(p) set_high_type_bit(T_CFn(p), T_Simple_Elements)
/* c_func case here refers to boolean? et al -- structure element type declaration that ensures a simple object */

#define T_Simple_Keys                  T_Simple_Elements
#define has_simple_keys(p)             has_high_type_bit(T_Hsh(p), T_Simple_Keys)
#define set_has_simple_keys(p)         set_high_type_bit(T_Hsh(p), T_Simple_Keys)
#define clear_has_simple_keys(p)       clear_high_type_bit(T_Hsh(p), T_Simple_Keys)

#define T_Safe_Setter                  T_Simple_Elements
#define is_safe_setter(p)              has_high_type_bit(T_Sym(p), T_Safe_Setter)
#define set_is_safe_setter(p)          set_high_type_bit(T_Sym(p), T_Safe_Setter)

#define T_Float_Optable                T_Simple_Elements
#define is_float_optable(p)            has_high_type_bit(T_Pair(p), T_Float_Optable)
#define set_is_float_optable(p)        set_high_type_bit(T_Pair(p), T_Float_Optable)

#define T_Full_Case_Key                (1LL << (48 + 9))
#define T_Case_Key                     (1 << 9)
#define is_case_key(p)                 has_high_type_bit(T_Ext(p), T_Case_Key)
#define set_case_key(p)                set_high_type_bit(T_Sym(p), T_Case_Key)

#define T_Opt1_Func_Listed             T_Case_Key
#define opt1_func_listed(p)            has_high_type_bit(T_Pair(p), T_Opt1_Func_Listed)
#define set_opt1_func_listed(p)        set_high_type_bit(T_Pair(p), T_Opt1_Func_Listed)

#define T_Full_True_Is_Done            (1LL << (48 + 10))
#define T_True_Is_Done                 (1 << 10)
#define true_is_done(p)                has_high_type_bit(T_Pair(p), T_True_Is_Done)
#define set_true_is_done(p)            set_high_type_bit(T_Pair(p), T_True_Is_Done)
#define set_a_is_cadr(p)               set_true_is_done(p)
#define a_is_cadr(p)                   true_is_done(p)

/* #define T_Full_Is_Saver             T_Full_True_Is_Done */
#define T_Is_Saver                     T_True_Is_Done
#define is_saver(p)                    has_high_type_bit(p, T_Is_Saver)
#define set_is_saver(p)                do {set_high_type_bit(T_Sym(p), T_Is_Saver); set_high_type_bit(T_Fnc(global_value(p)), T_Is_Saver);} while (0)

#define T_Full_Unknopt                 (1LL << (48 + 11))
#define T_Unknopt                      (1 << 11)
#define is_unknopt(p)                  has_high_type_bit(T_Pair(p), T_Unknopt)
#define set_is_unknopt(p)              set_high_type_bit(T_Pair(p), T_Unknopt)

/* #define T_Full_Is_Translucent       T_Full_Unknopt */
#define T_Is_Translucent               T_Unknopt
#define is_translucent(p)              (((is_symbol(p)) || (is_c_function(p))) && (has_high_type_bit(p, T_Is_Translucent)))
#define set_is_translucent(p)          do {set_high_type_bit(T_Sym(p), T_Is_Translucent); set_high_type_bit(T_Fnc(global_value(p)), T_Is_Translucent);} while (0)

#define T_Mac_Ok                       T_Unknopt
#define mac_is_ok(p)                   has_high_type_bit(T_Pair(p), T_Mac_Ok)
#define set_mac_is_ok(p)               set_high_type_bit(T_Pair(p), T_Mac_Ok)
/* marks a macro (via (macro...)) that has been checked -- easier (and slower) than making 4 or 5 more ops, op_macro_unchecked and so on */

#define T_Full_Safety_Checked          (1LL << (48 + 12))
#define T_Safety_Checked               (1 << 12)
#define is_safety_checked(p)           has_high_type_bit(T_Pair(p), T_Safety_Checked)
#define set_safety_checked(p)          do {if (in_heap(p)) set_high_type_bit(T_Pair(p), T_Safety_Checked);} while (0)

#define T_Setter                       T_Safety_Checked
#define set_is_setter(p)               do {set_high_type_bit(T_Sym(p), T_Setter); set_high_type_bit(global_value(p), T_Setter);} while (0)
#define is_setter(p)                   ((has_high_type_bit(p, T_Setter)) && (!is_pair(p)))
/* optimizer flag for a procedure that sets some variable (set-car! for example) */

#define T_Full_Has_Fn                  (1LL << (48 + 13))
#define T_Has_Fn                       (1 << 13)
#define set_has_fn(p)                  set_high_type_bit(T_Pair(p), T_Has_Fn)
#define has_fn(p)                      has_high_type_bit(T_Pair(p), T_Has_Fn)
#define clear_has_fn(p)                clear_high_type_bit(T_Pair(p), T_Has_Fn)

/* #define T_Full_Is_Escaper           T_Full_Has_Fn */
#define T_Is_Escaper                   T_Has_Fn
#define is_escaper(p)                  ((has_high_type_bit(p, T_Is_Escaper)) && (!is_pair(p)))
#define set_is_escaper_syntax(p)       do {set_high_type_bit(T_Sym(p), T_Is_Escaper); set_high_type_bit(T_Syn(global_value(p)), T_Is_Escaper);} while (0)
#define set_is_escaper_function(p)     do {set_high_type_bit(T_Sym(p), T_Is_Escaper); set_high_type_bit(T_Fnc(global_value(p)), T_Is_Escaper);} while (0)

#define T_Unheap                       0x4000000000000000
#define T_Short_Unheap                 (1 << 14)
#define in_heap(p)                     (((T_Pos(p))->tf.bits.high_bits & T_Short_Unheap) == 0) /* can be slot, make_starlet let_set_slot */
#define unheap(p)                      set_high_type_bit(T_Ext(p), T_Short_Unheap)

#define T_Gc_Mark                      0x8000000000000000
#define is_marked(p)                   has_type_bit(p, T_Gc_Mark)
#define set_mark(p)                    set_type_bit(T_Pos(p), T_Gc_Mark)
#define clear_mark(p)                  clear_type_bit(p, T_Gc_Mark)
/* using the sign bit, bit 23 (or 55) == 31 (or 63) for this makes a big difference in the GC */

#define is_eof(p)                      ((T_Ext(p)) == eof_object)
#define is_true(Sc, p)                 ((T_Ext(p)) != Sc->F)
#define is_false(Sc, p)                ((T_Ext(p)) == Sc->F)

#if MS_Windows
  static s7_pointer make_boolean(s7_scheme *sc, bool val) {if (val) return(sc->T); return(sc->F);}
#else
  #define make_boolean(Sc, Val)        ((Val) ? Sc->T : Sc->F)
#endif

#define is_pair(p)                     (type(p) == T_PAIR)
#define is_mutable_pair(p)             ((is_pair(p)) && (!is_immutable(p)))
   /* same speed: ((full_type(p) & (Type_Mask | T_Immutable)) == T_PAIR), list-set! set-car! set-cdr! */
#define is_null(p)                     ((T_Exs(p)) == sc->nil)
#define is_not_null(p)                 ((T_Exs(p)) != sc->nil)
#define is_list(p)                     ((is_pair(p)) || (type(p) == T_NIL))
#define is_quote(Sc, p)                (((p) == Sc->quote_symbol) || ((p) == Sc->quote_function)) /* order here apparently does not matter */
#define is_safe_quote(Sc, p)           ((((p) == Sc->quote_symbol) && (is_global(Sc->quote_symbol))) || ((p) == Sc->quote_function))
#define is_quoted_pair(Sc, p)          ((is_pair(p)) && (is_quote(Sc, car(p))))
#define is_safe_quoted_pair(Sc, p)     ((is_pair(p)) && (is_safe_quote(Sc, car(p))))
#define is_unquoted_pair(Sc, p)        ((is_pair(p)) && (!is_quote(Sc, car(p))))
#define is_quoted_symbol(Sc, p)        ((is_quoted_pair(Sc, p)) && (is_pair(cdr(p))) && (is_symbol(cadr(p))))

/* pair line/file/position */
#define Pair_Line_Bits                 24
#define Pair_File_Bits                 12
#define Pair_Position_Bits             28
#define Pair_Line_Offset               0
#define Pair_File_Offset               Pair_Line_Bits
#define Pair_Position_Offset           (Pair_Line_Bits + Pair_File_Bits)
#define Pair_Line_Mask                 ((1 << Pair_Line_Bits) - 1)
#define Pair_File_Mask                 ((1 << Pair_File_Bits) - 1)
#define Pair_Position_Mask             ((1 << Pair_Position_Bits) - 1)

#define port_location(Port)            (((port_line_number(Port) & Pair_Line_Mask) << Pair_Line_Offset) | \
                                        ((port_file_number(Port) & Pair_File_Mask) << Pair_File_Offset) | \
                                        ((port_position(Port) & Pair_Position_Mask) << Pair_Position_Offset))

#define location_to_line(Loc)          ((Loc >> Pair_Line_Offset) & Pair_Line_Mask)
#define location_to_file(Loc)          ((Loc >> Pair_File_Offset) & Pair_File_Mask)
#define location_to_position(Loc)      ((Loc >> Pair_Position_Offset) & Pair_Position_Mask)

#define pair_line_number(p)            location_to_line(pair_location(p))
#define pair_file_number(p)            location_to_file(pair_location(p))
#define pair_position(p)               location_to_position(pair_location(p))

#if !s7_Debugging
#define pair_location(p)               (p)->object.sym_cons.location
#define pair_set_location(p, X)        (p)->object.sym_cons.location = X
#define pair_raw_hash(p)               (p)->object.sym_cons.hash
#define pair_set_raw_hash(p, X)        (p)->object.sym_cons.hash = X
#define pair_raw_len(p)                (p)->object.sym_cons.location
#define pair_set_raw_len(p, X)         (p)->object.sym_cons.location = X
#define pair_raw_name(p)               (p)->object.sym_cons.fstr
#define pair_set_raw_name(p, X)        (p)->object.sym_cons.fstr = X
/* opt1 == raw_hash, opt2 == raw_name, opt3 == line|ctr + len, but hash/name/len only apply to the symbol table so there's no collision */

#define opt1(p, r)                     ((p)->object.cons.opt1)
#define set_opt1(p, x, r)              (p)->object.cons.opt1 = x
#define opt2(p, r)                     ((p)->object.cons.o2.opt2)
#define set_opt2(p, x, r)              (p)->object.cons.o2.opt2 = (s7_pointer)(x)
#define opt2_n(p, r)                   ((p)->object.cons.o2.n)
#define set_opt2_n(p, x, r)            (p)->object.cons.o2.n = x
#define opt3(p, r)                     ((p)->object.cons.o3.opt3)
#define set_opt3(p, x, r)              do {(p)->object.cons.o3.opt3 = x; clear_type_bit(p, T_Location);} while (0)
#define opt3_n(p, r)                   ((p)->object.cons.o3.n)
#define set_opt3_n(p, x, r)            do {(p)->object.cons.o3.n = x; clear_type_bit(p, T_Location);} while (0)

#else

/* the 3 opt fields hold most of the varigated optimizer info, so they are used in many conflicting ways.
 * the bits and funcs here try to track each such use, and report any cross-talk or collisions.
 * all of this machinery vanishes if debugging is turned off.
 */
#define Opt1_Set                       (1 << 0)
#define Opt2_Set                       (1 << 1)
#define Opt3_Set                       (1 << 2)

#define Opt1_Fast                      (1 << 3)   /* fast list in member/assoc circular list check */
#define Opt1_Cfunc                     (1 << 4)   /* c-function */
#define Opt1_Clause                    (1 << 5)   /* case clause */
#define Opt1_Lambda                    (1 << 6)   /* lambda(*) */
#define Opt1_Sym                       (1 << 7)   /* symbol */
#define Opt1_Pair                      (1 << 8)   /* pair */
#define Opt1_Con                       (1 << 9)   /* constant from eval's point of view */ /* 10 was opt1_goto, unused */
#define Opt1_Any                       (1 << 11)  /* anything -- deliberate unchecked case */
#define Opt1_Hash                      (1 << 12)  /* hash code used in the symbol table (pair_raw_hash) */
#define Opt1_Mask                      (Opt1_Fast | Opt1_Cfunc | Opt1_Clause | Opt1_Lambda | Opt1_Sym | Opt1_Pair | Opt1_Con | Opt1_Any | Opt1_Hash)

#define opt1_is_set(p)                 (((T_Pair(p))->debugger_bits & Opt1_Set) != 0)
#define set_opt1_is_set(p)             (T_Pair(p))->debugger_bits |= Opt1_Set
#define opt1_role_matches(p, Role)     (((T_Pair(p))->debugger_bits & Opt1_Mask) == Role)
#define set_opt1_role(p, Role)         (T_Pair(p))->debugger_bits = (Role | ((p)->debugger_bits & ~Opt1_Mask))
#define opt1(p, Role)                  opt1_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt1(p, x, Role)           set_opt1_1(T_Pair(p), x, Role, __func__, __LINE__)

#define Opt2_Key                       (1 << 13)  /* case key */
#define Opt2_Slow                      (1 << 14)  /* slow list in member/assoc circular list check */
#define Opt2_Sym                       (1 << 15)  /* symbol */
#define Opt2_Pair                      (1 << 16)  /* pair */
#define Opt2_Con                       (1 << 17)  /* constant as above */
#define Opt2_Fx                        (1 << 18)  /* fx (fx_*) func (sc, form) */
#define Opt2_Fn                        (1 << 19)  /* fn (s7_function) func (sc, arglist) */
#define Opt2_Lambda                    (1 << 20)  /* lambda form */
#define Opt2_Name                      (1 << 21)  /* named used by symbol table (pair_raw_name) */
#define Opt2_Direct                    (1LL << 32)
#define Opt2_Int                       (1LL << 33)
#define Opt2_Mask                      (Opt2_Key | Opt2_Slow | Opt2_Sym | Opt2_Pair | Opt2_Con | Opt2_Fx | \
                                        Opt2_Fn | Opt2_Lambda | Opt2_Direct | Opt2_Name | Opt2_Int)

#define opt2_is_set(p)                 (((T_Pair(p))->debugger_bits & Opt2_Set) != 0)
#define set_opt2_is_set(p)             (T_Pair(p))->debugger_bits |= Opt2_Set
#define opt2_role_matches(p, Role)     (((T_Pair(p))->debugger_bits & Opt2_Mask) == Role)
#define set_opt2_role(p, Role)         (T_Pair(p))->debugger_bits = (Role | ((p)->debugger_bits & ~Opt2_Mask))
#define opt2(p, Role)                  opt2_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt2(p, x, Role)           set_opt2_1(sc, T_Pair(p), (s7_pointer)(x), Role, __func__, __LINE__)
#define opt2_n(p, Role)                opt2_n_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt2_n(p, x, Role)         set_opt2_n_1(sc, T_Pair(p), x, Role, __func__, __LINE__)

#define Opt3_Arglen                    (1 << 22)  /* arglist length */
#define Opt3_Sym                       (1 << 23)  /* expression symbol access */
#define Opt3_And                       (1 << 24)  /* and second clause */
#define Opt3_Direct                    (1 << 25)  /* direct call info */
#define Opt3_Any                       (1 << 26)
#define Opt3_Let                       (1 << 27)  /* let or #f */
#define Opt3_Con                       (1 << 28)
#define Opt3_Location                  (1 << 29)
#define Opt3_Len                       (1 << 30)
#define Opt3_Byte                      (1LL << 31)
#define Opt3_Int                       (1LL << 34)
#define Opt3_Mask                      (Opt3_Arglen | Opt3_Sym | Opt3_And | Opt3_Any | Opt3_Let | Opt3_Byte | \
                                        Opt3_Location | Opt3_Len | Opt3_Direct | Opt3_Con | Opt3_Int)

#define opt3_is_set(p)                 (((T_Pair(p))->debugger_bits & Opt3_Set) != 0)
#define set_opt3_is_set(p)             (T_Pair(p))->debugger_bits |= Opt3_Set
#define opt3_role_matches(p, Role)     (((T_Pair(p))->debugger_bits & Opt3_Mask) == Role)
#define set_opt3_role(p, Role)         (T_Pair(p))->debugger_bits = (Role | ((p)->debugger_bits & ~Opt3_Mask))
#define opt3(p, Role)                  opt3_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt3(p, x, Role)           set_opt3_1(T_Pair(p), x, Role)
#define opt3_n(p, Role)                opt3_n_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt3_n(p, x, Role)         set_opt3_n_1(T_Pair(p), x, Role)

#define pair_location(p)               opt3_location_1(T_Pair(p), __func__, __LINE__)
#define pair_set_location(p, X)        set_opt3_location_1(T_Pair(p), X)
#define pair_raw_hash(p)               opt1_hash_1(T_Pair(p), __func__, __LINE__)
#define pair_set_raw_hash(p, X)        set_opt1_hash_1(T_Pair(p), X)
#define pair_raw_len(p)                opt3_len_1(T_Pair(p), __func__, __LINE__)
#define pair_set_raw_len(p, X)         set_opt3_len_1(T_Pair(p), X)
#define pair_raw_name(p)               opt2_name_1(T_Pair(p), __func__, __LINE__)
#define pair_set_raw_name(p, X)        set_opt2_name_1(T_Pair(p), X)

#define L_Hit                          (1LL << 40) /* "L_SET" is taken */
#define L_Func                         (1LL << 41)
#define L_Dox                          (1LL << 42)
#define L_Mask                         (L_Func | L_Dox)
#endif

#define opt1_fast(P)                   T_Lst(opt1(P,                Opt1_Fast))
#define set_opt1_fast(P, X)            set_opt1(P, T_Pair(X),       Opt1_Fast)
#define opt1_cfunc(P)                  T_Exs(opt1(P,                Opt1_Cfunc))
#define set_opt1_cfunc(P, X)           set_opt1(P, T_CFn(X),        Opt1_Cfunc)
#define opt1_lambda_unchecked(P)       opt1(P,                      Opt1_Lambda) /* can be free/null? from s7_call? */
#define opt1_lambda(P)                 T_Clo(opt1(P,                Opt1_Lambda))
#define set_opt1_lambda(P, X)          set_opt1(P, T_Clo(X),        Opt1_Lambda)
#define set_opt1_lambda_add(P, X)      do {set_opt1(P, T_Clo(X),    Opt1_Lambda); add_opt1_func(sc, P);} while (0)
#define opt1_clause(P)                 T_Exs(opt1(P,                Opt1_Clause))
#define set_opt1_clause(P, X)          set_opt1(P, T_Exs(X),        Opt1_Clause)
#define opt1_sym(P)                    T_Sym(opt1(P,                Opt1_Sym))
#define set_opt1_sym(P, X)             set_opt1(P, T_Sym(X),        Opt1_Sym)
#define opt1_pair(P)                   T_Lst(opt1(P,                Opt1_Pair))
#define set_opt1_pair(P, X)            set_opt1(P, T_Lst(X),        Opt1_Pair)
#define opt1_con(P)                    T_Exs(opt1(P,                Opt1_Con))
#define set_opt1_con(P, X)             set_opt1(P, T_Exs(X),        Opt1_Con)    /* can be #<unused> */
#define opt1_any(P)                    opt1(P,                      Opt1_Any)    /* can be free in closure_is_ok */
#define set_opt1_any(P, X)             set_opt1(P, X,               Opt1_Any)

#define opt2_any(P)                    opt2(P,                      Opt2_Key)
#define set_opt2_any(P, X)             set_opt2(P, X,               Opt2_Key)
#define opt2_int(P)                    opt2_n(P,                    Opt2_Int)
#define set_opt2_int(P, X)             set_opt2_n(P, X,             Opt2_Int)
#define opt2_slow(P)                   T_Lst(opt2(P,                Opt2_Slow))
#define set_opt2_slow(P, X)            set_opt2(P, T_Pair(X),       Opt2_Slow)
#define opt2_sym(P)                    T_Sym(opt2(P,                Opt2_Sym))
#define set_opt2_sym(P, X)             set_opt2(P, T_Sym(X),        Opt2_Sym)
#define opt2_pair(P)                   T_Lst(opt2(P,                Opt2_Pair))
#define set_opt2_pair(P, X)            set_opt2(P, T_Lst(X),        Opt2_Pair)
#define opt2_con(P)                    T_Exs(opt2(P,                Opt2_Con))
#define set_opt2_con(P, X)             set_opt2(P, T_Exs(X),        Opt2_Con)
#define opt2_lambda(P)                 T_Pair(opt2(P,               Opt2_Lambda))
#define set_opt2_lambda(P, X)          set_opt2(P, T_Pair(X),       Opt2_Lambda)
#define opt2_direct(P)                 opt2(P,                      Opt2_Direct)
#define set_opt2_direct(P, X)          set_opt2(P, (s7_pointer)(X), Opt2_Direct)

#define opt3_arglen(P)                 opt3_n(P,                    Opt3_Arglen)
#define set_opt3_arglen(P, X)          set_opt3_n(P, X,             Opt3_Arglen)
#define opt3_int(P)                    opt3_n(P,                    Opt3_Int)
#define set_opt3_int(P, X)             set_opt3_n(P, X,             Opt3_Int)
#define opt3_sym(P)                    T_Sym(opt3(P,                Opt3_Sym))
#define set_opt3_sym(P, X)             set_opt3(P, T_Sym(X),        Opt3_Sym)
#define opt3_con(P)                    T_Exs(opt3(P,                Opt3_Con))
#define set_opt3_con(P, X)             set_opt3(P, T_Exs(X),        Opt3_Con)
#define opt3_pair(P)                   T_Pair(opt3(P,               Opt3_And))
#define set_opt3_pair(P, X)            set_opt3(P, T_Pair(X),       Opt3_And)
#define opt3_any(P)                    opt3(P,                      Opt3_Any)
#define set_opt3_any(P, X)             set_opt3(P, X,               Opt3_Any)   /* can be -1 (arity), check_lambda[78617] */
#define opt3_let(P)                    T_Let(opt3(P,                Opt3_Let))
#define set_opt3_let(P, X)             set_opt3(P, T_Let(X),        Opt3_Let)
#define opt3_direct(P)                 opt3(P,                      Opt3_Direct)
#define set_opt3_direct(P, X)          set_opt3(P, (s7_pointer)(X), Opt3_Direct)

#if s7_Debugging
#define opt3_byte(p)                   opt3_byte_1(T_Pair(p), Opt3_Byte, __func__, __LINE__)
#define set_opt3_byte(p, x)            set_opt3_byte_1(T_Pair(p), x, Opt3_Byte, __func__, __LINE__)
#else
#define opt3_byte(P)                   T_Pair(P)->object.cons.o3.opt_type /* op_if_is_type, opt_type == opt3 in cons */
#define set_opt3_byte(P, X)            do {T_Pair(P)->object.cons.o3.opt_type = X; clear_type_bit(P, T_Location);} while (0)
#endif

#define pair_macro(P)                  opt2_sym(P)
#define set_pair_macro(P, Name)        set_opt2_sym(P, Name)

#define fn_proc(f)                     ((s7_function)(opt2(f, Opt2_Fn)))
#define fx_proc(f)                     ((s7_function)(opt2(f, Opt2_Fx)))
#define fn_proc_unchecked(f)           ((s7_function)(T_Pair(f)->object.cons.o2.opt2))

#define set_fx(f, _X_)                 do {s7_pointer X; X = (s7_pointer)(_X_); set_opt2(f, X, Opt2_Fx); if (X) set_has_fx(f); else clear_has_fx(f);} while (0)
#define set_fx_direct(f, X)            do {clear_has_fn(f); set_opt2(f, (s7_pointer)(X), Opt2_Fx); set_has_fx(f);} while (0)
#define set_fn(f, _X_)                 do {s7_pointer X; X = (s7_pointer)(_X_); set_opt2(f, X, Opt2_Fn); if (X) set_has_fn(f); else clear_has_fn(f);} while (0)
#define set_fn_direct(f, X)            do {set_opt2(f, (s7_pointer)(X), Opt2_Fn); set_has_fn(f);} while (0)
#define set_class_and_fn_proc(X, f)    do {set_opt1_cfunc(X, f); set_fn_direct(X, c_function_call(f));} while (0)

#if WITH_GCC
#if s7_Debugging
  /* make sure sc->code and sc->args are not changed by fx_call or fn_call (sc->value is changed by fx_tc_* via tc_z) */
  #define fx_call(Sc, F)               ({s7_pointer _P_, _C_ = Sc->code, _V_, _A_ = Sc->args; _P_ = F; _V_ = fx_proc(_P_)(Sc, car(_P_)); if (Sc->code != _C_) fprintf(stderr, "%s%s[%d]: %s clobbered sc->code%s\n", Bold_Text, __func__, __LINE__, display(_C_), Unbold_Text); if (Sc->args != _A_) fprintf(stderr, "%s%s[%d]: %s clobbered sc->args%s\n", Bold_Text, __func__, __LINE__, display(_A_), Unbold_Text); _V_;})
  #define fn_call(Sc, F)               ({s7_pointer _P_, _C_ = Sc->code, _V_, _A_ = Sc->args; _P_ = F; _V_ = fn_proc(_P_)(Sc, cdr(_P_)); if (Sc->code != _C_) fprintf(stderr, "%s%s[%d]: %s clobbered sc->code%s\n", Bold_Text, __func__, __LINE__, display(_C_), Unbold_Text); if (Sc->args != _A_) fprintf(stderr, "%s%s[%d]: %s clobbered sc->args%s\n", Bold_Text, __func__, __LINE__, display(_A_), Unbold_Text); _V_;})
#else
  #define fx_call(Sc, F)               ({s7_pointer _P_; _P_ = F; fx_proc(_P_)(Sc, car(_P_));})
  #define fn_call(Sc, F)               ({s7_pointer _P_; _P_ = F; fn_proc(_P_)(Sc, cdr(_P_));})
#endif
#else
#define fx_call(Sc, F)                 fx_proc(F)(Sc, car(F))
#define fn_call(Sc, F)                 fn_proc(F)(Sc, cdr(F))
#endif
/* fx_call can affect the stack and sc->value */

#define car(p)                         (T_Pair(p))->object.cons.car
#define car_unchecked(p)               (T_Pos(p))->object.cons.car
#define set_car(p, Val)                car(p) = T_Pos(Val)            /* Val can be a slot or #<unused> or #<catch> etc */

#define cdr(p)                         (T_Pair(p))->object.cons.cdr
#define set_cdr_unchecked(p, Val)      cdr(p) = T_Exs(Val)            /* #<unused> in g_gc */
#define cdr_unchecked(p)               (T_Exs(p))->object.cons.cdr
#if s7_Debugging
  static void checked_set_cdr(s7_pointer p, s7_pointer Val, const char *func, int32_t line);
  #define set_cdr(p, Val)              checked_set_cdr(p, Val, __func__, __LINE__)
#else
  #define set_cdr(p, Val)              cdr(p) = T_Ext(Val)
#endif

#define caar(p)                        car(car(p))
#define cadr(p)                        car(cdr(p))
#define set_cadr(p, Val)               car(cdr(p)) = T_Exs(Val) /* #<unused> in g_gc[8178], set_elist_2, we're clearing the elists */
#define cdar(p)                        cdr(car(p))
#define set_cdar(p, Val)               cdr(car(p)) = T_Ext(Val)
#define cddr(p)                        cdr(cdr(p))

#define caaar(p)                       car(car(car(p)))
#define cadar(p)                       car(cdr(car(p)))
#define cdadr(p)                       cdr(car(cdr(p)))
#define caddr(p)                       car(cdr(cdr(p)))
#define set_caddr(p, Val)              car(cdr(cdr(p))) = T_Ext(Val)
#define caadr(p)                       car(car(cdr(p)))
#define cdaar(p)                       cdr(car(car(p)))
#define cdddr(p)                       cdr(cdr(cdr(p)))
#define set_cdddr(p, Val)              cdr(cdr(cdr(p))) = T_Ext(Val)
#define cddar(p)                       cdr(cdr(car(p)))

#define caaadr(p)                      car(car(car(cdr(p))))
#define caadar(p)                      car(car(cdr(car(p))))
#define cadaar(p)                      car(cdr(car(car(p))))
#define cadddr(p)                      car(cdr(cdr(cdr(p))))
#define caaddr(p)                      car(car(cdr(cdr(p))))
#define cddddr(p)                      cdr(cdr(cdr(cdr(p))))
#define caddar(p)                      car(cdr(cdr(car(p))))
#define cdadar(p)                      cdr(car(cdr(car(p))))
#define cdaddr(p)                      cdr(car(cdr(cdr(p))))
#define caaaar(p)                      car(car(car(car(p))))
#define cadadr(p)                      car(cdr(car(cdr(p))))
#define cdaadr(p)                      cdr(car(car(cdr(p))))
#define cdaaar(p)                      cdr(car(car(car(p))))
#define cdddar(p)                      cdr(cdr(cdr(car(p))))
#define cddadr(p)                      cdr(cdr(car(cdr(p))))
#define cddaar(p)                      cdr(cdr(car(car(p))))

#define cadaddr(p)                     cadr(caddr(p))
#define caddadr(p)                     caddr(cadr(p))
#define caddaddr(p)                    caddr(caddr(p))

#if WITH_GCC
  /* slightly tricky because cons can be called recursively, macro here is faster than inline function */
  #define cons(Sc, A, B)   ({s7_pointer _X_, _A_, _B_; _A_ = A; _B_ = B; new_cell(Sc, _X_, T_PAIR | T_Safe_Procedure); set_car(_X_, _A_); set_cdr(_X_, _B_); _X_;})
#else
  #define cons(Sc, A, B)               s7_cons(Sc, A, B)
#endif

#define list_1(Sc, A)                  cons(Sc, A, Sc->nil)
#define list_1_unchecked(Sc, A)        cons_unchecked(Sc, A, Sc->nil)
#define list_2(Sc, A, B)               cons_unchecked(Sc, A, cons(Sc, B, Sc->nil))
#define list_2_unchecked(Sc, A, B)     cons_unchecked(Sc, A, cons_unchecked(Sc, B, Sc->nil))
#define list_3(Sc, A, B, C)            cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons(Sc, C, Sc->nil)))
#define list_4(Sc, A, B, C, D)         cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons_unchecked(Sc, C, cons(Sc, D, Sc->nil))))
#define with_list_t1(Sc, A)            (set_car(Sc->t1_1, A), Sc->t1_1) /* this is slower than explicit code, esp t3, procedures are same as this */
#define with_list_t2(Sc, A, B)         (set_car(Sc->t2_1, A), set_car(Sc->t2_2, B), Sc->t2_1)
#define with_list_t3(Sc, A, B, C)      (set_car(Sc->t3_1, A), set_car(Sc->t3_2, B), set_car(Sc->t3_3, C), Sc->t3_1)
/* #define with_list_t4(Sc, A, B, C, D) (set_car(Sc->t4_1, A), set_car(Sc->t3_1, B), set_car(Sc->t3_2, C), set_car(Sc->t3_3, D), Sc->t4_1) */

#define is_string(p)                   (type(p) == T_STRING)
#define is_mutable_string(p)           ((full_type(T_Ext(p)) & (Type_Mask | T_Immutable)) == T_STRING)
#define string_value(p)                (T_Str(p))->object.string.svalue
#define string_length(p)               (T_Str(p))->object.string.length
#define string_hash(p)                 (T_Str(p))->object.string.hash
#define string_block(p)                (T_Str(p))->object.string.block
#define string_block_unchecked(p)      p->object.string.block

#define character(p)                   (T_Chr(p))->object.chr.c
#define is_character(p)                (type(p) == T_CHARACTER)
#define upper_character(p)             (T_Chr(p))->object.chr.up_c
#define is_char_alphabetic(p)          (T_Chr(p))->object.chr.alpha_c
#define is_char_numeric(p)             (T_Chr(p))->object.chr.digit_c
#define is_char_whitespace(p)          (T_Chr(p))->object.chr.space_c
#define is_char_uppercase(p)           (T_Chr(p))->object.chr.upper_c
#define is_char_lowercase(p)           (T_Chr(p))->object.chr.lower_c
#define character_name(p)              (T_Chr(p))->object.chr.c_name
#define character_name_length(p)       (T_Chr(p))->object.chr.length

#define optimize_op(P)                 (T_Ext(P))->tf.bits.opt_bits
#define optimize_op_unchecked(P)       (P)->tf.bits.opt_bits
#define set_optimize_op(P, Op)         (T_Ext(P))->tf.bits.opt_bits = (Op) /* not T_Pair */
#define Op_Hop_Mask                    0xfffe
#define optimize_op_match(P, Q)        ((is_optimized(P)) && ((optimize_op(P) & Op_Hop_Mask) == (Q)))
#define op_no_hop(P)                   (optimize_op(P) & Op_Hop_Mask)
#define op_has_hop(P)                  ((optimize_op(P) & 1) != 0)
#define clear_optimize_op(P)           set_optimize_op(P, OP_UNOPT)
#define set_safe_optimize_op(P, Q)     do {set_optimized(P); set_optimize_op(P, Q);} while (0)
#define set_unsafe_optimize_op(P, Q)   do {set_unsafely_optimized(P); set_optimize_op(P, Q);} while (0)

#if s7_Debugging
#define s7_t_slot(Sc, Slot) s7_t_slot_1(Sc, Slot, __func__, __LINE__)
const char *display(s7_pointer obj);
static s7_pointer s7_t_slot_1(s7_scheme *sc, s7_pointer symbol, const char *func, int line)
{
  s7_pointer slot = s7_slot(sc, symbol);
  if (type(slot) == T_SLOT) return(slot);
  fprintf(stderr, "%s[%d] (via s7_t_slot): symbol: %s, slot: %s\n", func, line, display(symbol), display(slot));
  return(slot);
}
#else
#define s7_t_slot(Sc, Slot) s7_slot(Sc, Slot)
#endif

#define is_symbol(p)                   (type(p) == T_SYMBOL)
#define is_normal_symbol(p)            ((is_symbol(p)) && (!is_keyword(p)))  /* ((full_type(p) & (0xff | T_KEYWORD)) == T_SYMBOL) is exactly the same speed */
#define is_bound_symbol(Sc, p)         (is_slot(s7_slot(Sc, p))) /* (s7_slot(Sc, p) != Sc->undefined) is the same speed apparently */
#define is_safe_symbol(Sc, p)          ((is_symbol(p)) && (is_bound_symbol(Sc, p)))
#define symbol_name_cell(p)            T_Str((T_Sym(p))->object.sym.name)
#define symbol_set_name_cell(p, S)     (T_Sym(p))->object.sym.name = T_Str(S)
#define symbol_name(p)                 string_value(symbol_name_cell(p))
#define symbol_name_length(p)          string_length(symbol_name_cell(p))
#define gensym_block(p)                symbol_name_cell(p)->object.string.gensym_block
#define pointer_map(p)                 (s7_uint)((intptr_t)(p))
#define symbol_id(p)                   (T_Sym(p))->object.sym.id
#define symbol_set_id_unchecked(p, X)  (T_Sym(p))->object.sym.id = X
#if s7_Debugging
static void symbol_set_id(s7_pointer sym, s7_int id)
{
  if (id < symbol_id(sym))
    {fprintf(stderr, "%s[%d]: id mismatch: sym: %s %" ld64 ", let: %" ld64 "\n", __func__, __LINE__, symbol_name(sym), symbol_id(sym), id); abort();}
  (T_Sym(sym))->object.sym.id = id;
}
#else
#define symbol_set_id(p, X)            (T_Sym(p))->object.sym.id = X
#endif
/* we need 64-bits here, since we don't want this thing to wrap around, and lets are created at a great rate
 *    callgrind says this is faster than a uint32_t!
 */
#define symbol_info(p)                 (symbol_name_cell(p))->object.string.block
#define symbol_type(p)                 (block_size(symbol_info(p)) & 0xff)                    /* boolean function bool type, block_size is an s7_int */
#define symbol_set_type(p, Type)       block_size(symbol_info(p)) = ((block_size(symbol_info(p)) & ~0xff) | ((Type) & 0xff))
#define symbol_clear_type(p)           block_size(symbol_info(p)) = T_FREE
#define starlet_symbol_id(p)           ((uint8_t)((block_size(symbol_info(p)) >> 8) & 0xff))  /* *s7* id -- can be gensym, eval unopt section */
#define starlet_symbol_set_id(p, F)    block_size(symbol_info(p)) = ((block_size(symbol_info(p)) & ~0xff00) | (((F) & 0xff) << 8))
#define symbol_category(p)             ((uint8_t)((block_size(symbol_info(p)) >> 16) & 0xff))
#define symbol_set_category(p, Type)   block_size(symbol_info(p)) = ((block_size(symbol_info(p)) & ~0xff0000) | (((Type) & 0xff) << 16))

#define REPORT_ROOTLET_REDEF 0
#if REPORT_ROOTLET_REDEF
  /* to find who is stomping on our symbols: */
  static void set_local_1(s7_scheme *sc, s7_pointer symbol, const char *func, int32_t line);
  #define set_local(Symbol)            set_local_1(sc, T_Sym(Symbol), __func__, __LINE__)
#else
  #define set_local(p)                 full_type(T_Sym(p)) &= ~(T_Dont_Eval_Args | T_Syntactic)
  /* if symbol_increment_ctr, local shadowing value is not found? same if {} */
#endif
#define is_global(p)                   (symbol_id(p) == 0)
#define is_defined_global(p)           ((is_slot(global_slot(p))) && (symbol_id(p) == 0))

#define global_slot(p)                 T_Sld((T_Sym(p))->object.sym.global_slot)
#define set_global_slot(p, Val)        (T_Sym(p))->object.sym.global_slot = T_Sld(Val) /* #<undefined> from new_symbol and g_gensym */
#define local_slot(p)                  T_Slt((T_Sym(p))->object.sym.local_slot)        /* was T_Sld 1-Aug-25 */
#define set_local_slot(p, Val)         (T_Sym(p))->object.sym.local_slot = T_Slt(Val)

#define initial_value(p)               symbol_info(p)->ex.ex_ptr
#define set_initial_value(p, Val)      initial_value(p) = T_Ext(Val)
#define local_value(p)                 slot_value(local_slot(T_Sym(p)))
#define local_value_unchecked(p)       local_slot(p)->object.slt.val
#define global_value(p)                slot_value(global_slot(T_Sym(p)))
#define set_global_value(p, Val)       slot_set_value(global_slot(T_Sym(p)), Val) /* slot_set_value checks T_Ext */

#define keyword_symbol(p)              symbol_info(T_Key(p))->nx.ksym        /* keyword only, so does not collide with documentation */
#define keyword_symbol_unchecked(p)    symbol_info(p)->nx.ksym
#define keyword_set_symbol(p, Val)     symbol_info(T_Key(p))->nx.ksym = T_Sym(Val)
#define symbol_help(p)                 symbol_info(p)->nx.documentation
#define symbol_set_help(p, Doc)        symbol_info(p)->nx.documentation = Doc
#define big_symbol_tag(p)              symbol_info(p)->dx.tag
#define set_big_symbol_tag(p, Val)     symbol_info(p)->dx.tag = Val

#define small_symbol_tag(p)            (T_Sym(p))->object.sym.small_symbol_tag
#define set_small_symbol_tag(p, Val)   (T_Sym(p))->object.sym.small_symbol_tag = Val
#define symbol_shadows(p)              symbol_info(p)->ln.iter_or_size
/* #define symbol_set_shadows(p, Val)     symbol_info(p)->ln.iter_or_size = Val */ /* checked in new_symbol that it's 0 for a gensym */
/* field incremented in add_symbol_to_big_symbol_set and decremented in cleanup_big_symbol_set */
#define symbol_clear_shadows(p)        symbol_info(p)->ln.iter_or_size = 0

#define symbol_ctr(p)                  (T_Sym(p))->object.sym.ctr            /* needs to be in the symbol object (not symbol_info) for speed */
#define symbol_clear_ctr(p)            (T_Sym(p))->object.sym.ctr = 0        /* used only to set initial ctr value */
#define symbol_increment_ctr(p)        (T_Sym(p))->object.sym.ctr++          /* despite this expense, ctr does save a lot overall */
#define symbol_has_help(p)             (is_documented(symbol_name_cell(p)))
#define symbol_set_has_help(p)         set_documented(symbol_name_cell(p))

#define symbol_set_local_slot_unchecked(Symbol, Id, Slot) \
  do {(Symbol)->object.sym.local_slot = T_Sld(Slot); symbol_set_id_unchecked(Symbol, Id); symbol_increment_ctr(Symbol);} while (0)
#define symbol_set_local_slot_unchecked_and_unincremented(Symbol, Id, Slot) \
  do {(Symbol)->object.sym.local_slot = T_Sld(Slot); symbol_set_id_unchecked(Symbol, Id);} while (0)
#define symbol_set_local_slot(Symbol, Id, Slot) \
  do {set_local_slot(Symbol, Slot); symbol_set_id(Symbol, Id); symbol_increment_ctr(Symbol);} while (0)
#define symbol_set_local_slot_unincremented(Symbol, Id, Slot) \
  do {set_local_slot(Symbol, Slot); symbol_set_id(Symbol, Id);} while (0)
/* set slot before id in case Slot is an expression that tries to find the current Symbol slot (using its old Id obviously) */

/* block_index(symbol_info()) is free. We won't deallocate a symbol unless it is a gensym.  remove_gensym_from_symbol_table could be opt'd by using this as the location.
 * high bits of block_size(symbol_info()) might be free.  Maybe use the gensym block instead?
 */

#define is_slot(p)                     (type(p) == T_SLOT)
#define slot_symbol(p)                 T_Sym((T_Slt(p))->object.slt.sym)
#define slot_set_symbol(p, Sym)        (T_Slt(p))->object.slt.sym = T_Sym(Sym)
#define slot_value(p)                  T_Nmv((T_Slt(p))->object.slt.val)
#if s7_Debugging
/* how to see an unheaped and un-GC-checked slot with a heap value?  Can't do it here because unheap=most rootlet slots */
#define slot_set_value(slot, value) \
  do { \
       if (is_immutable_slot(slot)) fprintf(stderr, "%s[%d]: setting immutable slot %s\n", __func__, __LINE__, symbol_name(slot_symbol(slot))); \
       (T_Slt(slot))->object.slt.val = T_Nmv(value); \
     } while (0)
#else
#define slot_set_value(p, Val)         (T_Slt(p))->object.slt.val = T_Nmv(Val)
#endif
#define slot_set_symbol_and_value(Slot, Symbol, Value) do {slot_set_symbol(Slot, Symbol); slot_set_value(Slot, Value);} while (0)
#define slot_set_value_with_hook(Slot, Value) \
  do {if (hook_has_functions(sc->rootlet_redefinition_hook)) slot_set_value_with_hook_1(sc, T_Slt(Slot), T_Nmv(Value)); else slot_set_value(T_Slt(Slot), T_Nmv(Value));} while (0)
#define next_slot(p)                   T_Sln((T_Slt(p))->object.slt.nxt)
#define slot_set_next(p, Val)          (T_Slt(p))->object.slt.nxt = T_Sln(Val)
#define slot_set_pending_value(p, Val) do {(T_Slt(p))->object.slt.pending_value = T_Nmv(Val); slot_set_has_pending_value(p);} while (0)
#define slot_simply_set_pending_value(p, Val) (T_Slt(p))->object.slt.pending_value = T_Nmv(Val)
#if s7_Debugging
static s7_pointer slot_pending_value(s7_pointer p) \
  {if (slot_has_pending_value(p)) return(p->object.slt.pending_value); fprintf(stderr, "%s[%d]: slot: no pending value\n", __func__, __LINE__); abort(); return(NULL);}
static s7_pointer slot_expression(s7_pointer p)    \
  {if (slot_has_expression(p)) return(p->object.slt.expr); fprintf(stderr, "%s[%d]: slot: no expression\n", __func__, __LINE__); abort(); return(NULL);}
#else
#define slot_pending_value(p)          (T_Slt(p))->object.slt.pending_value
#define slot_expression(p)             (T_Slt(p))->object.slt.expr
#endif
#define slot_pending_value_unchecked(p) (T_Slt(p))->object.slt.pending_value

#define slot_set_expression(p, Val)    do {(T_Slt(p))->object.slt.expr = T_Ext(Val); slot_set_has_expression(p);} while (0)
#define slot_just_set_expression(p, Val) (T_Slt(p))->object.slt.expr = T_Ext(Val)
#define slot_setter(p)                 T_Prc((T_Slt(p)->object.slt.pending_value))
#define slot_set_setter_1(p, Val)      (T_Slt(p))->object.slt.pending_value = T_Prc(Val)

#if s7_Debugging
  #define is_not_slot_end(p)           ((p) && (T_Slt(p)))
#else
  #define is_not_slot_end(p)           (p) /* used for loop through let slots which end in null, not for general slot recognition */
#endif
#define Slot_End                       NULL
#define is_slot_end(p)                 (!(p))
#define curlet_has_slots(Sc)           is_not_slot_end(let_slots(Sc->curlet))

#define is_syntax(p)                   (type(p) == T_SYNTAX)
#define syntax_symbol(p)               T_Sym((T_Syn(p))->object.syn.symbol)
#define syntax_set_symbol(p, Sym)      (T_Syn(p))->object.syn.symbol = T_Sym(Sym)
#define syntax_opcode(p)               (T_Syn(p))->object.syn.op
#define syntax_min_args(p)             (T_Syn(p))->object.syn.min_args
#define syntax_max_args(p)             (T_Syn(p))->object.syn.max_args
#define syntax_documentation(p)        (T_Syn(p))->object.syn.documentation
#define pair_set_syntax_op(p, X)       do {set_optimize_op(p, X); set_syntactic_pair(p);} while (0)
#define symbol_syntax_op_checked(p)    ((is_syntactic_pair(p)) ? optimize_op(p) : symbol_syntax_op(car(p)))
#define symbol_syntax_op(p)            syntax_opcode(global_value(p))
#define is_syntax_or_qq(p)	       ((is_syntax(p)) || ((p) == sc->quasiquote_function)) /* qq is from s7_define_macro -> T_C_MACRO */

#define let_id(p)                      (T_Let(p))->object.let.id
#define is_let(p)                      (type(p) == T_LET)
#define is_let_unchecked(p)            (type_unchecked(p) == T_LET)
#define let_slots(p)                   T_Sln((T_Let(p))->object.let.slots)
#define let_outlet(p)                  T_Out((T_Let(p))->object.let.nxt)
#define let_set_outlet(p, ol)          (T_Let(p))->object.let.nxt = T_Out(ol)
#if s7_Debugging
  #define let_set_id(p, Id)            do {(T_Let(p))->object.let.id = Id; if ((p == sc->rootlet) && (Id != -1)) {fprintf(stderr, "%s[%d]: rootlet id: %" ld64 "\n", __func__, __LINE__, (s7_int)Id); abort();}} while (0)
  #define let_set_slots(p, Slot)       check_let_set_slots(sc, p, Slot, __func__, __LINE__)
  #define C_Let(p, role)               check_let_ref(p, role, __func__, __LINE__)
  #define S_Let(p, role)               check_let_set(p, role, __func__, __LINE__)
#else
  #define let_set_id(p, Id)            (T_Let(p))->object.let.id = Id
  #define let_set_slots(p, Slot)       (T_Let(p))->object.let.slots = T_Sln(Slot)
  #define C_Let(p, role)               p
  #define S_Let(p, role)               p
#endif
#define funclet_function(p)            T_Sym((C_Let(p, L_Func))->object.let.edat.efnc.function)
#define funclet_set_function(p, F)     (S_Let(p, L_Func))->object.let.edat.efnc.function = T_Sym(F)
#define set_curlet(Sc, P)              Sc->curlet = T_Let(P)

#define let_baffle_key(p)              (T_Let(p))->object.let.edat.key
#define let_set_baffle_key(p, K)       (T_Let(p))->object.let.edat.key = K

#define let_line(p)                    (C_Let(p, L_Func))->object.let.edat.efnc.line
#define let_set_line(p, L)             (S_Let(p, L_Func))->object.let.edat.efnc.line = L
#define let_file(p)                    (C_Let(p, L_Func))->object.let.edat.efnc.file
#define let_set_file(p, F)             (S_Let(p, L_Func))->object.let.edat.efnc.file = F

#define let_dox_slot1(p)               T_Slt((C_Let(p, L_Dox))->object.let.edat.dox.dox1)
#define let_set_dox_slot1(p, S)        do {(S_Let(p, L_Dox))->object.let.edat.dox.dox1 = T_Slt(S); set_has_dox_slot1(p);} while (0)
#define let_dox_slot2(p)               T_Sld((C_Let(p, L_Dox))->object.let.edat.dox.dox2) /* #<undefined> from opt_cell_do */
#define let_set_dox_slot2(p, S)        do {(S_Let(p, L_Dox))->object.let.edat.dox.dox2 = T_Slt(S); set_has_dox_slot2(p);} while (0)
#define let_dox_slot2_unchecked(p)     T_Sld(C_Let(p, L_Dox)->object.let.edat.dox.dox2)
#define let_set_dox_slot2_unchecked(p, S) do {S_Let(p, L_Dox)->object.let.edat.dox.dox2 = T_Sld(S); set_has_dox_slot2(p);} while (0)
#define let_dox1_value(p)              slot_value(let_dox_slot1(p))
#define let_dox2_value(p)              slot_value(let_dox_slot2(p))

#define unique_name(p)                 (p)->object.unq.name /* not T_Uniq(p) here -- see make_unique */
#define unique_name_length(p)          (p)->object.unq.len
#define is_unspecified(p)              (type(p) == T_UNSPECIFIED)
#define unique_car(p)                  (p)->object.unq.car
#define unique_cdr(p)                  (p)->object.unq.cdr

#define is_undefined(p)                (type(p) == T_UNDEFINED)
#define undefined_name(p)              (T_Undf(p))->object.undef.name
#define undefined_name_length(p)       (T_Undf(p))->object.undef.len
#define undefined_set_name_length(p, L) (T_Undf(p))->object.undef.len = L
#define eof_name(p)                    (T_Eof(p))->object.eof.name
#define eof_name_length(p)             (T_Eof(p))->object.eof.len

#define is_any_vector(p)               t_vector_p[type(p)]
#define is_t_vector(p)                 (type(p) == T_VECTOR)
#define vector_length(p)               (p)->object.vector.length
#define vector_elements_unchecked(p)   (p)->object.vector.elements.objects
#define vector_element_unchecked(p, i) ((p)->object.vector.elements.objects[i])
#define vector_element(p, i)           ((T_Nvc(p))->object.vector.elements.objects[i])
#define vector_elements(p)             (T_Nvc(p))->object.vector.elements.objects
#define any_vector_elements(p)         (T_Vec(p))->object.vector.elements.objects
#define vector_getter(p)               (T_Vec(p))->object.vector.vget
#define vector_setter(p)               (T_Vec(p))->object.vector.setv.vset
#define vector_block(p)                (T_Vec(p))->object.vector.block
#define vector_block_unchecked(p)      p->object.vector.block

#define typed_vector_typer(p)          T_Prc((T_Nvc(p))->object.vector.setv.fset)
#define typed_vector_set_typer(p, Fnc) (T_Nvc(p))->object.vector.setv.fset = T_Prc(Fnc)
#define typed_vector_gc_mark(p)        ((is_c_function(typed_vector_typer(p))) ? c_function_marker(typed_vector_typer(p)) : mark_typed_vector_1)
#define typed_vector_typer_call(Sc, p, Args) \
  ((is_c_function(typed_vector_typer(p))) ? c_function_call(typed_vector_typer(p))(Sc, Args) : s7_apply_function(Sc, typed_vector_typer(p), Args))

#define is_int_vector(p)               (type(p) == T_INT_VECTOR)
#define int_vector(p, i)               ((T_Ivc(p))->object.vector.elements.ints[i])
#define int_vector_ints(p)             (T_Ivc(p))->object.vector.elements.ints

#define is_float_vector(p)             (type(p) == T_FLOAT_VECTOR)
#define float_vector(p, i)             ((T_Fvc(p))->object.vector.elements.floats[i])
#define float_vector_floats(p)         (T_Fvc(p))->object.vector.elements.floats

#define is_complex_vector(p)           (type(p) == T_COMPLEX_VECTOR)
#define complex_vector(p, i)           ((T_Cvc(p))->object.vector.elements.complexes[i])
#define complex_vector_complexes(p)    (T_Cvc(p))->object.vector.elements.complexes

#define is_byte_vector(p)              (type(p) == T_BYTE_VECTOR)
#define byte_vector_length(p)          (T_BVc(p))->object.vector.length
#define byte_vector_bytes(p)           (T_BVc(p))->object.vector.elements.bytes
#define byte_vector(p, i)              ((T_BVc(p))->object.vector.elements.bytes[i])
#define is_string_or_byte_vector(p)    ((type(p) == T_STRING) || (type(p) == T_BYTE_VECTOR))

#define vector_dimension_info(p)       ((vdims_t *)(T_Vec(p))->object.vector.block->ex.ex_info)
#define vector_set_dimension_info(p, d) (T_Vec(p))->object.vector.block->ex.ex_info = (void  *)d
#define vector_ndims(p)                vdims_rank(vector_dimension_info(p))
#define vector_dimension(p, i)         vdims_dims(vector_dimension_info(p))[i]
#define vector_dimensions(p)           vdims_dims(vector_dimension_info(p))
#define vector_offset(p, i)            vdims_offsets(vector_dimension_info(p))[i]
#define vector_offsets(p)              vdims_offsets(vector_dimension_info(p))
#define vector_rank(p)                 ((vector_dimension_info(p)) ? vector_ndims(p) : 1)
#define vector_has_dimension_info(p)   (vector_dimension_info(p))

#define subvector_vector(p)            T_Vec(((vector_dimension_info(T_SVec(p))) ? vdims_original(vector_dimension_info(p)) : (p)->object.vector.block->nx.ksym))
#define subvector_set_vector(p, vect)  (T_SVec(p))->object.vector.block->nx.ksym = T_Vec(vect)

#define stack_element(p, i)            vector_element_unchecked(T_Stk(p), i)
#define stack_elements(p)              vector_elements_unchecked(T_Stk(p))
#define stack_block(p)                 vector_block_unchecked(T_Stk(p))
#define stack_top(Sc)                  ((Sc)->stack_end - (Sc)->stack_start)
#define temp_stack_top(p)              (T_Stk(p))->object.stk.top
/* #define stack_flags(p)              (T_Stk(p))->object.stk.flags */
#define stack_clear_flags(p)           (T_Stk(p))->object.stk.flags = 0
#define stack_has_pairs(p)             (((T_Stk(p))->object.stk.flags & 1) != 0)
#define stack_set_has_pairs(p)         (T_Stk(p))->object.stk.flags |= 1
#define stack_has_counters(p)          (((T_Stk(p))->object.stk.flags & 2) != 0)
#define stack_set_has_counters(p)      (T_Stk(p))->object.stk.flags |= 2

#define is_hash_table(p)               (type(p) == T_HASH_TABLE)
#define is_mutable_hash_table(p)       ((full_type(T_Ext(p)) & (Type_Mask | T_Immutable)) == T_HASH_TABLE)
#define hash_table_mask(p)             (T_Hsh(p))->object.hasher.mask
/* hash_table_mask comes from an earlier hash_map, but hash_table_size is still a power of 2, so hash_map's % wastes an entry, but
 *   the mask (pow2-1) is still useful -- in pointer_map for example if we divide by pow2, we can end up just tossing the 3 lower bits,
 *   (hash-table :a 1 :b 2 :c 3) -> hash-stats:empty|1|2|n|most (7 0 0 1 3)
 */
#define hash_table_size(p)             ((T_Hsh(p))->object.hasher.mask + 1)
#define hash_table_block(p)            (T_Hsh(p))->object.hasher.block
#define hash_table_block_unchecked(p)  p->object.hasher.block
#define hash_table_set_block(p, b)     (T_Hsh(p))->object.hasher.block = b
#define hash_table_element(p, i)       (T_Hsh(p))->object.hasher.elements[i]
#define hash_table_elements(p)         (T_Hsh(p))->object.hasher.elements /* block data (dx) */
#define hash_table_entries(p)          hash_table_block(p)->nx.nx_uint
#define hash_table_checker(p)          (T_Hsh(p))->object.hasher.hash_func
#define hash_table_mapper(p)           (T_Hsh(p))->object.hasher.loc
#define hash_table_procedures(p)       T_Lst(hash_table_block(p)->ex.ex_ptr)
#define hash_table_set_procedures(p, Lst)       hash_table_block(p)->ex.ex_ptr = T_Lst(Lst)  /* both the checker/mapper: car/cdr, and the two typers (opt1/opt2) */
#define hash_table_procedures_checker(p)        T_Prc(car(hash_table_procedures(p)))
#define hash_table_procedures_mapper(p)         T_Prc(cdr(hash_table_procedures(p)))
#define hash_table_set_procedures_checker(p, f) set_car(hash_table_procedures(p), T_Prc(f))
#define hash_table_set_procedures_mapper(p, f)  set_cdr(hash_table_procedures(p), T_Prc(f))
#define hash_table_key_typer(p)                 T_Prc(opt1_any(hash_table_procedures(p)))
#define hash_table_key_typer_unchecked(p)       hash_table_block(p)->ex.ex_ptr->object.cons.opt1
#define hash_table_set_key_typer(p, Fnc)        set_opt1_any(hash_table_procedures(T_Hsh(p)), T_Prc(Fnc))
#define hash_table_value_typer(p)               T_Prc(opt2_any(hash_table_procedures(p)))
#define hash_table_value_typer_unchecked(p)     hash_table_block(p)->ex.ex_ptr->object.cons.o2.opt2
#define hash_table_set_value_typer(p, Fnc)      set_opt2_any(hash_table_procedures(T_Hsh(p)), T_Prc(Fnc))
#define weak_hash_iters(p)                      hash_table_block(p)->ln.iter_or_size
#define missing_key_value(Sc)                   hash_entry_value(Sc->unentry)

#if s7_Debugging
  #define T_Itr_Pos(p)                 titr_pos(sc, T_Itr(p), __func__, __LINE__)
  #define T_Itr_Len(p)                 titr_len(sc, T_Itr(p), __func__, __LINE__)
  #define T_Itr_Hash(p)                titr_hash(sc, T_Itr(p), __func__, __LINE__)
  #define T_Itr_Let(p)                 titr_let(sc, T_Itr(p), __func__, __LINE__)
  #define T_Itr_Pair(p)                titr_pair(sc, T_Itr(p), __func__, __LINE__)
#else
  #define T_Itr_Pos(p)                 p
  #define T_Itr_Len(p)                 p
  #define T_Itr_Hash(p)                p
  #define T_Itr_Let(p)                 p
  #define T_Itr_Pair(p)                p
#endif

#define is_iterator(p)                 (type(p) == T_ITERATOR)
#define iterator_sequence(p)           (T_Itr(p))->object.iter.seq
#define iterator_position(p)           (T_Itr_Pos(p))->object.iter.lc.loc
#define iterator_length(p)             (T_Itr_Len(p))->object.iter.lw.len
#define iterator_next(p)               (T_Itr(p))->object.iter.next
#define iterator_current(p)            (T_Itr(p))->object.iter.cur
#define iterator_carrier(p)            (T_Itr(p))->object.iter.cur
#define iterator_is_at_end(p)          (!iter_ok(p))                /* ((full_type(T_Itr(p)) & T_Iter_Ok) == 0) */
#define iterator_at_end_value(Sc)      Sc->iterator_at_end_value

#define pair_iterator_slow(p)          T_Lst((T_Itr_Pair(p))->object.iter.lw.slow) /* applies only to pairs */
#define pair_iterator_set_slow(p, Val) (T_Itr_Pair(p))->object.iter.lw.slow = T_Lst(Val)
#define hash_iterator_entry(p)         (T_Itr_Hash(p))->object.iter.lw.entry       /* applies only to hash-tables */
#define let_iterator_slot(p)           T_Sln((T_Itr_Let(p))->object.iter.lc.slot)  /* applies only to lets */
#define let_iterator_set_slot(p, Val)  (T_Itr_Let(p))->object.iter.lc.slot = T_Sln(Val)

#define is_input_port(p)               (type(p) == T_INPUT_PORT)
#define is_output_port(p)              (type(p) == T_OUTPUT_PORT)
#define port_port(p)                   (T_Prt(p))->object.prt.port
#define is_string_port(p)              (port_type(p) == String_Port)
#define is_file_port(p)                (port_type(p) == File_Port)
#define is_function_port(p)            (port_type(p) == Function_Port)
#define port_filename_block(p)         port_port(p)->filename_block
#define port_filename(p)               port_port(p)->filename
#define port_filename_length(p)        port_port(p)->filename_length
#define port_file(p)                   port_port(p)->file
#define port_data_block(p)             port_port(p)->block
#define port_data_block_unchecked(p)   p->object.prt.port->block
#define port_line_number(p)            port_port(p)->line_number
#define port_file_number(p)            port_port(p)->file_number
#define port_data(p)                   (T_Prt(p))->object.prt.data
#define port_data_size(p)              (T_Prt(p))->object.prt.size
#define port_position(p)               (T_Prt(p))->object.prt.point
#define port_block(p)                  (T_Prt(p))->object.prt.block
#define port_type(p)                   port_port(p)->ptype
#define port_is_closed(p)              port_port(p)->is_closed
#define port_set_closed(p, Val)        port_port(p)->is_closed = Val
#define port_needs_free(p)             port_port(p)->needs_free
#define port_next(p)                   port_block(p)->nx.next
#define port_output_function(p)        port_port(p)->output_function /* these two are for function ports */
#define port_input_function(p)         port_port(p)->input_function
#define port_string_or_function(p)     port_port(p)->orig_str
#define port_set_string_or_function(p, S) port_port(p)->orig_str = S

#define current_input_port(Sc)         T_Pri(Sc->input_port)
#define set_current_input_port(Sc, P)  Sc->input_port = T_Pri(P)
#define current_output_port(Sc)        T_Pro(Sc->output_port)
#define set_current_output_port(Sc, P) Sc->output_port = T_Pro(P)
#define current_error_port(Sc)         T_Pro(Sc->error_port)
#define set_current_error_port(Sc, P)  Sc->error_port = T_Pro(P)

#define port_read_character(p)         port_port(p)->pf->read_character
#define port_read_line(p)              port_port(p)->pf->read_line
#define port_display(p)                port_port(p)->pf->displayer
#define port_write_character(p)        port_port(p)->pf->write_character
#define port_write_string(p)           port_port(p)->pf->write_string
#define port_read_semicolon(p)         port_port(p)->pf->read_semicolon
#define port_read_white_space(p)       port_port(p)->pf->read_white_space
#define port_read_name(p)              port_port(p)->pf->read_name
#define port_read_sharp(p)             port_port(p)->pf->read_sharp
#define port_close(p)                  port_port(p)->pf->close_port

#define is_c_function(f)               (type(f) >= T_C_FUNCTION)                   /* does not include T_C_FUNCTION_STAR */
#define is_c_function_star(f)          (type(f) == T_C_FUNCTION_STAR)
#define is_any_c_function(f)           (type(f) >= T_C_FUNCTION_STAR)
#define is_safe_c_function(f)          ((is_c_function(f)) && (is_safe_procedure(f)))
#define c_function_data(f)             (T_Fnc(f))->object.fnc.c_proc               /* not T_CFn -- this also applies to T_C_MACROs */
#define c_function_call(f)             (T_Fnc(f))->object.fnc.ff
#define c_function_min_args(f)         (T_Fnc(f))->object.fnc.required_args
#define c_function_optional_args(f)    (T_Fnc(f))->object.fnc.optional_args
#define c_function_max_args(f)         (T_Fnc(f))->object.fnc.all_args
#define c_function_is_aritable(f, N)   ((c_function_min_args(f) <= N) && (c_function_max_args(f) >= N))
#define c_function_name(f)             c_function_data(f)->name                    /* const char* */
#define c_function_name_length(f)      c_function_data(f)->name_length             /* int32_t */
#define c_function_documentation(f)    c_function_data(f)->doc                     /* const char* */
#define c_function_signature(f)        T_Prf(c_function_data(f)->signature)        /* pair or #f */
#define c_function_set_signature(f, Val) c_function_data(f)->signature = T_Prf(Val)
#define c_function_setter(f)           T_Prc(c_function_data(f)->setter)
#define c_function_set_setter(f, Val)  c_function_data(f)->setter = T_Prc(Val)
#define c_function_class(f)            c_function_data(f)->class_id                /* uint32_t */
#define c_function_chooser(f)          c_function_data(f)->chooser
#define c_function_base(f)             T_CFn(c_function_data(f)->generic_ff)
#define c_function_set_base(f, Val)    c_function_data(f)->generic_ff = T_CFn(Val)
#define c_function_marker(f)           c_function_data(f)->cam.marker              /* the mark function for the vector (mark_vector_1 etc) */
#define c_function_set_marker(f, Val)  c_function_data(f)->cam.marker = Val
#define c_function_symbol(f)           T_Sym(c_function_data(f)->sam.c_sym)        /* f is c_function or c_macro, but not c_function* -- doesn't fit current checks */
#define c_function_set_symbol(f, Sym)  c_function_data(f)->sam.c_sym = T_Sym(Sym)
#define c_function_let(f)              T_Let(c_function_data(f)->let)
#define c_function_set_let(f, Val)     c_function_data(f)->let = T_Let(Val)

#define c_function_bool_setter(f)      T_CFn(c_function_data(f)->dam.bool_setter)
#define c_function_set_bool_setter(f, Val) c_function_data(f)->dam.bool_setter = T_CFn(Val)

#define c_function_arg_defaults(f)     c_function_data(T_Fst(f))->dam.arg_defaults /* array of s7_pointer */
#define c_function_call_args(f)        c_function_data(T_Fst(f))->cam.call_args    /* pair or NULL */
#define c_function_par_names(f)        c_function_data(T_Fst(f))->sam.arg_names    /* array of s7_pointer */
#define c_function_opt_data(f)         c_function_data(f)->opt_data                /* opt_funcs_t (vunion) */

#define is_c_macro(p)                  (type(p) == T_C_MACRO)
#define c_macro_data(f)                (T_CMac(f))->object.fnc.c_proc
#define c_macro_call(f)                (T_CMac(f))->object.fnc.ff
#define c_macro_name(f)                c_macro_data(f)->name
#define c_macro_name_length(f)         c_macro_data(f)->name_length
#define c_macro_min_args(f)            (T_CMac(f))->object.fnc.required_args
#define c_macro_max_args(f)            (T_CMac(f))->object.fnc.all_args
#define c_macro_setter(f)              T_Prc(c_macro_data(f)->setter)
#define c_macro_set_setter(f, Val)     c_macro_data(f)->setter = T_Prc(Val)
#define could_be_macro_setter(Obj)     t_macro_setter_p[type(Obj)]

#define is_random_state(p)             (type(p) == T_RANDOM_STATE)
#define random_gmp_state(p)            (p)->object.rng.state /* sweep sees free cell in big_random_state gc_list and needs to call gmprandclear on its value */
#define random_seed(p)                 (T_Ran(p))->object.rng.seed
#define random_carry(p)                (T_Ran(p))->object.rng.carry

#define continuation_block(p)          (T_Con(p))->object.cwcc.block
#define continuation_stack(p)          T_Stk(T_Con(p)->object.cwcc.stack)
#define continuation_set_stack(p, Val) (T_Con(p))->object.cwcc.stack = T_Stk(Val)
#define continuation_stack_end(p)      (T_Con(p))->object.cwcc.stack_end
#define continuation_stack_start(p)    (T_Con(p))->object.cwcc.stack_start
#define continuation_stack_top(p)      (continuation_stack_end(p) - continuation_stack_start(p))
#define continuation_op_stack(p)       (T_Con(p))->object.cwcc.op_stack
#define continuation_stack_size(p)     continuation_block(p)->nx.ix.i1
#define continuation_op_loc(p)         continuation_block(p)->nx.ix.i2
#define continuation_op_size(p)        continuation_block(p)->ln.iter_or_size
#define continuation_key(p)            continuation_block(p)->ex.ckey
/* this can overflow int32_t -- baffle_key is s7_int, so ckey should be also */
#define continuation_name(p)           continuation_block(p)->dx.d_ptr

#define call_exit_goto_loc(p)          (T_Got(p))->object.rexit.goto_loc
#define call_exit_op_loc(p)            (T_Got(p))->object.rexit.op_stack_loc
#define call_exit_active(p)            (T_Got(p))->object.rexit.active
#define call_exit_name(p)              (T_Got(p))->object.rexit.name

#define is_continuation(p)             (type(p) == T_CONTINUATION)
#define is_goto(p)                     (type(p) == T_GOTO)
#define is_macro(p)                    (type(p) == T_MACRO)
#define is_macro_star(p)               (type(p) == T_MACRO_STAR)
#define is_bacro(p)                    (type(p) == T_BACRO)
#define is_bacro_star(p)               (type(p) == T_BACRO_STAR)
#define is_either_macro(p)             ((is_macro(p)) || (is_macro_star(p)))
#define is_either_bacro(p)             ((is_bacro(p)) || (is_bacro_star(p)))

#define is_closure(p)                  (type(p) == T_CLOSURE)
#define is_closure_star(p)             (type(p) == T_CLOSURE_STAR)
#define closure_pars(p)                T_Arg((T_Clo(p))->object.func.args)
#define closure_set_pars(p, Val)       (T_Clo(p))->object.func.args = T_Arg(Val)
#define closure_body(p)                (T_Pair((T_Clo(p))->object.func.body))
#define closure_set_body(p, Val)       (T_Clo(p))->object.func.body = T_Pair(Val)
#define closure_let(p)                 T_Let((T_Clo(p))->object.func.let)
#define closure_set_let(p, L)          (T_Clo(p))->object.func.let = T_Let(L)
#define closure_arity(p)               (T_Clo(p))->object.func.arity
#define closure_set_arity(p, A)        (T_Clo(p))->object.func.arity = A

#define closure_setter(p)              (T_Prc((T_Clo(p))->object.func.setter))
#define closure_set_setter(p, Val)     (T_Clo(p))->object.func.setter = T_Prc(Val)
#define closure_map_list(p)            (T_Pair((T_Clo(p))->object.func.setter))
#define closure_set_map_list(p, Val)   (T_Clo(p))->object.func.setter = T_Pair(Val)
#define closure_setter_or_map_list(p)  (T_Clo(p)->object.func.setter)
#define closure_set_setter_or_map_list(p, Val) T_Clo(p)->object.func.setter = Val
/* closure_map_list refers to a cyclic list detector in map */

#define Closure_Arity_Not_Set          0x40000000
#define Max_Arity                      0x20000000
#define closure_arity_unknown(p)       (closure_arity(p) == Closure_Arity_Not_Set)
#define is_thunk(Sc, Fnc)              ((type(Fnc) >= T_GOTO) && (s7_is_aritable(Sc, Fnc, 0)))

#define hook_has_functions(p)          (is_pair(s7_hook_functions(sc, T_Clo(p))))

#define catch_tag(p)                   (T_Cat(p))->object.rcatch.tag
#define catch_goto_loc(p)              (T_Cat(p))->object.rcatch.goto_loc
#define catch_op_loc(p)                (T_Cat(p))->object.rcatch.op_stack_loc
#define catch_cstack(p)                (T_Cat(p))->object.rcatch.cstack
#define catch_handler(p)               T_Ext((T_Cat(p))->object.rcatch.handler)
#define catch_set_handler(p, val)      (T_Cat(p))->object.rcatch.handler = T_Ext(val)

#define dynamic_wind_state(p)          (T_Dyn(p))->object.winder.state
#define dynamic_wind_in(p)             (T_Dyn(p))->object.winder.in
#define dynamic_wind_out(p)            (T_Dyn(p))->object.winder.out
#define dynamic_wind_body(p)           (T_Dyn(p))->object.winder.body

#define is_c_object(p)                 (type(p) == T_C_OBJECT)
#define c_object_value(p)              (T_Obj(p))->object.c_obj.value
#define c_object_type(p)               (T_Obj(p))->object.c_obj.type
#define c_object_let(p)                T_Let((T_Obj(p))->object.c_obj.let)
#define c_object_set_let(p, L)         (T_Obj(p))->object.c_obj.let = T_Let(L)
#define c_object_sc(p)                 (T_Obj(p))->object.c_obj.sc

#define c_object_info(Sc, p)           Sc->c_object_types[c_object_type(T_Obj(p))]
#define c_object_free(Sc, p)           c_object_info(Sc, p)->free
#define c_object_mark(Sc, p)           c_object_info(Sc, p)->mark
#define c_object_gc_mark(Sc, p)        c_object_info(Sc, p)->gc_mark
#define c_object_gc_free(Sc, p)        c_object_info(Sc, p)->gc_free
#define c_object_ref(Sc, p)            c_object_info(Sc, p)->ref
#define c_object_getf(Sc, p)           c_object_info(Sc, p)->getter
#define c_object_set(Sc, p)            c_object_info(Sc, p)->set
#define c_object_setf(Sc, p)           c_object_info(Sc, p)->setter
#if !Disable_Deprecated
  #define c_object_print(Sc, p)        c_object_info(Sc, p)->print
#endif
#define c_object_len(Sc, p)            c_object_info(Sc, p)->length
#define c_object_eql(Sc, p)            c_object_info(Sc, p)->eql
#define c_object_equal(Sc, p)          c_object_info(Sc, p)->equal
#define c_object_equivalent(Sc, p)     c_object_info(Sc, p)->equivalent
#define c_object_fill(Sc, p)           c_object_info(Sc, p)->fill
#define c_object_copy(Sc, p)           c_object_info(Sc, p)->copy
#define c_object_reverse(Sc, p)        c_object_info(Sc, p)->reverse
#define c_object_to_list(Sc, p)        c_object_info(Sc, p)->to_list
#define c_object_to_string(Sc, p)      c_object_info(Sc, p)->to_string
#define c_object_scheme_name(Sc, p)    T_Str(c_object_info(Sc, p)->scheme_name)

#define c_pointer(p)                   (T_Ptr(p))->object.cptr.c_pointer
#define c_pointer_type(p)              (T_Ptr(p))->object.cptr.c_type
#define c_pointer_info(p)              (T_Ptr(p))->object.cptr.info
#define c_pointer_weak1(p)             (T_Ptr(p))->object.cptr.weak1
#define c_pointer_weak2(p)             (T_Ptr(p))->object.cptr.weak2
#define c_pointer_set_weak1(p, q)      (T_Ptr(p))->object.cptr.weak1 = T_Ext(q)
#define c_pointer_set_weak2(p, q)      (T_Ptr(p))->object.cptr.weak2 = T_Ext(q)
#define is_c_pointer(p)                (type(p) == T_C_POINTER)

#define is_counter(p)                  (type(p) == T_COUNTER)
#define counter_result(p)              (T_Ctr(p))->object.ctr.result
#define counter_set_result(p, Val)     (T_Ctr(p))->object.ctr.result = T_Ext(Val)
#define counter_list(p)                (T_Ctr(p))->object.ctr.list
#define counter_set_list(p, Val)       (T_Ctr(p))->object.ctr.list = T_Ext(Val)
#define counter_capture(p)             (T_Ctr(p))->object.ctr.cap
#define counter_set_capture(p, Val)    (T_Ctr(p))->object.ctr.cap = Val
#define counter_let(p)                 T_Let((T_Ctr(p))->object.ctr.let)
#define counter_set_let(p, L)          (T_Ctr(p))->object.ctr.let = T_Let(L)
#define counter_slots(p)               T_Sln(T_Ctr(p)->object.ctr.slots)
#define counter_set_slots(p, Val)      (T_Ctr(p))->object.ctr.slots = T_Sln(Val)

#if s7_Debugging
#define begin_temp(P, Val)             do {s7_pointer __val__ = Val; begin_temp_1(sc, P, __func__, __LINE__); P = __val__;} while (0)
static void begin_temp_1(s7_scheme *sc, s7_pointer p, const char *func, int line)
{
  if(p != sc->unused)
    {
      char *str;
      fprintf(stderr, "%s[%d]: begin_temp %s %d %s\n", func, line,
	      (p == sc->y) ? "sc->y" : ((p == sc->v) ? "sc->v" : ((p == sc->x) ? "sc->x" : ((p == sc->temp6) ? "sc->temp6" : "???"))),
	      (p == sc->y) ? sc->y_line : ((p == sc->v) ? sc->v_line : ((p == sc->x) ? sc->x_line : sc->t_line)),
	      str = s7_object_to_c_string(sc, p));
      free(str);
      /* abort(); */ /* this happens when an error interrupts a loop (for-each etc) so end_temp is missed */
    }
  if (p == sc->y) sc->y_line = line;
  else
    if (p == sc->v) sc->v_line = line;
    else
      if (p == sc->x) sc->x_line = line;
      else sc->t_line = line;
}
#else
#define begin_temp(p, Val)              p = Val
#endif
#define end_temp(p)                     p = sc->unused
#define return_with_end_temp(Temp)      do {s7_pointer Result = Temp; end_temp(Temp); return(Result);} while (0)

#if __cplusplus && Have_Complex_Numbers
  static s7_double Real(complex<s7_double> x) {return(real(x));} /* protect the C++ name */
  static s7_double Imag(complex<s7_double> x) {return(imag(x));}
#endif

#define integer(p)                     (T_Int(p))->object.number.integer_value
#define set_integer(p, x)              integer(p) = x
#define real(p)                        (T_Rel(p))->object.number.real_value
#define set_real(p, x)                 real(p) = x
#define numerator(p)                   (T_Frc(p))->object.number.fraction_value.numerator
#define set_numerator(p, x)            numerator(p) = x
#define denominator(p)                 (T_Frc(p))->object.number.fraction_value.denominator
#define set_denominator(p, x)          denominator(p) = x
#define fraction(p)                    (((long_double)numerator(p)) / ((long_double)denominator(p)))
#define inverted_fraction(p)           (((long_double)denominator(p)) / ((long_double)numerator(p)))
#define real_part(p)                   (T_Cmp(p))->object.number.cz.complex_value.rl
#define set_real_part(p, x)            real_part(p) = x
#define imag_part(p)                   (T_Cmp(p))->object.number.cz.complex_value.im
#define set_imag_part(p, x)            imag_part(p) = x
#if WITH_CLANG_PP
#define a_bi(p)                        CMPLX((T_Cmp(p))->object.number.cz.complex_value.rl, p->object.number.cz.complex_value.im)
#else
#define a_bi(p)                        (T_Cmp(p))->object.number.cz.z
#define set_a_bi(p, x)                 a_bi(p) = x
#endif
#if Have_Complex_Numbers
  #define to_c_complex(p)              CMPLX(real_part(p), imag_part(p))
#endif

#if With_Gmp
#define big_integer(p)                 ((T_Bgi(p))->object.number.bgi->n)
#define big_integer_nxt(p)             (T_Bgi(p))->object.number.bgi->nxt
#define big_integer_bgi(p)             (T_Bgi(p))->object.number.bgi
#define big_ratio(p)                   ((T_Bgf(p))->object.number.bgr->q)
#define big_ratio_nxt(p)               (T_Bgf(p))->object.number.bgr->nxt
#define big_ratio_bgr(p)               (T_Bgf(p))->object.number.bgr
#define big_real(p)                    ((T_Bgr(p))->object.number.bgf->x)
#define big_real_nxt(p)                (T_Bgr(p))->object.number.bgf->nxt
#define big_real_bgf(p)                (T_Bgr(p))->object.number.bgf
#define big_complex(p)                 ((T_Bgz(p))->object.number.bgc->z)
#define big_complex_nxt(p)             (T_Bgz(p))->object.number.bgc->nxt
#define big_complex_bgc(p)             (T_Bgz(p))->object.number.bgc
#endif

#if s7_Debugging
const char *display(s7_pointer obj)
{
  const char *result;
  if (!has_methods_unchecked(obj))
    return(string_value(s7_object_to_string(cur_sc, obj, false)));
  clear_type_bit(obj, T_Has_Methods); /* clear_has_methods calls T_Met -> check_ref_met */
  result = string_value(s7_object_to_string(cur_sc, obj, false));
  set_type_bit(obj, T_Has_Methods);   /* same for set_has_methods */
  return(result);
}
#else
#define display(Obj) string_value(s7_object_to_string(sc, Obj, false))
#endif
#define display_truncated(Obj) string_value(object_to_string_truncated(sc, Obj))

#if s7_Debugging
static void check_mutable_bit(s7_pointer p)
{
  if (!is_mutable(p))
    fprintf(stderr, "%s[%d]: mutable cleared already?\n", p->gc_func, p->gc_line);
}

static void set_alloc_info(s7_pointer p, const char *func, int32_t line)
{
  p->alloc_line = line;
  p->alloc_func = func;
  p->explicit_free_line = 0;
}

static void set_type_1(s7_pointer p, s7_uint typ, const char *func, int32_t line)
{
  p->alloc_line = line;
  p->alloc_func = func;
  p->alloc_type = typ;
  p->explicit_free_line = 0;
  p->uses++;
  if (((typ) & Type_Mask) == T_FREE)
    fprintf(stderr, "%d: set free, %p type to #x%" PRIx64 "\n", __LINE__, p, (s7_int)(typ));
  else
    if (((typ) & Type_Mask) >= Num_Types)
      fprintf(stderr, "%d: set invalid type, %p type to #x%" PRIx64 "\n", __LINE__, p, (s7_int)(typ));
    else
      {
	if (((full_type(p) & T_Immutable) != 0) && ((full_type(p) != (s7_uint)(typ))))
	  {fprintf(stderr, "%s[%d]: set immutable %p type %d to %" ld64 "\n", __func__, __LINE__, p, type_unchecked(p), (s7_int)(typ)); abort();}
	if (((full_type(p) & T_Unheap) != 0) && (((typ) & T_Unheap) == 0))
	  fprintf(stderr, "%s[%d]: clearing unheap in set type!\n", __func__, __LINE__);
      }
  full_type(p) = typ;
}
#else
  #define set_alloc_info(P, Func, Line)
#endif

static int32_t s7_int_digits_by_radix[17];

#define s7_Int_Bits 63

#define s7_Int64_Max 9223372036854775807LL
/* #define s7_Int64_Min -9223372036854775808LL */   /* why is this disallowed in C? "warning: integer constant is so large that it is unsigned" */
#define s7_Int64_Min (int64_t)(-s7_Int64_Max - 1LL) /* in gcc 9 we had to assign this to an s7_int, then use that! */

#define s7_Int32_Max 2147483647LL
#define s7_Int32_Min (-s7_Int32_Max - 1LL)

static void init_int_limits(void)
{
#if With_Gmp
  #define s7_Log_Int64_Max 36.736800
#else
  /* actually not safe = (log (- (expt 2 63) 1)) and (log (- (expt 2 31) 1)) (using 63 and 31 bits) */
  #define s7_Log_Int64_Max 43.668274
#endif
  s7_int_digits_by_radix[0] = 0;
  s7_int_digits_by_radix[1] = 0;
  for (int32_t i = 2; i < 17; i++)
    s7_int_digits_by_radix[i] = (int32_t)(floor(s7_Log_Int64_Max / log((double)i)));
}

static s7_pointer make_permanent_integer(s7_int i)
{
  s7_pointer p = (s7_pointer)Malloc(sizeof(s7_cell)); /* was using Calloc to clear name. 22-May-25 */
  full_type(p) = T_Immutable | T_INTEGER | T_Unheap;
  set_alloc_info(p, __func__, __LINE__);
  set_integer(p, i);
  return(p);
}

#define Num_Chars 256
#ifndef NUM_SMALL_INTS
  #define NUM_SMALL_INTS 8192
#else
#if (NUM_SMALL_INTS < Num_Chars) /* g_char_to_integer assumes this is at least Num_Chars, as does the byte_vector stuff (256) */
  #error NUM_SMALL_INTS is less than Num_Chars which will not work
#endif
#endif
/* if NUM_SMALL_INTS 256: tvect +240, trclo +200, tfft +170, trec +300, etc -- mostly gc + various adds */
/*                 16384: tvect 0,    trclo -20,  tfft 0,    trec -50 */
#define Num_Small_Ints NUM_SMALL_INTS

static bool t_number_separator_p[Num_Chars];
static s7_cell *small_ints = NULL;
#define small_int(Val) &small_ints[Val]
#define is_small_int(n) ((n & ~(Num_Small_Ints - 1)) == 0)   /* ((n >= 0) && (n < Num_Small_Ints)) is slower */

static s7_pointer real_zero, real_NaN, complex_NaN, real_pi, real_one, arity_not_set, max_arity, real_infinity, real_minus_infinity;
static s7_pointer int_zero, int_one, int_two, int_three, minus_one, minus_two, mostfix, leastfix;

static void init_small_ints(void)
{
  s7_cell *cells;
  small_ints = (s7_cell *)Malloc(Num_Small_Ints * sizeof(s7_cell)); /* was calloc 14-Apr-22, used a pointless intermediate s7_pointer array until 22-May-25 */
  /* we only use 16 bytes of s7_cell for integers and floats, but if we used s7_small_cell here we'd need a ton of casts and repetitive
   *   functions (using s7_small_pointer not s7_pointer, etc);  the gain in space might be 256k bytes -- not worth the complexity.
   *   we could I suppose preset these built-in ints/floats at compile-time -- too ugly.
   */
  for (int32_t i = 0; i < Num_Small_Ints; i++)
    {
      s7_pointer p;
      p = &small_ints[i];
      full_type(p) = T_Immutable | T_INTEGER | T_Unheap;
      set_integer(p, i);
      set_alloc_info(p, __func__, __LINE__);
    }
  /* setup a few other numbers while we're here */
  #define Extra_Numbers 11
  cells = (s7_cell *)Malloc(Extra_Numbers * sizeof(s7_cell));

  #define init_integer(Ptr, Num) \
    do {full_type(Ptr) = T_INTEGER | T_Immutable | T_Unheap; set_integer(Ptr, Num); set_alloc_info(Ptr, __func__, __LINE__);} while (0)
  #define init_real(Ptr, Num) \
    do {full_type(Ptr) = T_REAL | T_Immutable | T_Unheap; set_real(Ptr, Num); set_alloc_info(Ptr, __func__, __LINE__);} while (0)
  #define init_complex(Ptr, Real, Imag) \
    do {full_type(Ptr) = T_COMPLEX | T_Immutable | T_Unheap; set_real_part(Ptr, Real); set_imag_part(Ptr, Imag); set_alloc_info(Ptr, __func__, __LINE__);} while (0)

  real_zero = &cells[0]; init_real(real_zero, 0.0);
  real_one = &cells[1]; init_real(real_one, 1.0);
  real_NaN = &cells[2]; init_real(real_NaN, Nan);
  complex_NaN = &cells[10]; init_complex(complex_NaN, Nan, Nan);
  real_infinity = &cells[3]; init_real(real_infinity, Infinity);
  real_minus_infinity = &cells[4]; init_real(real_minus_infinity, -Infinity);
  real_pi = &cells[5]; init_real(real_pi, 3.1415926535897932384626433832795029L);

  arity_not_set = &cells[6]; init_integer(arity_not_set, Closure_Arity_Not_Set);
  max_arity = &cells[7]; init_integer(max_arity, Max_Arity);
  minus_one = &cells[8]; init_integer(minus_one, -1);
  minus_two = &cells[9]; init_integer(minus_two, -2);
  int_zero = &small_ints[0];
  int_one = &small_ints[1];
  int_two = &small_ints[2];
  int_three = &small_ints[3];

  mostfix = make_permanent_integer(s7_Int64_Max);
  leastfix = make_permanent_integer(s7_Int64_Min);
  for (int32_t i = 0; i < Num_Chars; i++) t_number_separator_p[i] = true;
  t_number_separator_p[(uint8_t)'i'] = false;
  t_number_separator_p[(uint8_t)'+'] = false;
  t_number_separator_p[(uint8_t)'-'] = false;
  t_number_separator_p[(uint8_t)'/'] = false;
  t_number_separator_p[(uint8_t)'@'] = false;
  t_number_separator_p[(uint8_t)'.'] = false;
  t_number_separator_p[(uint8_t)'e'] = false;
  t_number_separator_p[(uint8_t)'E'] = false;
}

#define clamp_length(NLen, Len) (((NLen) < (Len)) ? (NLen) : (Len))


/* -------------------------------------------------------------------------------- */
#if (defined(__FreeBSD__)) || ((defined(__linux__)) && (__GLIBC__ >= 2) && (__GLIBC_MINOR__ > 17)) || (defined(__OpenBSD__)) || (defined(__NetBSD__))
  static inline s7_int my_clock(void)
  {
    struct timespec ts;
    clock_gettime(CLOCK_MONOTONIC, &ts);
    /* coarse: 0.057u 0.007s, monotonic: 0.083u 0.007s, clock(): 0.624u 0.372s -- coarse since Linux 2.6.32, glibc > 2.17
     *   FreeBSD has CLOCK_MONOTONIC_FAST in place of COARSE, OpenBSD and netBSD have neither
     *   clock_getres places 1 in tv_nsec in linux, so I assume I divide billion/tv_nsec
     *   MacOSX has clock_get_time, and after Sierra 10.12 has clock_gettime
     *     apparently we include /usr/include/AvailabilityMacros.h, then #if MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
     *   Windows has QueryPerformanceCounter or something
     * maybe just check for POSIX compatibility?
     */
    return(ts.tv_sec * 1000000000 + ts.tv_nsec); /* accumulated into s7_int so this should be ok: s7.h gives it 64 bits */
  }

  static s7_int ticks_per_second(void)
  {
    struct timespec ts;
    clock_getres(CLOCK_MONOTONIC, &ts);
    return((ts.tv_nsec == 0) ? 1000000000 : (1000000000 / ts.tv_nsec));
  }
#else
  #define my_clock clock /* but this is cpu time? */
  #define ticks_per_second() CLOCKS_PER_SEC
#endif

#ifndef GC_TRIGGER_SIZE
  #define GC_TRIGGER_SIZE 64
#endif
#define Gc_Trigger_Size GC_TRIGGER_SIZE

#if s7_Debugging
  static void try_to_call_gc_1(s7_scheme *sc, const char *func, int32_t line);
  #define try_to_call_gc(Sc) try_to_call_gc_1(Sc, __func__, __LINE__)
#else
  static void try_to_call_gc(s7_scheme *sc);
#endif

#define Gc_Stats 1
#define Heap_Stats 2
#define Stack_Stats 4
#define Protected_Objects_Stats 8

#define show_gc_stats(Sc)                ((Sc->gc_stats & Gc_Stats) != 0)
#define show_stack_stats(Sc)             ((Sc->gc_stats & Stack_Stats) != 0)
#define show_heap_stats(Sc)              ((Sc->gc_stats & Heap_Stats) != 0)
#define show_protected_objects_stats(Sc) ((Sc->gc_stats & Protected_Objects_Stats) != 0)


/* new_cell has to include the new cell's type.  In the free list, it is 0 (T_FREE).  If we remove it here,
 *   but then hit some error before setting the type, the GC sweep thinks it is a free cell already and
 *   does not return it to the free list: a memory leak.
 */
#if !s7_Debugging
#define new_cell(Sc, Obj, Type)			\
  do {						\
    if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc); \
    Obj = (*(--(Sc->free_heap_top))); \
    set_full_type(Obj, Type);	      \
    } while (0)

#define new_cell_unchecked(Sc, Obj, Type) do {Obj = (*(--(Sc->free_heap_top))); set_full_type(Obj, Type);} while (0)
  /* since sc->free_heap_trigger is Gc_Trigger_Size above the free heap base, we don't need
   *   to check it repeatedly after the first such check.
   */
#else

#define Cell_Field_Init NULL /* or sc->unused */
#define new_cell(Sc, Obj, Type)						\
  do {									\
    if (Sc->gc_in_progress) fprintf(stderr, "%s[%d]: new_cell during GC\n", __func__, __LINE__); \
    if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc); \
    Obj = (*(--(Sc->free_heap_top)));					\
    Obj->debugger_bits = 0; Obj->gc_func = NULL; Obj->gc_line = 0;	\
    Obj->object.cons.car = Cell_Field_Init; Obj->object.cons.cdr = Cell_Field_Init; \
    Obj->object.cons.opt1 = Cell_Field_Init; Obj->object.cons.o2.opt2 = Cell_Field_Init; Obj->object.cons.o3.opt3 = Cell_Field_Init; \
    set_full_type(Obj, Type);						\
  } while (0)

#define new_cell_unchecked(Sc, Obj, Type)				\
  do {									\
    if (Sc->gc_in_progress) fprintf(stderr, "%s[%d]: new_cell_unchecked during GC\n", __func__, __LINE__); \
    Obj = (*(--(Sc->free_heap_top)));					\
    if (Sc->free_heap_top < Sc->free_heap) {fprintf(stderr, "%s[%d]: free heap exhausted\n", __func__, __LINE__); abort();}\
    Obj->debugger_bits = 0; Obj->gc_func = NULL; Obj->gc_line = 0;	\
    Obj->object.cons.car = Cell_Field_Init; Obj->object.cons.cdr = Cell_Field_Init; \
    Obj->object.cons.opt1 = Cell_Field_Init; Obj->object.cons.o2.opt2 = Cell_Field_Init; Obj->object.cons.o3.opt3 = Cell_Field_Init; \
    set_full_type(Obj, Type);						\
    } while (0)
#endif

/* #define gc_if_at_trigger(Sc) if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc) */


#if WITH_GCC
#define make_integer(Sc, N) \
  ({ s7_int _N_; _N_ = (N); (is_small_int(_N_) ? small_int(_N_) : ({ s7_pointer _I_; new_cell(Sc, _I_, T_INTEGER); set_integer(_I_, _N_); _I_;}) ); })
#define make_integer_unchecked(Sc, N) \
  ({ s7_int _N_; _N_ = (N); (is_small_int(_N_) ? small_int(_N_) : ({ s7_pointer _I_; new_cell_unchecked(Sc, _I_, T_INTEGER); set_integer(_I_, _N_); _I_;}) ); })

#define make_real(Sc, X) ({ s7_pointer _R_; s7_double _N_ = (X); new_cell(Sc, _R_, T_REAL); set_real(_R_, _N_); _R_;})
#define make_real_unchecked(Sc, X) ({ s7_pointer _R_; s7_double _N_ = (X); new_cell_unchecked(Sc, _R_, T_REAL); set_real(_R_, _N_); _R_;})

#if s7_Debugging
#define make_complex_not_0i(Sc, R, I) \
  ({ s7_double _im_; _im_ = (I); if (_im_ == 0.0) fprintf(stderr, "%s[%d]: make_complex i: %f\n", __func__, __LINE__, _im_); \
     ({ s7_pointer _C_; new_cell(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, _im_); _C_;}); })
#else
#define make_complex_not_0i(Sc, R, I) ({ s7_pointer _C_; new_cell(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, I); _C_;})
#endif
#define make_complex(Sc, R, I) \
  ({ s7_double _im_; _im_ = (I); ((_im_ == 0.0) ? make_real(Sc, R) : \
				  ({ s7_pointer _C_; new_cell(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, _im_); _C_;}) ); })
#define make_complex_unchecked(Sc, R, I) \
  ({ s7_double _im_; _im_ = (I); ((_im_ == 0.0) ? make_real_unchecked(Sc, R) : \
				  ({ s7_pointer _C_; new_cell_unchecked(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, _im_); _C_;}) ); })

#define real_to_double(Sc, X, Caller) ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_REAL) ? real(_x_) : s7_number_to_real_with_caller(Sc, _x_, Caller)); })
#define rational_to_double(Sc, X)     ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_INTEGER) ? (s7_double)integer(_x_) : (s7_double)fraction(_x_)); })

#else

#define make_integer(Sc, N)              s7_make_integer(Sc, N)
#define make_integer_unchecked(Sc, N)    s7_make_integer(Sc, N)
#define make_real(Sc, X)                 s7_make_real(Sc, X)
#define make_real_unchecked(Sc, X)       s7_make_real(Sc, X)
#define make_complex(Sc, R, I)           s7_make_complex(Sc, R, I)
#define make_complex_unchecked(Sc, R, I) s7_make_complex(Sc, R, I)
#define make_complex_not_0i(Sc, R, I)    s7_make_complex(Sc, R, I)
#define real_to_double(Sc, X, Caller)    s7_number_to_real_with_caller(Sc, X, Caller)
#define rational_to_double(Sc, X)        s7_number_to_real(Sc, X)
#endif


/* --------------------------------------------------------------------------------
 * local versions of some standard C library functions
 * timing tests involving these are very hard to interpret, local_memset is faster using s7_int than int32_t
 * but don't replace local_memset and memclr64 with memset! tbig -> 208! (if aligned, other timings go up or down by 1-2%, no obvious winner)
 *   [tmac +55, tstar +40, concordance +40, tlist -30, tgc -40, tset -40, tform -20, tmv +0.2, tbig +63.1!]
 */

static void local_memset(void *s, uint8_t val, size_t n)
{
  uint8_t *s2;
#if s7_Aligned
  s2 = (uint8_t *)s;
#else
#if (defined(__x86_64__) || defined(__i386__))
  if (n >= 8)
    {
      s7_int *s1 = (s7_int *)s;
      size_t n8 = n >> 3;
      s7_int ival = val | (val << 8) | (val << 16) | (((s7_uint)val) << 24); /* s7_uint casts make gcc/clang/fsanitize happy */
      ival = (((s7_uint)ival) << 32) | ival;
      if ((n8 & 0x3) == 0)
	while (n8 > 0) {Loop_4(*s1++ = ival); n8 -= 4;}
      else do {*s1++ = ival;} while (--n8 > 0);
      n &= 7;
      s2 = (uint8_t *)s1;
    }
  else s2 = (uint8_t *)s;
#else
  s2 = (uint8_t *)s;
#endif
#endif
  while (n > 0)
    {
      *s2++ = val;
      n--;
    }
}

static inline s7_int safe_strlen(const char *str) /* this is safer than strlen, and slightly faster */
{
  const char *tmp = str;
  /* if ((!tmp) || (!*tmp)) return(0); */ /* removing *tmp check doesn't save any time, but looks better? */
  if (!tmp) return(0);
  for (; *tmp; ++tmp);
  return(tmp - str);
}

static char *copy_string_with_length(const char *str, s7_int len)
{
  char *newstr;
#if s7_Debugging
  if ((len <= 0) || (!str))
    {fprintf(stderr, "%s[%d]: len: %" ld64 ", str: %s\n", __func__, __LINE__, len, str); abort();}
#endif
  if (len > (1LL << 48)) return(NULL); /* squelch an idiotic warning */
  newstr = (char *)Malloc(len + 1);
  memcpy((void *)newstr, (const void *)str, len); /* we check len != 0 above -- 24-Jan-22 */
  newstr[len] = '\0';
  return(newstr);
}

static char *copy_string(const char *str) {return(copy_string_with_length(str, safe_strlen(str)));}

#define local_strcmp(S1, S2) (strcmp(S1, S2) == 0)
#define c_strings_are_equal(Str1, Str2) (local_strcmp(Str1, Str2))  /* scheme strings can have embedded nulls */

static bool safe_strcmp(const char *s1, const char *s2)
{
  if ((!s1) || (!s2)) return(s1 == s2);
  return(local_strcmp(s1, s2));
}

static bool local_strncmp(const char *s1, const char *s2, size_t n) /* not strncmp because scheme strings can have embedded nulls */
{
#if ((!s7_Aligned) && (defined(__x86_64__) || defined(__i386__)))   /* unaligned accesses are safe on i386 hardware, sez everyone */
  if (n >= 8)
    {
      size_t n8 = n >> 3;
      s7_int *is1 = (s7_int *)s1, *is2 = (s7_int *)s2;
      do {if (*is1++ != *is2++) return(false);} while (--n8 > 0); /* in tbig Loop_4 is slower? */
      s1 = (const char *)is1;
      s2 = (const char *)is2;
      n &= 7;
    }
#endif
  while (n > 0)
    {
      if (*s1++ != *s2++) return(false); /* 45B in tbig!! v-big38 */
      n--;
    }
  return(true);
}

#define strings_are_equal_with_length(Str1, Str2, Len) (local_strncmp(Str1, Str2, Len))

static Sentinel size_t catstrs(char *dst, size_t len, ...) /* NULL-terminated arg list */
{
  const char *dend = (const char *)(dst + len - 1); /* -1 for null at end? */
  char *d = dst;
  va_list ap;
  while ((*d) && (d < dend)) d++; /* stop at NULL or end-of-buffer */
  va_start(ap, len);
  for (const char *s = va_arg(ap, const char *); s != NULL; s = va_arg(ap, const char *))
    while ((*s) && (d < dend)) {*d++ = *s++;}
  *d = '\0';
  va_end (ap);
  return(d - dst);
}

static Sentinel size_t catstrs_direct(char *dst, const char *str1, ...)
{ /* NULL-terminated arg list, dst is destination only (assumed empty), all args known to fit in dst */
  char *d = dst;
  va_list ap;
  va_start(ap, str1);
  for (const char *s = str1; s != NULL; s = va_arg(ap, const char *))
    while (*s) {*d++ = *s++;}
  *d = '\0';
  va_end (ap);
  return(d - dst);
}

static char *pos_int_to_str(s7_scheme *sc, s7_int num, s7_int *len, char endc)
{
  char *p = (char *)(sc->int_to_str3 + Int_To_Str_Size - 1); /* str[31] */
  char *op = p;
  if ((s7_Debugging) && (num < 0)) {fprintf(stderr, "%s[%d]: num=%" ld64, __func__, __LINE__, num); abort();}
  *p-- = '\0';
  if (endc != '\0') *p-- = endc;
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  (*len) = op - p;           /* this includes the trailing #\null */
  return((char *)(p + 1));
}

static char *pos_int_to_str_direct(s7_scheme *sc, s7_int num)
{
  char *p = (char *)(sc->int_to_str4 + Int_To_Str_Size - 1);
  if ((s7_Debugging) && (num < 0)) {fprintf(stderr, "%s[%d]: num=%" ld64, __func__, __LINE__, num); abort();}
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  return((char *)(p + 1));
}

static char *pos_int_to_str_direct_1(s7_scheme *sc, s7_int num)
{
  char *p = (char *)(sc->int_to_str5 + Int_To_Str_Size - 1);
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  return((char *)(p + 1));
}

#if WITH_GCC
  #if s7_Debugging
    static s7_pointer lookup_1(s7_scheme *sc, const s7_pointer symbol);
    #define lookup(Sc, Sym) check_null_sym(Sc, lookup_1(Sc, T_Sym(Sym)), Sym, __LINE__, __func__)
    static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int32_t line, const char *func);
    #define lookup_unexamined(Sc, Sym) lookup_1(Sc, T_Sym(Sym))
    #define lookup_checked(Sc, Sym) ({s7_pointer _x_; _x_ = lookup_1(Sc, T_Sym(Sym)); ((_x_) ? _x_ : unbound_variable(Sc, T_Sym(Sym)));})
  #else
    static inline s7_pointer lookup(s7_scheme *sc, const s7_pointer symbol);
    #define lookup_unexamined(Sc, Sym) lookup(Sc, Sym)
    #define lookup_checked(Sc, Sym) ({s7_pointer _x_; _x_ = lookup(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #endif
#else
  static inline s7_pointer lookup(s7_scheme *sc, const s7_pointer symbol);
  #define lookup_unexamined(Sc, Sym) s7_symbol_value(Sc, Sym)  /* changed 3-Nov-22 -- we're using lookup_unexamined below to avoid the unbound_variable check */
  #define lookup_checked(Sc, Sym) lookup(Sc, Sym)
#endif
static s7_pointer symbol_to_local_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer let);


/* ---------------- evaluator ops ---------------- */
/* C=constant, S=symbol, A=fx-callable, Q=quote, N=any number of next >= 1, FX=list of A's, P=parlous?, O=one form, M=multiform */
enum {OP_UNOPT, OP_GC_PROTECT, /* must be an even number of ops here, op_gc_protect used below as lower boundary marker */

      OP_SAFE_C_NC, HOP_SAFE_C_NC, OP_SAFE_C_S, HOP_SAFE_C_S,
      OP_SAFE_C_SS, HOP_SAFE_C_SS, OP_SAFE_C_SC, HOP_SAFE_C_SC, OP_SAFE_C_CS, HOP_SAFE_C_CS, OP_SAFE_C_CQ, HOP_SAFE_C_CQ,
      OP_SAFE_C_SSS, HOP_SAFE_C_SSS, OP_SAFE_C_SCS, HOP_SAFE_C_SCS, OP_SAFE_C_SSC, HOP_SAFE_C_SSC, OP_SAFE_C_CSS, HOP_SAFE_C_CSS,
      OP_SAFE_C_SCC, HOP_SAFE_C_SCC, OP_SAFE_C_CSC, HOP_SAFE_C_CSC, OP_SAFE_C_CCS, HOP_SAFE_C_CCS,
      OP_SAFE_C_NS, HOP_SAFE_C_NS, OP_SAFE_C_opNCq, HOP_SAFE_C_opNCq, OP_SAFE_C_opSq, HOP_SAFE_C_opSq,
      OP_SAFE_C_opSSq, HOP_SAFE_C_opSSq, OP_SAFE_C_opSCq, HOP_SAFE_C_opSCq,
      OP_SAFE_C_opCSq, HOP_SAFE_C_opCSq, OP_SAFE_C_S_opSq, HOP_SAFE_C_S_opSq,
      OP_SAFE_C_C_opSCq, HOP_SAFE_C_C_opSCq, OP_SAFE_C_S_opSCq, HOP_SAFE_C_S_opSCq, OP_SAFE_C_S_opCSq, HOP_SAFE_C_S_opCSq,
      OP_SAFE_C_opSq_S, HOP_SAFE_C_opSq_S, OP_SAFE_C_opSq_C, HOP_SAFE_C_opSq_C,
      OP_SAFE_C_opSq_opSq, HOP_SAFE_C_opSq_opSq, OP_SAFE_C_S_opSSq, HOP_SAFE_C_S_opSSq, OP_SAFE_C_C_opSq, HOP_SAFE_C_C_opSq,
      OP_SAFE_C_opCSq_C, HOP_SAFE_C_opCSq_C, OP_SAFE_C_opSSq_C, HOP_SAFE_C_opSSq_C, OP_SAFE_C_C_opSSq, HOP_SAFE_C_C_opSSq,
      OP_SAFE_C_opSSq_opSSq, HOP_SAFE_C_opSSq_opSSq, OP_SAFE_C_opSSq_opSq, HOP_SAFE_C_opSSq_opSq, OP_SAFE_C_opSq_opSSq, HOP_SAFE_C_opSq_opSSq,
      OP_SAFE_C_opSSq_S, HOP_SAFE_C_opSSq_S, OP_SAFE_C_opCSq_S, HOP_SAFE_C_opCSq_S, OP_SAFE_C_opSCq_C, HOP_SAFE_C_opSCq_C,
      OP_SAFE_C_op_opSSqq_S, HOP_SAFE_C_op_opSSqq_S, OP_SAFE_C_op_opSqq, HOP_SAFE_C_op_opSqq,
      OP_SAFE_C_op_S_opSqq, HOP_SAFE_C_op_S_opSqq, OP_SAFE_C_op_opSq_Sq, HOP_SAFE_C_op_opSq_Sq, OP_SAFE_C_opSq_CS, HOP_SAFE_C_opSq_CS,

      OP_SAFE_C_A, HOP_SAFE_C_A, OP_SAFE_C_AA, HOP_SAFE_C_AA, OP_SAFE_C_SA, HOP_SAFE_C_SA, OP_SAFE_C_AS, HOP_SAFE_C_AS,
      OP_SAFE_C_CA, HOP_SAFE_C_CA, OP_SAFE_C_AC, HOP_SAFE_C_AC, OP_SAFE_C_AAA, HOP_SAFE_C_AAA, OP_SAFE_C_4A, HOP_SAFE_C_4A,
      OP_SAFE_C_NA, HOP_SAFE_C_NA, OP_SAFE_C_ALL_CA, HOP_SAFE_C_ALL_CA,
      OP_SAFE_C_SSA, HOP_SAFE_C_SSA, OP_SAFE_C_SAS, HOP_SAFE_C_SAS, OP_SAFE_C_SAA, HOP_SAFE_C_SAA,
      OP_SAFE_C_CSA, HOP_SAFE_C_CSA, OP_SAFE_C_SCA, HOP_SAFE_C_SCA, OP_SAFE_C_ASS, HOP_SAFE_C_ASS,
      OP_SAFE_C_CAC, HOP_SAFE_C_CAC, OP_SAFE_C_AGG, HOP_SAFE_C_AGG,
      OP_SAFE_C_opAq, HOP_SAFE_C_opAq, OP_SAFE_C_opAAq, HOP_SAFE_C_opAAq, OP_SAFE_C_opAAAq, HOP_SAFE_C_opAAAq,
      OP_SAFE_C_S_opAq, HOP_SAFE_C_S_opAq, OP_SAFE_C_opAq_S, HOP_SAFE_C_opAq_S, OP_SAFE_C_S_opAAq, HOP_SAFE_C_S_opAAq,
      OP_SAFE_C_STAR, HOP_SAFE_C_STAR, OP_SAFE_C_STAR_A, HOP_SAFE_C_STAR_A,
      OP_SAFE_C_STAR_AA, HOP_SAFE_C_STAR_AA, OP_SAFE_C_STAR_NA, HOP_SAFE_C_STAR_NA,

      OP_SAFE_C_P, HOP_SAFE_C_P, OP_SAFE_C_PP, HOP_SAFE_C_PP, OP_SAFE_C_FF, HOP_SAFE_C_FF, OP_SAFE_C_SP, HOP_SAFE_C_SP,
      OP_SAFE_C_CP, HOP_SAFE_C_CP, OP_SAFE_C_AP, HOP_SAFE_C_AP, OP_SAFE_C_PA, HOP_SAFE_C_PA, OP_SAFE_C_PS, HOP_SAFE_C_PS,
      OP_SAFE_C_PC, HOP_SAFE_C_PC, OP_SAFE_C_SPS, HOP_SAFE_C_SPS, OP_SAFE_C_AAP, HOP_SAFE_C_AAP,
      OP_SAFE_C_3P, HOP_SAFE_C_3P, OP_ANY_C_NP, HOP_ANY_C_NP,

      OP_THUNK, HOP_THUNK, OP_THUNK_O, HOP_THUNK_O, OP_THUNK_C, HOP_THUNK_C, OP_THUNK_ANY, HOP_THUNK_ANY,
      OP_SAFE_THUNK, HOP_SAFE_THUNK, OP_SAFE_THUNK_A, HOP_SAFE_THUNK_A, OP_SAFE_THUNK_ANY, HOP_SAFE_THUNK_ANY,

      OP_CLOSURE_S, HOP_CLOSURE_S, OP_CLOSURE_S_O, HOP_CLOSURE_S_O,
      OP_CLOSURE_A, HOP_CLOSURE_A, OP_CLOSURE_A_O, HOP_CLOSURE_A_O, OP_CLOSURE_P, HOP_CLOSURE_P,
      OP_CLOSURE_AP, HOP_CLOSURE_AP, OP_CLOSURE_PA, HOP_CLOSURE_PA, OP_CLOSURE_PP, HOP_CLOSURE_PP,
      OP_CLOSURE_FA, HOP_CLOSURE_FA, OP_CLOSURE_SS, HOP_CLOSURE_SS, OP_CLOSURE_SS_O, HOP_CLOSURE_SS_O,
      OP_CLOSURE_SC, HOP_CLOSURE_SC, OP_CLOSURE_SC_O, HOP_CLOSURE_SC_O,
      OP_CLOSURE_3S, HOP_CLOSURE_3S, OP_CLOSURE_3S_O, HOP_CLOSURE_3S_O, OP_CLOSURE_4S, HOP_CLOSURE_4S, OP_CLOSURE_4S_O, HOP_CLOSURE_4S_O, OP_CLOSURE_5S, HOP_CLOSURE_5S,
      OP_CLOSURE_AA, HOP_CLOSURE_AA, OP_CLOSURE_AA_O, HOP_CLOSURE_AA_O, OP_CLOSURE_3A, HOP_CLOSURE_3A, OP_CLOSURE_4A, HOP_CLOSURE_4A,
      OP_CLOSURE_NA, HOP_CLOSURE_NA, OP_CLOSURE_ASS, HOP_CLOSURE_ASS, OP_CLOSURE_SAS, HOP_CLOSURE_SAS ,OP_CLOSURE_AAS, HOP_CLOSURE_AAS,
      OP_CLOSURE_SAA, HOP_CLOSURE_SAA, OP_CLOSURE_ASA, HOP_CLOSURE_ASA, OP_CLOSURE_NS, HOP_CLOSURE_NS,

      OP_SAFE_CLOSURE_S, HOP_SAFE_CLOSURE_S, OP_SAFE_CLOSURE_S_O, HOP_SAFE_CLOSURE_S_O,
      OP_SAFE_CLOSURE_S_A, HOP_SAFE_CLOSURE_S_A, OP_SAFE_CLOSURE_S_TO_S, HOP_SAFE_CLOSURE_S_TO_S, OP_SAFE_CLOSURE_S_TO_SC, HOP_SAFE_CLOSURE_S_TO_SC,
      OP_SAFE_CLOSURE_P, HOP_SAFE_CLOSURE_P, OP_SAFE_CLOSURE_P_A, HOP_SAFE_CLOSURE_P_A,
      OP_SAFE_CLOSURE_AP, HOP_SAFE_CLOSURE_AP, OP_SAFE_CLOSURE_PA, HOP_SAFE_CLOSURE_PA, OP_SAFE_CLOSURE_PP, HOP_SAFE_CLOSURE_PP,
      OP_SAFE_CLOSURE_A, HOP_SAFE_CLOSURE_A, OP_SAFE_CLOSURE_A_O, HOP_SAFE_CLOSURE_A_O, OP_SAFE_CLOSURE_A_A, HOP_SAFE_CLOSURE_A_A,
      OP_SAFE_CLOSURE_A_TO_SC, HOP_SAFE_CLOSURE_A_TO_SC,
      OP_SAFE_CLOSURE_SS, HOP_SAFE_CLOSURE_SS, OP_SAFE_CLOSURE_SS_O, HOP_SAFE_CLOSURE_SS_O, OP_SAFE_CLOSURE_SS_A, HOP_SAFE_CLOSURE_SS_A,
      OP_SAFE_CLOSURE_SC, HOP_SAFE_CLOSURE_SC, OP_SAFE_CLOSURE_SC_O, HOP_SAFE_CLOSURE_SC_O,
      OP_SAFE_CLOSURE_AA, HOP_SAFE_CLOSURE_AA, OP_SAFE_CLOSURE_AA_O, HOP_SAFE_CLOSURE_AA_O, OP_SAFE_CLOSURE_AA_A, HOP_SAFE_CLOSURE_AA_A,
      OP_SAFE_CLOSURE_SAA, HOP_SAFE_CLOSURE_SAA, OP_SAFE_CLOSURE_SSA, HOP_SAFE_CLOSURE_SSA,
      OP_SAFE_CLOSURE_AGG, HOP_SAFE_CLOSURE_AGG, OP_SAFE_CLOSURE_3A, HOP_SAFE_CLOSURE_3A, OP_SAFE_CLOSURE_NA, HOP_SAFE_CLOSURE_NA,
      OP_SAFE_CLOSURE_3S, HOP_SAFE_CLOSURE_3S, OP_SAFE_CLOSURE_NS, HOP_SAFE_CLOSURE_NS, /* safe_closure_4s gained very little */
      OP_SAFE_CLOSURE_3S_A, HOP_SAFE_CLOSURE_3S_A,

      OP_ANY_CLOSURE_3P, HOP_ANY_CLOSURE_3P, OP_ANY_CLOSURE_4P, HOP_ANY_CLOSURE_4P, OP_ANY_CLOSURE_NP, HOP_ANY_CLOSURE_NP,
      OP_ANY_CLOSURE_SYM, HOP_ANY_CLOSURE_SYM, OP_ANY_CLOSURE_A_SYM, HOP_ANY_CLOSURE_A_SYM,

      OP_CLOSURE_STAR_A, HOP_CLOSURE_STAR_A, OP_CLOSURE_STAR_NA, HOP_CLOSURE_STAR_NA,
      OP_SAFE_CLOSURE_STAR_A, HOP_SAFE_CLOSURE_STAR_A, OP_SAFE_CLOSURE_STAR_AA, HOP_SAFE_CLOSURE_STAR_AA,
      OP_SAFE_CLOSURE_STAR_AA_O, HOP_SAFE_CLOSURE_STAR_AA_O, OP_SAFE_CLOSURE_STAR_A1, HOP_SAFE_CLOSURE_STAR_A1,
      OP_SAFE_CLOSURE_STAR_KA, HOP_SAFE_CLOSURE_STAR_KA, OP_CLOSURE_STAR_KA, HOP_CLOSURE_STAR_KA, OP_SAFE_CLOSURE_STAR_3A, HOP_SAFE_CLOSURE_STAR_3A,
      OP_SAFE_CLOSURE_STAR_NA, HOP_SAFE_CLOSURE_STAR_NA, OP_SAFE_CLOSURE_STAR_NA_0, HOP_SAFE_CLOSURE_STAR_NA_0,
      OP_SAFE_CLOSURE_STAR_NA_1, HOP_SAFE_CLOSURE_STAR_NA_1, OP_SAFE_CLOSURE_STAR_NA_2, HOP_SAFE_CLOSURE_STAR_NA_2,

      OP_C_SS, HOP_C_SS, OP_C_S, HOP_C_S, OP_C_SC, HOP_C_SC, OP_READ_S, HOP_READ_S, OP_C_P, HOP_C_P, OP_C_AP, HOP_C_AP,
      OP_C_A, HOP_C_A, OP_C_AA, HOP_C_AA, OP_C, HOP_C, OP_C_NC, HOP_C_NC, OP_C_NA, HOP_C_NA,

      OP_CL_S, HOP_CL_S, OP_CL_SS, HOP_CL_SS, OP_CL_A, HOP_CL_A, OP_CL_AA, HOP_CL_AA,
      OP_CL_NA, HOP_CL_NA, OP_CL_FA, HOP_CL_FA, OP_CL_SAS, HOP_CL_SAS,
      /* end of h_opts */

      OP_APPLY_SS, OP_APPLY_SA, OP_APPLY_SL, OP_MACRO_D, OP_MACRO_STAR_D,
      OP_WITH_IO, OP_WITH_IO_1, OP_WITH_OUTPUT_TO_STRING, OP_WITH_IO_C, OP_CALL_WITH_OUTPUT_STRING,
      OP_S, OP_S_G, OP_S_A, OP_S_AA, OP_A_A, OP_A_AA, OP_A_SC, OP_P_S, OP_P_S_1, OP_MAP_FOR_EACH_FA, OP_MAP_FOR_EACH_FAA,
      OP_F, OP_F_A, OP_F_AA, OP_F_NP, OP_F_NP_1,

      OP_IMPLICIT_GOTO, OP_IMPLICIT_GOTO_A, OP_IMPLICIT_CONTINUATION_A, OP_IMPLICIT_ITERATE,
      OP_IMPLICIT_VECTOR_REF_A, OP_IMPLICIT_VECTOR_REF_AA,
      OP_IMPLICIT_STRING_REF_A, OP_IMPLICIT_C_OBJECT_REF_A, OP_IMPLICIT_PAIR_REF_A, OP_IMPLICIT_PAIR_REF_AA,
      OP_IMPLICIT_HASH_TABLE_REF_A, OP_IMPLICIT_HASH_TABLE_REF_AA,
      OP_IMPLICIT_LET_REF_C, OP_IMPLICIT_LET_REF_A, OP_IMPLICIT_STARLET_REF_S, OP_IMPLICIT_STARLET_SET_S,
      OP_UNKNOWN, OP_UNKNOWN_NS, OP_UNKNOWN_NA, OP_UNKNOWN_S, OP_UNKNOWN_GG, OP_UNKNOWN_A, OP_UNKNOWN_AA, OP_UNKNOWN_NP,

      OP_SYMBOL, OP_CONSTANT, OP_PAIR_SYM, OP_PAIR_PAIR, OP_PAIR_PAIR_S, OP_PAIR_ANY, HOP_HASH_TABLE_INCREMENT, OP_CLEAR_OPTS,

      OP_READ_INTERNAL, OP_EVAL, OP_EVAL_ARGS, OP_EVAL_ARGS1, OP_EVAL_ARGS2, OP_EVAL_ARGS3, OP_EVAL_ARGS4, OP_EVAL_ARGS5,
      OP_EVAL_SET1_NO_MV, OP_EVAL_SET2, OP_EVAL_SET2_MV, OP_EVAL_SET2_NO_MV, OP_EVAL_SET3, OP_EVAL_SET3_MV, OP_EVAL_SET3_NO_MV,
      OP_APPLY, OP_EVAL_MACRO, OP_LAMBDA, OP_QUOTE, OP_QUOTE_UNCHECKED, OP_MACROEXPAND, OP_CALL_CC, OP_CALL_WITH_EXIT, OP_CALL_WITH_EXIT_O,
      OP_C_CATCH, OP_C_CATCH_ALL, OP_C_CATCH_ALL_O, OP_C_CATCH_ALL_A,

      OP_DEFINE, OP_DEFINE1, OP_BEGIN, OP_BEGIN_HOOK, OP_BEGIN_NO_HOOK, OP_BEGIN_UNCHECKED, OP_BEGIN_2_UNCHECKED, OP_BEGIN_NA, OP_BEGIN_AA,
      OP_IF, OP_IF1, OP_WHEN, OP_UNLESS, OP_SET, OP_SET1, OP_SET2,
      OP_LET, OP_LET1, OP_LET_STAR, OP_LET_STAR1, OP_LET_STAR2, OP_LET_STAR_SHADOWED,
      OP_LETREC, OP_LETREC1, OP_LETREC_STAR, OP_LETREC_STAR1,
      OP_LET_TEMPORARILY, OP_LET_TEMP_UNCHECKED, OP_LET_TEMP_INIT1, OP_LET_TEMP_INIT2, OP_LET_TEMP_DONE, OP_LET_TEMP_DONE1,
      OP_LET_TEMP_S7, OP_LET_TEMP_NA, OP_LET_TEMP_A, OP_LET_TEMP_SETTER, OP_LET_TEMP_UNWIND, OP_LET_TEMP_S7_UNWIND, OP_LET_TEMP_SETTER_UNWIND,
      OP_LET_TEMP_A_A, OP_LET_TEMP_S7_OPENLETS, OP_LET_TEMP_S7_OPENLETS_UNWIND,
      OP_COND, OP_COND1, OP_FEED_TO_1, OP_COND_SIMPLE, OP_COND1_SIMPLE, OP_COND_SIMPLE_O, OP_COND1_SIMPLE_O,
      OP_AND, OP_OR,
      OP_DEFINE_MACRO, OP_DEFINE_MACRO_STAR, OP_DEFINE_EXPANSION, OP_DEFINE_EXPANSION_STAR, OP_MACRO, OP_MACRO_STAR,
      OP_CASE,
      OP_READ_LIST, OP_READ_NEXT, OP_READ_DOT, OP_READ_QUOTE,
      OP_READ_QUASIQUOTE, OP_READ_UNQUOTE, OP_READ_APPLY_VALUES,
      OP_READ_VECTOR, OP_READ_BYTE_VECTOR, OP_READ_INT_VECTOR, OP_READ_FLOAT_VECTOR, OP_READ_COMPLEX_VECTOR, OP_READ_DONE,
      OP_LOAD_RETURN_IF_EOF, OP_LOAD_CLOSE_AND_POP_IF_EOF, OP_EVAL_DONE, OP_SPLICE_VALUES, OP_NO_VALUES,
      OP_CATCH, OP_DYNAMIC_WIND, OP_DYNAMIC_UNWIND, OP_DYNAMIC_UNWIND_PROFILE, OP_PROFILE_IN,
      OP_DEFINE_CONSTANT, OP_DEFINE_CONSTANT1,
      OP_DO, OP_DO_END, OP_DO_END1, OP_DO_STEP, OP_DO_STEP2, OP_DO_INIT,
      OP_DEFINE_STAR, OP_LAMBDA_STAR, OP_LAMBDA_STAR_DEFAULT, OP_ERROR_QUIT, OP_UNWIND_INPUT, OP_UNWIND_OUTPUT, OP_ERROR_HOOK_QUIT,
      OP_WITH_LET, OP_WITH_LET1, OP_WITH_LET_UNCHECKED, OP_WITH_LET_S,
      OP_WITH_BAFFLE, OP_WITH_BAFFLE_UNCHECKED, OP_EXPANSION,
      OP_FOR_EACH, OP_FOR_EACH_1, OP_FOR_EACH_2, OP_FOR_EACH_3,
      OP_MAP, OP_MAP_1, OP_MAP_2, OP_MAP_GATHER, OP_MAP_GATHER_1, OP_MAP_GATHER_2, OP_MAP_GATHER_3, OP_MAP_UNWIND,
      OP_BARRIER, OP_DEACTIVATE_GOTO,
      OP_DEFINE_BACRO, OP_DEFINE_BACRO_STAR, OP_BACRO, OP_BACRO_STAR,
      OP_GET_OUTPUT_STRING,
      OP_SORT, OP_SORT1, OP_SORT2, OP_SORT3, OP_SORT_PAIR_END, OP_SORT_VECTOR_END, OP_SORT_STRING_END,
      OP_EVAL_STRING,
      OP_MEMBER_IF, OP_ASSOC_IF, OP_MEMBER_IF1, OP_ASSOC_IF1,
      OP_LAMBDA_UNCHECKED, OP_LET_UNCHECKED, OP_CATCH_1, OP_CATCH_2, OP_CATCH_ALL,

      OP_SET_UNCHECKED, OP_SET_S_C, OP_SET_S_S, OP_SET_S_P, OP_SET_S_A,
      OP_SET_NORMAL, OP_SET_opSq_A, OP_SET_opSAq_A, OP_SET_opSAq_P, OP_SET_opSAq_P_1, OP_SET_opSAAq_A, OP_SET_opSAAq_P, OP_SET_opSAAq_P_1,
      OP_SET_FROM_SETTER, OP_SET_FROM_LET_TEMP, OP_SET_SP_1,
      OP_INCREMENT_BY_1, OP_DECREMENT_BY_1, OP_INCREMENT_SS, OP_INCREMENT_SA, OP_INCREMENT_SAA, OP_SET_CONS,

      OP_LETREC_UNCHECKED, OP_LETREC_STAR_UNCHECKED, OP_COND_UNCHECKED,
      OP_LAMBDA_STAR_UNCHECKED, OP_DO_UNCHECKED, OP_DEFINE_UNCHECKED, OP_DEFINE_STAR_UNCHECKED, OP_DEFINE_FUNCHECKED, OP_DEFINE_CONSTANT_UNCHECKED,
      OP_DEFINE_WITH_SETTER,

      OP_LET_NO_VARS, OP_NAMED_LET, OP_NAMED_LET_NO_VARS, OP_NAMED_LET_A, OP_NAMED_LET_AA, OP_NAMED_LET_NA, OP_NAMED_LET_STAR,
      OP_LET_NA_OLD, OP_LET_NA_NEW, OP_LET_2A_OLD, OP_LET_2A_NEW, OP_LET_3A_OLD, OP_LET_3A_NEW,
      OP_LET_opaSSq_OLD, OP_LET_opaSSq_NEW, OP_LET_ONE_OLD, OP_LET_ONE_NEW, OP_LET_ONE_P_OLD, OP_LET_ONE_P_NEW,
      OP_LET_ONE_OLD_1, OP_LET_ONE_NEW_1, OP_LET_ONE_P_OLD_1, OP_LET_ONE_P_NEW_1,
      OP_LET_A_OLD, OP_LET_A_NEW, OP_LET_A_P_OLD, OP_LET_A_P_NEW,
      OP_LET_A_A_OLD, OP_LET_A_A_NEW, OP_LET_A_NA_OLD, OP_LET_A_NA_NEW, OP_LET_A_OLD_2, OP_LET_A_NEW_2,
      OP_LET_STAR_NA, OP_LET_STAR_NA_A,

      OP_CASE_A_E_S, OP_CASE_A_I_S, OP_CASE_A_G_S, OP_CASE_A_E_G, OP_CASE_A_G_G, OP_CASE_A_S_G,
      OP_CASE_P_E_S, OP_CASE_P_I_S, OP_CASE_P_G_S, OP_CASE_P_E_G, OP_CASE_P_G_G,
      OP_CASE_E_S, OP_CASE_I_S, OP_CASE_G_S, OP_CASE_E_G, OP_CASE_G_G,
      OP_CASE_A_I_S_A, OP_CASE_A_E_S_A, OP_CASE_A_G_S_A, OP_CASE_A_S_G_A,

      OP_IF_UNCHECKED, OP_AND_P, OP_AND_P1, OP_AND_AP, OP_AND_PAIR_P,
      OP_AND_SAFE_P1, OP_AND_SAFE_P2, OP_AND_SAFE_P3, OP_AND_SAFE_P_REST, OP_AND_2A, OP_AND_3A, OP_AND_N, OP_AND_S_2,
      OP_OR_P, OP_OR_P1, OP_OR_AP, OP_OR_2A, OP_OR_3A, OP_OR_N, OP_OR_S_2, OP_OR_S_TYPE_2,
      OP_WHEN_S, OP_WHEN_A, OP_WHEN_P, OP_WHEN_AND_AP, OP_WHEN_AND_2A, OP_WHEN_AND_3A, OP_UNLESS_S, OP_UNLESS_A, OP_UNLESS_P,

      OP_IF_A_C_C, OP_IF_A_A, OP_IF_A_A_A, OP_IF_S_A_A, OP_IF_AND2_S_A, OP_IF_NOT_A_A, OP_IF_NOT_A_A_A,
      OP_IF_B_A, OP_IF_B_P, OP_IF_B_R, OP_IF_B_A_P, OP_IF_B_P_A, OP_IF_B_P_P, OP_IF_B_N_N,
      OP_IF_A_A_P, OP_IF_A_P_A, OP_IF_S_P_A, OP_IF_S_A_P, OP_IF_S_P, OP_IF_S_P_P, OP_IF_S_R, OP_IF_S_N, OP_IF_S_N_N,
      OP_IF_opSq_P, OP_IF_opSq_P_P, OP_IF_opSq_R, OP_IF_opSq_N, OP_IF_opSq_N_N,
      OP_IF_IS_TYPE_S_P, OP_IF_IS_TYPE_S_P_P, OP_IF_IS_TYPE_S_R, OP_IF_IS_TYPE_S_N, OP_IF_IS_TYPE_S_N_N, OP_IF_IS_TYPE_S_P_A, OP_IF_IS_TYPE_S_A_A, OP_IF_IS_TYPE_S_A_P,
      OP_IF_A_P, OP_IF_A_P_P, OP_IF_A_R, OP_IF_A_N, OP_IF_A_N_N,
      OP_IF_AND2_P, OP_IF_AND2_P_P, OP_IF_AND2_R, OP_IF_AND2_N, OP_IF_AND2_N_N,
      OP_IF_AND3_P, OP_IF_AND3_P_P, OP_IF_AND3_R, OP_IF_AND3_N, OP_IF_AND3_N_N,  /* or3 got few hits */
      OP_IF_P_P, OP_IF_P_P_P, OP_IF_P_R, OP_IF_P_N, OP_IF_P_N_N,
      OP_IF_ANDP_P, OP_IF_ANDP_P_P, OP_IF_ANDP_R, OP_IF_ANDP_N, OP_IF_ANDP_N_N,
      OP_IF_ORP_P, OP_IF_ORP_P_P, OP_IF_ORP_R, OP_IF_ORP_N, OP_IF_ORP_N_N,
      OP_IF_OR2_P, OP_IF_OR2_P_P, OP_IF_OR2_R, OP_IF_OR2_N, OP_IF_OR2_N_N,
      OP_IF_PP, OP_IF_PPP, OP_IF_PN, OP_IF_PR, OP_IF_PRR, OP_WHEN_PP, OP_UNLESS_PP,

      OP_COND_NA_NA, OP_COND_NA_NP, OP_COND_NA_NP_1, OP_COND_NA_2E, OP_COND_NA_3E, OP_COND_NA_NP_O,
      OP_COND_FEED, OP_COND_FEED_1,

      OP_SIMPLE_DO, OP_SIMPLE_DO_STEP, OP_SAFE_DOTIMES, OP_SAFE_DOTIMES_STEP, OP_SAFE_DOTIMES_STEP_O,
      OP_SAFE_DO, OP_SAFE_DO_STEP, OP_DOX, OP_DOX_STEP, OP_DOX_STEP_O, OP_DOX_NO_BODY, OP_DOX_PENDING_NO_BODY, OP_DOX_INIT,
      OP_DOTIMES_P, OP_DOTIMES_STEP_O,
      OP_DO_NO_VARS, OP_DO_NO_VARS_NO_OPT, OP_DO_NO_VARS_NO_OPT_1,
      OP_DO_NO_BODY_NA_VARS, OP_DO_NO_BODY_NA_VARS_STEP, OP_DO_NO_BODY_NA_VARS_STEP_1,

      OP_SAFE_C_P_1, OP_SAFE_C_PP_1, OP_SAFE_C_PP_3_MV, OP_SAFE_C_PP_5,
      OP_SAFE_C_3P_1, OP_SAFE_C_3P_2, OP_SAFE_C_3P_3, OP_SAFE_C_3P_1_MV, OP_SAFE_C_3P_2_MV, OP_SAFE_C_3P_3_MV,
      OP_SAFE_C_SP_1, OP_SAFE_CONS_SP_1, OP_SAFE_ADD_SP_1, OP_SAFE_MULTIPLY_SP_1, OP_SAFE_C_PS_1, OP_SAFE_C_PC_1,
      OP_EVAL_MACRO_MV, OP_MACROEXPAND_1, OP_APPLY_LAMBDA,
      OP_ANY_C_NP_1, OP_ANY_C_NP_MV, OP_SAFE_C_SPS_1, OP_SAFE_C_AAP_1,
      OP_C_P_1, OP_C_AP_1, OP_ANY_C_NP_2, OP_SAFE_C_PA_1,
      OP_SET_WITH_LET_1, OP_SET_WITH_LET_2,

      OP_CLOSURE_AP_1, OP_CLOSURE_PA_1, OP_CLOSURE_PP_1, OP_CLOSURE_P_1,
      OP_SAFE_CLOSURE_P_1, OP_SAFE_CLOSURE_P_A_1, OP_SAFE_CLOSURE_AP_1, OP_SAFE_CLOSURE_PA_1, OP_SAFE_CLOSURE_PP_1,
      OP_ANY_CLOSURE_3P_1, OP_ANY_CLOSURE_3P_2, OP_ANY_CLOSURE_3P_3, OP_ANY_CLOSURE_NP_1,
      OP_ANY_CLOSURE_4P_1, OP_ANY_CLOSURE_4P_2, OP_ANY_CLOSURE_4P_3, OP_ANY_CLOSURE_4P_4, OP_ANY_CLOSURE_NP_2,

      OP_TC_AND_A_OR_A_LA, OP_TC_OR_A_AND_A_LA, OP_TC_AND_A_OR_A_L2A, OP_TC_OR_A_AND_A_L2A, OP_TC_AND_A_OR_A_L3A, OP_TC_OR_A_AND_A_L3A,
      OP_TC_OR_A_A_AND_A_A_LA, OP_TC_OR_A_AND_A_A_L3A, OP_TC_AND_A_OR_A_A_LA, OP_TC_OR_A_AND_A_A_LA,
      OP_TC_WHEN_LA, OP_TC_WHEN_L2A, OP_TC_WHEN_L3A, OP_TC_LET_WHEN_L2A,
      OP_TC_COND_A_Z_A_L2A_L2A, OP_TC_LET_COND, OP_TC_COND_N,
      OP_TC_IF_A_Z_LA, OP_TC_IF_A_Z_L2A, OP_TC_IF_A_Z_L3A,
      OP_TC_IF_A_Z_IF_A_Z_LA, OP_TC_IF_A_Z_IF_A_LA_Z, OP_TC_IF_A_Z_IF_A_Z_L2A, OP_TC_IF_A_Z_IF_A_L2A_Z,
      OP_TC_IF_A_Z_IF_A_Z_L3A, OP_TC_IF_A_Z_IF_A_L3A_Z, OP_TC_IF_A_Z_IF_A_L3A_L3A,
      OP_TC_LET_IF_A_Z_LA, OP_TC_LET_IF_A_Z_L2A, OP_TC_IF_A_Z_LET_IF_A_Z_L2A,
      OP_TC_AND_A_IF_A_Z_LA, OP_TC_AND_A_IF_A_LA_Z,
      OP_TC_CASE_LA, OP_TC_CASE_L2A, OP_TC_CASE_L3A, /* treat this as last tc op (see below) */

      OP_RECUR_IF_A_A_opLA_LAq, OP_RECUR_IF_A_A_opL2A_L2Aq, OP_RECUR_IF_A_A_opL3A_L3Aq,
      OP_RECUR_IF_A_A_opA_LAq, OP_RECUR_IF_A_A_opA_L2Aq, OP_RECUR_IF_A_A_opA_L3Aq,
      OP_RECUR_IF_A_A_opLA_LA_LAq, OP_RECUR_IF_A_A_AND_A_L2A_L2A, OP_RECUR_IF_A_A_opA_LA_LAq,
      OP_RECUR_IF_A_A_IF_A_A_opLA_LAq, OP_RECUR_IF_A_A_IF_A_A_opL2A_L2Aq, OP_RECUR_IF_A_A_IF_A_A_opL3A_L3Aq,
      OP_RECUR_IF_A_A_IF_A_L2A_opA_L2Aq, OP_RECUR_COND_A_A_A_A_opA_L2Aq,
      OP_RECUR_COND_A_A_A_L2A_LopA_L2Aq, OP_RECUR_AND_A_OR_A_L2A_L2A,

      Num_Ops};

#define is_tc_op(Op) ((Op >= OP_TC_AND_A_OR_A_LA) && (Op <= OP_TC_CASE_L3A))

typedef enum {Combine_p, Combine_pp, Combine_cp, Combine_sp, Combine_pc, Combine_ps} combine_op_t;

#if s7_Debugging
static const char *op_names[Num_Ops] =
     {"unopt", "gc_protect",

      "safe_c_nc", "h_safe_c_nc", "safe_c_s", "h_safe_c_s",
      "safe_c_ss", "h_safe_c_ss", "safe_c_sc", "h_safe_c_sc", "safe_c_cs", "h_safe_c_cs", "safe_c_cq", "h_safe_c_cq",
      "safe_c_sss", "h_safe_c_sss", "safe_c_scs", "h_safe_c_scs", "safe_c_ssc", "h_safe_c_ssc", "safe_c_css", "h_safe_c_css",
      "safe_c_scc", "h_safe_c_scc", "safe_c_csc", "h_safe_c_csc", "safe_c_ccs", "h_safe_c_ccs",
      "safe_c_ns", "h_safe_c_ns", "safe_c_opncq", "h_safe_c_opncq", "safe_c_opsq", "h_safe_c_opsq",
      "safe_c_opssq", "h_safe_c_opssq", "safe_c_opscq", "h_safe_c_opscq",
      "safe_c_opcsq", "h_safe_c_opcsq", "safe_c_s_opsq", "h_safe_c_s_opsq",
      "safe_c_c_opscq", "h_safe_c_c_opscq", "safe_c_s_opscq", "h_safe_c_s_opscq", "safe_c_s_opcsq", "h_safe_c_s_opcsq",
      "safe_c_opsq_s", "h_safe_c_opsq_s", "safe_c_opsq_c", "h_safe_c_opsq_c",
      "safe_c_opsq_opsq", "h_safe_c_opsq_opsq", "safe_c_s_opssq", "h_safe_c_s_opssq", "safe_c_c_opsq", "h_safe_c_c_opsq",
      "safe_c_opcsq_c", "h_safe_c_opcsq_c", "safe_c_opssq_c", "h_safe_c_opssq_c", "safe_c_c_opssq", "h_safe_c_c_opssq",
      "safe_c_opssq_opssq", "h_safe_c_opssq_opssq", "safe_c_opssq_opsq", "h_safe_c_opssq_opsq", "safe_c_opsq_opssq", "h_safe_c_opsq_opssq",
      "safe_c_opssq_s", "h_safe_c_opssq_s", "safe_c_opcsq_s", "h_safe_c_opcsq_s", "safe_c_opscq_c", "h_safe_c_opscq_c",
      "safe_c_op_opssqq_s", "h_safe_c_op_opssqq_s", "safe_c_op_opsqq", "h_safe_c_op_opsqq",
      "safe_c_op_s_opsqq", "h_safe_c_op_s_opsqq", "safe_c_op_opsq_sq", "h_safe_c_op_opsq_sq", "safe_c_opsq_cs", "h_safe_c_opsq_cs",

      "safe_c_a", "h_safe_c_a", "safe_c_aa", "h_safe_c_aa", "safe_c_sa", "h_safe_c_sa", "safe_c_as", "h_safe_c_as",
      "safe_c_ca", "h_safe_c_ca", "safe_c_ac", "h_safe_c_ac", "safe_c_aaa", "h_safe_c_aaa", "safe_c_4a", "h_safe_c_4a",
      "safe_c_na", "h_safe_c_na", "safe_c_all_ca", "h_safe_c_all_ca",
      "safe_c_ssa", "h_safe_c_ssa", "safe_c_sas", "h_safe_c_sas", "safe_c_saa", "h_safe_c_saa",
      "safe_c_csa", "h_safe_c_csa", "safe_c_sca", "h_safe_c_sca", "safe_c_ass", "h_safe_c_ass",
      "safe_c_cac", "h_safe_c_cac", "safe_c_agg", "h_safe_c_agg",
      "safe_c_opaq", "h_safe_c_opaq", "safe_c_opaaq", "h_safe_c_opaaq", "safe_c_opaaaq", "h_safe_c_opaaaq",
      "safe_c_s_opaq", "h_safe_c_s_opaq", "safe_c_opaq_s", "h_safe_c_opaq_s", "safe_c_s_opaaq", "h_safe_c_s_opaaq",
      "safe_c*", "h_safe_c*", "safe_c*_a", "h_safe_c*_a", "safe_c*_aa", "h_safe_c*_aa", "safe_c*_na", "h_safe_c*_na",

      "safe_c_p", "h_safe_c_p", "safe_c_pp", "h_safe_c_pp", "safe_c_ff", "h_safe_c_ff", "safe_c_sp", "h_safe_c_sp",
      "safe_c_cp", "h_safe_c_cp", "safe_c_ap", "h_safe_c_ap", "safe_c_pa", "h_safe_c_pa", "safe_c_ps", "h_safe_c_ps",
      "safe_c_pc", "h_safe_c_pc", "safe_c_sps", "h_safe_c_sps", "safe_c_aap", "h_safe_c_aap",
      "safe_c_3p", "h_safe_c_3p", "any_c_np", "h_any_c_np",

      "thunk", "h_thunk", "thunk_o", "h_thunk_o", "thunk_c", "h_thunk_c", "thunk_any", "h_thunk_any",
      "safe_thunk", "h_safe_thunk", "safe_thunk_a", "h_safe_thunk_a", "safe_thunk_any", "h_safe_thunk_any",

      "closure_s", "h_closure_s", "closure_s_o", "h_closure_s_o",
      "closure_a", "h_closure_a", "closure_a_o", "h_closure_a_o", "closure_p", "h_closure_p",
      "closure_ap", "h_closure_ap", "closure_pa", "h_closure_pa", "closure_pp", "h_closure_pp",
      "closure_fa", "h_closure_fa", "closure_ss", "h_closure_ss", "closure_ss_o", "h_closure_ss_o",
      "closure_sc", "h_closure_sc", "closure_sc_o", "h_closure_sc_o",
      "closure_3s", "h_closure_3s", "closure_3s_o", "h_closure_3s_o", "closure_4s", "h_closure_4s", "closure_4s_o", "h_closure_4s_o", "closure_5s", "h_closure_5s",
      "closure_aa", "h_closure_aa", "closure_aa_o", "h_closure_aa_o", "closure_3a", "h_closure_3a", "closure_4a", "h_closure_4a",
      "closure_na", "h_closure_na", "closure_ass", "h_closure_ass", "closure_sas", "h_closure_sas ","closure_aas", "h_closure_aas",
      "closure_saa", "h_closure_saa", "closure_asa", "h_closure_asa", "closure_ns", "h_closure_ns",

      "safe_closure_s", "h_safe_closure_s", "safe_closure_s_o", "h_safe_closure_s_o",
      "safe_closure_s_a", "h_safe_closure_s_a", "safe_closure_s_to_s", "h_safe_closure_s_to_s", "safe_closure_s_to_sc", "h_safe_closure_s_to_sc",
      "safe_closure_p", "h_safe_closure_p", "safe_closure_p_a", "h_safe_closure_p_a",
      "safe_closure_ap", "h_safe_closure_ap", "safe_closure_pa", "h_safe_closure_pa", "safe_closure_pp", "h_safe_closure_pp",
      "safe_closure_a", "h_safe_closure_a", "safe_closure_a_o", "h_safe_closure_a_o", "safe_closure_a_a", "h_safe_closure_a_a",
      "safe_closure_a_to_sc", "h_safe_closure_a_to_sc",
      "safe_closure_ss", "h_safe_closure_ss", "safe_closure_ss_o", "h_safe_closure_ss_o", "safe_closure_ss_a", "h_safe_closure_ss_a",
      "safe_closure_sc", "h_safe_closure_sc", "safe_closure_sc_o", "h_safe_closure_sc_o",
      "safe_closure_aa", "h_safe_closure_aa", "safe_closure_aa_o", "h_safe_closure_aa_o", "safe_closure_aa_a", "h_safe_closure_aa_a",
      "safe_closure_saa", "h_safe_closure_saa", "safe_closure_ssa", "h_safe_closure_ssa",
      "safe_closure_agg", "h_safe_closure_agg", "safe_closure_3a", "h_safe_closure_3a", "safe_closure_na", "h_safe_closure_na",
      "safe_closure_3s", "h_safe_closure_3s", "safe_closure_ns", "h_safe_closure_ns",
      "safe_closure_3s_a", "h_safe_closure_3s_a",

      "any_closure_3p", "h_any_closure_3p", "any_closure_4p", "h_any_closure_4p", "any_closure_np", "h_any_closure_np",
      "any_closure_sym", "h_any_closure_sym", "any_closure_a_sym", "h_any_closure_a_sym",

      "closure*_a", "h_closure*_a", "closure*_na", "h_closure*_na",
      "safe_closure*_a", "h_safe_closure*_a", "safe_closure*_aa", "h_safe_closure*_aa",
      "safe_closure*_aa_o", "h_safe_closure*_aa_o", "safe_closure*_a1", "h_safe_closure*_a1",
      "safe_closure*_ka", "h_safe_closure*_ka", "closure*_ka", "h_closure*_ka", "safe_closure*_3a", "h_safe_closure*_3a",
      "safe_closure*_na", "h_safe_closure*_na", "safe_closure*_na_0", "h_safe_closure*_na_0",
      "safe_closure*_na_1", "h_safe_closure*_na_1", "safe_closure*_na_2", "h_safe_closure*_na_2",

      "c_ss", "h_c_ss", "c_s", "h_c_s", "c_sc", "h_c_sc", "read_s", "h_read_s", "c_p", "h_c_p", "c_ap", "h_c_ap",
      "c_a", "h_c_a", "c_aa", "h_c_aa", "c", "h_c", "c_nc", "h_c_nc", "c_na", "h_c_na",

      "cl_s", "h_cl_s", "cl_ss", "h_cl_ss", "cl_a", "h_cl_a", "cl_aa", "h_cl_aa",
      "cl_na", "h_cl_na", "cl_fa", "h_cl_fa", "cl_sas", "h_cl_sas",

      "apply_ss", "apply_sa", "apply_sl", "macro_d", "macro*_d",
      "with_input_from_string", "with_input_from_string_1", "with_output_to_string", "with_input_from_string_c", "call_with_output_string",
      "s", "s_g", "s_a", "s_aa", "a_a", "a_aa", "a_sc", "p_s", "p_s_1", "map_for_each_fa", "map_for_each_faa",
      "f", "f_a", "f_aa", "f_np", "f_np_1",

      "implicit_goto", "implicit_goto_a", "implicit_continuation_a","implicit_iterate",
      "implicit_vector_ref_a", "implicit_vector_ref_aa",
      "implicit_string_ref_a", "implicit_c_object_ref_a", "implicit_pair_ref_a", "implicit_pair_ref_aa",
      "implicit_hash_table_ref_a", "implicit_hash_table_ref_aa",
      "implicit_let_ref_c", "implicit_let_ref_a", "implicit_*s7*_ref_s", "implicit_*s7*_set_s",
      "unknown_thunk", "unknown_ns", "unknown_na", "unknown_s", "unknown_gg", "unknown_a", "unknown_aa", "unknown_np",

      "symbol", "constant", "pair_sym", "pair_pair", "pair_pair_s", "pair_any", "h_hash_table_increment", "clear_opts",

      "read_internal", "eval", "eval_args", "eval_args1", "eval_args2", "eval_args3", "eval_args4", "eval_args5",
      "eval_set1_no_mv", "eval_set2", "eval_set2_mv", "eval_set2_no_mv", "eval_set3", "eval_set3_mv", "eval_set3_no_mv",
      "apply", "eval_macro", "lambda", "quote", "quote_unchecked", "macroexpand", "call/cc", "call_with_exit", "call_with_exit_o",
      "c_catch", "c_catch_all", "c_catch_all_o", "c_catch_all_a",

      "define", "define1", "begin", "begin_hook", "begin_no_hook", "begin_unchecked", "begin_2_unchecked", "begin_na", "begin_aa",
      "if", "if1", "when", "unless", "set", "set1", "set2",
      "let", "let1", "let*", "let*1", "let*2", "let*-shadowed",
      "letrec", "letrec1", "letrec*", "letrec*1",
      "let_temporarily", "let_temp_unchecked", "let_temp_init1", "let_temp_init2", "let_temp_done", "let_temp_done1",
      "let_temp_s7", "let_temp_na", "let_temp_a", "let_temp_setter", "let_temp_unwind", "let_temp_s7_unwind", "let_temp_setter_unwind",
      "let_temp_a_a", "let_temp_s7_openlets", "let_temp_s7_openlets_unwind",
      "cond", "cond1", "feed_to_1", "cond_simple", "cond1_simple", "cond_simple_o", "cond1_simple_o",
      "and", "or",
      "define_macro", "define_macro*", "define_expansion", "define_expansion*", "macro", "macro*",
      "case", "read_list", "read_next", "read_dot", "read_quote",
      "read_quasiquote", "read_unquote", "read_apply_values",
      "read_vector", "read_byte_vector", "read_int_vector", "read_float_vector", "read_complex_vector", "read_done",
      "load_return_if_eof", "load_close_and_pop_if_eof", "eval_done", "splice_values", "no_values",
      "catch", "dynamic_wind", "dynamic_unwind", "dynamic_unwind_profile", "profile_in",
      "define_constant", "define_constant1",
      "do", "do_end", "do_end1", "do_step", "do_step2", "do_init",
      "define*", "lambda*", "lambda*_default", "error_quit", "unwind_input", "unwind_output", "error_hook_quit",
      "with_let", "with_let1", "with_let_unchecked", "with_let_s",
      "with_baffle", "with_baffle_unchecked", "expansion",
      "for_each", "for_each_1", "for_each_2", "for_each_3",
      "map", "map_1", "map_2", "map_gather", "map_gather_1", "map_gather_2", "map_gather_3", "map_unwind",
      "barrier", "deactivate_goto",
      "define_bacro", "define_bacro*", "bacro", "bacro*",
      "get_output_string",
      "sort", "sort1", "sort2", "sort3", "sort_pair_end", "sort_vector_end", "sort_string_end",
      "eval_string",
      "member_if", "assoc_if", "member_if1", "assoc_if1",
      "lambda_unchecked", "let_unchecked", "catch_1", "catch_2", "catch_all",
      "set_unchecked", "set_s_c", "set_s_s", "set_s_p", "set_s_a",
      "set_normal", "set_opsq_a", "set_opsaq_a", "set_opsaq_p", "set_opsaq_p_1", "set_opsaaq_a", "set_opsaaq_p", "set_opsaaq_p_1",
      "set_from_setter", "set_from_let_temp", "set_sp_1",
      "increment_1", "decrement_1", "increment_ss", "increment_sa", "increment_saa", "set_cons",
      "letrec_unchecked", "letrec*_unchecked", "cond_unchecked",
      "lambda*_unchecked", "do_unchecked", "define_unchecked", "define*_unchecked", "define_funchecked", "define_constant_unchecked",
      "define_with_setter",

      "let_no_vars", "named_let", "named_let_no_vars", "named_let_a", "named_let_aa", "named_let_na", "named_let*",
      "let_na_old", "let_na_new", "let_2a_old", "let_2a_new", "let_3a_old", "let_3a_new",
      "let_opassq_old", "let_opassq_new", "let_one_old", "let_one_new", "let_one_p_old", "let_one_p_new",
      "let_one_old_1", "let_one_new_1", "let_one_p_old_1", "let_one_p_new_1",
      "let_a_old", "let_a_new", "let_a_p_old", "let_a_p_new",
      "let_a_a_old", "let_a_a_new", "let_a_na_old", "let_a_na_new", "let_a_old_2", "let_a_new_2",
      "let*_na", "let*_na_a",

      "case_a_e_s", "case_a_i_s", "case_a_g_s", "case_a_e_g", "case_a_g_g", "case_a_s_g",
      "case_p_e_s", "case_p_i_s", "case_p_g_s", "case_p_e_g", "case_p_g_g",
      "case_e_s", "case_i_s", "case_g_s", "case_e_g", "case_g_g",
      "case_a_i_s_a", "case_a_e_s_a", "case_a_g_s_a", "case_a_s_g_a",

      "if_unchecked", "and_p", "and_p1", "and_ap", "and_pair_p",
      "and_safe_p1", "and_safe_p2", "and_safe_p3", "and_safe_p_rest", "and_2a", "and_3a", "and_n", "and_s_2",
      "or_p", "or_p1", "or_ap", "or_2a", "or_3a", "or_n", "or_s_2", "or_s_type_2",
      "when_s", "when_a", "when_p", "when_and_ap", "when_and_2a", "when_and_3a", "unless_s", "unless_a", "unless_p",

      "if_a_c_c", "if_a_a", "if_a_a_a", "if_s_a_a", "if_and2_s_a", "if_not_a_a", "if_not_a_a_a",
      "if_b_a", "if_b_p", "if_b_r",  "if_b_a_p", "if_b_p_a", "if_b_p_p", "if_b_n_n",
      "if_a_a_p", "if_a_p_a", "if_s_p_a", "if_s_a_p", "if_s_p", "if_s_p_p", "if_s_r", "if_s_n", "if_s_n_n",
      "if_opsq_p", "if_opsq_p_p", "if_opsq_r", "if_opsq_n", "if_opsq_n_n",
      "if_is_type_s_p", "if_is_type_s_p_p", "if_is_type_s_r", "if_is_type_s_n", "if_is_type_s_n_n", "if_is_type_s_p_a", "if_is_type_s_a_a", "if_is_type_s_a_p",
      "if_a_p", "if_a_p_p", "if_a_r", "if_a_n", "if_a_n_n",
      "if_and2_p", "if_and2_p_p", "if_and2_r", "if_and2_n", "if_and2_n_n",
      "if_and3_p", "if_and3_p_p", "if_and3_r", "if_and3_n", "if_and3_n_n",
      "if_p_p", "if_p_p_p", "if_p_r", "if_p_n", "if_p_n_n",
      "if_andp_p", "if_andp_p_p", "if_andp_r", "if_andp_n", "if_andp_n_n",
      "if_orp_p", "if_orp_p_p", "if_orp_r", "if_orp_n", "if_orp_n_n",
      "if_or2_p", "if_or2_p_p", "if_or2_r", "if_or2_n", "if_or2_n_n",
      "if_pp", "if_ppp", "if_pn", "if_pr", "if_prr", "when_pp", "unless_pp",

      "cond_na_na", "cond_na_np", "cond_na_np_1", "cond_na_2e", "cond_na_3e", "cond_na_np_o",
      "cond_feed", "cond_feed_1",

      "simple_do", "simple_do_step", "safe_dotimes", "safe_dotimes_step", "safe_dotimes_step_o",
      "safe_do", "safe_do_step", "dox", "dox_step", "dox_step_o", "dox_no_body", "dox_pending_no_body", "dox_init",
      "dotimes_p", "dotimes_step_o",
      "do_no_vars", "do_no_vars_no_opt", "do_no_vars_no_opt_1",
      "do_no_body_na_vars", "do_no_body_na_vars_step", "do_no_body_na_vars_step_1",

      "safe_c_p_1", "safe_c_pp_1", "safe_c_pp_3_mv", "safe_c_pp_5",
      "safe_c_3p_1", "safe_c_3p_2", "safe_c_3p_3", "safe_c_3p_1_mv", "safe_c_3p_2_mv", "safe_c_3p_3_mv",
      "safe_c_sp_1", "safe_cons_sp_1", "safe_add_sp_1", "safe_multiply_sp_1", "safe_c_ps_1", "safe_c_pc_1",
      "eval_macro_mv", "macroexpand_1", "apply_lambda",
      "any_c_np_1", "any_c_np_mv", "safe_c_sps_1", "safe_c_aap_1",
      "c_p_1", "c_ap_1", "any_c_np_2", "safe_c_pa_1",
      "set_with_let_1", "set_with_let_2",

      "closure_ap_1", "closure_pa_1", "closure_pp_1", "closure_p_1",
      "safe_closure_p_1", "safe_closure_p_a_1", "safe_closure_ap_1", "safe_closure_pa_1", "safe_closure_pp_1",
      "any_closure_3p_1", "any_closure_3p_2", "any_closure_3p_3", "any_closure_np_1",
      "any_closure_4p_1", "any_closure_4p_2", "any_closure_4p_3", "any_closure_4p_4", "any_closure_np_2",

      "tc_and_a_or_a_la", "tc_or_a_and_a_la", "tc_and_a_or_a_l2a", "tc_or_a_and_a_l2a", "tc_and_a_or_a_l3a", "tc_or_a_and_a_l3a",
      "tc_or_a_a_and_a_a_la", "tc_or_a_and_a_a_l3a", "tc_and_a_or_a_a_la", "tc_or_a_and_a_a_la",
      "tc_when_la", "tc_when_l2a", "tc_when_l3a", "tc_let_when_l2a",
      "tc_cond_a_z_a_l2a_l2a", "tc_let_cond", "tc_cond_n",
      "tc_if_a_z_la", "tc_if_a_z_l2a", "tc_if_a_z_l3a",
      "tc_if_a_z_if_a_z_la", "tc_if_a_z_if_a_la_z", "tc_if_a_z_if_a_z_l2a", "tc_if_a_z_if_a_l2a_z",
      "tc_if_a_z_if_a_z_l3a", "tc_if_a_z_if_a_l3a_z", "tc_if_a_z_if_a_l3a_l3a",
      "tc_let_if_a_z_la", "tc_let_if_a_z_l2a", "if_a_z_let_if_a_z_l2a",
      "tc_and_a_if_a_z_la", "tc_and_a_if_a_la_z",
      "tc_case_la", "tc_case_l2a", "tc_case_l3a",

      "recur_if_a_a_opla_laq", "recur_if_a_a_opl2a_l2aq", "recur_if_a_a_opl3a_l3aq",
      "recur_if_a_a_opa_laq", "recur_if_a_a_opa_l2aq", "recur_if_a_a_opa_l3aq",
      "recur_if_a_a_opla_la_laq", "recur_if_a_a_and_a_l2a_l2a", "recur_if_a_a_opa_la_laq",
      "recur_if_a_a_if_a_a_opla_laq", "recur_if_a_a_if_a_a_opl2a_l2aq", "recur_if_a_a_if_a_a_opl3a_l3aq",
      "recur_if_a_a_if_a_l2a_opa_l2aq", "recur_cond_a_a_a_a_opa_l2aq",
      "recur_cond_a_a_a_l2a_lopa_l2aq", "recur_and_a_or_a_l2a_l2a"
};
#endif

#define is_safe_c_op(op)            ((op >= OP_SAFE_C_NC) && (op < OP_THUNK))
#define is_safe_closure_op(op)      ((op >= OP_SAFE_CLOSURE_S) && (op < OP_ANY_CLOSURE_3P))
#define is_safe_closure_star_op(op) ((op >= OP_SAFE_CLOSURE_STAR_A) && (op < OP_C_SS))
#define is_unknown_op(op)           ((op >= OP_UNKNOWN) && (op <= OP_UNKNOWN_NP))
#define is_h_safe_c_nc(P)           (optimize_op(P) == HOP_SAFE_C_NC)
#define is_safe_c_s(P)              ((optimize_op(P) == OP_SAFE_C_S) || (optimize_op(P) == HOP_SAFE_C_S))
#define is_h_safe_c_s(P)            (optimize_op(P) == HOP_SAFE_C_S)
#define First_Unhoppable_Op OP_APPLY_SS

static bool is_h_optimized(s7_pointer p)
{
  return((is_optimized(p)) &&
	 (op_has_hop(p)) &&
	 (optimize_op(p) < First_Unhoppable_Op) &&  /* was OP_S? */
	 (optimize_op(p) > OP_GC_PROTECT));
}

/* if this changes, remember to change lint.scm */
typedef enum {sl_No_Field=0, sl_Accept_All_Keyword_Arguments, sl_Autoloading, sl_Bignum_Precision, sl_Catches, sl_Cpu_Time, sl_C_Types,
	      sl_Debug, sl_Default_Hash_Table_Length, sl_Default_Random_State, sl_Default_Rationalize_Error, sl_Equivalent_Float_Epsilon,
	      sl_Expansions, sl_Filenames, sl_File_Names, sl_Float_Format_Precision, sl_Free_Heap_Size, sl_gc_Freed, sl_Gc_Info,
	      sl_Gc_Protected_Objects, sl_Gc_Resize_Heap_By_4_Fraction, sl_Gc_Resize_Heap_Fraction, sl_Gc_Stats, sl_Gc_Temps_Size,
	      sl_Gc_Total_Freed, sl_Hash_Table_Float_Epsilon, sl_Hash_Table_Missing_Key_Value, sl_Heap_Size, sl_History, sl_History_enabled,
	      sl_History_size, sl_Initial_String_Port_Length, sl_Iterator_At_End_Value, sl_Major_Version, sl_Max_Heap_Size, sl_Max_List_Length,
	      sl_Max_Stack_Size, sl_Max_String_Length, sl_Max_String_Port_Length, sl_Max_Vector_Dimensions, sl_Max_Vector_Length,
	      sl_Memory_Usage, sl_Minor_Version, sl_Most_Negative_Fixnum, sl_Most_Positive_Fixnum, sl_Muffle_Warnings,
	      sl_Number_Separator, sl_Openlets, sl_Output_File_Port_Length, sl_Print_Length, sl_Profile, sl_Profile_info,
	      sl_Profile_prefix, sl_Rootlet_Size, sl_Safety, sl_Scheme_Version, sl_Stack, sl_Stacktrace_defaults, sl_Stack_size, sl_Stack_top,
	      sl_Symbol_Quote, sl_Symbol_Printer, sl_Undefined_Constant_Warnings, sl_Undefined_Identifier_Warnings, sl_Version,
	      sl_Num_Fields} starlet_t;
/* default_bignum_precision|print_length|history-size|random-state? */

static const char *starlet_names[sl_Num_Fields] =
  {"no-field", "accept-all-keyword-arguments", "autoloading?", "bignum-precision", "catches", "cpu-time", "c-types",
   "debug", "default-hash-table-length", "default-random-state", "default-rationalize-error", "equivalent-float-epsilon",
   "expansions?", "filenames", "file-names", "float-format-precision", "free-heap-size", "gc-freed", "gc-info",
   "gc-protected-objects", "gc-resize-heap-by-4-fraction", "gc-resize-heap-fraction", "gc-stats", "gc-temps-size",
   "gc-total-freed", "hash-table-float-epsilon", "hash-table-missing-key-value", "heap-size", "history", "history-enabled",
   "history-size", "initial-string-port-length", "iterator-at-end-value", "major-version", "max-heap-size", "max-list-length",
   "max-stack-size", "max-string-length", "max-string-port-length", "max-vector-dimensions", "max-vector-length",
   "memory-usage", "minor-version", "most-negative-fixnum", "most-positive-fixnum", "muffle-warnings?",
   "number-separator", "openlets", "output-file-port-length", "print-length", "profile", "profile-info",
   "profile-prefix", "rootlet-size", "safety", "scheme-version", "stack", "stacktrace-defaults", "stack-size", "stack-top",
   "symbol-quote?", "symbol-printer", "undefined-constant-warnings", "undefined-identifier-warnings", "version"};

static s7_pointer object_to_string_truncated(s7_scheme *sc, s7_pointer p);
static const char *type_name(s7_scheme *sc, s7_pointer arg, article_t article);
static s7_pointer cons_unchecked(s7_scheme *sc, s7_pointer a, s7_pointer b);
static s7_pointer unbound_variable(s7_scheme *sc, s7_pointer sym); /* see lookup_checked */


#define Bold_Text    "\033[1m"
#define Unbold_Text  "\033[22m"
#define Red_Text     "\033[31m"
#define Green_Text   "\033[32m"
#define Blue_Text    "\033[34m"
#define Uncolor_Text "\033[0m" /* yellow=33 */


/* -------------------------------- internal debugging apparatus -------------------------------- */
static s7_int heap_location(s7_scheme *sc, s7_pointer p)
{
  for (heap_block_t *hp = sc->heap_blocks; hp; hp = hp->next)
    if (((intptr_t)p >= hp->start) && ((intptr_t)p < hp->end))
      return(hp->offset + (((intptr_t)p - hp->start) / sizeof(s7_cell)));
  return(((s7_big_pointer)p)->big_hloc);
}

#if TRAP_SEGFAULT
#include <signal.h>
static Jmp_Buf siglet; /* global here is not a problem -- it is used only to protect s7_is_valid */
static volatile sig_atomic_t can_jump = 0;
static void segv(int32_t unused) {if (can_jump) LongJmp(siglet, 1);}
#endif

#if s7_Debugging
static bool s7_is_in_heap(s7_scheme *sc, s7_pointer arg) /* for heap analysis where we only care about stuff in the heap */
{
  s7_pointer heap0 = *(sc->heap);
  s7_pointer heap1 = (s7_pointer)(heap0 + sc->heap_size);
  return((arg >= heap0) && (arg < heap1));
}
#endif

bool s7_is_valid(s7_scheme *sc, s7_pointer arg)
{
  bool result = false;
  if (!arg) return(false);
  {
    s7_pointer heap0 = *(sc->heap);
    s7_pointer heap1 = (s7_pointer)(heap0 + sc->heap_size);
    if ((arg >= heap0) && (arg < heap1)) return(true);
  }
#if TRAP_SEGFAULT
  if (SetJmp(siglet, 1) == 0)
    {
      void (*old_segv)(int32_t sig);
      can_jump = 1;
      old_segv = signal(SIGSEGV, segv);
#endif
      if ((type_unchecked(arg) > T_FREE) &&
	  (type_unchecked(arg) < Num_Types))
	{
	  if (!in_heap(arg))
	    result = true;
	  else
	    {
	      s7_int loc = heap_location(sc, arg);
	      if ((loc >= 0) && (loc < sc->heap_size))
		result = (sc->heap[loc] == arg);
	    }}
#if TRAP_SEGFAULT
      signal(SIGSEGV, old_segv);
    }
  else result = false;
  can_jump = 0;
#endif
  return(result);
}

#define safe_print(Code)	   \
  do {				   \
    bool _Old_Open_ = sc->has_openlets;  \
    sc->has_openlets = false;      \
    Code;			   \
    sc->has_openlets = _Old_Open_;   \
  } while (0)

void s7_show_history(s7_scheme *sc);
void s7_show_history(s7_scheme *sc)
{
#if With_History
  if (sc->cur_code == sc->history_sink)
    fprintf(stderr, "history diabled\n");
  else
    {
      s7_pointer p = cdr(sc->cur_code);
      fprintf(stderr, "history:\n");
      for (int32_t i = 0, size = sc->history_size; i < size; i++, p = cdr(p)) /* stepper "i" is not redundant */
	safe_print(fprintf(stderr, "%d: %s\n", i, display_truncated(car(p))));
      fprintf(stderr, "\n");
    }
#else
  fprintf(stderr, "%s\n", display(sc->cur_code));
#endif
}

#if s7_Debugging
#define Unused_Bits 0x000fc00000000c0 /* high 6 bits of optimizer code + high 2 bits of type */

static char *describe_type_bits(s7_scheme *sc, s7_pointer obj)
{
  const s7_uint full_typ = full_type(obj);
  const uint8_t typ = type_unchecked(obj);
  char str[900];

  str[0] = '\0';
  catstrs(str, 900,	  /* if debugging, all of these bits are being watched, so we need to access them directly */
	  /* bit 8 (the first 8 bits (after the 8 type bits) are easy...) */
	  ((full_typ & T_Multiform) != 0) ?      ((is_any_closure(obj)) ?
						  (((full_typ & T_One_Form) != 0) ? " closure-one-form-has-fx" : " closure-multiform") :
						  " ?0?") : "",
	  /* bit 9 */
	  ((full_typ & T_Syntactic) != 0) ?      (((is_pair(obj)) || (is_syntax(obj)) || (is_normal_symbol(obj))) ?
						  " syntactic" :
						  " ?1?") : "",
	  /* bit 10 */
	  ((full_typ & T_Simple_Arg_Defaults) != 0) ? ((is_pair(obj)) ? " simple-args|in-use" :
						       ((is_any_closure(obj)) ? " closure-one-form" :
							" ?2?")) : "",
	  /* bit 11 */
	  ((full_typ & T_Optimized) != 0) ?      ((is_c_function(obj)) ? " scope-safe" :
						  ((is_pair(obj)) ? " optimized" :
						   " ?3?")) : "",
	  /* bit 12 */
	  ((full_typ & T_Safe_Closure) != 0) ?   (((has_closure_let(obj)) || (is_pair(obj))) ? " safe-closure" : " ?4?") : "",
	  /* bit 13 */
	  ((full_typ & T_Dont_Eval_Args) != 0) ? (((is_any_macro(obj)) || (is_syntax(obj))) ? " dont-eval-args" : " ?5?") : "",
	  /* bit 14 */
	  ((full_typ & T_Expansion) != 0) ?      (((is_normal_symbol(obj)) || (is_any_macro(obj))) ? " expansion" :
						  " ?6?") : "",
	  /* bit 15 */
	  ((full_typ & T_Multiple_Value) != 0) ? ((is_symbol(obj)) ? " matched" :
						  ((is_pair(obj)) ? " values|matched" :
						   " ?7?")) : "",
	  /* bit 16 */
	  ((full_typ & T_Unsafe_Do) != 0) ?      ((is_pair(obj)) ? " unsafe-do" :
						  ((is_let(obj)) ? " dox-slot1" :
						   ((is_any_c_function(obj)) ? " even-args" :
						    ((is_symbol(obj)) ? " maybe-shadowed" :
						     " ?8?")))) : "",
	  /* bit 17 */
	  ((full_typ & T_Collected) != 0) ?      " collected" : "",
	  /* bit 18 */
	  ((full_typ & T_Location) != 0) ?       ((is_pair(obj)) ? " line-number" :
						  ((is_input_port(obj)) ? " loader-port" :
						   ((is_let(obj)) ? " with-let" :
						    ((is_any_procedure(obj)) ? " simple-defaults" :
						     ((is_slot(obj)) ? " has-setter" :
						      " ?10?"))))) : "",
	  /* bit 19 */
	  ((full_typ & T_Shared) != 0) ?         ((is_sequence(obj)) ? " shared" : " ?11?") : "",
	  /* bit 20 */
	  ((full_typ & T_Low_Count) != 0) ?      ((is_pair(obj)) ? " low-count" : " init-value") : "",
	  /* bit 21 */
	  ((full_typ & T_Safe_Procedure) != 0) ? ((is_applicable(obj)) ? " safe-procedure" : " ?13?") : "",
	  /* bit 22 */
	  ((full_typ & T_Checked) != 0) ?        (((is_pair(obj)) || (is_slot(obj))) ? " checked" :
						  ((is_symbol(obj)) ? " all-integer" :
						   " ?14?")) : "",
	  /* bit 23 */
	  ((full_typ & T_Unsafe) != 0) ?         ((is_symbol(obj)) ? " clean-symbol" :
						  ((is_slot(obj)) ? " has-stepper" :
						   ((is_pair(obj)) ? " unsafely-opt|no-float-opt" :
						    ((is_let(obj)) ? " dox-slot2" :
						     " ?15?")))) : "",
	  /* bit 24 */
	  ((full_typ & T_Immutable) != 0) ?      " immutable" : "",
	  /* bit 25 */
	  ((full_typ & T_Allow_Other_Keys) != 0) ? ((is_pair(obj)) ? " allow-other-keys|no-int-opt" :
						    ((is_slot(obj)) ? " has-expression" :
						     ((is_c_function_star(obj)) ? " allow-other-keys" :
						      ((is_let(obj)) ? " let-removed-from-heap" :
						       " ?17?")))) : "",
	  /* bit 26 */
	  ((full_typ & T_Mutable) != 0) ?        ((is_number(obj)) ? " mutable" :
						  ((is_symbol(obj)) ? " has-keyword" :
						   ((is_let(obj)) ? " ref-fallback" :
						    ((is_iterator(obj)) ? " mark-sequence" :
						     ((is_slot(obj)) ? " step-end" :
						      ((is_pair(obj)) ? " no-opt" :
						       " ?18?")))))) : "",
	  /* bit 27 */
	  ((full_typ & T_Safe_Stepper) != 0) ?   ((is_let(obj)) ? " set-fallback" :
						  ((is_slot(obj)) ? " safe-stepper" :
						   ((is_c_function(obj)) ? " maybe-safe" :
						    ((is_pair(obj)) ? " direct-opt" :
						     ((is_hash_table(obj)) ? " weak-hash" :
						      ((is_any_macro(obj)) ? " pair-macro-set" :
						       ((is_symbol(obj)) ? " all-float" :
							((is_string(obj)) ? " wrapper" :
							 " ?19?")))))))) : "",

	  /* bit 28, for c_function case see sc->apply */
	  ((full_typ & T_Copy_Args) != 0) ?      (((is_pair(obj)) || (is_any_macro(obj)) || (is_syntax(obj)) ||
						   (is_any_closure(obj)) || (is_c_function(obj))) ? " copy-args" :
						  " ?20?") : "",
	  /* bit 29 */
	  ((full_typ & T_Gensym) != 0) ?         ((is_let(obj)) ? " funclet" :
						  ((is_normal_symbol(obj)) ? " gensym" :
						   ((is_string(obj)) ? " documented-symbol" :
						    ((is_hash_table(obj)) ? " hash-chosen" :
						     ((is_pair(obj)) ? " fx-treed" :
						      ((is_any_vector(obj)) ? " subvector" :
						       ((is_slot(obj)) ? " has-pending-value" :
							((is_any_closure(obj)) ? " unknopt" :
							 " ?21?")))))))) : "",
	  /* bit 30 */
	  ((full_typ & T_Has_Methods) != 0) ?    (((is_let(obj)) || (is_c_object(obj)) || (is_any_closure(obj)) ||
						   (is_any_macro(obj)) || (is_c_pointer(obj))) ? " has-methods" :
						  " ?22?") : "",
	  /* bit 31 */
	  ((full_typ & T_Iter_Ok) != 0) ?        ((is_iterator(obj)) ? " iter-ok" :
						  ((is_pair(obj)) ? " loop-end-possible" :
						   ((is_slot(obj)) ? " in-rootlet" :
						    ((is_c_function(obj)) ? " bool-function" :
						     ((is_symbol(obj)) ? " symbol-from-symbol" :
						      " ?23?"))))) : "",
	  /* bit 24+24 */
	  ((full_typ & T_Full_Symcons) != 0) ?   ((is_symbol(obj)) ? " possibly-constant" :
						  ((is_any_procedure(obj)) ? " has-let-arg" :
						   ((is_hash_table(obj)) ? " has-value-type" :
						    ((is_pair(obj)) ? " int-optable" :
						     ((is_let(obj)) ? " unlet" :
						      ((is_t_vector(obj)) ? " symbol-table" :
						       " ?24?")))))) : "",
	  /* bit 25+24 */
	  ((full_typ & T_Full_Has_Let_File) != 0) ? ((is_let(obj)) ? " has-let-file" :
						     ((is_t_vector(obj)) ? " typed-vector" :
						      ((is_hash_table(obj)) ? " typed-hash-table" :
						       ((is_c_function(obj)) ? " has-bool-setter" :
							((is_slot(obj)) ? " rest-slot" :
							 (((is_pair(obj)) || (is_closure_star(obj))) ? " no-defaults" :
							  " ?25?")))))) : "",
	  /* bit 26+24 */
	  ((full_typ & T_Full_Definer) != 0) ?   ((is_normal_symbol(obj)) ? " definer" :
						  ((is_pair(obj)) ? " has-fx" :
						   ((is_slot(obj)) ? " slot-defaults" :
						    ((is_iterator(obj)) ? " weak-hash-iterator" :
						     ((is_hash_table(obj)) ? " has-key-type" :
						      ((is_let(obj)) ? " maclet" :
						       ((is_c_function(obj)) ? " func-definer" :
							((is_syntax(obj)) ? " syntax-definer" :
							 " ?26?")))))))) : "",
	  /* bit 27+24 */
	  ((full_typ & T_Full_Binder) != 0) ?    ((is_pair(obj)) ? " tree-collected" :
						  ((is_hash_table(obj)) ? " simple-values" :
						   ((is_normal_symbol(obj)) ? " binder" :
						    ((is_c_function(obj)) ? " safe-args" :
						     ((is_syntax(obj)) ?  " syntax-binder" :
						      " ?27?"))))) : "",
	  /* bit 28+24 */
	  ((full_typ & T_Very_Safe_Closure) != 0) ? (((is_pair(obj)) || (is_any_closure(obj))) ? " very-safe-closure" :
						     ((is_let(obj)) ? " baffle-let" :
						      " ?28?")) : "",
	  /* bit 29+24 */
	  ((full_typ & T_Cyclic) != 0) ?         (((is_simple_sequence(obj)) || (t_structure_p[type(obj)]) ||
						   (is_any_closure(obj))) ? " cyclic" : " ?29?") : "",
	  /* bit 30+24 */
	  ((full_typ & T_Cyclic_Set) != 0) ?     (((is_simple_sequence(obj)) || (t_structure_p[type(obj)]) ||
						   (is_any_closure(obj))) ? " cyclic-set" : " ?30?") : "",
	  /* bit 31+24 */
	  ((full_typ & T_Keyword) != 0) ?        ((is_symbol(obj)) ? " keyword" :
						  ((is_pair(obj)) ? " fx-treeable" :
						   " ?31?")) : "",
	  /* bit 32+24 */
	  ((full_typ & T_Full_Simple_Elements) != 0) ? ((is_t_vector(obj)) ? " simple-elements" :
							((is_hash_table(obj)) ? " simple-keys" :
							 ((is_normal_symbol(obj)) ? " safe-setter" :
							  ((is_pair(obj)) ? " float-optable" :
							   ((typ >= T_C_MACRO) ? " function-simple-elements" :
							    " 32?"))))) : "",
	  /* bit 33+24 */
	  ((full_typ & T_Full_Case_Key) != 0) ?  ((is_symbol(obj)) ? " case-key" :
						  ((is_pair(obj)) ? " opt1-func-listed" :
						   " ?33?")) : "",
	  /* bit 34+24 */
	  ((full_typ & T_Full_True_Is_Done) != 0) ? ((is_pair(obj)) ? " #t-is-done" :
						     ((is_symbol(obj)) ? " saver-symbol" :
						      ((is_c_function(obj)) ? " saver-c-function" :
						       " ?34?"))) : "",
	  /* bit 35+24 */
	  ((full_typ & T_Full_Unknopt) != 0) ?   ((is_pair(obj)) ? " unknopt" :
						  ((is_symbol(obj)) ? " translucent-symbol" :
						   ((is_c_function(obj)) ? " translucent-c-function" :
						    " ?35?"))) : "",
	  /* bit 36+24 */
	  ((full_typ & T_Full_Safety_Checked) != 0) ? ((is_pair(obj)) ? " safety-checked" :
						       ((is_symbol(obj)) ? " setter" :
							((is_c_function(obj)) ? " setter-c-function" :
							 ((is_syntax(obj)) ? " setter-syntax" :
							  " ?36?")))) : "",
	  /* bit 37+24 */
	  ((full_typ & T_Full_Has_Fn) != 0) ?    ((is_pair(obj)) ? " has-fn" :
						  ((is_symbol(obj)) ? " escaper-symbol" :
						   ((is_syntax(obj)) ? " escaper-syntax" :
						    ((is_c_function(obj)) ? " escaper-c-function" :
						     " ?37")))) : "",
	  /* bit 62 */
	  ((full_typ & T_Unheap) != 0) ?         " unheap" : "",
	  /* bit 63 */
	  ((full_typ & T_Gc_Mark) != 0) ?        " gc-marked" : "",

	  ((full_typ & Unused_Bits) != 0) ?      " unused bits set?" : "",

	  ((is_symbol(obj)) && (((uint8_t)(symbol_type(obj) & 0xff) >= Num_Types) || ((symbol_type(obj) & ~0xffff) != 0))) ? " bad-symbol-type" : "",
	  (((is_any_macro(obj)) || (is_syntax(obj))) && ((full_typ & T_Dont_Eval_Args) == 0)) ? " dont-eval-args not set" : "",
	  /* for is_applicable, e.g. goto is not a safe procedure -- maybe check goto/continuation...? */
	  NULL);
  {
    #define Bits_Bufsize 1024
    char *buf = (char *)Malloc(Bits_Bufsize);
    snprintf(buf, Bits_Bufsize, "%s? (type: %d), opt_op: %d %s, full_type: #x%" PRIx64 "%s",
	     type_name(sc, obj, No_Article), typ,
	     optimize_op_unchecked(obj), (optimize_op_unchecked(obj) < Num_Ops) ? op_names[optimize_op_unchecked(obj)] : "", full_typ,
	     str);
    /* snprintf returns the number of bytes that would have been written: (display (c-pointer 123123123 (symbol (make-string 130 #\a)))) */
    return(buf);
  }
}

void s7_decode_type(s7_pointer p, s7_uint new_type);
void s7_decode_type(s7_pointer p, s7_uint new_type) /* just pass any valid s7_pointer! This is for GC debugging (in gdb primarily). */
{
  s7_uint old_type = p->tf.u64_type;
  char *str;
  p->tf.u64_type = new_type;
  str = describe_type_bits(cur_sc, p);
  p->tf.u64_type = old_type;
  fprintf(stderr, "%s\n", str);
  free(str);
}

static bool never_unheaped[Num_Types];
static void init_never_unheaped(void)
{
  #define Heaped_Size 20
  const int32_t heaped[Heaped_Size] = {
    T_BACRO, T_BACRO_STAR, T_CATCH, T_CLOSURE, T_CLOSURE_STAR, T_CONTINUATION, T_COUNTER, T_C_OBJECT, T_C_POINTER, T_DYNAMIC_WIND,
    T_FREE, T_GOTO, T_HASH_TABLE, T_ITERATOR, T_MACRO, T_MACRO_STAR, T_RANDOM_STATE, T_SLOT, T_STACK, T_VECTOR};
  /* T_UNUSED, like T_NIL, is never in the heap, but can be unheaped slot value */
  for (int32_t i = 0; i < Num_Types; i++) never_unheaped[i] = false;
  for (int32_t i = 0; i < Heaped_Size; i++) never_unheaped[heaped[i]] = true;
}

static bool has_odd_bits(s7_pointer obj)
{
  const s7_uint full_typ = full_type(obj);
  if ((full_typ & Unused_Bits) != 0) return(true);
  if (((full_typ & T_Multiform) != 0) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_Keyword) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_Syntactic) != 0) && (!is_syntax(obj)) && (!is_pair(obj)) && (!is_normal_symbol(obj))) return(true);
  if (((full_typ & T_Simple_Arg_Defaults) != 0) && (!is_pair(obj)) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_Optimized) != 0) && (!is_c_function(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_Safe_Closure) != 0) && (!is_any_closure(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_Safe_Procedure) != 0) && (!is_applicable(obj))) return(true);
  if (((full_typ & T_Expansion) != 0) && (!is_normal_symbol(obj)) && (!is_any_macro(obj))) return(true);
  if (((full_typ & T_Multiple_Value) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_Unsafe_Do) != 0) && (!is_pair(obj)) && (!is_let(obj)) && (!is_any_c_function(obj)) && (!is_symbol(obj))) return(true);
  if (((full_typ & T_Iter_Ok) != 0) && (!is_iterator(obj)) && (!is_pair(obj)) && (!is_slot(obj)) && (!is_c_function(obj)) && (!is_symbol(obj))) return(true);
  /* if (((full_typ & T_Low_Count) != 0) && (!is_pair(obj))) return(true); */
  if (((full_typ & T_Unsafe) != 0) && (!is_symbol(obj)) && (!is_slot(obj)) && (!is_let(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_Very_Safe_Closure) != 0) && (!is_pair(obj)) && (!is_any_closure(obj)) && (!is_let(obj))) return(true);
  if (((full_typ & T_Full_Case_Key) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_Full_Unknopt) != 0) && (!is_pair(obj)) && (!is_symbol(obj)) && (!is_c_function(obj))) return(true);
  if (((full_typ & T_Full_Safety_Checked) != 0) && (!is_pair(obj)) && (!is_normal_symbol(obj)) && (!is_c_function(obj)) && (!is_syntax(obj))) return(true);
  if (((full_typ & T_Dont_Eval_Args) != 0) && (!is_any_macro(obj)) && (!is_syntax(obj))) return(true);
  if (((full_typ & T_Checked) != 0) && (!is_slot(obj)) && (!is_pair(obj)) && (!is_symbol(obj))) return(true);
  if (((full_typ & T_Shared) != 0) && (!t_sequence_p[type(obj)]) && (!t_structure_p[type(obj)]) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_Full_Allow_Other_Keys) != 0) && (!is_slot(obj)) && (!is_pair(obj)) && (!is_let(obj)) && (!is_c_function_star(obj)) && (!is_let(obj))) return(true);
  if (((full_typ & T_Copy_Args) != 0) && (!is_pair(obj)) &&
      (!is_any_macro(obj)) && (!is_any_closure(obj)) && (!is_c_function(obj)) && (!is_syntax(obj)))
    return(true);
  if (((full_typ & T_Full_Symcons) != 0) &&
      (!is_symbol(obj)) && (!is_any_procedure(obj)) && (!is_hash_table(obj)) && (!is_pair(obj)) && (!is_let(obj)) && (!is_t_vector(obj)))
    return(true);
  if (((full_typ & T_Full_Binder) != 0) &&
      (!is_pair(obj)) && (!is_hash_table(obj)) && (!is_normal_symbol(obj)) && (!is_c_function(obj)) && (!is_syntax(obj)))
    return(true);
  if (((full_typ & T_Full_Definer) != 0) &&
      (!is_normal_symbol(obj)) && (!is_c_function(obj)) && (!is_pair(obj)) && (!is_slot(obj)) && (!is_iterator(obj)) &&
      (!is_hash_table(obj)) && (!is_let(obj)) && (!is_syntax(obj)))
    return(true);
  if (((full_typ & T_Full_Has_Let_File) != 0) &&
      (!is_let(obj)) && (!is_t_vector(obj)) && (!is_hash_table(obj)) && (!is_c_function(obj)) &&
      (!is_slot(obj)) && (!is_pair(obj)) && (!is_closure_star(obj)))
    return(true);
  if (((full_typ & T_Safe_Stepper) != 0) &&
      (!is_let(obj)) && (!is_slot(obj)) && (!is_c_function(obj)) && (!is_string(obj)) &&
      (!is_pair(obj)) && (!is_hash_table(obj)) && (!is_any_macro(obj)) && (!is_symbol(obj)))
    return(true);
  if (((full_typ & T_Location) != 0) &&
      (!is_pair(obj)) && (!is_input_port(obj)) && (!is_let(obj)) && (!is_any_procedure(obj)) && (!is_slot(obj)))
    return(true);
  if (((full_typ & T_Mutable) != 0) &&
      (!is_number(obj)) && (!is_symbol(obj)) && (!is_let(obj)) && (!is_iterator(obj)) && (!is_slot(obj)) && (!is_let(obj)) && (!is_pair(obj)))
    return(true);
  if (((full_typ & T_Gensym) != 0) && (!is_slot(obj)) && (!is_any_closure(obj)) &&
      (!is_let(obj)) && (!is_symbol(obj)) && (!is_string(obj)) && (!is_hash_table(obj)) && (!is_pair(obj)) && (!is_any_vector(obj)))
    return(true);
  if (((full_typ & T_Full_Simple_Elements) != 0) &&
      (!is_t_vector(obj)) && (!is_hash_table(obj)) && (!is_normal_symbol(obj)) && (!is_pair(obj)) && (type_unchecked(obj) < T_C_MACRO))
    return(true);
  if (((full_typ & T_Has_Methods) != 0) &&
      (!is_let(obj)) && (!is_c_object(obj)) && (!is_any_closure(obj)) && (!is_any_macro(obj)) && (!is_c_pointer(obj)))
    return(true);
  if (((full_typ & T_Cyclic) != 0) && (!is_simple_sequence(obj)) && (!t_structure_p[type(obj)]) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_Cyclic_Set) != 0) && (!is_simple_sequence(obj)) && (!t_structure_p[type(obj)]) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_Full_Has_Fn) != 0) && (!is_pair(obj)) && (!is_symbol(obj)) && (!is_syntax(obj)) && (!is_c_function(obj))) return(true);
  if (((full_typ & T_Full_True_Is_Done) != 0) && (!is_pair(obj)) && (!is_symbol(obj)) && (!is_c_function(obj))) return(true);
  if (is_symbol(obj))
    {
      if (((block_size(symbol_info(obj))) & ~0xffffff) != 0) /* boolean function bool type, *s7*_let field id, symbol_catetgory */
	return(true);
      if ((uint8_t)(symbol_type(obj) & 0xff) >= Num_Types)
	return(true);
      if (symbol_category(obj) > 14) /* e_boolean see init_rootlet */
	return(true);
      if (starlet_symbol_id(obj) >= sl_Num_Fields)
	return(true);
    }
  if ((signed_type(obj) == 0) && ((full_typ & T_Gc_Mark) != 0)) return(true);
  if (!in_heap(obj))
    {
      uint8_t typ = type_unchecked(obj);
      if (never_unheaped[typ]) {fprintf(stderr, "unheap %s!\n", s7_type_names[typ]); print_gc_info(cur_sc, obj, __func__, __LINE__); return(true);}
    }
  /* all the hash_table bits seem to be compatible, symbols? (all_float/all_integer only apply to sc->divide_symbol et al at init time) */
  return(false);
}

void s7_show_let(s7_scheme *sc);
void s7_show_let(s7_scheme *sc) /* debugging convenience */
{
  for (s7_pointer let = sc->curlet; let; let = let_outlet(let))
    {
      if (let == sc->owlet)
	fprintf(stderr, "(owlet): ");
      else
	if (let == sc->rootlet)
	  fprintf(stderr, "(rootlet): ");
	else
	  if (is_funclet(let))
	    fprintf(stderr, "(%s funclet): ", display(funclet_function(let)));
	  else
	    if (let == sc->shadow_rootlet)
	      fprintf(stderr, "(shadow rootlet): ");
      fprintf(stderr, "%s\n", display(let));
    }
}

static const char *checked_type_name(s7_scheme *sc, int32_t typ)
{
  if ((typ >= 0) && (typ < Num_Types))
    {
      s7_pointer p = sc->type_names[typ];
      if (is_string(p)) return(string_value(p));
    }
  return("unknown type!");
}

#if REPORT_ROOTLET_REDEF
static void set_local_1(s7_scheme *sc, s7_pointer symbol, const char *func, int32_t line)
{
  if (is_defined_global(symbol))
    fprintf(stderr, "%s[%d]: %s%s%s in %s\n", func, line, Bold_Text, display(symbol), Unbold_Text, display_truncated(sc->cur_code));
  full_type(symbol) = (full_type(symbol) & ~(T_Dont_Eval_Args | T_Syntactic));
}
#endif

static char *object_raw_type_to_string(s7_pointer p)
{
  #define Raw_Bufsize 128
  char *buf = (char *)Malloc(Raw_Bufsize);
  snprintf(buf, Raw_Bufsize, "type: %d", type_unchecked(p));
  return(buf);
}

static void complain(s7_scheme *sc, const char *complaint, s7_pointer p, const char *func, int32_t line, uint8_t typ)
{
  char *pstr = object_raw_type_to_string(p);
  fprintf(stderr, complaint, Bold_Text, func, line, checked_type_name(sc, typ), pstr, Unbold_Text);
  free(pstr);
  abort();
}

static char *show_debugger_bits(s7_pointer p)
{
  #define Show_Bufsize 512
  char *bits_str = (char *)Malloc(Show_Bufsize);
  const s7_int bits = p->debugger_bits;
  snprintf(bits_str, Show_Bufsize, " %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
	   ((bits & Opt1_Set) != 0) ? " opt1_set" : "",
	   ((bits & Opt1_Fast) != 0) ? " opt1_fast" : "",
	   ((bits & Opt1_Cfunc) != 0) ? " opt1_cfunc" : "",
	   ((bits & Opt1_Clause) != 0) ? " opt1_clause" : "",
	   ((bits & Opt1_Lambda) != 0) ? " opt1_lambda" : "",
	   ((bits & Opt1_Sym) != 0) ? " opt1_sym" : "",
	   ((bits & Opt1_Pair) != 0) ? " opt1_pair" : "",
	   ((bits & Opt1_Con) != 0) ? " opt1_con" : "",
	   ((bits & Opt1_Any) != 0) ? " opt1_any" : "",
	   ((bits & Opt1_Hash) != 0) ? " opt1_hash" : "",

	   ((bits & Opt2_Set) != 0) ? " opt2_set" : "",
	   ((bits & Opt2_Key) != 0) ? " opt2_any" : "",
	   ((bits & Opt2_Slow) != 0) ? " opt2_slow" : "",
	   ((bits & Opt2_Sym) != 0) ? " opt2_sym" : "",
	   ((bits & Opt2_Pair) != 0) ? " opt2_pair" : "",
	   ((bits & Opt2_Con) != 0) ? " opt2_con" : "",
	   ((bits & Opt2_Fx) != 0) ? " opt2_fx" : "",
	   ((bits & Opt2_Fn) != 0) ? " opt2_fn" : "",
	   ((bits & Opt2_Lambda) != 0) ? " opt2_lambda" : "",
	   ((bits & Opt2_Direct) != 0) ? " opt2_direct" : "",
	   ((bits & Opt2_Name) != 0) ? " opt2_raw_name" : "",
	   ((bits & Opt2_Int) != 0) ? " opt2_int" : "",

	   ((bits & Opt3_Set) != 0) ? " opt3_set" : "",
	   ((bits & Opt3_Arglen) != 0) ? " opt3_arglen" : "",
	   ((bits & Opt3_Sym) != 0) ? " opt3_sym" : "",
	   ((bits & Opt3_Con) != 0) ? " opt3_con" : "",
	   ((bits & Opt3_And) != 0) ? " opt3_pair " : "",
	   ((bits & Opt3_Any) != 0) ? " opt3_any " : "",
	   ((bits & Opt3_Let) != 0) ? " opt3_let " : "",
	   ((bits & Opt3_Byte) != 0) ? " opt3_byte " : "",
	   ((bits & Opt3_Direct) != 0) ? " opt3_direct" : "",
	   ((bits & Opt3_Location) != 0) ? " opt3_location" : "",
	   ((bits & Opt3_Len) != 0) ? " opt3_len" : "",
	   ((bits & Opt3_Int) != 0) ? " opt3_int" : "",

	   ((bits & L_Hit) != 0) ? " let_set" : "",
	   ((bits & L_Func) != 0) ? " let_func" : "",
	   ((bits & L_Dox) != 0) ? " let_dox" : "");
  return(bits_str);
}

static s7_pointer check_ref_one(s7_pointer p, uint8_t expected_type, const char *func, int32_t line, const char *func1, const char *func2)
{
  if (!p)
    {fprintf(stderr, "%s%s[%d]: null pointer passed to check_ref_one%s\n", Bold_Text, func, line, Unbold_Text); abort();}
  {
    const uint8_t typ = type_unchecked(p);
    if (typ != expected_type)
      {
	if ((!func1) || (typ != T_FREE))
	  {
	    fprintf(stderr, "%s%s[%d]: not %s, but %s (%s)%s\n",
		    Bold_Text,
		    func, line, checked_type_name(cur_sc, expected_type), checked_type_name(cur_sc, typ), object_raw_type_to_string(p),
		    Unbold_Text);
	    abort();
	  }
	if ((strcmp(func, func1) != 0) &&
	    ((!func2) || (strcmp(func, func2) != 0)))
	  {fprintf(stderr, "%s%s[%d]: free cell, not %s%s\n", Bold_Text, func, line, checked_type_name(cur_sc, expected_type), Unbold_Text); abort();}
      }}
  return(p);
}

static void check_let_set_slots(s7_scheme *sc, s7_pointer let, s7_pointer slot, const char *func, int32_t line)
{
  if ((!in_heap(let)) && (slot) && (in_heap(slot))) fprintf(stderr, "%s[%d]: let+slot mismatch\n", func, line);
  if ((let == sc->rootlet) && (slot != Slot_End))
    {fprintf(stderr, "%s[%d]: setting rootlet slots!\n", func, line); abort();}
  T_Let(let)->object.let.slots = T_Sln(slot);
}

static s7_pointer check_let_ref(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_ref_one(p, T_LET, func, line, NULL, NULL);
  if ((p->debugger_bits & L_Hit) == 0) fprintf(stderr, "%s[%d]: let not set\n", func, line);
  if ((p->debugger_bits & L_Mask) != role) fprintf(stderr, "%s[%d]: let bad role\n", func, line);
  return(p);
}

static s7_pointer check_let_set(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_ref_one(p, T_LET, func, line, NULL, NULL);
  p->debugger_bits &= (~L_Mask);
  p->debugger_bits |= (L_Hit | role);
  return(p);
}

static s7_pointer check_ref_two(s7_pointer p, uint8_t expected_type, int32_t other_type, const char *func, int32_t line, const char *func1, const char *func2)
{
  if (!p)
    fprintf(stderr, "%s[%d]: null pointer passed to check_ref_two\n", func, line);
  else
    {
      uint8_t typ = type_unchecked(p);
      if ((typ != expected_type) && (typ != other_type))
	return(check_ref_one(p, expected_type, func, line, func1, func2));
    }
  return(p);
}

static s7_pointer check_ref_prf(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ != T_PAIR) && (p != cur_sc->F))
    complain(cur_sc, "%s%s[%d]: not a pair or #f, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_prt(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ != T_INPUT_PORT) && (typ != T_OUTPUT_PORT) && (typ != T_FREE))
    complain(cur_sc, "%s%s[%d]: not a port, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_pri(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ != T_INPUT_PORT) && (p != cur_sc->F))
    complain(cur_sc, "%s%s[%d]: not an input port or #f, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_pro(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ != T_OUTPUT_PORT) && (p != cur_sc->F))
    complain(cur_sc, "%s%s[%d]: not an output port or #f, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_vec(s7_pointer p, const char *func, int32_t line)
{
  if ((strcmp(func, "sweep") != 0) &&
      (strcmp(func, "process_multivector") != 0))
    {
      uint8_t typ = type_unchecked(p);
      if (!t_vector_p[typ]) complain(cur_sc, "%s%s[%d]: not a vector, but %s (%s)%s\n", p, func, line, typ);
    }
  return(p);
}

static s7_pointer check_ref_clo(s7_pointer p, const char *func, int32_t line)
{
  if (!p)
    fprintf(stderr, "%s[%d]: null pointer passed to check_ref_clo\n", func, line);
  else
    {
      uint8_t typ = type_unchecked(p);
      if (!t_has_closure_let[typ]) complain(cur_sc, "%s%s[%d]: not a closure, but %s (%s)%s\n", p, func, line, typ);
    }
  return(p);
}

static s7_pointer check_ref_cfn(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if (typ < T_C_FUNCTION_STAR) complain(cur_sc, "%s%s[%d]: not a c-function (type < T_C_FUNCTION_STAR, from T_CFn), but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_fnc(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if (typ < T_C_MACRO) complain(cur_sc, "%s%s[%d]: not a c-function or c-macro (type < T_C_MACRO, from T_Fnc), but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_num(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ < T_INTEGER) || (typ > T_COMPLEX))
    complain(cur_sc, "%s%s[%d]: not a number, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_seq(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((!t_sequence_p[typ]) && (!t_structure_p[typ]) && (!is_any_closure(p))) /* closure as iterator -- see s7test */
    complain(cur_sc, "%s%s[%d]: not a sequence or structure, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_met(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ != T_LET) && (typ != T_C_OBJECT) && (!is_any_closure(p)) && (!is_any_macro(p)) && (typ != T_C_POINTER))
    complain(cur_sc, "%s%s[%d]: not a possible method holder, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_arg(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((typ != T_PAIR) && (typ != T_NIL) && (typ != T_SYMBOL))
    complain(cur_sc, "%s%s[%d]: arglist is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_app(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ = type_unchecked(p);
  if ((!t_applicable_p[typ]) && (p != cur_sc->F))
    complain(cur_sc, "%s%s[%d]: applicable object is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_sln(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  if (is_slot_end(p)) return(p);
  typ = type_unchecked(p);
  if ((typ != T_SLOT) && (typ != T_UNDEFINED)) /* unset slots are #<undefined> */
    complain(cur_sc, "%s%s[%d]: slot is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_out(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  if (!p) return(NULL);
  typ = type_unchecked(p);
  if (typ != T_LET)
    complain(cur_sc, "%s%s[%d]: outlet is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref_svec(s7_pointer p, const char *func, int32_t line)
{
  if (!is_any_vector(p)) complain(cur_sc, "%s%s[%d]: subvector is %s (%s)%s?\n", p, func, line, type_unchecked(p));
  if (!is_subvector(p)) complain(cur_sc, "%s%s[%d]: subvector is %s (%s), but not a subvector?%s\n", p, func, line, type_unchecked(p));
  return(p);
}

static s7_pointer check_ref_prc(s7_pointer p, const char *func, int32_t line)
{
  if ((!is_any_procedure(p)) && (!is_boolean(p)))
    complain(cur_sc, "%s%s[%d]: setter is %s (%s)%s?\n", p, func, line, type_unchecked(p));
  return(p);
}

static void print_gc_info(s7_scheme *sc, s7_pointer obj, const char *func, int32_t line)
{
  if (!obj)
    fprintf(stderr, "[%d]: obj is %p\n", line, obj);
  else
    if (!is_free(obj))
      fprintf(stderr, "%s from %s[%d]: %p type is %d?\n", __func__, func, line, obj, type_unchecked(obj));
    else
      {
	const s7_int free_type = full_type(obj);
	char *bits;
	#define Info_Bufsize 128
	char fline[Info_Bufsize];
	full_type(obj) = obj->alloc_type; /* not set_full_type here!  it clobbers existing alloc/free info */
	sc->printing_gc_info = true;
	bits = describe_type_bits(sc, obj); /* this func called in type macro */
	sc->printing_gc_info = false;
	full_type(obj) = free_type;
	if (obj->explicit_free_line > 0)
	  snprintf(fline, Info_Bufsize, ", freed at %d, ", obj->explicit_free_line);
	fprintf(stderr, "%s%p is free (%s[%d], alloc type: %s %" ld64 " #x%" PRIx64 " (%s)), alloc: %s[%d], %sgc: %s[%d], uses: %d%s",
		Bold_Text, obj, func, line, s7_type_names[obj->alloc_type & 0xff], obj->alloc_type, obj->alloc_type,
		bits, obj->alloc_func, obj->alloc_line,
		(obj->explicit_free_line > 0) ? fline : "", obj->gc_func, obj->gc_line,	obj->uses, Unbold_Text);
	fprintf(stderr, "\n");
	free(bits);
      }
  abort();
}

static s7_pointer check_nref(s7_pointer p, const char *func, int32_t line)
{
  if (!p)
    {fprintf(stderr, "%s%s[%d]: null pointer!%s\n", Bold_Text, func, line, Unbold_Text); abort();}
  if (type_unchecked(p) >= Num_Types)
    {fprintf(stderr, "%s%s[%d]: attempt to use messed up cell (type: %d)%s\n", Bold_Text, func, line, type_unchecked(p), Unbold_Text); abort();}
  if (is_free(p))
    {
      fprintf(stderr, "%s%s[%d]: attempt to use free cell%s\n", Bold_Text, func, line, Unbold_Text);
      print_gc_info(cur_sc, p, func, line);
      abort();
    }
  return(p);
}

static s7_pointer check_ref_nmv(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  check_nref(p, func, line);
  typ = type_unchecked(p);  /* must follow check_nref -- p might be NULL */
  if ((is_multiple_value(p)) &&
      (!safe_strcmp(func, "mark_slot"))) /* match == multiple-values which causes false error messages */
    complain(cur_sc, "%s%s[%d]: slot value is a multiple-value, %s (%s)%s?\n", p, func, line, typ);
  if (has_odd_bits(p))
    {char *str; fprintf(stderr, "%s[%d]: odd bits: %s\n", __func__, __LINE__, str = describe_type_bits(cur_sc, p)); free(str);}
  if (t_exs_p[typ]) /* can be t_slot, make_funclet[81614] */
    {fprintf(stderr, "%s%s[%d]: slot_value is %s?%s\n", Bold_Text, func, line, s7_type_names[typ], Unbold_Text); abort();}
  return(p);
}

static s7_pointer check_ref_mac(s7_pointer p, const char *func, int32_t line)
{
  if ((!is_any_macro(p)) || (is_c_macro(p)))
    complain(cur_sc, "%s%s[%d]: macro is %s (%s)%s?\n", p, func, line, type_unchecked(p));
  return(p);
}

static s7_pointer check_ref_key(s7_pointer p, const char *func, int32_t line)
{
  if (!is_symbol_and_keyword(p))
    complain(cur_sc, "%s%s[%d]: not a keyword: %s (%s)%s?\n", p, func, line, type_unchecked(p));
  if (strcmp(func, "new_symbol") != 0)
    {
      if (global_value(p) != p)
	{
	  fprintf(stderr, "%s%s[%d]: keyword %s value is not itself (type: %s)%s\n",
		  Bold_Text, func, line, display(p), s7_type_names[type_unchecked(global_value(p))], Unbold_Text);
	  abort();
	}
      if (in_heap(keyword_symbol_unchecked(p)))
	fprintf(stderr, "%s%s[%d]: keyword %s symbol is in the heap%s\n", Bold_Text, func, line, display(p), Unbold_Text);
      if (has_odd_bits(p))
	{char *str; fprintf(stderr, "odd bits: %s\n", str = describe_type_bits(cur_sc, p)); free(str);}
    }
  return(p);
}

static s7_pointer check_ref_ext(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  check_nref(p, func, line);
  typ = type_unchecked(p);
  if (t_ext_p[typ])
    {fprintf(stderr, "%s%s[%d]: attempt to use (internal) %s cell%s\n", Bold_Text, func, line, s7_type_names[typ], Unbold_Text); abort();}
  return(p);
}

static s7_pointer check_ref_exs(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  check_nref(p, func, line);
  typ = type_unchecked(p);
  if (t_exs_p[typ])
    {fprintf(stderr, "%s%s[%d]: attempt to use (internal) %s cell%s\n", Bold_Text, func, line, s7_type_names[typ], Unbold_Text); abort();}
  return(p);
}

static s7_pointer check_opcode(s7_pointer p, const char *func, int32_t line)
{
  s7_int op = (s7_int)(intptr_t)p;
  if ((op < 0) || (op >= Num_Ops))
    {fprintf(stderr, "%s%s[%d]: opcode_t: %" ld64 " == %p?%s\n", Bold_Text, func, line, op, p, Unbold_Text); abort();}
  return(p);
}

static void checked_set_cdr(s7_pointer p, s7_pointer val, const char *func, int32_t line)
{
  if ((is_immutable(p)) && (!in_heap(p)))
    {fprintf(stderr, "%s[%d]: set-cdr! target is immutable and not in the heap, %p %s\n", func, line, p, display(p)); abort();}
  if ((!in_heap(p)) && (in_heap(val)))
    fprintf(stderr, "%s[%d]: set-cdr! target is not in the heap, but the new value is, %p %s, %p %s\n", func, line, p, display(p), val, display(val));
  cdr(p) = val;
}

static const char *opt1_role_name(s7_uint role)
{
  if (role == Opt1_Fast) return("opt1_fast");
  if (role == Opt1_Cfunc) return("opt1_cfunc");
  if (role == Opt1_Lambda) return("opt1_lambda");
  if (role == Opt1_Clause) return("opt1_clause");
  if (role == Opt1_Sym) return("opt1_sym");
  if (role == Opt1_Pair) return("opt1_pair");
  if (role == Opt1_Con) return("opt1_con");
  if (role == Opt1_Any) return("opt1_any");
  return((role == Opt1_Hash) ? "opt1_hash" : "opt1_unknown");
}

static const char *opt2_role_name(s7_uint role)
{
  if (role == Opt2_Fx) return("opt2_fx");
  if (role == Opt2_Fn) return("opt2_fn");
  if (role == Opt2_Key) return("opt2_any");
  if (role == Opt2_Slow) return("opt2_slow");
  if (role == Opt2_Sym) return("opt2_sym");
  if (role == Opt2_Pair) return("opt2_pair");
  if (role == Opt2_Con) return("opt2_con");
  if (role == Opt2_Lambda) return("opt2_lambda");
  if (role == Opt2_Direct) return("opt2_direct");
  if (role == Opt2_Int) return("opt2_int");
  return((role == Opt2_Name) ? "opt2_raw_name" : "opt2_unknown");
}

static const char *opt3_role_name(s7_uint role)
{
  if (role == Opt3_Arglen) return("opt3_arglen");
  if (role == Opt3_Sym) return("opt3_sym");
  if (role == Opt3_Con) return("opt3_con");
  if (role == Opt3_And) return("opt3_pair");
  if (role == Opt3_Any) return("opt3_any");
  if (role == Opt3_Let) return("opt3_let");
  if (role == Opt3_Byte) return("opt3_byte");
  if (role == Opt3_Direct) return("direct_opt3");
  if (role == Opt3_Len) return("opt3_len");
  if (role == Opt3_Int) return("opt3_int");
  return((role == Opt3_Location) ? "opt3_location" : "opt3_unknown");
}

static void show_opt1_bits(s7_pointer p, const char *func, int32_t line, s7_uint role)
{
  char *bits = show_debugger_bits(p);
  fprintf(stderr, "%s%s[%d]%s: opt1: %p->%p wants %s, debugger bits are #x%" PRIx64 "%s but expects #x%" PRIx64,
	  Bold_Text, func, line, Unbold_Text,
	  p, p->object.cons.opt1, opt1_role_name(role), p->debugger_bits, bits, (s7_int)role);
  free(bits);
}

static s7_pointer opt1_1(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  if ((!opt1_is_set(p)) ||
      ((!opt1_role_matches(p, role)) &&
       (role != Opt1_Any)))
    {
      show_opt1_bits(p, func, line, role);
      abort();
    }
  return(p->object.cons.opt1);
}

static void base_opt1(s7_pointer p, s7_uint role)
{
  set_opt1_role(p, role);
  set_opt1_is_set(p);
}

static s7_pointer set_opt1_1(s7_pointer p, s7_pointer x, s7_uint role, const char *func, int32_t line)
{
  if (((p->debugger_bits & Opt1_Mask) != role) &&
      ((p->debugger_bits & Opt1_Mask) == Opt1_Lambda) &&
      (role != Opt1_Cfunc))
    fprintf(stderr, "%s[%d]: opt1_lambda -> %s, op: %s, x: %s,\n    %s\n",
	    func, line, opt1_role_name(role),
	    (is_optimized(x)) ? op_names[optimize_op(x)] : "unopt",
	    display(x), display(p));
  p->object.cons.opt1 = x;
  base_opt1(p, role);
  return(x);
}

static s7_uint opt1_hash_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt1_is_set(p)) || (!opt1_role_matches(p, Opt1_Hash)))
    {
      show_opt1_bits(p, func, line, (s7_uint)Opt1_Hash);
      abort();
    }
  return(p->object.sym_cons.hash);
}

static void set_opt1_hash_1(s7_pointer p, s7_uint x)
{
  p->object.sym_cons.hash = x;
  base_opt1(p, Opt1_Hash);
}

static void show_opt2_bits(s7_pointer p, const char *func, int32_t line, s7_uint role)
{
  char *bits = show_debugger_bits(p);
  fprintf(stderr, "%s%s[%d]%s: %s opt2: %p->%p wants %s, debugger bits are #x%" PRIx64 "%s but expects #x%" PRIx64 " %s",
	  Bold_Text, func, line, Unbold_Text,
	  display(p), p, p->object.cons.o2.opt2, opt2_role_name(role), p->debugger_bits, bits, (s7_int)role, opt2_role_name(role));
  free(bits);
}

static bool f_call_func_mismatch(const char *func)
{
  return((!safe_strcmp(func, "check_and")) &&  /* these reflect set_fx|unchecked where the destination checks for null fx_proc */
	 (!safe_strcmp(func, "check_or")) &&
	 (!safe_strcmp(func, "eval")) &&
	 (!safe_strcmp(func, "set_any_c_np")) &&
	 (!safe_strcmp(func, "set_any_closure_np")) &&
	 (!safe_strcmp(func, "optimize_func_two_args")) &&
	 (!safe_strcmp(func, "optimize_func_many_args")) &&
	 (!safe_strcmp(func, "optimize_func_three_args")) &&
	 (!safe_strcmp(func, "fx_c_ff")) &&
	 (!safe_strcmp(func, "op_map_for_each_fa")) &&
	 (!safe_strcmp(func, "op_map_for_each_faa")));
}

static void check_opt2_bits(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  if (!p)
    {fprintf(stderr, "%s%s[%d]: opt2 null!\n%s", Bold_Text, func, line, Unbold_Text); abort();}
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, role)))
    {
      show_opt2_bits(p, func, line, role);
      abort();
    }
}

static s7_pointer opt2_1(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_opt2_bits(p, role, func, line);
  return(p->object.cons.o2.opt2);
}

static s7_int opt2_n_1(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_opt2_bits(p, role, func, line);
  return(p->object.cons.o2.n);
}

static void base_opt2(s7_pointer p, s7_uint role)
{
  set_opt2_role(p, role);
  set_opt2_is_set(p);
}

static void set_opt2_1(s7_scheme *sc, s7_pointer p, s7_pointer x, s7_uint role, const char *func, int32_t line)
{
  if ((role == Opt2_Fx) &&
      (x == NULL) &&
      (f_call_func_mismatch(func)))
    fprintf(stderr, "%s[%d]: set fx_proc for %s to null (%s%s%s)\n", func, line,
	    string_value(object_to_string_truncated(sc, p)),
	    ((is_h_optimized(car(p))) && (is_safe_c_op(optimize_op(car(p))))) ? Bold_Text : "",
	    op_names[optimize_op(car(p))],
	    ((is_h_optimized(car(p))) && (is_safe_c_op(optimize_op(car(p))))) ? Unbold_Text : "");
  if ((role != Opt2_Fx) && (role != Opt2_Direct) && (has_fx(p))) /* sometimes opt2_direct just specializes fx */
    {fprintf(stderr, "%s[%d]: overwrite has_fx: %s %s\n", func, line, opt2_role_name(role), display_truncated(p)); abort();}
  if ((role != Opt2_Fn) && (has_fn(p)))
    {fprintf(stderr, "%s[%d]: overwrite has_fn: %s %s\n", func, line, opt2_role_name(role), display_truncated(p)); abort();}
  p->object.cons.o2.opt2 = x;
  base_opt2(p, role);
}

static void set_opt2_n_1(s7_scheme *unused_sc, s7_pointer p, s7_int x, s7_uint role, const char *unused_func, int32_t unused_line)
{
  p->object.cons.o2.n = x;
  base_opt2(p, role);
}

static const char *opt2_name_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, Opt2_Name)))
    {
      show_opt2_bits(p, func, line, (s7_uint)Opt2_Name);
      abort();
    }
  return(p->object.sym_cons.fstr);
}

static void set_opt2_name_1(s7_pointer p, const char *str)
{
  p->object.sym_cons.fstr = str;
  base_opt2(p, Opt2_Name);
}

static void show_opt3_bits(s7_pointer p, const char *func, int32_t line, s7_uint role)
{
  char *bits = show_debugger_bits(p);
  fprintf(stderr, "%s%s[%d]%s: opt3: %s #x%" PRIx64 "%s", Bold_Text, func, line, Unbold_Text, opt3_role_name(role), p->debugger_bits, bits);
  free(bits);
}

static void check_opt3_bits(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  if (!p)
    {fprintf(stderr, "%s%s[%d]: opt3 null!\n%s", Bold_Text, func, line, Unbold_Text); abort();}
  if ((!opt3_is_set(p)) ||
      (!opt3_role_matches(p, role)))
    {
      show_opt3_bits(p, func, line, role);
      abort();
    }
}

static s7_pointer opt3_1(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_opt3_bits(p, role, func, line);
  return(p->object.cons.o3.opt3);
}

static s7_int opt3_n_1(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_opt3_bits(p, role, func, line);
  return(p->object.cons.o3.n);
}

static void base_opt3(s7_pointer p, s7_uint role)
{
  set_opt3_role(p, role);
  set_opt3_is_set(p);
}

static void set_opt3_1(s7_pointer p, s7_pointer x, s7_uint role)
{
  clear_type_bit(p, T_Location);
  p->object.cons.o3.opt3 = x;
  base_opt3(p, role);
}

static void set_opt3_n_1(s7_pointer p, s7_int x, s7_uint role)
{
  clear_type_bit(p, T_Location);
  p->object.cons.o3.n = x;
  base_opt3(p, role);
}

static uint8_t opt3_byte_1(s7_pointer p, s7_uint role, const char *func, int32_t line)
{
  check_opt3_bits(p, role, func, line);
  return(p->object.cons.o3.opt_type);
}

static void set_opt3_byte_1(s7_pointer p, uint8_t x, s7_uint role, const char *unused_func, int32_t unused_line)
{
  clear_type_bit(p, T_Location);
  p->object.cons.o3.opt_type = x;
  base_opt3(p, role);
}

static s7_uint opt3_location_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      ((p->debugger_bits & Opt3_Location) == 0) ||
      (!has_location(p)))
    {
      show_opt3_bits(p, func, line, (s7_uint)Opt3_Location);
      abort();
    }
  return(p->object.sym_cons.location); /* don't use pair_location macro here or below (infinite recursion if s7_Debugging via opt3_location_1) */
}

static void set_opt3_location_1(s7_pointer p, s7_uint x)
{
  p->object.sym_cons.location = x;
  (p)->debugger_bits = (Opt3_Location | (p->debugger_bits & ~Opt3_Len)); /* turn on line, cancel len */
  set_opt3_is_set(p);
}

static s7_uint opt3_len_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      ((p->debugger_bits & Opt3_Len) == 0) ||
      (has_location(p)))
    {
      show_opt3_bits(p, func, line, (s7_uint)Opt3_Len);
      abort();
    }
  return(p->object.sym_cons.location);
}

static void set_opt3_len_1(s7_pointer p, s7_uint x)
{
  clear_type_bit(p, T_Location);
  p->object.sym_cons.location = x;
  (p)->debugger_bits = (Opt3_Len | (p->debugger_bits & ~(Opt3_Location)));
  set_opt3_is_set(p);
}

static void print_debugging_state(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  /* show current state, current allocated state */
  char *allocated_bits, *str;
  const s7_int save_full_type = full_type(obj);
  s7_int len, nlen;
  const char *excl_name = (is_free(obj)) ? "free cell!" : "unknown object!";
  block_t *b;
  char *current_bits = describe_type_bits(sc, obj);

  set_full_type(obj, obj->alloc_type);
  allocated_bits = describe_type_bits(sc, obj);
  set_full_type(obj, save_full_type);

  len = safe_strlen(excl_name) + safe_strlen(current_bits) + safe_strlen(allocated_bits) + safe_strlen(obj->alloc_func) + 512;
  b = mallocate(sc, len);
  str = (char *)block_data(b);
  nlen = snprintf(str, len, "\n<%s %s,\n  alloc: %s[%d] %s, uses: %d>", excl_name, current_bits, obj->alloc_func, obj->alloc_line, allocated_bits, obj->uses);
  free(current_bits);
  free(allocated_bits);
  if (is_null(port))
    fprintf(stderr, "%s[%d]: %p: %s\n", __func__, __LINE__, obj, str);
  else port_write_string(port)(sc, str, clamp_length(nlen, len), port);
  liberate(sc, b);
}

static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int32_t line, const char *func)
{
  if (!p) /* the result of lookup_1 */
    {
      const s7_pointer slot = symbol_to_local_slot(sc, sym, sc->curlet);
      char *str = describe_type_bits(sc, sym);
      fprintf(stderr, "%s%s[%d]: %s unbound%s\n", Bold_Text, func, line, symbol_name(sym), Unbold_Text);
      fprintf(stderr, "  symbol_id: %" ld64 ", let_id: %" ld64 ", %s", symbol_id(sym), let_id(sc->curlet), str);
      free(str);
      if (is_slot(slot)) fprintf(stderr, ", slot: %s", display(slot));
      fprintf(stderr, "\n");
      abort();
    }
  return(p);
}
#endif /* s7_Debugging */


/* -------- wrappers -------- */

static s7_pointer wrap_mutable_integer(s7_scheme *sc, s7_int x) /* wrap_integer without small_int possibility -- usable as a mutable integer for example */
{
  /* using circular lists here is much faster than using an array of s7_cells and an integer location */
  s7_pointer wrapped_int = car(sc->integer_wrappers);
#if s7_Debugging
  if ((full_type(wrapped_int) & (~T_Gc_Mark)) != (T_INTEGER | T_Unheap))
    fprintf(stderr, "%s[%d]: %s\n", __func__, __LINE__, describe_type_bits(sc, wrapped_int));
  sc->integer_wrapper_allocs++;
#endif
  set_integer(wrapped_int, x);
  sc->integer_wrappers = cdr(sc->integer_wrappers);
  return(wrapped_int);
}

static s7_pointer wrap_integer(s7_scheme *sc, s7_int x)
{
  if (is_small_int(x)) return(small_int(x));
  return(wrap_mutable_integer(sc, x));
}

static s7_pointer wrap_real(s7_scheme *sc, s7_double x)
{
  s7_pointer wrapped_real = car(sc->real_wrappers);
#if s7_Debugging
  if ((full_type(wrapped_real) & (~T_Gc_Mark)) != (T_REAL | T_Unheap))
    fprintf(stderr, "%s[%d]: %s\n", __func__, __LINE__, describe_type_bits(sc, wrapped_real));
  sc->real_wrapper_allocs++;
#endif
  set_real(wrapped_real, x);
  sc->real_wrappers = cdr(sc->real_wrappers);
  return(wrapped_real);
}

#if !With_Gmp
static s7_pointer wrap_complex(s7_scheme *sc, s7_double rl, s7_double im)
{
  s7_pointer wrapped_complex = car(sc->complex_wrappers);
#if s7_Debugging
  if ((full_type(wrapped_complex) & (~T_Gc_Mark)) != (T_COMPLEX | T_Unheap))
    fprintf(stderr, "%s[%d]: %s\n", __func__, __LINE__, describe_type_bits(sc, wrapped_complex));
  sc->complex_wrapper_allocs++;
#endif
  set_real_part(wrapped_complex, rl);
  set_imag_part(wrapped_complex, im);
  sc->complex_wrappers = cdr(sc->complex_wrappers);
  return(wrapped_complex);
}

static s7_pointer wrap_real_or_complex(s7_scheme *sc, s7_double rl, s7_double im)
{
  if (im == 0.0) return(wrap_real(sc, rl));
  return(wrap_complex(sc, rl, im));
}
#else
#define wrap_complex(Sc, A, B) make_complex(Sc, A, B)
#define wrap_real_or_complex(Sc, A, B) make_complex(Sc, A, B)
#endif

static s7_pointer wrap_let(s7_scheme *sc, s7_pointer old_let)
{
  s7_pointer wrapped_let = car(sc->let_wrappers);
#if s7_Debugging
  if ((full_type(wrapped_let) & (~T_Gc_Mark)) != (T_LET | T_Safe_Procedure | T_Unheap))
    fprintf(stderr, "%s[%d]: %s\n", __func__, __LINE__, describe_type_bits(sc, wrapped_let));
  sc->let_wrapper_allocs++;
#endif
  let_set_id(wrapped_let, ++sc->let_number);
  let_set_slots(wrapped_let, Slot_End);
  let_set_outlet(wrapped_let, old_let);
  sc->let_wrappers = cdr(sc->let_wrappers);
  return(wrapped_let);
}

static s7_pointer wrap_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  s7_pointer wrapped_slot = car(sc->slot_wrappers);
#if s7_Debugging
  sc->slot_wrapper_allocs++;
#endif
  slot_set_symbol_and_value(wrapped_slot, symbol, value);
  sc->slot_wrappers = cdr(sc->slot_wrappers);
  return(wrapped_slot);
}

/* -------- prebuilt lists -------- */
static s7_pointer set_elist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->elist_1, x1);
  return(sc->elist_1);
}

static s7_pointer set_elist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->elist_2, x1);
  set_cadr(sc->elist_2, x2);
  return(sc->elist_2);
}

static s7_pointer set_elist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p = sc->elist_3;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(sc->elist_3);
}

static s7_pointer set_elist_4(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p = sc->elist_4;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(sc->elist_4);
}

static s7_pointer set_elist_5(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5)
{
  set_car(sc->elist_5, x1);
  set_elist_4(sc, x2, x3, x4, x5);
  return(sc->elist_5);
}

static s7_pointer set_elist_6(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5, s7_pointer x6)
{
  set_car(sc->elist_6, x1);
  set_elist_5(sc, x2, x3, x4, x5, x6);
  return(sc->elist_6);
}

static s7_pointer set_elist_7(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5, s7_pointer x6, s7_pointer x7)
{
  set_car(sc->elist_7, x1);
  set_elist_6(sc, x2, x3, x4, x5, x6, x7);
  return(sc->elist_7);
}

static s7_pointer set_wlist_3(s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(lst);
}

static s7_pointer set_wlist_4(s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(lst);
}

static s7_pointer set_mlist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->mlist_1, x1);
  return(sc->mlist_1);
}

static s7_pointer set_mlist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2) /* mlist_3 saves 3 in tmock -- see ~/old/s7-mlist_3.c */
{
  set_car(sc->mlist_2, x1);
  set_cadr(sc->mlist_2, x2);
  return(sc->mlist_2);
}

static s7_pointer set_plist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->plist_1, x1);
  return(sc->plist_1);
}

static s7_pointer set_plist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->plist_2, x1);
  set_car(sc->plist_2_2, x2);
  return(sc->plist_2);
}

static s7_pointer set_plist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  return(set_wlist_3(sc->plist_3, x1, x2, x3));
}

static s7_pointer set_plist_4(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  return(set_wlist_4(sc->plist_4, x1, x2, x3, x4));
}

static s7_pointer set_qlist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2) /* let_ref_fallback */
{
  set_car(sc->qlist_2, x1);
  set_cadr(sc->qlist_2, x2);
  return(sc->qlist_2);
}

static s7_pointer set_qlist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3) /* let_set_fallback */
{
  return(set_wlist_3(sc->qlist_3, x1, x2, x3));
}

static s7_pointer set_clist_1(s7_scheme *sc, s7_pointer x1) /* for c_object length method etc, a "weak" list */
{
  set_car(sc->clist_1, x1);
  return(sc->clist_1);
}

static s7_pointer set_clist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2) /* for c_object equal method etc, a "weak" list */
{
  set_car(sc->clist_2, x1);
  set_cadr(sc->clist_2, x2);
  return(sc->clist_2);
}

static s7_pointer set_dlist_1(s7_scheme *sc, s7_pointer x1) /* another like clist: temp usage, "weak" (not gc_marked), but semipermanent list */
{
  set_car(sc->dlist_1, x1);
  return(sc->dlist_1);
}

static s7_pointer set_ulist_1(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->u1_1, x1);
  set_cdr_unchecked(sc->u1_1, x2);
  return(sc->u1_1);
}


/* ---------------- error handlers ---------------- */
static const char *make_type_name(s7_scheme *sc, const char *name, article_t article)
{
  s7_int i;
  const s7_int slen = safe_strlen(name);
  const s7_int len = slen + 8;
  if (len > sc->typnam_len)
    {
      if (sc->typnam) free(sc->typnam);
      sc->typnam = (char *)Malloc(len);
      sc->typnam_len = len;
    }
  if (article == Indefinite_Article)
    {
      i = 1;
      sc->typnam[0] = 'a';
      if ((name[0] == 'a') || (name[0] == 'e') || (name[0] == 'i') || (name[0] == 'o') || (name[0] == 'u'))
	sc->typnam[i++] = 'n';
      sc->typnam[i++] = ' ';
    }
  else i = 0;
  memcpy((void *)(sc->typnam + i), (const void *)name, slen);
  sc->typnam[i + slen] = '\0';
  return(sc->typnam);
}

static const char *type_name_from_type(int32_t typ, article_t article)
{
  static const char *articled[] =
    {"a free cell", "a pair", "nil", "the unused object", "an undefined object", "the unspecified object", "the end-of-file object",
     "boolean", "a character", "syntactic", "a symbol", "an integer", "a ratio", "a real", "a complex number", "a big integer",
     "a big ratio", "a big real", "a big complex number", "a string", "a c-object", "a vector", "an int-vector", "a float-vector",
     "a byte-vector", "a complex-vector", "a catch", "a dynamic-wind", "a hash-table", "a let", "an iterator", "a stack",
     "an internal-counter", "a slot (variable binding)", "a c-pointer", "an output-port", "an input-port", "a random-state",
     "a continuation", "a goto (from call-with-exit)", "a function", "a function*", "a macro", "a macro*", "a bacro", "a bacro*",
     "a c-macro", "a c-function*", "a c-function", "a c-function"};

  static const char *unarticled[] =
    {"free-cell", "pair", "nil", "#<unused>", "#<undefined>", "#<unspecified>", "#<eof>", "boolean", "character", "syntax", "symbol",
     "integer", "ratio", "real", "complex-number", "big-integer", "big-ratio", "big-real", "big-complex-number", "string", "c-object",
     "vector", "int-vector", "float-vector", "byte-vector", "complex-vector", "catch", "dynamic-wind", "hash-table", "let", "iterator",
     "stack", "internal-counter", "slot", "c-pointer", "output-port", "input-port", "random-state", "continuation", "goto",
     "function", "function*", "macro", "macro*", "bacro", "bacro*", "c-macro", "c-function*", "c-function", "c-function"};

  if (typ >= Num_Types) return(NULL);
  return((article == No_Article) ? unarticled[typ] : articled[typ]);
}

static s7_pointer find_let(s7_scheme *sc, s7_pointer obj)
{
  if ((s7_Debugging) && (is_let(obj))) {fprintf(stderr, "let passed to find_let: %s\n", display(obj)); abort();}
  if (has_closure_let(obj)) return(closure_let(obj)); /* some of these are immutable -- they hold the parameter names */
  switch (type(obj))
    {
    case T_C_OBJECT:
      if (is_let(c_object_let(obj))) return(c_object_let(obj));
      return(sc->rootlet);
    case T_C_POINTER:
      if (is_let(c_pointer_info(obj))) return(c_pointer_info(obj));
      return(sc->rootlet);
    case T_C_MACRO: case T_C_FUNCTION_STAR: case T_C_FUNCTION: case T_C_RST_NO_REQ_FUNCTION:
      return(c_function_let(obj));
    }
  return(sc->nil);
}

s7_pointer s7_function_let(s7_scheme *sc, s7_pointer obj) {return(c_function_let(obj));}

static inline s7_pointer lookup_slot_from(s7_pointer symbol, s7_pointer let);

static s7_pointer find_method(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  s7_pointer slot;
  if (is_global(symbol)) /* this means the symbol has never been bound locally, so how can it be a method? */
    return(sc->undefined);
  slot = lookup_slot_from(symbol, let);
  if (slot != global_slot(symbol))
    return(slot_value(slot));
  return(sc->undefined);
}

static s7_pointer find_method_with_let(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  if (!is_let(let)) let = find_let(sc, let);
  return(find_method(sc, let, symbol));
}

static s7_pointer find_method_with_c_object(s7_scheme *sc, s7_pointer c_obj, s7_pointer symbol)
{
  s7_pointer let = c_object_let(c_obj);
  return(find_method(sc, (is_let(let)) ? let : sc->rootlet, symbol));
}

static const char *type_name(s7_scheme *sc, s7_pointer arg, article_t article)
{
  switch (type_unchecked(arg))
    {
    case T_C_OBJECT:    return(make_type_name(sc, string_value(c_object_scheme_name(sc, arg)), article));
    case T_INPUT_PORT:  return(make_type_name(sc, (is_file_port(arg)) ? "input file port" : ((is_string_port(arg)) ? "input string port" : "input port"), article));
    case T_OUTPUT_PORT: return(make_type_name(sc, (is_file_port(arg)) ? "output file port" : ((is_string_port(arg)) ? "output string port" : "output port"), article));
    case T_LET:
      if (has_active_methods(sc, arg))
	{
	  s7_pointer class_name = find_method(sc, arg, sc->class_name_symbol);
	  if (is_symbol(class_name))
	    return(make_type_name(sc, symbol_name(class_name), article));
	}
    default:
      {
	const char *str = type_name_from_type(type_unchecked(arg), article);
	if (str) return(str);
      }}
  return("messed up object");
}

static s7_pointer object_type_name(s7_scheme *sc, s7_pointer obj) /* used only by the error handlers */
{
  uint8_t typ;
  if (has_active_methods(sc, obj))
    {
      s7_pointer func = find_method_with_let(sc, obj, sc->class_name_symbol);
      if (func != sc->undefined)
	return(s7_apply_function(sc, func, set_plist_1(sc, obj)));
      if (is_symbol(func))
	return(symbol_name_cell(func));
    }
  typ = type(obj);
  if (typ < Num_Types)
    {
      if (typ == T_C_OBJECT) return(c_object_scheme_name(sc, obj));
      return(sc->type_names[typ]);
    }
  return(wrap_string(sc, "unknown type!", 13));
}

static s7_pointer type_name_string(s7_scheme *sc, s7_pointer arg)
{
  if (type(arg) < Num_Types)
    {
      s7_pointer p = sc->type_names[type(arg)]; /* these use Indefinite_Article */
      if (is_string(p)) return(p);
    }
  return(s7_make_string_wrapper(sc, type_name(sc, arg, Indefinite_Article)));
}


static no_return void sole_arg_wrong_type_error_nr(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer typ)
{
  set_wlist_4(cdr(sc->sole_arg_wrong_type_info), caller, arg, object_type_name(sc, arg), typ);
  error_nr(sc, sc->wrong_type_arg_symbol, sc->sole_arg_wrong_type_info);
}

static /* Inline */ no_return void wrong_type_error_nr(s7_scheme *sc, s7_pointer caller, s7_int arg_num, s7_pointer arg, s7_pointer typ)
{
  s7_pointer p = cdr(sc->wrong_type_arg_info);  /* info list is '(format_string caller arg_n arg type_name descr) */
  set_car(p, caller);                    p = cdr(p);
  set_car(p, (is_small_int(arg_num)) ? small_int(arg_num) : wrap_integer(sc, arg_num)); p = cdr(p);
  set_car(p, arg);                       p = cdr(p);
  set_car(p, object_type_name(sc, arg)); p = cdr(p);
  set_car(p, typ);
  error_nr(sc, sc->wrong_type_arg_symbol, sc->wrong_type_arg_info);
}

s7_pointer s7_wrong_type_arg_error(s7_scheme *sc, const char *caller, s7_int arg_n, s7_pointer arg, const char *descr)
{
  if (arg_n > 0)
    wrong_type_error_nr(sc, wrap_string(sc, caller, safe_strlen(caller)), arg_n, arg, wrap_string(sc, descr, safe_strlen(descr)));
  sole_arg_wrong_type_error_nr(sc, wrap_string(sc, caller, safe_strlen(caller)), arg, wrap_string(sc, descr, safe_strlen(descr)));
  return(sc->wrong_type_arg_symbol);
}

s7_pointer s7_wrong_type_error(s7_scheme *sc, s7_pointer caller, s7_int arg_n, s7_pointer arg, s7_pointer descr)
{
  if (arg_n > 0) wrong_type_error_nr(sc, caller, arg_n, arg, descr);
  sole_arg_wrong_type_error_nr(sc, caller, arg, descr);
  return(sc->wrong_type_arg_symbol); /* never happens */
}

static no_return void sole_arg_out_of_range_error_nr(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer descr)
{
  set_wlist_3(cdr(sc->sole_arg_out_of_range_info), caller, arg, descr);
  error_nr(sc, sc->out_of_range_symbol, sc->sole_arg_out_of_range_info);
}

static no_return void out_of_range_error_nr(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer descr)
{
  set_wlist_4(cdr(sc->out_of_range_info), caller, arg_n, arg, descr);
  error_nr(sc, sc->out_of_range_symbol, sc->out_of_range_info);
}

s7_pointer s7_out_of_range_error(s7_scheme *sc, const char *caller, s7_int arg_n, s7_pointer arg, const char *descr)
{
  if (arg_n > 0)
    {
      set_wlist_4(cdr(sc->out_of_range_info), wrap_string(sc, caller, safe_strlen(caller)),
		  wrap_integer(sc, arg_n), arg, wrap_string(sc, descr, safe_strlen(descr)));
      error_nr(sc, sc->out_of_range_symbol, sc->out_of_range_info);
    }
  set_wlist_3(cdr(sc->sole_arg_out_of_range_info), wrap_string(sc, caller, safe_strlen(caller)),
	      arg, wrap_string(sc, descr, safe_strlen(descr)));
  error_nr(sc, sc->out_of_range_symbol, sc->sole_arg_out_of_range_info);
  return(sc->out_of_range_symbol);
}

static no_return void wrong_number_of_arguments_error_nr(s7_scheme *sc, const char *errmsg, s7_int len, s7_pointer args)
{
  error_nr(sc, sc->wrong_number_of_args_symbol, set_elist_2(sc, wrap_string(sc, errmsg, len), args));
}

s7_pointer s7_wrong_number_of_args_error(s7_scheme *sc, const char *caller, s7_pointer args)
{
  error_nr(sc, sc->wrong_number_of_args_symbol,
	   set_elist_2(sc, wrap_string(sc, caller, safe_strlen(caller)), args)); /* "caller" includes the format directives */
  return(sc->wrong_number_of_args_symbol);
}


static no_return void syntax_error_nr(s7_scheme *sc, const char *errmsg, s7_int len, s7_pointer obj)
{
  error_nr(sc, sc->syntax_error_symbol, set_elist_2(sc, wrap_string(sc, errmsg, len), obj));
}

static no_return void syntax_error_with_caller_nr(s7_scheme *sc, const char *errmsg, s7_int len, s7_pointer caller, s7_pointer obj)
{
  error_nr(sc, sc->syntax_error_symbol, set_elist_3(sc, wrap_string(sc, errmsg, len), caller, obj));
}

static no_return void syntax_error_with_caller2_nr(s7_scheme *sc, const char *errmsg, s7_int len, s7_pointer caller, s7_pointer name, s7_pointer obj)
{
  error_nr(sc, sc->syntax_error_symbol, set_elist_4(sc, wrap_string(sc, errmsg, len), caller, name, obj));
}

static s7_pointer make_symbol(s7_scheme *sc, const char *name, s7_int len); /* calls new_symbol */
#define make_symbol_with_strlen(Sc, Name) make_symbol(Sc, Name, safe_strlen(Name))

static s7_pointer missing_method_class_name(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer class_name = find_method(sc, obj, sc->class_name_symbol);
  if (is_symbol(class_name)) return(class_name);
  return(sc->is_openlet_symbol);
}

static no_return void missing_method_error_nr(s7_scheme *sc, s7_pointer method, s7_pointer obj)
{
  error_nr(sc, sc->missing_method_symbol,
	   set_elist_4(sc, wrap_string(sc, "~S method is not defined in ~A ~A", 33), method,
		       (is_c_object(obj)) ? c_object_scheme_name(sc, obj) :
                         (((is_let(obj)) && (is_openlet(obj))) ? missing_method_class_name(sc, obj) :
                          s7_make_string_wrapper(sc, type_name(sc, obj, No_Article))),
		       object_to_string_truncated(sc, obj)));
}

static no_return void immutable_object_error_nr(s7_scheme *sc, s7_pointer info) {error_nr(sc, sc->immutable_error_symbol, info);}


/* -------- method handlers -------- */
s7_pointer s7_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  if (has_active_methods(sc, obj)) return(find_method_with_let(sc, obj, method));
  return(sc->undefined);
}

/* if a method is shadowing a built-in like abs, it should expect the same args as abs and behave the same -- no multiple values etc */
#define if_method_exists_return_value(Sc, Obj, Method, Args)		\
  {							\
    s7_pointer _Func_;					\
    if ((has_active_methods(Sc, Obj)) &&				\
	((_Func_ = find_method_with_let(Sc, Obj, Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, _Func_, Args)); \
  }

#define if_let_method_exists_return_value(Sc, Let, Method, Args)	\
  {							\
    s7_pointer _Func_;					\
    if ((has_active_methods(Sc, T_Let(Let))) &&				\
	((_Func_ = find_method(Sc, Let, Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, _Func_, Args)); \
  }

#define if_c_object_method_exists_return_value(Sc, C_Obj, Method, Args)	\
  {							\
    s7_pointer _Func_;					\
    if ((has_active_methods(Sc, T_Obj(C_Obj))) &&				\
	((_Func_ = find_method_with_c_object(Sc, C_Obj, Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, _Func_, Args)); \
  }

static s7_pointer apply_boolean_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  s7_pointer func = find_method_with_let(sc, obj, method);
  if (func == sc->undefined) return(sc->F);
  return(s7_apply_function(sc, func, set_mlist_1(sc, obj))); /* plist here and below will probably not work (_pp case known bad) */
}

/* this is a macro mainly to simplify the Checker handling */
#define check_boolean_method(Sc, Checker, Method, Args)	       \
  {							       \
    s7_pointer _P_ = car(Args);				       \
    if (Checker(_P_)) return(Sc->T);			       \
    if (!has_active_methods(Sc, _P_)) return(Sc->F);	       \
    return(apply_boolean_method(Sc, _P_, Method));	       \
  }

static s7_pointer apply_method_closure(s7_scheme *sc, s7_pointer func, s7_pointer args);

static s7_pointer find_and_apply_method(s7_scheme *sc, s7_pointer obj, s7_pointer sym, s7_pointer args) /* slower if inline */
{
  s7_pointer func = find_method_with_let(sc, obj, sym); /* perhaps find_and_apply_c_object_method for g_c_object_let */
  if (is_closure(func)) return(apply_method_closure(sc, func, args));
  if (func == sc->undefined) missing_method_error_nr(sc, sym, obj);
  if ((s7_Debugging) && (func == global_value(sym))) {fprintf(stderr, "loop in %s?\n", __func__); abort();}
  return(s7_apply_function(sc, func, args));
}

static s7_pointer method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj)) wrong_type_error_nr(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, args));
}

static s7_pointer mutable_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (has_active_methods(sc, obj)) return(find_and_apply_method(sc, obj, method, args));
  if (sc->type_names[type(obj)] != typ) wrong_type_error_nr(sc, method, num, obj, typ);
  if (!is_immutable(obj)) wrong_type_error_nr(sc, method, num, obj, typ);
  immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, method, obj));
  return(NULL);
}

static s7_pointer mutable_method_or_bust_ppp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
					     s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer typ, int32_t num)
{
  return(mutable_method_or_bust(sc, obj, method, set_qlist_3(sc, x1, x2, x3), typ, num)); /* was list_3, plist_3 not safe */
}

static s7_pointer method_or_bust_p(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer typ)
{
  if (!has_active_methods(sc, obj)) sole_arg_wrong_type_error_nr(sc, method, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_mlist_1(sc, obj)));
}

static s7_pointer method_or_bust_pp(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer x1, s7_pointer x2, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj)) wrong_type_error_nr(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_mlist_2(sc, x1, x2)));
}

static s7_pointer methods_or_bust_pp(s7_scheme *sc, s7_pointer obj, s7_pointer method1, s7_pointer method2, s7_pointer x1, s7_pointer x2, s7_pointer typ, int32_t num)
{ /* this is for the memq/memv and assq/assv equivalence in r7rs */
  s7_pointer func;
  if (!has_active_methods(sc, obj)) wrong_type_error_nr(sc, method1, num, obj, typ);
  func = find_method_with_let(sc, obj, method1);
  if (func == sc->undefined) func = find_method_with_let(sc, obj, method2);
  if (func == sc->undefined) missing_method_error_nr(sc, method1, obj);
  return(s7_apply_function(sc, func, set_mlist_2(sc, x1, x2)));
}

static s7_pointer method_or_bust_ppp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
				     s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer typ, int32_t num)
{
  if (!has_active_methods(sc, obj)) wrong_type_error_nr(sc, method, num, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_qlist_3(sc, x1, x2, x3))); /* was list_3, plist not safe */
}

static s7_pointer method_or_bust_with_type_and_loc_pp(s7_scheme *sc, s7_pointer obj, s7_pointer method,
						      s7_pointer x1, s7_pointer x2, s7_pointer typ, int32_t num)
{
  int32_t loc = sc->error_argnum + num;
  sc->error_argnum = 0;
  if (!has_active_methods(sc, obj)) wrong_type_error_nr(sc, method, loc, obj, typ);
  return(find_and_apply_method(sc, obj, method, set_mlist_2(sc, x1, x2)));
}

static s7_pointer sole_arg_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, s7_pointer typ)
{
  if (!has_active_methods(sc, obj))  sole_arg_wrong_type_error_nr(sc, method, obj, typ);
  return(find_and_apply_method(sc, obj, method, args));
}


/* -------------------------------- constants -------------------------------- */
/* #f and #t */
s7_pointer s7_f(s7_scheme *sc) {return(sc->F);}
s7_pointer s7_t(s7_scheme *sc) {return(sc->T);}


/* () */
s7_pointer s7_nil(s7_scheme *sc) {return(sc->nil);}               /* should this be "s7_null" ? */
bool s7_is_null(s7_scheme *sc, s7_pointer p) {return(is_null(p));}
static bool is_null_b_p(s7_pointer p) {return(type(p) == T_NIL);} /* faster than b_7p because opt_b_p is faster */

static s7_pointer g_is_null(s7_scheme *sc, s7_pointer args)
{
  #define H_is_null "(null? obj) returns #t if obj is the empty list"
  #define Q_is_null sc->pl_bt
  check_boolean_method(sc, is_null, sc->is_null_symbol, args);
}


/* #<undefined> and #<unspecified> */
s7_pointer s7_undefined(s7_scheme *sc)   {return(sc->undefined);}
s7_pointer s7_unspecified(s7_scheme *sc) {return(sc->unspecified);}

bool s7_is_unspecified(s7_scheme *sc, s7_pointer val) {return(is_unspecified(val));}

static s7_pointer g_is_undefined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_undefined "(undefined? val) returns #t if val is #<undefined> or some other #... value that s7 does not recognize; (undefined? #asdf): #t.\
This is not the same as (not (defined? val)) which refers to whether a symbol has a binding: (undefined? 'asdf): #f, but (not (defined? 'asdf)): #t"
  #define Q_is_undefined sc->pl_bt
  check_boolean_method(sc, is_undefined, sc->is_undefined_symbol, args);
}

static s7_pointer g_is_unspecified(s7_scheme *sc, s7_pointer args)
{
  #define H_is_unspecified "(unspecified? val) returns #t if val is #<unspecified>"
  #define Q_is_unspecified sc->pl_bt
  check_boolean_method(sc, is_unspecified, sc->is_unspecified_symbol, args);
}

static s7_pointer nil_string; /* permanent "" */
/* nil_vector is complicated by the many vector types, and s7test assumes it is mutable! and not eq? to other nil_vectors (watch out for add_vector!) */


/* -------------------------------- eof-object? -------------------------------- */
s7_pointer eof_object = NULL;          /* #<eof> is an entry in the chars array, so it's not a part of sc */

s7_pointer s7_eof_object(s7_scheme *sc) {return(eof_object);}

static s7_pointer g_is_eof_object(s7_scheme *sc, s7_pointer args)
{
  #define H_is_eof_object "(eof-object? val) returns #t if val is the end-of-file object, #<eof>.  It is the same as (eq? val #<eof>)"
  #define Q_is_eof_object sc->pl_bt
  check_boolean_method(sc, is_eof, sc->is_eof_object_symbol, args);
}

static bool is_eof_object_b_p(s7_pointer p) {return(p == eof_object);}


/* -------------------------------- not -------------------------------- */
static bool not_b_7p(s7_scheme *sc, s7_pointer p) {return(p == sc->F);}

static s7_pointer g_not(s7_scheme *sc, s7_pointer args)
{ /* this doesn't need method handling */
  #define H_not "(not obj) returns #t if obj is #f, otherwise #f: (not ()) -> #f"
  #define Q_not sc->pl_bt
  return((car(args) == sc->F) ? sc->T : sc->F);
}


/* -------------------------------- boolean? -------------------------------- */
bool s7_boolean(s7_scheme *sc, s7_pointer obj)      {return(obj != sc->F);}
s7_pointer s7_make_boolean(s7_scheme *sc, bool obj) {return(make_boolean(sc, obj));}

bool s7_is_boolean(s7_pointer x) {return(type(x) == T_BOOLEAN);}

static s7_pointer g_is_boolean(s7_scheme *sc, s7_pointer args)
{
  #define H_is_boolean "(boolean? obj) returns #t if obj is #f or #t: (boolean? ()) -> #f"
  #define Q_is_boolean sc->pl_bt
  check_boolean_method(sc, is_boolean, sc->is_boolean_symbol, args);
}


/* -------------------------------- constant? -------------------------------- */
static inline bool is_constant_symbol(s7_scheme *sc, s7_pointer sym) /* inline: 7 in cb, 5 in tgen */
{
  if (is_immutable_symbol(sym))    /* for keywords */
    return(true);
  if (is_possibly_constant(sym))
    {
      s7_pointer slot = s7_slot(sc, sym);
      return((is_slot(slot)) && (is_immutable_slot(slot)));
    }
  return(false);
}

#define is_constant(sc, p) ((type(p) != T_SYMBOL) || (is_constant_symbol(sc, p)))

static s7_pointer g_is_constant(s7_scheme *sc, s7_pointer args)
{
  #define H_is_constant "(constant? obj) returns #t if obj either evaluates to itself, or is a symbol whose binding is constant"
  #define Q_is_constant sc->pl_bt
  return(make_boolean(sc, is_constant(sc, car(args))));
}

static bool is_constant_b_7p(s7_scheme *sc, s7_pointer p) {return(is_constant(sc, p));}
static s7_pointer is_constant_p_p(s7_scheme *sc, s7_pointer p) {return(make_boolean(sc, is_constant(sc, p)));}


/* -------------------------------- immutable? -------------------------------- */

static no_return void find_let_error_nr(s7_scheme *sc, s7_pointer caller, s7_pointer let, s7_pointer new_let, s7_int arg_num, s7_pointer args)
{
  if (new_let == sc->rootlet)
    {
      if ((arg_num > 1) || (is_pair(cdr(args))))
	error_nr(sc, sc->wrong_type_arg_symbol,
		 set_elist_5(sc, wrap_string(sc, "(~A~{~^ ~$~}) ~:D argument is ~A, but it does not have its own let", 66),
			     caller, args, wrap_integer(sc, arg_num), object_type_name(sc, let)));
      error_nr(sc, sc->wrong_type_arg_symbol,
	       set_elist_4(sc, wrap_string(sc, "(~A~{~^ ~$~}) argument is ~A, but it does not have its own let", 62),
			   caller, args, object_type_name(sc, let)));
    }
  wrong_type_error_nr(sc, caller, arg_num, s7_list_ref(sc, args, arg_num - 1), wrap_string(sc, "a let or an object that has its own let", 39));
}

bool s7_is_immutable(s7_pointer p) {return(is_immutable(p));}
#define has_let_signature(sc) s7_make_signature(sc, 5, sc->is_let_symbol, sc->is_c_object_symbol, sc->is_c_pointer_symbol, sc->is_procedure_symbol, sc->is_macro_symbol)

static s7_pointer g_is_immutable(s7_scheme *sc, s7_pointer args)
{
  #define H_is_immutable "(immutable? obj (env (curlet))) returns #t if obj (or obj in the environment env) is immutable"
  #define Q_is_immutable s7_make_signature(sc, 3, sc->is_boolean_symbol, sc->T, has_let_signature(sc))
  const s7_pointer obj = car(args);
  if (is_symbol(obj))
    {
      s7_pointer slot;
      if (is_keyword(obj)) return(sc->T);
      if (is_pair(cdr(args)))
	{
	  s7_pointer let = cadr(args);
	  if (!is_let(let))
	    {
	      s7_pointer new_let = find_let(sc, let);
	      if ((!is_let(new_let)) || (new_let == sc->rootlet))
		find_let_error_nr(sc, sc->is_immutable_symbol, let, new_let, 2, args);
	      let = new_let;
	    }
	  if (let == sc->rootlet)
	    slot = global_slot(obj);
	  else slot = lookup_slot_from((is_keyword(obj)) ? keyword_symbol(obj) : obj, let);
	}
      else slot = s7_slot(sc, obj);
      if (is_slot(slot)) /* might be #<undefined> */
	return(make_boolean(sc, is_immutable_slot(slot)));
    }
  else
    if ((is_pair(cdr(args))) && (!is_let(cadr(args)))) /* (immutable? 1 2) */
      wrong_type_error_nr(sc, sc->is_immutable_symbol, 2, cadr(args), a_let_string);
  return(make_boolean(sc, (is_immutable(obj)) || (t_immutable_p[type(obj)]) ||
		           ((is_any_vector(obj)) && (vector_length(obj) == 0))));
}


/* -------------------------------- immutable! -------------------------------- */
s7_pointer s7_set_immutable(s7_scheme *sc, s7_pointer obj)
{
  if (is_symbol(obj)) /* trying to mimic g_immutable */
    {
      s7_pointer slot;
      if (is_keyword(obj)) return(obj);
      slot = s7_slot(sc, obj);
      if (is_slot(slot))
	set_immutable_slot(slot);
      /* symbol is not set immutable (as below) */
    }
  else set_immutable(obj);
  return(obj);
}

#if (!Disable_Deprecated) && (s7_Debugging || Disable_File_Output || Pointer_32)
  s7_pointer s7_immutable(s7_pointer obj) {return(s7_set_immutable(cur_sc, obj));}
#endif

#if WITH_GCC
static __attribute__ ((format (printf, 3, 4))) void s7_warn(s7_scheme *sc, s7_int len, const char *ctrl, ...);
#else
static void s7_warn(s7_scheme *sc, s7_int len, const char *ctrl, ...);
#endif
#define s7_Warn_Bufsize 256

static s7_pointer make_string_with_length(s7_scheme *sc, const char *str, s7_int len);

static s7_pointer g_immutable(s7_scheme *sc, s7_pointer args)
{
  #define H_immutable "(immutable! obj (env (curlet))) declares that the object obj (or obj in the environment env) can't be changed. obj is returned."
  #define Q_immutable s7_make_signature(sc, 3, sc->T, sc->T, sc->T)
  /* the sig second argument should be has_let_signature if the first is a symbol, and is_integer_symbol if the first is a pair, else the second argument is unused */
  const s7_pointer obj = car(args);
  if (is_symbol(obj))
    {
      s7_pointer slot;
      if (is_pair(cdr(args)))
	{
	  s7_pointer let = cadr(args);
	  if ((!is_let(let)) || (let == sc->rootlet))
	    {
	      s7_pointer new_let = find_let(sc, let);
	      if ((!is_let(new_let)) || (new_let == sc->rootlet))
		find_let_error_nr(sc, sc->immutable_symbol, let, new_let, 2, args);
	      let = new_let;
	    }
	  slot = symbol_to_local_slot(sc, (is_keyword(obj)) ? keyword_symbol(obj) : obj, let); /* different from immutable? */
	}
      else
	{
	  if (is_keyword(obj)) return(obj);
	  slot = s7_slot(sc, obj);
	}
      if (is_slot(slot))
	set_immutable_slot(slot);
      return(obj);  /* symbol is not set immutable ? */
    }
  if (is_pair(cdr(args)))
    {
      s7_pointer let_int = cadr(args);
      if ((is_let(let_int)) || (!is_pair(obj)))
	error_nr(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "in (immutable! ~{~S~^ ~}), the second argument is pointless", 59), args));
      if (!s7_is_integer(let_int))
	wrong_type_error_nr(sc, sc->immutable_symbol, 2, let_int, a_non_negative_integer_string);
    }
  if ((sc->safety > No_Safety) && (ignores_immutable(obj)))
    s7_warn(sc, 256, "%s, %s, ignores immutable!", display(obj), type_name(sc, obj, Indefinite_Article));

  if (is_pair(obj)) /* the second arg saying how many pairs to set immutable makes object->string :readable of a circular patchwork list workable */
    {
      s7_int end = (is_pair(cdr(args))) ? s7_integer(cadr(args)) : sc->max_list_length;
      s7_int i = 1;
      if (end < 0)
	wrong_type_error_nr(sc, sc->immutable_symbol, 2, cadr(args), a_non_negative_integer_string);
      if (end == 0) return(obj);
      set_immutable(obj);
      for (s7_pointer p = cdr(obj), slow = obj; (i < end) && (is_pair(p)) && (slow != p); p = cdr(p), i++)
	{
	  set_immutable(p);
	  i++;
	  if ((i < end) && (is_pair(cdr(p))))
	    {
	      p = cdr(p);
	      set_immutable(p);
	      slow = cdr(slow);
	    }}}
  else set_immutable(obj);
  return(obj);
}

/* there's no way to make a slot setter (as setter) immutable (t_multiform as bit) */


/* -------------------------------- GC -------------------------------- */
/* in most code, pairs, lets, and slots dominate the heap -- each about 25% to 40% of the
 *   total cell allocations.  In snd-test, reals are 50%. slots need not be in the heap,
 *   but moving them out to their own free list was slower because we need (in that
 *   case) to manage them in the sweep process by tracking lets.
 */

#if s7_Debugging
static s7_int gc_protect_2(s7_scheme *sc, s7_pointer x, int32_t line)
{
  static bool already_warned = false;
  s7_int loc = s7_gc_protect(sc, x);
  if ((sc->safety > No_Safety) && (!already_warned) && (loc > 8192))
    {
      already_warned = true;
      fprintf(stderr, "s7_gc_protect has protected more than 8192 values? (line: %d, code: %s, loc: %" ld64 ")\n",
	      line, string_value(s7_object_to_string(sc, current_code(sc), false)), loc);
      abort();
    }
  return(loc);
}
#define gc_protect_1(Sc, X) gc_protect_2(Sc, X, __LINE__)
#else
#define gc_protect_1(Sc, X) s7_gc_protect(Sc, X)
#endif

static void resize_gc_protect(s7_scheme *sc)
{
  const s7_int size = sc->protected_objects_size;
  block_t *old_b = vector_block(sc->protected_objects);
  const s7_int new_size = 2 * size;
  block_t *new_b = reallocate(sc, old_b, new_size * sizeof(s7_pointer));
  block_info(new_b) = NULL;
  vector_block(sc->protected_objects) = new_b;
  vector_elements(sc->protected_objects) = (s7_pointer *)block_data(new_b);
  vector_length(sc->protected_objects) = new_size;
  sc->protected_objects_size = new_size;
  sc->protected_objects_free_list = (s7_int *)Realloc(sc->protected_objects_free_list, new_size * sizeof(s7_int));
  for (s7_int i = size; i < new_size; i++)
    {
      vector_element(sc->protected_objects, i) = sc->unused;
      sc->protected_objects_free_list[++sc->protected_objects_free_list_loc] = i;
    }
}

s7_int s7_gc_protect(s7_scheme *sc, s7_pointer x)
{
  s7_int loc;
  if (sc->protected_objects_free_list_loc < 0)
    resize_gc_protect(sc);
  loc = sc->protected_objects_free_list[sc->protected_objects_free_list_loc--];
  vector_element(sc->protected_objects, loc) = x;
  return(loc);
}

void s7_gc_unprotect_at(s7_scheme *sc, s7_int loc)
{
  if (loc < sc->protected_objects_size)
    {
      if (vector_element(sc->protected_objects, loc) != sc->unused) /* ?? */
	sc->protected_objects_free_list[++sc->protected_objects_free_list_loc] = loc;
      else if (s7_Debugging) fprintf(stderr, "redundant gc_unprotect_at location %" ld64 "\n", loc);
      vector_element(sc->protected_objects, loc) = sc->unused;
    }
}

s7_pointer s7_gc_protected_at(s7_scheme *sc, s7_int loc)
{
  s7_pointer obj = sc->unspecified;
  if (loc < sc->protected_objects_size)
    obj = vector_element(sc->protected_objects, loc);
  if (obj == sc->unused)
    return(sc->unspecified);
  return(obj);
}

#define gc_protected_at(Sc, Loc) vector_element(Sc->protected_objects, Loc)

s7_pointer s7_gc_protect_via_location(s7_scheme *sc, s7_pointer x, s7_int loc)
{
  vector_element(sc->protected_objects, loc) = x;
  return(x);
}

s7_pointer s7_gc_unprotect_via_location(s7_scheme *sc, s7_int loc)
{
  vector_element(sc->protected_objects, loc) = sc->unused;
  sc->protected_objects_free_list[++sc->protected_objects_free_list_loc] = loc; /* added 13-Feb-25 */
  return(sc->F);
}


/* these 3 are needed by sweep */
static void (*mark_function[Num_Types])(s7_pointer p);
void s7_mark(s7_pointer p) {if (!is_marked(p)) (*mark_function[type_unchecked(p)])(p);}
static void mark_noop(s7_pointer unused_p) {}

static void process_iterator(s7_scheme *unused_sc, s7_pointer iter)
{
  if (is_weak_hash_iterator(iter))
    {
      s7_pointer seq = iterator_sequence(iter);
      clear_weak_hash_iterator(iter);
      if (type_unchecked(seq) == T_HASH_TABLE)
	weak_hash_iters(seq)--;
    }
}

static void process_multivector(s7_scheme *sc, s7_pointer vect)
{
  vdims_t *info = vector_dimension_info(vect);  /* a multidimensional empty vector can have dimension info, wrapped vectors always have dimension info */
  if ((info) && (info != sc->wrap_only))
    {
      if (vector_elements_should_be_freed(info)) /* a kludge for foreign code convenience */
	{
	  free(any_vector_elements(vect));
	  vector_elements_should_be_freed(info) = false;
	}
      liberate(sc, info);
      vector_set_dimension_info(vect, NULL);
    }
  liberate(sc, vector_block(vect));
}

static void process_input_string_port(s7_scheme *sc, s7_pointer port)
{
#if s7_Debugging
  /* this set of ports is a subset of the ports that respond true to is_string_port --
   *   the latter include file ports fully read into local memory; see read_file which uses add_input_port, not add_input_string_port
   */
  if (port_filename(port))
    fprintf(stderr, "%s[%d]: string input port has a filename: %s\n", __func__, __LINE__, port_filename(port));
  if (port_needs_free(port))
    {fprintf(stderr, "%s[%d]: string input port needs data release\n", __func__, __LINE__); abort();}
#endif
  liberate(sc, port_block(port));
}

static void free_port_data(s7_scheme *sc, s7_pointer port)
{
  if (port_data(port))
    {
      liberate(sc, port_data_block(port));
      port_data_block(port) = NULL;
      port_data(port) = NULL;
      port_data_size(port) = 0;
    }
  port_needs_free(port) = false;
}

static void close_input_function_port(s7_scheme *sc, s7_pointer port);
static void close_output_port(s7_scheme *sc, s7_pointer port);

static void process_input_port(s7_scheme *sc, s7_pointer port)
{
  if (!port_is_closed(port))
    {
      if (is_file_port(port))
	{
	  if (port_file(port))
	    {
	      fclose(port_file(port));
	      port_file(port) = NULL;
	    }}
      else
	if (is_function_port(port))
	  close_input_function_port(sc, port);
    }
  if (port_needs_free(port))
    free_port_data(sc, port);

  if (port_filename(port))
    {
      liberate(sc, port_filename_block(port));
      port_filename(port) = NULL;
    }
  liberate(sc, port_block(port));
}

static void process_output_port(s7_scheme *sc, s7_pointer port)
{
  close_output_port(sc, port); /* needed for free filename, etc */
  liberate(sc, port_block(port));
  if (port_needs_free(port))
    {
      port_needs_free(port) = false;
      if (port_data_block(port))
	{
	  liberate(sc, port_data_block(port));
	  port_data_block(port) = NULL;
	}}
}

static void process_continuation(s7_scheme *sc, s7_pointer cc)
{
  continuation_op_stack(cc) = NULL;
  liberate_block(sc, continuation_block(cc)); /* from mallocate_block (s7_make_continuation) */
}


#if With_Gmp
#if ((__GNU_MP_VERSION < 6) || ((__GNU_MP_VERSION == 6) && (__GNU_MP_VERSION_MINOR == 0)))
static int32_t mpq_cmp_z(const mpq_t op1, const mpz_t op2)
{
  mpq_t z;
  int32_t result;
  mpq_init(z);
  mpq_set_z(z, op2);
  result = mpq_cmp(op1, z);
  mpq_clear(z);
  return(result);
}
#endif

static s7_int big_integer_to_s7_int(s7_scheme *sc, mpz_t n);

static s7_int s7_integer_clamped_if_gmp(s7_scheme *sc, s7_pointer num)
{
  if (is_t_integer(num))
    return(integer(num));
  if (is_t_big_integer(num))
    return(big_integer_to_s7_int(sc, big_integer(num)));
  return(0);
}

static void free_big_integer(s7_scheme *sc, s7_pointer num)
{
  big_integer_nxt(num) = sc->bigints;
  sc->bigints = big_integer_bgi(num);
  big_integer_bgi(num) = NULL;
}

static void free_big_ratio(s7_scheme *sc, s7_pointer num)
{
  big_ratio_nxt(num) = sc->bigrats;
  sc->bigrats = big_ratio_bgr(num);
  big_ratio_bgr(num) = NULL;
}

static void free_big_real(s7_scheme *sc, s7_pointer num)
{
  big_real_nxt(num) = sc->bigflts;
  sc->bigflts = big_real_bgf(num);
  big_real_bgf(num) = NULL;
}

static void free_big_complex(s7_scheme *sc, s7_pointer num)
{
  big_complex_nxt(num) = sc->bigcmps;
  sc->bigcmps = big_complex_bgc(num);
  big_complex_bgc(num) = NULL;
}
#else
#define s7_integer_clamped_if_gmp(Sc, P) integer(P)
#endif


static void free_hash_table(s7_scheme *sc, s7_pointer table);
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym);
static void cull_weak_hash_table(s7_scheme *sc, s7_pointer table);

static void sweep(s7_scheme *sc)
{
  s7_int i, j;
  gc_list_t *gp;

  #define process_gc_list(Code)				\
    if (gp->loc > 0)					\
      {							\
        for (i = 0, j = 0; i < gp->loc; i++)		\
          {						\
            s7_pointer gc_obj = gp->list[i];		\
            if (is_free_and_clear(gc_obj))		\
              {						\
                Code; /* may access gc_obj internally */	\
              }							\
            else if (in_heap(gc_obj)) gp->list[j++] = gc_obj;	\
          }						\
        gp->loc = j;					\
      }							\

  gp = sc->strings;
  process_gc_list(liberate(sc, string_block(gc_obj)));

  gp = sc->gensyms;
  process_gc_list(remove_gensym_from_symbol_table(sc, gc_obj); liberate(sc, gensym_block(gc_obj)));
  if (gp->loc == 0) mark_function[T_SYMBOL] = mark_noop;

  gp = sc->undefineds;
  process_gc_list(free(undefined_name(gc_obj)));

  gp = sc->c_objects;
  process_gc_list(if (c_object_gc_free(sc, gc_obj)) (*(c_object_gc_free(sc, gc_obj)))(sc, gc_obj); else (*(c_object_free(sc, gc_obj)))(c_object_value(gc_obj)));

  gp = sc->vectors;
  process_gc_list(liberate(sc, vector_block(gc_obj)));

  gp = sc->multivectors;
  process_gc_list(process_multivector(sc, gc_obj));

  gp = sc->hash_tables;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer gc_obj = gp->list[i];
	  if (is_free_and_clear(gc_obj))
	    free_hash_table(sc, gc_obj);
	  else
	    {
	      if ((is_weak_hash_table(gc_obj)) &&
		  (weak_hash_iters(gc_obj) == 0) &&
		  (hash_table_entries(gc_obj) > 0))
		cull_weak_hash_table(sc, gc_obj);
	      gp->list[j++] = gc_obj;
	    }}
      gp->loc = j;
    }

  gp = sc->weak_hash_iterators;
  process_gc_list(process_iterator(sc, gc_obj));

  gp = sc->opt1_funcs;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer gc_obj = gp->list[i];
	  if (!is_free_and_clear(gc_obj))
	    gp->list[j++] = gc_obj;
	}
      gp->loc = j;
    }

  gp = sc->input_ports;
  process_gc_list(process_input_port(sc, gc_obj));

  gp = sc->input_string_ports;
  process_gc_list(process_input_string_port(sc, gc_obj));

  gp = sc->output_ports;
  process_gc_list(process_output_port(sc, gc_obj));

  gp = sc->continuations;
  process_gc_list(process_continuation(sc, gc_obj));

  gp = sc->weak_refs;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer gc_obj = gp->list[i];
	  if (!is_free_and_clear(gc_obj))
	    {
	      if (is_free_and_clear(c_pointer_weak1(gc_obj)))
		c_pointer_weak1(gc_obj) = sc->F;
	      if (is_free_and_clear(c_pointer_weak2(gc_obj)))
		c_pointer_weak2(gc_obj) = sc->F;
	      if ((c_pointer_weak1(gc_obj) != sc->F) ||
		  (c_pointer_weak2(gc_obj) != sc->F))
		gp->list[j++] = gc_obj;
	    }}
      gp->loc = j;
    }

#if With_Gmp
  gp = sc->big_integers;
  process_gc_list(free_big_integer(sc, gc_obj)) /* gc_obj == gp->list[i] */

  gp = sc->big_ratios;
  process_gc_list(free_big_ratio(sc, gc_obj))

  gp = sc->big_reals;
  process_gc_list(free_big_real(sc, gc_obj))

  gp = sc->big_complexes;
  process_gc_list(free_big_complex(sc, gc_obj))

  gp = sc->big_random_states;
  process_gc_list(gmp_randclear(random_gmp_state(gc_obj)))
#endif
}

static void add_to_gc_list(s7_scheme *sc, gc_list_t *gp, s7_pointer p)
{
#if s7_Debugging
  if ((!in_heap(p)) && (gp != sc->opt1_funcs))
    {
      char *str = describe_type_bits(sc, p);
      fprintf(stderr, "%s[%d]: %s not in heap, %s\n", __func__, __LINE__, display(p), str);
      free(str);
      abort();
    }
#endif
  if (gp->loc == gp->size)
    {
      gp->size *= 2;
      gp->list = (s7_pointer *)Realloc(gp->list, gp->size * sizeof(s7_pointer));
    }
  gp->list[gp->loc++] = p;
}

static gc_list_t *make_gc_list(void)
{
  gc_list_t *gp = (gc_list_t *)Malloc(sizeof(gc_list_t));
  #define Init_Gc_Cache_Size 4
  gp->size = Init_Gc_Cache_Size;
  gp->loc = 0;
  gp->list = (s7_pointer *)Malloc(gp->size * sizeof(s7_pointer));
  return(gp);
}

static void just_mark(s7_pointer p) {set_mark(p);}

static void add_gensym(s7_scheme *sc, s7_pointer p)
{
  add_to_gc_list(sc, sc->gensyms, p);
  mark_function[T_SYMBOL] = just_mark;
}

#define add_c_object(sc, p)          add_to_gc_list(sc, sc->c_objects, p)
#define add_hash_table(sc, p)        add_to_gc_list(sc, sc->hash_tables, p)
#define add_string(sc, p)            add_to_gc_list(sc, sc->strings, p)
#define add_input_port(sc, p)        add_to_gc_list(sc, sc->input_ports, p)
#define add_input_string_port(sc, p) add_to_gc_list(sc, sc->input_string_ports, p)
#define add_output_port(sc, p)       add_to_gc_list(sc, sc->output_ports, p)
#define add_continuation(sc, p)      add_to_gc_list(sc, sc->continuations, p)
#define add_undefined(sc, p)         add_to_gc_list(sc, sc->undefineds, p)
#define add_vector(sc, p)            add_to_gc_list(sc, sc->vectors, p)
#define add_multivector(sc, p)       add_to_gc_list(sc, sc->multivectors, p)
#define add_weak_ref(sc, p)          add_to_gc_list(sc, sc->weak_refs, p)
#define add_weak_hash_iterator(sc, p) add_to_gc_list(sc, sc->weak_hash_iterators, p)
#define add_opt1_func(sc, p) do {if (!opt1_func_listed(p)) add_to_gc_list(sc, sc->opt1_funcs, p); set_opt1_func_listed(p);} while (0) /* called by set_opt1_lambda_add */

#if With_Gmp
#define add_big_integer(sc, p)       add_to_gc_list(sc, sc->big_integers, p)
#define add_big_ratio(sc, p)         add_to_gc_list(sc, sc->big_ratios, p)
#define add_big_real(sc, p)          add_to_gc_list(sc, sc->big_reals, p)
#define add_big_complex(sc, p)       add_to_gc_list(sc, sc->big_complexes, p)
#define add_big_random_state(sc, p)  add_to_gc_list(sc, sc->big_random_states, p)
#endif

static void init_gc_caches(s7_scheme *sc)
{
  sc->strings = make_gc_list();
  sc->gensyms = make_gc_list();
  sc->undefineds = make_gc_list();
  sc->vectors = make_gc_list();
  sc->multivectors = make_gc_list();
  sc->hash_tables = make_gc_list();
  sc->input_ports = make_gc_list();
  sc->input_string_ports = make_gc_list();
  sc->output_ports = make_gc_list();
  sc->continuations = make_gc_list();
  sc->c_objects = make_gc_list();
  sc->weak_refs = make_gc_list();
  sc->weak_hash_iterators = make_gc_list();
  sc->opt1_funcs = make_gc_list();
#if With_Gmp
  sc->big_integers = make_gc_list();
  sc->big_ratios = make_gc_list();
  sc->big_reals = make_gc_list();
  sc->big_complexes = make_gc_list();
  sc->big_random_states = make_gc_list();
  sc->ratloc = NULL;
#endif
  /* slightly unrelated... */
  sc->setters_size = 4;
  sc->setters_loc = 0;
  sc->setters = (s7_pointer *)Malloc(sc->setters_size * sizeof(s7_pointer));
}

#if s7_Debugging
#define semipermanent_cons(Sc, A, B, Type) semipermanent_cons_1(Sc, A, B, Type, __func__, __LINE__)
static s7_pointer semipermanent_cons_1(s7_scheme *sc, s7_pointer a, s7_pointer b, s7_uint type, const char *func, int line);
#else
static s7_pointer semipermanent_cons(s7_scheme *sc, s7_pointer a, s7_pointer b, s7_uint type);
#endif

static void add_setter(s7_scheme *sc, s7_pointer func, s7_pointer setter)
{
  /* setters GC-protected. The c_function_setter field can't be used because the built-in functions
   *   are often removed from the heap and never thereafter marked.  Only closures and macros are protected here.
   */
  for (s7_int i = 0; i < sc->setters_loc; i++)
    {
      s7_pointer x = sc->setters[i];
      if (car(x) == func)
	{
	  set_cdr_unchecked(x, T_Clo(setter)); /* T_Clo else no GC protection needed */
	  return;
 	}}
  if (sc->setters_loc == sc->setters_size)
    {
      sc->setters_size *= 2;
      sc->setters = (s7_pointer *)Realloc(sc->setters, sc->setters_size * sizeof(s7_pointer));
    }
  sc->setters[sc->setters_loc++] = semipermanent_cons(sc, func, T_Prc(setter), T_PAIR | T_Immutable);
}


static inline void gc_mark(s7_pointer p) {if (!is_marked(p)) (*mark_function[type_unchecked(p)])(p);}

static void mark_symbol_vector(s7_pointer vec, s7_int len)
{
  set_mark(vec);
  if (mark_function[T_SYMBOL] != mark_noop) /* else no gensyms */
    {
      s7_pointer *els = vector_elements(vec);
      for (s7_int i = 0; i < len; i++)
	if ((is_symbol(els[i])) && (is_gensym(els[i]))) /* need is_symbol: make-vector + set! vector-typer symbol? where init is not a symbol */
	  set_mark(els[i]);
    }
}

static void mark_simple_vector(s7_pointer vec, s7_int len)
{
  s7_pointer *vels = vector_elements(vec);
  set_mark(vec);
  for (s7_int i = 0; i < len; i++)
    set_mark(vels[i]);
}

static void just_mark_vector(s7_pointer vect, s7_int unused_len) {set_mark(vect);}

static void mark_vector_1(s7_pointer vect, s7_int top)
{
  s7_pointer *tp = (s7_pointer *)(vector_elements(vect));
  s7_pointer *tend, *tend4;
  set_mark(vect);
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);
  tend4 = (s7_pointer *)(tend - 16);
  while (tp <= tend4) {Loop_8(gc_mark(*tp++)); Loop_8(gc_mark(*tp++));} /* faster if large vectors in use, maybe slower otherwise? */
  while (tp < tend)
    gc_mark(*tp++);
}

static void mark_typed_vector_1(s7_pointer vect, s7_int top) /* for typed vectors with closure setters */
{
  gc_mark(typed_vector_typer(vect));
  mark_vector_1(vect, top);
}

static inline void mark_slot(s7_pointer slot)
{
  set_mark(T_Slt(slot));
  gc_mark(slot_value(slot));
  if (slot_has_setter_or_pending_value(slot))
    gc_mark(slot_pending_value_unchecked(slot)); /* setter field == pending_value */
  set_mark(slot_symbol(slot));
}

static void mark_let(s7_pointer let)
{
  for (; (let) && (!is_marked(let)); let = let_outlet(let))
    {
      set_mark(let);
      if (has_dox_slot1(let)) mark_slot(let_dox_slot1(let));
      if ((has_dox_slot2(let)) && (is_slot(let_dox_slot2(let)))) mark_slot(let_dox_slot2(let));
      /* it can happen (call/cc related) that let_dox_slot2 is a slot but invalid, but in that case has_dox_slot2 will not be set(?) */
      for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
	if (!is_marked(slot)) /* slot value might be the enclosing let */
	  mark_slot(slot);
    }
}

static void mark_wrappers(s7_scheme *sc)
{
  s7_pointer p = sc->let_wrappers;
  s7_pointer end_p = p;
  do {
    for (s7_pointer slot = let_slots(car(p)); is_not_slot_end(slot); slot = next_slot(slot))
      if (!is_marked(slot)) mark_slot(slot);
    p = cdr(p);
  } while (p != end_p);

  /* dox1|2? gensyms? maybe don't wrap gensym-slot */
}

static void unmark_wrappers(s7_scheme *sc)
{
  s7_pointer p = sc->let_wrappers;
  s7_pointer end_p = p;
  do {
    for (s7_pointer slot = let_slots(car(p)); is_not_slot_end(slot); slot = next_slot(slot)) clear_mark(slot);
    p = cdr(p);
  } while (p != end_p);
}

#if With_History
static void gc_owlet_mark(s7_pointer tp)
{
  /* gc_mark but if tp is a pair ignore the marked bit on unheaped entries */
  if (is_pair(tp))
    {
      s7_pointer p = tp;
      do {
	set_mark(p);
	gc_mark(car(p)); /* does this need to be gc_owlet_mark? I can't find a case */
	p = cdr(p);
      } while ((is_pair(p)) && (p != tp) && ((!in_heap(p)) || (!is_marked(p)))); /* ((full_type(p) & (Type_Mask | T_Gc_Mark)) == T_PAIR) is much slower */
      gc_mark(p);
    }
  else
    if (!is_marked(tp))
      (*mark_function[type_unchecked(tp)])(tp);
}
#endif

static void mark_owlet(s7_scheme *sc)
{
#if With_History
  {
    for (s7_pointer p1 = sc->eval_history1, p2 = sc->eval_history2, p3 = sc->history_pairs; ; p2 = cdr(p2), p3 = cdr(p3))
      {
	gc_owlet_mark(car(p1));
	gc_owlet_mark(car(p2));
	gc_owlet_mark(car(p3));
	p1 = cdr(p1);
	if (p1 == sc->eval_history1) break; /* these are circular lists */
      }}
#endif
  /* sc->error_type and friends are slots in owlet */
  mark_slot(sc->error_type);
  slot_set_value(sc->error_data, sc->F); /* or maybe mark_tree(slot_value(sc->error_data)) ? */
  mark_slot(sc->error_data);
  mark_slot(sc->error_code);
  mark_slot(sc->error_line);
  mark_slot(sc->error_file);
  mark_slot(sc->error_position);
#if With_History
  mark_slot(sc->error_history);
#endif
  set_mark(sc->owlet);
  mark_let(let_outlet(sc->owlet));
}

static void mark_c_pointer(s7_pointer cp)
{
  set_mark(cp);
  gc_mark(c_pointer_type(cp));
  gc_mark(c_pointer_info(cp));
}

static void mark_c_proc_star(s7_pointer proc)
{
  set_mark(proc);
  if ((!c_func_has_simple_defaults(proc)) &&
      (c_function_call_args(proc))) /* NULL if not a safe function */
    for (s7_pointer arg = c_function_call_args(proc); is_pair(arg); arg = cdr(arg))
      gc_mark(car(arg));
}

static void mark_pair(s7_pointer p)
{
  do {
    set_mark(p);
    gc_mark(car(p)); /* expanding this to avoid recursion is slower */
    p = cdr(p);
  } while ((is_pair(p)) && (!is_marked(p))); /* ((full_type(p) & (Type_Mask | T_Gc_Mark)) == T_PAIR) is much slower */
  gc_mark(p);
}

static void mark_counter(s7_pointer ctr)
{
  set_mark(ctr);
  gc_mark(counter_result(ctr));
  gc_mark(counter_list(ctr));
  gc_mark(counter_let(ctr));
}

static void mark_closure(s7_pointer clo)
{
  set_mark(clo);
  gc_mark(closure_pars(clo));
  gc_mark(closure_body(clo));
  mark_let(closure_let(clo));
  /* because we can't tell if a closure is live, we can't clear closure_let slot_values that are not currently in play (all gc roots are live!) */
  gc_mark(closure_setter_or_map_list(clo));
}

static void mark_stack_1(s7_pointer stack, s7_int top)
{
  s7_pointer *tp = (s7_pointer *)(stack_elements(stack)), *tend;
  set_mark(stack);
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);
  while (tp < tend)
    {
      gc_mark(*tp++); /* sc->code */
      gc_mark(*tp++); /* sc->curlet */
      gc_mark(*tp++); /* sc->args */
      tp++;           /* sc->cur_op */
    }
}

static void mark_stack(s7_pointer stack)
{
  /* we can have a bare stack waiting for a continuation to hold it if the new_cell for the continuation triggers the GC!  But we need a top-of-stack?? */
  mark_stack_1(stack, temp_stack_top(stack));
}

static void mark_continuation(s7_pointer cc)
{
  set_mark(cc);
  if (!is_marked(continuation_stack(cc))) /* can these be cyclic? */
    mark_stack_1(continuation_stack(cc), continuation_stack_top(cc));
  gc_mark(continuation_op_stack(cc));
}

static void mark_vector(s7_pointer vect)
{
  if (is_typed_vector(vect))
    typed_vector_gc_mark(vect)(vect, vector_length(vect));
  else mark_vector_1(vect, vector_length(vect));
}

static void mark_vector_possibly_shared(s7_pointer vect)
{
  /* If a subvector (an inner dimension) of a vector is the only remaining reference
   *    to the main vector, we want to make sure the main vector is not GC'd until
   *    the subvector is also GC-able.  The subvector field either points to the
   *    parent vector, or it is sc->F, so we need to check for a vector parent if
   *    the current is multidimensional (this will include 1-dim slices).  We need
   *    to keep the parent case separate (i.e. sc->F means the current is the original)
   *    so that we only free once (or remove_from_heap once).
   *
   * If we have a subvector of a subvector, and the middle and original are not otherwise
   *   in use, we mark the middle one, but (since it itself is not in use anywhere else)
   *   we don't mark the original!  So we need to follow the share-vector chain marking every one.
   *
   * To remove a cell from the heap, we need its current heap location so that we can replace it.
   *   The heap is allocated as needed in monolithic blocks of (say) 1/2M s7_cells. When a cell
   *   is replaced, the new cell (at heap[x] say) is no longer from the original block. Since the
   *   GC clears all type bits when it frees a cell, we can't use a type bit to distinguish the
   *   replacements from the originals, but we need that info because in the base case, we use
   *   the distance of the cell from the base cell to get "x", its location.  In the replacement
   *   case, we add the location at the end of the s7_cell (s7_big_cell).  We track the current
   *   heap blocks via the sc->heap_blocks list.  To get the location of "p" above, we run through
   *   that list looking for a block it fits in.  If none is found, we assume it is an s7_big_cell
   *   and use the saved location.
   */
  if (is_subvector(vect))
    mark_vector_possibly_shared(subvector_vector(vect));

  /* mark_vector_1 does not check the marked bit, so if subvector below is in a cycle involving
   *   the calling vector, we get infinite recursion unless we check the mark bit here.
   */
  if (!is_marked(vect))
    mark_vector_1(vect, vector_length(vect));
}

static void mark_int_or_float_vector(s7_pointer vect) {set_mark(vect);}

static void mark_int_or_float_vector_possibly_shared(s7_pointer vect) /* also complex_vector */
{
  if (is_subvector(vect))
    mark_int_or_float_vector_possibly_shared(subvector_vector(vect));
  set_mark(vect);
}

static void mark_c_object(s7_pointer cobj)
{
  set_mark(cobj);
  mark_let(c_object_let(cobj));  /* I think this is guaranteed to be a let, added 5-Apr-25 */
  if (c_object_gc_mark(c_object_sc(cobj), cobj))  /* c_object_sc = s7_scheme pointer */
    (*(c_object_gc_mark(c_object_sc(cobj), cobj)))(c_object_sc(cobj), cobj);
  else (*(c_object_mark(c_object_sc(cobj), cobj)))(c_object_value(cobj));
}

static void mark_catch(s7_pointer catcher) /* C++ reserves "catch" */
{
  set_mark(catcher);
  gc_mark(catch_tag(catcher));
  gc_mark(catch_handler(catcher));
}

static void mark_dynamic_wind(s7_pointer dw)
{
  set_mark(dw);
  gc_mark(dynamic_wind_in(dw));
  gc_mark(dynamic_wind_out(dw));
  gc_mark(dynamic_wind_body(dw));
}

static void mark_hash_table(s7_pointer table)
{
  set_mark(table);
  gc_mark(hash_table_procedures(table));
  if (is_pair(hash_table_procedures(table)))
    {
      gc_mark(hash_table_key_typer_unchecked(table));  /* unchecked to avoid s7-debugger's reference to sc */
      gc_mark(hash_table_value_typer_unchecked(table));
    }
  if (hash_table_entries(table) > 0)
    {
      const s7_int len = (s7_int)hash_table_size(table);
      hash_entry_t **entries = hash_table_elements(table);
      hash_entry_t **last = (hash_entry_t **)(entries + len);

      if ((is_weak_hash_table(table)) &&
	  (weak_hash_iters(table) == 0))
	while (entries < last)
	  {
	    hash_entry_t *entry;
	    for (entry = *entries++; entry; entry = hash_entry_next(entry))
	      gc_mark(hash_entry_value(entry));
	    for (entry = *entries++; entry; entry = hash_entry_next(entry))
	      gc_mark(hash_entry_value(entry));
	  }
      else
	while (entries < last) /* counting entries here was slightly faster */
	  {
	    hash_entry_t *entry;
	    for (entry = *entries++; entry; entry = hash_entry_next(entry))
	      {
		gc_mark(hash_entry_key(entry));
		gc_mark(hash_entry_value(entry));
	      }
	    for (entry = *entries++; entry; entry = hash_entry_next(entry))
	      {
		gc_mark(hash_entry_key(entry));
		gc_mark(hash_entry_value(entry));
	      }}}
}

static void mark_iterator(s7_pointer iter)
{
  set_mark(iter);
  gc_mark(iterator_sequence(iter));
  if (has_carrier(iter))
    {
      if (iterator_carrier(iter))
	gc_mark(iterator_carrier(iter));
#if s7_Debugging
      else fprintf(stderr, "mark_iterator[%d]: has_carrier set (at line %d), but no carrier!\n", __LINE__, iter->carrier_line);
#endif
    }
}

static void mark_input_port(s7_pointer port)
{
  set_mark(port);
  gc_mark(port_string_or_function(port));
}

static void mark_output_port(s7_pointer port)
{
  set_mark(port);
  if (is_function_port(port))
    gc_mark(port_string_or_function(port));
}

static void mark_free(s7_pointer unused_p) {/* if (s7_Debugging) {fprintf(stderr, "mark_free!\n"); abort();} */} /* set_mark also checks */
/* this can happen if safe_list partially filled when GC called etc -- how to catch these cases and not abort? */

static void init_mark_functions(void)
{
  mark_function[T_BACRO]               = mark_closure;
  mark_function[T_BACRO_STAR]          = mark_closure;
  mark_function[T_BIG_COMPLEX]         = just_mark;
  mark_function[T_BIG_INTEGER]         = just_mark;
  mark_function[T_BIG_RATIO]           = just_mark;
  mark_function[T_BIG_REAL]            = just_mark;
  mark_function[T_BOOLEAN]             = mark_noop;
  mark_function[T_BYTE_VECTOR]         = just_mark;
  mark_function[T_CATCH]               = mark_catch;
  mark_function[T_CHARACTER]           = mark_noop;
  mark_function[T_CLOSURE]             = mark_closure;
  mark_function[T_CLOSURE_STAR]        = mark_closure;
  mark_function[T_COMPLEX]             = just_mark;
  mark_function[T_COMPLEX_VECTOR]      = mark_int_or_float_vector;
  mark_function[T_CONTINUATION]        = mark_continuation;
  mark_function[T_COUNTER]             = mark_counter;
  mark_function[T_C_FUNCTION]          = just_mark;
  mark_function[T_C_FUNCTION_STAR]     = just_mark; /* changes to mark_c_proc_star if defaults involve an expression */
  mark_function[T_C_MACRO]             = just_mark;
  mark_function[T_C_OBJECT]            = mark_c_object;
  mark_function[T_C_POINTER]           = mark_c_pointer;
  mark_function[T_C_RST_NO_REQ_FUNCTION] = just_mark;
  mark_function[T_DYNAMIC_WIND]        = mark_dynamic_wind;
  mark_function[T_EOF]                 = mark_noop;
  mark_function[T_FLOAT_VECTOR]        = mark_int_or_float_vector;
  mark_function[T_FREE]                = mark_free;
  mark_function[T_GOTO]                = just_mark;
  mark_function[T_HASH_TABLE]          = mark_hash_table;
  mark_function[T_INPUT_PORT]          = mark_input_port;
  mark_function[T_INTEGER]             = just_mark;
  mark_function[T_INT_VECTOR]          = mark_int_or_float_vector;
  mark_function[T_ITERATOR]            = mark_iterator;
  mark_function[T_LET]                 = mark_let;
  mark_function[T_MACRO]               = mark_closure;
  mark_function[T_MACRO_STAR]          = mark_closure;
  mark_function[T_NIL]                 = mark_noop;
  mark_function[T_OUTPUT_PORT]         = just_mark; /* changed to mark_output_port if output function ports are active */
  mark_function[T_PAIR]                = mark_pair;
  mark_function[T_RANDOM_STATE]        = just_mark;
  mark_function[T_RATIO]               = just_mark;
  mark_function[T_REAL]                = just_mark;
  mark_function[T_SLOT]                = mark_slot;
  mark_function[T_STACK]               = mark_stack;
  mark_function[T_STRING]              = just_mark;
  mark_function[T_SYMBOL]              = mark_noop; /* this changes to just_mark when gensyms are in the heap */
  mark_function[T_SYNTAX]              = mark_noop;
  mark_function[T_UNDEFINED]           = just_mark;
  mark_function[T_UNSPECIFIED]         = mark_noop;
  mark_function[T_UNUSED]              = mark_noop;
  mark_function[T_VECTOR]              = mark_vector; /* this changes if subvector created (similarly below) */
}

static void mark_op_stack(s7_scheme *sc)
{
  s7_pointer *p = sc->op_stack;
  s7_pointer *tp = sc->op_stack_now;
  while (p < tp)
    gc_mark(*p++);
}

static void mark_input_port_stack(s7_scheme *sc)
{
  s7_pointer *tp = (s7_pointer *)(sc->input_port_stack + sc->input_port_stack_loc);
  for (s7_pointer *p = sc->input_port_stack; p < tp; p++)
    gc_mark(*p);
}

static void mark_rootlet(s7_scheme *sc)
{
  for (s7_pointer slot = sc->rootlet_slots; is_not_slot_end(slot); slot = next_slot(slot))
    gc_mark(slot_value(slot)); /* slot is semipermanent? does this assume slot_value is not rootlet? or that rootlet is marked? */
  /* slot_setter is handled below with an explicit list -- more code than its worth probably */
  /* we're not marking slot_symbol above which makes me worry that a top-level gensym won't be protected
   *   (apply define (gensym) '(32)), then try to get the GC to clobber {gensym}-0,
   *   but I can't get it to break, so they must be protected somehow; apparently they are
   *   removed from the heap!  At least: (define-macro (defit) (let ((n (gensym))) `(define (,n) (format #t "fun")))) (defit)
   *   removes the function from the heap (protecting the gensym).
   */
}

/* mark_closure calls mark_let on closure_let(func) which marks slot values.
 *   if we move rootlet to end, unmarked closures at that point could mark let/slot but not slot value?
 *   or save safe-closure lets to handle all at end?  or a gc_list of safe closure lets and only mark let if not safe?
 */

static void mark_semipermanent_objects(s7_scheme *sc)
{
  for (gc_obj_t *g = sc->semipermanent_objects; g; g = (gc_obj_t *)(g->nxt))
    gc_mark(g->p);
  /* semipermanent_objects also has lets (removed from heap) -- should they be handled like semipermanent_lets?
   *    if unmarked should either be removed from the list and perhaps placed on a free list?
   *    if outlet is free can the let potentially be in use?
   *    there are many more semipermanent_lets(slots) than semipermanent objects
   */
}
/* do we mark funclet slot values from the function as root?  Maybe treat them like semipermanent_lets here? */

static void unmark_semipermanent_objects(s7_scheme *sc)
{
  for (gc_obj_t *g = sc->semipermanent_objects; g; g = (gc_obj_t *)(g->nxt))
    clear_mark(g->p);
  for (gc_obj_t *g = sc->semipermanent_lets; g; g = (gc_obj_t *)(g->nxt)) /* there are lets and slots in this list */
    clear_mark(g->p);
}

#if !MS_Windows
  #include <time.h>
  #include <sys/time.h>
#endif

#if s7_Debugging
static s7_int gc(s7_scheme *sc, const char *func, int32_t line)
#else
static s7_int gc(s7_scheme *sc)
#endif
{
  s7_cell **old_free_heap_top;

  if (sc->gc_in_progress)
    error_nr(sc, sc->error_symbol, set_elist_1(sc, wrap_string(sc, "GC called recursively", 21)));
  sc->gc_in_progress = true;
  sc->gc_start = my_clock();
  sc->gc_calls++;
  sc->gc_true_calls++;

  mark_rootlet(sc);
  mark_owlet(sc);
  gc_mark(sc->code);
  if ((s7_Debugging) && (!(sc->args))) {fprintf(stderr, "%d: sc->args is NULL\n", __LINE__); abort();}
  gc_mark(sc->args);
  gc_mark(sc->curlet);   /* not mark_let because op_any_closure_3p uses sc->curlet as a temp!! */
  mark_current_code(sc); /* probably redundant if with_history */
  gc_mark(sc->value);

  mark_stack_1(sc->stack, stack_top(sc));
  set_mark(current_input_port(sc));
  mark_input_port_stack(sc);
  set_mark(current_output_port(sc));
  set_mark(current_error_port(sc));
  mark_pair(sc->stacktrace_defaults);
  gc_mark(sc->autoload_table);        /* () or a hash-table */
  set_mark(sc->default_random_state); /* always a random_state object */
  gc_mark(sc->temp_error_hook);

  gc_mark(sc->v);
  gc_mark(sc->w);
  gc_mark(sc->x);
  gc_mark(sc->y);
  gc_mark(sc->z);
  gc_mark(sc->temp1);
  gc_mark(sc->temp2);
  gc_mark(sc->temp3);
  gc_mark(sc->temp4);
  gc_mark(sc->temp5);
  gc_mark(sc->temp6);
  gc_mark(sc->temp7);
  gc_mark(sc->temp8);
  gc_mark(sc->temp9);
  just_mark(sc->read_dims);

  gc_mark(car(sc->t1_1));
  gc_mark(car(sc->t2_1)); gc_mark(car(sc->t2_2));
  gc_mark(car(sc->t3_1)); gc_mark(car(sc->t3_2)); gc_mark(car(sc->t3_3));
  gc_mark(car(sc->t4_1));
  gc_mark(car(sc->mlist_1));
  gc_mark(car(sc->mlist_2)); gc_mark(cadr(sc->mlist_2));
  gc_mark(car(sc->plist_1));
  gc_mark(car(sc->plist_2)); gc_mark(car(sc->plist_2_2));
  gc_mark(car(sc->plist_3)); gc_mark(cadr(sc->plist_3)); gc_mark(caddr(sc->plist_3)); gc_mark(car(sc->plist_4));
  gc_mark(car(sc->qlist_2)); gc_mark(cadr(sc->qlist_2));
  gc_mark(car(sc->qlist_3));
  gc_mark(car(sc->u1_1));
  gc_mark(sc->rec_p1);
  gc_mark(sc->rec_p2);

  /* these do need to be marked, at least protecting "info" for the duration of the error handler procedure */
  for (s7_pointer p = cdr(sc->wrong_type_arg_info); is_pair(p); p = cdr(p)) gc_mark(car(p));
  for (s7_pointer p = cdr(sc->sole_arg_wrong_type_info); is_pair(p); p = cdr(p)) gc_mark(car(p));
  for (s7_pointer p = cdr(sc->out_of_range_info); is_pair(p); p = cdr(p)) gc_mark(car(p));
  for (s7_pointer p = cdr(sc->sole_arg_out_of_range_info); is_pair(p); p = cdr(p)) gc_mark(car(p));

  gc_mark(car(sc->elist_1));
  gc_mark(car(sc->elist_2)); gc_mark(cadr(sc->elist_2));
  gc_mark(car(sc->elist_3)); gc_mark(cadr(sc->elist_3)); gc_mark(caddr(sc->elist_3));
  gc_mark(car(sc->elist_4));
  gc_mark(car(sc->elist_5));
  gc_mark(car(sc->elist_6));
  gc_mark(car(sc->elist_7));

  for (s7_int i = 1; i < Num_Safe_Lists; i++) /* see tgen.scm ([18] often in use) -- we can't just check sc->current_safe_list */
    if ((is_pair(sc->safe_lists[i])) &&
	(safe_list_is_in_use(sc->safe_lists[i]))) /* safe_lists are semipermanent, so we have to mark contents by hand */
      for (s7_pointer p = sc->safe_lists[i]; is_pair(p); p = cdr(p))
	gc_mark(car(p));

  for (s7_int i = 0; i < sc->setters_loc; i++)
    gc_mark(cdr(sc->setters[i]));

  for (s7_int i = 0; i <= sc->format_depth; i++) /* sc->num_fdats is size of array */
    if (sc->fdats[i])
      gc_mark(sc->fdats[i]->curly_arg);

  if (sc->rec_stack)
    {
      set_mark(sc->rec_stack);
      for (s7_int i = 0; i < sc->rec_loc; i++)
	gc_mark(sc->rec_els[i]);
    }
  mark_vector(sc->protected_objects);
  mark_vector(sc->protected_setters);
  set_mark(sc->protected_setter_symbols);
  if ((is_symbol(sc->profile_prefix)) && (is_gensym(sc->profile_prefix))) set_mark(sc->profile_prefix);
  gc_mark(sc->symbol_printer);
  gc_mark(sc->saved_sharp_readers);

  /* protect recent allocations using the free_heap cells above the current free_heap_top (if any).
   * cells above sc->free_heap_top might be malloc'd garbage (after heap reallocation), so we keep track of
   *   where the last actually freed cells were after the previous GC call.  We're trying to
   *   GC protect the previous Gc_Temps_Size allocated pointers so that the caller doesn't have
   *   to gc-protect every temporary cell.
   */
  {
    s7_pointer *tmps = sc->free_heap_top;
    s7_pointer *tmps_top = tmps + sc->gc_temps_size;
    if (tmps_top > sc->previous_free_heap_top)
      tmps_top = sc->previous_free_heap_top;
    while (tmps < tmps_top)
      gc_mark(*tmps++);
  }
  mark_op_stack(sc);
  mark_semipermanent_objects(sc);
  mark_wrappers(sc);

  if (sc->profiling_gensyms)
    {
      profile_data_t *pd = sc->profile_data;
      for (s7_int i = 0; i < pd->top; i++)
	if ((pd->funcs[i]) && (is_gensym(pd->funcs[i])))
	  set_mark(pd->funcs[i]);
    }

  {
    gc_list_t *gp = sc->opt1_funcs;
    for (s7_int i = 0; i < gp->loc; i++)
      {
	s7_pointer gc_obj = T_Pair(gp->list[i]);
	if ((is_marked(gc_obj)) && (!is_marked(opt1_any(gc_obj)))) /* opt1_lambda, but op_unknown* can change to opt1_cfunc etc */
	  gc_mark(opt1_any(gc_obj));                               /* not set_mark -- need to protect let/body/args as well */
      }}

  /* free up all unmarked objects */
  old_free_heap_top = sc->free_heap_top;
  {
    s7_pointer *fp = sc->free_heap_top;
    s7_pointer *tp = sc->heap;
    s7_pointer *heap_top = (s7_pointer *)(sc->heap + sc->heap_size);

#if s7_Debugging
  #define gc_object(Tp)							\
    p = (*Tp++);							\
    if (signed_type(p) > 0)						\
      {								        \
        p->debugger_bits = 0; p->gc_func = func; p->gc_line = line;	\
        if (has_odd_bits(p)) {char *str; fprintf(stderr, "odd bits: %s\n", str = describe_type_bits(sc, p)); free(str);} \
	if (!in_heap(p)) {char *str; fprintf(stderr, "not in heap: %s\n", str = describe_type_bits(sc, p)); free(str);} \
        clear_type(p);							\
        (*fp++) = p;							\
      }									\
    else if (signed_type(p) < 0) clear_mark(p);
#else
  #define gc_object(Tp) p = (*Tp++); if (signed_type(p) > 0) {clear_type(p); (*fp++) = p;} else if (signed_type(p) != 0) clear_mark(p);
  /* this appears to be about 10% faster than the previous form, using !=0 is about the same as <0
   *   if the sign bit is on, but no other bits, this version will take no action (it thinks the cell is on the free list), but
   *   it means we've marked a free cell as in-use: since types are set as soon as removed from the free list, this has to be a bug
   *   (this case is caught by has_odd_bits).  If ignored, the type will be set, and later the bit cleared, so no problem?
   *   An alternate form that simply calls clear_mark (no check for < 0) appears to be the same speed even in cases with lots
   *   of long-lived objects.
   */
#endif
    while (tp < heap_top)          /* != here or ^ makes no difference, and going to 64 (from 32) doesn't matter */
      {
	s7_pointer p; /* gc_object defined above needs this pointer */
	Loop_8(gc_object(tp));
	Loop_8(gc_object(tp));
	Loop_8(gc_object(tp));
	Loop_8(gc_object(tp));
      }
    /* I tried using pthreads here, since there is no need for a lock in this loop, but the *fp++ part needs to
     *   be local to each thread, then merged at the end.  In my timing tests, the current version was faster.
     *   If NUM_THREADS=2, and all thread variables are local, surely there's no "false sharing"?
     */
    sc->free_heap_top = fp;
    sweep(sc);
  }

  unmark_semipermanent_objects(sc);
  unmark_wrappers(sc);

  sc->gc_freed = (s7_int)(sc->free_heap_top - old_free_heap_top);
  sc->gc_total_freed += sc->gc_freed;
  sc->gc_end = my_clock();
  sc->gc_total_time += (sc->gc_end - sc->gc_start);
  sc->gc_true_total_time += (sc->gc_end - sc->gc_start);

  if (show_gc_stats(sc))
    {
#if !MS_Windows
      s7_warn(sc, s7_Warn_Bufsize, "gc freed %" ld64 "/%" ld64 " (free: %" p64 "), time: %f\n",
	      sc->gc_freed, sc->heap_size, (intptr_t)(sc->free_heap_top - sc->free_heap), (double)(sc->gc_end - sc->gc_start) / ticks_per_second());
#else
      s7_warn(sc, s7_Warn_Bufsize, "gc freed %" ld64 "/%" ld64 "\n", sc->gc_freed, sc->heap_size);
#endif
    }
  if (show_protected_objects_stats(sc))
    s7_warn(sc, s7_Warn_Bufsize, "gc-protected-objects: %" ld64 " in use of %" ld64 "\n",
	    sc->protected_objects_size - 1 - sc->protected_objects_free_list_loc,
	    sc->protected_objects_size);
  sc->previous_free_heap_top = sc->free_heap_top;
  sc->gc_in_progress = false;
  return(sc->gc_freed);
}


#ifndef GC_RESIZE_HEAP_FRACTION
  #define GC_RESIZE_HEAP_FRACTION 0.8
/* 1/2 is ok, 3/4 speeds up some GC benchmarks, 7/8 is a bit faster, 95/100 comes to a halt (giant heap)
 *    in my tests, only tvect.scm ends up larger if 3/4 used
 */
#endif
#define Gc_Resize_Heap_Fraction GC_RESIZE_HEAP_FRACTION

#define Gc_Resize_Heap_By_4_Fraction 0.67
/*   .5+.1: test -3?, dup +86, tmap +45, tsort -3, thash +305.  .85+.7: dup -5 */

#if s7_Debugging
#define resize_heap_to(Sc, Size) resize_heap_to_1(Sc, Size, __func__, __LINE__)
static void resize_heap_to_1(s7_scheme *sc, s7_int size, const char *func, int line)
#else
static void resize_heap_to(s7_scheme *sc, s7_int size)
#endif
{
  const s7_int old_size = sc->heap_size;
  const s7_int old_free = sc->free_heap_top - sc->free_heap;

#if s7_Debugging && (!MS_Windows)
  if (show_gc_stats(sc))
    s7_warn(sc, 512, "%s from %s[%d]: old: %" ld64 " / %" ld64 ", new: %" ld64 ", fraction: %.3f -> %" ld64 "\n",
	    __func__, func, line, old_free, old_size, size, sc->gc_resize_heap_fraction, (s7_int)(floor(sc->heap_size * sc->gc_resize_heap_fraction)));
#endif

  if (size == 0)
    {
      if ((old_free < old_size * sc->gc_resize_heap_by_4_fraction) &&
	  (sc->max_heap_size > (sc->heap_size * 4)))
	sc->heap_size *= 4;          /* *8 if < 1M (or whatever) doesn't make much difference */
      else sc->heap_size *= 2;
      if ((s7_Debugging) && (sc->heap_size >= sc->max_heap_size))
	fprintf(stderr, "%s[%d]: heap_size: %" ld64 ", max: %" ld64 "\n", __func__, __LINE__, sc->heap_size, sc->max_heap_size);
      if (sc->gc_resize_heap_fraction > .4)
	sc->gc_resize_heap_fraction *= .95;
    }
  else
    {
      if (size > sc->heap_size)
	while (sc->heap_size < size) sc->heap_size *= 2;
      else return;
      if ((s7_Debugging) && (sc->heap_size >= sc->max_heap_size))
	fprintf(stderr, "%s[%d]: heap_size: %" ld64 ", max: %" ld64 "\n", __func__, __LINE__, sc->heap_size, sc->max_heap_size);
    }
  if (sc->heap_size >= sc->max_heap_size)
    {
      const s7_int new_heap_size = 32 * (s7_int)floor(sc->max_heap_size / 32.0);
      if (new_heap_size > old_size)
	{
	  s7_warn(sc, s7_Warn_Bufsize, "heap size requested is greater than (*s7* 'max-heap-size); trying %" ld64 "\n", new_heap_size);
	  sc->heap_size = new_heap_size;
	  if ((s7_Debugging) && (sc->heap_size >= sc->max_heap_size))
	    fprintf(stderr, "%s[%d]: heap_size: %" ld64 ", max: %" ld64 "\n", __func__, __LINE__, sc->heap_size, sc->max_heap_size);
	}
      else
	{
	  const s7_int new_size = sc->heap_size;
	  sc->heap_size = old_size; /* needed if user catches this error and (for example) runs (*s7* 'memory-usage) in the error handler */
	  error_nr(sc, make_symbol(sc, "heap-too-big", 12),
		   set_elist_3(sc, wrap_string(sc, "heap has grown past (*s7* 'max-heap-size): ~D > ~D", 50),
			       wrap_integer(sc, new_size),
			       wrap_integer(sc, sc->max_heap_size)));
	  return;
	}}

  /* do not call new_cell here! */
#if Pointer_32
  if (((2 * sc->heap_size * sizeof(s7_cell *)) + ((sc->heap_size - old_size) * sizeof(s7_cell))) >= SIZE_MAX)
    { /* can this happen in 64-bit land?  SIZE_MAX is unsigned int in 32-bit, unsigned long in 64 bit = UINTPTR_MAX = 18446744073709551615UL */
      s7_warn(sc, s7_Warn_Bufsize, "heap size requested, %" ld64 " => %" ld64 " bytes, is greater than size_t: %" ld64 "\n",
	      sc->heap_size,
	      (2 * sc->heap_size * sizeof(s7_cell *)) + ((sc->heap_size - old_size) * sizeof(s7_cell)),
	      (s7_int)SIZE_MAX);
      sc->heap_size = old_size + 64000;
      if ((s7_Debugging) && (sc->heap_size >= sc->max_heap_size))
	fprintf(stderr, "%s[%d]: heap_size: %" ld64 ", max: %" ld64 "\n", __func__, __LINE__, sc->heap_size, sc->max_heap_size);
    }
#endif
  {
    s7_cell **cp = (s7_cell **)Realloc(sc->heap, sc->heap_size * sizeof(s7_cell *));
    if (cp)
      sc->heap = cp;
    else /* can this happen? */
      {
	s7_warn(sc, s7_Warn_Bufsize, "heap reallocation failed! tried to get %" ld64 " bytes (will retry with a smaller amount)\n",
		(s7_int)(sc->heap_size * sizeof(s7_cell *)));
	sc->heap_size = old_size + 64000;
	if ((s7_Debugging) && (sc->heap_size >= sc->max_heap_size))
	  fprintf(stderr, "%s[%d]: heap_size: %" ld64 ", max: %" ld64 "\n", __func__, __LINE__, sc->heap_size, sc->max_heap_size);
	sc->heap = (s7_cell **)Realloc(sc->heap, sc->heap_size * sizeof(s7_cell *));
      }}
  sc->free_heap = (s7_cell **)Realloc(sc->free_heap, sc->heap_size * sizeof(s7_cell *));
  sc->free_heap_trigger = (s7_cell **)(sc->free_heap + Gc_Trigger_Size);
  sc->free_heap_top = sc->free_heap + old_free; /* incremented below, added old_free 21-Aug-12?!? */

  {
    s7_cell *cells = (s7_cell *)Calloc(sc->heap_size - old_size, sizeof(s7_cell)); /* Malloc + clear_type below is much slower?! */
    add_saved_pointer(sc, (void *)cells);
    {
      s7_pointer p = cells;
      for (s7_int k = old_size; k < sc->heap_size;)
	{
	  Loop_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	  Loop_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	  Loop_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	  Loop_8(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);
	}}
    {
      heap_block_t *hp = (heap_block_t *)Malloc(sizeof(heap_block_t));
      hp->start = (intptr_t)cells;
      hp->end = (intptr_t)cells + ((sc->heap_size - old_size) * sizeof(s7_cell));
      hp->offset = old_size;
      hp->next = sc->heap_blocks;
      sc->heap_blocks = hp;
    }}
  sc->previous_free_heap_top = sc->free_heap_top;

  if (show_heap_stats(sc))
    {
      if (size != 0)
	s7_warn(sc, 512, "heap grows to %" ld64 " (old free/size: %" ld64 "/%" ld64 ", requested %" ld64 ")\n",
		sc->heap_size, old_free, old_size, size);
      else s7_warn(sc, 512, "heap grows to %" ld64 " (old free/size: %" ld64 "/%" ld64 ", %.3f)\n",
		   sc->heap_size, old_free, old_size, sc->gc_resize_heap_fraction);
    }
}

#define resize_heap(Sc) resize_heap_to(Sc, 0)

#if s7_Debugging
#define call_gc(Sc) gc(Sc, __func__, __LINE__)
static void try_to_call_gc_1(s7_scheme *sc, const char *func, int32_t line)
#else
#define call_gc(Sc) gc(Sc)
static void try_to_call_gc(s7_scheme *sc)
#endif
{
  /* called only from new_cell */
  if (sc->gc_off)     /* we can't just return here!  Someone needs a new cell, and once the heap free list is exhausted, segfault */
    {
#if s7_Debugging
      fprintf(stderr, "%s[%d]: forced resize from %s[%d]\n", __func__, __LINE__, func, line);
#endif
      resize_heap(sc);
    }
  else
    {
      if ((sc->gc_resize_heap_fraction > 0.5) && (sc->heap_size >= 4194304))
	  sc->gc_resize_heap_fraction = 0.5;
#if s7_Debugging
      gc(sc, func, line); /* not call_gc! */
      /* describe_gc_strings(sc); */
#else
      gc(sc);
#endif
      if ((s7_int)(sc->free_heap_top - sc->free_heap) < (sc->heap_size * sc->gc_resize_heap_fraction)) /* changed 21-Jul-22 */
	resize_heap(sc);
    }
}
  /* originally I tried to mark each temporary value until I was done with it, but that way madness lies... By delaying
   *   GC of _every_ %$^#%@ pointer, I can dispense with hundreds of individual protections.  So the free_heap's last
   *   Gc_Temps_Size allocated pointers are protected during the mark sweep.
   */

static s7_pointer g_gc(s7_scheme *sc, s7_pointer args)
{
  #define H_gc "(gc (on #t)) runs the garbage collector.  If 'on' (a boolean) is supplied, it turns the GC on or off. \
Evaluation produces a surprising amount of garbage, so don't leave the GC off for very long!"
  #define Q_gc s7_make_signature(sc, 2, sc->T, sc->is_boolean_symbol)

  set_elist_1(sc, sc->unused);
  set_elist_2(sc, sc->unused, sc->unused);
  set_elist_3(sc, sc->unused, sc->unused, sc->unused);
  set_car(sc->elist_4, sc->unused);
  set_car(sc->elist_5, sc->unused);
  set_car(sc->elist_6, sc->unused);
  set_car(sc->elist_7, sc->unused); /* clist and dlist are weak references */
  if (is_pair(args))
    {
      if (!is_boolean(car(args)))
	return(sole_arg_method_or_bust(sc, car(args), sc->gc_symbol, args, sc->type_names[T_BOOLEAN]));
      sc->gc_off = (car(args) == sc->F);
      if (sc->gc_off)
	return(sc->F);
    }
  call_gc(sc);
  return(sc->unspecified);
}

s7_pointer s7_gc_on(s7_scheme *sc, bool on)
{
  sc->gc_off = !on;
  return(make_boolean(sc, on));
}

#if s7_Debugging
static void check_free_heap_size_1(s7_scheme *sc, s7_int size, const char *func, int32_t line)
#define check_free_heap_size(Sc, Size) check_free_heap_size_1(Sc, Size, __func__, __LINE__)
#else
static void check_free_heap_size(s7_scheme *sc, s7_int size)
#endif
{
  s7_int free_cells = sc->free_heap_top - sc->free_heap;
  if (free_cells < size)
    {
#if s7_Debugging
      gc(sc, func, line);
#else
      gc(sc);
#endif
      while ((sc->free_heap_top - sc->free_heap) < (s7_int)(size * 1.5))
	resize_heap(sc);
    }
}

#define Alloc_Pointer_Size 256
static s7_cell *alloc_pointer(s7_scheme *sc)
{
  if (sc->alloc_pointer_k == Alloc_Pointer_Size)     /* if either no current block or the block is used up, make a new block */
    {
      sc->semipermanent_cells += Alloc_Pointer_Size;
      sc->alloc_pointer_cells = (s7_cell *)Calloc(Alloc_Pointer_Size, sizeof(s7_cell)); /* not Malloc here or below (maybe set full type to 0 if Malloc) */
      add_saved_pointer(sc, sc->alloc_pointer_cells);
      sc->alloc_pointer_k = 0;
    }
  return(&(sc->alloc_pointer_cells[sc->alloc_pointer_k++]));
}

#define Alloc_Big_Pointer_Size 256
static s7_big_cell *alloc_big_pointer(s7_scheme *sc, s7_int loc)
{
  s7_big_pointer p;
  if (sc->alloc_big_pointer_k == Alloc_Big_Pointer_Size)
    {
      sc->semipermanent_cells += Alloc_Big_Pointer_Size;
      sc->alloc_big_pointer_cells = (s7_big_cell *)Calloc(Alloc_Big_Pointer_Size, sizeof(s7_big_cell));
      add_saved_pointer(sc, sc->alloc_big_pointer_cells);
      sc->alloc_big_pointer_k = 0;
    }
  p = (&(sc->alloc_big_pointer_cells[sc->alloc_big_pointer_k++]));
  p->big_hloc = loc;
  /* needed if this new pointer is itself petrified later -- it's not from one of the heap blocks,
   *   but it's in the heap, and we'll need to know where it is in the heap to replace it
   */
  return(p);
}

static void add_semipermanent_object(s7_scheme *sc, s7_pointer obj) /* called by remove_from_heap */
{
  gc_obj_t *g = (gc_obj_t *)Malloc(sizeof(gc_obj_t));
  g->p = obj;
  g->nxt = sc->semipermanent_objects;
  sc->semipermanent_objects = g;
}

static void add_semipermanent_let_or_slot(s7_scheme *sc, s7_pointer obj)
{
  gc_obj_t *g = (gc_obj_t *)Malloc(sizeof(gc_obj_t));
  g->p = obj;
  g->nxt = sc->semipermanent_lets;
  sc->semipermanent_lets = g;
}

static inline s7_pointer petrify(s7_scheme *sc, s7_pointer x)
{
  const s7_int loc = heap_location(sc, x);
  const s7_pointer p = (s7_pointer)alloc_big_pointer(sc, loc);
  sc->heap[loc] = p;
  (*(sc->free_heap_top++)) = p;
  unheap(x);        /* set_immutable(x); */ /* if there are GC troubles, this might catch them? */
  return(x);
}

#if s7_Debugging
#define remove_gensym_from_heap(Sc, Gensym) remove_gensym_from_heap_1(Sc, Gensym, __func__, __LINE__)
static void remove_gensym_from_heap_1(s7_scheme *sc, s7_pointer x, const char *func, int line)
#else
static void remove_gensym_from_heap(s7_scheme *sc, s7_pointer x) /* x known to be a symbol and in the heap */
#endif
{
  const s7_int loc = heap_location(sc, x);
  sc->heap[loc] = (s7_pointer)alloc_big_pointer(sc, loc);
  (*(sc->free_heap_top++)) = sc->heap[loc];
#if s7_Debugging
  x->gc_func = func; /* main culprit in s7test/t725 is (essentially) (symbol->keyword (gensym)) */
  x->gc_line = line;
#endif
  unheap(x); /* set T_Unheap bit in type(x) */
  {
    gc_list_t *gp = sc->gensyms;
    for (s7_int i = 0; i < gp->loc; i++) /* sc->gensyms reaches size 512 during s7test, but this search is called 3 times and costs nothing */
      if (gp->list[i] == x)
	{
	  for (s7_int j = i + 1; i < gp->loc - 1; i++, j++)
	    gp->list[i] = gp->list[j];
	  gp->list[i] = NULL;
	  gp->loc--;
	  if (gp->loc == 0) mark_function[T_SYMBOL] = mark_noop;
	  break;
	}}
}

static inline void remove_from_heap(s7_scheme *sc, s7_pointer x)
{
  /* global functions are very rarely redefined, so we can remove the function body from the heap when it is defined */
  if (!in_heap(x)) return;
  if (is_pair(x))   /* all the compute time is here, might be faster to go down a level explicitly */
    {
      s7_pointer p = x;
      do {
	petrify(sc, p);
	remove_from_heap(sc, car(p));
	p = cdr(p);
      } while (is_pair(p) && (in_heap(p)));
      if (in_heap(p)) petrify(sc, p);
      return;
    }
  switch (type(x))
    {
    case T_LET: /* very rare */
      if (is_funclet(x)) set_immutable_let(x);
    case T_HASH_TABLE:
    case T_VECTOR:
      /* not byte|int|float|complex_vector or string because none of their elements are GC-able (so unheap below is ok)
       *   but hash-table and let seem like they need protection? And let does happen via define-class.
       */
      add_semipermanent_object(sc, x);
      return;
    case T_SYMBOL:
      if (is_gensym(x))
	remove_gensym_from_heap(sc, x);
      return;
    case T_CLOSURE: case T_CLOSURE_STAR:
    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
      /* these need to be GC-protected! */
      add_semipermanent_object(sc, x);
      return;
    default: break;
    }
  petrify(sc, x);
}


/* -------------------------------- stacks -------------------------------- */

/* -------- op stack -------- */
#define Op_Stack_Initial_Size 64

#define op_stack_entry(Sc) (*(Sc->op_stack_now - 1))

#if s7_Debugging
static void push_op_stack(s7_scheme *sc, s7_pointer op)
{
  (*sc->op_stack_now++) = T_Ext(op); /* not T_App etc -- args can be pushed */
  if (sc->op_stack_now > (sc->op_stack + sc->op_stack_size))
    {fprintf(stderr, "%sop_stack overflow%s\n", Bold_Text, Unbold_Text); abort();}
}

static s7_pointer pop_op_stack(s7_scheme *sc)
{
  s7_pointer op = T_Ext(*(--(sc->op_stack_now)));
  if (sc->op_stack_now < sc->op_stack)
    {fprintf(stderr, "%sop_stack underflow%s\n", Bold_Text, Unbold_Text); abort();}
  return(T_Ext(op));
}
#else
#define push_op_stack(Sc, Op) (*Sc->op_stack_now++) = Op
#define pop_op_stack(Sc)      (*(--(Sc->op_stack_now)))
#endif

static void initialize_op_stack(s7_scheme *sc)
{
  sc->op_stack = (s7_pointer *)Malloc(Op_Stack_Initial_Size * sizeof(s7_pointer));
  sc->op_stack_size = Op_Stack_Initial_Size;
  sc->op_stack_now = sc->op_stack;
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
  for (int32_t i = 0; i < Op_Stack_Initial_Size; i++) sc->op_stack[i] = sc->unused;
}

static void resize_op_stack(s7_scheme *sc)
{
  const uint32_t new_size = sc->op_stack_size * 2;
  const uint32_t loc = (uint32_t)(sc->op_stack_now - sc->op_stack);
  if (new_size > sc->max_stack_size)
#if s7_Debugging
    {fprintf(stderr, "%s%s[%d]: op stack will be too big after resize, %u > %u%s\n", Bold_Text, __func__, __LINE__, new_size, sc->max_stack_size, Unbold_Text); abort();}
#else
    error_nr(sc, make_symbol(sc, "stack-too-big", 13),
	     set_elist_3(sc, wrap_string(sc, "op stack has grown past (*s7* 'max-stack-size): ~D > ~D", 55),
			 wrap_integer(sc, (s7_int)new_size),
			 wrap_integer(sc, (s7_int)sc->max_stack_size)));
#endif
  sc->op_stack = (s7_pointer *)Realloc((void *)(sc->op_stack), new_size * sizeof(s7_pointer));
  for (uint32_t i = sc->op_stack_size; i < new_size; i++) sc->op_stack[i] = sc->unused;
  sc->op_stack_size = (uint32_t)new_size;
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + loc);
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
}


/* -------- main stack -------- */
/* stack_top_code changes.  If a function has a tail-call, the stack_top_code that form sees
 *   if stack_top_op==op-begin1 can change from call to call -- the begin actually refers
 *   to the caller, which is dependent on where the current function was called, so we can't hard-wire
 *   any optimizations based on that sequence.
 */

#define stack_op(Stack, Loc)         ((opcode_t)T_Op(stack_element(Stack, Loc)))
#define stack_args(Stack, Loc)       stack_element(Stack, Loc - 1)
#define stack_let(Stack, Loc)        stack_element(Stack, Loc - 2)
#define stack_code(Stack, Loc)       stack_element(Stack, Loc - 3)
#define set_stack_op(Stack, Loc, Op) stack_element(Stack, Loc) = (s7_pointer)(opcode_t)(Op)

#define stack_top_op(Sc)             ((opcode_t)T_Op(Sc->stack_end[-1]))
#define stack_top_op_unchecked(Sc)   ((opcode_t)(Sc->stack_end[-1]))
#define stack_top_args(Sc)           (Sc->stack_end[-2])
#define stack_top_let(Sc)            (Sc->stack_end[-3])
#define stack_top_code(Sc)           (Sc->stack_end[-4])
#define set_stack_top_op(Sc, Op)     Sc->stack_end[-1] = (s7_pointer)(opcode_t)(Op)
#define set_stack_top_args(Sc, Args) Sc->stack_end[-2] = Args
#define set_stack_top_code(Sc, Code) Sc->stack_end[-4] = Code

#define stack_end_code(Sc) Sc->stack_end[0]
#define stack_end_let(Sc)  Sc->stack_end[1]
#define stack_end_args(Sc) Sc->stack_end[2]
#define stack_end_op(Sc)   Sc->stack_end[3]

#if s7_Debugging
void s7_show_stack(s7_scheme *sc);

#define pop_stack(Sc) pop_stack_1(Sc, __func__, __LINE__)
static void pop_stack_1(s7_scheme *sc, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start)
    {fprintf(stderr, "%s%s[%d]: stack underflow%s\n", Bold_Text, func, line, Unbold_Text); abort();}
  /* here and in push_stack, both code and args might be non-free only because they've been retyped
   *   inline (as in named let) -- they actually don't make sense in these cases, but are ignored,
   *   and are carried around as GC protection in other cases.
   */
  sc->code = T_Pos(stack_end_code(sc));
  sc->curlet = stack_end_let(sc);  /* not T_Let|Pos (hence not set_curlet), see op_any_closure_3p_end et al (stack used to pass args, not curlet) */
  sc->args = stack_end_args(sc);
  sc->cur_op = (opcode_t)T_Op(stack_end_op(sc));
  if ((sc->cur_op != OP_GC_PROTECT) &&
      (!is_let(stack_end_let(sc))) && (!is_null(stack_end_let(sc))) &&
      (sc->cur_op != OP_ANY_CLOSURE_3P_3)) /* used as third GC protection field */
    fprintf(stderr, "%s[%d]: curlet not a let: %s\n", func, line, op_names[sc->cur_op]);
}

#define pop_stack_no_op(Sc) pop_stack_no_op_1(Sc, __func__, __LINE__)
static void pop_stack_no_op_1(s7_scheme *sc, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start)
    {fprintf(stderr, "%s%s[%d]: stack underflow%s\n", Bold_Text, func, line, Unbold_Text); abort();}
  sc->code = T_Pos(stack_end_code(sc));
  if ((sc->cur_op != OP_GC_PROTECT) && (!is_let(stack_end_let(sc))))
    fprintf(stderr, "%s[%d]: curlet not a let\n", func, line);
  sc->curlet = stack_end_let(sc); /* not T_Let|Pos: gc_protect can set this directly (not through push_stack) to anything */
  sc->args = stack_end_args(sc);
}

static void push_stack_1(s7_scheme *sc, opcode_t op, s7_pointer args, s7_pointer code, s7_pointer *end, const char *func, int32_t line)
{
  if ((Show_Eval_Ops) && (op == OP_EVAL_DONE)) fprintf(stderr, "  %s[%d]: push eval_done\n", func, line);
  if (sc->stack_end >= sc->stack_start + sc->stack_size)
    {
      fprintf(stderr, "%s%s[%d]: stack overflow, %u > %u, trigger: %u %s\n",
	      Bold_Text, func, line,
	      (uint32_t)((intptr_t)(sc->stack_end - sc->stack_start)), sc->stack_size,
	      (uint32_t)((intptr_t)(sc->stack_resize_trigger - sc->stack_start)),
	      Unbold_Text);
      s7_show_stack(sc);
      abort();
    }
  if (sc->stack_end >= sc->stack_resize_trigger)
    {
      fprintf(stderr, "%s%s[%d] from %s: stack resize skipped, stack at %u of %u %s%s\n",
	      Bold_Text, func, line, op_names[op],
	      (uint32_t)((intptr_t)(sc->stack_end - sc->stack_start)),
	      sc->stack_size, display_truncated(code), Unbold_Text);
      s7_show_stack(sc);
    }
  if (sc->stack_end != end)
    fprintf(stderr, "%s[%d]: stack changed in push_stack\n", func, line);
  if (op >= Num_Ops)
    {fprintf(stderr, "%s%s[%d]: push_stack invalid opcode: %" p64 " %s\n", Bold_Text, func, line, sc->cur_op, Unbold_Text); abort();}
  if (code) stack_end_code(sc) = T_Pos(code);
  stack_end_let(sc) = T_Let(sc->curlet);
  if ((args) && (!is_free(args))) stack_end_args(sc) = T_Pos(args);
  stack_end_op(sc) = (s7_pointer)op;
  sc->stack_end += 4;
}

#define push_stack(Sc, Op, Args, Code)	\
  do {s7_pointer *_end_; _end_ = Sc->stack_end; push_stack_1(Sc, Op, Args, Code, _end_, __func__, __LINE__);} while (0)

#define push_stack_no_code(Sc, Op, Args)        push_stack(Sc, Op, Args, Sc->unused)
#define push_stack_no_let_no_code(Sc, Op, Args) push_stack(Sc, Op, Args, Sc->unused)
#define push_stack_no_args(Sc, Op, Code)        push_stack(Sc, Op, Sc->unused, Code)
#define push_stack_no_let(Sc, Op, Args, Code)   push_stack(Sc, Op, Args, Code)
#define push_stack_op(Sc, Op)                   push_stack(Sc, Op, Sc->unused, Sc->unused)
#define push_stack_op_let(Sc, Op)               push_stack(Sc, Op, Sc->unused, Sc->unused)
#define push_stack_direct(Sc, Op)               push_stack(Sc, Op, Sc->args, Sc->code)
#define push_stack_no_args_direct(Sc, Op)       push_stack(Sc, Op, Sc->unused, Sc->code)
/* in the non-debugging case, the sc->unused's here are not set, so we can (later) pop free cells */

#else

#define pop_stack(Sc)       do {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 4 * sizeof(s7_pointer));} while (0)
#define pop_stack_no_op(Sc) do {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 3 * sizeof(s7_pointer));} while (0)

#define push_stack(Sc, Op, Args, Code) \
  do { \
      stack_end_code(Sc) = Code; \
      stack_end_let(Sc) = Sc->curlet; \
      stack_end_args(Sc) = Args; \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_direct(Sc, Op) \
  do { \
      Sc->cur_op = Op; \
      memcpy((void *)(Sc->stack_end), (void *)Sc, 4 * sizeof(s7_pointer)); \
      /* stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); */ \
      Sc->stack_end += 4; \
  } while (0)
/* is this faster with cur_op because of the cast to s7_pointer, or is callgrind messing up memcpy stats?
 *   time's output is all over the map.  I think the cur_op form should be slower, but callgrind disagrees.
 */

#define push_stack_no_code(Sc, Op, Args) \
  do { \
      stack_end_let(Sc) = Sc->curlet; \
      stack_end_args(Sc) = Args; \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_let_no_code(Sc, Op, Args) \
  do { \
      stack_end_args(Sc) = Args; \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_args(Sc, Op, Code) \
  do { \
      stack_end_code(Sc) = Code; \
      stack_end_let(Sc) = Sc->curlet; \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_args_direct(Sc, Op) \
  do { \
      memcpy((void *)(Sc->stack_end), (void *)Sc, 2 * sizeof(s7_pointer));	\
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_let(Sc, Op, Args, Code) \
  do { \
      stack_end_code(Sc) = Code; \
      stack_end_args(Sc) = Args; \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_op(Sc, Op) \
  do { \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_op_let(Sc, Op) \
  do { \
      stack_end_let(Sc) = Sc->curlet; \
      stack_end_op(Sc) = (s7_pointer)(opcode_t)(Op); \
      Sc->stack_end += 4; \
  } while (0)
#endif
/* since we don't GC mark the stack past the stack_top, push_stack_no_args and friends can cause pop_stack to set
 *   sc->code and sc->args to currently free objects.
 */

#if s7_Debugging
#define unstack_with(Sc, Op) unstack_1(Sc, Op, __func__, __LINE__)
static void unstack_1(s7_scheme *sc, opcode_t op, const char *func, int32_t line)
{
  sc->stack_end -= 4;
  if ((opcode_t)T_Op(stack_end_op(sc)) != op)
    {
      fprintf(stderr, "%s%s[%d]: popped %s? (expected %s)%s\n", Bold_Text, func, line, op_names[(opcode_t)T_Op(stack_end_op(sc))], op_names[op], Unbold_Text);
      /* "popped apply" means we called something that went to eval+apply when we thought it was a safe function */
      fprintf(stderr, "    code: %s\n    args: %s\n", display(sc->code), display(sc->args));
      fprintf(stderr, "    cur_code: %s\n    estr: %s\n", display(current_code(sc)), display(s7_name_to_value(sc, "estr")));
      s7_show_stack(sc);
      abort();
    }
}
#define unstack_gc_protect(Sc) unstack_with(Sc, OP_GC_PROTECT)
#else
#define unstack_gc_protect(Sc) Sc->stack_end -= 4
#define unstack_with(Sc, op) Sc->stack_end -= 4
#endif

static void stack_reset(s7_scheme *sc)
{
  sc->stack_end = sc->stack_start;
  push_stack_op(sc, OP_EVAL_DONE);
}

static uint32_t resize_stack_unchecked(s7_scheme *sc)
{
  const s7_uint loc = stack_top(sc);
  const uint32_t new_size = sc->stack_size * 2;
  block_t *old_b = stack_block(sc->stack);
  block_t *new_b = reallocate(sc, old_b, new_size * sizeof(s7_pointer));
  block_info(new_b) = NULL;
  stack_block(sc->stack) = new_b;
  stack_elements(sc->stack) = (s7_pointer *)block_data(new_b);
  {
    s7_pointer *orig = stack_elements(sc->stack);
    s7_int i = sc->stack_size;
    const s7_int left = new_size - i - 8;
    while (i <= left)
      Loop_8(orig[i++] = sc->unused);
    for (; i < new_size; i++)
      orig[i] = sc->unused;
  }
  vector_length(sc->stack) = new_size;
  sc->stack_size = new_size;
  sc->stack_start = stack_elements(sc->stack);
  sc->stack_end = (s7_pointer *)(sc->stack_start + loc);
  sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + (new_size - Stack_Resize_Trigger));
  return(new_size);
}

#if s7_Debugging
void s7_show_stack(s7_scheme *sc)
{
  if (sc->stack_end >= sc->stack_resize_trigger)
    resize_stack_unchecked(sc);
  fprintf(stderr, "stack:\n");
  for (s7_int op_loc = stack_top(sc) - 1, j = 0; (op_loc >= 3) && (j < sc->max_show_stack_frames); op_loc -= 4, j++) /* s7_int (or s7_uint?) is correct -- not uint32_t */
    fprintf(stderr, "  %s\n", op_names[stack_op(sc->stack, op_loc)]);
}

void s7_show_full_stack(s7_scheme *sc);
void s7_show_full_stack(s7_scheme *sc)
{
  if (sc->stack_end >= sc->stack_resize_trigger)
    resize_stack_unchecked(sc);
  fprintf(stderr, "stack:\n");
  for (s7_int op_loc = stack_top(sc) - 1, j = 0; (op_loc >= 3) && (j < sc->max_show_stack_frames); op_loc -= 4, j++)
    {
      fprintf(stderr, "  %s: ", op_names[stack_op(sc->stack, op_loc)]);
      if (s7_is_valid(sc, stack_code(sc->stack, op_loc)))
	fprintf(stderr, "code: %s, ", display_truncated(stack_code(sc->stack, op_loc)));
      if (s7_is_valid(sc, stack_args(sc->stack, op_loc)))
	fprintf(stderr, "args: %s, ", display_truncated(stack_args(sc->stack, op_loc)));
      if ((stack_op(sc->stack, op_loc) != OP_GC_PROTECT) && (s7_is_valid(sc, stack_let(sc->stack, op_loc)))) /* this probably won't work */
	fprintf(stderr, "let: %s", display_truncated(stack_let(sc->stack, op_loc)));
      fprintf(stderr, "\n");
    }
}

#define resize_stack(Sc) resize_stack_1(Sc, __func__, __LINE__)
static void resize_stack_1(s7_scheme *sc, const char *func, int line)
{
  if ((sc->stack_size * 2) > sc->max_stack_size)
    {
      fprintf(stderr, "%s%s[%d]: stack will be too big after resize, %u > %u, trigger: %" ld64 "%s\n",
	      Bold_Text, func, line, sc->stack_size * 2, sc->max_stack_size,
	      (s7_int)((intptr_t)(sc->stack_resize_trigger - sc->stack_start)),
	      Unbold_Text);
      s7_show_stack(sc);
      abort();
    }
  resize_stack_unchecked(sc);
}
#else
static void resize_stack(s7_scheme *sc)
{
  const uint32_t new_size = resize_stack_unchecked(sc);
  if (show_stack_stats(sc))
    s7_warn(sc, s7_Warn_Bufsize, "stack grows to %u\n", new_size);
  if (new_size > sc->max_stack_size)
    error_nr(sc, make_symbol(sc, "stack-too-big", 13),
	     set_elist_3(sc, wrap_string(sc, "stack has grown past (*s7* 'max-stack-size): ~D > ~D", 52),
			 wrap_integer(sc, new_size),
			 wrap_integer(sc, sc->max_stack_size)));
  /* error needs to follow realloc, else error -> catchers in error_nr -> let_temp* -> eval_done -> stack_resize -> infinite loop */
}
#endif

#define check_stack_size(Sc) do {if (Sc->stack_end >= Sc->stack_resize_trigger) resize_stack(Sc);} while (0)

s7_pointer s7_gc_protect_via_stack(s7_scheme *sc, s7_pointer x)
{
  check_stack_size(sc); /* this can be called externally, so we need to be careful about this */
  push_stack_no_code(sc, OP_GC_PROTECT, x);
  return(x);
}

s7_pointer s7_gc_protect_2_via_stack(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  check_stack_size(sc);
  push_stack(sc, OP_GC_PROTECT, x, y);
  return(x);
}

s7_pointer s7_gc_unprotect_via_stack(s7_scheme *sc, s7_pointer x)
{
  unstack_gc_protect(sc); /* this might not be related to 'x' -- something got unprotected */
  return(x);
}

#if s7_Debugging
  static s7_pointer stack_protected1_1(s7_scheme *sc, opcode_t op, const char *func, int line)
  {
    if (stack_top_op(sc) != op)
      {fprintf(stderr, "%s[%d]: stack_protected1 %s\n", func, line, op_names[stack_top_op(sc)]); abort();}
    return(stack_top_args(sc));
  }

  static s7_pointer stack_protected2_1(s7_scheme *sc, opcode_t op, const char *func, int line)
  {
    if (stack_top_op(sc) != op)
      {fprintf(stderr, "%s[%d]: stack_protected2 %s\n", func, line, op_names[stack_top_op(sc)]); abort();}
    return(stack_top_code(sc));
  }

  static s7_pointer stack_protected3_1(s7_scheme *sc, opcode_t op, const char *func, int line)
  {
    if (stack_top_op(sc) != op)
      {fprintf(stderr, "%s[%d]: stack_protected3 %s\n", func, line, op_names[stack_top_op(sc)]); abort();}
    return(stack_top_let(sc));
  }

  #define stack_protected1(Sc, Op) stack_protected1_1(Sc, Op, __func__, __LINE__)
  #define stack_protected2(Sc, Op) stack_protected2_1(Sc, Op, __func__, __LINE__)
  #define stack_protected3(Sc, Op) stack_protected3_1(Sc, Op, __func__, __LINE__)

  #define set_stack_protected1(Sc, Val, Op) do {if (stack_top_op(Sc) != Op) fprintf(stderr, "%s[%d]: set_stack_protected1 %s\n", __func__, __LINE__, op_names[stack_top_op(Sc)]); stack_top_args(Sc) = Val;} while (0)
  #define set_stack_protected2(Sc, Val, Op) do {if (stack_top_op(Sc) != Op) fprintf(stderr, "%s[%d]: set_stack_protected2 %s\n", __func__, __LINE__, op_names[stack_top_op(Sc)]); stack_top_code(Sc) = Val;} while (0)
  #define set_stack_protected3(Sc, Val, Op) do {if (stack_top_op(Sc) != Op) fprintf(stderr, "%s[%d]: set_stack_protected3 %s\n", __func__, __LINE__, op_names[stack_top_op(Sc)]); stack_top_let(Sc) = Val;} while (0)

#else

  #define stack_protected1(Sc, Op) stack_top_args(Sc)
  #define stack_protected2(Sc, Op) stack_top_code(Sc)
  #define stack_protected3(Sc, Op) stack_top_let(Sc)

  #define set_stack_protected1(Sc, Val, Op) stack_top_args(Sc) = Val
  #define set_stack_protected2(Sc, Val, Op) stack_top_code(Sc) = Val
  #define set_stack_protected3(Sc, Val, Op) stack_top_let(Sc) = Val
#endif

#define gc_protected1(Sc) stack_protected1(Sc, OP_GC_PROTECT)
#define gc_protected2(Sc) stack_protected2(Sc, OP_GC_PROTECT)
#define gc_protected3(Sc) stack_protected3(Sc, OP_GC_PROTECT)

#define set_gc_protected1(Sc, Val) set_stack_protected1(Sc, Val, OP_GC_PROTECT)
#define set_gc_protected2(Sc, Val) set_stack_protected2(Sc, Val, OP_GC_PROTECT)
#define set_gc_protected3(Sc, Val) set_stack_protected3(Sc, Val, OP_GC_PROTECT)

#define map_unwind_list(Sc)          stack_protected3(Sc, OP_MAP_UNWIND)
#define set_map_unwind_list(Sc, Val) set_stack_protected3(Sc, Val, OP_MAP_UNWIND)

#define gc_protect_curlet_via_stack(Sc)      push_stack_op_let(Sc, OP_GC_PROTECT)
#define gc_protected_let_via_stack(Sc)       stack_top_let(Sc)
#define gc_protect_via_stack(Sc, Obj)        push_stack_no_code(Sc, OP_GC_PROTECT, Obj)
#define gc_protect_via_stack_no_let(Sc, Obj) push_stack_no_let_no_code(Sc, OP_GC_PROTECT, Obj)
#define gc_protect_2_via_stack(Sc, X, Y)     do {gc_protect_via_stack(Sc, X); set_gc_protected2(Sc, Y);} while (0)
  /* often X and Y are fx_calls, so push X, then set Y */
#define gc_protect_2_via_stack_no_let(Sc, X, Y) do {push_stack_no_let_no_code(Sc, OP_GC_PROTECT, X); set_gc_protected2(Sc, Y);} while (0)


/* -------------------------------- symbols -------------------------------- */
static inline s7_uint raw_string_hash(const uint8_t *key, s7_int len) /* used in symbols, hash-tables */
{
  if (len <= 8)
    {
      s7_uint xs[1] = {0};
      memcpy((void *)xs, (const void *)key, len);
      return(xs[0]);
    }
  if (len <= 16)
    {
      s7_uint xs[2] = {0, 0};
      memcpy((void *)xs, (const void *)key, len);
      return(xs[0] + xs[1]);
    }
  {
    s7_uint xs[3] = {0, 0, 0};
    memcpy((void *)xs, (const void *)key, (len > 24) ? 24 : len);
    return(xs[0] + xs[1] + xs[2]);
  }
}

static uint8_t *alloc_symbol(s7_scheme *sc)
{
  #define Symbol_Size (3 * sizeof(s7_cell) + sizeof(block_t))
  #define Alloc_Symbol_Size (64 * Symbol_Size)
  uint8_t *result;
  if (sc->alloc_symbol_k == Alloc_Symbol_Size)
    {
      sc->alloc_symbol_cells = (uint8_t *)Malloc(Alloc_Symbol_Size);
      add_saved_pointer(sc, sc->alloc_symbol_cells);
      sc->alloc_symbol_k = 0;
    }
  result = &(sc->alloc_symbol_cells[sc->alloc_symbol_k]);
  sc->alloc_symbol_k += Symbol_Size;
  return(result);
}

static s7_pointer make_semipermanent_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot = alloc_pointer(sc);
  set_full_type(slot, T_SLOT | T_Unheap);
  slot_set_symbol_and_value(slot, symbol, value);
  return(slot);
}

static /* inline */ s7_pointer new_symbol(s7_scheme *sc, const char *name, s7_int len, s7_uint hash, uint32_t location) /* inline useless here 20-Oct-22 */
{
  /* name might not be null-terminated, these are semipermanent symbols even in s7_gensym; g_gensym handles everything separately */
  uint8_t *base = alloc_symbol(sc);
  const s7_pointer new_sym = (s7_pointer)base;
  const s7_pointer str = (s7_pointer)(base + sizeof(s7_cell));
  const s7_pointer p = (s7_pointer)(base + 2 * sizeof(s7_cell));
  uint8_t *val = (uint8_t *)permalloc(sc, len + 1);
  memcpy((void *)val, (const void *)name, len);
  val[len] = '\0';

  full_type(str) = T_STRING | T_Immutable | T_Unheap; /* avoid debugging confusion involving set_type (also below) */
  set_alloc_info(str, __func__, __LINE__);
  string_length(str) = len;
  string_value(str) = (char *)val;
  string_hash(str) = hash;

  full_type(new_sym) = T_SYMBOL | T_Unheap;
  set_alloc_info(new_sym, __func__, __LINE__);
  symbol_set_name_cell(new_sym, str);
  set_global_slot(new_sym, sc->undefined);  /* undefined_slot? */
  symbol_info(new_sym) = (block_t *)(base + 3 * sizeof(s7_cell));
  set_initial_value(new_sym, sc->undefined);
  symbol_set_local_slot_unchecked_and_unincremented(new_sym, 0LL, sc->undefined);
  set_big_symbol_tag(new_sym, 0);
  set_small_symbol_tag(new_sym, 0);
  symbol_clear_shadows(new_sym);
  symbol_clear_ctr(new_sym); /* alloc_symbol uses malloc */
  symbol_clear_type(new_sym);

  if ((len > 1) &&                                    /* not 0, otherwise : is a keyword */
      ((name[0] == ':') || (name[len - 1] == ':')))   /* see s7test under keyword? for troubles if both colons are present */
    {
      s7_pointer slot, ksym;
      set_type_bit(new_sym, T_Immutable | T_Keyword);
      set_optimize_op(str, OP_CONSTANT);
      ksym = make_symbol(sc, (name[0] == ':') ? (const char *)(name + 1) : name, len - 1);
      keyword_set_symbol(new_sym, ksym);
      set_has_keyword(ksym);
      /* the keyword symbol needs to be semipermanent (not a gensym) else we have to laboriously gc-protect it */
      if ((is_gensym(ksym)) &&
	  (in_heap(ksym)))
	remove_gensym_from_heap(sc, ksym);
      slot = make_semipermanent_slot(sc, new_sym, new_sym);
      set_global_slot(new_sym, slot);
      set_local_slot(new_sym, slot);
      set_immutable_slot(slot);
      /* we need to include this keyword in the symbol-table */
    }
  full_type(p) = T_PAIR | T_Immutable | T_Unheap;  /* add x to the symbol table */
  set_car(p, new_sym);
  set_cdr_unchecked(p, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = p;
  pair_set_raw_hash(p, hash);
  pair_set_raw_len(p, (s7_uint)len); /* symbol name length, so it ought to fit! */
  pair_set_raw_name(p, string_value(str));

#if 0
  { /* high 6 bytes are probably unused, and index */
    uint32_t high_bits = block_size(symbol_info(new_sym)) >> 32;
    int32_t index = block_index(symbol_info(new_sym));
    if (high_bits != 0) fprintf(stderr, "%x ", high_bits);
    /* for use of index, need this above:
     * block_set_index(symbol_info(new_sym), location); symbol_location(new_sym), if gensym clear before free, or use only if not gensym
     *    or use sz bits instead -- only useful in remove_gensym_from_symbol_table.
     * if (index != location) fprintf(stderr, "index: %d ", index);
     */
  }
#endif
  return(new_sym);
}

static Inline s7_pointer inline_make_symbol(s7_scheme *sc, const char *name, s7_int len) /* inline out: ca 40=2% in tload */
{ /* name here might not be null-terminated or aligned */
  const s7_uint hash = raw_string_hash((const uint8_t *)name, len);
  const uint32_t location = hash % Symbol_Table_Size;

  if (len <= 8)
    {
      for (s7_pointer syms = vector_element(sc->symbol_table, location); is_pair(syms); syms = cdr(syms))
	if ((hash == pair_raw_hash(syms)) &&
	    ((s7_uint)len == pair_raw_len(syms)))
	  return(car(syms));
    }
  else /* checking name[len=='\0' and using strcmp if so was not a big win */
    for (s7_pointer syms = vector_element(sc->symbol_table, location); is_pair(syms); syms = cdr(syms))
      if ((hash == pair_raw_hash(syms)) &&
	  ((s7_uint)len == pair_raw_len(syms)) &&
	  (strings_are_equal_with_length(name, pair_raw_name(syms), len))) /* length here because name might not be null-terminated or aligned */
	return(car(syms));
  return(new_symbol(sc, name, len, hash, location));
}

static s7_pointer make_symbol(s7_scheme *sc, const char *name, s7_int len) {return(inline_make_symbol(sc, name, len));}

s7_pointer s7_make_symbol(s7_scheme *sc, const char *name) {return(inline_make_symbol(sc, name, safe_strlen(name)));}

static s7_pointer symbol_table_find_by_name(s7_scheme *sc, const char *name, s7_uint hash, uint32_t location, s7_int len)
{
  for (s7_pointer syms = vector_element(sc->symbol_table, location); is_pair(syms); syms = cdr(syms))
    if ((hash == pair_raw_hash(syms)) &&
	(strings_are_equal_with_length(name, pair_raw_name(syms), len)))
      return(car(syms));
  return(sc->nil);
}

s7_pointer s7_symbol_table_find_name(s7_scheme *sc, const char *name)
{
  s7_int len = safe_strlen(name);
  s7_uint hash = raw_string_hash((const uint8_t *)name, len);
  s7_pointer result = symbol_table_find_by_name(sc, name, hash, hash % Symbol_Table_Size, len);
  return((is_null(result)) ? NULL : result);
}


/* -------------------------------- symbol-table -------------------------------- */
static inline s7_pointer make_simple_vector(s7_scheme *sc, s7_int len);

static s7_pointer g_symbol_table(s7_scheme *sc, s7_pointer unused_args)
{
  #define H_symbol_table "(symbol-table) returns a vector containing the current contents (symbols) of s7's symbol-table"
  #define Q_symbol_table s7_make_signature(sc, 1, sc->is_vector_symbol)

  int32_t syms = 0;
  s7_pointer *entries = vector_elements(sc->symbol_table);

  /* this can't be optimized by returning the actual symbol-table (a vector of lists), because
   *    gensyms can cause the table's lists and symbols to change at any time.  This wreaks havoc
   *    on traversals like for-each.  So, symbol-table returns a snap-shot of the table contents
   *    at the time it is called.
   * can be called in gdb: p display(s7_eval_c_string(sc, "(for-each (lambda (x) (when (gensym? x) (format *stderr* \"~A \" x))) (symbol-table))"))
   */
  for (int32_t i = 0; i < Symbol_Table_Size; i++)
    for (s7_pointer symlist = entries[i]; is_pair(symlist); symlist = cdr(symlist))
      syms++;
  if (syms > sc->max_vector_length)
    error_nr(sc, sc->out_of_range_symbol,
	     set_elist_3(sc, wrap_string(sc, "symbol-table size, ~D, is greater than (*s7* 'max-vector-length), ~D", 68),
			 wrap_integer(sc, syms), wrap_integer(sc, sc->max_vector_length)));
  {
    const s7_pointer vec = make_simple_vector(sc, syms);
    s7_pointer *els = vector_elements(vec);
    set_is_symbol_table(vec);
    for (int32_t i = 0, j = 0; i < Symbol_Table_Size; i++)
      for (s7_pointer symlist = entries[i]; is_pair(symlist); symlist = cdr(symlist))
	els[j++] = car(symlist);
    return(vec);
  }
}

bool s7_for_each_symbol_name(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  /* this includes the special constants #<unspecified> and so on for simplicity -- are there any others? */
  for (int32_t i = 0; i < Symbol_Table_Size; i++)
    for (s7_pointer syms = vector_element(sc->symbol_table, i); is_pair(syms); syms = cdr(syms))
      if (symbol_func(symbol_name(car(syms)), data))
	return(true);
  return((symbol_func("#t", data))             || (symbol_func("#f", data))             ||
	 (symbol_func("#<unspecified>", data)) || (symbol_func("#<undefined>", data))   ||
	 (symbol_func("#<eof>", data))         ||
	 (symbol_func("#true", data))          || (symbol_func("#false", data)));
}

bool s7_for_each_symbol(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  for (int32_t i = 0; i < Symbol_Table_Size; i++)
    for (s7_pointer syms = vector_element(sc->symbol_table, i); is_pair(syms); syms = cdr(syms))
      if (symbol_func(symbol_name(car(syms)), data))
	return(true);
  return(false);
}


/* -------------------------------- gensym -------------------------------- */
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym)
{
  /* sym is a free cell at this point (we're called after the GC), but the name_cell is still intact */
  const uint32_t location = string_hash(symbol_name_cell(sym)) % Symbol_Table_Size;
  s7_pointer symbols = vector_element(sc->symbol_table, location);
  if (car(symbols) == sym)
    vector_element(sc->symbol_table, location) = cdr(symbols);
  else
    for (s7_pointer syms = cdr(symbols); is_pair(syms); symbols = syms, syms = cdr(syms))
      if (car(syms) == sym)
	{
	  set_cdr_unchecked(symbols, cdr(syms)); /* delete z */
	  return;
	}
}

s7_pointer s7_gensym(s7_scheme *sc, const char *prefix)
{
  const s7_int len = safe_strlen(prefix) + 32;
  block_t *b = mallocate(sc, len);
  char *name = (char *)block_data(b);
  /* there's no point in heroic efforts here to avoid name collisions -- the user can screw up no matter what we do */
  name[0] = '\0';
  {
    s7_int slen = catstrs(name, len, "{", (prefix) ? prefix : "", "}-", pos_int_to_str_direct(sc, sc->gensym_counter++), (char *)NULL);
    s7_uint hash = raw_string_hash((const uint8_t *)name, slen);
    int32_t location = hash % Symbol_Table_Size;
    s7_pointer x = new_symbol(sc, name, slen, hash, location);  /* not T_Gensym -- might be called from outside so should not be GC'd(?) */
    liberate(sc, b);
    return(x);
  }
}

static bool is_gensym_b_p(s7_pointer g) {return((is_symbol(g)) && (is_gensym(g)));}

static s7_pointer g_is_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_is_gensym "(gensym? sym) returns #t if sym is a gensym"
  #define Q_is_gensym sc->pl_bt
  check_boolean_method(sc, is_gensym_b_p, sc->is_gensym_symbol, args);
}

static s7_pointer g_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_gensym "(gensym (prefix \"gensym\")) returns a new, unused symbol"
  #define Q_gensym s7_make_signature(sc, 2, sc->is_gensym_symbol, sc->is_string_symbol)

  const char *prefix;
  s7_int plen;

  /* get symbol name */
  if (is_pair(args))
    {
      s7_pointer gname = car(args);
      if (!is_string(gname))
	return(sole_arg_method_or_bust(sc, gname, sc->gensym_symbol, args, sc->type_names[T_STRING]));
      prefix = string_value(gname);
      plen = string_length(gname); /* was safe_strlen(prefix): were we stopping at #\null deliberately? */
    }
  else
    {
      prefix = "gensym";
      plen = 6;
    }

  {
    s7_int len = plen + 32; /* why 32 -- we need room for the gensym_counter integer, but (length "9223372036854775807") = 19, see gensym name collision loop below */
    /* it might be better (less predictable) to use a random number instead of gensym_counter, but that looks messy */
    block_t *b = mallocate(sc, len + sizeof(block_t) + 2 * sizeof(s7_cell));
    char *base = (char *)block_data(b);
    s7_pointer str = (s7_cell *)base;
    s7_pointer stc = (s7_cell *)(base + sizeof(s7_cell));
    block_t *ib = (block_t *)(base + 2 * sizeof(s7_cell));
    char *name = (char *)(base + sizeof(block_t) + 2 * sizeof(s7_cell));
    name[0] = '{';
    memcpy((void *)(name + 1), prefix, plen); /* memcpy is ok with plen==0, I think */
    name[plen + 1] = '}';
    name[plen + 2] = '-'; /* {gensym}-nnn */

    {
      s7_pointer new_gensym;
      s7_uint hash;
      uint32_t location;
      s7_int nlen;
      while (true)
	{
	  const char *p = pos_int_to_str(sc, sc->gensym_counter++, &len, '\0');
	  memcpy((void *)(name + plen + 3), (void *)p, len);
	  nlen = len + plen + 2;
	  name[nlen] = '\0';
	  hash = raw_string_hash((const uint8_t *)name, nlen);
	  location = hash % Symbol_Table_Size;
	  if (is_null(symbol_table_find_by_name(sc, name, hash, location, nlen))) break;
	  if (sc->safety > No_Safety)
	    s7_warn(sc, nlen + 25, "%s collides with gensym?\n", name);
	}

      /* make-string for symbol name */
      if (s7_Debugging) full_type(str) = 0; /* here and below, this is needed to avoid set_type check errors (mallocate above) */
      set_full_type(str, T_STRING | T_Immutable); /* was T_Unheap? 17-Mar-25 */
      string_length(str) = nlen;
      string_value(str) = name;
      string_hash(str) = hash;

      /* allocate the symbol in the heap so GC'd when inaccessible */
      new_cell(sc, new_gensym, T_SYMBOL | T_Gensym);
      symbol_set_name_cell(new_gensym, str);
      symbol_info(new_gensym) = ib;
      set_global_slot(new_gensym, sc->undefined);   /* undefined_slot? */
      set_initial_value(new_gensym, sc->undefined);
      symbol_set_local_slot_unchecked(new_gensym, 0LL, sc->undefined);
      symbol_clear_ctr(new_gensym);
      set_big_symbol_tag(new_gensym, 0);
      set_small_symbol_tag(new_gensym, 0);
      symbol_clear_shadows(new_gensym);
      symbol_clear_type(new_gensym);
      gensym_block(new_gensym) = b;

      /* place new symbol in symbol-table */
      if (s7_Debugging) full_type(stc) = 0;
      set_full_type(stc, T_PAIR | T_Immutable); /* was T_Unheap? 17-Mar-25 */
      set_car(stc, new_gensym);
      set_cdr_unchecked(stc, vector_element(sc->symbol_table, location));
      vector_element(sc->symbol_table, location) = stc;
      pair_set_raw_hash(stc, hash);
      pair_set_raw_len(stc, (s7_uint)string_length(str));
      pair_set_raw_name(stc, string_value(str));

      add_gensym(sc, new_gensym);
      return(new_gensym);
    }}
}


/* -------------------------------- syntax? -------------------------------- */
bool s7_is_syntax(s7_pointer p) {return(is_syntax(p));}

static s7_pointer g_is_syntax(s7_scheme *sc, s7_pointer args)
{
  #define H_is_syntax "(syntax? obj) returns #t if obj is a syntactic value (e.g. lambda)"
  #define Q_is_syntax sc->pl_bt
  check_boolean_method(sc, is_syntax, sc->is_syntax_symbol, args);
}


/* -------------------------------- symbol? -------------------------------- */
bool s7_is_symbol(s7_pointer p) {return(is_symbol(p));}

static s7_pointer g_is_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_is_symbol "(symbol? obj) returns #t if obj is a symbol"
  #define Q_is_symbol sc->pl_bt
  check_boolean_method(sc, is_symbol, sc->is_symbol_symbol, args);
}

const char *s7_symbol_name(s7_pointer sym) {return(symbol_name(sym));}

s7_pointer s7_name_to_value(s7_scheme *sc, const char *name) {return(s7_symbol_value(sc, make_symbol_with_strlen(sc, name)));}
/* should this also handle non-symbols such as "+nan.0"? */


/* -------------------------------- symbol->string -------------------------------- */
static Inline s7_pointer inline_make_string_with_length(s7_scheme *sc, const char *str, s7_int len)
{
  s7_pointer new_string;
  new_cell(sc, new_string, T_STRING | T_Safe_Procedure);
  string_block(new_string) = inline_mallocate(sc, len + 1);
  string_value(new_string) = (char *)block_data(string_block(new_string));
  if (str) memcpy((void *)string_value(new_string), (const void *)str, len);
  string_value(new_string)[len] = 0;
  string_length(new_string) = len;
  string_hash(new_string) = 0;
  add_string(sc, new_string);
  return(new_string);
}

static s7_pointer make_string_with_length(s7_scheme *sc, const char *str, s7_int len)
{
  return(inline_make_string_with_length(sc, str, len)); /* packaged to avoid inlining everywhere */
}

static s7_pointer g_symbol_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_string "(symbol->string sym) returns the symbol sym converted to a string"
  #define Q_symbol_to_string s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_symbol_symbol)

  const s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(sole_arg_method_or_bust(sc, sym, sc->symbol_to_string_symbol, args, sc->type_names[T_SYMBOL]));
  /* s7_make_string uses strlen which stops at an embedded null */
  if (symbol_name_length(sym) > sc->max_string_length)
    error_nr(sc, sc->out_of_range_symbol,
	     set_elist_3(sc, wrap_string(sc, "symbol->string symbol name is too large: (> ~D ~D) (*s7* 'max-string-length)", 76),
			 wrap_integer(sc, symbol_name_length(sym)), wrap_integer(sc, sc->max_string_length)));
  return(inline_make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy */
}

static s7_pointer g_symbol_to_string_uncopied(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(sole_arg_method_or_bust(sc, sym, sc->symbol_to_string_symbol, args, sc->type_names[T_SYMBOL]));
  if (is_gensym(sym))
    return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy of gensym name (which will be freed) */
  return(symbol_name_cell(sym));
}

static s7_pointer symbol_to_string_p_p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    return(sole_arg_method_or_bust(sc, sym, sc->symbol_to_string_symbol, set_plist_1(sc, sym), sc->type_names[T_SYMBOL]));
  if (symbol_name_length(sym) > sc->max_string_length)
    error_nr(sc, sc->out_of_range_symbol,
	     set_elist_3(sc, wrap_string(sc, "symbol->string symbol name is too large: (> ~D ~D) (*s7* 'max-string-length)", 76),
			 wrap_integer(sc, symbol_name_length(sym)), wrap_integer(sc, sc->max_string_length)));
  return(inline_make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));
}

static s7_pointer symbol_to_string_uncopied_p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    return(sole_arg_method_or_bust(sc, sym, sc->symbol_to_string_symbol, set_plist_1(sc, sym), sc->type_names[T_SYMBOL]));
  if (is_gensym(sym))
    return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));
  return(symbol_name_cell(sym));
}


/* -------------------------------- string->symbol -------------------------------- */
static inline s7_pointer string_to_symbol_1(s7_scheme *sc, s7_pointer str, s7_pointer caller)
{
  if (!is_string(str))
    return(method_or_bust_p(sc, str, caller, sc->type_names[T_STRING]));
  if (string_length(str) <= 0)
    sole_arg_wrong_type_error_nr(sc, caller, str, wrap_string(sc, "a non-null string", 17));
  return(make_symbol(sc, string_value(str), string_length(str)));
}

static s7_pointer g_string_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_symbol "(string->symbol str) returns the string str converted to a symbol"
  #define Q_string_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_string_symbol)
  return(string_to_symbol_1(sc, car(args), sc->string_to_symbol_symbol));
}

static s7_pointer string_to_symbol_p_p(s7_scheme *sc, s7_pointer p) {return(string_to_symbol_1(sc, p, sc->string_to_symbol_symbol));}


/* -------------------------------- symbol -------------------------------- */
static s7_pointer string_append_1(s7_scheme *sc, s7_pointer args, s7_pointer caller);

static s7_pointer mark_as_symbol_from_symbol(s7_pointer sym)
{
  set_is_symbol_from_symbol(sym);
  return(sym);
}

static s7_pointer g_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol "(symbol str ...) returns its string arguments concatenated and converted to a symbol"
  #define Q_symbol s7_make_circular_signature(sc, 1, 2, sc->is_symbol_symbol, sc->is_string_symbol)

  /* (let ((x 0)) (set! (symbol "x") 12)) ;symbol (a c-function) does not have a setter: (set! (symbol "x") 12)
   *   (let (((symbol "x") 3)) x) ; bad variable ((symbol "x")
   *   (let ((x 2)) (+ (symbol "x") 1)) ;+ first argument, x, is a symbol but should be a number
   *   maybe document this: (symbol...) just returns the symbol
   *   (let ((x 3)) (+ (symbol->value (symbol "x")) 1)) -> 4, (let ((x 0)) (apply set! (symbol "x") (list 32)) x) -> 32
   */
  s7_int len = 0;
  s7_pointer p;

  for (p = args; is_pair(p); p = cdr(p))
    if (is_string(car(p)))
      len += string_length(car(p));
    else break;
  if (is_pair(p))
    {
      if (is_null(cdr(args)))
	return(mark_as_symbol_from_symbol(string_to_symbol_1(sc, car(args), sc->symbol_symbol)));
      return(mark_as_symbol_from_symbol(string_to_symbol_1(sc, string_append_1(sc, args, sc->symbol_symbol), sc->symbol_symbol)));
    }
  if (len == 0)
    sole_arg_wrong_type_error_nr(sc, sc->symbol_symbol, car(args), wrap_string(sc, "a non-null string", 17));

  { /* can't use catstrs_direct here because it stops at embedded null */
    block_t *b = mallocate(sc, len + 1);
    char *name = (char *)block_data(b);
    s7_pointer sym;
    p = args;
    for (s7_int cur_len = 0; is_pair(p); p = cdr(p))
      {
	s7_pointer str = car(p);
	if (string_length(str) > 0)
	  {
	    memcpy((void *)(name + cur_len), (void *)string_value(str), string_length(str));
	    cur_len += string_length(str);
	  }}
    name[len] = '\0';
    sym = mark_as_symbol_from_symbol(inline_make_symbol(sc, name, len));
    liberate(sc, b);
    return(sym);
  }
}

static s7_pointer symbol_p_pp(s7_scheme *sc, s7_pointer str1, s7_pointer str2)
{
  #define Sym_Bufsize 256
  char buf[Sym_Bufsize];
  s7_int len;
  if ((!is_string(str1)) || (!is_string(str2))) return(g_symbol(sc, set_plist_2(sc, str1, str2)));
  len = string_length(str1) + string_length(str2);
  if ((len == 0) || (len >= Sym_Bufsize)) return(g_symbol(sc, set_plist_2(sc, str1, str2)));
  memcpy((void *)buf, (void *)string_value(str1), string_length(str1));
  memcpy((void *)(buf + string_length(str1)), (void *)string_value(str2), string_length(str2));
  return(mark_as_symbol_from_symbol(inline_make_symbol(sc, buf, len)));
}

/* -------- symbol-initial-value -------- */
static s7_pointer g_symbol_initial_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_initial_value "(symbol-initial-value sym) returns the initial binding of the symbol sym"
  #define Q_symbol_initial_value s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)

  s7_pointer symbol = car(args);
  if (!is_symbol(symbol)) /* or is_normal_symbol? now (symbol-initial-value :hi) -> #<undefined> */
    return(sole_arg_method_or_bust(sc, symbol, sc->symbol_initial_value_symbol, set_plist_1(sc, symbol), sc->type_names[T_SYMBOL]));
  return(initial_value(symbol));
}

static s7_pointer g_symbol_set_initial_value(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer symbol = car(args), value = cadr(args);
  if (!is_symbol(symbol))
    wrong_type_error_nr(sc, wrap_string(sc, "set! symbol-initial-value", 25), 1, symbol, sc->type_names[T_SYMBOL]);
  if (initial_value_is_defined(sc, symbol))
    immutable_object_error_nr(sc, set_elist_2(sc, wrap_string(sc, "can't set! (symbol-initial-value '~S); it is immutable", 54), symbol));
  set_initial_value(symbol, value);
  if (in_heap(value)) add_semipermanent_object(sc, value);
  /* should this tie into unlet? */
  return(value);
}

s7_pointer s7_symbol_initial_value(s7_pointer symbol) {return(initial_value(symbol));}

static bool is_eq_initial_value(s7_pointer symbol, s7_pointer other)
{
  const s7_pointer init = initial_value(symbol);
  if (init == other) return(true);
  if (is_c_function(init)) return((is_c_function(other)) && (c_function_data(init) == c_function_data(other)));
  if (is_syntax(init)) return((is_syntax(other)) && (syntax_symbol(init) == syntax_symbol(other)));
  if (is_unspecified(init)) return(is_unspecified(other));
  if (is_c_macro(init)) return((is_c_macro(other)) && (c_macro_data(init) == c_macro_data(other)));
  return(false);
}

static bool is_eq_initial_c_function_data(s7_pointer symbol, s7_pointer s_func)
{
  return((is_c_function(s_func)) && (c_function_data(s_func) == c_function_data(initial_value(symbol))));
}

static void copy_initial_value(s7_scheme *sc, s7_pointer sym)
{
  s7_pointer curval = initial_value(sym);
  if (!is_symbol(curval)) /* otherwise we end up with a copied symbol */
    {
      s7_pointer newval = alloc_pointer(sc);
      memcpy((void *)newval, (void *)curval, sizeof(s7_cell));
      set_is_initial_value(newval);
      set_initial_value(sym, newval);
    }
}

s7_pointer s7_symbol_set_initial_value(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  if (initial_value(symbol) == sc->undefined)
    {
      set_initial_value(symbol, value);
      /* copy_initial_value(sc, symbol); */
      if (in_heap(value)) add_semipermanent_object(sc, value);
    }
  return(initial_value(symbol));
}


/* -------- small symbol set -------- */

#if s7_Debugging
enum {Set_Ignore, Set_Begin, Set_End};

#define symbol_is_in_small_symbol_set(Sc, Sym) symbol_is_in_small_symbol_set_1(Sc, Sym, __func__, __LINE__)
static bool symbol_is_in_small_symbol_set_1(s7_scheme *sc, s7_pointer sym, const char *func, int line)
{
  if (sc->small_symbol_set_state == Set_End)
    fprintf(stderr, "%s[%d]: small_symbol_set membership test but it's not running\n", func, line);
  return(small_symbol_tag(sym) == sc->small_symbol_tag);
}

#define add_symbol_to_small_symbol_set(Sc, Sym) add_symbol_to_small_symbol_set_1(Sc, Sym, __func__, __LINE__)
static s7_pointer add_symbol_to_small_symbol_set_1(s7_scheme *sc, s7_pointer sym, const char *func, int line)
{
  if (sc->small_symbol_set_state == Set_End)
    fprintf(stderr, "%s[%d]: small_symbol_set add member but it's not running\n", func, line);
  set_small_symbol_tag(sym, sc->small_symbol_tag);
  return(sym);
}

#define clear_small_symbol_set(Sc) clear_small_symbol_set_1(Sc, Set_Ignore, __func__, __LINE__)
static void clear_small_symbol_set_1(s7_scheme *sc, int status, const char *func, int line)
{
  /* if running end is ok, begin is an error, if not running end is error, begin is ok */
  if (status == Set_Begin)
    {
      if (sc->small_symbol_set_state == Set_Begin)
	{
	  fprintf(stderr, "%s[%d]: small_symbol_set is running but begin requested (started at %s[%d])\n",
		  func, line, sc->small_symbol_set_func, sc->small_symbol_set_line);
	  abort();
	}
      sc->small_symbol_set_func = func;
      sc->small_symbol_set_line = line;
    }
  if ((status == Set_End) && (sc->small_symbol_set_state == Set_End))
    fprintf(stderr, "%s[%d]: small_symbol_set is not running but end requested (started at %s[%d])\n",
	    func, line, sc->small_symbol_set_func, sc->small_symbol_set_line);
  sc->small_symbol_set_state = status;

  if (sc->small_symbol_tag == 0) /* see comment below */
    {
      s7_pointer *els = vector_elements(sc->symbol_table);
      for (int32_t i = 0; i < Symbol_Table_Size; i++) /* clear old small_symbol_tags */
	for (s7_pointer syms = els[i]; is_pair(syms); syms = cdr(syms))
	  set_small_symbol_tag(car(syms), 0);
      sc->small_symbol_tag = 1;
    }
  else sc->small_symbol_tag++;
}

#define begin_small_symbol_set(Sc) clear_small_symbol_set_1(Sc, Set_Begin, __func__, __LINE__)
#define end_small_symbol_set(Sc)   clear_small_symbol_set_1(Sc, Set_End, __func__, __LINE__)

#else

#define symbol_is_in_small_symbol_set(Sc, Sym) (small_symbol_tag(Sym) == Sc->small_symbol_tag)

static /* inline */ s7_pointer add_symbol_to_small_symbol_set(s7_scheme *sc, s7_pointer sym)
{
  set_small_symbol_tag(sym, sc->small_symbol_tag);
  return(sym);
}

static /* inline */ void clear_small_symbol_set(s7_scheme *sc)
{
  if (sc->small_symbol_tag == 0) /* or 2^32 - 1, but that's much slower than checking for 0 -- unsigned wrap around is defined in C */
    {
      s7_pointer *els = vector_elements(sc->symbol_table);
      for (int32_t i = 0; i < Symbol_Table_Size; i++) /* clear old small_symbol_tags */
	for (s7_pointer syms = els[i]; is_pair(syms); syms = cdr(syms))
	  set_small_symbol_tag(car(syms), 0);
      sc->small_symbol_tag = 1;
    }
  else sc->small_symbol_tag++;
}

#define begin_small_symbol_set(Sc) clear_small_symbol_set(Sc)
#define end_small_symbol_set(Sc)
#endif

/* -------- big symbol set -------- */
#define symbol_is_in_big_symbol_set(Sc, Sym) (big_symbol_tag(Sym) == Sc->big_symbol_tag)
#define clear_big_symbol_set(Sc) Sc->big_symbol_tag++

static s7_pointer add_symbol_to_big_symbol_set(s7_scheme *sc, s7_pointer sym)
{
  if (symbol_is_in_big_symbol_set(sc, sym)) symbol_shadows(sym)++; else symbol_clear_shadows(sym);
  set_big_symbol_tag(sym, sc->big_symbol_tag);
  return(sym);
}


/* -------------------------------- lets/slots -------------------------------- */
static Inline s7_pointer inline_make_let(s7_scheme *sc, s7_pointer old_let)
{
  s7_pointer new_let;
  new_cell(sc, new_let, T_LET | T_Safe_Procedure);
  let_set_id(new_let, ++sc->let_number);
  let_set_slots(new_let, Slot_End);
  let_set_outlet(new_let, old_let);
  return(new_let);
}

static inline s7_pointer make_let(s7_scheme *sc, s7_pointer old_let) {return(inline_make_let(sc, old_let));}

static Inline s7_pointer inline_make_let_with_slot(s7_scheme *sc, s7_pointer old_let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer new_let, slot;
  sc->value = value;
  new_cell(sc, new_let, T_LET | T_Safe_Procedure);
  let_set_id(new_let, ++sc->let_number);
  let_set_outlet(new_let, old_let);
  new_cell_unchecked(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  symbol_set_local_slot(symbol, sc->let_number, slot);
  slot_set_next(slot, Slot_End);
  let_set_slots(new_let, slot);
  return(new_let);
}

static s7_pointer wrap_let_with_slot(s7_scheme *sc, s7_pointer old_let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer let = wrap_let(sc, old_let); /* increments let_number */
  s7_pointer slot = wrap_slot(sc, symbol, value);
  symbol_set_local_slot(symbol, sc->let_number, slot);
  slot_set_next(slot, Slot_End);
  let_set_slots(let, slot);
  return(let);
}

static s7_pointer make_let_with_slot(s7_scheme *sc, s7_pointer old_let, s7_pointer symbol, s7_pointer value)
{
  return(inline_make_let_with_slot(sc, old_let, symbol, value));
}

static Inline s7_pointer inline_make_let_with_two_slots(s7_scheme *sc, s7_pointer old_let,
							s7_pointer symbol1, s7_pointer value1, s7_pointer symbol2, s7_pointer value2)
{
  /* we leave value1/value2 computation order to the C compiler here -- in the old macro, it was explicitly value1 then value2
   *   this means any let in old scheme code that actually depends on the order may break -- it should be let*.
   */
  s7_pointer new_let, slot1, slot2;
  new_cell(sc, new_let, T_LET | T_Safe_Procedure);
  let_set_id(new_let, ++sc->let_number);
  let_set_outlet(new_let, old_let);

  new_cell_unchecked(sc, slot1, T_SLOT);
  slot_set_symbol_and_value(slot1, symbol1, value1);
  symbol_set_local_slot(symbol1, sc->let_number, slot1);
  let_set_slots(new_let, slot1);

  new_cell_unchecked(sc, slot2, T_SLOT);
  slot_set_symbol_and_value(slot2, symbol2, value2);
  symbol_set_local_slot(symbol2, sc->let_number, slot2);
  slot_set_next(slot2, Slot_End);
  slot_set_next(slot1, slot2);
  return(new_let);
}

static s7_pointer make_let_with_two_slots(s7_scheme *sc, s7_pointer old_let, s7_pointer symbol1, s7_pointer value1, s7_pointer symbol2, s7_pointer value2)
{
  return(inline_make_let_with_two_slots(sc, old_let, symbol1, value1, symbol2, value2));
}

/* in all these functions, symbol_set_local_slot should follow slot_set_value so that we can evaluate the slot's value in its old state */
static inline void add_slot_unchecked(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value, s7_uint id)
{
  s7_pointer slot;
  new_cell_unchecked(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  set_local(symbol);
  symbol_set_local_slot(symbol, id, slot);
}

static s7_pointer add_slot_unchecked_no_local_slot(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_unchecked(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  set_local(symbol);
  return(slot);
}

#define add_slot(Sc, Let, Symbol, Value) add_slot_unchecked(Sc, Let, Symbol, Value, let_id(Let))

static inline s7_pointer add_slot_checked(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  symbol_set_local_slot(symbol, let_id(let), slot);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static inline s7_pointer add_slot_checked_with_id(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  set_local(symbol);
  if (let_id(let) >= symbol_id(symbol))
    symbol_set_local_slot(symbol, let_id(let), slot);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static inline s7_pointer add_slot_no_local(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value) /* no symbol_set_local_slot, no set_local */
{
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static s7_pointer add_slot_unchecked_with_id(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_unchecked(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  set_local(symbol);
  if (let_id(let) >= symbol_id(symbol))
    symbol_set_local_slot(symbol, let_id(let), slot);
  slot_set_next(slot, let_slots(let));
  let_set_slots(let, slot);
  return(slot);
}

static inline s7_pointer add_slot_at_end(s7_scheme *sc, s7_uint id, s7_pointer last_slot, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_unchecked(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, Slot_End);
  symbol_set_local_slot(symbol, id, slot);
  slot_set_next(last_slot, slot);
  return(slot);
}

static s7_pointer add_slot_checked_at_end(s7_scheme *sc, s7_uint id, s7_pointer last_slot, s7_pointer symbol, s7_pointer value)
{ /* same as above but new_cell is checked */
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, Slot_End);
  symbol_set_local_slot(symbol, id, slot);
  slot_set_next(last_slot, slot);
  return(slot);
}

static s7_pointer add_slot_at_end_no_local(s7_scheme *sc, s7_pointer last_slot, s7_pointer symbol, s7_pointer value)
{
  s7_pointer slot;
  new_cell_unchecked(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, symbol, value);
  slot_set_next(slot, Slot_End);
  slot_set_next(last_slot, slot);
  return(slot);
}

static inline void make_let_with_three_slots(s7_scheme *sc, s7_pointer func, s7_pointer val1, s7_pointer val2, s7_pointer val3)
{
  s7_pointer last_slot;
  const s7_pointer pars = closure_pars(func);
  set_curlet(sc, inline_make_let_with_two_slots(sc, closure_let(func), car(pars), val1, cadr(pars), val2));
  last_slot = next_slot(let_slots(sc->curlet));
  add_slot_at_end(sc, let_id(sc->curlet), last_slot, caddr(pars), val3);
}

static inline void make_let_with_four_slots(s7_scheme *sc, s7_pointer func, s7_pointer val1, s7_pointer val2, s7_pointer val3, s7_pointer val4)
{
  s7_pointer last_slot, pars = closure_pars(func);
  set_curlet(sc, inline_make_let_with_two_slots(sc, closure_let(func), car(pars), val1, cadr(pars), val2));
  pars = cddr(pars);
  last_slot = next_slot(let_slots(sc->curlet));
  last_slot = add_slot_at_end(sc, let_id(sc->curlet), last_slot, car(pars), val3);
  add_slot_at_end(sc, let_id(sc->curlet), last_slot, cadr(pars), val4);
}

static inline void make_let_with_five_slots(s7_scheme *sc, s7_pointer func, s7_pointer val1, s7_pointer val2, s7_pointer val3, s7_pointer val4, s7_pointer val5)
{
  s7_pointer last_slot, pars = closure_pars(func);
  set_curlet(sc, inline_make_let_with_two_slots(sc, closure_let(func), car(pars), val1, cadr(pars), val2));
  pars = cddr(pars);
  last_slot = next_slot(let_slots(sc->curlet));
  last_slot = add_slot_at_end(sc, let_id(sc->curlet), last_slot, car(pars), val3);
  pars = cdr(pars);
  last_slot = add_slot_at_end(sc, let_id(sc->curlet), last_slot, car(pars), val4);
  add_slot_at_end(sc, let_id(sc->curlet), last_slot, cadr(pars), val5);
}

#define update_slot(Slot, Val, Id) do {s7_pointer sym; slot_set_value(Slot, Val); sym = slot_symbol(Slot); symbol_set_local_slot_unincremented(sym, Id, Slot);} while (0)

static s7_pointer update_let_with_slot(s7_scheme *sc, s7_pointer let, s7_pointer val)
{
  s7_pointer slot = let_slots(let);
  s7_int id = ++sc->let_number;
  if ((s7_Debugging) && (slot == Slot_End)) fprintf(stderr, "%s[%d]: no slot!\n", __func__, __LINE__);
  let_set_id(let, id);
  update_slot(slot, val, id);
  return(let);
}

static s7_pointer update_let_with_two_slots(s7_scheme *sc, s7_pointer let, s7_pointer val1, s7_pointer val2)
{
  s7_pointer slot = let_slots(let);
  s7_int id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val1, id); slot = next_slot(slot);
  update_slot(slot, val2, id);
  return(let);
}

static s7_pointer update_let_with_three_slots(s7_scheme *sc, s7_pointer let, s7_pointer val1, s7_pointer val2, s7_pointer val3)
{
  s7_pointer slot = let_slots(let);
  const s7_int id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val1, id); slot = next_slot(slot);
  update_slot(slot, val2, id); slot = next_slot(slot);
  update_slot(slot, val3, id);
  return(let);
}

static s7_pointer update_let_with_four_slots(s7_scheme *sc, s7_pointer let, s7_pointer val1, s7_pointer val2, s7_pointer val3, s7_pointer val4)
{
  s7_pointer slot = let_slots(let);
  const s7_int id = ++sc->let_number;
  let_set_id(let, id);
  update_slot(slot, val1, id); slot = next_slot(slot);
  update_slot(slot, val2, id); slot = next_slot(slot);
  update_slot(slot, val3, id); slot = next_slot(slot);
  update_slot(slot, val4, id);
  return(let);
}

static s7_pointer make_semipermanent_let(s7_scheme *sc, s7_pointer vars)
{
  s7_pointer slot;
  const s7_pointer let = alloc_pointer(sc);
  set_full_type(let, T_LET | T_Safe_Procedure | T_Unheap);
  let_set_id(let, ++sc->let_number);
  let_set_outlet(let, sc->curlet);
  slot = make_semipermanent_slot(sc, caar(vars), sc->F);
  add_semipermanent_let_or_slot(sc, slot);
  symbol_set_local_slot(caar(vars), sc->let_number, slot);
  let_set_slots(let, slot);
  for (s7_pointer var = cdr(vars); is_pair(var); var = cdr(var))
    {
      s7_pointer last_slot = slot;
      slot = make_semipermanent_slot(sc, caar(var), sc->F);
      add_semipermanent_let_or_slot(sc, slot);
      symbol_set_local_slot(caar(var), sc->let_number, slot);
      slot_set_next(last_slot, slot);
    }
  slot_set_next(slot, Slot_End);
  add_semipermanent_let_or_slot(sc, let); /* need to mark outlet and maybe slot values */
  return(let);
}

static s7_pointer call_setter(s7_scheme *sc, s7_pointer slot, s7_pointer old_value);

static inline s7_pointer checked_slot_set_value(s7_scheme *sc, s7_pointer slot, s7_pointer value)
{
  if (slot_has_setter(slot))
    slot_set_value(slot, call_setter(sc, slot, value));
  else
    {
      if (is_immutable_slot(slot))
	immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, sc->let_set_symbol, slot_symbol(slot)));
      slot_set_value(slot, value);
    }
  return(slot_value(slot));
}

static s7_pointer let_fill(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer let = car(args);
  s7_pointer val;
  if (let == sc->rootlet)
    out_of_range_error_nr(sc, sc->fill_symbol, int_one, let, wrap_string(sc, "can't fill! rootlet", 19));
  if (let == sc->starlet)
    out_of_range_error_nr(sc, sc->fill_symbol, int_one, let, wrap_string(sc, "can't fill! *s7*", 16));
  /* (owlet) copies sc->owlet, so let can't be sc->owlet */
  if (is_funclet(let))
    out_of_range_error_nr(sc, sc->fill_symbol, int_one, let, wrap_string(sc, "can't fill! a funclet", 21));
  val = cadr(args);
  for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
    checked_slot_set_value(sc, slot, val);
  return(val);
}

static s7_int starlet_length(void);

static s7_int let_length(s7_scheme *sc, s7_pointer let)
{
  /* used by length, applicable_length, copy, and some length optimizations */
  if (let == sc->rootlet)
    {
      s7_int i = 0;
      for (s7_pointer slot = sc->rootlet_slots; is_not_slot_end(slot); i++, slot = next_slot(slot));
      return(i);
    }
  if (let == sc->starlet)
    return(starlet_length());
  if (has_active_methods(sc, let))
    {
      s7_pointer length_func = find_method(sc, let, sc->length_symbol);
      if (length_func != sc->undefined)
	{
	  s7_pointer num = s7_apply_function(sc, length_func, set_plist_1(sc, let));
	  return((s7_is_integer(num)) ? s7_integer(num) : -1); /* ?? */
	}}
  {
    s7_int i = 0;
    for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); i++, slot = next_slot(slot));
    return(i);
  }
}

static void slot_set_setter(s7_pointer slot, s7_pointer val)
{
  if ((type(val) == T_C_FUNCTION) &&
      (c_function_has_bool_setter(val)))
    slot_set_setter_1(slot, c_function_bool_setter(val));
  else slot_set_setter_1(slot, val);
}

static void slot_set_value_with_hook_1(s7_scheme *sc, s7_pointer slot, s7_pointer value)
{
  /* (set! (hook-functions *rootlet-redefinition-hook*) (list (lambda (hook) (format *stderr* "~A ~A~%" (hook 'name) (hook 'value))))) */
  s7_pointer symbol = slot_symbol(slot);
  if ((global_slot(symbol) == slot) &&
      (value != slot_value(slot)))
    s7_call(sc, sc->rootlet_redefinition_hook, set_plist_2(sc, symbol, value));
  slot_set_value(slot, value);
}

static void remove_function_from_heap(s7_scheme *sc, s7_pointer value); /* calls remove_let_from_heap */

static void remove_let_from_heap(s7_scheme *sc, s7_pointer let)
{
  for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
    {
      s7_pointer val = slot_value(slot);
      if ((has_closure_let(val)) &&
	  (in_heap(closure_pars(val))))
	remove_function_from_heap(sc, val);
    }
  let_set_removed(let);
}

static s7_pointer funclet_entry(s7_scheme *sc, s7_pointer x, s7_pointer sym)
{
  if ((has_closure_let(x)) && (is_let(closure_let(x))) && (closure_let(x) != sc->rootlet))
    {
      s7_pointer val = symbol_to_local_slot(sc, sym, closure_let(x));
      if ((!is_slot(val)) && (let_outlet(closure_let(x)) != sc->rootlet))
	val = symbol_to_local_slot(sc, sym, let_outlet(closure_let(x)));
      if (is_slot(val))
	return(slot_value(val));
    }
  return(NULL);
}

static void remove_function_from_heap(s7_scheme *sc, s7_pointer value)
{
  remove_from_heap(sc, closure_pars(value));
  remove_from_heap(sc, closure_body(value)); /* this is where the compute time goes */
  /* remove closure if it's local to current func (meaning (define f (let ...) (lambda ...)) removes the enclosing let) */

  { /* not sure this is worth the effort (finds 46 strings during s7test, checks 407 functions) */
    s7_pointer val = funclet_entry(sc, value, sc->local_documentation_symbol);
    if ((val) && (is_string(val)) && (in_heap(val))) petrify(sc, val);
  }
  {
    s7_pointer let = closure_let(value);
    if ((is_let(let)) && (!let_removed(let)) && (let != sc->shadow_rootlet))
      {
	let = let_outlet(let);
	if ((is_let(let)) && (!let_removed(let)) && (let != sc->shadow_rootlet))
	  {
	    remove_let_from_heap(sc, let);
	    let = let_outlet(let);
	    if ((is_let(let)) && (!let_removed(let)) && (let != sc->shadow_rootlet))
	      remove_let_from_heap(sc, let);
	  }}}
}

static void add_slot_to_rootlet(s7_scheme *sc, s7_pointer slot)
{
  set_in_rootlet(slot);
  slot_set_next(slot, sc->rootlet_slots);
  sc->rootlet_slots = slot;
}

static void add_to_unlet(s7_scheme *sc, s7_pointer symbol)
{
  unlet_entry_t *new_entry = (unlet_entry_t *)permalloc(sc, sizeof(unlet_entry_t));
  new_entry->symbol = symbol;
  new_entry->next = sc->unlet_entries;
  sc->unlet_entries = new_entry;
}

s7_pointer s7_make_slot(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if ((!is_let(let)) || (let == sc->rootlet))
    {
      s7_pointer slot;
      if (is_immutable(sc->rootlet))
	immutable_object_error_nr(sc, set_elist_2(sc, wrap_string(sc, "can't define '~S; rootlet is immutable", 38), symbol));
      if ((sc->safety <= No_Safety) &&
	  (has_closure_let(value)))
	remove_function_from_heap(sc, value); /* optimization of access pointers happens later so presumably this is safe */

      /* first look for existing slot -- this is not always checked before calling s7_make_slot */
      if (is_slot(global_slot(symbol)))
	{
	  slot = global_slot(symbol);
	  if (is_immutable_slot(slot))        /* 2-Oct-23: (immutable! 'abs) (set! abs 3) */
	    immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, sc->set_symbol, symbol));
	  symbol_increment_ctr(symbol);
	  slot_set_value_with_hook(slot, value);
	  return(slot);
	}

      slot = make_semipermanent_slot(sc, symbol, value);
      add_slot_to_rootlet(sc, slot);
      set_global_slot(symbol, slot);
      if (is_global(symbol))                  /* never defined locally (symbol_id tracks let_id) */
	{
	  if ((!is_gensym(symbol)) &&
	      (!initial_value_is_defined(sc, symbol)) &&
	      (!in_heap(value)) &&         /* else initial_value can be GC'd if symbol set! (initial != global, initial unprotected) */
	       ((!sc->string_signature) || /* from init_signatures -- maybe need a boolean for this */
	       (is_c_function(value))))    /* || (is_syntax(value)) -- we need 'else as a special case? */
	    /* the string_signature business means only the initial rootlet c_functions take part in unlet.  It would be neat if any
	     *   cload library's c_functions could be there as well, but then (unlet) needs to know which lets are in the chain.
	     *   The current shadow_rootlet could be saved in each initial_value, these could be marked in some way, then the chain
	     *   searched in (unlet) to get the currently active lets -- maybe too complex?  We could also provide a way to overrule
	     *   the string_signature check, but then symbol collisions would probably be resolved as the last loaded (which might not
	     *   be in the active chain).
	     * Also, the c_function check is overly paranoid -- all we need is that the value is semipermanent (T_Unheap?).
	     * But I don't see any interesting omissions.
	     */
	    {
	      set_initial_value(symbol, value);
	      if ((!sc->string_signature) && ((is_c_function(value)) || (is_syntax(value)))) /* syntax probably can't happen here (handled explicitly in syntax procedure) */
		add_to_unlet(sc, symbol);
	    }
	  set_local_slot(symbol, slot);
	}
      symbol_increment_ctr(symbol);
      if (is_gensym(symbol))
	remove_gensym_from_heap(sc, symbol);
      return(slot);
    }
  return(add_slot_checked_with_id(sc, let, symbol, value));
  /* there are about as many lets as local variables -- this strikes me as surprising, but it holds up across a lot of code */
}

static s7_pointer make_slot(s7_scheme *sc, s7_pointer variable, s7_pointer value)
{
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol_and_value(slot, variable, value);
  return(slot);
}


/* -------------------------------- let? -------------------------------- */
bool s7_is_let(s7_pointer let) {return(is_let(let));}

static s7_pointer g_is_let(s7_scheme *sc, s7_pointer args)
{
  #define H_is_let "(let? obj) returns #t if obj is a let."
  #define Q_is_let sc->pl_bt
  check_boolean_method(sc, is_let, sc->is_let_symbol, args);
}


/* -------------------------------- funclet? -------------------------------- */
static s7_pointer g_is_funclet(s7_scheme *sc, s7_pointer args)
{
  #define H_is_funclet "(funclet? obj) returns #t if obj is a funclet (a function's environment)."
  #define Q_is_funclet sc->pl_bt

  s7_pointer let = car(args);
  if (let == sc->rootlet) return(sc->F);
  if ((is_let(let)) && ((is_funclet(let)) || (is_maclet(let))))
    return(sc->T);
  if (!has_active_methods(sc, let))
    return(sc->F);
  return(apply_boolean_method(sc, let, sc->is_funclet_symbol));
}


/* -------------------------------- unlet -------------------------------- */
static s7_pointer g_unlet(s7_scheme *sc, s7_pointer unused_args)
{
  /* add sc->unlet bindings to the current environment */
  #define H_unlet "(unlet) returns a let that establishes the original bindings of all the predefined functions"
  #define Q_unlet s7_make_signature(sc, 1, sc->is_let_symbol)

  const s7_pointer result = make_let(sc, sc->curlet);
  begin_temp(sc->y, result);
  set_is_unlet(result);
  if (global_value(sc->else_symbol) != sc->else_symbol)
    add_slot_checked_with_id(sc, result, sc->else_symbol, initial_value(sc->else_symbol));
  for (unlet_entry_t *p = sc->unlet_entries; p; p = p->next)
    {
      s7_pointer sym = p->symbol;
      if ((!is_eq_initial_value(sym, global_value(sym))) ||  /* it has been changed globally */
	  ((!is_global(sym)) &&        /* it might be shadowed locally */
	   (s7_symbol_local_value(sc, sym, sc->curlet) != global_value(sym))))
	add_slot_checked_with_id(sc, result, sym, initial_value(sym));
    }
  end_temp(sc->y);
  return(result);
}


/* -------------------------------- openlet? -------------------------------- */
bool s7_is_openlet(s7_pointer let) {return(has_methods(let));}

static s7_pointer g_is_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_is_openlet "(openlet? obj) returns #t if 'obj' has methods."
  #define Q_is_openlet sc->pl_bt

  s7_pointer let = car(args);  /* if let is not a let, should this raise an error? -- no, easier to use this way in cond */
  if_method_exists_return_value(sc, let, sc->is_openlet_symbol, args);
  return(make_boolean(sc, has_methods(let)));
}


/* -------------------------------- openlet -------------------------------- */
s7_pointer s7_openlet(s7_scheme *sc, s7_pointer let)
{
  /* if e is not a let, the openlet bit is still set on it (c-pointer etc) */
  set_has_methods(let);
  return(let);
}

static s7_pointer g_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_openlet "(openlet e) tells the built-in functions that the let e might have an over-riding method. e is returned."
  #define Q_openlet s7_make_signature(sc, 2, has_let_signature(sc), has_let_signature(sc))

  const s7_pointer let = car(args);
  s7_pointer new_let, func;
  if (!is_let(let))
    {
      new_let = find_let(sc, let);
      if ((!is_let(new_let)) || (new_let == sc->rootlet))
	find_let_error_nr(sc, sc->openlet_symbol, let, new_let, 1, args);
    }
  else new_let = let;
  if ((new_let == sc->rootlet) || (new_let == sc->starlet))
    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "can't openlet ~S", 17), let));
  if (is_unlet(new_let)) /* protect against infinite loop: (let () (define + -) (with-let (unlet) (+ (openlet (unlet)) 2))) */
    error_nr(sc, sc->out_of_range_symbol, set_elist_1(sc, wrap_string(sc, "can't openlet unlet", 19)));
  if ((has_active_methods(sc, let)) &&
      ((func = find_method(sc, new_let, sc->openlet_symbol)) != sc->undefined))
    return(s7_apply_function(sc, func, args));
  set_has_methods(let);
  return(let); /* openlet and coverlet return their argument */
}

/* -------------------------------- coverlet -------------------------------- */
static s7_pointer g_coverlet(s7_scheme *sc, s7_pointer args)
{
  #define H_coverlet "(coverlet e) undoes an earlier openlet.  e is returned."
  #define Q_coverlet s7_make_signature(sc, 2, has_let_signature(sc), has_let_signature(sc))

  const s7_pointer let = car(args);
  s7_pointer new_let, func;
  if (!is_let(let))
    {
      new_let = find_let(sc, let);
      if ((!is_let(new_let))  || (new_let == sc->rootlet))
	find_let_error_nr(sc, sc->coverlet_symbol, let, new_let, 1, args);
    }
  else new_let = let;
  if ((new_let == sc->rootlet) || (new_let == sc->starlet))
    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "can't coverlet ~S", 17), let));
  if (is_unlet(new_let))
    error_nr(sc, sc->out_of_range_symbol, set_elist_1(sc, wrap_string(sc, "can't coverlet unlet", 20)));
  if ((has_active_methods(sc, let)) &&
      ((func = find_method(sc, new_let, sc->coverlet_symbol)) != sc->undefined))
    return(s7_apply_function(sc, func, args));
  clear_has_methods(let);
  return(let); /* mimic openlet in everything */
}


/* -------------------------------- varlet -------------------------------- */
static void check_let_fallback(s7_scheme *sc, const s7_pointer symbol, s7_pointer let)
{
  if (symbol == sc->let_ref_fallback_symbol)
    set_has_let_ref_fallback(let);
  else
    if (symbol == sc->let_set_fallback_symbol)
      set_has_let_set_fallback(let);
}

static void append_let(s7_scheme *sc, s7_pointer new_let, s7_pointer old_let)
{
  if (new_let == sc->rootlet)
    for (s7_pointer slot = let_slots(old_let); is_not_slot_end(slot); slot = next_slot(slot))
      {
	s7_pointer sym = slot_symbol(slot), val = slot_value(slot);
	if (is_slot(global_slot(sym)))
	  set_global_value(sym, val);
	else s7_make_slot(sc, sc->rootlet, sym, val);
      }
  else
    if (old_let == sc->starlet)
      {
	const s7_pointer iter = s7_make_iterator(sc, sc->starlet);
	const s7_int gc_loc = gc_protect_1(sc, iter);
	iterator_carrier(iter) = cons_unchecked(sc, sc->F, sc->F);
	set_has_carrier(iter); /* so carrier is GC protected by mark_iterator */
	while (true)
	  {
	    s7_pointer field = s7_iterate(sc, iter);
	    if (iterator_is_at_end(iter)) break;
	    add_slot_checked_with_id(sc, new_let, car(field), cdr(field));
	  }
	s7_gc_unprotect_at(sc, gc_loc);
      }
    else
      for (s7_pointer slot = let_slots(old_let); is_not_slot_end(slot); slot = next_slot(slot))
	add_slot_checked_with_id(sc, new_let, slot_symbol(slot), slot_value(slot)); /* not add_slot here because it might run off the free heap end */
}

s7_pointer s7_varlet(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(let))
    wrong_type_error_nr(sc, sc->varlet_symbol, 1, let, a_let_string);
  if (!is_symbol(symbol))
    wrong_type_error_nr(sc, sc->varlet_symbol, 2, symbol, a_symbol_string);
  if ((is_slot(global_slot(symbol))) &&
      (is_syntax(global_value(symbol))))
    wrong_type_error_nr(sc, sc->varlet_symbol, 2, symbol, wrap_string(sc, "a non-syntactic symbol", 22));

  if (let == sc->rootlet)
    {
      if (is_slot(global_slot(symbol)))
	set_global_value(symbol, value);
      else s7_make_slot(sc, sc->rootlet, symbol, value);
    }
  else
    {
      add_slot_checked_with_id(sc, let, symbol, value);
      check_let_fallback(sc, symbol, let);
    }
  return(value);
}

static int32_t position_of(const s7_pointer p, s7_pointer args)
{
  int32_t i;
  for (i = 1; p != args; i++, args = cdr(args));
  return(i);
}

static s7_pointer g_varlet(s7_scheme *sc, s7_pointer args)   /* varlet = with-let + define */
{
  #define H_varlet "(varlet target-let ...) adds its arguments (a let, a cons: (symbol . value), or two arguments, the symbol and its value) \
to the let target-let, and returns target-let.  (varlet (curlet) 'a 1) adds 'a to the current environment with the value 1."
  #define Q_varlet s7_make_circular_signature(sc, 2, 4, sc->is_let_symbol, has_let_signature(sc), \
                     s7_make_signature(sc, 3, sc->is_pair_symbol, sc->is_symbol_symbol, sc->is_let_symbol), sc->T)
  s7_pointer let = car(args);
  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if ((!is_let(new_let)) || (new_let == sc->rootlet))
	find_let_error_nr(sc, sc->varlet_symbol, let, new_let, 1, args);
      let = new_let;
    }
  if ((is_immutable_let(let)) || (let == sc->starlet))
    immutable_object_error_nr(sc, set_elist_3(sc, wrap_string(sc, "can't (varlet ~{~S~^ ~}), ~S is immutable", 41), args, let));

  for (s7_pointer arglist = cdr(args); is_pair(arglist); arglist = cdr(arglist))
    {
      s7_pointer sym, val;
      const s7_pointer arg = car(arglist);
      if (is_symbol(arg))
	{
	  sym = (is_keyword(arg)) ? keyword_symbol(arg) : arg;
	  if (!is_pair(cdr(arglist)))
	    error_nr(sc, sc->syntax_error_symbol, set_elist_3(sc, wrap_string(sc, "varlet: symbol ~S, but no value: ~S", 35), arg, args));
	  if (is_constant_symbol(sc, sym))
	    wrong_type_error_nr(sc, sc->varlet_symbol, position_of(arglist, args), sym, a_non_constant_symbol_string);
	  arglist = cdr(arglist);
	  val = car(arglist);
	}
      else
	if (is_let(arg))
	  {
	    if ((arg != sc->rootlet) && (let != sc->starlet))   /* (varlet (inlet 'a 1) (rootlet)) is trouble */
	      {
		append_let(sc, let, arg);
		if (has_let_set_fallback(arg)) set_has_let_set_fallback(let);
		if (has_let_ref_fallback(arg)) set_has_let_ref_fallback(let);
	      }
	    continue;
	  }
	else
	  if (is_pair(arg))
	    {
	      sym = car(arg);
	      if (!is_symbol(sym))
		wrong_type_error_nr(sc, sc->varlet_symbol, position_of(arglist, args), arg, a_symbol_string);
	      if (is_constant_symbol(sc, sym))
		wrong_type_error_nr(sc, sc->varlet_symbol, position_of(arglist, args), sym, a_non_constant_symbol_string);
	      val = cdr(arg);
	    }
	  else wrong_type_error_nr(sc, sc->varlet_symbol, position_of(arglist, args), arg, wrap_string(sc, "a symbol, let, or cons", 22));

      if (let == sc->rootlet)
	{
	  s7_pointer gslot = global_slot(sym);
	  if (is_slot(gslot))
	    {
	      if (is_immutable(gslot)) /* (immutable! 'abs) (varlet (rootlet) 'abs 1) */
		immutable_object_error_nr(sc, set_elist_5(sc, wrap_string(sc, "~S is immutable in (varlet ~S '~S ~S)", 37), sym, car(args), arg, val));
	      slot_set_value_with_hook(global_slot(sym), val);
	    }
	  else s7_make_slot(sc, sc->rootlet, sym, val);
	}
      else
	{
	  check_let_fallback(sc, sym, let);
	  add_slot_checked_with_id(sc, let, sym, val);
	  /* this used to check for sym already defined, and set its value, but that greatly slows down
	   *   the most common use (adding a slot), and makes it hard to shadow explicitly.  Don't use
	   *   varlet as a substitute for set!/let-set!.
	   */
	}}
  return(let);
}


/* -------------------------------- cutlet -------------------------------- */
static s7_pointer g_cutlet(s7_scheme *sc, s7_pointer args)
{
  #define H_cutlet "(cutlet e symbol ...) removes symbols from the let e."
  #define Q_cutlet s7_make_circular_signature(sc, 2, 3, sc->is_let_symbol, has_let_signature(sc), sc->is_symbol_symbol)

  s7_pointer let = car(args);
  s7_int the_un_id;
  if (let != sc->rootlet)
    {
      if_method_exists_return_value(sc, let, sc->cutlet_symbol, args);
      if (!is_let(let))
	{
	  s7_pointer new_let = find_let(sc, let);
	  if ((!is_let(new_let)) || (new_let == sc->rootlet))
	    find_let_error_nr(sc, sc->cutlet_symbol, let, new_let, 1, args);
	  let = new_let;
	}}
  if ((is_immutable_let(let)) || (let == sc->starlet))
    immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, sc->cutlet_symbol, let));

  /* besides removing the slot we have to make sure the symbol_id does not match, else
   *   let-ref and others will use the old slot!  So use the next (unused) id.
   *   (let ((b 1)) (let ((b 2)) (cutlet (curlet) 'b)) b)
   */
  the_un_id = ++sc->let_number;

  for (s7_pointer syms = cdr(args); is_pair(syms); syms = cdr(syms))
    {
      s7_pointer sym = car(syms);
      if (!is_symbol(sym))
	wrong_type_error_nr(sc, sc->cutlet_symbol, position_of(syms, args), sym, a_symbol_string);
      if (is_keyword(sym))
	sym = keyword_symbol(sym);

      if (let == sc->rootlet)
	{
	  if (!is_slot(global_slot(sym)))
	    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "cutlet can't remove ~S", 22), sym));
	  if (is_immutable(global_slot(sym)))
	    immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, sc->cutlet_symbol, sym));
	  symbol_set_id(sym, the_un_id);
	  set_global_value(sym, sc->undefined);
	  /* here we need to at least clear bits: syntactic binder clean-symbol(?) etc, maybe also locally */
	}
      else
	{
	  s7_pointer slot;
	  if ((has_let_fallback(let)) &&
	      ((sym == sc->let_ref_fallback_symbol) || (sym == sc->let_set_fallback_symbol)))
	    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "cutlet can't remove ~S", 22), sym));
	  slot = let_slots(let);
	  if (is_not_slot_end(slot))
	    {
	      if (slot_symbol(slot) == sym)
		{
		  if (is_immutable_slot(slot))
		    immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, sc->cutlet_symbol, sym));
		  let_set_slots(let, next_slot(let_slots(let)));
		  symbol_set_id(sym, the_un_id);
		}
	      else
		{
		  s7_pointer last_slot = slot;
		  for (slot = next_slot(let_slots(let)); is_not_slot_end(slot); last_slot = slot, slot = next_slot(slot))
		    if (slot_symbol(slot) == sym)
		      {
			if (is_immutable_slot(slot))
			  immutable_object_error_nr(sc, set_elist_3(sc, immutable_error_string, sc->cutlet_symbol, sym));
			symbol_set_id(sym, the_un_id);
			slot_set_next(last_slot, next_slot(slot));
			break;
		      }}}}}
  return(let);
}


/* -------------------------------- sublet -------------------------------- */
static s7_pointer sublet_1(s7_scheme *sc, s7_pointer let, s7_pointer bindings, s7_pointer caller)
{
  const s7_pointer new_let = make_let(sc, let);
  set_all_methods(new_let, let);

  if (!is_null(bindings))
    {
      sc->temp3 = new_let;
      for (s7_pointer slot = NULL, entries = bindings; is_pair(entries); entries = cdr(entries))
	{
	  s7_pointer entry = car(entries), sym, val;

	  switch (type(entry))
	    {
	    case T_SYMBOL:
	      sym = (is_keyword(entry)) ? keyword_symbol(entry) : entry;
	      if (!is_pair(cdr(entries)))
		error_nr(sc, sc->syntax_error_symbol,
			 set_elist_4(sc, wrap_string(sc, "~A: entry ~S, but no value: ~S", 30), caller, entry, bindings));
	      entries = cdr(entries);
	      val = car(entries);
	      break;

	    case T_PAIR:  /* (cons sym val) */
	      sym = car(entry);
	      if (!is_symbol(sym))
		wrong_type_error_nr(sc, caller, 1 + position_of(entries, bindings), entry, a_symbol_string);
	      if (is_keyword(sym))
		sym = keyword_symbol(sym);
	      val = cdr(entry);
	      break;

	    case T_LET:
	      if ((entry == sc->rootlet) || (new_let == sc->starlet)) continue;
	      append_let(sc, new_let, entry);
	      if (is_not_slot_end(let_slots(new_let))) /* make sure the end slot (slot) is correct */
		for (slot = let_slots(new_let); is_not_slot_end(next_slot(slot)); slot = next_slot(slot)); /* slot can't be local -- see below */
	      continue;

	    default:
	      wrong_type_error_nr(sc, caller, 1 + position_of(entries, bindings), entry, a_symbol_string);
	    }
	  if (is_constant_symbol(sc, sym))
	    wrong_type_error_nr(sc, caller, 1 + position_of(entries, bindings), sym, a_non_constant_symbol_string);
#if 0
	  if ((is_slot(global_slot(sym))) &&
	      (is_syntax_or_qq(global_value(sym))))
	    wrong_type_error_nr(sc, caller, 2, sym, wrap_string(sc, "a non-syntactic symbol", 22));
	  /* this is a local redefinition which we accept elsewhere: (let ((if 3)) if) -> 3 */
	  /*   so s7_inlet (which calls sublet) differs from g_inlet? which is correct? */
	  /*   (define (f1) (with-let (sublet (curlet)) (inlet 'quasiquote 1))) (f1) */

#endif
	  /* here we know new_let is a let and is not rootlet */
	  if (!slot)
	    slot = add_slot_checked_with_id(sc, new_let, sym, val);
	  else
	    {
	      /* if (sc->free_heap_top <= sc->free_heap_trigger) try_to_call_gc(sc);*/ /* or maybe add add_slot_at_end_checked? */
	      slot = add_slot_checked_at_end(sc, let_id(new_let), slot, sym, val);
	      set_local(sym); /* ? */
	    }
	  check_let_fallback(sc, sym, new_let);
	}
      if ((s7_Debugging) && (sc->temp3 != new_let)) fprintf(stderr, "%s[%d]: temp3: %s\n", __func__, __LINE__, display(sc->temp3));
      sc->temp3 = sc->unused;
    }
  return(new_let);
}

s7_pointer s7_sublet(s7_scheme *sc, s7_pointer let, s7_pointer bindings) {return(sublet_1(sc, let, bindings, sc->sublet_symbol));}

static s7_pointer g_sublet(s7_scheme *sc, s7_pointer args)
{
  #define H_sublet "(sublet let ...) makes a new let (an environment) within the environment 'let', initializing it with the bindings"
  #define Q_sublet Q_varlet

  s7_pointer let = car(args);
  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if ((!is_let(new_let)) || (new_let == sc->rootlet))  /* not rootlet because find_let -> rootlet means c_object|pointer did not have a let?? */
	find_let_error_nr(sc, sc->sublet_symbol, let, new_let, 1, args);
      let = new_let;
    }
  return(sublet_1(sc, let, cdr(args), sc->sublet_symbol));
}

static s7_pointer g_sublet_curlet(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym = cadr(args), new_let;
  if_let_method_exists_return_value(sc, sc->curlet, sc->sublet_symbol, args); /* curlet is a let so... */
  new_let = inline_make_let_with_slot(sc, sc->curlet, sym, caddr(args));
  set_all_methods(new_let, sc->curlet);
  check_let_fallback(sc, sym, new_let);
  return(new_let);
}

static s7_pointer sublet_chooser(s7_scheme *sc, s7_pointer func, int32_t num_args, s7_pointer expr)
{
  if (num_args == 3)
    {
      s7_pointer args = cdr(expr);
      if ((is_pair(car(args))) && (caar(args) == sc->curlet_symbol) && (is_null(cdar(args))) &&
	  (is_quoted_symbol(sc, cadr(args))))
	return(sc->sublet_curlet);
    }
  return(func);
}


/* -------------------------------- inlet -------------------------------- */
s7_pointer s7_inlet(s7_scheme *sc, s7_pointer args)
{
  #define H_inlet "(inlet ...) adds its arguments, each a let, a cons: '(symbol . value), or a symbol/value pair, \
to a new let, and returns the new let. (inlet :a 1 :b 2) or (inlet 'a 1 'b 2)"
  #define Q_inlet s7_make_circular_signature(sc, 1, 2, sc->is_let_symbol, sc->T)
  return(sublet_1(sc, sc->rootlet, args, sc->inlet_symbol));
}

#define g_inlet s7_inlet

static s7_pointer g_simple_inlet(s7_scheme *sc, s7_pointer args)
{
  /* here all args are paired with normal symbol/value, no fallbacks, no immutable symbols, no syntax, etc */
  const s7_pointer new_let = make_let(sc, sc->rootlet);
  const s7_int id = let_id(new_let);

  begin_temp(sc->temp6, new_let);
  for (s7_pointer x = args, last_slot = NULL; is_pair(x); x = cddr(x))
    {
      s7_pointer symbol = car(x);
      if (is_keyword(symbol))                 /* (inlet ':allow-other-keys 3) */
	symbol = keyword_symbol(symbol);
      if (is_constant_symbol(sc, symbol))     /* (inlet 'pi 1) */
	{
	  end_temp(sc->temp6);
	  wrong_type_error_nr(sc, sc->inlet_symbol, 1, symbol, a_non_constant_symbol_string);
	}
      if (!last_slot)
	{
	  add_slot_unchecked(sc, new_let, symbol, cadr(x), id);
	  last_slot = let_slots(new_let);
	}
      else last_slot = add_slot_checked_at_end(sc, id, last_slot, symbol, cadr(x));
    }
  end_temp(sc->temp6);
  return(new_let);
}

static s7_pointer inlet_p_pp(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  if (!is_symbol(symbol))
    return(sublet_1(sc, sc->rootlet, set_plist_2(sc, symbol, value), sc->inlet_symbol));
  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);
  if (is_constant_symbol(sc, symbol))
    wrong_type_error_nr(sc, sc->inlet_symbol, 1, symbol, a_non_constant_symbol_string);
  if ((is_defined_global(symbol)) &&
      (is_syntax_or_qq(global_value(symbol))))
    wrong_type_error_nr(sc, sc->inlet_symbol, 1, symbol, wrap_string(sc, "a non-syntactic symbol", 22));
  {
    s7_pointer new_let;
    new_cell(sc, new_let, T_LET | T_Safe_Procedure);
    begin_temp(sc->x, new_let);
    let_set_id(new_let, ++sc->let_number);
    let_set_outlet(new_let, sc->rootlet);
    let_set_slots(new_let, Slot_End);
    add_slot_unchecked(sc, new_let, symbol, value, let_id(new_let));
    end_temp(sc->x);
    return(new_let);
  }
}

static s7_pointer internal_inlet(s7_scheme *sc, s7_int num_args, ...) /* used in *->let */
{
  va_list ap;
  const s7_pointer new_let = make_let(sc, sc->rootlet);
  const s7_int id = let_id(new_let);
  s7_pointer last_slot = NULL;

  begin_temp(sc->x, new_let);
  va_start(ap, num_args);
  for (s7_int i = 0; i < num_args; i += 2)
    {
      s7_pointer symbol = T_Sym(va_arg(ap, s7_pointer));
      s7_pointer value = T_Ext(va_arg(ap, s7_pointer));
      if (!last_slot)
	{
	  add_slot_unchecked(sc, new_let, symbol, value, id);
	  last_slot = let_slots(new_let);
	}
      else last_slot = add_slot_at_end(sc, id, last_slot, symbol, value);
    }
  va_end(ap);
  end_temp(sc->x);
  return(new_let);
}

static bool is_proper_quote(s7_scheme *sc, s7_pointer p)
{
  return((is_safe_quoted_pair(sc, p)) &&
	 (is_pair(cdr(p))) &&
	 (is_null(cddr(p))));
}

static s7_pointer inlet_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  if ((args > 0) && ((args % 2) == 0))
    {
      for (s7_pointer p = cdr(expr); is_pair(p); p = cddr(p))
	{
	  s7_pointer sym;
	  if (is_symbol_and_keyword(car(p)))                  /* (inlet :if ...) */
	    sym = keyword_symbol(car(p));
	  else
	    {
	      if (!is_proper_quote(sc, car(p))) return(func); /* (inlet abs ...) */
	      sym = cadar(p);                                 /* looking for (inlet 'a ...) */
	      if (!is_symbol(sym)) return(func);              /* (inlet '(a . 3) ...) */
	      if (is_keyword(sym)) sym = keyword_symbol(sym); /* (inlet ':abs ...) */
	    }
	  if ((is_possibly_constant(sym)) ||                  /* (inlet 'define-constant ...) or (inlet 'pi ...) */
	      (is_syntactic_symbol(sym))  ||                  /* (inlet 'if 3) */
	      ((is_slot(global_slot(sym))) &&
	       (is_syntax_or_qq(global_value(sym)))) ||       /* (inlet 'quasiquote 1) */
	      (sym == sc->let_ref_fallback_symbol) ||
	      (sym == sc->let_set_fallback_symbol))
	    return(func);
	}
      return(sc->simple_inlet);
    }
  return(func);
}


/* -------------------------------- let->list -------------------------------- */
static s7_pointer proper_list_reverse_in_place(s7_scheme *sc, s7_pointer list);

static s7_pointer abbreviate_let(s7_scheme *sc, s7_pointer val)
{
  if (is_let(val))
    return(make_symbol(sc, "<inlet...>", 11));
  return(val);
}

s7_pointer s7_let_to_list(s7_scheme *sc, s7_pointer let)
{
  if (let == sc->rootlet)
    {
      begin_temp(sc->temp6, sc->nil);
      for (s7_pointer lib = global_value(sc->libraries_symbol); is_pair(lib); lib = cdr(lib))
	sc->temp6 = cons(sc, caar(lib), sc->temp6);
      sc->temp6 = cons(sc, cons(sc, sc->libraries_symbol, sc->temp6), sc->nil);
      for (s7_pointer slot = sc->rootlet_slots; is_not_slot_end(slot); slot = next_slot(slot))
	if (slot_symbol(slot) != sc->libraries_symbol)
	  sc->temp6 = cons_unchecked(sc, cons(sc, slot_symbol(slot), abbreviate_let(sc, slot_value(slot))), sc->temp6);
      {
	s7_pointer result = proper_list_reverse_in_place(sc, sc->temp6);
	end_temp(sc->temp6);
	return(result);
      }}
  else
    {
      s7_pointer iter, func;
      s7_int gc_loc = -1;
      /* need to check make-iterator method before dropping into let->list */
      sc->temp3 = sc->w;
      sc->w = sc->nil;

      if ((has_active_methods(sc, let)) &&
	  ((func = find_method(sc, let, sc->make_iterator_symbol)) != sc->undefined))
	iter = s7_apply_function(sc, func, set_plist_1(sc, let));
      else
	if (let == sc->starlet) /* (let->list *s7*) via starlet_make_iterator */
	  {
	    iter = s7_make_iterator(sc, let);
	    gc_loc = gc_protect_1(sc, iter);
	  }
	else iter = sc->nil;

      if (is_null(iter))
	for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
	  sc->w = cons_unchecked(sc, cons(sc, slot_symbol(slot), slot_value(slot)), sc->w);
      else
	/* (begin (load "mockery.scm") (let ((lt ((*mock-pair* 'mock-pair) 1 2 3))) (format *stderr* "~{~A ~}" lt))) */
	while (true)
	  {
	    s7_pointer val = s7_iterate(sc, iter);
	    if (iterator_is_at_end(iter)) break;
	    sc->w = cons(sc, val, sc->w);
	  }
      sc->w = proper_list_reverse_in_place(sc, sc->w);
      if (gc_loc != -1)
	s7_gc_unprotect_at(sc, gc_loc);
      {
	s7_pointer result = sc->w;
	sc->w = sc->temp3;
	sc->temp3 = sc->unused;
	return(result);
      }}
}

#if !With_Pure_s7
static s7_pointer g_let_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_let_to_list "(let->list let) returns let's bindings as a list of cons's: '(symbol . value)."
  #define Q_let_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, has_let_signature(sc))

  s7_pointer let = car(args);
  if_method_exists_return_value(sc, let, sc->let_to_list_symbol, args);
  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if ((!is_let(new_let)) || (new_let == sc->rootlet))
	find_let_error_nr(sc, sc->let_to_list_symbol, let, new_let, 1, args);
      /* this is not (let->list (rootlet)) but (say) (let->list func) which defaults in find_let to rootlet */
      let = new_let;
    }
  return(s7_let_to_list(sc, let));
}
/* *s7* in gdb: p display(s7_let_to_list(sc, sc->starlet)) */
#endif


/* -------------------------------- let-ref -------------------------------- */
static s7_pointer call_let_ref_fallback(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  s7_pointer result;
  const s7_pointer val = find_method(sc, let, sc->let_ref_fallback_symbol);
  /* (let ((x #f)) (let begin ((x 1234)) (begin 1) 2)) -> stack overflow eventually, but should we try to catch it? */
  if (!is_applicable(val)) return(val);
  push_stack_no_let(sc, OP_GC_PROTECT, sc->value, sc->code);
  result = s7_apply_function(sc, val, set_qlist_2(sc, let, symbol));
  unstack_gc_protect(sc);
  sc->code = T_Pos(stack_end_code(sc)); /* can be #<unused> */
  sc->value = T_Ext(stack_end_args(sc));
  return(result);
}

static s7_pointer call_let_set_fallback(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  s7_pointer result;
  push_stack_no_let(sc, OP_GC_PROTECT, sc->value, sc->code);
  result = s7_apply_function(sc, find_method(sc, let, sc->let_set_fallback_symbol), set_qlist_3(sc, let, symbol, value));
  unstack_gc_protect(sc);
  sc->code = T_Pos(stack_end_code(sc));
  sc->value = T_Ext(stack_end_args(sc));
  return(result);
}

static s7_pointer g_unlet_disabled(s7_scheme *sc, s7_pointer unused_args) {return(sc->unlet_disabled);}
/* we need a self-id here for let_ref, but it needs to be a real s7_cell, not g_unlet_disabled itself, hence sc->unlet_disabled */

static /* inline */ s7_pointer let_ref(s7_scheme *sc, s7_pointer let, s7_pointer symbol)
{
  /* (let ((a 1)) ((curlet) 'a)) or ((rootlet) 'abs) */
  if (!is_let(let))
    {
      s7_pointer new_let;
      if (let == sc->unlet_disabled) return(initial_value(symbol));
      new_let = find_let(sc, let);
      if ((!is_let(new_let)) || (new_let == sc->rootlet))
	find_let_error_nr(sc, sc->let_ref_symbol, let, new_let, 1, set_mlist_2(sc, let, symbol));
      let = new_let;
    }
  if (!is_symbol(symbol))
    {
      if ((let != sc->rootlet) && (has_let_ref_fallback(let))) /* let-ref|set-fallback refer to (explicit) let-ref in various forms, not the method lookup process */
	return(call_let_ref_fallback(sc, let, symbol));
      wrong_type_error_nr(sc, sc->let_ref_symbol, 2, symbol, a_symbol_string);
    }
  /* a let-ref method is almost impossible to write without creating an infinite loop:
   *   any reference to the let will probably call let-ref somewhere, calling us again, and looping.
   *   This is not a problem in c-objects and funclets because c-object-ref and funclet-ref don't exist.
   *   After much wasted debugging, I decided to make let-ref and let-set! immutable.
   *   What about other let-as-first-arg funcs?
   */

  if (let_id(let) == symbol_id(symbol))
    return(local_value(symbol)); /* this has to follow the rootlet check(?) */

  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);
  if (let == sc->rootlet)
    return((is_slot(global_slot(symbol))) ? global_value(symbol) : sc->undefined);

  for (s7_pointer e = let; e; e = let_outlet(e))
    for (s7_pointer slot = let_slots(e); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == symbol)
	return(slot_value(slot));

  if (is_openlet(let))
    {
      /* If a let is a mock-hash-table (for example), implicit indexing of the hash-table collides with the same thing for the let (field names
       *   versus keys), and we can't just try again here because that makes it too easy to get into infinite recursion.  So, 'let-ref-fallback...
       */
      if (has_let_ref_fallback(let))
	return(call_let_ref_fallback(sc, let, symbol));
    }
  return((is_slot(global_slot(symbol))) ? global_value(symbol) : sc->undefined); /* (let () ((curlet) 'pi)) */
}

s7_pointer s7_let_ref(s7_scheme *sc, s7_pointer let, s7_pointer symbol) {return(let_ref(sc, let, symbol));}

static s7_pointer g_let_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_let_ref "(let-ref let sym) returns the value of the symbol sym in the let"
  #define Q_let_ref s7_make_signature(sc, 3, sc->T, has_let_signature(sc), sc->is_symbol_symbol)
  if (!is_pair(cdr(args)))
    error_nr(sc, sc->syntax_error_symbol,
	     set_elist_2(sc, wrap_string(sc, "let-ref: symbol missing: ~S", 27), set_ulist_1(sc, sc->let_ref_symbol, args)));
  return(let_ref(sc, car(args), cadr(args)));
}

static s7_pointer slot_in_let(s7_scheme *sc, s7_pointer let, const s7_pointer sym)
{
  for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
    if (slot_symbol(slot) == sym)
      return(slot);
  return(sc->undefined);
}

static s7_pointer let_ref_p_pp(s7_scheme *sc, s7_pointer let, s7_pointer sym)
{
  if (let_id(let) == symbol_id(sym))
    return(local_value(sym)); /* see add in tlet! */
  if (let == sc->rootlet) /* op_implicit_let_ref_c can pass rootlet */
    return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);
  for (s7_pointer e = let; e; e = let_outlet(e))
    for (s7_pointer slot = let_slots(e); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == sym)
	return(slot_value(slot));
  if (has_let_ref_fallback(let))
    return(call_let_ref_fallback(sc, let, sym));
  return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);
}

static inline s7_pointer g_cdr_let_ref(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer let = car(args), sym = cadr(args);
  if (!is_let(let))
    wrong_type_error_nr(sc, sc->let_ref_symbol, 1, let, a_let_string);
  if (let_id(let) == symbol_id(sym))
    return(local_value(sym));
  if (let == sc->rootlet)
    return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);
  for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
    if (slot_symbol(slot) == sym)
      return(slot_value(slot));
  return(let_ref_p_pp(sc, let_outlet(let), sym));
}

static s7_pointer starlet(s7_scheme *sc, s7_int choice);
static s7_pointer g_starlet_ref(s7_scheme *sc, s7_pointer args) {return(starlet(sc, starlet_symbol_id(cadr(args))));}
static s7_pointer g_curlet_ref(s7_scheme *sc, s7_pointer args) {return(lookup(sc, cadr(args)));}
static s7_pointer g_unlet_ref(s7_scheme *unused_sc, s7_pointer args) {return(initial_value(cadr(args)));}

static s7_pointer g_rootlet_ref(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym = cadr(args);
  return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);
}

static s7_pointer let_ref_chooser(s7_scheme *sc, s7_pointer func, int32_t unused_args, s7_pointer expr)
{
  const s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
  if ((is_quoted_symbol(sc, arg2)) && (!is_keyword(cadr(arg2))))
    {
      if (is_pair(arg1))
	{
	  if ((optimize_op(expr) == HOP_SAFE_C_opSq_C) && (car(arg1) == sc->cdr_symbol))
	    {
	      set_opt3_sym(cdr(expr), cadr(arg2));
	      return(sc->cdr_let_ref);
	    }
	  if (car(arg1) == sc->rootlet_symbol) return(sc->rootlet_ref);
	  if (car(arg1) == sc->curlet_symbol) return(sc->curlet_ref);
	  if (car(arg1) == sc->unlet_symbol)
	    {
	      set_fn_direct(arg1, g_unlet_disabled);
	      return(sc->unlet_ref);
	    }}
      if (arg1 == sc->starlet_symbol) return(sc->starlet_ref); /* should *curlet* be added? */
    }
  return(func);
}

static bool op_implicit_let_ref_c(s7_scheme *sc)
{
  s7_pointer let = lookup_checked(sc, car(sc->code));
  if (!is_let(let)) {sc->last_function = let; return(false);}
  sc->value = let_ref_p_pp(sc, let, opt3_con(sc->code));
  return(true);
}

static bool op_implicit_let_ref_a(s7_scheme *sc)
{
  s7_pointer sym, let = lookup_checked(sc, car(sc->code));
  if (!is_let(let)) {sc->last_function = let; return(false);}
  sym = fx_call(sc, cdr(sc->code));
  if (is_symbol(sym))
    sc->value = let_ref_p_pp(sc, let, (is_keyword(sym)) ? keyword_symbol(sym) : sym);
  else sc->value = let_ref(sc, let, sym);
  return(true);
}

static s7_pointer fx_implicit_let_ref_c(s7_scheme *sc, s7_pointer arg)
{
  s7_pointer let = lookup_checked(sc, car(arg)); /* the let */
  if (!is_let(let))
    return(s7_apply_function(sc, let, set_mlist_1(sc, opt3_con(arg))));
  return(let_ref_p_pp(sc, let, opt3_con(arg)));
}


/* -------------------------------- let-set! -------------------------------- */
static s7_pointer let_set_1(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);

  if (let == sc->rootlet)
    {
      s7_pointer slot;
      if (is_constant_symbol(sc, symbol))  /* (let-set! (rootlet) 'pi #f) */
	wrong_type_error_nr(sc, sc->let_set_symbol, 2, symbol, a_non_constant_symbol_string);
      /* it would be nice if safety>0 to add an error check for bad arity if a built-in method is set (set! (lt 'write) hash-table-set!),
       *   built_in being (initial_value_is_defined(sc, sym)), but this function is called a ton, and this error can't easily be
       *   checked by the optimizer (we see the names, but not the values, so bad arity check requires assumptions about those values).
       */
      slot = global_slot(symbol);
      if (!is_slot(slot))
	error_nr(sc, sc->wrong_type_arg_symbol,
		 set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), symbol, let));
      if (is_syntax(slot_value(slot)))
	wrong_type_error_nr(sc, sc->let_set_symbol, 2, symbol, wrap_string(sc, "a non-syntactic symbol", 22));
      if (is_immutable(slot))
	immutable_object_error_nr(sc, set_elist_2(sc, wrap_string(sc, "~S is immutable in (rootlet)", 28), symbol)); /* also (set! (with-let...)...) */
      symbol_increment_ctr(symbol);
      slot_set_value(slot, (slot_has_setter(slot)) ? call_setter(sc, slot, value) : value);
      return(slot_value(slot));
    }
  if (is_unlet(let))
    immutable_object_error_nr(sc, set_elist_2(sc, wrap_string(sc, "~S is immutable in (unlet)", 26), symbol));
  if (let_id(let) == symbol_id(symbol))
   {
     s7_pointer slot = local_slot(symbol);
     if (is_slot(slot))
       {
	 symbol_increment_ctr(symbol);
	 return(checked_slot_set_value(sc, slot, value));
       }}
  for (s7_pointer e = let; e; e = let_outlet(e))
    for (s7_pointer slot = let_slots(e); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == symbol)
	{
	  symbol_increment_ctr(symbol);
	  return(checked_slot_set_value(sc, slot, value));
	}
  if (!has_let_set_fallback(let))
    error_nr(sc, sc->wrong_type_arg_symbol,
	     set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), symbol, let));
  /* not sure about this -- what's the most useful choice? */
  return(call_let_set_fallback(sc, let, symbol, value));
}

static s7_pointer let_set_2(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if (!is_let(new_let))
	find_let_error_nr(sc, sc->let_set_symbol, let, new_let, 1, set_plist_3(sc, let, symbol, value));
      let = new_let;
    }
  if (!is_symbol(symbol))
    {
      if ((let != sc->rootlet) && (has_let_set_fallback(let)))
	return(call_let_set_fallback(sc, let, symbol, value));
      wrong_type_error_nr(sc, sc->let_set_symbol, 2, symbol, a_symbol_string);
    }
  /* currently let-set! is immutable, so we don't have to check for a let-set! method (so let_set! is always global) */
  return(let_set_1(sc, let, symbol, value));
}

s7_pointer s7_let_set(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value) {return(let_set_2(sc, let, symbol, value));}

static s7_pointer g_let_set(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (set! ((curlet) 'a) 32) a) */
  #define H_let_set "(let-set! let sym val) sets the symbol sym's value in the let to val"
  #define Q_let_set s7_make_signature(sc, 4, sc->T, has_let_signature(sc), sc->is_symbol_symbol, sc->T)

  if (!is_pair(cdr(args))) /* (let ((a 123.0)) (define (f) (set! (let-ref) a)) (catch #t f (lambda args #f)) (f)) */
    error_nr(sc, sc->wrong_number_of_args_symbol,
	     set_elist_3(sc, wrap_string(sc, "~S: not enough arguments: ~S", 28), sc->let_set_symbol, sc->code));

  return(let_set_2(sc, car(args), cadr(args), caddr(args)));
}

static s7_pointer let_set_p_ppp_2(s7_scheme *sc, s7_pointer let, s7_pointer sym, s7_pointer val)
{
  if (!is_symbol(sym))
    wrong_type_error_nr(sc, sc->let_set_symbol, 2, sym, a_symbol_string);
  return(let_set_1(sc, let, sym, val));
}

static s7_pointer g_cdr_let_set(s7_scheme *sc, s7_pointer args)
{
  s7_pointer let = car(args);
  const s7_pointer sym = cadr(args), val = caddr(args);
  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if (!is_let(new_let))
	find_let_error_nr(sc, sc->let_set_symbol, let, new_let, 1, args);
      let = new_let;
    }
  if (let != sc->rootlet)
    {
      for (s7_pointer e = let; e; e = let_outlet(e))
	for (s7_pointer slot = let_slots(e); is_not_slot_end(slot); slot = next_slot(slot))
	  if (slot_symbol(slot) == sym)
	    {
	      slot_set_value(slot, (slot_has_setter(slot)) ? call_setter(sc, slot, val) : val);
	      return(slot_value(slot));
	    }
      if ((let != sc->rootlet) && (has_let_set_fallback(let)))
	return(call_let_set_fallback(sc, let, sym, val));
    }
  {
    s7_pointer slot = global_slot(sym);
    if (!is_slot(slot))
      error_nr(sc, sc->wrong_type_arg_symbol, set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), sym, let));
    slot_set_value(slot, (slot_has_setter(slot)) ? call_setter(sc, slot, val) : val);
    return(slot_value(slot));
  }
}

static s7_pointer starlet_set_1(s7_scheme *sc, s7_pointer sym, s7_pointer val);

static s7_pointer g_starlet_set(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym = cadr(args);
  if (!is_symbol(sym)) /* (let () (define (func) (let-set! *s7* '(1 . 2) (hash-table))) (func) (func)) */
    error_nr(sc, sc->wrong_type_arg_symbol,
	     set_elist_3(sc, wrap_string(sc, "(let-set! *s7* ~A ...) second argument is ~A but should be a symbol", 67),
			 sym, object_type_name(sc, sym)));
  if (is_keyword(sym))
    sym = keyword_symbol(sym);
  if (starlet_symbol_id(sym) == sl_No_Field)
    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "can't set (*s7* '~S); no such field in *s7*", 43), sym));
  return(starlet_set_1(sc, sym, caddr(args)));
}

static s7_pointer g_unlet_set(s7_scheme *sc, s7_pointer args)
{
  immutable_object_error_nr(sc, set_elist_2(sc, wrap_string(sc, "~S is immutable in (unlet)", 26), cadr(args)));
  return(sc->F);
}

static s7_pointer let_set_chooser(s7_scheme *sc, s7_pointer func, int32_t unused_args, s7_pointer expr)
{
  const s7_pointer arg1 = cadr(expr);
  if (optimize_op(expr) == HOP_SAFE_C_opSq_CS)
    {
      const s7_pointer arg2 = caddr(expr), arg3 = cadddr(expr);
      if ((car(arg1) == sc->cdr_symbol) &&
	  (is_quoted_symbol(sc, arg2)) &&
	  (!is_possibly_constant(cadr(arg2))) && /* assumes T_Sym */
	  (!is_possibly_constant(arg3)))
	return(sc->cdr_let_set);
      if (car(arg1) == sc->unlet_symbol)
	{
	  set_fn_direct(arg1, g_unlet_disabled);
	  return(sc->unlet_set);
	}}
  if (arg1 == sc->starlet_symbol) return(sc->starlet_set);
  return(func);
}


static s7_pointer reverse_slots(s7_pointer let_slots)
{
  s7_pointer slot = let_slots, result = Slot_End;
  while (is_not_slot_end(slot))
    {
      s7_pointer nextslot = next_slot(slot);
      slot_set_next(slot, result);
      result = slot;
      slot = nextslot;
    }
  return(result);
}

static s7_pointer let_copy(s7_scheme *sc, s7_pointer let)
{
  s7_pointer new_let;
  if (T_Let(let) == sc->rootlet)   /* (copy (rootlet)) or (copy (funclet abs)) etc */
    return(sc->rootlet);
  /* we can't make copy handle lets-as-objects specially because the make-object function in define-class uses copy to make a new object!
   *   So if it is present, we get it here, and then there's almost surely trouble.
   */
  new_let = make_let(sc, let_outlet(let));
  set_all_methods(new_let, let);
  begin_temp(sc->x, new_let);
  if (is_not_slot_end(let_slots(let)))
    {
      const s7_int id = let_id(new_let);
      for (s7_pointer last_slot = NULL, slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
	{
	  s7_pointer new_slot;
	  new_cell(sc, new_slot, T_SLOT);
	  slot_set_symbol_and_value(new_slot, slot_symbol(slot), slot_value(slot));
	  if (symbol_id(slot_symbol(new_slot)) != id) /* keep shadowing intact */
	    symbol_set_local_slot(slot_symbol(slot), id, new_slot);
	  if (slot_has_setter(slot))
	    {
	      slot_set_setter(new_slot, slot_setter(slot));
	      slot_set_has_setter(new_slot);
	    }
	  if (last_slot)
	    slot_set_next(last_slot, new_slot);
	  else let_set_slots(new_let, new_slot);
	  slot_set_next(new_slot, Slot_End);        /* in case GC runs during this loop */
	  last_slot = new_slot;
	}}
  /* We can't do a (normal) loop here then reverse the slots later because the symbol's local_slot has to
   *    match the unshadowed slot, not the last in the list:
   *    (let ((e1 (inlet 'a 1 'a 2))) (let ((e2 (copy e1))) (list (equal? e1 e2) (equal? (e1 'a) (e2 'a)))))
   */
  end_temp(sc->x);
  return(new_let);
}


/* -------------------------------- rootlet -------------------------------- */
static s7_pointer g_rootlet(s7_scheme *sc, s7_pointer unused)
{
  #define H_rootlet "(rootlet) returns the current top-level definitions (symbol bindings)."
  #define Q_rootlet s7_make_signature(sc, 1, sc->is_let_symbol)
  return(sc->rootlet);
}

s7_pointer s7_rootlet(s7_scheme *sc) {return(sc->rootlet);}

/* shadow_rootlet is a convenience for foreign function writers -- the C code can act as if it were loading everything into rootlet,
 *   but when actually loaded, everything can be shunted into a separate namespace (*motif* for example).
 */
s7_pointer s7_shadow_rootlet(s7_scheme *sc) {return(sc->shadow_rootlet);}

s7_pointer s7_set_shadow_rootlet(s7_scheme *sc, s7_pointer let)
{
  s7_pointer old_let = sc->shadow_rootlet;
  sc->shadow_rootlet = let;
  return(old_let); /* like s7_set_curlet below */
}


/* -------------------------------- curlet -------------------------------- */
s7_pointer s7_curlet(s7_scheme *sc) /* see also fx_curlet */
{
  sc->capture_let_counter++;
  return(sc->curlet);
}

static s7_pointer g_curlet(s7_scheme *sc, s7_pointer unused_args)
{
  #define H_curlet "(curlet) returns the current definitions (symbol bindings)"
  #define Q_curlet s7_make_signature(sc, 1, sc->is_let_symbol)
  sc->capture_let_counter++;
  return(sc->curlet);
}

static void update_symbol_ids(s7_scheme *sc, s7_pointer let)
{
  for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
    {
      s7_pointer sym = slot_symbol(slot);
      if (symbol_id(sym) != sc->let_number)
	symbol_set_local_slot_unincremented(sym, sc->let_number, slot);
    }
}

s7_pointer s7_set_curlet(s7_scheme *sc, s7_pointer let)
{
  const s7_pointer old_let = sc->curlet;
  if (is_let(let))
    {
      set_curlet(sc, let);
      if (let_id(let) > 0)
	{
	  let_set_id(let, ++sc->let_number);
	  update_symbol_ids(sc, let);
	}}
  return(old_let);
}


/* -------------------------------- outlet -------------------------------- */
s7_pointer s7_outlet(s7_scheme *sc, s7_pointer let) {return(let_outlet(let));}

static s7_pointer outlet_p_p(s7_scheme *sc, s7_pointer let)
{
  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if (!is_let(new_let))
	find_let_error_nr(sc, sc->outlet_symbol, let, new_let, 1, set_mlist_1(sc, let));
      let = new_let;
    }
  return((let == sc->rootlet) ? sc->rootlet : let_outlet(let)); /* rootlet check is needed(!) */
}

static s7_pointer g_outlet_unlet(s7_scheme *sc, s7_pointer unused_args) {return(sc->curlet);}

static s7_pointer g_outlet(s7_scheme *sc, s7_pointer args)
{
  #define H_outlet "(outlet let) is the environment that contains let."
  #define Q_outlet s7_make_signature(sc, 2, sc->is_let_symbol, has_let_signature(sc))
  return(outlet_p_p(sc, car(args)));
}

static s7_pointer outlet_chooser(s7_scheme *sc, s7_pointer func, int32_t num_args, s7_pointer expr)
{
  if ((num_args == 1) && (is_pair(cadr(expr))) && (caadr(expr) == sc->unlet_symbol))
    {
      set_fn_direct(cadr(expr), g_unlet_disabled);
      return(sc->outlet_unlet);
    }
  return(func);
}

static s7_pointer g_set_outlet(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (let ((b 2)) (set! (outlet (curlet)) (rootlet)) ((curlet) 'a))) */
  s7_pointer let = car(args), new_outer;

  if (!is_let(let))
    {
      s7_pointer new_let = find_let(sc, let);
      if (!is_let(new_let))
	find_let_error_nr(sc, wrap_string(sc, "set! outlet", 11), let, new_let, 1, args);
      let = new_let;
    }
  if (let == sc->starlet)
    error_nr(sc, sc->out_of_range_symbol, set_elist_1(sc, wrap_string(sc, "can't set! (outlet *s7*)", 24)));
  if (is_immutable_let(let))
    immutable_object_error_nr(sc, set_elist_4(sc, wrap_string(sc, "can't (set! (outlet ~S) ~S), ~S is immutable", 44), let, cadr(args), let));
  new_outer = cadr(args);
  if (!is_let(new_outer))
    {
      s7_pointer new_let = find_let(sc, new_outer);
      if (!is_let(new_let))
	find_let_error_nr(sc, wrap_string(sc, "set! outlet", 11), new_outer, new_let, 2, args);
      new_outer = new_let;
    }
  if (let != sc->rootlet)
    {
      /* here it's possible to get cyclic let chains; maybe do this check only if safety>0 */
      for (s7_pointer new_let = new_outer; new_let; new_let = let_outlet(new_let))
	if (let == new_let)
	  error_nr(sc, make_symbol(sc, "cyclic-let", 10),
		   set_elist_2(sc, wrap_string(sc, "set! (outlet ~A) creates a cyclic let chain", 43), let));
      let_set_outlet(let, new_outer);
    }
  return(new_outer);
}

/* -------------------------------- symbol lookup -------------------------------- */
static Inline s7_pointer inline_lookup_from(s7_scheme *sc, const s7_pointer symbol, s7_pointer let)
{
  /* splitting out the no-sc WITH_GCC case made no difference in speed, same if using s7_int id = symbol_id(symbol) */
  if (let_id(let) == symbol_id(symbol))
    return(local_value(symbol));
  if (let_id(let) > symbol_id(symbol)) /* let is newer so look back in the outlet chain */
    {
      do {let = let_outlet(let);} while (let_id(let) > symbol_id(symbol));
      if (let_id(let) == symbol_id(symbol))
	return(local_value(symbol));
    }
  for (; let; let = let_outlet(let))
    for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == symbol)
	return(slot_value(slot));

  if (is_slot(global_slot(symbol)))
    return(global_value(symbol));
#if WITH_GCC && ((!__cplusplus) || (!__clang__))
  return(NULL); /* much faster than various alternatives */
#else
  return(unbound_variable(sc, symbol)); /* only use of sc */
#endif
}

#if WITH_GCC && s7_Debugging
static s7_pointer lookup_1(s7_scheme *sc, const s7_pointer symbol)
#else
static inline s7_pointer lookup(s7_scheme *sc, const s7_pointer symbol) /* lookup_checked includes the unbound_variable call */
#endif
{
  return(inline_lookup_from(sc, symbol, sc->curlet));
}

static inline s7_pointer lookup_slot_from(s7_pointer symbol, s7_pointer let)
{
  if (let_id(let) == symbol_id(symbol))
    return(T_Slt(local_slot(symbol)));
  if (let_id(let) > symbol_id(symbol))
    {
      do {let = let_outlet(let);} while (let_id(let) > symbol_id(symbol));
      if (let_id(let) == symbol_id(symbol))
	return(T_Slt(local_slot(symbol)));
    }
  for (; let; let = let_outlet(let))
    for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == symbol)
	return(T_Slt(slot));
  return(T_Sld(global_slot(symbol))); /* #<undefined> when (define x ...) and x is not yet defined */
}

s7_pointer s7_slot(s7_scheme *sc, s7_pointer symbol) {return(lookup_slot_from(symbol, sc->curlet));}

static s7_pointer lookup_slot_with_let(s7_scheme *sc, s7_pointer symbol, s7_pointer let) {return(lookup_slot_from(symbol, let));}

s7_pointer s7_slot_value(s7_pointer slot) {return(slot_value(slot));}

s7_pointer s7_slot_set_value(s7_scheme *sc, s7_pointer slot, s7_pointer value) {slot_set_value(slot, value); return(value);}

void s7_slot_set_real_value(s7_scheme *sc, s7_pointer slot, s7_double value) {set_real(slot_value(slot), value);}

static s7_pointer symbol_to_local_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer let)
{ /* assumes e is a let, only let is searched */
  if (T_Let(let) == sc->rootlet)
    return(global_slot(symbol));
  if (!is_global(symbol))  /* i.e. rootlet is not the desired let, and the symbol might have a local value */
    for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == symbol)
	return(slot);
  return(sc->undefined);
}

s7_pointer s7_symbol_value(s7_scheme *sc, s7_pointer sym)
{
  s7_pointer slot = s7_slot(sc, sym);
  return((is_slot(slot)) ? slot_value(slot) : sc->undefined);
}

s7_pointer s7_symbol_local_value(s7_scheme *sc, s7_pointer sym, s7_pointer let)
{
  if (let_id(let) == symbol_id(sym))
    return(local_value(sym));
  if (let_id(let) > symbol_id(sym))
    {
      do {let = let_outlet(let);} while (let_id(let) > symbol_id(sym));
      if (let_id(let) == symbol_id(sym))
	return(local_value(sym));
    }
  for (; let; let = let_outlet(let))
    for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == sym)
	return(slot_value(slot));

  /* maybe let is local but sym is global but previously shadowed */
  if (is_slot(global_slot(sym)))
    return(global_value(sym));

  /* (let ((e (curlet))) (let ((a 1)) (symbol->value 'a e))) -> #<undefined> not 1 */
  return(sc->undefined); /* 29-Nov-17 */
}


/* -------------------------------- symbol->value -------------------------------- */
#define lookup_global(Sc, Sym) ((is_defined_global(Sym)) ? global_value(Sym) : lookup_checked(Sc, Sym))

static s7_pointer g_symbol_to_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_value "(symbol->value sym (let (curlet))) returns the binding of (the value associated with) the \
symbol sym in the given let: (let ((x 32)) (symbol->value 'x)) -> 32"
  #define Q_symbol_to_value s7_make_signature(sc, 3, sc->T, sc->is_symbol_symbol, has_let_signature(sc))

  const s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->symbol_to_value_symbol, args, sc->type_names[T_SYMBOL], 1));
  if (is_keyword(sym))
    {
      if ((is_pair(cdr(args))) && (!is_let(cadr(args))) && (!is_let(find_let(sc, cadr(args)))))
	wrong_type_error_nr(sc, sc->symbol_to_value_symbol, 2, cadr(args), sc->type_names[T_LET]);
      return(sym);
    }
  if (is_pair(cdr(args)))
    {
      s7_pointer local_let = cadr(args);
      if (!is_let(local_let))
	{
	  local_let = find_let(sc, local_let);
	  if (!is_let(local_let))
	    return(method_or_bust(sc, cadr(args), sc->symbol_to_value_symbol, args, a_let_string, 2)); /* not local_let */
	}
      if (local_let == sc->rootlet) return((is_slot(global_slot(sym))) ? global_value(sym) : sc->undefined);
      if (is_unlet(local_let)) return(initial_value(sym));
      if (local_let == sc->starlet) return(starlet(sc, starlet_symbol_id(sym)));
      return(s7_symbol_local_value(sc, sym, local_let));
    }
  if (is_defined_global(sym))
    return(global_value(sym));
  return(s7_symbol_value(sc, sym));
}

s7_pointer s7_symbol_set_value(s7_scheme *sc, s7_pointer sym, s7_pointer val)
{
  s7_pointer slot = s7_slot(sc, sym);   /* if immutable should this return an error? */
  if (is_slot(slot))
    slot_set_value(slot, val); /* with_hook? */
  return(val);
}

static s7_pointer g_sv_unlet_ref(s7_scheme *unused_sc, s7_pointer args) {return(initial_value(car(args)));}

static s7_pointer symbol_to_value_chooser(s7_scheme *sc, s7_pointer func, int32_t unused_args, s7_pointer expr)
{
  s7_pointer arg1 = cadr(expr), arg2 = (is_pair(cddr(expr))) ? caddr(expr) : sc->F;
  if ((is_quoted_symbol(sc, arg1)) && (!is_keyword(cadr(arg1))) && (is_pair(arg2)) && (car(arg2) == sc->unlet_symbol)) /* old-style (obsolete) unlet as third arg(!) */
    {
      set_fn_direct(arg2, g_unlet_disabled);
      return(sc->sv_unlet_ref);
    }
  return(func);
}


/* -------------------------------- symbol->dynamic-value -------------------------------- */
static s7_pointer find_dynamic_value(s7_scheme *sc, s7_pointer let, s7_pointer sym, s7_int *id)
{
  for (; let_id(let) > symbol_id(sym); let = let_outlet(let));
  if (let_id(let) == symbol_id(sym))
    {
      (*id) = let_id(let);
      return(local_value(sym));
    }
  for (; (let) && (let_id(let) > (*id)); let = let_outlet(let))
    for (s7_pointer slot = let_slots(let); is_not_slot_end(slot); slot = next_slot(slot))
      if (slot_symbol(slot) == sym)
	{
	  (*id) = let_id(let);
	  return(slot_value(slot));
	}
  return(sc->unused);
}

static s7_pointer g_symbol_to_dynamic_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_dynamic_value "(symbol->dynamic-value sym) returns the dynamic binding of the symbol sym"
  #define Q_symbol_to_dynamic_value s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)

  /* symbol->dynamic-value assumes the "true" curlet is saved on the stack if a function is called that
   *   calls the code with symbol->dynamic-value.  But the function might simply reset curlet to its own, then return,
   *   as in op_safe_thunk_a, leaving the incoming curlet inaccessible.  If we push op_gc_protect+let=incoming curlet,
   *   everything is ok.
   *   (let ((x 32))
   *     (define (gx) (symbol->dynamic-value 'x))
   *     (define (ok?) (format #t "gx -> ~S~%" (gx)))
   *     (let ((x 12))
   *       (ok? gx) ; 32, but 12 if we include the line below (since x=12 let is on the stack)
   *       ;(newline)
   *       ))
   */
  const s7_pointer sym = car(args);
  s7_pointer val;
  s7_int top_id = -1;

  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->symbol_to_dynamic_value_symbol, args, sc->type_names[T_SYMBOL], 1));

  if (is_defined_global(sym))
    return(global_value(sym));

  if (let_id(sc->curlet) == symbol_id(sym))
    return(local_value(sym));

  val = find_dynamic_value(sc, sc->curlet, sym, &top_id);
  if (top_id == symbol_id(sym))
    return(val);

  for (s7_int op_loc = stack_top(sc) - 1; op_loc > 0; op_loc -= 4)
    if (is_let_unchecked(stack_let(sc->stack, op_loc))) /* OP_GC_PROTECT let slot can be anything (even free) */
      {
	s7_pointer cur_val = find_dynamic_value(sc, stack_let(sc->stack, op_loc), sym, &top_id);
	if (cur_val != sc->unused)
	  val = cur_val;
	if (top_id == symbol_id(sym))
	  return(val);
      }
  /* what about call/cc stacks? */
  return((val == sc->unused) ? s7_symbol_value(sc, sym) : val);
}

static bool direct_memq(const s7_pointer symbol, s7_pointer symbols)
{
  for (s7_pointer syms = symbols; is_pair(syms); syms = cdr(syms))
    if (car(syms) == symbol)
      return(true);
  return(false);
}

static bool direct_translucent_member(const s7_pointer symbol, s7_pointer symbols)
{
  for (s7_pointer syms = symbols; is_pair(syms); syms = cdr(syms))
    {
      if (car(syms) == symbol) return(true);
      if ((is_pair(car(syms))) && (is_translucent(caar(syms))) && (is_pair(cdar(syms))) && (cadar(syms) == symbol)) return(true);
    }
  return(false);
}

static bool direct_assq(const s7_pointer symbol, s7_pointer symbols) /* used only below in do_symbol_is_safe */
{
  for (s7_pointer syms = symbols; is_pair(syms); syms = cdr(syms))
    if (caar(syms) == symbol)
      return(true);
  return(false);
}

static bool do_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer var_list)
{
  return((is_slot(global_slot(sym))) ||
	 (direct_assq(sym, var_list)) ||
	 (is_bound_symbol(sc, sym)));
}

static bool let_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer var_list)
{
  if (is_slot(global_slot(sym)))
    return(true);
  if (var_list == sc->rootlet)
    return(false);
  return((!is_with_let_let(var_list)) && (is_bound_symbol(sc, sym)));
}

static bool let_symbol_is_safe_or_listed(s7_scheme *sc, s7_pointer sym, s7_pointer var_list)
{
  return((symbol_is_in_big_symbol_set(sc, sym)) ||
	 (let_symbol_is_safe(sc, sym, var_list)));
}

static bool let_star_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer var_list)
{
  return((symbol_is_in_big_symbol_set(sc, sym)) ||
	 (is_slot(global_slot(sym))) ||
	 ((is_let(var_list)) && (!is_with_let_let(var_list)) && (is_bound_symbol(sc, sym))));
}

static bool pair_symbol_is_safe(s7_scheme *unused_sc, s7_pointer sym, s7_pointer symbols)
{
  return((is_slot(global_slot(sym))) ||
	 (direct_memq(sym, symbols)));  /* optimize_syntax pushes :if (and others like () I think) on this list */
}

static s7_pointer collect_variables(s7_scheme *sc, s7_pointer lst, s7_pointer var_list)
{
  /* collect local variable names from let/do (pre-error-check), 20 overhead in tgen -> 14 if cons_unchecked below */
  begin_temp(sc->y, var_list);
  for (s7_pointer p = lst; is_pair(p); p = cdr(p))
    sc->y = cons(sc, add_symbol_to_big_symbol_set(sc, caar(p)), sc->y);
  return_with_end_temp(sc->y);
}

static s7_pointer collect_parameters(s7_scheme *sc, s7_pointer lst, s7_pointer var_list)
{
  /* collect local variable names from lambda arglists (pre-error-check) */
  s7_pointer pars;
  const s7_int the_un_id = ++sc->let_number;
  if (is_normal_symbol(lst))
    {
      symbol_set_id(lst, the_un_id);
      return(cons(sc, add_symbol_to_big_symbol_set(sc, lst), var_list));
    }
  begin_temp(sc->y, var_list);
  for (pars = lst; is_pair(pars); pars = cdr(pars))
    {
      s7_pointer par = car(pars);
      if (is_pair(par))
	par = car(par);
      if (is_normal_symbol(par))
	{
	  symbol_set_id(par, the_un_id);
	  sc->y = cons(sc, add_symbol_to_big_symbol_set(sc, par), sc->y);
	}}
  if (is_normal_symbol(pars)) /* rest arg */
    {
      symbol_set_id(pars, the_un_id);
      sc->y = cons(sc, add_symbol_to_big_symbol_set(sc, pars), sc->y);
    }
  return_with_end_temp(sc->y);
}

static void clear_all_optimizations(s7_scheme *sc, s7_pointer p)
{
  if (is_unquoted_pair(sc, p))
    {
      if ((is_optimized(p)) &&
 	  (((optimize_op(p) >= First_Unhoppable_Op) ||  /* avoid clearing hop ops, fx_function and op_unknown* need to be cleared */
	    (!op_has_hop(p)))))
	{
	  clear_optimized(p);     /* includes T_Syntactic */
	  clear_optimize_op(p);
	}
      clear_all_optimizations(sc, cdr(p));
      clear_all_optimizations(sc, car(p));
    }
}

static s7_pointer add_trace(s7_scheme *sc, s7_pointer code)
{
  if ((is_pair(car(code))) && (caar(code) == sc->trace_in_symbol))
    return(code);
  return(cons_unchecked(sc, list_2(sc, sc->trace_in_symbol, list_1(sc, sc->curlet_symbol)), code));
}

static s7_pointer add_profile(s7_scheme *sc, s7_pointer code)
{
  s7_pointer result;
  if ((is_pair(car(code))) && (caar(code) == sc->profile_in_symbol))
    return(code);
  result = cons_unchecked(sc, list_3(sc, sc->profile_in_symbol, make_integer_unchecked(sc, sc->profile_position), list_1(sc, sc->curlet_symbol)), code);
  sc->profile_position++;
  set_unsafe_optimize_op(car(result), OP_PROFILE_IN);
  return(result);
}

static bool tree_has_definer(s7_scheme *sc, s7_pointer tree)
{
  for (s7_pointer p = tree; is_pair(p); p = cdr(p))
    if (tree_has_definer(sc, car(p)))
      return(true);
  return((is_symbol(tree)) && (is_definer(tree)));
}

static s7_pointer cur_op_to_caller(s7_scheme *sc, opcode_t op)
{
  switch (op)
    {
    case OP_DEFINE_MACRO:          return(sc->define_macro_symbol);
    case OP_DEFINE_MACRO_STAR:     return(sc->define_macro_star_symbol);
    case OP_DEFINE_BACRO:          return(sc->define_bacro_symbol);
    case OP_DEFINE_BACRO_STAR:     return(sc->define_bacro_star_symbol);
    case OP_DEFINE_EXPANSION:      return(sc->define_expansion_symbol);
    case OP_DEFINE_EXPANSION_STAR: return(sc->define_expansion_star_symbol);
    case OP_MACRO:                 return(sc->macro_symbol);
    case OP_MACRO_STAR:            return(sc->macro_star_symbol);
    case OP_BACRO:                 return(sc->bacro_symbol);
    case OP_BACRO_STAR:            return(sc->bacro_star_symbol);
    default:
#if s7_Debugging
      fprintf(stderr, "%s[%d]: %s?\n", __func__, __LINE__, op_names[op]);
#endif
      break;
    }
  return(NULL);
}

typedef enum {Opt_Bad, Opt_Ok, Opt_Oops} opt_t;
static opt_t optimize(s7_scheme *sc, s7_pointer code, int32_t hop, s7_pointer let_or_list);

static s7_pointer make_macro(s7_scheme *sc, opcode_t op, bool named)
{
  s7_pointer mac, body, mac_name = NULL;
  s7_uint typ;
  if (Show_Eval_Ops) fprintf(stderr, "  %s[%d]: %d, %s\n", __func__, __LINE__, named, display_truncated(sc->code));
  switch (op)
    {
    case OP_DEFINE_MACRO:      case OP_MACRO:      typ = T_MACRO;      break;
    case OP_DEFINE_MACRO_STAR: case OP_MACRO_STAR: typ = T_MACRO_STAR; break;
    case OP_DEFINE_BACRO:      case OP_BACRO:      typ = T_BACRO;      break;
    case OP_DEFINE_BACRO_STAR: case OP_BACRO_STAR: typ = T_BACRO_STAR; break;
    case OP_DEFINE_EXPANSION:      typ = T_MACRO | ((sc->curlet != sc->rootlet) ? 0 : T_Expansion); break; /* local expansions are just normal macros */
    case OP_DEFINE_EXPANSION_STAR: typ = T_MACRO_STAR | ((sc->curlet != sc->rootlet) ? 0 : T_Expansion); break;
    default:
#if s7_Debugging
      fprintf(stderr, "%s[%d]: got %s\n", __func__, __LINE__, op_names[op]);
#endif
      typ = T_MACRO;
      break;
    }
  new_cell(sc, mac, typ | T_Dont_Eval_Args);
  closure_set_pars(mac, (named) ? cdar(sc->code) : car(sc->code));
  body = cdr(sc->code);
  closure_set_body(mac, body);
  closure_set_setter(mac, sc->F);
  closure_set_let(mac, sc->curlet);
  closure_set_arity(mac, Closure_Arity_Not_Set);
  sc->capture_let_counter++;
  gc_protect_via_stack(sc, mac);

  if (named)
    {
      s7_pointer mac_slot;
      mac_name = caar(sc->code);
      if (((op == OP_DEFINE_EXPANSION) || (op == OP_DEFINE_EXPANSION_STAR)) &&
	  (sc->curlet == sc->rootlet))
	set_full_type(mac_name, T_Expansion | T_SYMBOL | (full_type(mac_name) & T_Unheap));

      /* symbol? macro name has already been checked, find name in let, and define it */
      mac_slot = symbol_to_local_slot(sc, mac_name, sc->curlet);

      if (is_slot(mac_slot))
	{
	  if (is_immutable_slot(mac_slot))
	    immutable_object_error_nr(sc, set_elist_3(sc, wrap_string(sc, "can't ~S ~S; it is immutable", 28), cur_op_to_caller(sc, op), mac_name));

	  if ((sc->curlet == sc->nil) && (!in_rootlet(mac_slot)))
	    add_slot_to_rootlet(sc, mac_slot);
	  slot_set_value_with_hook(mac_slot, mac);
	}
      else s7_make_slot(sc, sc->curlet, mac_name, mac); /* was current but we've checked immutable already */
      if (tree_has_definer(sc, body))
	set_is_definer(mac_name);                       /* (list-values 'define ...) t101-13 */
    }
  clear_big_symbol_set(sc);
  if ((!is_either_bacro(mac)) &&
      (optimize(sc, body, 1, collect_parameters(sc, closure_pars(mac), sc->nil)) == Opt_Oops))
    clear_all_optimizations(sc, body);
  clear_big_symbol_set(sc);

  if (sc->debug > 1) /* no profile here */
    closure_set_body(mac, add_trace(sc, body));

  unstack_gc_protect(sc);
  if (named)
    {
      set_pair_macro(closure_body(mac), mac_name);
      set_has_pair_macro(mac);
      if (has_location(car(sc->code)))
	{
	  pair_set_location(closure_body(mac), pair_location(car(sc->code)));
	  set_has_location(closure_body(mac));
	}}
  /* passed to maclet in apply_macro et al, copied in copy_closure */

  /* we can't add the T_Expansion bit ourselves if
   *  ((mac_name) && (!is_bacro(mac_name)) && (!is_expansion(mac_name)) && (sc->curlet == sc->rootlet) && (is_global(mac_name)))
   * because the user might reuse mac_name locally later, and our hidden expansion setting will cause the s7 reader to try to
   * treat that reuse as a call of the original macro.
   */
  return(mac);
}

static s7_pointer make_closure_unchecked(s7_scheme *sc, s7_pointer args, s7_pointer code, s7_uint type, int32_t arity)
{
  s7_pointer new_func;
  new_cell_unchecked(sc, new_func, (type | closure_bits(code)));
  closure_set_pars(new_func, args);
  closure_set_let(new_func, sc->curlet);
  closure_set_setter(new_func, sc->F);
  closure_set_arity(new_func, arity);
  closure_set_body(new_func, code);
  if (is_pair(cdr(code))) set_closure_has_multiform(new_func); else set_closure_has_one_form(new_func);
  sc->capture_let_counter++;
  return(new_func);
}

static inline s7_pointer make_closure_gc_checked(s7_scheme *sc, s7_pointer args, s7_pointer code, s7_uint type, int32_t arity) /* inline 100>1% tgc, 35=2% texit */
{ /* used in op_lambda_unchecked to avoid enormous call overhead if using make_closure  -- this code is repetitive but faster */
  s7_pointer new_func;
  new_cell(sc, new_func, (type | closure_bits(code)));
  closure_set_pars(new_func, args);
  closure_set_let(new_func, sc->curlet);
  closure_set_setter(new_func, sc->F);
  closure_set_arity(new_func, arity);
  closure_set_body(new_func, code);
  if (is_pair(cdr(code))) set_closure_has_multiform(new_func); else set_closure_has_one_form(new_func);
  sc->capture_let_counter++;
  return(new_func);
}

static s7_pointer make_closure(s7_scheme *sc, s7_pointer args, s7_pointer code, s7_uint type, int32_t arity)
{
  /* this is called (almost?) every time a lambda form is evaluated, or during letrec, etc */
  s7_pointer new_func;
  new_cell(sc, new_func, (type | closure_bits(code)));
  closure_set_pars(new_func, args);
  closure_set_let(new_func, sc->curlet);
  closure_set_setter(new_func, sc->F);
  closure_set_arity(new_func, arity);
  closure_set_body(new_func, code);                /* in case add_trace triggers GC, new func (x) needs some legit body for mark_closure */
  if (sc->debug_or_profile)
    {
      gc_protect_via_stack(sc, new_func);          /* GC protect func during add_trace */
      closure_set_body(new_func, (sc->debug > 1) ? add_trace(sc, code) : add_profile(sc, code));
      set_closure_has_multiform(new_func);
      unstack_gc_protect(sc);
    }
  else
    if (is_pair(cdr(code)))
      set_closure_has_multiform(new_func);
    else set_closure_has_one_form(new_func);
  sc->capture_let_counter++;
  return(new_func);
}

static int32_t closure_length(s7_scheme *sc, s7_pointer clo)
{
  /* we can't use let_length(sc, closure_let(clo)) because the closure_let(closure)
   *   changes.  So the open bit is not always on.  Besides, the fallbacks need to be for closures, not lets.
   */
  s7_pointer length_func = find_method(sc, closure_let(clo), sc->length_symbol);
  if (length_func != sc->undefined)
    return((int32_t)s7_integer(s7_apply_function(sc, length_func, set_plist_1(sc, clo))));
  /* there are cases where this should raise a wrong-type-arg error, but for now... */
  return(-1);
}

static s7_pointer cons_unchecked_with_type(s7_scheme *sc, s7_pointer p, s7_pointer a, s7_pointer b) /* (used only in copy_tree_with_type) */
{
  s7_pointer new_pair;
  new_cell_unchecked(sc, new_pair, full_type(p) & (Type_Mask | T_Immutable | T_Safe_Procedure));
  set_car(new_pair, a);
  set_cdr(new_pair, b);
  return(new_pair);
}

static s7_pointer copy_tree_with_type(s7_scheme *sc, s7_pointer tree)
{
  /* if sc->safety > No_Safety, '(1 2) is set immutable by the reader, but eval (in that safety case) calls
   *   copy_body on the incoming tree, so we have to preserve T_Immutable in that case.
   * if tree is something like (+ 1 (car '#1=(2 . #1#))), we have to see the quoted list and not copy it.
   * Before getting here, we have checked that there is room for the entire tree (in copy_body), or 8192 cells (in list_values) in the free heap.
   */
#if WITH_GCC
  #define Copy_Tree_With_Type(P) ({s7_pointer _p; _p = P; \
                                   cons_unchecked_with_type(sc, _p, (is_unquoted_pair(sc, car(_p))) ? copy_tree_with_type(sc, car(_p)) : car(_p), \
                                                                    (is_unquoted_pair(sc, cdr(_p))) ? copy_tree_with_type(sc, cdr(_p)) : cdr(_p));})
#else
  #define Copy_Tree_With_Type(P) copy_tree_with_type(sc, P)
#endif
  return(cons_unchecked_with_type(sc, tree,
				  (is_unquoted_pair(sc, car(tree))) ? Copy_Tree_With_Type(car(tree)) : car(tree),
				  (is_unquoted_pair(sc, cdr(tree))) ? Copy_Tree_With_Type(cdr(tree)) : cdr(tree)));
}

static inline s7_pointer copy_tree(s7_scheme *sc, s7_pointer tree)
{
#if WITH_GCC
  #define Copy_Tree(P) ({s7_pointer _p; _p = P; \
                         cons_unchecked(sc, (is_unquoted_pair(sc, car(_p))) ? copy_tree(sc, car(_p)) : car(_p), \
                                            (is_pair(cdr(_p))) ? copy_tree(sc, cdr(_p)) : cdr(_p));})
#else
  #define Copy_Tree(P) copy_tree(sc, P)
#endif
  return(cons_unchecked(sc,
			(is_unquoted_pair(sc, car(tree))) ? Copy_Tree(car(tree)) : car(tree),
			(is_pair(cdr(tree))) ? Copy_Tree(cdr(tree)) : cdr(tree)));
}


/* -------------------------------- tree-cyclic? -------------------------------- */
#define Tree_Not_Cyclic 0
#define Tree_Cyclic 1
#define Tree_Has_Pairs 2

static int32_t tree_is_cyclic_or_has_pairs(s7_scheme *sc, s7_pointer tree)
{
  s7_pointer fast = tree, slow = tree; /* we assume tree is a pair */
  bool has_pairs = false;
  while (true)
    {
      if (tree_is_collected(fast)) return(Tree_Cyclic);
      if ((!has_pairs) && (is_unquoted_pair(sc, car(fast)))) has_pairs = true;
      fast = cdr(fast);
      if (!is_pair(fast)) return((has_pairs) ? Tree_Has_Pairs : Tree_Not_Cyclic);

      if (tree_is_collected(fast)) return(Tree_Cyclic);
      if ((!has_pairs) && (is_unquoted_pair(sc, car(fast)))) has_pairs = true;
      fast = cdr(fast);
      if (!is_pair(fast)) return((has_pairs) ? Tree_Has_Pairs : Tree_Not_Cyclic);

      slow = cdr(slow);
      if (fast == slow) return(Tree_Cyclic);
    }
  return(Tree_Has_Pairs); /* not reached */
}

/* we can't use shared_info here because tree_is_cyclic may be called in the midst of output that depends on sc->circle_info */

static bool tree_is_cyclic_1(s7_scheme *sc, s7_pointer tree)
{
  for (s7_pointer p = tree; is_pair(p); p = cdr(p))
    {
      tree_set_collected(p);
      if (sc->tree_pointers_top == sc->tree_pointers_size)
	{
	  if (sc->tree_pointers_size == 0)
	    {
	      sc->tree_pointers_size = 8;
	      sc->tree_pointers = (s7_pointer *)Malloc(sc->tree_pointers_size * sizeof(s7_pointer));
	    }
	  else
	    {
	      sc->tree_pointers_size *= 2;
	      sc->tree_pointers = (s7_pointer *)Realloc(sc->tree_pointers, sc->tree_pointers_size * sizeof(s7_pointer));
	    }}
      sc->tree_pointers[sc->tree_pointers_top++] = p;
      if (is_unquoted_pair(sc, car(p)))
	{
	  const int32_t old_top = sc->tree_pointers_top;
	  const int32_t result = tree_is_cyclic_or_has_pairs(sc, car(p));
	  if ((result == Tree_Cyclic) || (tree_is_cyclic_1(sc, car(p))))
	    return(true);
	  for (int32_t i = old_top; i < sc->tree_pointers_top; i++)
	    tree_clear_collected(sc->tree_pointers[i]);
	  sc->tree_pointers_top = old_top;
	}}
  return(false);
}

static bool tree_is_cyclic(s7_scheme *sc, s7_pointer tree)
{
  int32_t result;
  if (!is_pair(tree)) return(false);
  result = tree_is_cyclic_or_has_pairs(sc, tree);
  if (result == Tree_Not_Cyclic) return(false);
  if (result == Tree_Cyclic) return(true);
  result = tree_is_cyclic_1(sc, tree);
  for (int32_t i = 0; i < sc->tree_pointers_top; i++)
    tree_clear_collected(sc->tree_pointers[i]);
  sc->tree_pointers_top = 0;
  return(result); /* this means Tree_Has_Pairs -> true? */
}

static s7_pointer g_tree_is_cyclic(s7_scheme *sc, s7_pointer args)
{
  #define H_tree_is_cyclic "(tree-cyclic? tree) returns #t if the tree has a cycle."
  #define Q_tree_is_cyclic sc->pl_bt
  return(make_boolean(sc, tree_is_cyclic(sc, car(args))));
}

static inline s7_int tree_len(s7_scheme *sc, s7_pointer p);

static s7_pointer copy_body(s7_scheme *sc, s7_pointer p)
{
  sc->w = p;
  if (tree_is_cyclic(sc, p)) /* don't wrap this in is_safety_checked */
    error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "copy: tree is cyclic: ~S", 24), p));
  check_free_heap_size(sc, tree_len(sc, p) * 2);
  return((sc->safety > No_Safety) ? copy_tree_with_type(sc, p) : copy_tree(sc, p));
}

static s7_pointer copy_closure(s7_scheme *sc, s7_pointer fnc)
{
  /* copy the source tree annotating (for eventual optimization), return a thing of the same type as fnc */
  s7_pointer new_fnc;
  const s7_pointer body = copy_body(sc, closure_body(fnc));
  if ((is_any_macro(fnc)) && (has_pair_macro(fnc)))
    {
      set_pair_macro(body, pair_macro(closure_body(fnc)));
      set_has_pair_macro(fnc);
    }
  new_cell(sc, new_fnc, full_type(fnc) & (~T_Collected)); /* I'm paranoid about that is_collected bit */
  closure_set_pars(new_fnc, closure_pars(fnc));
  closure_set_body(new_fnc, body);
  closure_set_setter_or_map_list(new_fnc, closure_setter_or_map_list(fnc));
  closure_set_arity(new_fnc, closure_arity(fnc));
  closure_set_let(new_fnc, closure_let(fnc));
  return(new_fnc);
}


/* -------------------------------- defined? -------------------------------- */
static s7_pointer g_is_defined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_defined "(defined? symbol (let (curlet)) only-search-local-let) returns #t if symbol has a binding (a value) in the let. \
Only the let is searched if only-search-local-let is #t."
  #define Q_is_defined s7_make_signature(sc, 4, sc->is_boolean_symbol, sc->is_symbol_symbol, has_let_signature(sc), sc->is_boolean_symbol)
  /* if the symbol has a global slot and e is unset or rootlet, this returns #t */

  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->is_defined_symbol, args, sc->type_names[T_SYMBOL], 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer let = cadr(args);
      const s7_pointer ignore_globals = (is_pair(cddr(args))) ? caddr(args) : sc->F;
      if (!is_let(let))
	{
	  const s7_pointer new_let = find_let(sc, let);  /* returns () if none */
	  if (!is_let(new_let))
	    find_let_error_nr(sc, sc->is_defined_symbol, let, new_let, 2, args);
	  if ((new_let == sc->rootlet) && (is_pair(cddr(args))) && (ignore_globals != sc->F))
	    {
	      if (ignore_globals != sc->T) /* signature claims this should be a boolean */
		return(method_or_bust(sc, ignore_globals, sc->is_defined_symbol, args, a_boolean_string, 3));
	      return(sc->F);
	    }
	  let = new_let;
	}
      /* if (is_unlet(let)) return(make_boolean(sc, initial_value_is_defined(sc, sym))); */
      /* this ^ is wrong: (with-let (unlet) (define xx 1) (list (defined? 'xx) (defined? 'xx (curlet)))) should be (#t #t) */

      if (is_keyword(sym))                       /* if no "let", is global -> #t */
	{                                        /* we're treating :x as 'x outside rootlet, but consider all keywords defined (as themselves) in rootlet? */
	  if (let == sc->rootlet) return(sc->T); /* (defined? x (rootlet)) where x value is a keyword */
	  sym = keyword_symbol(sym);             /* (defined? :print-length *s7*) */
	}
      if (let == sc->starlet)
	return(make_boolean(sc, starlet_symbol_id(sym) != sl_No_Field));
      if (!is_boolean(ignore_globals))
	return(method_or_bust(sc, ignore_globals, sc->is_defined_symbol, args, a_boolean_string, 3));
      if (let == sc->rootlet) /* we checked (let? let) above */
	{
	  if (ignore_globals == sc->F)
	    return(make_boolean(sc, is_slot(global_slot(sym)))); /* new_symbol and gensym initialize global_slot to #<undefined> */
	  return(sc->F);
	}
      if (ignore_globals == sc->T)
	return(make_boolean(sc, is_slot(symbol_to_local_slot(sc, sym, T_Let(let)))));
      return(make_boolean(sc, is_slot(lookup_slot_from(sym, T_Let(let)))));
    }
  return((is_defined_global(sym)) ? sc->T : make_boolean(sc, is_bound_symbol(sc, sym))); /* is_bound_symbol is (s7_slot() != sc->undefined) */
}

static s7_pointer g_is_defined_in_unlet(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    wrong_type_error_nr(sc, sc->is_defined_symbol, 1, car(args), a_symbol_string);
  return(make_boolean(sc, initial_value_is_defined(sc, sym)));
}

static s7_pointer g_is_defined_in_rootlet(s7_scheme *sc, s7_pointer args) /* aimed at lint.scm */
{
  /* (defined? bigi1 (rootlet)) can be optimized to opt_p_call_sf */
  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    wrong_type_error_nr(sc, sc->is_defined_symbol, 1, sym, a_symbol_string);
  return(make_boolean(sc, (is_slot(global_slot(sym))) && (global_value(sym) != sc->undefined)));
}

static s7_pointer is_defined_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  if (args == 2)
    {
      const s7_pointer arg2 = caddr(expr);
      if ((is_pair(arg2)) && (is_null(cdr(arg2))))
	{
	  if (car(arg2) == sc->rootlet_symbol)
	    return(sc->is_defined_in_rootlet);
	  if (car(arg2) == sc->unlet_symbol)
	    {
	      set_fn_direct(arg2, g_unlet_disabled);
	      return(sc->is_defined_in_unlet);
	    }}}
  return(func);
}

bool s7_is_defined(s7_scheme *sc, const char *name)
{
  s7_pointer symbol = s7_symbol_table_find_name(sc, name);
  if (!symbol) return(false);
  return(is_bound_symbol(sc, symbol));
}

static bool is_defined_b_7p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym)) return(method_or_bust(sc, sym, sc->is_defined_symbol, set_plist_1(sc, sym), sc->type_names[T_SYMBOL], 1) != sc->F);
  return(is_bound_symbol(sc, sym));
}

static bool is_defined_b_7pp(s7_scheme *sc, s7_pointer sym, s7_pointer let) {return(g_is_defined(sc, set_plist_2(sc, sym, let)) != sc->F);}


void s7_define(s7_scheme *sc, s7_pointer let, s7_pointer symbol, s7_pointer value) /* assumes let is a let */
{
  s7_pointer slot;
  if (T_Let(let) == sc->rootlet) let = sc->shadow_rootlet; /* if symbol is a gensym should we issue a warning? */
  slot = symbol_to_local_slot(sc, symbol, let);            /* x can be #<undefined> */
  if (is_slot(slot))
    slot_set_value_with_hook(slot, value);
  else
    {
      s7_make_slot(sc, let, symbol, value); /* I think this means C code can override "constant" defs */
      /* if let is rootlet, s7_make_slot makes a semipermanent_slot */
      if ((let == sc->shadow_rootlet) &&
	  (!is_slot(global_slot(symbol))))
	set_global_slot(symbol, local_slot(symbol));
    }
}

s7_pointer s7_define_variable(s7_scheme *sc, const char *name, s7_pointer value)
{
  s7_pointer sym = make_symbol_with_strlen(sc, name);
  s7_define(sc, sc->rootlet, sym, value);
  return(sym);
}

s7_pointer s7_define_variable_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym = s7_define_variable(sc, name, value);
  symbol_set_has_help(sym);
  symbol_set_help(sym, copy_string(help));
  add_saved_pointer(sc, symbol_help(sym));
  return(sym);
}

s7_pointer s7_define_constant_with_environment(s7_scheme *sc, s7_pointer let, const char *name, s7_pointer value)
{
  const s7_pointer sym = make_symbol_with_strlen(sc, name);
  s7_define(sc, T_Let(let), sym, value);
  set_immutable(sym);
  set_possibly_constant(sym);
  set_immutable(global_slot(sym)); /* might also be #<undefined> */
  set_immutable_slot(local_slot(sym));
  return(sym);
}

s7_pointer s7_define_constant(s7_scheme *sc, const char *name, s7_pointer value)
{
  return(s7_define_constant_with_environment(sc, sc->rootlet, name, value));
}

/* (define (func a) (let ((cvar (+ a 1))) cvar)) (define-constant cvar 23) (func 1) -> ;can't bind an immutable object: cvar
 * (let ((aaa 1)) (define-constant aaa 32) (set! aaa 3)) -> set!: can't alter immutable object: aaa
 */

s7_pointer s7_define_constant_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym = s7_define_constant(sc, name, value);
  symbol_set_has_help(sym);
  symbol_set_help(sym, copy_string(help));
  add_saved_pointer(sc, symbol_help(sym));
  return(value); /* inconsistent with variable above, but consistent with define_function? */
}


/* -------------------------------- keyword? -------------------------------- */
bool s7_is_keyword(s7_pointer obj) {return(is_symbol_and_keyword(obj));}

static s7_pointer g_is_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_is_keyword "(keyword? obj) returns #t if obj is a keyword, (keyword? :rest) -> #t"
  #define Q_is_keyword sc->pl_bt
  check_boolean_method(sc, is_symbol_and_keyword, sc->is_keyword_symbol, args);
}


/* -------------------------------- string->keyword -------------------------------- */
s7_pointer s7_make_keyword(s7_scheme *sc, const char *key)
{
  const size_t slen = (size_t)safe_strlen(key);
  block_t *b = inline_mallocate(sc, slen + 2);
  char *name = (char *)block_data(b);
  name[0] = ':';
  memcpy((void *)(name + 1), (const void *)key, slen);
  name[slen + 1] = '\0';
  {
    s7_pointer sym = inline_make_symbol(sc, name, slen + 1); /* keyword slot etc taken care of here (in new_symbol actually) */
    liberate(sc, b);
    return(sym);
  }
}

static s7_pointer g_string_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_keyword "(string->keyword str) prepends ':' to str and defines that as a keyword"
  #define Q_string_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_string_symbol)

  const s7_pointer str = car(args);
  if (!is_string(str))
    return(sole_arg_method_or_bust(sc, str, sc->string_to_keyword_symbol, args, sc->type_names[T_STRING]));
  if ((string_length(str) == 0) ||
      (string_value(str)[0] == '\0'))
    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "string->keyword wants a non-null string: ~S", 43), str));
  return(s7_make_keyword(sc, string_value(str)));
}


/* -------------------------------- keyword->symbol -------------------------------- */
static s7_pointer g_keyword_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_keyword_to_symbol "(keyword->symbol key) returns a symbol with the same name as key but no prepended or appended colon"
  #define Q_keyword_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_keyword_symbol)
  s7_pointer sym = car(args);
  if (!is_symbol_and_keyword(sym))
    return(method_or_bust_p(sc, sym, sc->keyword_to_symbol_symbol, wrap_string(sc, "a keyword", 9)));
  do {sym = keyword_symbol(sym);} while (is_keyword(sym)); /* loop for: (keyword->symbol ::hi) -> 'hi, : is not a keyword */
  return(sym);
}

s7_pointer s7_keyword_to_symbol(s7_scheme *sc, s7_pointer key) {return(keyword_symbol(key));}


/* -------------------------------- symbol->keyword -------------------------------- */
#define symbol_to_keyword(Sc, Sym) s7_make_keyword(Sc, symbol_name(Sym))

static s7_pointer g_symbol_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_keyword "(symbol->keyword sym) returns a keyword with the same name as sym, but with a colon prepended"
  #define Q_symbol_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_symbol_symbol)
  s7_pointer sym = car(args);
  if (!is_symbol(sym))
    return(sole_arg_method_or_bust(sc, sym, sc->symbol_to_keyword_symbol, args, sc->type_names[T_SYMBOL]));
  if (is_keyword(sym)) return(sym);
  return(symbol_to_keyword(sc, sym));
}


/* -------------------------------- c-pointer? -------------------------------- */
bool s7_is_c_pointer(s7_pointer arg) {return(is_c_pointer(arg));}

bool s7_is_c_pointer_of_type(s7_pointer arg, s7_pointer type) {return((is_c_pointer(arg)) && (c_pointer_type(arg) == type));}

static s7_pointer g_is_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_c_pointer "(c-pointer? obj type) returns #t if obj is a C pointer being held in s7.  \
If type is given, the c_pointer's type is also checked."
  #define Q_is_c_pointer s7_make_signature(sc, 3, sc->is_boolean_symbol, sc->T, sc->T)

  s7_pointer obj = car(args);
  if (is_c_pointer(obj))
    return((is_pair(cdr(args))) ? make_boolean(sc, c_pointer_type(obj) == cadr(args)) : sc->T);
  if (!has_active_methods(sc, obj)) return(sc->F);
  return(apply_boolean_method(sc, obj, sc->is_c_pointer_symbol));
}


/* -------------------------------- c-pointer -------------------------------- */
void *s7_c_pointer(s7_pointer p) {return(c_pointer(p));}

void *s7_c_pointer_with_type(s7_scheme *sc, s7_pointer cptr, s7_pointer expected_type, const char *caller, s7_int argnum)
{
  if (!is_c_pointer(cptr))
    wrong_type_error_nr(sc, wrap_string(sc, caller, safe_strlen(caller)), argnum, cptr, sc->type_names[T_C_POINTER]);
  if ((c_pointer(cptr) != NULL) &&
      (c_pointer_type(cptr) != expected_type))
    error_nr(sc, sc->wrong_type_arg_symbol,
	     (argnum == 0) ?
	     set_elist_4(sc, wrap_string(sc, "~S argument is a pointer of type ~S, but expected ~S", 52),
			 wrap_string(sc, caller, safe_strlen(caller)), c_pointer_type(cptr), expected_type) :
	     set_elist_5(sc, wrap_string(sc, "~S ~:D argument got a pointer of type ~S, but expected ~S", 57),
			 wrap_string(sc, caller, safe_strlen(caller)),
			 wrap_integer(sc, argnum), c_pointer_type(cptr), expected_type));
  return(c_pointer(cptr));
}

s7_pointer s7_make_c_pointer_with_type(s7_scheme *sc, void *ptr, s7_pointer type, s7_pointer info)
{
  s7_pointer new_cptr;
  new_cell(sc, new_cptr, T_C_POINTER);
  c_pointer(new_cptr) = ptr;
  c_pointer_type(new_cptr) = type;
  c_pointer_info(new_cptr) = info;
  c_pointer_weak1(new_cptr) = sc->F;
  c_pointer_weak2(new_cptr) = sc->F;
  return(new_cptr);
}

s7_pointer s7_make_c_pointer(s7_scheme *sc, void *ptr) {return(s7_make_c_pointer_with_type(sc, ptr, sc->F, sc->F));}

#define Num_C_Pointer_Wrappers 16 /* need at least 9 for gsl */

s7_pointer s7_make_c_pointer_wrapper_with_type(s7_scheme *sc, void *ptr, s7_pointer type, s7_pointer info) /* for FFI, see libgsl_s7.c */
{
  s7_pointer new_cptr = car(sc->c_pointer_wrappers);
#if s7_Debugging
  if ((full_type(new_cptr) & (~T_Gc_Mark)) != (T_C_POINTER | T_Unheap)) fprintf(stderr, "%s[%d]: %s\n", __func__, __LINE__, describe_type_bits(sc, new_cptr));
  sc->c_pointer_wrapper_allocs++;
#endif
  sc->c_pointer_wrappers = cdr(sc->c_pointer_wrappers);
  c_pointer(new_cptr) = ptr;
  c_pointer_type(new_cptr) = type;
  c_pointer_info(new_cptr) = info;
  c_pointer_weak1(new_cptr) = sc->F;
  c_pointer_weak2(new_cptr) = sc->F;
  return(new_cptr);
}

static s7_pointer g_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer "(c-pointer int type info weak1 weak2) returns a c-pointer object. The type and info args are optional, defaulting to #f."
  #define Q_c_pointer s7_make_circular_signature(sc, 2, 3, sc->is_c_pointer_symbol, sc->is_integer_symbol, sc->T)

  const s7_pointer ptr_as_int = car(args);
  s7_pointer type = sc->F, info = sc->F, weak1 = sc->F, weak2 = sc->F;
  intptr_t cptr;

  if (!s7_is_integer(ptr_as_int))
    return(method_or_bust(sc, ptr_as_int, sc->c_pointer_symbol, args, sc->type_names[T_INTEGER], 1));
  cptr = (intptr_t)s7_integer_clamped_if_gmp(sc, ptr_as_int);     /* (c-pointer (bignum "1234")) */
  args = cdr(args);
  if (is_pair(args))
    {
      type = car(args);
      args = cdr(args);
      if (is_pair(args))
	{
	  info = car(args);
	  args = cdr(args);
	  if (is_pair(args))
	    {
	      weak1 = car(args);
	      args = cdr(args);
	      if (is_pair(args))
		weak2 = car(args);
	    }}}
  {
    s7_pointer cp = s7_make_c_pointer_with_type(sc, (void *)cptr, type, info);
    c_pointer_set_weak1(cp, weak1);
    c_pointer_set_weak2(cp, weak2);
    if ((weak1 != sc->F) || (weak2 != sc->F))
      add_weak_ref(sc, cp);
    return(cp);
  }
}


/* -------------------------------- c-pointer-info -------------------------------- */
static s7_pointer c_pointer_info_p_p(s7_scheme *sc, s7_pointer cptr)
{
  if (!is_c_pointer(cptr))
    return(method_or_bust_p(sc, cptr, sc->c_pointer_info_symbol, sc->type_names[T_C_POINTER]));
  return(c_pointer_info(cptr));
}

static s7_pointer g_c_pointer_info(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_info "(c-pointer-info obj) returns the c-pointer info field"
  #define Q_c_pointer_info s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_info_p_p(sc, car(args)));
}


/* -------------------------------- c-pointer-type -------------------------------- */
static s7_pointer method_or_bust_lp(s7_scheme *sc, s7_pointer obj, s7_pointer method, uint8_t typ)
{ /* weird -- overhead goes berserk in callgrind if using the simpler method_or_bust_p! */
  if (!has_active_methods(sc, obj))
    wrong_type_error_nr(sc, method, 1, obj, sc->type_names[typ]);
  return(find_and_apply_method(sc, obj, method, set_plist_1(sc, obj)));
}

s7_pointer s7_c_pointer_type(s7_pointer cptr) {return((is_c_pointer(cptr)) ? c_pointer_type(cptr) : NULL);}

static s7_pointer c_pointer_type_p_p(s7_scheme *sc, s7_pointer cptr)
{
  return((is_c_pointer(cptr)) ? c_pointer_type(cptr) : method_or_bust_lp(sc, cptr, sc->c_pointer_type_symbol, T_C_POINTER));
}

static s7_pointer g_c_pointer_type(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_type "(c-pointer-type obj) returns the c-pointer type field"
  #define Q_c_pointer_type s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_type_p_p(sc, car(args)));
}


/* -------------------------------- c-pointer-weak1/2 -------------------------------- */
static s7_pointer c_pointer_weak1_p_p(s7_scheme *sc, s7_pointer cptr)
{
  return((is_c_pointer(cptr)) ? c_pointer_weak1(cptr) : method_or_bust_lp(sc, cptr, sc->c_pointer_weak1_symbol, T_C_POINTER));
}

static s7_pointer g_c_pointer_weak1(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_weak1 "(c-pointer-weak1 obj) returns the c-pointer weak1 field"
  #define Q_c_pointer_weak1 s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_weak1_p_p(sc, car(args)));
}

static s7_pointer c_pointer_weak2_p_p(s7_scheme *sc, s7_pointer cptr)
{
  return((is_c_pointer(cptr)) ? c_pointer_weak2(cptr) : method_or_bust_lp(sc, cptr, sc->c_pointer_weak2_symbol, T_C_POINTER));
}

static s7_pointer g_c_pointer_weak2(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_weak2 "(c-pointer-weak2 obj) returns the c-pointer weak2 field"
  #define Q_c_pointer_weak2 s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_weak2_p_p(sc, car(args)));
}


/* -------------------------------- c-pointer->list -------------------------------- */
static s7_pointer g_c_pointer_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_to_list "(c-pointer->list obj) returns the c-pointer data as (list pointer-as-int type info)"
  #define Q_c_pointer_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_c_pointer_symbol)

  s7_pointer cptr = car(args);
  if (!is_c_pointer(cptr))
    return(method_or_bust(sc, cptr, sc->c_pointer_to_list_symbol, args, sc->type_names[T_C_POINTER], 1));
  return(list_3(sc, make_integer(sc, (s7_int)((intptr_t)c_pointer(cptr))), c_pointer_type(cptr), c_pointer_info(cptr)));
}


/* -------------------------------- continuations and gotos -------------------------------- */

/* ----------------------- continuation? -------------------------------- */
static s7_pointer g_is_continuation(s7_scheme *sc, s7_pointer args)
{
  #define H_is_continuation "(continuation? obj) returns #t if obj is a continuation"
  #define Q_is_continuation sc->pl_bt
  check_boolean_method(sc, is_continuation, sc->is_continuation_symbol, args);
  /* is this the right thing?  It returns #f for call-with-exit ("goto") because
   *   that form of continuation can't continue (via a jump back to its context).
   */
}

static bool is_continuation_b_p(s7_pointer p) {return(is_continuation(p));}

#if s7_Debugging
static s7_pointer check_wrap_return(s7_pointer lst)
{
  for (s7_pointer fast = lst, slow = lst; is_pair(fast); slow = cdr(slow), fast = cdr(fast))
    {
      if (is_matched_pair(fast)) fprintf(stderr, "%s[%d]: matched_pair not cleared\n", __func__, __LINE__);
      fast = cdr(fast);
      if (!is_pair(fast)) return(lst);
      if (fast == slow) return(lst);
      if (is_matched_pair(fast)) fprintf(stderr, "%s[%d]: matched_pair not cleared\n", __func__, __LINE__);
    }
  return(lst);
}
#endif

static s7_pointer copy_any_list(s7_scheme *sc, s7_pointer a)
{
  s7_pointer slow = cdr(a);
  s7_pointer fast = slow;
  s7_pointer p;
#if s7_Debugging
  #define wrap_return(W) do {fast = W; W = sc->unused; end_temp(sc->y); return(check_wrap_return(fast));} while (0)
#else
  #define wrap_return(W) do {fast = W; W = sc->unused; end_temp(sc->y); return(fast);} while (0)
#endif
  begin_temp(sc->y, a); /* gc_protect_via_stack doesn't work here because we're called in copy_stack, I think (trouble is in call/cc stuff) */
  sc->w = list_1(sc, car(a));
  p = sc->w;
  while (true)
    {
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    wrap_return(sc->w);
	  set_cdr(p, fast);
	  wrap_return(sc->w);
	}

      set_cdr(p, list_1(sc, car(fast)));
      p = cdr(p);
      fast = cdr(fast);
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    wrap_return(sc->w);
	  set_cdr(p, fast);
	  wrap_return(sc->w);
	}
      /* if unrolled further, it's a lot slower? */
      set_cdr(p, list_1_unchecked(sc, car(fast)));
      p = cdr(p);
      fast = cdr(fast);
      slow = cdr(slow);
      if (fast == slow)
	{
	  /* try to preserve the original cyclic structure */
	  s7_pointer p1, f1, p2, f2;
	  set_match_pair(a);
	  for (p1 = sc->w, f1 = a; !(is_matched_pair(cdr(f1))); f1 = cdr(f1), p1 = cdr(p1))
	    set_match_pair(f1);
	  for (p2 = sc->w, f2 = a; cdr(f1) != f2; f2 = cdr(f2), p2 = cdr(p2))
	    clear_match_pair(f2);
	  for (f1 = f2; is_pair(f1); f1 = cdr(f1), f2 = cdr(f2))
	    {
	      clear_match_pair(f1);
	      f1 = cdr(f1);
	      clear_match_pair(f1);
	      if (f1 == f2) break;
	    }
	  clear_match_pair(a);
	  if (is_null(p1))
	    set_cdr(p2, p2);
	  else set_cdr(p1, p2);
	  wrap_return(sc->w);
	}}
  wrap_return(sc->w);
}

static s7_pointer copy_counter(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer nobj;
  new_cell(sc, nobj, T_COUNTER);
  counter_set_result(nobj, counter_result(obj));
  counter_set_list(nobj, counter_list(obj));
  counter_set_capture(nobj, counter_capture(obj));
  counter_set_let(nobj, counter_let(obj));
  counter_set_slots(nobj, counter_slots(obj));
  return(nobj);
}

static void stack_list_set_immutable(s7_pointer pold, s7_pointer pnew)
{
  for (s7_pointer p1 = pold, p2 = pnew, slow = pold; is_pair(p2); p1 = cdr(p1), p2 = cdr(p2))
    {
      if (is_immutable(p1)) set_immutable_pair(p2);
      if (is_pair(cdr(p1)))
	{
	  p1 = cdr(p1);
	  p2 = cdr(p2);
	  if (is_immutable(p1)) set_immutable_pair(p2);
	  if (p1 == slow) break;
	  slow = cdr(slow);
	}}
}

static s7_pointer copy_stack(s7_scheme *sc, s7_pointer new_v, s7_pointer old_v, s7_int top)
{
  bool has_pairs = false;
  s7_pointer *nv = stack_elements(new_v);
  s7_pointer *ov = stack_elements(old_v);
  memcpy((void *)nv, (void *)ov, top * sizeof(s7_pointer));
  stack_clear_flags(new_v);

  s7_gc_on(sc, false);
  if (stack_has_counters(old_v))
    {
      for (s7_int i = 2; i < top; i += 4)
	{
	  const s7_pointer p = ov[i];               /* args */
	  /* if op_gc_protect, any ov[i] (except op) can be a list, but it isn't the arglist, so it seems to be safe */
	  if (is_pair(p))                           /* args need not be a list (it can be a port or #f, etc) */
	    {
	      has_pairs = true;
	      if (is_null(cdr(p)))
		nv[i] = cons_unchecked(sc, car(p), sc->nil); /* GC is off -- could uncheck list_2 et al also */
	      else
		if ((is_pair(cdr(p))) && (is_null(cddr(p))))
		  nv[i] = list_2_unchecked(sc, car(p), cadr(p));
		else nv[i] = copy_any_list(sc, p);  /* args (copy is needed -- see s7test.scm) */
	      /* if op=eval_args4 for example, this has to be a proper list, and in many cases it doesn't need to be copied */
	      stack_list_set_immutable(p, nv[i]);
	    }
	  /* lst can be dotted or circular here.  The circular list only happens in a case like:
	   *    (dynamic-wind (lambda () (eq? (let ((lst (cons 1 2))) (set-cdr! lst lst) lst) (call/cc (lambda (k) k)))) (lambda () #f) (lambda () #f))
	   *    proper_list_reverse_in_place(sc->args) is one reason we need to copy
	   */
	  else
	    if (is_counter(p))                     /* these can only occur in this context (not in a list etc) */
	      {
		stack_set_has_counters(new_v);
		nv[i] = copy_counter(sc, p);
	      }}}
  else
    for (s7_int i = 2; i < top; i += 4)
      if (is_pair(ov[i]))
	{
	  const s7_pointer p = ov[i];
	  has_pairs = true;
	  if (is_null(cdr(p)))
	    nv[i] = cons_unchecked(sc, car(p), sc->nil);
	  else
	    if ((is_pair(cdr(p))) && (is_null(cddr(p))))
	      nv[i] = list_2_unchecked(sc, car(p), cadr(p));
	    else nv[i] = copy_any_list(sc, p);  /* args (copy is needed -- see s7test.scm) */
	  stack_list_set_immutable(p, nv[i]);
	}
  if (has_pairs) stack_set_has_pairs(new_v);
  s7_gc_on(sc, true);
  return(new_v);
}

static s7_pointer copy_op_stack(s7_scheme *sc)
{
  int32_t len = (int32_t)(sc->op_stack_now - sc->op_stack);
  s7_pointer nv = make_simple_vector(sc, len); /* not sc->op_stack_size */
  if (len > 0)
    {
      s7_pointer *src = sc->op_stack;
      s7_pointer *dst = (s7_pointer *)vector_elements(nv);
      for (int32_t i = len; i > 0; i--) *dst++ = *src++;
    }
  return(nv);
}

/* -------------------------------- with-baffle -------------------------------- */
/* (with-baffle . body) calls body guaranteeing that there can be no jumps into the
 *    middle of it from outside -- no outer evaluation of a continuation can jump across this
 *    barrier:  The flip-side of call-with-exit.
 */

static bool find_baffle(s7_scheme *sc, s7_int key)
{
  /* search backwards through sc->curlet for baffle_let with (continuation_)key as its baffle_key value */
  if (sc->baffle_ctr > 0)
    for (s7_pointer let = sc->curlet; let; let = let_outlet(let))
      if ((is_baffle_let(let)) &&
	  (let_baffle_key(let) == key))
	return(true);
  return(false);
}

#define Not_Baffled -1

static s7_int find_any_baffle(s7_scheme *sc)
{
  /* search backwards through sc->curlet for any sc->baffle_symbol -- called by s7_make_continuation to set continuation_key */
  if (sc->baffle_ctr > 0)
    for (s7_pointer let = sc->curlet; let; let = let_outlet(let))
      if (is_baffle_let(let))
	return(let_baffle_key(let));
  return(Not_Baffled);
}

static void check_with_baffle(s7_scheme *sc)
{
  if (!s7_is_proper_list(sc, sc->code))
    syntax_error_nr(sc, "with-baffle: unexpected dot? ~A", 31, sc->code);
  pair_set_syntax_op(sc->code, OP_WITH_BAFFLE_UNCHECKED);
}

static bool op_with_baffle_unchecked(s7_scheme *sc)
{
  sc->code = cdr(sc->code);
  if (is_null(sc->code))
    {
      sc->value = sc->nil;
      return(true);
    }
  set_curlet(sc, make_let(sc, sc->curlet));
  set_baffle_let(sc->curlet);
  let_set_baffle_key(sc->curlet, sc->baffle_ctr++);
  return(false);
}


/* -------------------------------- call/cc -------------------------------- */
static void make_room_for_cc_stack(s7_scheme *sc)
{
  if ((s7_int)(sc->free_heap_top - sc->free_heap) < (s7_int)(sc->heap_size / 32)) /* we probably never need this much space (8 becomes enormous, 512 seems ok) */
    {                                                                               /*  but this doesn't seem to make much difference in timings */
      call_gc(sc);
      if ((s7_int)(sc->free_heap_top - sc->free_heap) < (s7_int)(sc->heap_size / 32))
	resize_heap(sc);
    }
}

s7_pointer s7_make_continuation(s7_scheme *sc)
{
  /* precede this with make_room_for_cc_stack(sc); */
  const s7_int loc = stack_top(sc);
  const s7_pointer stack = make_simple_vector(sc, loc);
  s7_pointer new_cc;
  block_t *block;

  set_full_type(stack, T_STACK);
  temp_stack_top(stack) = loc;
  begin_temp(sc->x, stack);
  copy_stack(sc, stack, sc->stack, loc);

  new_cell(sc, new_cc, T_CONTINUATION);
  block = mallocate_block(sc);
#if s7_Debugging
  sc->blocks_mallocated[Block_List]++;
#endif
  continuation_block(new_cc) = block;
  continuation_set_stack(new_cc, stack);
  continuation_stack_size(new_cc) = vector_length(continuation_stack(new_cc));
  continuation_stack_start(new_cc) = stack_elements(continuation_stack(new_cc));
  continuation_stack_end(new_cc) = (s7_pointer *)(continuation_stack_start(new_cc) + loc);
  continuation_op_stack(new_cc) = copy_op_stack(sc);
  continuation_op_loc(new_cc) = (int32_t)(sc->op_stack_now - sc->op_stack);
  continuation_op_size(new_cc) = sc->op_stack_size;
  continuation_key(new_cc) = find_any_baffle(sc);
  continuation_name(new_cc) = sc->F;
  end_temp(sc->x);
  add_continuation(sc, new_cc);
  return(new_cc);
}

static void let_temp_done(s7_scheme *sc, s7_pointer args, s7_pointer let);
static void let_temp_unwind(s7_scheme *sc, s7_pointer slot, s7_pointer new_value);
static s7_pointer dynamic_unwind(s7_scheme *sc, s7_pointer func, s7_pointer let);

static bool check_for_dynamic_winds(s7_scheme *sc, s7_pointer cont)
{
  /* called only from call_with_current_continuation.
   *   if call/cc jumps into a dynamic-wind, the init/finish funcs are wrapped in with-baffle
   *   so they'll complain.  Otherwise we're supposed to re-run the init func before diving
   *   into the body.  Similarly for let-temporarily.  If a call/cc jumps out of a dynamic-wind
   *   body-func, we're supposed to call the finish-func.  The continuation is called at
   *   stack_top(sc); the continuation form is at continuation_stack_top(cont).
   *
   * check sc->stack for dynamic-winds we're jumping out of
   *    we need to check from the current stack top down to where the continuation stack matches the current stack??
   *    this was (i > 0), but that goes too far back; perhaps s7 should save the position of the call/cc invocation.
   *    also the two stacks can be different sizes (either can be larger)
   */
  const s7_pointer cc_stack = continuation_stack(cont);
  const s7_int cc_top = continuation_stack_top(cont);
  for (s7_int op_loc = stack_top(sc) - 1; (op_loc > 0) && ((op_loc >= cc_top) || (stack_code(sc->stack, op_loc) != stack_code(cc_stack, op_loc))); op_loc -= 4)
    {
      const opcode_t op = stack_op(sc->stack, op_loc);
      switch (op)
	{
	case OP_DYNAMIC_WIND:
	case OP_LET_TEMP_DONE:
	  {
	    const s7_pointer code = stack_code(sc->stack, op_loc);
	    s7_int s_base = 0;
	    for (s7_int j = 3; j < cc_top; j += 4)
	      if (((stack_op(cc_stack, j) == OP_DYNAMIC_WIND) ||
		   (stack_op(cc_stack, j) == OP_LET_TEMP_DONE)) &&
		  (code == stack_code(cc_stack, j)))
		{
		  s_base = op_loc;
		  break;
		}
	    if (s_base == 0)
	      {
		if (op == OP_DYNAMIC_WIND)
		  {
		    if (dynamic_wind_state(code) == Dwind_Body)
		      {
			dynamic_wind_state(code) = Dwind_Finish;
			if (dynamic_wind_out(code) != sc->F)
			  sc->value = s7_call(sc, dynamic_wind_out(code), sc->nil);
		      }}
		else let_temp_done(sc, stack_args(sc->stack, op_loc), T_Let(stack_let(sc->stack, op_loc)));
	      }}
	  break;

	case OP_DYNAMIC_UNWIND:
	  {
	    s7_pointer func = stack_code(sc->stack, op_loc);
	    s7_pointer args = stack_args(sc->stack, op_loc);
	    if ((is_pair(cdr(args))) && (is_pair(cddr(args))) && (caddr(args) == sc->T))
	      dynamic_unwind(sc, func, args);
	  }
	case OP_DYNAMIC_UNWIND_PROFILE:
	  set_stack_op(sc->stack, op_loc, OP_GC_PROTECT);
	  break;

	case OP_LET_TEMP_UNWIND:
	  let_temp_unwind(sc, stack_code(sc->stack, op_loc), stack_args(sc->stack, op_loc));
	  break;

	case OP_LET_TEMP_S7_UNWIND:
	  starlet_set_1(sc, stack_code(sc->stack, op_loc), stack_args(sc->stack, op_loc));
	  break;

	case OP_LET_TEMP_S7_OPENLETS_UNWIND:
	  sc->has_openlets = (stack_args(sc->stack, op_loc) != sc->F);
	  break;

	case OP_BARRIER:
	  if (op_loc > cc_top)                /* otherwise it's some unproblematic outer eval-string? */
	    return(false);                    /*    but what if we've already evaluated a dynamic-wind closer? */
	  break;

	case OP_DEACTIVATE_GOTO:              /* here we're jumping out of an unrelated call-with-exit block */
	  if (op_loc > cc_top)
	    call_exit_active(stack_args(sc->stack, op_loc)) = false;
	  break;

	case OP_UNWIND_INPUT:
	  if (stack_args(sc->stack, op_loc) != sc->unused)
	    set_current_input_port(sc, stack_args(sc->stack, op_loc));    /* "args" = port that we shadowed */
	  break;

	case OP_UNWIND_OUTPUT:
	  if (stack_args(sc->stack, op_loc) != sc->unused)
	    set_current_output_port(sc, stack_args(sc->stack, op_loc));   /* "args" = port that we shadowed */
	  break;

	default:
	  if ((s7_Debugging) && (op == OP_MAP_UNWIND)) fprintf(stderr, "%s[%d]: unwind %" ld64 "\n", __func__, __LINE__, sc->map_call_ctr);
	  break;
	}}

  /* check continuation-stack for dynamic-winds we're jumping into */
  for (s7_int op_loc = stack_top(sc) - 1; op_loc < cc_top; op_loc += 4)
    {
      const opcode_t op = stack_op(cc_stack, op_loc);
      if (op == OP_DYNAMIC_WIND)
	{
	  s7_pointer dw = T_Dyn(stack_code(cc_stack, op_loc));
	  if (dynamic_wind_in(dw) != sc->F)
	    sc->value = s7_call(sc, dynamic_wind_in(dw), sc->nil);
	  dynamic_wind_state(dw) = Dwind_Body;
	}
      else
	if (op == OP_DEACTIVATE_GOTO)
	  call_exit_active(stack_args(cc_stack, op_loc)) = true;
      /* not let_temp_done here! */
      /* if op == OP_LET_TEMP_DONE, we're jumping back into a let-temporarily.  MIT and Chez scheme say they remember the
       *   let-temp vars (fluid-let or parameters in their terminology) at the point of the call/cc, and restore them
       *   on re-entry; that strikes me as incoherently complex -- they've wrapped a hidden dynamic-wind around the
       *   call/cc to restore all let-temp vars!  I think let-temp here should be the same as let -- if you jump back
       *   in, nothing hidden happens. So,
       *     (let ((x #f) (cc #f))
       *       (let-temporarily ((x 1))
       *         (set! x 2) (call/cc (lambda (r) (set! cc r))) (display x) (unless (= x 2) (newline) (exit)) (set! x 3) (cc)))
       *   behaves the same (in this regard) if let-temp is replaced with let.
       */
    }
  return(true);
}

static s7_pointer splice_in_values(s7_scheme *sc, s7_pointer args);

static void call_with_current_continuation(s7_scheme *sc)
{
  s7_pointer cont = sc->code;  /* sc->args are the returned values */

  /* check for (baffle ...) blocking the current attempt to continue */
  if ((continuation_key(cont) != Not_Baffled) &&
      (!find_baffle(sc, continuation_key(cont))))
    error_nr(sc, sc->baffled_symbol,
	     (is_symbol(continuation_name(sc->code))) ?
	     set_elist_2(sc, wrap_string(sc, "continuation ~S can't jump into with-baffle", 43), continuation_name(cont)) :
	     set_elist_1(sc, wrap_string(sc, "continuation can't jump into with-baffle", 40)));

  if (check_for_dynamic_winds(sc, cont))
    {
      /* make_room_for_cc_stack(sc); */ /* 28-May-21 */
      /* we push_stack sc->code before calling an embedded eval above, so sc->code should still be cont here, etc */
      if ((stack_has_pairs(continuation_stack(cont))) ||
	  (stack_has_counters(continuation_stack(cont))))
	{
	  make_room_for_cc_stack(sc);
	  copy_stack(sc, sc->stack, continuation_stack(cont), continuation_stack_top(cont));
	}
      else
	{
	  s7_pointer *nv = stack_elements(sc->stack);
	  s7_pointer *ov = stack_elements(continuation_stack(cont));
	  memcpy((void *)nv, (void *)ov, continuation_stack_top(cont) * sizeof(s7_pointer));
	}
      /* copy_stack(sc, sc->stack, continuation_stack(cont), continuation_stack_top(cont)); */
      sc->stack_end = (s7_pointer *)(sc->stack_start + continuation_stack_top(cont));

      {
	const int32_t top = continuation_op_loc(cont);
	s7_pointer *src, *dst;
	sc->op_stack_now = (s7_pointer *)(sc->op_stack + top);
	sc->op_stack_size = continuation_op_size(cont);
	sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
	src = (s7_pointer *)vector_elements(continuation_op_stack(cont));
	dst = sc->op_stack;
	for (int32_t i = 0; i < top; i++) dst[i] = src[i];
      }
      sc->value = (is_null(sc->args)) ? sc->nil : ((is_null(cdr(sc->args))) ? car(sc->args) : splice_in_values(sc, sc->args));
    }
}

static s7_pointer g_call_cc(s7_scheme *sc, s7_pointer args)
{
  #define H_call_cc "(call-with-current-continuation (lambda (continuer) ...)) evaluates the body with continuer as a way to goto to the continuation of the body"
  #define Q_call_cc s7_make_signature(sc, 2, sc->values_symbol, sc->is_procedure_symbol)

  const s7_pointer func = car(args);         /* this is the procedure passed to call/cc */
  if (!is_t_procedure(func))                 /* this includes continuations */
    {
      if_method_exists_return_value(sc, func, sc->call_cc_symbol, args);
      if_method_exists_return_value(sc, func, sc->call_with_current_continuation_symbol, args);
      sole_arg_wrong_type_error_nr(sc, sc->call_cc_symbol, func, a_procedure_string);
    }
  if (((!is_closure(func)) ||
       (closure_arity(func) != 1)) &&
      (!s7_is_aritable(sc, func, 1)))
    error_nr(sc, sc->wrong_type_arg_symbol,
	     set_elist_2(sc, wrap_string(sc, "call/cc procedure, ~A, should take one argument", 47), func));

  make_room_for_cc_stack(sc);
  begin_temp(sc->y, s7_make_continuation(sc));
  if ((is_any_closure(func)) && (is_pair(closure_pars(func))) && (is_symbol(car(closure_pars(func)))))
    continuation_name(sc->y) = car(closure_pars(func));
  push_stack(sc, OP_APPLY, list_1_unchecked(sc, sc->y), func); /* apply func to continuation */
  end_temp(sc->y);
  return(sc->nil);
}

static void op_call_cc(s7_scheme *sc) /* OP_CALL_CC in eval via optimize_c_function_one_arg */
{
  make_room_for_cc_stack(sc);
  begin_temp(sc->y, s7_make_continuation(sc));
  continuation_name(sc->y) = caar(opt2_pair(sc->code)); /* caadadr(sc->code) */
  set_curlet(sc, inline_make_let_with_slot(sc, sc->curlet, continuation_name(sc->y), sc->y));
  end_temp(sc->y);
  sc->code = cdr(opt2_pair(sc->code)); /* cddadr(sc->code) */
}

static bool op_implicit_continuation_a(s7_scheme *sc)
{
  s7_pointer code = sc->code; /* dumb-looking code, but it's faster than the pretty version, according to callgrind */
  s7_pointer cont = lookup_checked(sc, car(code));
  if (!is_continuation(cont)) {sc->last_function = cont; return(false);}
  sc->code = cont;
  sc->args = set_plist_1(sc, fx_call(sc, cdr(code)));
  call_with_current_continuation(sc);
  return(true);
}


/* -------------------------------- call-with-exit -------------------------------- */
static void pop_input_port(s7_scheme *sc);

static void call_with_exit(s7_scheme *sc)
{
  s7_int op_loc, new_stack_top, quit = 0;

  if (!call_exit_active(sc->code))
    error_nr(sc, sc->invalid_exit_function_symbol,
	     (is_symbol(call_exit_name(sc->code))) ?
	       set_elist_2(sc, wrap_string(sc, "call-with-exit exit procedure, ~A, called outside its block", 59), call_exit_name(sc->code)) :
	       set_elist_1(sc, wrap_string(sc, "call-with-exit exit procedure called outside its block", 54)));

  call_exit_active(sc->code) = false;
  new_stack_top = call_exit_goto_loc(sc->code);
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + call_exit_op_loc(sc->code));

  /* look for dynamic-wind in the stack section that we are jumping out of */
  op_loc = stack_top(sc) - 1;
  /* op is entirely op_deactivate_goto tgc, for_each_2|3 tcase, dox_step_o texit, lots of ops s7test.scm */
  /* if (stack_op(sc->stack, op_loc) == OP_DEACTIVATE_GOTO) {call_exit_active(stack_args(sc->stack, op_loc)) = false; goto Set_Value;} saves >54 in tgc */

  do {
    switch (stack_op(sc->stack, op_loc)) /* the hit rate here is good; exiters[op] slowed us down! (see tmp) tgc/texit slower, tcase faster */
      {
      case OP_DYNAMIC_WIND:
	{
	  const s7_pointer lx = T_Dyn(stack_code(sc->stack, op_loc));
	  if (dynamic_wind_state(lx) == Dwind_Body)
	    {
	      dynamic_wind_state(lx) = Dwind_Finish;
	      if (dynamic_wind_out(lx) != sc->F)
		{
		  s7_pointer arg = (sc->args == sc->plist_1) ? car(sc->plist_1) : sc->unused;  /* might also need GC protection here */
		  /* protect the sc->args value across this call if it is sc->plist_1 -- I can't find a broken case */
		  sc->value = s7_call(sc, dynamic_wind_out(lx), sc->nil);
		  if (arg != sc->unused) set_plist_1(sc, arg);
		}}}
	break;

      case OP_DYNAMIC_UNWIND:
      case OP_DYNAMIC_UNWIND_PROFILE:
	set_stack_op(sc->stack, op_loc, OP_GC_PROTECT);
	dynamic_unwind(sc, stack_code(sc->stack, op_loc), stack_args(sc->stack, op_loc));
	break;

      case OP_EVAL_STRING:
	s7_close_input_port(sc, current_input_port(sc));
	pop_input_port(sc);
	break;

      case OP_BARRIER:                /* oops -- we almost certainly went too far */
	goto Set_Value;

      case OP_DEACTIVATE_GOTO:        /* here we're jumping into an unrelated call-with-exit block */
	call_exit_active(stack_args(sc->stack, op_loc)) = false;
	break;

      case OP_LET_TEMP_DONE:
	{
	  s7_pointer old_args = sc->args;
	  let_temp_done(sc, stack_args(sc->stack, op_loc), T_Let(stack_let(sc->stack, op_loc)));
	  sc->args = old_args;
	}
	break;

      case OP_LET_TEMP_UNWIND:
	let_temp_unwind(sc, stack_code(sc->stack, op_loc), stack_args(sc->stack, op_loc));
	break;

      case OP_LET_TEMP_S7_UNWIND:
	starlet_set_1(sc, stack_code(sc->stack, op_loc), stack_args(sc->stack, op_loc));
	break;

      case OP_LET_TEMP_S7_OPENLETS_UNWIND:
	sc->has_openlets = (stack_args(sc->stack, op_loc) != sc->F);
	break;

	/* call/cc does not close files, but I think call-with-exit should */
      case OP_GET_OUTPUT_STRING:
      case OP_UNWIND_OUTPUT:
	{
	  s7_pointer port = T_Pro(stack_code(sc->stack, op_loc));  /* "code" = port that we opened */
	  s7_close_output_port(sc, port);
	  port = stack_args(sc->stack, op_loc);                    /* "args" = port that we shadowed, if not #<unused> */
	  if (port != sc->unused)
	    set_current_output_port(sc, port);
	}
	break;

      case OP_UNWIND_INPUT:
	s7_close_input_port(sc, T_Pri(stack_code(sc->stack, op_loc))); /* "code" = port that we opened */
	if (stack_args(sc->stack, op_loc) != sc->unused)
	  set_current_input_port(sc, stack_args(sc->stack, op_loc));   /* "args" = port that we shadowed */
	break;

      case OP_EVAL_DONE: /* goto called in a method -- put off the inner eval return(s) until we clean up the stack */
	quit++;
	break;

      default:
	if ((s7_Debugging) && (stack_op(sc->stack, op_loc) == OP_MAP_UNWIND)) fprintf(stderr, "%s[%d]: unwind %" ld64 "\n", __func__, __LINE__, sc->map_call_ctr);
	break;
      }
    op_loc -= 4;
  } while (op_loc > new_stack_top);

 Set_Value:
  sc->stack_end = (s7_pointer *)(sc->stack_start + new_stack_top);

  /* the return value should have an implicit values call, just as in call/cc */
  sc->value = (is_null(sc->args)) ? sc->nil : ((is_null(cdr(sc->args))) ? car(sc->args) : splice_in_values(sc, sc->args));
  if (quit > 0)
    {
      if (sc->longjmp_ok)
	{
	  pop_stack(sc);
	  LongJmp(*(sc->Goto_Start), Call_With_Exit_Jump);
	}
      for (s7_int i = 0; i < quit; i++)
	push_stack_op_let(sc, OP_EVAL_DONE);
    }
}

static s7_pointer g_is_goto(s7_scheme *sc, s7_pointer args)
{
  #define H_is_goto "(goto? obj) returns #t if obj is a call-with-exit exit function"
  #define Q_is_goto sc->pl_bt
  return(make_boolean(sc, is_goto(car(args))));
}

static inline s7_pointer make_goto(s7_scheme *sc, s7_pointer name) /* inline for 73=1% in tgc */
{
  s7_pointer new_goto;
  new_cell(sc, new_goto, T_GOTO);
  call_exit_goto_loc(new_goto) = stack_top(sc);
  call_exit_op_loc(new_goto) = (int32_t)(sc->op_stack_now - sc->op_stack);
  call_exit_active(new_goto) = true;
  call_exit_name(new_goto) = name;
  return(new_goto);
}

static s7_pointer g_call_with_exit(s7_scheme *sc, s7_pointer args)   /* (call-with-exit (lambda (return) ...)) */
{
  #define H_call_with_exit "(call-with-exit (lambda (exiter) ...)) is call/cc without the ability to jump back into a previous computation."
  #define Q_call_with_exit s7_make_signature(sc, 2, sc->values_symbol, sc->is_procedure_symbol)

  const s7_pointer func = car(args);
  s7_pointer new_goto;
  if (is_any_closure(func)) /* lambda or lambda* */
    {
      new_goto = make_goto(sc, ((is_pair(closure_pars(func))) && (is_symbol(car(closure_pars(func))))) ? car(closure_pars(func)) : sc->F);
      push_stack(sc, OP_DEACTIVATE_GOTO, new_goto, func); /* this means call-with-exit is not tail-recursive */
      push_stack(sc, OP_APPLY, cons_unchecked(sc, new_goto, sc->nil), func);
      return(sc->nil);
    }
  /* maybe just return an error here -- these gotos as args are stupid; also an error above if closure not aritable 1 */
  if (!is_t_procedure(func))
    return(method_or_bust_p(sc, func, sc->call_with_exit_symbol, a_procedure_string));
  if (!s7_is_aritable(sc, func, 1))
    error_nr(sc, sc->wrong_type_arg_symbol,
	     set_elist_2(sc, wrap_string(sc, "call-with-exit argument should be a function of one argument: ~S", 64), func));
  if (is_continuation(func)) /* (call/cc call-with-exit) ! */
    error_nr(sc, sc->wrong_type_arg_symbol,
	     set_elist_2(sc, wrap_string(sc, "call-with-exit argument should be a normal function (not a continuation: ~S)", 76), func));
  new_goto = make_goto(sc, sc->F);
  call_exit_active(new_goto) = false;
  return((is_c_function(func)) ? c_function_call(func)(sc, set_plist_1(sc, new_goto)) : s7_apply_function_star(sc, func, set_plist_1(sc, new_goto)));
}

static inline void op_call_with_exit(s7_scheme *sc)
{
  s7_pointer args = opt2_pair(sc->code);
  s7_pointer go = make_goto(sc, caar(args));
  push_stack_no_let_no_code(sc, OP_DEACTIVATE_GOTO, go); /* was also pushing code */
  set_curlet(sc, inline_make_let_with_slot(sc, sc->curlet, caar(args), go));
  sc->code = T_Pair(cdr(args));
  /* goto Begin */
}

static void op_call_with_exit_o(s7_scheme *sc)
{
  op_call_with_exit(sc);
  sc->code = car(sc->code);
  /* goto Eval */
}

static bool op_implicit_goto(s7_scheme *sc)
{
  s7_pointer g = lookup_checked(sc, car(sc->code));
  if (!is_goto(g)) {sc->last_function = g; return(false);}
  sc->args = sc->nil;
  sc->code = g;
  call_with_exit(sc);
  return(true);
}

static bool op_implicit_goto_a(s7_scheme *sc)
{
  s7_pointer g = lookup_checked(sc, car(sc->code));
  if (!is_goto(g)) {sc->last_function = g; return(false);}
  sc->args = set_plist_1(sc, fx_call(sc, cdr(sc->code)));
  sc->code = g;
  call_with_exit(sc);
  return(true);
}


/* -------------------------------- numbers -------------------------------- */
static block_t *string_to_block(s7_scheme *sc, const char *p, s7_int len)
{
  block_t *b = inline_mallocate(sc, len + 1);
  char *bp = (char *)block_data(b);
  memcpy((void *)bp, (const void *)p, len);
  bp[len] = '\0';
  return(b);
}

static Inline s7_pointer inline_block_to_string(s7_scheme *sc, block_t *block, s7_int len)
{
  s7_pointer new_string;
  new_cell(sc, new_string, T_STRING | T_Safe_Procedure);
  string_block(new_string) = block;
  string_value(new_string) = (char *)block_data(block);
  string_length(new_string) = len;
  string_value(new_string)[len] = '\0';
  string_hash(new_string) = 0;
  add_string(sc, new_string);
  return(new_string);
}

static s7_pointer block_to_string(s7_scheme *sc, block_t *block, s7_int len) {return(inline_block_to_string(sc, block, len));}

static /* inline */ s7_pointer make_simple_ratio(s7_scheme *sc, s7_int num, s7_int den) /* no gcd needed in this case */
{
  s7_pointer new_ratio;
  if (den < 0)
    {
      if ((num == s7_Int64_Min) || (den == s7_Int64_Min))  /* assume no gcd involved */
	return(make_real(sc, (long_double)num / (long_double)den));
      if (den == -1)
	return(make_integer(sc, -num));
      new_cell(sc, new_ratio, T_RATIO);
      set_numerator(new_ratio, -num);
      set_denominator(new_ratio, -den);
    }
  else
    {
      if (den == 1)
	return(make_integer(sc, num));
      new_cell(sc, new_ratio, T_RATIO);
      set_numerator(new_ratio, num);
      set_denominator(new_ratio, den);
    }
  return(new_ratio);
}

static /* inline */ s7_pointer make_simpler_ratio(s7_scheme *sc, s7_int num, s7_int den) /* no gcd needed, and den > 1 */
{
  s7_pointer new_ratio;
  if ((s7_Debugging) && (den < 2)) fprintf(stderr, "%s[%d]: denominator: %" ld64 "/n", __func__, __LINE__, den);
  new_cell(sc, new_ratio, T_RATIO);
  set_numerator(new_ratio, num);
  set_denominator(new_ratio, den);
  return(new_ratio);
}

static inline s7_pointer make_simpler_ratio_or_integer(s7_scheme *sc, s7_int num, s7_int den) /* nom gcd needed and den > 0 (might be 1) */
{
  s7_pointer new_ratio;
  if ((s7_Debugging) && (den <= 0)) fprintf(stderr, "%s[%d]: denominator: %" ld64 "/n", __func__, __LINE__, den);
  if (den == 1)
    return(make_integer(sc, num));
  new_cell(sc, new_ratio, T_RATIO);
  set_numerator(new_ratio, num);
  set_denominator(new_ratio, den);
  return(new_ratio);
}

static bool is_zero(s7_pointer x);
static bool is_positive(s7_scheme *sc, s7_pointer x);
static bool is_negative(s7_scheme *sc, s7_pointer x);
static s7_pointer make_ratio(s7_scheme *sc, s7_int a, s7_int b);

static bool is_NaN(s7_double x) {return(x != x);}
/* callgrind says this is faster than isnan, I think (very confusing data...) */

#if defined(__sun) && defined(__SVR4)
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));} /* there's no isinf in Solaris */
#else
#if !MS_Windows
  #if __cplusplus
    #define is_inf(x) std::isinf(x)
  #else
    #define is_inf(x) isinf(x)
  #endif
#else
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));}  /* Another possibility: (x * 0) != 0 */

#if (_MSC_VER < 1700)
  /* in MS C, we need to provide inverse hyperbolic trig funcs and cbrt */
  static double asinh(double x) {return(log(x + sqrt(1.0 + x * x)));}
  static double acosh(double x) {return(log(x + sqrt(x * x - 1.0)));}
  /* perhaps less prone to numerical troubles (untested): 2.0 * log(sqrt(0.5 * (x + 1.0)) + sqrt(0.5 * (x - 1.0))) */
  static double atanh(double x) {return(log((1.0 + x) / (1.0 - x)) / 2.0);}
  static double cbrt(double x) {if (x >= 0.0) return(pow(x, 1.0 / 3.0)); return(-pow(-x, 1.0 / 3.0));}
#endif
#endif /* windows */
#endif /* not sun */


/* -------------------------------- NaN payloads -------------------------------- */
typedef union {s7_int ix; double fx;} decode_float_t;

static double nan_with_payload(s7_int payload)
{
  decode_float_t num;
  if (payload <= 0) return(Nan);
  num.fx = Nan;
  num.ix = num.ix | payload;
  return(num.fx);
}

static s7_pointer make_nan_with_payload(s7_scheme *sc, s7_int payload)
{
  return(make_real(sc, nan_with_payload(payload)));
}

static s7_pointer g_nan(s7_scheme *sc, s7_pointer args)
{
  #define H_nan "(nan (int 0)) returns a NaN with payload int"
  #define Q_nan s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_integer_symbol)
  #define Nan_Payload_Limit (1LL << 51LL) /* 53 is probably ok, (nan (- (ash 1 53) 1)): +nan.9007199254740991 -- 52 bits available? */
  s7_pointer payload;
  if (is_null(args)) return(real_NaN);    /* payload defaults to 0 */
  payload = car(args);
  if (!is_t_integer(payload))
    return(method_or_bust_p(sc, payload, sc->nan_symbol, sc->type_names[T_INTEGER]));
  if (integer(payload) < 0)
    sole_arg_out_of_range_error_nr(sc, sc->nan_symbol, set_elist_1(sc, payload), it_is_negative_string);
  if (integer(payload) >= Nan_Payload_Limit)
    sole_arg_out_of_range_error_nr(sc, sc->nan_symbol, set_elist_1(sc, payload), it_is_too_large_string);
  return(make_nan_with_payload(sc, integer(payload)));
}

static s7_int nan_payload(double x)
{
  decode_float_t num;
  num.fx = x;
  return(num.ix & 0xffffffffffff);
}

static s7_pointer g_nan_payload(s7_scheme *sc, s7_pointer args)
{
  #define H_nan_payload "(nan-payload x) returns the payload associated with the NaN x"
  #define Q_nan_payload s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  s7_pointer nan = car(args);
  if (!is_t_real(nan))
    return(method_or_bust_p(sc, nan, sc->nan_payload_symbol, sc->type_names[T_REAL]));
  if (!is_NaN(real(nan))) /* for complex case, use real-part etc (see s7test.scm) */
    sole_arg_wrong_type_error_nr(sc, sc->nan_payload_symbol, nan, wrap_string(sc, "a NaN", 5));
  return(make_integer(sc, nan_payload(real(nan))));
}

/* no similar support for +inf.0 because inf is just a single bit pattern in ieee754 */


/* -------- gmp stuff -------- */
#if With_Gmp
static mp_prec_t mpc_precision = Default_Bignum_Precision;
static mp_prec_t mpc_set_default_precision(mp_prec_t prec) {mpc_precision = prec; return(prec);}
#define mpc_init(Z) mpc_init2(Z, mpc_precision)

static bigint *alloc_bigint(s7_scheme *sc)
{
  bigint *p;
  if (sc->bigints)
    {
      p = sc->bigints;
      sc->bigints = p->nxt;
    }
  else
    {
      p = (bigint *)Malloc(sizeof(bigint));
      /* not permalloc here: gmp must be playing tricky games with realloc or something.  permalloc can lead
       *   to mpz_set_si overwriting adjacent memory (valgrind does not catch this), clobbering at least the
       *   bigint nxt field.  Someday I need to look at the source.
       */
      mpz_init(p->n);
    }
  return(p);
}

static bigrat *alloc_bigrat(s7_scheme *sc)
{
  bigrat *p;
  if (sc->bigrats)
    {
      p = sc->bigrats;
      sc->bigrats = p->nxt;
    }
  else
    {
      p = (bigrat *)Malloc(sizeof(bigrat));
      mpq_init(p->q);
    }
  return(p);
}

static bigflt *alloc_bigflt(s7_scheme *sc)
{
  bigflt *p;
  if (sc->bigflts)
    {
      p = sc->bigflts;
      sc->bigflts = p->nxt;
      mpfr_set_prec(p->x, sc->bignum_precision);
    }
  else
    {
      p = (bigflt *)Malloc(sizeof(bigflt));
      mpfr_init2(p->x, sc->bignum_precision);
    }
  return(p);
}

static bigcmp *alloc_bigcmp(s7_scheme *sc)
{
  bigcmp *p;
  if (sc->bigcmps)
    {
      p = sc->bigcmps;
      sc->bigcmps = p->nxt;
      mpc_set_prec(p->z, sc->bignum_precision);
    }
  else
    {
      p = (bigcmp *)Malloc(sizeof(bigcmp));
      mpc_init(p->z);
    }
  return(p);
}

static s7_pointer mpz_to_big_integer(s7_scheme *sc, mpz_t val)
{
  s7_pointer new_bgi;
  new_cell(sc, new_bgi, T_BIG_INTEGER);
  big_integer_bgi(new_bgi) = alloc_bigint(sc);
  mpz_set(big_integer(new_bgi), val);
  add_big_integer(sc, new_bgi);
  return(new_bgi);
}

static s7_pointer mpz_to_integer(s7_scheme *sc, mpz_t val)
{
  if (mpz_fits_slong_p(val))
    return(make_integer(sc, mpz_get_si(val)));
  return(mpz_to_big_integer(sc, val));
}

#if !With_Pure_s7
static s7_pointer mpz_to_big_real(s7_scheme *sc, mpz_t val)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  add_big_real(sc, new_bgf);
  mpfr_set_z(big_real(new_bgf), val, Mpfr_Rndn);
  return(new_bgf);
}
#endif

static s7_pointer mpq_to_big_ratio(s7_scheme *sc, mpq_t val)
{
  s7_pointer new_bgr;
  new_cell(sc, new_bgr, T_BIG_RATIO);
  big_ratio_bgr(new_bgr) = alloc_bigrat(sc);
  add_big_ratio(sc, new_bgr);
  mpq_set(big_ratio(new_bgr), val);
  return(new_bgr);
}

static s7_pointer mpq_to_rational(s7_scheme *sc, mpq_t val)
{
  if (mpz_cmp_ui(mpq_denref(val), 1) == 0)
    return(mpz_to_integer(sc, mpq_numref(val)));
#if s7_Debugging
  mpq_canonicalize(val);
  if (mpz_cmp_ui(mpq_denref(val), 1) == 0)
    {
      fprintf(stderr, "mpq_to_rational: missing canonicalize\n");
      return(mpz_to_integer(sc, mpq_numref(val)));
    }
#endif
 if ((mpz_fits_slong_p(mpq_numref(val))) && (mpz_fits_slong_p(mpq_denref(val))))
    return(make_simple_ratio(sc, mpz_get_si(mpq_numref(val)), mpz_get_si(mpq_denref(val))));
  return(mpq_to_big_ratio(sc, val));
}

static s7_pointer mpq_to_canonicalized_rational(s7_scheme *sc, mpq_t mpq)
{
  mpq_canonicalize(mpq);
  return(mpq_to_rational(sc, mpq));
}

static s7_pointer mpz_to_rational(s7_scheme *sc, mpz_t n, mpz_t d) /* mpz_3 and mpz_4 */
{
  if (mpz_cmp_ui(d, 1) == 0)
    return(mpz_to_integer(sc, n));
  mpq_set_num(sc->mpq_1, n);
  mpq_set_den(sc->mpq_1, d);
  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
}

#if !With_Pure_s7
static s7_pointer mpq_to_big_real(s7_scheme *sc, mpq_t val)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  add_big_real(sc, new_bgf);
  mpfr_set_q(big_real(new_bgf), val, Mpfr_Rndn);
  return(new_bgf);
}
#endif

static s7_pointer any_rational_to_mpq(s7_scheme *sc, s7_pointer z, mpq_t bigq)
{
  switch (type(z))
    {
    case T_INTEGER:     mpq_set_si(bigq, integer(z), 1);                break;
    case T_BIG_INTEGER: mpq_set_z(bigq, big_integer(z));                break;
    case T_RATIO:       mpq_set_si(bigq, numerator(z), denominator(z)); break;
    case T_BIG_RATIO:   mpq_set(bigq, big_ratio(z));                    break;
    }
  return(z);
}

static s7_pointer mpfr_to_integer(s7_scheme *sc, mpfr_t val)
{
  mpfr_get_z(sc->mpz_4, val, Mpfr_Rndn);
  return(mpz_to_integer(sc, sc->mpz_4));
}

static s7_pointer mpfr_to_big_real(s7_scheme *sc, mpfr_t val)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL);
  add_big_real(sc, new_bgf);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  mpfr_set(big_real(new_bgf), val, Mpfr_Rndn);
  return(new_bgf);
}

static s7_pointer mpc_to_number(s7_scheme *sc, mpc_t val)
{
  s7_pointer new_bgc;
  if (mpfr_zero_p(mpc_imagref(val)))
    return(mpfr_to_big_real(sc, mpc_realref(val)));
  new_cell(sc, new_bgc, T_BIG_COMPLEX);
  big_complex_bgc(new_bgc) = alloc_bigcmp(sc);
  add_big_complex(sc, new_bgc);
  mpc_set(big_complex(new_bgc), val, MPC_RNDNN);
  return(new_bgc);
}

/* s7.h */
mpz_t  *s7_big_integer(s7_pointer x) {return(&big_integer(x));}
mpq_t  *s7_big_ratio(s7_pointer x)   {return(&big_ratio(x));}
mpfr_t *s7_big_real(s7_pointer x)    {return(&big_real(x));}
mpc_t  *s7_big_complex(s7_pointer x) {return(&big_complex(x));}

bool s7_is_big_integer(s7_pointer x) {return(is_t_big_integer(x));}
bool s7_is_big_ratio(s7_pointer x)   {return(is_t_big_ratio(x));}
bool s7_is_big_real(s7_pointer x)    {return(is_t_big_real(x));}
bool s7_is_big_complex(s7_pointer x) {return(is_t_big_complex(x));}

s7_pointer s7_make_big_integer(s7_scheme *sc, mpz_t *val) {return(mpz_to_integer(sc, *val));}
s7_pointer s7_make_big_ratio(s7_scheme *sc, mpq_t *val)   {return(mpq_to_rational(sc, *val));}
s7_pointer s7_make_big_real(s7_scheme *sc, mpfr_t *val)   {return(mpfr_to_big_real(sc, *val));}
s7_pointer s7_make_big_complex(s7_scheme *sc, mpc_t *val) {return(mpc_to_number(sc, *val));}

#if !With_Pure_s7
static s7_pointer big_integer_to_big_real(s7_scheme *sc, s7_pointer x) {return(mpz_to_big_real(sc, big_integer(x)));}
static s7_pointer big_ratio_to_big_real(s7_scheme *sc, s7_pointer x)   {return(mpq_to_big_real(sc, big_ratio(x)));}
#endif

static s7_pointer s7_int_to_big_integer(s7_scheme *sc, s7_int val)
{
  s7_pointer new_bgi;
  new_cell(sc, new_bgi, T_BIG_INTEGER);
  big_integer_bgi(new_bgi) = alloc_bigint(sc);
  mpz_set_si(big_integer(new_bgi), val);
  add_big_integer(sc, new_bgi);
  return(new_bgi);
}

static s7_pointer s7_int_to_big_ratio(s7_scheme *sc, s7_int num, s7_int den)
{
  /* (called only in g_bignum), den here always comes from denominator(x) or some positive constant so it is not negative */
  s7_pointer new_bgr;
  new_cell(sc, new_bgr, T_BIG_RATIO);
  big_ratio_bgr(new_bgr) = alloc_bigrat(sc);
  add_big_ratio(sc, new_bgr);
  mpq_set_si(big_ratio(new_bgr), num, den);
  return(new_bgr);
}

static s7_pointer s7_double_to_big_real(s7_scheme *sc, s7_double rl)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  add_big_real(sc, new_bgf);
  mpfr_set_d(big_real(new_bgf), rl, Mpfr_Rndn);
  return(new_bgf);
}

static s7_pointer s7_double_to_big_complex(s7_scheme *sc, s7_double rl, s7_double im)
{
  s7_pointer new_bgc;
  new_cell(sc, new_bgc, T_BIG_COMPLEX);
  add_big_complex(sc, new_bgc);
  big_complex_bgc(new_bgc) = alloc_bigcmp(sc);
  mpc_set_d_d(big_complex(new_bgc), rl, im, MPC_RNDNN);
  return(new_bgc);
}

static s7_pointer big_pi(s7_scheme *sc)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL | T_Immutable);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  add_big_real(sc, new_bgf);
  mpfr_const_pi(big_real(new_bgf), Mpfr_Rndn);
  return(new_bgf);
}

static bool is_integer_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_integer(p))
    return(true);
  if (has_active_methods(sc, p))
    {
      s7_pointer func = find_method_with_let(sc, p, sc->is_integer_symbol);
      if (func != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, func, set_plist_1(sc, p))));
    }
  return(false);
}

#if !With_Pure_s7
static s7_pointer s7_number_to_big_real(s7_scheme *sc, s7_pointer num)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  add_big_real(sc, new_bgf);

  switch (type(num))
    {
    case T_INTEGER:
      mpfr_set_si(big_real(new_bgf), integer(num), Mpfr_Rndn);
      break;
    case T_RATIO:
      /* here we can't use fraction(number(num)) even though that uses long_double division because
       *   there are lots of s7_int ratios that will still look the same. We have to do the bignum divide by hand.
       */
      mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
      mpfr_set_q(big_real(new_bgf), sc->mpq_1, Mpfr_Rndn);
      break;
    default:
      mpfr_set_d(big_real(new_bgf), s7_real(num), Mpfr_Rndn);
      break;
    }
  return(new_bgf);
}
#endif

static s7_pointer s7_number_to_big_complex(s7_scheme *sc, s7_pointer num)
{
  s7_pointer new_bgc;
  new_cell(sc, new_bgc, T_BIG_COMPLEX);
  big_complex_bgc(new_bgc) = alloc_bigcmp(sc);
  add_big_complex(sc, new_bgc);

  switch (type(num))
    {
    case T_INTEGER:
      mpc_set_si(big_complex(new_bgc), integer(num), MPC_RNDNN);
      break;
    case T_RATIO:
      /* can't use fraction here */
      mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
      mpfr_set_q(sc->mpfr_1, sc->mpq_1, Mpfr_Rndn);
      mpc_set_fr(big_complex(new_bgc), sc->mpfr_1, MPC_RNDNN);
      break;
    case T_REAL:
      mpc_set_d(big_complex(new_bgc), s7_real(num), MPC_RNDNN);
      break;
    default:
      mpc_set_d_d(big_complex(new_bgc), real_part(num), imag_part(num), MPC_RNDNN);
      break;
    }
  return(new_bgc);
}

static s7_pointer any_real_to_mpfr(s7_scheme *sc, s7_pointer num, mpfr_t bigx)
{
  switch (type(num))
    {
    case T_INTEGER:
      mpfr_set_si(bigx, integer(num), Mpfr_Rndn);
      break;
    case T_RATIO:
      mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
      mpfr_set_q(bigx, sc->mpq_1, Mpfr_Rndn);
      break;
    case T_REAL:
      mpfr_set_d(bigx, real(num), Mpfr_Rndn);
      if (is_NaN(real(num))) return(make_nan_with_payload(sc, __LINE__));
      if (is_inf(real(num))) return(real_infinity);
      break;
    case T_BIG_INTEGER:
      mpfr_set_z(bigx, big_integer(num), Mpfr_Rndn);
      break;
    case T_BIG_RATIO:
      mpfr_set_q(bigx, big_ratio(num), Mpfr_Rndn);
      break;
    case T_BIG_REAL:
      mpfr_set(bigx, big_real(num), Mpfr_Rndn);
      if (mpfr_nan_p(big_real(num))) return(make_nan_with_payload(sc, __LINE__));
      if (mpfr_inf_p(big_real(num))) return(real_infinity);
      break;
    }
  return(NULL);
}

#define mpc_zero_p(z) ((mpfr_zero_p(mpc_realref(z))) && (mpfr_zero_p(mpc_imagref(z))))

static s7_pointer any_number_to_mpc(s7_scheme *sc, s7_pointer num, mpc_t bigz)
{
  switch (type(num))
    {
    case T_INTEGER:
      mpc_set_si(bigz, integer(num), MPC_RNDNN);
      break;
    case T_RATIO:
      mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
      mpc_set_q(bigz, sc->mpq_1, MPC_RNDNN);
      break;
    case T_REAL:
      if (is_NaN(real(num))) return(make_nan_with_payload(sc, __LINE__));
      if (is_inf(real(num))) return(real_infinity);
      mpc_set_d(bigz, real(num), MPC_RNDNN);
      break;
    case T_COMPLEX:
      if (is_NaN(imag_part(num))) return(complex_NaN);
      if (is_NaN(real_part(num))) return(make_nan_with_payload(sc, __LINE__));
     mpc_set_d_d(bigz, real_part(num), imag_part(num), MPC_RNDNN);
      break;
    case T_BIG_INTEGER:
      mpc_set_z(bigz, big_integer(num), MPC_RNDNN);
      break;
    case T_BIG_RATIO:
      mpc_set_q(bigz, big_ratio(num), MPC_RNDNN);
      break;
    case T_BIG_REAL:
      mpc_set_fr(bigz, big_real(num), MPC_RNDNN);
      if (mpfr_nan_p(big_real(num))) return(make_nan_with_payload(sc, __LINE__));
      if (mpfr_inf_p(big_real(num))) return(real_infinity);
      break;
    case T_BIG_COMPLEX:
      if (mpfr_nan_p(mpc_imagref(big_complex(num)))) return(complex_NaN);
      if (mpfr_nan_p(mpc_realref(big_complex(num)))) return(make_nan_with_payload(sc, __LINE__));
      mpc_set(bigz, big_complex(num), MPC_RNDNN);
      break;
    }
  return(NULL);
}

static s7_pointer make_big_complex(s7_scheme *sc, mpfr_t rl, mpfr_t im)
{
  /* there is no mpc_get_str equivalent, so we need to split up str, use make_big_real to get the 2 halves, then mpc_init, then mpc_set_fr_fr */
  s7_pointer new_bgc;
  new_cell(sc, new_bgc, T_BIG_COMPLEX);
  big_complex_bgc(new_bgc) = alloc_bigcmp(sc);
  add_big_complex(sc, new_bgc);
  mpc_set_fr_fr(big_complex(new_bgc), rl ,im, MPC_RNDNN);
  return(new_bgc);
}

static block_t *mpfr_to_string(s7_scheme *sc, mpfr_t val, int32_t radix)
{
  if (mpfr_zero_p(val))
    return(string_to_block(sc, "0.0", 3));
  if (mpfr_nan_p(val))
    return(string_to_block(sc, "+nan.0", 6));
  if (mpfr_inf_p(val))
    return((mpfr_signbit(val) == 0) ? string_to_block(sc, "+inf.0", 6) : string_to_block(sc, "-inf.0", 6));
  {
    mp_exp_t expptr;
    block_t *b = callocate(sc, sc->bignum_precision + 32);
    char *str = mpfr_get_str((char *)block_data(b), &expptr, radix, 0, val, Mpfr_Rndn);
    int32_t ep = (int32_t)expptr;
    s7_int i, len = safe_strlen(str);

    /* remove trailing 0's */
    for (i = len - 1; i > 3; i--)
      if (str[i] != '0')
	break;
    if (i < len - 1)
      str[i + 1] = '\0';
    {
      #define Btmp_Bufsize (len + 64)
      block_t *btmp = mallocate(sc, Btmp_Bufsize);
      if (str[0] == '-')
	snprintf((char *)block_data(btmp), Btmp_Bufsize, "-%c.%s%c%d", str[1], (char *)(str + 2), (radix <= 10) ? 'E' : '@', ep - 1);
      else snprintf((char *)block_data(btmp), Btmp_Bufsize, "%c.%s%c%d", str[0], (char *)(str + 1), (radix <= 10) ? 'E' : '@', ep - 1);
      liberate(sc, b);
      return(btmp);
    }}
}

static block_t *mpc_to_string(s7_scheme *sc, mpc_t val, int32_t radix, use_write_t use_write)
{
  block_t *rl, *im, *tmp;
  s7_int len;

  mpc_real(sc->mpfr_1, val, Mpfr_Rndn);
  rl = mpfr_to_string(sc, sc->mpfr_1, radix);
  mpc_imag(sc->mpfr_2, val, Mpfr_Rndn);
  im = mpfr_to_string(sc, sc->mpfr_2, radix);

  len = safe_strlen((char *)block_data(rl)) + safe_strlen((char *)block_data(im)) + 128;
  tmp = mallocate(sc, len);
  snprintf((char *)block_data(tmp), len, "%s%s%si",
	   (char *)block_data(rl),
	   ((((char *)block_data(im))[0] == '-') || (((char *)block_data(im))[0] == '+')) ? "" : "+", (char *)block_data(im));

  liberate(sc, rl);
  liberate(sc, im);
  return(tmp);
}

static block_t *big_number_to_string_with_radix(s7_scheme *sc, s7_pointer num, int32_t radix, s7_int width, s7_int *nlen, use_write_t use_write)
{
  block_t *str;
  switch (type(num))
    {
    case T_BIG_INTEGER:
      str = callocate(sc, mpz_sizeinbase(big_integer(num), radix) + 64);
      mpz_get_str((char *)block_data(str), radix, big_integer(num));
      break;
    case T_BIG_RATIO:
      mpz_set(sc->mpz_1, mpq_numref(big_ratio(num)));
      mpz_set(sc->mpz_2, mpq_denref(big_ratio(num)));
      str = callocate(sc, mpz_sizeinbase(sc->mpz_1, radix) + mpz_sizeinbase(sc->mpz_2, radix) + 64);
      mpq_get_str((char *)block_data(str), radix, big_ratio(num));
      break;
    case T_BIG_REAL:
      str = mpfr_to_string(sc, big_real(num), radix);
      break;
    default:
      str = mpc_to_string(sc, big_complex(num), radix, use_write);
      break;
    }
  if (width > 0)
    {
      const s7_int len = safe_strlen((char *)block_data(str));
      if (width > len)
	{
	  const int32_t spaces = width - len;
	  block_t *tmp = (block_t *)mallocate(sc, width + 1);
	  ((char *)block_data(tmp))[width] = '\0';
	  memmove((void *)((char *)block_data(tmp) + spaces), (void *)block_data(str), len);
	  local_memset((void *)block_data(tmp), (int)' ', spaces);
	  (*nlen) = width;
	  liberate(sc, str);
	  return(tmp);
	}
      (*nlen) = len;
    }
  else (*nlen) = safe_strlen((char *)block_data(str));
  return(str);
}

static s7_pointer string_to_big_integer(s7_scheme *sc, const char *str, int32_t radix)
{
  mpz_set_str(sc->mpz_4, (str[0] == '+') ? (const char *)(str + 1) : str, radix);
  return(mpz_to_integer(sc, sc->mpz_4));
}

static s7_pointer string_to_big_ratio(s7_scheme *sc, const char *str, int32_t radix)
{
  s7_pointer new_bgr;
  mpq_set_str(sc->mpq_1, str, radix);
  mpq_canonicalize(sc->mpq_1);
  if (mpz_cmp_ui(mpq_denref(sc->mpq_1), 1) == 0)
    return(mpz_to_integer(sc, mpq_numref(sc->mpq_1)));
  new_cell(sc, new_bgr, T_BIG_RATIO);
  big_ratio_bgr(new_bgr) = alloc_bigrat(sc);
  add_big_ratio(sc, new_bgr);
  mpq_set(big_ratio(new_bgr), sc->mpq_1);
  return(new_bgr);
}

static s7_pointer string_to_big_real(s7_scheme *sc, const char *str, int32_t radix)
{
  s7_pointer new_bgf;
  new_cell(sc, new_bgf, T_BIG_REAL);
  big_real_bgf(new_bgf) = alloc_bigflt(sc);
  add_big_real(sc, new_bgf);
  mpfr_set_str(big_real(new_bgf), str, radix, Mpfr_Rndn);
  return(new_bgf);
}

static s7_int string_to_integer(const char *str, int32_t radix, bool *overflow);

static s7_pointer string_to_either_integer(s7_scheme *sc, const char *str, int32_t radix)
{
  bool overflow = false;
  s7_int val = string_to_integer(str, radix, &overflow);
  if (!overflow)
    return(make_integer(sc, val));
  return(string_to_big_integer(sc, str, radix));
}

static s7_pointer string_to_either_ratio(s7_scheme *sc, const char *nstr, const char *dstr, int32_t radix)
{
  bool overflow = false;
  /* gmp segfaults if passed a bignum/0 so this needs to check first that the denominator is not 0 before letting gmp screw up.
   *   Also, if the first character is '+', gmp returns 0!
   */
  const s7_int d = string_to_integer(dstr, radix, &overflow);
  if (!overflow)
    {
      s7_int n;
      if (d == 0) return(make_nan_with_payload(sc, __LINE__)); /* this NaN can end up as a hash-table key -- maybe the payload is confusing? */
      n = string_to_integer(nstr, radix, &overflow);
      if (!overflow)
	return(make_ratio(sc, n, d));
    }
  if (nstr[0] == '+')
    return(string_to_big_ratio(sc, (const char *)(nstr + 1), radix));
  return(string_to_big_ratio(sc, nstr, radix));
}

static s7_double string_to_double_with_radix(const char *ur_str, int32_t radix, bool *overflow); /* gmp version */

static s7_pointer string_to_either_real(s7_scheme *sc, const char *str, int32_t radix)
{
  bool overflow = false;
  s7_double val = string_to_double_with_radix((char *)str, radix, &overflow);
  if (!overflow) return(make_real(sc, val));
  return(string_to_big_real(sc, str, radix));
}

static s7_pointer string_to_either_complex_1(s7_scheme *sc, char *q, char *slash1, char *ex1, bool has_dec_point1, int32_t radix, s7_double *d_rl)
{
  bool overflow = false;
  /* there's a real problem here -- we don't want to promote s7_double .1 to a bignum because
   *    its low order digits are garbage, causing (rationalize .1 0) to return 3602879701896397/36028797018963968
   *    no matter what the bignum-precision.  But we can't just fallback on gmp's reader because (for example)
   *    it reads 1/2+i or 1+0/0i as 1.0.  Also format gets screwed up.  And string->number signals an error
   *    where it should return #f.  I wonder what to do.
   */
  if ((has_dec_point1) ||
      (ex1))
    {
      (*d_rl) = string_to_double_with_radix(q, radix, &overflow);
      if (overflow) return(string_to_big_real(sc, q, radix));
    }
  else
    {
      if (slash1)
	{
	  s7_int d, n = string_to_integer(q, radix, &overflow);  /* q can include the slash and denominator */
	  if (overflow) return(string_to_big_ratio(sc, q, radix));
	  d = string_to_integer(slash1, radix, &overflow);
	  if (overflow) return(string_to_big_ratio(sc, q, radix));
	  (*d_rl) = (s7_double)n / (s7_double)d;
	}
      else
	{
	  s7_int val = string_to_integer(q, radix, &overflow);
	  if (overflow) return(string_to_big_integer(sc, q, radix));
	  (*d_rl) = (s7_double)val;
	}}
  if ((*d_rl) == -0.0) (*d_rl) = 0.0;
  return(NULL);
}

static s7_pointer string_to_either_complex(s7_scheme *sc, char *q, char *slash1, char *ex1, bool has_dec_point1,
					   char *plus, char *slash2, char *ex2, bool has_dec_point2,
					   int32_t radix, int32_t has_plus_or_minus)
{
  /* this can be just about anything involving 2 real/ratio/int portions, +/- in between and 'i' at the end */
  double d_rl = 0.0, d_im = 0.0;
  s7_pointer p_rl = string_to_either_complex_1(sc, q, slash1, ex1, has_dec_point1, radix, &d_rl);
  s7_pointer p_im = string_to_either_complex_1(sc, plus, slash2, ex2, has_dec_point2, radix, &d_im);

  if ((d_im == 0.0) &&                     /* 1.0+0.0000000000000000000000000000i */
      ((!p_im) || (is_zero(p_im))))
    return((p_rl) ? p_rl : make_real(sc, d_rl));
  if ((!p_rl) && (!p_im))
    return(make_complex_not_0i(sc, d_rl, (has_plus_or_minus == -1) ? (-d_im) : d_im));
  if (p_rl)
    any_real_to_mpfr(sc, p_rl, sc->mpfr_1);
  else mpfr_set_d(sc->mpfr_1, d_rl, Mpfr_Rndn);
  if (p_im)
    any_real_to_mpfr(sc, p_im, sc->mpfr_2);
  else mpfr_set_d(sc->mpfr_2, d_im, Mpfr_Rndn);
  if (has_plus_or_minus == -1)
    mpfr_neg(sc->mpfr_2, sc->mpfr_2, Mpfr_Rndn);
  return(make_big_complex(sc, sc->mpfr_1, sc->mpfr_2));
}

static bool big_numbers_are_eqv(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* either or both can be big here, but not neither, and types might not match at all */
  switch (type(x))
    {
    case T_INTEGER:
      return((is_t_big_integer(y)) && (mpz_cmp_si(big_integer(y), integer(x)) == 0));
    case T_BIG_INTEGER:
      if (is_t_big_integer(y)) return(mpz_cmp(big_integer(x), big_integer(y)) == 0);
      return((is_t_integer(y)) && (mpz_cmp_si(big_integer(x), integer(y)) == 0));
    case T_RATIO:
      if (!is_t_big_ratio(y)) return(false);
      mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
      return(mpq_equal(sc->mpq_1, big_ratio(y)));
    case T_BIG_RATIO:
      if (is_t_big_ratio(y)) return(mpq_equal(big_ratio(x), big_ratio(y)));
      if (!is_t_ratio(y)) return(false);
      mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
      return(mpq_equal(sc->mpq_1, big_ratio(x)));
    case T_REAL:
      if (is_NaN(real(x))) return(false);
      return((is_t_big_real(y)) && (!mpfr_nan_p(big_real(y))) && (mpfr_cmp_d(big_real(y), real(x)) == 0));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x))) return(false);
      if (is_t_big_real(y)) return((!mpfr_nan_p(big_real(y))) && (mpfr_equal_p(big_real(x), big_real(y))));
      return((is_t_real(y)) && (!is_NaN(real(y))) && (mpfr_cmp_d(big_real(x), real(y)) == 0));
    case T_COMPLEX:
      if ((is_NaN(real_part(x))) || (is_NaN(imag_part(x)))) return(false);
      if (!is_t_big_complex(y)) return(false);
      if ((mpfr_nan_p(mpc_realref(big_complex(y)))) || (mpfr_nan_p(mpc_imagref(big_complex(y)))))
	return(false);
      mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
      return(mpc_cmp(sc->mpc_1, big_complex(y)) == 0);
    case T_BIG_COMPLEX:
      if ((mpfr_nan_p(mpc_realref(big_complex(x)))) || (mpfr_nan_p(mpc_imagref(big_complex(x)))))
	return(false);
      if (is_t_big_complex(y))
	{
	  if ((mpfr_nan_p(mpc_realref(big_complex(y)))) || (mpfr_nan_p(mpc_imagref(big_complex(y)))))
	    return(false);
	  return(mpc_cmp(big_complex(x), big_complex(y)) == 0);
	}
      if (is_t_complex(y))
	{
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))) return(false);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  return(mpc_cmp(big_complex(x), sc->mpc_1) == 0);
	}}
  return(false);
}

static s7_int big_integer_to_s7_int(s7_scheme *sc, mpz_t n)
{
  if (!mpz_fits_slong_p(n))
    error_nr(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "bigint does not fit in s7_int: ~S", 33), mpz_to_big_integer(sc, n)));
  return(mpz_get_si(n));
}
#endif

#ifndef HAVE_OVERFLOW_CHECKS
  #if ((defined(__clang__) && (!Pointer_32) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4))) || (defined(__GNUC__) && (__GNUC__ >= 5)))
    #define HAVE_OVERFLOW_CHECKS 1
  #else
    #define HAVE_OVERFLOW_CHECKS 0
    #pragma message("no arithmetic overflow checks in this version of s7")
    /* these are untested */
    static bool add_overflow(s7_int A, s7_int B, s7_int *C) {*C = A + B; return(false);}        /* #define add_overflow(A, B, C) 0 */
    static bool subtract_overflow(s7_int A, s7_int B, s7_int *C) {*C = A - B; return(false);}   /* #define subtract_overflow(A, B, C) 0 */
    static bool multiply_overflow(s7_int A, s7_int B, s7_int *C) {*C = A * B; return(false);}   /* #define multiply_overflow(A, B, C) 0 */
  #endif
#endif
#define Have_Overflow_Checks HAVE_OVERFLOW_CHECKS

#if (defined(__clang__) && (!Pointer_32) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4)))
  #define subtract_overflow(A, B, C)       __builtin_ssubll_overflow((long long)A, (long long)B, (long long *)C)
  #define add_overflow(A, B, C)            __builtin_saddll_overflow((long long)A, (long long)B, (long long *)C)
  #define multiply_overflow(A, B, C)       __builtin_smulll_overflow((long long)A, (long long)B, (long long *)C)
  #define int32_add_overflow(A, B, C)      __builtin_sadd_overflow(A, B, C)
  #define int32_multiply_overflow(A, B, C) __builtin_smul_overflow(A, B, C)
#else
#if (defined(__GNUC__) && (__GNUC__ >= 5))
  #define subtract_overflow(A, B, C)       __builtin_sub_overflow(A, B, C)
  #define add_overflow(A, B, C)            __builtin_add_overflow(A, B, C)
  #define multiply_overflow(A, B, C)       __builtin_mul_overflow(A, B, C)
  #define int32_add_overflow(A, B, C)      __builtin_add_overflow(A, B, C)
  #define int32_multiply_overflow(A, B, C) __builtin_mul_overflow(A, B, C)
#endif
#endif

#if WITH_GCC
#define s7_int_abs(x) ({s7_int _X_; _X_ = x; _X_ >= 0 ? _X_ : -_X_;})
#else
#define s7_int_abs(x) ((x) >= 0 ? (x) : -(x))
#endif
/* can't use abs even in gcc -- it doesn't work with s7_ints! */

#if !__NetBSD__
  #define s7_fabsl(X) fabsl(X)
#else
  static double s7_fabsl(long_double x) {return((signbit(x)) ? -x : x);}
#endif

/* for g_log, we also need round. this version is from stackoverflow, see also r5rs_round below */
static double s7_round(double number) {return((number < 0.0) ? ceil(number - 0.5) : floor(number + 0.5));}

#if Have_Complex_Numbers
#if __cplusplus
  #define _Complex_I (complex<s7_double>(0.0, 1.0))
  #define creal(x) Real(x)
  #define cimag(x) Imag(x)
  #define carg(x) arg(x)
  #define cabs(x) abs(x)
  #define csqrt(x) sqrt(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define cexp(x) exp(x)
  #define csin(x) sin(x)
  #define ccos(x) cos(x)
  #define ctan(x) tan(x)
  #define csinh(x) sinh(x)
  #define ccosh(x) cosh(x)
  #define ctanh(x) tanh(x)
  #define casin(x) asin(x)
  #define cacos(x) acos(x)
  #define catan(x) atan(x)
  #define casinh(x) asinh(x)
  #define cacosh(x) acosh(x)
  #define catanh(x) atanh(x)
#endif


#if !Have_Complex_Trig
#if __cplusplus

  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-s7_complex_i * clog(s7_complex_i * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-s7_complex_i * clog(z + s7_complex_i * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(s7_complex_i * clog((s7_complex_i + z) / (s7_complex_i - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
#else

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 12)
static s7_complex clog(s7_complex z) {return(log(fabs(cabs(z))) + carg(z) * s7_complex_i);}
static s7_complex cpow(s7_complex x, s7_complex y)
{
  s7_double r = cabs(x);
  s7_double theta = carg(x);
  s7_double yre = creal(y);
  s7_double yim = cimag(y);
  s7_double nr = exp(yre * log(r) - yim * theta);
  s7_double ntheta = yre * theta + yim * log(r);
  return(nr * cos(ntheta) + (nr * sin(ntheta)) * s7_complex_i);
}
#endif
#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 9) /* untested -- this orignally looked at __FreeBSD_version which apparently no longer exists */
  static s7_complex cexp(s7_complex z) {return(exp(creal(z)) * cos(cimag(z)) + (exp(creal(z)) * sin(cimag(z))) * s7_complex_i);}
#endif

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 10)
  static s7_complex csin(s7_complex z)   {return(sin(creal(z)) * cosh(cimag(z)) + (cos(creal(z)) * sinh(cimag(z))) * s7_complex_i);}
  static s7_complex ccos(s7_complex z)   {return(cos(creal(z)) * cosh(cimag(z)) + (-sin(creal(z)) * sinh(cimag(z))) * s7_complex_i);}
  static s7_complex csinh(s7_complex z)  {return(sinh(creal(z)) * cos(cimag(z)) + (cosh(creal(z)) * sin(cimag(z))) * s7_complex_i);}
  static s7_complex ccosh(s7_complex z)  {return(cosh(creal(z)) * cos(cimag(z)) + (sinh(creal(z)) * sin(cimag(z))) * s7_complex_i);}
  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-s7_complex_i * clog(s7_complex_i * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-s7_complex_i * clog(z + s7_complex_i * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(s7_complex_i * clog((s7_complex_i + z) / (s7_complex_i - z)) / 2.0);}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
#endif /* not FreeBSD 10 */
#endif /* not c++ */
#endif /* not Have_Complex_Trig */

#else  /* not Have_Complex_Numbers */
  #define _Complex_I 1.0
  #define creal(x) 0.0
  #define cimag(x) 0.0
  #define csin(x) sin(x)
  #define casin(x) x
  #define ccos(x) cos(x)
  #define cacos(x) x
  #define ctan(x) x
  #define catan(x) x
  #define csinh(x) x
  #define casinh(x) x
  #define ccosh(x) x
  #define cacosh(x) x
  #define ctanh(x) x
  #define catanh(x) x
  #define cexp(x) exp(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define csqrt(x) sqrt(x)
  #define conj(x) x
#endif

#ifdef __OpenBSD__
  /* openbsd's builtin versions of these functions are not usable */
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh_1(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
#endif
#ifdef __NetBSD__
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
#endif

bool s7_is_number(s7_pointer p)   {return(is_number(p));}
bool s7_is_complex(s7_pointer p)  {return(is_number(p));}
bool s7_is_real(s7_pointer p)     {return(is_real(p));}
bool s7_is_rational(s7_pointer p) {return(is_rational(p));}

bool s7_is_integer(s7_pointer p)
{
#if With_Gmp
  return((is_t_integer(p)) || (is_t_big_integer(p)));
#else
  return(is_t_integer(p));
#endif
}

bool s7_is_ratio(s7_pointer p)
{
#if With_Gmp
  return((is_t_ratio(p)) || (is_t_big_ratio(p)));
#else
  return(is_t_ratio(p));
#endif
}

static s7_int c_gcd_1(s7_int u, s7_int v)
{
  /* can't take abs of these so do it by hand */
  s7_int divisor = 1;
  if (u == v) return(u);
  while (((u & 1) == 0) && ((v & 1) == 0))
    {
      u /= 2;
      v /= 2;
      divisor *= 2;
    }
  return(divisor);
}

static s7_int c_gcd(s7_int u, s7_int v)
{
  /* #if __cplusplus\n return std::gcd(u, v);\n #else... but this requires #include <algorithm> (else gcd is not defined in std::)
   *   and C++'s gcd returns negative results sometimes -- isn't gcd defined to be positive?  std::gcd is ca 25% faster than the code below.
   */
  s7_int a, b;
  if (u < 0)
    {
      if (u == s7_Int64_Min) return(c_gcd_1(u, v));
      a = -u;
    }
  else a = u;
  if (v < 0)
    {
      if (v == s7_Int64_Min) return(c_gcd_1(u, v));
      b = -v;
    }
  else b = v;
  while (b != 0)
    {
      s7_int temp = a % b;
      a = b;
      b = temp;
    }
  return(a);
}

#define Rationalize_Limit 1.0e12

static bool c_rationalize(s7_double ux, s7_double error, s7_int *numer, s7_int *denom)
{
  /* from CL code in Canny, Donald, Ressler, "A Rational Rotation Method for Robust Geometric Algorithms" */
  double x0, x1;
  s7_int i, p0, q0 = 1, p1, q1 = 1;
  double e0, e1, e0p, e1p;
  int32_t tries = 0;
  /* don't use long_double: the loop below will hang */

  /* #e1e19 is a killer -- it's bigger than most-positive-fixnum, but if we ceil(ux) below
   *   it turns into most-negative-fixnum.  1e19 is trouble in many places.
   */
  if (fabs(ux) > Rationalize_Limit)
    {
      /* (rationalize most-positive-fixnum) should not return most-negative-fixnum
       *   but any number > 1e14 here is so inaccurate that rationalize is useless
       *   for example,
       *     default: (rationalize (/ (*s7* 'most-positive-fixnum) 31111.0)) -> 1185866354261165/4
       *     gmp:     (rationalize (/ (*s7* 'most-positive-fixnum) 31111.0)) -> 9223372036854775807/31111
       * can't return false here because that confuses some of the callers!
       */
      (*numer) = (s7_int)ux;
      (*denom) = 1;
      return(true);
    }

  if (error < 0.0) error = -error;
  x0 = ux - error;
  x1 = ux + error;
  i = (s7_int)ceil(x0);

  if (error >= 1.0) /* aw good grief! */
    {
      if (x0 < 0.0)
	(*numer) = (x1 < 0.0) ? (s7_int)floor(x1) : 0;
      else (*numer) = i;
      (*denom) = 1;
      return(true);
    }
  if (x1 >= i)
    {
      (*numer) = (i >= 0) ? i : (s7_int)floor(x1);
      (*denom) = 1;
      return(true);
    }

  p0 = (s7_int)floor(x0);
  p1 = (s7_int)ceil(x1);
  e0 = p1 - x0;
  e1 = x0 - p0;
  e0p = p1 - x1;
  e1p = x1 - p0;
  while (true)
    {
      s7_int old_p1, old_q1;
      double old_e0, old_e1, old_e0p, r, r1;
      const double val = (double)p0 / (double)q0;

      if (((x0 <= val) && (val <= x1)) || (e1 == 0.0) || (e1p == 0.0) || (tries > 100))
	{
	  if ((q0 == s7_Int64_Min) && (p0 == 1)) /* (rationalize 1.000000004297917e-12) when error is 1e-12 */
	    {
	      (*numer) = 0;
	      (*denom) = 1;
	    }
	  else
	    {
	      (*numer) = p0;
	      (*denom) = q0;
	      if ((s7_Debugging) && (q0 == 0)) fprintf(stderr, "%s[%d]: %f %" ld64 "/0\n", __func__, __LINE__, ux, p0);
	    }
	  if ((s7_Debugging) && (*denom < 0)) fprintf(stderr, "%s[%d]: denominator is %" ld64 "?\n", __func__, __LINE__, *denom);
	  return(true);
	}
      tries++;
      r = (s7_int)floor(e0 / e1);
      r1 = (s7_int)ceil(e0p / e1p);
      if (r1 < r) r = r1;
      /* do handles all step vars in parallel */
      old_p1 = p1;
      p1 = p0;
      old_q1 = q1;
      q1 = q0;
      old_e0 = e0;
      e0 = e1p;
      old_e0p = e0p;
      e0p = e1;
      old_e1 = e1;
      p0 = old_p1 + r * p0;
      q0 = old_q1 + r * q0;
      e1 = old_e0p - r * e1p;  /* if the error is set too low, we can get e1 = 0 here: (rationalize (/ pi) 1e-17) */
      e1p = old_e0 - r * old_e1;
    }
  return(false);
}

s7_pointer s7_rationalize(s7_scheme *sc, s7_double x, s7_double error)
{
  s7_int numer = 0, denom = 1;
  if (c_rationalize(x, error, &numer, &denom))
    return(make_simpler_ratio_or_integer(sc, numer, denom));
  return(make_real(sc, x));
}

s7_pointer s7_make_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer new_int;
  if (is_small_int(n))
    return(small_int(n));
  new_cell(sc, new_int, T_INTEGER);
  set_integer(new_int, n);
  return(new_int);
}

#if s7_Debugging
#define make_mutable_integer(Sc, N) make_mutable_integer_1(Sc, N, __func__, __LINE__)
static s7_pointer make_mutable_integer_1(s7_scheme *sc, s7_int n, const char *func, int line)
#else
static s7_pointer make_mutable_integer(s7_scheme *sc, s7_int n)
#endif
{
  s7_pointer new_int;
  new_cell(sc, new_int, T_INTEGER | T_Mutable);
#if s7_Debugging
  new_int->carrier_line = __LINE__;
  new_int->gc_line = line;
  new_int->gc_func = func;
#endif
  set_integer(new_int, n);
  return(new_int);
}

s7_pointer s7_make_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL);
  set_real(x, n);
  return(x);
}

#if s7_Debugging
#define make_mutable_real(Sc, N) make_mutable_real_1(Sc, N, __func__, __LINE__)
static s7_pointer make_mutable_real_1(s7_scheme *sc, s7_double n, const char *func, int line)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL | T_Mutable);
  x->carrier_line = __LINE__;
  x->gc_line = line;
  x->gc_func = func;
  set_real(x, n);
  return(x);
}
#else
#define make_mutable_real(Sc, N) s7_make_mutable_real(Sc, N)
#endif

s7_pointer s7_make_mutable_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL | T_Mutable);
  set_real(x, n);
  return(x);
}

s7_pointer s7_make_complex(s7_scheme *sc, s7_double a, s7_double b)
{
  s7_pointer x;
  if (b == 0.0)
    {
      new_cell(sc, x, T_REAL);
      set_real(x, a);
    }
  else
    {
      new_cell(sc, x, T_COMPLEX);
      set_real_part(x, a);
      set_imag_part(x, b);
    }
  return(x);
}

static s7_pointer make_mutable_complex(s7_scheme *sc, s7_double rl, s7_double im)
{
  s7_pointer x;
  new_cell(sc, x, T_COMPLEX | T_Mutable); /* do we need to change to real if imag==0? */
  set_real_part(x, rl);
  set_imag_part(x, im);
  return(x);
}

static s7_complex s7_to_c_complex(s7_pointer z)
{
#if Have_Complex_Numbers
  return(CMPLX(s7_real_part(z), s7_imag_part(z)));
#else
  return(0.0);
#endif
}

static inline s7_pointer c_complex_to_s7(s7_scheme *sc, s7_complex z) {return(make_complex(sc, creal(z), cimag(z)));}

static no_return void division_by_zero_error_1_nr(s7_scheme *sc, s7_pointer caller, s7_pointer x)
{
  error_nr(sc, sc->division_by_zero_symbol,
	   set_elist_4(sc, wrap_string(sc, "~A: division by zero, (~A ~S)", 29), caller, caller, x));
}

static no_return void division_by_zero_error_2_nr(s7_scheme *sc, s7_pointer caller, s7_pointer x, s7_pointer y)
{
  error_nr(sc, sc->division_by_zero_symbol,
	   set_elist_5(sc, wrap_string(sc, "~A: division by zero, (~A ~S ~S)", 32), caller, caller, x, y));
}

static s7_pointer make_ratio(s7_scheme *sc, s7_int a, s7_int b)
{
  if (b < 0)
    {
      if (b == s7_Int64_Min)
	{
	  /* This should not trigger an error during reading -- we might have the
	   *   ratio on a switch with-bignums or whatever, so its mere occurrence is just an annoyance.
	   */
	  /* if (a == b) return(int_one); */
	  if (a & 1)
	    return(make_real(sc, (long_double)a / (long_double)b));
	  a /= 2;
	  b /= 2;
	}
      if (a == s7_Int64_Min)
	{
	  if (b & 1)
	    return(make_real(sc, (long_double)a / (long_double)b));
	  a /= 2;
	  b /= 2;
	}
      a = -a;
      b = -b;
    }
  if (a == s7_Int64_Min)
    {
      while (((a & 1) == 0) && ((b & 1) == 0))
	{
	  a /= 2;
	  b /= 2;
	}}
  else
    {
      s7_int b1 = b, divisor = s7_int_abs(a);
      do {
	s7_int temp = divisor % b1;
	divisor = b1;
	b1 = temp;
      } while (b1 != 0);
      if (divisor != 1)
	{
	  a /= divisor;
	  b /= divisor;
	}}
  if (b == 1)
    return(make_integer(sc, a));
  {
    s7_pointer x;
    new_cell(sc, x, T_RATIO);
    set_numerator(x, a);
    set_denominator(x, b);
    return(x);
  }
}

/* using make_ratio here is a desperate kludge trying to maintain backwards compatibility; internally we use make_ratio_with_div_check below */
s7_pointer s7_make_ratio(s7_scheme *sc, s7_int a, s7_int b)
{
  if (b == 0)
    division_by_zero_error_2_nr(sc, wrap_string(sc, "s7_make_ratio", 13), wrap_integer(sc, a), int_zero);
  return(make_ratio(sc, a, b));
}

static s7_pointer make_ratio_with_div_check(s7_scheme *sc, s7_pointer caller, s7_int a, s7_int b)
{
  if (b == 0)
    division_by_zero_error_2_nr(sc, caller, wrap_integer(sc, a), int_zero);
  return(make_ratio(sc, a, b));
}


#define With_Overflow_Error true
#define Without_Overflow_Error false

#define Int64_To_Double_Limit (1LL << 53)
#define Double_To_Int64_Limit (1LL << 53)

/* 9007199254740991LL is where a truncated double starts to skip integers (expt 2 53) = ca 1e16
 *   (ceiling (+ 1e16 1)) -> 10000000000000000
 *   (> 9007199254740993.0 9007199254740992.0) -> #f ; in non-gmp 64-bit doubles
 * but we can't fix this except in the gmp case because:
 *   (integer-decode-float (+ (expt 2.0 62) 100)) -> (4503599627370496 10 1)
 *   (integer-decode-float (+ (expt 2.0 62) 500)) -> (4503599627370496 10 1)
 *   (> (+ (expt 2.0 62) 500) (+ (expt 2.0 62) 100)) -> #f ; non-gmp again
 * i.e. the bits are identical.  We can't even detect when it has happened (without tedious effort), so should
 *   we just give an error for any floor (or whatever) of an arg>1e16?  (sin has a similar problem)?
 *   I think in the non-gmp case I'll throw an error in these cases because the results are bogus:
 *   (floor (+ (expt 2.0 62) 512)) -> 4611686018427387904
 *   (floor (+ (expt 2.0 62) 513)) -> 4611686018427388928
 * another case at the edge: (round 9007199254740992.51) -> 9007199254740992
 * This spells trouble for normal arithmetic in this range.  If no gmp,
 *    (- (+ (expt 2.0 62) 512) (+ (expt 2.0 62) 513)) = -1024.0 (should be -1.0)
 *    but we don't currently give an error in this case -- not sure what the right thing is.
 */

s7_double s7_number_to_real_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_real(x)) return(real(x));
  switch (type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return(fraction(x));
#if With_Gmp
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(sc, big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long_double)big_integer_to_s7_int(sc, mpq_numref(big_ratio(x))) /
					   (long_double)big_integer_to_s7_int(sc, mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), Mpfr_Rndn));
#endif
    default:
      sole_arg_wrong_type_error_nr(sc, wrap_string(sc, caller, safe_strlen(caller)), x, sc->type_names[T_REAL]);
    }
  return(0.0);
}

s7_double s7_number_to_real_with_location(s7_scheme *sc, s7_pointer x, s7_pointer caller)
{
  if (is_t_real(x)) return(real(x));
  switch (type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return(fraction(x));
#if With_Gmp
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(sc, big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long_double)big_integer_to_s7_int(sc, mpq_numref(big_ratio(x))) /
					   (long_double)big_integer_to_s7_int(sc, mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), Mpfr_Rndn));
#endif
    default:
      sole_arg_wrong_type_error_nr(sc, caller, x, sc->type_names[T_REAL]);
    }
  return(0.0);
}

s7_double s7_number_to_real(s7_scheme *sc, s7_pointer x) {return(s7_number_to_real_with_location(sc, x, sc->number_to_real_symbol));}

s7_int s7_number_to_integer_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_integer(x)) return(integer(x));
#if With_Gmp
  if (is_t_big_integer(x)) return(big_integer_to_s7_int(sc, big_integer(x)));
#endif
  sole_arg_wrong_type_error_nr(sc, wrap_string(sc, caller, safe_strlen(caller)), x, sc->type_names[T_INTEGER]);
  return(0);
}

s7_int s7_number_to_integer(s7_scheme *sc, s7_pointer x) {return(s7_number_to_integer_with_caller(sc, x, "s7_number_to_integer"));}

s7_int s7_numerator(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x));
    case T_RATIO:       return(numerator(x));
#if With_Gmp
    case T_BIG_INTEGER: return(mpz_get_si(big_integer(x))); /* big_integer_to_s7_int but no sc -- no error if out of range */
    case T_BIG_RATIO:   return(mpz_get_si(mpq_numref(big_ratio(x))));
#endif
    }
  return(0);
}

s7_int s7_denominator(s7_pointer x)
{
  if (is_t_ratio(x)) return(denominator(x));
#if With_Gmp
  if (is_t_big_ratio(x)) return(mpz_get_si(mpq_denref(big_ratio(x))));
#endif
  return(1);
}

s7_int s7_integer(s7_pointer p)
{
  if (is_t_integer(p)) return(integer(p));
#if With_Gmp
  if (is_t_big_integer(p)) return(mpz_get_si(big_integer(p)));
#endif
  return(0);
}

s7_double s7_real(s7_pointer x)
{
  if (is_t_real(x)) return(real(x));
  switch (type(x))
    {
    case T_RATIO:       return(fraction(x));
    case T_INTEGER:     return((s7_double)integer(x));
#if With_Gmp
    case T_BIG_INTEGER: return((s7_double)mpz_get_si(big_integer(x)));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), Mpfr_Rndn));
    case T_BIG_RATIO:
      {
	s7_double result;
	mpfr_t bx;
	mpfr_init2(bx, Default_Bignum_Precision);
	mpfr_set_q(bx, big_ratio(x), Mpfr_Rndn);
	result = mpfr_get_d(bx, Mpfr_Rndn);
	mpfr_clear(bx);
	return(result);
      }
#endif
    }
  return(0.0);
}

static bool is_one(s7_pointer x)
{
  return(((is_t_integer(x)) && (integer(x) == 1)) ||
	 ((is_t_real(x)) && (real(x) == 1.0)));
}


/* -------- optimize exponents -------- */

#define Max_Pow 64 /* faster startup if 32, but much slower in tbig; also waiting until use to init_pows is faster at startup, but slower in tbig */
static double **pepow = NULL; /* [17][Max_Pow * 2]; */

static void init_pows(void)
{
  pepow = (double **)Malloc(17 * sizeof(double *));
  pepow[0] = NULL;
  pepow[1] = NULL;
  for (int32_t i = 2; i < 17; i++) pepow[i] = (double *)Malloc((Max_Pow * 2) * sizeof(double));
  for (int32_t i = 2; i < 17; i++)               /* radix between 2 and 16 */
    for (int32_t j = -Max_Pow; j < Max_Pow; j++) /* saved exponent between 0 and +/- Max_Pow */
      pepow[i][j + Max_Pow] = pow((double)i, (double)j);
}

static inline double dpow(int32_t x, int32_t y)
{
  if ((y >= Max_Pow) || (y < -Max_Pow)) /* this can happen */
    return(pow((double)x, (double)y));
  return(pepow[x][y + Max_Pow]);
}


/* -------------------------------- number->string -------------------------------- */
#ifndef WITH_DTOA
  #define WITH_DTOA 1
#endif
/* there was a time when libc was so slow that this code was mandatory, but now (Oct-2024) the difference is smaller (still a ca. factor of 4):
 *   in tbig/callgrind with dtoa 6254M, with C's printf stuff instead 24410M
 */

#if WITH_DTOA
/* fpconv, revised to fit the local coding style

   The MIT License

Copyright (c) 2013 Andreas Samoljuk

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#define dtoa_npowers     87
#define dtoa_steppowers  8
#define dtoa_firstpower -348 /* 10 ^ -348 */
#define dtoa_expmax     -32
#define dtoa_expmin     -60

typedef struct dtoa_np {uint64_t frac; int32_t exp;} dtoa_np;

static const dtoa_np dtoa_powers_ten[] = {
    { 18054884314459144840U, -1220 }, { 13451937075301367670U, -1193 }, { 10022474136428063862U, -1166 }, { 14934650266808366570U, -1140 },
    { 11127181549972568877U, -1113 }, { 16580792590934885855U, -1087 }, { 12353653155963782858U, -1060 }, { 18408377700990114895U, -1034 },
    { 13715310171984221708U, -1007 }, { 10218702384817765436U, -980 }, { 15227053142812498563U, -954 },  { 11345038669416679861U, -927 },
    { 16905424996341287883U, -901 },  { 12595523146049147757U, -874 }, { 9384396036005875287U,  -847 },  { 13983839803942852151U, -821 },
    { 10418772551374772303U, -794 },  { 15525180923007089351U, -768 }, { 11567161174868858868U, -741 },  { 17236413322193710309U, -715 },
    { 12842128665889583758U, -688 },  { 9568131466127621947U,  -661 }, { 14257626930069360058U, -635 },  { 10622759856335341974U, -608 },
    { 15829145694278690180U, -582 },  { 11793632577567316726U, -555 }, { 17573882009934360870U, -529 },  { 13093562431584567480U, -502 },
    { 9755464219737475723U,  -475 },  { 14536774485912137811U, -449 }, { 10830740992659433045U, -422 },  { 16139061738043178685U, -396 },
    { 12024538023802026127U, -369 },  { 17917957937422433684U, -343 }, { 13349918974505688015U, -316 },  { 9946464728195732843U,  -289 },
    { 14821387422376473014U, -263 },  { 11042794154864902060U, -236 }, { 16455045573212060422U, -210 },  { 12259964326927110867U, -183 },
    { 18268770466636286478U, -157 },  { 13611294676837538539U, -130 }, { 10141204801825835212U, -103 },  { 15111572745182864684U, -77 },
    { 11258999068426240000U, -50 },   { 16777216000000000000U, -24 }, { 12500000000000000000U,   3 },   { 9313225746154785156U,   30 },
    { 13877787807814456755U,  56 },   { 10339757656912845936U,  83 }, { 15407439555097886824U, 109 },   { 11479437019748901445U, 136 },
    { 17105694144590052135U, 162 },   { 12744735289059618216U, 189 }, { 9495567745759798747U,  216 },   { 14149498560666738074U, 242 },
    { 10542197943230523224U, 269 },   { 15709099088952724970U, 295 }, { 11704190886730495818U, 322 },   { 17440603504673385349U, 348 },
    { 12994262207056124023U, 375 },   { 9681479787123295682U,  402 }, { 14426529090290212157U, 428 },   { 10748601772107342003U, 455 },
    { 16016664761464807395U, 481 },   { 11933345169920330789U, 508 }, { 17782069995880619868U, 534 },   { 13248674568444952270U, 561 },
    { 9871031767461413346U,  588 },   { 14708983551653345445U, 614 }, { 10959046745042015199U, 641 },   { 16330252207878254650U, 667 },
    { 12166986024289022870U, 694 },   { 18130221999122236476U, 720 }, { 13508068024458167312U, 747 },   { 10064294952495520794U, 774 },
    { 14996968138956309548U, 800 },   { 11173611982879273257U, 827 }, { 16649979327439178909U, 853 },   { 12405201291620119593U, 880 },
    { 9242595204427927429U,  907 },   { 13772540099066387757U, 933 }, { 10261342003245940623U, 960 },   { 15290591125556738113U, 986 },
    { 11392378155556871081U, 1013 },  { 16975966327722178521U, 1039 },
    { 12648080533535911531U, 1066 }};

static dtoa_np dtoa_find_cachedpow10(int exp, int *k)
{
  const double one_log_ten = 0.30102999566398114;
  int32_t approx = -(exp + dtoa_npowers) * one_log_ten;
  int32_t idx = (approx - dtoa_firstpower) / dtoa_steppowers;
  while (true)
    {
      int32_t current = exp + dtoa_powers_ten[idx].exp + 64;
      if (current < dtoa_expmin)
	{
	  idx++;
	  continue;
        }
      if (current > dtoa_expmax)
	{
	  idx--;
	  continue;
        }
      *k = (dtoa_firstpower + idx * dtoa_steppowers);
      return(dtoa_powers_ten[idx]);
    }
}

#define dtoa_fracmask  0x000FFFFFFFFFFFFFU
#define dtoa_expmask   0x7FF0000000000000U
#define dtoa_hiddenbit 0x0010000000000000U
#define dtoa_signmask  0x8000000000000000U
#define dtoa_expbias   (1023 + 52)
#define dtoa_absv(n)   ((n) < 0 ? -(n) : (n))
#define dtoa_minv(a, b) ((a) < (b) ? (a) : (b))

static uint64_t dtoa_tens[] =
  { 10000000000000000000U, 1000000000000000000U, 100000000000000000U,
    10000000000000000U, 1000000000000000U, 100000000000000U,
    10000000000000U, 1000000000000U, 100000000000U,
    10000000000U, 1000000000U, 100000000U,
    10000000U, 1000000U, 100000U,
    10000U, 1000U, 100U,
    10U, 1U};

static uint64_t dtoa_get_dbits(double d)
{
  union {double dbl; uint64_t i;} dbl_bits = {d};
  return(dbl_bits.i);
}

static dtoa_np dtoa_build_np(double d)
{
  uint64_t bits = dtoa_get_dbits(d);
  dtoa_np fp;
  fp.frac = bits & dtoa_fracmask;
  fp.exp = (bits & dtoa_expmask) >> 52;
  if (fp.exp)
    {
      fp.frac += dtoa_hiddenbit;
      fp.exp -= dtoa_expbias;
    }
  else fp.exp = -dtoa_expbias + 1;
  return(fp);
}

static void dtoa_normalize(dtoa_np *fp)
{
  int32_t shift = 64 - 52 - 1;
  while ((fp->frac & dtoa_hiddenbit) == 0)
    {
      fp->frac <<= 1;
      fp->exp--;
    }
  fp->frac <<= shift;
  fp->exp -= shift;
}

static void dtoa_get_normalized_boundaries(dtoa_np *fp, dtoa_np *lower, dtoa_np *upper)
{
  int32_t u_shift, l_shift;
  upper->frac = (fp->frac << 1) + 1;
  upper->exp  = fp->exp - 1;
  while ((upper->frac & (dtoa_hiddenbit << 1)) == 0)
    {
      upper->frac <<= 1;
      upper->exp--;
    }
  u_shift = 64 - 52 - 2;
  upper->frac <<= u_shift;
  upper->exp = upper->exp - u_shift;
  l_shift = (fp->frac == dtoa_hiddenbit) ? 2 : 1;
  lower->frac = (fp->frac << l_shift) - 1;
  lower->exp = fp->exp - l_shift;
  lower->frac <<= lower->exp - upper->exp;
  lower->exp = upper->exp;
}

static dtoa_np dtoa_multiply(dtoa_np *a, dtoa_np *b) /* const dtoa_np* here and elsewhere is slower!  perverse */
{
  dtoa_np fp;
  const uint64_t lomask = 0x00000000FFFFFFFF;
  uint64_t ah_bl = (a->frac >> 32)    * (b->frac & lomask);
  uint64_t al_bh = (a->frac & lomask) * (b->frac >> 32);
  uint64_t al_bl = (a->frac & lomask) * (b->frac & lomask);
  uint64_t ah_bh = (a->frac >> 32)    * (b->frac >> 32);
  uint64_t tmp = (ah_bl & lomask) + (al_bh & lomask) + (al_bl >> 32);
  /* round up */
  tmp += 1U << 31;
  fp.frac = ah_bh + (ah_bl >> 32) + (al_bh >> 32) + (tmp >> 32);
  fp.exp = a->exp + b->exp + 64;
  return(fp);
}

static void dtoa_round_digit(char *digits, int32_t ndigits, uint64_t delta, uint64_t rem, uint64_t kappa, uint64_t frac)
{
  while ((rem < frac) && (delta - rem >= kappa) &&
	 ((rem + kappa < frac) || (frac - rem > rem + kappa - frac)))
    {
      digits[ndigits - 1]--;
      rem += kappa;
    }
}

static int32_t dtoa_generate_digits(dtoa_np *fp, dtoa_np *upper, dtoa_np *lower, char *digits, int *K)
{
  uint64_t part1, part2, wfrac = upper->frac - fp->frac, delta = upper->frac - lower->frac;
  uint64_t *unit;
  int32_t idx = 0, kappa = 10;
  dtoa_np one;

  one.frac = 1ULL << -upper->exp;
  one.exp  = upper->exp;
  part1 = upper->frac >> -one.exp;
  part2 = upper->frac & (one.frac - 1);

  /* 1000000000 */
  for (uint64_t *divp = dtoa_tens + 10; kappa > 0; divp++)
    {
      uint64_t tmp, div = *divp;
      unsigned digit = part1 / div;
      if (digit || idx)
	digits[idx++] = digit + '0';
      part1 -= digit * div;
      kappa--;
      tmp = (part1 << -one.exp) + part2;
      if (tmp <= delta)
	{
	  *K += kappa;
	  dtoa_round_digit(digits, idx, delta, tmp, div << -one.exp, wfrac);
	  return(idx);
        }}

  /* 10 */
  unit = dtoa_tens + 18;
  while(true)
    {
      unsigned digit;
      part2 *= 10;
      delta *= 10;
      kappa--;
      digit = part2 >> -one.exp;
      if (digit || idx)
	digits[idx++] = digit + '0';
      part2 &= one.frac - 1;
      if (part2 < delta)
	{
	  *K += kappa;
	  dtoa_round_digit(digits, idx, delta, part2, one.frac, wfrac * *unit);
	  return(idx);
	}
      unit--;
    }
}

static int32_t dtoa_grisu2(double d, char *digits, int *K)
{
  int32_t k;
  dtoa_np cp, lower, upper;
  dtoa_np w = dtoa_build_np(d);
  dtoa_get_normalized_boundaries(&w, &lower, &upper);
  dtoa_normalize(&w);
  cp = dtoa_find_cachedpow10(upper.exp, &k);
  w = dtoa_multiply(&w, &cp);
  upper = dtoa_multiply(&upper, &cp);
  lower = dtoa_multiply(&lower, &cp);
  lower.frac++;
  upper.frac--;
  *K = -k;
  return(dtoa_generate_digits(&w, &upper, &lower, digits, K));
}

static int32_t dtoa_emit_digits(char *digits, int32_t ndigits, char *dest, int32_t K, bool neg)
{
  int32_t idx, cent;
  char sign;
  int32_t exp = dtoa_absv(K + ndigits - 1);

  /* write plain integer */
  if ((K >= 0) && (exp < (ndigits + 7)))
    {
      memcpy(dest, digits, ndigits);
      local_memset(dest + ndigits, '0', K); /* unaligned */
      dest[ndigits + K] = '.';
      dest[ndigits + K + 1] = '0';
      return(ndigits + K + 2);
    }

  /* write decimal w/o scientific notation */
  if ((K < 0) && (K > -7 || exp < 4))
    {
      int32_t offset = ndigits - dtoa_absv(K);
      /* fp < 1.0 -> write leading zero */
      if (offset <= 0)
	{
	  offset = -offset;
	  dest[0] = '0';
	  dest[1] = '.';
	  local_memset(dest + 2, '0', offset); /* unaligned */
	  memcpy(dest + offset + 2, digits, ndigits);
	  return(ndigits + 2 + offset);
	  /* fp > 1.0 */
	}
      else
	{
	  memcpy(dest, digits, offset);
	  dest[offset] = '.';
	  memcpy(dest + offset + 1, digits + offset, ndigits - offset);
	  return(ndigits + 1);
	}}

  /* write decimal w/ scientific notation */
  ndigits = dtoa_minv(ndigits, 18 - neg);
  idx = 0;
  dest[idx++] = digits[0];
  if (ndigits > 1)
    {
      dest[idx++] = '.';
      memcpy(dest + idx, digits + 1, ndigits - 1);
      idx += ndigits - 1;
    }
  dest[idx++] = 'e';
  sign = K + ndigits - 1 < 0 ? '-' : '+';
  dest[idx++] = sign;
  cent = 0;
  if (exp > 99)
    {
      cent = exp / 100;
      dest[idx++] = cent + '0';
      exp -= cent * 100;
    }
  if (exp > 9)
    {
      int32_t dec = exp / 10;
      dest[idx++] = dec + '0';
      exp -= dec * 10;
    }
  else
    if (cent)
      dest[idx++] = '0';

  dest[idx++] = exp % 10 + '0';
  return(idx);
}

static int32_t dtoa_filter_special(double fp, char *dest, bool neg)
{
  uint64_t bits;
  bool nan;
  if (fp == 0.0)
    {
      dest[0] = '0'; dest[1] = '.'; dest[2] = '0';
      return(3);
    }
  bits = dtoa_get_dbits(fp);
  nan = (bits & dtoa_expmask) == dtoa_expmask;
  if (!nan) return(0);

  if (!neg)
    {
      dest[0] = '+'; /* else 1.0-nan...? */
      dest++;
    }
  if (bits & dtoa_fracmask)
    {
      s7_int payload = nan_payload(fp);
      int32_t len;
      len = (int32_t)snprintf(dest, 22, "nan.%" ld64, payload); /* dest size = 23, below */
      return((neg) ? len : len + 1);
    }
  dest[0] = 'i'; dest[1] = 'n'; dest[2] = 'f'; dest[3] = '.'; dest[4] = '0';
  return((neg) ? 5 : 6);
}

static inline int32_t fpconv_dtoa(double d, char dest[24])
{
  char digit[23];
  int32_t str_len = 0, spec, K, ndigits;
  bool neg = false;

  if (dtoa_get_dbits(d) & dtoa_signmask)
    {
      dest[0] = '-';
      str_len++;
      neg = true;
    }
  spec = dtoa_filter_special(d, dest + str_len, neg);
  if (spec) return(str_len + spec);
  K = 0;
  ndigits = dtoa_grisu2(d, digit, &K);
  str_len += dtoa_emit_digits(digit, ndigits, dest + str_len, K, neg);
  return(str_len);
}
#endif


/* -------------------------------- number->string -------------------------------- */
static const char dignum[] = "0123456789abcdef";

static size_t integer_to_string_any_base(char *p, s7_int n, int32_t radix)  /* called by number_to_string_with_radix */
{
  s7_int i, len, end;
  const bool sign = (n < 0);
  s7_int pown;

  if ((radix < 2) || (radix > 16))
    return(0);
  if (sign)
    {
      if (n == s7_Int64_Min) /* can't negate this, so do it by hand */
	{
	  static const char *mnfs[17] = {"","",
					 "-1000000000000000000000000000000000000000000000000000000000000000", "-2021110011022210012102010021220101220222",
					 "-20000000000000000000000000000000", "-1104332401304422434310311213", "-1540241003031030222122212",
					 "-22341010611245052052301", "-1000000000000000000000", "-67404283172107811828", "-9223372036854775808",
					 "-1728002635214590698", "-41a792678515120368", "-10b269549075433c38", "-4340724c6c71dc7a8", "-160e2ad3246366808",
					 "-8000000000000000"};
	  len = safe_strlen(mnfs[radix]);
	  memcpy((void *)p, (const void *)mnfs[radix], len);
	  p[len] = '\0';
	  return(len);
	}
      n = -n;
    }
  /* the previous version that counted up to n, rather than dividing down below n, as here, could be confused by large ints on 64 bit machines  */
  pown = n;
  for (i = 1; i < 100; i++)
    {
      if (pown < radix)
	break;
      pown /= (s7_int)radix;
    }
  len = i - 1;
  if (sign) len++;
  end = 0;
  if (sign)
    {
      p[0] = '-';
      end++;
    }
  for (i = len; i >= end; i--)
    {
      p[i] = dignum[n % radix];
      n /= radix;
    }
  p[len + 1] = '\0';
  return(len + 1);
}

static const char *integer_to_string(s7_scheme *sc, s7_int num, s7_int *nlen) /* do not free the returned string */
{
  char *p, *op;
  bool sign = (num < 0);
  if (sign)
    {
      if (num == s7_Int64_Min)
	{
	  (*nlen) = 20;
	  return((const char *)"-9223372036854775808");
	}
      num = -num;  /* we need a positive index below */
    }
  p = (char *)(sc->int_to_str1 + Int_To_Str_Size - 1);
  op = p;
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      (*nlen) = op - p;
      return(p);
    }
  (*nlen) = op - p - 1;
  return(++p);
}

static const char *integer_to_string_no_length(s7_scheme *sc, s7_int num) /* do not free the returned string */
{
  char *p;
  const bool sign = (num < 0);
  if (sign)
    {
      if (num == s7_Int64_Min)
	return("-9223372036854775808");
      num = -num;
    }
  p = (char *)(sc->int_to_str2 + Int_To_Str_Size - 1);
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      return(p);
    }
  return(++p);
}

static char *floatify(char *str, s7_int *nlen)
{
  if ((!strchr(str, '.')) && (!strchr(str, 'e'))) /* faster than (strcspn(str, ".e") >= (size_t)(*nlen)) */
    {
      s7_int len = *nlen;
      /* snprintf returns "nan" and "inf" but we (stupidly) want "+nan.0" and "+inf.0"; "-nan" and "-inf" will be handled by the normal case */
      if (len == 3)
	{
	  if (str[0] == 'n')
	    {
	      str[0] = '+'; str[1] = 'n'; str[2] = 'a'; str[3] = 'n'; /* need to retain 'str' as output */
	      len = 4;
	    }
	  else
	    if (str[0] == 'i')
	      {
		str[0] = '+'; str[1] = 'i'; str[2] = 'n'; str[3] = 'f';
		len = 4;
	      }}
      str[len]='.';
      str[len + 1]='0';
      str[len + 2]='\0';
      (*nlen) = len + 2;
    }
  return(str);
}

static void insert_spaces(s7_scheme *sc, const char *src, s7_int width, s7_int len)
{
  s7_int spaces = width - len;
  if (width >= sc->num_to_str_size)
    {
      sc->num_to_str_size = width + 1;
      sc->num_to_str = (char *)Realloc(sc->num_to_str, sc->num_to_str_size);
    }
  sc->num_to_str[width] = '\0';
  memmove((void *)(sc->num_to_str + spaces), (const void *)src, len);
  local_memset((void *)(sc->num_to_str), (int)' ', spaces);
}

static char *number_to_string_base_10(s7_scheme *sc, s7_pointer obj, s7_int width, s7_int precision,
				      char float_choice, s7_int *nlen, use_write_t choice) /* don't free result */
{
  /* called by number_to_string_with_radix g_number_to_string, number_to_string_p_p number_to_port format_number */
  /* the rest of s7 assumes nlen is set to the correct length
   *   a tricky case: (format #f "~f" 1e308) -- tries to print 308 digits! so 256 as default len is too small.
   *   but then even worse: (format #f "~F" 1e308+1e308i)!
   */
  s7_int len = width + precision;
  len = (len > 512) ? (512 + 2 * len) : 1024;
  if (len > sc->num_to_str_size)
    {
      sc->num_to_str = (sc->num_to_str) ? (char *)Realloc(sc->num_to_str, len) : (char *)Malloc(len);
      sc->num_to_str_size = len;
    }

  /* bignums can't happen here */
  if ((s7_Debugging) && (is_big_number(obj))) fprintf(stderr, "%s[%d]: we got a bignum: %s\n", __func__, __LINE__, display(obj));
  if (is_t_integer(obj))
    {
      const char *p;
      if (width == 0)
	return((char *)integer_to_string(sc, integer(obj), nlen));
      p = integer_to_string(sc, integer(obj), &len);
      if (width > len)
	{
	  insert_spaces(sc, p, width, len); /* writes sc->num_to_str */
	  (*nlen) = width;
	  return(sc->num_to_str);
	}
      (*nlen) = len;
      return((char *)p);
    }

  if (is_t_real(obj))
    {
      if (width == 0)
	{
#if WITH_DTOA
	  if ((float_choice == 'g') &&
	      (precision == Write_Real_Precision)) /* set to 6 in format! need ~,16G to hit this code */
	    {
	      /* (number->string 0.0000001) is sensitive to (*s7* 'float-format-precision) and inconsistent: either 1e-7 or 0.0000001
	       *    because fpconv_dtoa has some complicated decision about 'g' vs 'f' -- not sure if this is a bug.
	       */
	      if (obj == real_pi)
		{
		  static const char pi_name[] = "3.141592653589793"; /* array form for ISO C++ */
		  (*nlen) = 17;
		  memcpy((void *)(sc->num_to_str), (void *)pi_name, 17);
		  return(sc->num_to_str); /* code below assumes we return sc->num_to_str in this case -- ugly! */
		}
	      len = fpconv_dtoa(real(obj), sc->num_to_str);
	      sc->num_to_str[len] = '\0';
	      (*nlen) = len;
	      return(sc->num_to_str);
	    }
#endif
	  len = snprintf(sc->num_to_str, sc->num_to_str_size - 4,
			 (float_choice == 'g') ? "%.*g" : ((float_choice == 'f') ? "%.*f" : "%.*e"),
			 (int32_t)precision, real(obj)); /* -4 for floatify */
	}
      else len = snprintf(sc->num_to_str, sc->num_to_str_size - 4,
			  (float_choice == 'g') ? "%*.*g" : ((float_choice == 'f') ? "%*.*f" : "%*.*e"),
			  (int32_t)width, (int32_t)precision, real(obj)); /* -4 for floatify */
      (*nlen) = len;
      floatify(sc->num_to_str, nlen);
      return(sc->num_to_str);
    }

  if (is_t_complex(obj))
    {
      char *imag;
      sc->num_to_str[0] = '\0';
      imag = copy_string(number_to_string_base_10(sc, wrap_real(sc, imag_part(obj)), 0, precision, float_choice, &len, choice));

      sc->num_to_str[0] = '\0';
      number_to_string_base_10(sc, wrap_real(sc, real_part(obj)), 0, precision, float_choice, &len, choice);

      sc->num_to_str[len] = '\0';
      len = catstrs(sc->num_to_str, sc->num_to_str_size, ((imag[0] == '+') || (imag[0] == '-')) ? "" : "+", imag, "i", (char *)NULL);
      free(imag);

      if (width > len)  /* (format #f "~20g" 1+i) */
	{
	  insert_spaces(sc, sc->num_to_str, width, len); /* this checks sc->num_to_str_size */
	  (*nlen) = width;
	}
      else (*nlen) = len;
      return(sc->num_to_str);
    }

  /* ratio */
  len = catstrs_direct(sc->num_to_str, integer_to_string_no_length(sc, numerator(obj)), "/", pos_int_to_str_direct(sc, denominator(obj)), (const char *)NULL);
  if (width > len)
    {
      insert_spaces(sc, sc->num_to_str, width, len);
      (*nlen) = width;
    }
  else (*nlen) = len;
  return(sc->num_to_str);
}

#define Base_10 10

static block_t *number_to_string_with_radix(s7_scheme *sc, s7_pointer obj, int32_t radix, s7_int width, s7_int precision, char float_choice, s7_int *nlen)
{
  /* called by s7_number_to_string (char*), g_number_to_string (strp), number_to_string_p_pp (strp), format_number (strp basically) */
  /* the rest of s7 assumes nlen is set to the correct length */
  block_t *b;
  char *p;
  s7_int len, str_len;

#if With_Gmp
  if (s7_is_bignum(obj))
    return(big_number_to_string_with_radix(sc, obj, radix, width, nlen, p_Write));
  /* this ignores precision because it's way too hard to get the mpfr string to look like
   *   C's output -- we either have to call mpfr_get_str twice (the first time just to
   *   find out what the exponent is and how long the string actually is), or we have
   *   to do messy string manipulations.  So (format #f "",3F" pi) ignores the "3" and
   *   prints the full string.  And don't even think about mpfr_snprintf!
   */
#endif
  if (radix == Base_10)
    {
      p = number_to_string_base_10(sc, obj, width, precision, float_choice, nlen, p_Write);
      return(string_to_block(sc, p, *nlen));
    }

  switch (type(obj))
    {
    case T_INTEGER:
      {
	size_t len1;
	b = inline_mallocate(sc, (128 + width));
	p = (char *)block_data(b);
	len1 = integer_to_string_any_base(p, integer(obj), radix);
	if ((size_t)width > len1)
	  {
	    size_t start = width - len1;
	    memmove((void *)(p + start), (void *)p, len1);
	    local_memset((void *)p, (int)' ', start);
	    p[width] = '\0';
	    *nlen = width;
	  }
	else *nlen = len1;
	return(b);
      }
    case T_RATIO:
      {
	size_t len1, len2;
	str_len = 256 + width;
	b = inline_mallocate(sc, str_len);
	p = (char *)block_data(b);
	len1 = integer_to_string_any_base(p, numerator(obj), radix);
	p[len1] = '/';
	len2 = integer_to_string_any_base((char *)(p + len1 + 1), denominator(obj), radix);
        len = len1 + 1 + len2;
        p[len] = '\0';
      }
      break;
    case T_REAL:
      {
	int32_t i;
	s7_int int_part, nsize;
	s7_double x = real(obj), frac_part, min_frac, base;
	bool sign = false;
	char n[128], d[256];

	if (is_NaN(x))
	  return(string_to_block(sc, "+nan.0", *nlen = 6));
	if (is_inf(x))
	  {
	    if (x < 0.0)
	      return(string_to_block(sc, "-inf.0", *nlen = 6));
	    return(string_to_block(sc, "+inf.0", *nlen = 6));
	  }
	if (x < 0.0)
	  {
	    sign = true;
	    x = -x;
	  }
	if (x > 1.0e18) /* i.e. close to or greater than most-positive-fixnum (9.22e18), so the code below is unlikely to work, (format #f "~X" 1e19) */
	  {
	    int32_t ep = (int32_t)floor(log(x) / log((double)radix));
	    block_t *b1;
	    len = 0;
	    b = number_to_string_with_radix(sc, wrap_real(sc, x / pow((double)radix, (double)ep)), /* divide it down to one digit, then the fractional part */
					    radix, width, precision, float_choice, &len);
	    b1 = inline_mallocate(sc, len + 8);
	    p = (char *)block_data(b1);
	    p[0] = '\0';
	    (*nlen) = catstrs(p, len + 8, (sign) ? "-" : "", (char *)block_data(b), (radix == 16) ? "@" : "e", integer_to_string_no_length(sc, ep), (char *)NULL);
	    liberate(sc, b);
	    return(b1);
	  }
	int_part = (s7_int)floor(x);
	frac_part = x - int_part;
	nsize = integer_to_string_any_base(n, int_part, radix);
	min_frac = dpow(radix, -precision);

	/* doesn't this assume precision < 128/256 and that we can fit in 256 digits (1e308)? */
	for (i = 0, base = radix; (i < precision) && (frac_part > min_frac); i++, base *= radix)
	  {
	    s7_int ipart = (s7_int)(frac_part * base);
	    if (ipart >= radix)         /* rounding confusion */
	      ipart = radix - 1;
	    frac_part -= (ipart / base);
	    /* d[i] = ((const char *)"0123456789abcdef")[ipart]; */
	    d[i] = dignum[ipart];
	  }
	if (i == 0)
	  d[i++] = '0';
	d[i] = '\0';
	#define Str_Len 256
	b = inline_mallocate(sc, Str_Len);
        p = (char *)block_data(b);
	/* much faster than catstrs because we know the string lengths */
	{
	  char *pt = p;
	  if (sign) {pt[0] = '-'; pt++;}
	  memcpy(pt, n, nsize);
	  pt += nsize;
	  pt[0] = '.';
	  pt++;
	  memcpy(pt, d, i);
	  pt[i] = '\0';
	  /* len = ((sign) ? 1 : 0) + 1 + nsize + i; */
	  len = pt + i - p;
	}
	str_len = Str_Len;
      }
      break;

    default:
      {
	char *pt;
	s7_int real_len = 0, imag_len = 0;
	block_t *n = number_to_string_with_radix(sc, wrap_real(sc, real_part(obj)), radix, 0, precision, float_choice, &real_len); /* include floatify */
	block_t *d = number_to_string_with_radix(sc, wrap_real(sc, imag_part(obj)), radix, 0, precision, float_choice, &imag_len);
	const char *dp = (const char *)block_data(d);
	#define Another_Str_Len 512
	b = inline_mallocate(sc, Another_Str_Len);
	p = (char *)block_data(b);
	pt = p;
	memcpy(pt, (void *)block_data(n), real_len);
	pt += real_len;
	if ((dp[0] != '+') && (dp[0] != '-')) {pt[0] = '+'; pt++;}
	memcpy(pt, dp, imag_len);
	pt[imag_len] = 'i';
	pt[imag_len + 1] = '\0';
	len = pt + imag_len + 1 - p;
	str_len = Another_Str_Len;
	liberate(sc, n);
	liberate(sc, d);
      }
      break;
    }
  if (width > len)
    {
      s7_int spaces;
      if (width >= str_len)
	{
	  str_len = width + 1;
	  b = reallocate(sc, b, str_len);
	  p = (char *)block_data(b);
	}
      spaces = width - len;
      p[width] = '\0';
      memmove((void *)(p + spaces), (void *)p, len);
      local_memset((void *)p, (int)' ', spaces);
      (*nlen) = width;
    }
  else (*nlen) = len;
  return(b);
}

char *s7_number_to_string(s7_scheme *sc, s7_pointer obj, s7_int radix)
{
  s7_int nlen = 0;
  block_t *b = number_to_string_with_radix(sc, obj, radix, 0, 20, 'g', &nlen);  /* (log top 10) so we get all the digits in base 10 (??) */
  char *str = copy_string_with_length((char *)block_data(b), nlen);
  liberate(sc, b);
  return(str);
}

static s7_pointer g_number_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_number_to_string "(number->string num (radix 10)) converts the number num into a string."
  #define Q_number_to_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_number_symbol, sc->is_integer_symbol)

  s7_int nlen = 0, radix; /* ignore cppcheck complaint about radix! */
  const char *result;
  s7_pointer x = car(args);

  if (!is_number(x))
    return(method_or_bust(sc, x, sc->number_to_string_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer base = cadr(args);
      if (s7_is_integer(base))
	radix = s7_integer_clamped_if_gmp(sc, base);
      else return(method_or_bust(sc, base, sc->number_to_string_symbol, args, sc->type_names[T_INTEGER], 2));
      if ((radix < 2) || (radix > 16))
	out_of_range_error_nr(sc, sc->number_to_string_symbol, int_two, base, a_valid_radix_string);
#if With_Gmp
      if (!s7_is_bignum(x))
#endif
	{
	  block_t *b = number_to_string_with_radix(sc, x, radix, 0, sc->float_format_precision, 'g', &nlen);
	  return(block_to_string(sc, b, nlen));
	}}
#if With_Gmp
  else radix = 10;
  if (s7_is_bignum(x))
    {
      block_t *b = big_number_to_string_with_radix(sc, x, radix, 0, &nlen, p_Write);
      return(block_to_string(sc, b, nlen));
    }
  result = number_to_string_base_10(sc, x, 0, sc->float_format_precision, 'g', &nlen, p_Write);
#else
  if (is_t_integer(x))
    result = integer_to_string(sc, integer(x), &nlen);
  else result = number_to_string_base_10(sc, x, 0, sc->float_format_precision, 'g', &nlen, p_Write);
#endif
  return(inline_make_string_with_length(sc, result, nlen));
}

static s7_pointer number_to_string_p_p(s7_scheme *sc, s7_pointer p)
{
#if With_Gmp
  return(g_number_to_string(sc, set_plist_1(sc, p)));
#else
  s7_int nlen = 0;
  char *result;
  if (!is_number(p))
    return(method_or_bust_p(sc, p, sc->number_to_string_symbol, a_number_string));
  result = number_to_string_base_10(sc, p, 0, sc->float_format_precision, 'g', &nlen, p_Write);
  return(inline_make_string_with_length(sc, result, nlen));
#endif
}

static s7_pointer number_to_string_p_i(s7_scheme *sc, s7_int p)
{
  s7_int nlen = 0;
  const char *result = integer_to_string(sc, p, &nlen);
  return(inline_make_string_with_length(sc, result, nlen));
}
/* not number_to_string_p_d! */

static s7_pointer number_to_string_p_pp(s7_scheme *sc, s7_pointer num, s7_pointer base)
{
#if With_Gmp
  return(g_number_to_string(sc, set_plist_2(sc, num, base)));
#else
  s7_int nlen = 0, radix;
  block_t *b;

  if (!is_number(num))
    wrong_type_error_nr(sc, sc->number_to_string_symbol, 1, num, a_number_string);
  if (!is_t_integer(base))
    wrong_type_error_nr(sc, sc->number_to_string_symbol, 2, base, sc->type_names[T_INTEGER]);
  radix = integer(base);
  if ((radix < 2) || (radix > 16))
    out_of_range_error_nr(sc, sc->number_to_string_symbol, int_two, base, a_valid_radix_string);
  b = number_to_string_with_radix(sc, num, radix, 0, sc->float_format_precision, 'g', &nlen);
  return(block_to_string(sc, b, nlen));
#endif
}


/* -------------------------------------------------------------------------------- */
#define Ctable_Size 256
static bool *exponent_table, *slashify_table, *char_ok_in_a_name, *white_space, *number_table, *symbol_slashify_table;
static int32_t *digits;

static void init_ctables(void)
{
  exponent_table = (bool *)Calloc(Ctable_Size, sizeof(bool));
  slashify_table = (bool *)Calloc(Ctable_Size, sizeof(bool));
  symbol_slashify_table = (bool *)Calloc(Ctable_Size, sizeof(bool));
  char_ok_in_a_name = (bool *)Malloc(Ctable_Size * sizeof(bool));
  white_space = (bool *)Calloc(Ctable_Size + 1, sizeof(bool));
  white_space++;      /* leave white_space[-1] false for white_space[EOF] */
  number_table = (bool *)Calloc(Ctable_Size, sizeof(bool));
  digits = (int32_t *)Malloc(Ctable_Size * sizeof(int32_t));

  for (int32_t i = 0; i < Ctable_Size; i++)
    {
      char_ok_in_a_name[i] = true;
      /* white_space[i] = false; */
      digits[i] = 256;
      /* number_table[i] = false; */
    }

  char_ok_in_a_name[0] = false;
  char_ok_in_a_name[(uint8_t)'('] = false;  /* cast for C++ */
  char_ok_in_a_name[(uint8_t)')'] = false;
  char_ok_in_a_name[(uint8_t)';'] = false;
  char_ok_in_a_name[(uint8_t)'\t'] = false;
  char_ok_in_a_name[(uint8_t)'\n'] = false;
  char_ok_in_a_name[(uint8_t)'\r'] = false;
  char_ok_in_a_name[(uint8_t)' '] = false;
  char_ok_in_a_name[(uint8_t)'"'] = false;

  white_space[(uint8_t)'\t'] = true;
  white_space[(uint8_t)'\n'] = true;
  white_space[(uint8_t)'\r'] = true;
  white_space[(uint8_t)'\f'] = true;
  white_space[(uint8_t)'\v'] = true;
  white_space[(uint8_t)' '] = true;
  white_space[(uint8_t)'\205'] = true; /* 133 */
  white_space[(uint8_t)'\240'] = true; /* 160 */

  /* surely only 'e' is needed... */
  exponent_table[(uint8_t)'e'] = true; exponent_table[(uint8_t)'E'] = true;
  exponent_table[(uint8_t)'@'] = true;
#if With_Extra_Exponent_Markers
  exponent_table[(uint8_t)'s'] = true; exponent_table[(uint8_t)'S'] = true;
  exponent_table[(uint8_t)'f'] = true; exponent_table[(uint8_t)'F'] = true;
  exponent_table[(uint8_t)'d'] = true; exponent_table[(uint8_t)'D'] = true;
  exponent_table[(uint8_t)'l'] = true; exponent_table[(uint8_t)'L'] = true;
#endif
  for (int32_t i = 0; i < 32; i++) slashify_table[i] = true;
  /* for (int32_t i = 127; i < 160; i++) slashify_table[i] = true; */ /* 6-Apr-24 for utf-8, but this has no effect on s7test?? */
  slashify_table[(uint8_t)'\\'] = true;
  slashify_table[(uint8_t)'"'] = true;
#if With_r7rs
  /* In R7RS mode, newlines should be escaped to ensure proper serialization */
  slashify_table[(uint8_t)'\n'] = true;
#else
   slashify_table[(uint8_t)'\n'] = false;
#endif

  for (int32_t i = 0; i < Ctable_Size; i++)
    symbol_slashify_table[i] = ((slashify_table[i]) || (!char_ok_in_a_name[i])); /* force use of (symbol ...) for cases like '(ab) as symbol */

  digits[(uint8_t)'0'] = 0; digits[(uint8_t)'1'] = 1; digits[(uint8_t)'2'] = 2; digits[(uint8_t)'3'] = 3; digits[(uint8_t)'4'] = 4;
  digits[(uint8_t)'5'] = 5; digits[(uint8_t)'6'] = 6; digits[(uint8_t)'7'] = 7; digits[(uint8_t)'8'] = 8; digits[(uint8_t)'9'] = 9;
  digits[(uint8_t)'a'] = 10; digits[(uint8_t)'A'] = 10;
  digits[(uint8_t)'b'] = 11; digits[(uint8_t)'B'] = 11;
  digits[(uint8_t)'c'] = 12; digits[(uint8_t)'C'] = 12;
  digits[(uint8_t)'d'] = 13; digits[(uint8_t)'D'] = 13;
  digits[(uint8_t)'e'] = 14; digits[(uint8_t)'E'] = 14;
  digits[(uint8_t)'f'] = 15; digits[(uint8_t)'F'] = 15;

  number_table[(uint8_t)'0'] = true; number_table[(uint8_t)'1'] = true; number_table[(uint8_t)'2'] = true; number_table[(uint8_t)'3'] = true;
  number_table[(uint8_t)'4'] = true; number_table[(uint8_t)'5'] = true; number_table[(uint8_t)'6'] = true; number_table[(uint8_t)'7'] = true;
  number_table[(uint8_t)'8'] = true; number_table[(uint8_t)'9'] = true; number_table[(uint8_t)'.'] = true;
  number_table[(uint8_t)'+'] = true;
  number_table[(uint8_t)'-'] = true;
  number_table[(uint8_t)'#'] = true;
}

#define is_white_space(C) white_space[C]
  /* this is much faster than C's isspace, and does not depend on the current locale.
   * if c == EOF (-1), it indexes into the empty (0) slot we preallocated below white_space
   */

/* -------------------------------- *#readers* -------------------------------- */
static s7_pointer check_sharp_readers(s7_scheme *sc, const char *name)
{
  s7_pointer value = sc->F;
  const bool need_loader_port = is_loader_port(current_input_port(sc));

  /* *#reader* is assumed to be an alist of (char . proc)
   *    where each proc takes one argument, the string from just beyond the "#" to the next delimiter.
   *    The procedure can call read-char to read ahead in the current-input-port.
   *    If it returns anything other than #f, that is the value of the sharp expression.
   *    Since #f means "nothing found", it is tricky to handle #F:
   *       (cons #\F (lambda (str) (and (string=? str "F") (list 'not #t)))) ; or ''#f used in lint.scm
   * This search happens after #|, #t, and #f (and #nD for multivectors?). #! has a fallback.  Added #_ later)
   */
  if (need_loader_port)
    clear_loader_port(current_input_port(sc));

  /* normally read* can't read from current_input_port(sc) if it is in use by the loader, but here we are deliberately making that possible */
  for (s7_pointer args = sc->F, reader = slot_value(sc->sharp_readers); is_pair(reader); reader = cdr(reader))
    if (name[0] == s7_character(caar(reader)))
      {
	if (args == sc->F)
	  args = set_plist_1(sc, wrap_string(sc, name, safe_strlen(name)));
	/* args is GC protected by s7_apply_function?? (placed on the stack) */
	value = s7_apply_function(sc, cdar(reader), args); /* this is much less error-safe than s7_call */
	if (value != sc->F)
	  break;
      }
  if (need_loader_port)
    set_loader_port(current_input_port(sc));
  return(value);
}

static s7_pointer g_sharp_readers_set(s7_scheme *sc, s7_pointer args)
{
  /* new value must be either () or a proper list of conses (char . func) */
  s7_pointer readers;
  if (is_null(cadr(args))) return(sc->nil);
  if (!is_pair(cadr(args)))
    error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args)));
  for (readers = cadr(args); is_pair(readers); readers = cdr(readers))
    if ((!is_pair(car(readers))) ||
	(!is_character(caar(readers))) ||
	(!is_procedure(cdar(readers))))
      error_nr(sc, sc->wrong_type_arg_symbol,
	       set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args)));
  if (!is_null(readers))
    error_nr(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args)));
  return(cadr(args));
}

static s7_pointer make_undefined(s7_scheme *sc, const char *name)
{
  const s7_int len = safe_strlen(name);
  char *newstr = (char *)Malloc(len + 2);
  s7_pointer undef;
  new_cell(sc, undef, T_UNDEFINED | T_Immutable);
  newstr[0] = '#';
  memcpy((void *)(newstr + 1), (const void *)name, len);
  newstr[len + 1] = '\0';
  if (sc->undefined_constant_warnings) s7_warn(sc, len + 32, "%s is undefined\n", newstr);
  undefined_set_name_length(undef, len + 1);
  undefined_name(undef) = newstr;
  add_undefined(sc, undef);
  return(undef);
}

static int32_t inchar(s7_pointer port)
{
  int32_t c;
  if (is_file_port(port))
    c = fgetc(port_file(port)); /* not uint8_t! -- could be EOF */
  else
    {
      if (port_data_size(port) <= port_position(port))
	return(EOF);
      c = (uint8_t)port_data(port)[port_position(port)++];
    }
  if (c == '\n')
    port_line_number(port)++;
  return(c);
}

static void backchar(char c, s7_pointer port)
{
  if (c == '\n')
    port_line_number(port)--;
  if (is_file_port(port))
    ungetc(c, port_file(port));
  else
    if (port_position(port) > 0)
      port_position(port)--;
}

static void resize_strbuf(s7_scheme *sc, s7_int needed_size)
{
  s7_int old_size = sc->strbuf_size;
  while (sc->strbuf_size <= needed_size) sc->strbuf_size *= 2;
  sc->strbuf = (char *)Realloc(sc->strbuf, sc->strbuf_size);
  for (s7_int i = old_size; i < sc->strbuf_size; i++) sc->strbuf[i] = '\0';
}

static s7_pointer *chars;

static s7_pointer unknown_sharp_constant(s7_scheme *sc, const char *name, s7_pointer port)
{
  /* if name[len - 1] != '>' there's no > delimiter at the end */
  if (hook_has_functions(sc->read_error_hook))  /* check *read-error-hook* */
    {
      bool old_history_enabled = s7_set_history_enabled(sc, false); /* see sc->error_hook for a more robust way to handle this */
      s7_pointer result = s7_call(sc, sc->read_error_hook, set_plist_2(sc, sc->T, wrap_string(sc, name, safe_strlen(name))));
      s7_set_history_enabled(sc, old_history_enabled);
      if (result != sc->unspecified)
	return(result);
    }
  if (port) /* #<"..."> which gets here as name="#<" */
    {
      const s7_int len = safe_strlen(name);
      if ((name[len - 1] != '>') &&
	  (is_input_port(port)) &&
	  (port != sc->standard_input))
	{
	  if (s7_peek_char(sc, port) != chars[(uint8_t)'"']) /* if not #<"...">, just return it */
	    return(make_undefined(sc, name));
	  /* PERHAPS: strchr port-data '>'?? it might be #<x y> etc -- what would this break? maybe extend section below */

	  if (is_string_port(port)) /* probably unnecessary (see below) */
	    {
	      s7_int c = inchar(port);
	      const char *pstart = (const char *)(port_data(port) + port_position(port));
	      const char *p = strchr(pstart, (int)'"');
	      s7_int added_len;
	      char *buf;
	      if (!p)
		{
		  backchar(c, port);
		  return(make_undefined(sc, name));
		}
	      p++;
	      while (char_ok_in_a_name[(uint8_t)(*p)]) {p++;}
	      added_len = (s7_int)(p - pstart);                 /* p is one past '>' presumably */
	      /* we can't use strbuf here -- it might be the source of the "name" argument! */
	      buf = (char *)Malloc(len + added_len + 2);
	      memcpy((void *)buf, (const void *)name, len);
	      buf[len] = '"';                                   /* from inchar */
	      memcpy((void *)(buf + len + 1), (const void *)pstart, added_len);
	      buf[len + added_len + 1] = 0;
	      port_position(port) += added_len;
	      {
		s7_pointer result = make_undefined(sc, (const char *)buf);
		free(buf);
		return(result);
	      }}}}
  return(make_undefined(sc, name));
}

static s7_pointer make_atom(s7_scheme *sc, char *q, int32_t radix, bool want_symbol, bool with_error);
#define Symbol_Ok true
#define No_Symbols false

static s7_pointer make_sharp_constant(s7_scheme *sc, const char *name, bool with_error, s7_pointer port, bool error_if_bad_number)
{
  /* name is the stuff after the '#', return sc->nil if not a recognized #... entity */
  if ((!name) || (!*name)) /* (string->number "#") for example */
    return(make_undefined(sc, name));

  /* stupid r7rs special cases */
  if ((name[0] == 't') &&
      ((name[1] == '\0') || (c_strings_are_equal(name, "true"))))
    return(sc->T);

  if ((name[0] == 'f') &&
      ((name[1] == '\0') || (c_strings_are_equal(name, "false"))))
    return(sc->F);

  if (name[0] == '_')
    {
      /* we handle #_ before looking at *#readers* below (via check_sharp_readers) because #_ needs to be unsettable via *#readers*:
       *    (set! *#readers* (list (cons #\_ (lambda (str) (string->symbol (substring str 1))))))
       *    (let ((+ -)) (#_+ 1 2)): -1
       */
      s7_pointer sym = make_symbol_with_strlen(sc, (const char *)(name + 1));
      if ((!is_gensym(sym)) && (initial_value_is_defined(sc, sym)))
#if 0
	return(initial_value(sym));
#else
	{
	  if (!is_initial_value(initial_value(sym)))
	    copy_initial_value(sc, sym);
	  return(initial_value(sym));
	}
#endif
      /* here we should not necessarily raise an error that *_... is undefined.  reader-cond, for example, needs to
       *    read undefined #_ vals that it will eventually discard.
       */
      return(make_undefined(sc, name));    /* (define x (with-input-from-string "(#_asdf 1 2)" read)) (type-of (car x)) -> undefined? */
    }

  if (is_pair(slot_value(sc->sharp_readers)))
    {
      s7_pointer x = check_sharp_readers(sc, name);
      if (x != sc->F)
	return(x);
    }

  if ((name[0] == '\0') || name[1] == '\0')
    return(unknown_sharp_constant(sc, name, port)); /* port here because #<"..."> comes here as "<" so name[1] is '\0'! */

  switch (name[0])
    {
      /* -------- #< ... > -------- */
    case '<':
      if (c_strings_are_equal(name, "<unspecified>")) return(sc->unspecified);
      if (c_strings_are_equal(name, "<undefined>"))   return(sc->undefined);
      if (c_strings_are_equal(name, "<eof>"))         return(eof_object);
      return(unknown_sharp_constant(sc, name, port));

      /* -------- #o #x #b -------- */
    case 'o':   /* #o (octal) */
    case 'x':   /* #x (hex) */
    case 'b':   /* #b (binary) */
      {
	s7_pointer result = make_atom(sc, (char *)(name + 1), (name[0] == 'o') ? 8 : ((name[0] == 'x') ? 16 : 2), No_Symbols, with_error);
	if ((error_if_bad_number) && (result == sc->F)) /* #b32 etc but not if called from string->number */
	  error_nr(sc, sc->read_error_symbol, set_elist_2(sc, wrap_string(sc, "#~A is not a number", 19), wrap_string(sc, name, safe_strlen(name))));
	return(result);
      }

      /* -------- #\... -------- */
    case '\\':
      if (name[2] == 0)                             /* the most common case: #\a */
	return(chars[(uint8_t)(name[1])]);
      /* not uint32_t here!  (uint32_t)255 (as a char) returns -1!! */
      switch (name[1])
	{
	case 'n':
	  if ((c_strings_are_equal(name + 1, "null")) ||
	      (c_strings_are_equal(name + 1, "nul")))
	    return(chars[0]);

	  if (c_strings_are_equal(name + 1, "newline"))
	    return(chars[(uint8_t)'\n']);
	  break;

	case 'a': if (c_strings_are_equal(name + 1, "alarm"))     return(chars[7]);             break;
	case 'b': if (c_strings_are_equal(name + 1, "backspace")) return(chars[8]);             break;
	case 'd': if (c_strings_are_equal(name + 1, "delete"))    return(chars[0x7f]);          break;
	case 'e': if (c_strings_are_equal(name + 1, "escape"))    return(chars[0x1b]);          break;
	case 'l': if (c_strings_are_equal(name + 1, "linefeed"))  return(chars[(uint8_t)'\n']); break;
	case 'r': if (c_strings_are_equal(name + 1, "return"))    return(chars[(uint8_t)'\r']); break;
	case 's': if (c_strings_are_equal(name + 1, "space"))     return(chars[(uint8_t)' ']);  break;
	case 't': if (c_strings_are_equal(name + 1, "tab"))       return(chars[(uint8_t)'\t']); break;
	  /* to print something in bold-face: (format *stderr* "~Ahiho~A~%" (string-append (string #\escape) "[1m") (string-append (string #\escape) "[22m")) */

	case 'x':
	  /* #\x is just x, but apparently #\x<num> is int->char? #\x65 -> #\e, and #\xcebb is lambda? */
	  {
	    /* sscanf here misses errors like #\x1.4, but make_atom misses #\x6/3,
	     *   #\x#b0, #\x#e0.0, #\x-0, #\x#e0e100 etc, so we have to do it at an even lower level.
	     * another problem: #\xbdca2cbec overflows so new_int is -593310740 -> segfault unless caught
	     */
	    bool happy = true;
	    const char *tmp = (const char *)(name + 2);
	    int32_t new_int = 0;

	    while ((*tmp) && (happy) && (new_int >= 0) && (new_int < 256))
	      {
		int32_t dig = digits[(int32_t)(*tmp++)];
		if (dig < 16)
		  new_int = dig + (new_int * 16);
		else happy = false;
	      }
	    if ((happy) &&
		(new_int < 256) &&
		(new_int >= 0))
	      return(chars[new_int]);
	  }
	  break;
	}}
  return(unknown_sharp_constant(sc, name, NULL));
}

static s7_int string_to_integer(const char *str, int32_t radix, bool *overflow)
{
  bool negative = false;
  s7_int new_int = 0;
  int32_t dig;
  const char *tmp = (const char *)str;
#if With_Gmp
  const char *tmp1;
#endif
  if (str[0] == '+')
    tmp++;
  else
    if (str[0] == '-')
      {
	negative = true;
	tmp++;
      }
  while (*tmp == '0') {tmp++;};
#if With_Gmp
  tmp1 = tmp;
#endif
 if (radix == Base_10)
    {
      while (true)
	{
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig > 9) break;
#if Have_Overflow_Checks
	  if ((multiply_overflow(new_int, (s7_int)10, &new_int)) ||
	      (add_overflow(new_int, (s7_int)dig, &new_int)))
	    {
	      if ((radix == Base_10) &&
		  (strncmp(str, "-9223372036854775808", 20) == 0) &&
		  (digits[(uint8_t)(*tmp++)] > 9)) /* i.e. next thing is not a digit -- "/" for example */
		return(s7_Int64_Min);
	      *overflow = true;
	      return((negative) ? s7_Int64_Min : s7_Int64_Max);
	    }
#else
	  new_int = dig + (new_int * 10);
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig > 9) break;
	  new_int = dig + (new_int * 10);
#endif
	}}
  else
    while (true)
      {
	dig = digits[(uint8_t)(*tmp++)];
	if (dig >= radix) break;
#if Have_Overflow_Checks && (!With_Gmp)
	{
	  s7_int oval = 0;
	  if (multiply_overflow(new_int, (s7_int)radix, &oval))
	    {
	      /* maybe a bad idea!  #xffffffffffffffff -> -1??? this is needed for 64-bit number hacks (see s7test.scm bit-reverse) */
	      if ((radix == 16) &&
		  (digits[(uint8_t)(*tmp)] >= radix))
		{
		  new_int -= 576460752303423488LL; /* turn off sign bit */
		  new_int *= radix;
		  new_int += dig;
		  new_int -= 9223372036854775807LL;
		  return(new_int - 1);
		}
	      new_int = oval; /* old case */
	      if ((new_int == s7_Int64_Min)  && (digits[(uint8_t)(*tmp++)] > 9))
		return(new_int);
	      *overflow = true;
	      break;
	    }
	  else new_int = oval;
	  if (add_overflow(new_int, (s7_int)dig, &new_int))
	    {
	      if (new_int == s7_Int64_Min) return(new_int);
	      *overflow = true;
	      break;
	    }}
#else
	new_int = dig + (new_int * radix);
	dig = digits[(uint8_t)(*tmp++)];
	if (dig >= radix) break;
	new_int = dig + (new_int * radix);
#endif
      }

#if With_Gmp
 if (!*overflow)
   (*overflow) = ((new_int > s7_Int32_Max) ||
		  ((tmp - tmp1) > s7_int_digits_by_radix[radix]));
  /* this tells the string->number readers to create a bignum.  We need to be very conservative here to catch contexts such as (/ 1/524288 19073486328125) */
#endif
  return((negative) ? -new_int : new_int);
}

static const char *radstr[17] = {NULL, NULL, "01", "012", "0123", "01234", "012345", "0123456", "01234567", "012345678", "0123456789",
  "0123456789aA", "0123456789aAbB", "0123456789aAbBcC", "0123456789aAbBcCdD", "0123456789aAbBcCdDeE", "0123456789aAbBcCdDeEfF"};

#if With_Gmp
static s7_double string_to_double_with_radix(const char *ur_str, int32_t radix, bool *overflow)
#else
static s7_double string_to_double_with_radix(const char *ur_str, int32_t radix)
#endif
{
  /* strtod follows LANG which is not what we want (only "." is decimal point in Scheme).
   *   To overcome LANG in strtod would require screwing around with setlocale which never works.
   *   So we use our own code -- according to valgrind, this function is much faster than strtod.
   * comma as decimal point causes ambiguities: `(+ ,1 2) etc
   */
  int32_t sign = 1, frac_len, int_len, dig, exponent = 0;
  const int32_t max_len = s7_int_digits_by_radix[radix];
  s7_int int_part = 0, frac_part = 0;
  const char *str = ur_str;
  const char *ipart, *fpart;
  s7_double dval = 0.0;

  /* there's an ambiguity in number notation here if we allow "1e1" or "1.e1" in base 16 (or 15) -- is e a digit or an exponent marker?
   *   but 1e+1, for example disambiguates it -- kind of messy! -- the scheme spec says "e" can only occur in base 10.
   * '@' can now be used as the exponent marker (26-Mar-12).
   * Another slight ambiguity: 1+1/2i is parsed as 1 + 0.5i, not 1+1/(2i), or (1+1)/(2i) or (1+1/2)i etc
   */
  if (*str == '-')
    {
      str++;
      sign = -1;
    }
  else
    if (*str == '+')
      str++;
  while (*str == '0') {str++;};

  ipart = str;
  /* while (digits[(int32_t)(*str)] < radix) str++; */
  /* int_len = str - ipart; */
  int_len = strspn((const char *)str, radstr[radix]); /* this is faster than the while loop with digits[] */
  str += int_len;

  if (*str == '.') str++;
  fpart = str;
  /* while (digits[(int32_t)(*str)] < radix) str++; */
  /* frac_len = str - fpart; */
  frac_len = strspn((const char *)str, radstr[radix]);
  str += frac_len;

  if ((*str) && (exponent_table[(uint8_t)(*str)]))
    {
      bool exp_negative = false;
      str++;
      if (*str == '+')
	str++;
      else
	if (*str == '-')
	  {
	    str++;
	    exp_negative = true;
	  }
      while ((dig = digits[(int32_t)(*str++)]) < 10) /* exponent itself is always base 10 */
	{
#if Have_Overflow_Checks
	  if ((int32_multiply_overflow(exponent, 10, &exponent)) ||
	      (int32_add_overflow(exponent, dig, &exponent)))
	    {
	      exponent = 1000000; /* see below */
	      break;
	    }
#else
	  exponent = dig + (exponent * 10);
#endif
	}
#if (!defined(__GNUC__)) || ((__GNUC__ < 5) && (!defined(__clang__)))
      if (exponent < 0)         /* we overflowed, so make sure we notice it below (need to check for 0.0e... first) (Brian Damgaard) */
	exponent = 1000000;     /*   see below for examples -- this number needs to be very big but not too big for add */
#endif
      if (exp_negative)
	exponent = -exponent;

      /*           2e12341234123123123123213123123123 -> 0.0
       * but exp len is not the decider: 2e00000000000000000000000000000000000000001 -> 20.0
       * first zero: 2e123412341231231231231
       * then:     2e12341234123123123123123123 -> inf
       * then:     2e123412341231231231231231231231231231 -> 0.0
       *           2e-123412341231231231231 -> inf
       * but:      0e123412341231231231231231231231231231
       */
    }

#if With_Gmp
  /* 9007199254740995.0 since 2^53=9007199254740992 */
  if (int_len + frac_len >= max_len)
    {
      (*overflow) = true;
      return(0.0);
    }
  /* trouble in non-gmp case if int_len>19 (9223372036854775808), or frac_len>0 and int_len>16 (2^53 above) -- should we warn
   *   about this if sc->safety != No_Safety? (need to ignore leading and (in frac case) trailing zeros, of course).
   */
#endif
  str = ipart;
  if ((int_len + exponent) > max_len)
    {
      /*  12341234.56789e12                   12341234567889999872.0              1.234123456789e+19
       * -1234567890123456789.0              -1234567890123456768.0              -1.2345678901235e+18
       *  12345678901234567890.0              12345678901234567168.0              1.2345678901235e+19
       *  123.456e30                          123456000000000012741097792995328.0 1.23456e+32
       *  12345678901234567890.0e12           12345678901234569054409354903552.0  1.2345678901235e+31
       *  1.234567890123456789012e30          1234567890123456849145940148224.0   1.2345678901235e+30
       *  1e20                                100000000000000000000.0             1e+20
       *  1234567890123456789.0               1234567890123456768.0               1.2345678901235e+18
       *  123.456e16                          1234560000000000000.0               1.23456e+18
       *  98765432101234567890987654321.0e-5  987654321012345728401408.0          9.8765432101235e+23
       *  98765432101234567890987654321.0e-10 9876543210123456512.0               9.8765432101235e+18
       *  0.00000000000000001234e20           1234.0
       *  0.000000000000000000000000001234e30 1234.0
       *  0.0000000000000000000000000000000000001234e40 1234.0
       *  0.000000000012345678909876543210e15 12345.678909877
       *  0e1000                              0.0
       */

      for (int32_t i = 0; i < max_len; i++)
	{
	  dig = digits[(int32_t)(*str++)];
	  if (dig < radix)
	    int_part = dig + (int_part * radix);
	  else break;
	}

      /* if the exponent is huge, check for 0 int_part and frac_part before complaining (0e1000 or 0.0e1000) */
      if ((int_part == 0) &&
	  (exponent > max_len))
	{
	  /* if frac_part is also 0, return 0.0 */
	  if (frac_len == 0) return(0.0);
	  str = fpart;
	  while ((dig = digits[(int32_t)(*str++)]) < radix)
	    frac_part = dig + (frac_part * radix);
	  if (frac_part == 0) return(0.0);
#if With_Gmp
	  (*overflow) = true;
#endif
	}
#if With_Gmp
      (*overflow) = ((int_part > 0) || (exponent > 20));    /* .1e310 is a tricky case */
#endif
      if (int_part != 0) /* 0.<310 zeros here>1e310 for example -- pow (via dpow) thinks it has to be too big, returns Nan,
			  *   then Nan * 0 -> Nan and the NaN propagates
			  */
	{
	  if (int_len <= max_len)
	    dval = int_part * dpow(radix, exponent);
	  else dval = int_part * dpow(radix, exponent + int_len - max_len);
	}
      else dval = 0.0;

      /* shift by exponent, but if int_len > max_len then we assumed (see below) int_len - max_len 0's on the left */
      /*   using int_to_int or table lookups here instead of pow did not make any difference in speed */

      if (int_len < max_len)
	{
	  str = fpart;
	  for (int32_t k = 0; (frac_len > 0) && (k < exponent); k += max_len)
	    {
	      int32_t flen = (frac_len > max_len) ? max_len : frac_len; /* ? */
	      frac_len -= max_len;
	      frac_part = 0;
	      for (int32_t i = 0; i < flen; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	      if (frac_part != 0)                                /* same pow->NaN problem as above can occur here */
		dval += frac_part * dpow(radix, exponent - flen - k);
	    }}
      else
	/* some of the fraction is in the integer part before the negative exponent shifts it over */
	if (int_len > max_len)
	  {
	    int32_t ilen = int_len - max_len;                          /* we read these above */
	    /* str should be at the last digit we read */
	    if (ilen > max_len)
	      ilen = max_len;
	    for (int32_t i = 0; i < ilen; i++)
	      frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	    dval += frac_part * dpow(radix, exponent - ilen);
	  }
      return(sign * dval);
    }

  /* int_len + exponent <= max_len */
  if (int_len <= max_len)
    {
      int32_t int_exponent = exponent;
      /* a better algorithm (since the inaccuracies are in the radix^exponent portion):
       *   strip off leading zeros and possible sign,
       *   strip off digits beyond max_len, then remove any trailing zeros.
       *     (maybe fiddle with the lowest order digit here for rounding, but I doubt it matters)
       *   read digits until end of number or max_len reached, ignoring the decimal point
       *   get exponent and use it and decimal point location to position the current result integer
       * this always combines the same integer and the same exponent no matter how the number is expressed.
       */
      if (int_len > 0)
	{
	  const char *iend = (const char *)(str + int_len - 1);
	  while ((*iend == '0') && (iend != str)) {iend--; int_exponent++;}
	  while (str <= iend)
	    int_part = digits[(int32_t)(*str++)] + (int_part * radix);
	}
      dval = (int_exponent == 0) ? (s7_double)int_part : int_part * dpow(radix, int_exponent);
    }
  else
    {
      int32_t flen, len = int_len + exponent;
      s7_int frpart = 0;

      /* 98765432101234567890987654321.0e-20    987654321.012346
       * 98765432101234567890987654321.0e-29    0.98765432101235
       * 98765432101234567890987654321.0e-30    0.098765432101235
       * 98765432101234567890987654321.0e-28    9.8765432101235
       */
      for (int32_t i = 0; i < len; i++)
	int_part = digits[(int32_t)(*str++)] + (int_part * radix);
      flen = -exponent;
      if (flen > max_len)
	flen = max_len;
      for (int32_t i = 0; i < flen; i++)
	frpart = digits[(int32_t)(*str++)] + (frpart * radix);
      if (len <= 0)
	dval = int_part + frpart * dpow(radix, len - flen);
      else dval = int_part + frpart * dpow(radix, -flen);
    }

  if (frac_len > 0)
    {
      str = fpart;
      if (frac_len <= max_len)
	{
	  /* splitting out base 10 case saves very little here */
	  /* this ignores trailing zeros, so that 0.3 equals 0.300 */
	  const char *fend = (const char *)(str + frac_len - 1);

	  while ((*fend == '0') && (fend != str)) {fend--; frac_len--;} /* (= .6 0.6000) */
	  if ((frac_len & 1) == 0)
	    {
	      while (str <= fend)
		{
		  frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
		  frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
		}}
	  else
	    while (str <= fend)
	      frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	  dval += frac_part * dpow(radix, exponent - frac_len);

	  /* 0.6:    frac:    6, exp: 0.10000000000000000555, val: 0.60000000000000008882
	   * 0.60:   frac:   60, exp: 0.01000000000000000021, val: 0.59999999999999997780
	   * 0.6000: frac: 6000, exp: 0.00010000000000000000, val: 0.59999999999999997780
	   * (= 0.6 0.60): #f
	   * (= #i3/5 0.6): #f
	   * so (string->number (number->string num)) == num only if both num's are the same text (or you get lucky)
	   * (= 0.6 6e-1): #t ; but not 60e-2
	   * to fix the 0.60 case, we need to ignore trailing post-dot zeros.
	   */
	}
      else
	{
	  if (exponent <= 0)
	    {
	      for (int32_t i = 0; i < max_len; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	      dval += frac_part * dpow(radix, exponent - max_len);
	    }
	  else
	    {
	      /* 1.0123456789876543210e1         10.12345678987654373771
	       * 1.0123456789876543210e10        10123456789.87654304504394531250
	       * 0.000000010000000000000000e10   100.0
	       * 0.000000010000000000000000000000000000000000000e10 100.0
	       * 0.000000012222222222222222222222222222222222222e10 122.22222222222222
	       * 0.000000012222222222222222222222222222222222222e17 1222222222.222222
	       */
	      int_part = 0;
	      for (int32_t i = 0; i < exponent; i++)
		int_part = digits[(int32_t)(*str++)] + (int_part * radix);
	      frac_len -= exponent;
	      if (frac_len > max_len)
		frac_len = max_len;
	      for (int32_t i = 0; i < frac_len; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	      dval += int_part + frac_part * dpow(radix, -frac_len);
	    }}}
#if With_Gmp
  if ((int_part == 0) &&
      (frac_part == 0))
    return(0.0);
  (*overflow) = ((frac_len - exponent) > max_len);
#endif
  return(sign * dval);
}

#if !With_Gmp
static s7_pointer make_undefined_bignum(s7_scheme *sc, const char *name)
{
  s7_int len = safe_strlen(name) + 16;
  block_t *b = mallocate(sc, len);
  char *buf = (char *)block_data(b);
  s7_pointer result;
  snprintf(buf, len, "<bignum: %s>", name);
  result = make_undefined(sc, (const char *)buf); /* 123123123123123123123123123123 -> +inf.0 originally, but now #<bignum: 123123...> */
  liberate(sc, b);
  return(result);
}
#endif

static s7_pointer nan1_or_bust(s7_scheme *sc, s7_double x, const char *p, const char *q, int32_t radix, bool want_symbol, int32_t offset)
{
  const s7_int len = safe_strlen(p);
  if (p[len - 1] == 'i')       /* +nan.0[+/-]...i */
    {
      if (len == (offset + 2))            /* +nan.0+i */
	return(make_complex_not_0i(sc, x, (p[offset] == '+') ? 1.0 : -1.0));
      if ((len > (offset + 1)) && (len < 1024)) /* make compiler happy */
	{
	  char *ip = copy_string_with_length((const char *)(p + offset), len - offset - 1);
	  s7_pointer imag = make_atom(sc, ip, radix, No_Symbols, Without_Overflow_Error);
	  free(ip);
	  if (is_real(imag))
	    return(make_complex(sc, x, real_to_double(sc, imag, __func__))); /* +nan.0+2/3i etc */
	}}
  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
}

static s7_pointer nan2_or_bust(s7_scheme *sc, s7_double x, const char *q, int32_t radix, bool want_symbol, s7_int rl_len)
{
  const s7_int len = safe_strlen(q);
  if ((len > rl_len) && (len < 1024)) /* make compiler happy */
    {
      char *ip = copy_string_with_length(q, rl_len);
      s7_pointer rl = make_atom(sc, ip, radix, No_Symbols, Without_Overflow_Error);
      free(ip);
      if (is_real(rl))
	return(make_complex(sc, real_to_double(sc, rl, __func__), x));
    }
  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
}

#if With_Number_Separator
static s7_pointer string_to_number(s7_scheme *sc, char *str, int32_t radix);

static s7_pointer make_symbol_or_number(s7_scheme *sc, const char *name, int32_t radix, bool want_symbol)
{
  block_t *b;
  char *new_name;
  const char sep = sc->number_separator;
  s7_int len, j = 0;

  if (name[0] == sep)
    return((want_symbol) ? make_symbol_with_strlen(sc, name) : sc->F);
  len = safe_strlen(name);
  b = mallocate(sc, len + 1);
  new_name = (char *)block_data(b);
  memcpy((void *)new_name, (const void *)name, len);
  new_name[len] = 0;

  for (s7_int i = 0; i < len; i++)
    if (name[i] != sep)
      {
	if ((digits[(uint8_t)(name[i])] < radix) || (!t_number_separator_p[(uint8_t)name[i]]))
	  new_name[j++] = name[i];
	else
	  {
	    liberate(sc, b);
	    return((want_symbol) ? make_symbol_with_strlen(sc, name) : sc->F);
	  }}
    else  /* sep has to be between two digits */
      if ((digits[(uint8_t)(name[i - 1])] >= radix) || (digits[(uint8_t)(name[i + 1])] >= radix))
	{
	  liberate(sc, b);
	  return((want_symbol) ? make_symbol_with_strlen(sc, name) : sc->F);
	}
  new_name[j] = '\0';
  {
    s7_pointer result = string_to_number(sc, new_name, radix);
    liberate(sc, b);
    return(result);
  }
}
#endif

static s7_pointer make_atom(s7_scheme *sc, char *q, int32_t radix, bool want_symbol, bool with_error)
{
  /* make symbol or number from string, a number starts with + - . or digit, but so does 1+ for example */
#if With_Number_Separator
  #define is_digit(Chr, Rad) ((digits[(uint8_t)Chr] < Rad) || ((Chr == sc->number_separator) && (sc->number_separator != '\0')))
#else
  #define is_digit(Chr, Rad) (digits[(uint8_t)Chr] < Rad)
#endif
  char c, *p = q;
  bool has_dec_point1 = false;

  c = *p++;
  switch (c)
    {
    case '#':
      /* from string->number, (string->number #xc) */
      return(make_sharp_constant(sc, p, with_error, NULL, false)); /* make_sharp_constant expects the '#' to be removed */

    case '+':
    case '-':
      c = *p++;
      if (c == '.')
	{
	  has_dec_point1 = true;
	  c = *p++;
	}
      if (!c)
	return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
      if (!is_digit(c, radix))
	{
	  if (has_dec_point1)
	    return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
	  if (c == 'n')
	    {
	      if (local_strcmp(p, "an.0"))            /* +nan.0, even if we read -nan.0 -- what's the point of a negative NaN? */
		return(real_NaN);                     /* not make_nan_with_payload(sc, __LINE__) here since it says "0" */
	      if ((local_strncmp(p, "an.0", 4)) &&    /* +nan.0[+/-]...i */
		  ((p[4] == '+') || (p[4] == '-')))
		return(nan1_or_bust(sc, Nan, p, q, radix, want_symbol, 4));
	      /* read +/-nan.<int> or +/-nan.<int>+/-...i */
	      if (local_strncmp(p, "an.", 3))         /* +nan.<int> */
		{
		  bool overflow = false;
		  int32_t i;
		  for (i = 3; is_digit(p[i], 10); i++);
		  if ((p[i] == '+') || (p[i] == '-')) /* complex case */
		    {
		      s7_int payload = string_to_integer((char *)(p + 3), 10, &overflow);
		      return(nan1_or_bust(sc, nan_with_payload(payload), p, q, radix, want_symbol, i));
		    }
		  /* check for "+nan." or "nan.a" -> symbol but "+nan.0+i" is a number */
		  if ((i == 3) || ((p[i] != '\0') && (!white_space[(uint8_t)(p[i])]))) /* check for +nan.0i etc, '\0' is not white_space apparently */
		    return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
		  return(make_nan_with_payload(sc, string_to_integer((char *)(p + 3), 10, &overflow)));
		}}
	  if (c == 'i')
	    {
	      if (local_strcmp(p, "nf.0"))  /* +inf.0 */
		return((q[0] == '+') ? real_infinity : real_minus_infinity);
	      if ((local_strncmp(p, "nf.0", 4)) && /* unaligned */
		  ((p[4] == '+') || (p[4] == '-')))
		return(nan1_or_bust(sc, (q[0] == '-') ? -Infinity : Infinity, p, q, radix, want_symbol, 4));
	    }
	  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
	}
      break;

    case '.':
      has_dec_point1 = true;
      c = *p++;
      if ((!c) || (!is_digit(c, radix)))
	return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
      break;

    case 'n':
      return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

    case 'i':
      return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

    case '0': case '1':    /* these two are always digits */
      break;

    default:
      if (!is_digit(c, radix))
	return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
      break;
    }

  /* now it's possibly a number -- the first character(s) could be part of a number in the current radix */
  {
    char *slash1 = NULL, *slash2 = NULL, *plus = NULL, *ex1 = NULL, *ex2 = NULL;
    bool has_i = false, has_dec_point2 = false;
    int32_t has_plus_or_minus = 0, current_radix;
#if !With_Gmp
    bool overflow = false; /* for string_to_integer */
#endif
    current_radix = radix;  /* current_radix is 10 for the exponent portions, but radix for all the rest */

    for ( ; (c = *p) != 0; ++p)
      {
	/* what about embedded null? (string->number (string #\1 (integer->char 0) #\0))
	 *   currently we stop and return 1, but Guile returns #f.
	 *   this also means we can't use substring_uncopied if (string->number (substring...))
	 */
	if (!is_digit(c, current_radix))         /* moving this inside the switch statement was much slower */
	  {
	    current_radix = radix;

	    switch (c)
	      {
		/* -------- decimal point -------- */
	      case '.':
		if ((!is_digit(p[1], current_radix)) &&
		    (!is_digit(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    if ((has_dec_point1) || (slash1))
		      return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
		    has_dec_point1 = true;
		  }
		else
		  {
		    if ((has_dec_point2) || (slash2))
		      return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
		    has_dec_point2 = true;
		  }
		continue;

		/* -------- exponent marker -------- */
#if With_Extra_Exponent_Markers
		/* 1st 3d-perspective 0.0f 128.0f 3d 1s -- in 2 million lines of public scheme code, not one actual use! */
	      case 's': case 'S':
	      case 'd': case 'D':
	      case 'f': case 'F':
	      case 'l': case 'L':
#endif
	      case 'e': case 'E':
		if (current_radix > 10) /* see above */
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
		/* fall through -- if '@' used, radices>10 are ok */

	      case '@':
		current_radix = 10;

		if (((ex1) ||
		     (slash1)) &&
		    (has_plus_or_minus == 0)) /* ee */
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		if (((ex2) ||
		     (slash2)) &&
		    (has_plus_or_minus != 0)) /* 1+1.0ee */
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		if ((!is_digit(p[-1], radix)) && /* was current_radix but that's always 10! */
		    (p[-1] != '.'))
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    ex1 = p;
		    has_dec_point1 = true; /* decimal point illegal from now on */
		  }
		else
		  {
		    ex2 = p;
		    has_dec_point2 = true;
		  }
		p++;
		if ((*p == '-') || (*p == '+')) p++;
		if (is_digit(*p, current_radix))
		  continue;
		break;

		/* -------- internal + or - -------- */
	      case '+':
	      case '-':
		if (has_plus_or_minus != 0) /* already have the separator */
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		has_plus_or_minus = (c == '+') ? 1 : -1;
		plus = (char *)(p + 1);
		/* now check for nan/inf as imaginary part */

		if ((plus[0] == 'n') &&
		    (local_strncmp(plus, "nan.", 4)))
		  {
		    bool overflow1 = false;
		    s7_int payload = string_to_integer((char *)(p + 5), 10, &overflow1);
		    return(nan2_or_bust(sc, nan_with_payload(payload), q, radix, want_symbol, (intptr_t)(p - q)));
		  }
		if ((plus[0] == 'i') &&
		    (local_strcmp(plus, "inf.0i")))
		  return(nan2_or_bust(sc, (c == '+') ? Infinity : -Infinity, q, radix, want_symbol, (intptr_t)(p - q)));
		continue;

		/* ratio marker */
	      case '/':
		if ((has_plus_or_minus == 0) &&
		    ((ex1) ||
		     (slash1) ||
		     (has_dec_point1)))
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		if ((has_plus_or_minus != 0) &&
		    ((ex2) ||
		     (slash2) ||
		     (has_dec_point2)))
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  slash1 = (char *)(p + 1);
		else slash2 = (char *)(p + 1);

		if ((!is_digit(p[1], current_radix)) ||
		    (!is_digit(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

		continue;

		/* -------- i for the imaginary part -------- */
	      case 'i':
		if ((has_plus_or_minus != 0) &&
		    (!has_i))
		  {
		    has_i = true;
		    continue;
		  }
		break;

	      default: break;
	      }
	    return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);
	  }}

    if ((has_plus_or_minus != 0) &&        /* that is, we have an internal + or - */
	(!has_i))                          /*   but no i for the imaginary part */
      return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

#if With_Number_Separator
    if ((sc->number_separator != '\0') && (strchr(q, (int)(sc->number_separator))))
      return(make_symbol_or_number(sc, q, radix, want_symbol));
#endif

    if (has_i)
      {
#if !With_Gmp
	s7_double rl = 0.0, im = 0.0;
#else
	char e1 = 0, e2 = 0;
#endif
	s7_pointer result;
	s7_int len = safe_strlen(q);
	char ql1, pl1;

	if (q[len - 1] != 'i')
	  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

	/* save original string */
	ql1 = q[len - 1];
	pl1 = (*(plus - 1));
#if With_Gmp
	if (ex1) {e1 = *ex1; (*ex1) = '@';} /* for mpfr */
	if (ex2) {e2 = *ex2; (*ex2) = '@';}
#endif
	/* look for cases like 1+i */
	q[len - 1] = ((q[len - 2] == '+') || (q[len - 2] == '-')) ? '1' : '\0'; /* remove 'i' */

	(*((char *)(plus - 1))) = '\0';

#if !With_Gmp
	if ((has_dec_point1) ||
	    (ex1))  /* (string->number "1100.1+0.11i" 2) -- need to split into 2 honest reals before passing to non-base-10 str->dbl */
	  rl = string_to_double_with_radix(q, radix);
	else /* no decimal point, no exponent, a ratio (1/2+i for example, but 1+2/3i is handled below) */
	  {
	    if (slash1)
	      {
		/* here the overflow could be innocuous if it's in the denominator and the numerator is 0: 0/100000000000000000000000000000000000000 */
		s7_int den, num = string_to_integer(q, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
		den = string_to_integer(slash1, radix, &overflow);
		if (den == 0)
		  rl = Nan;        /* real_part if complex */
		else
		  {
		    if (num == 0)
		      {
			rl = 0.0;
			overflow = false;
		      }
		    else
		      {
			if (overflow) return(make_undefined_bignum(sc, q)); /* denominator overflow */
			rl = (long_double)num / (long_double)den; /* no gmp, so we do what we can */
		      }}}
	    else
	      {
		rl = (s7_double)string_to_integer(q, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
	      }}
	if (rl == -0.0) rl = 0.0;

	if ((has_dec_point2) ||
	    (ex2))
	  im = string_to_double_with_radix(plus, radix);
	else
	  {
	    if (slash2) /* complex part I think */
	      {
		/* same as above: 0-0/100000000000000000000000000000000000000i */
		s7_int den;
		const s7_int num = string_to_integer(plus, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
		den = string_to_integer(slash2, radix, &overflow);
		if (den == 0)
		  im = Nan;
		else
		  {
		    if (num == 0)
		      {
			im = 0.0;
			overflow = false;
		      }
		    else
		      {
			if (overflow) return(make_undefined_bignum(sc, q)); /* denominator overflow */
			im = (long_double)num / (long_double)den;
		      }}}
	    else
	      {
		im = (s7_double)string_to_integer(plus, radix, &overflow);
		if (overflow) return(make_undefined_bignum(sc, q));
	      }}
	if ((has_plus_or_minus == -1) &&
	    (im != 0.0))
	  im = -im;
	result = make_complex(sc, rl, im);
#else
	result = string_to_either_complex(sc, q, slash1, ex1, has_dec_point1, plus, slash2, ex2, has_dec_point2, radix, has_plus_or_minus);
#endif
	/* restore original string */
	q[len - 1] = ql1;
	(*((char *)(plus - 1))) = pl1;
#if With_Gmp
	if (ex1) (*ex1) = e1;
	if (ex2) (*ex2) = e2;
#endif
	return(result);
      }

    /* not complex */
    if ((has_dec_point1) ||
	(ex1))
      {
	s7_pointer result;
	if (slash1)  /* not complex, so slash and "." is not a number */
	  return((want_symbol) ? make_symbol_with_strlen(sc, q) : sc->F);

#if !With_Gmp
	result = make_real(sc, string_to_double_with_radix(q, radix));
#else
	{
	  char old_e = 0;
	  if (ex1)
	    {
	      old_e = (*ex1);
	      (*ex1) = '@';
	    }
	  result = string_to_either_real(sc, q, radix);
	  if (ex1)
	    (*ex1) = old_e;
	}
#endif
	return(result);
      }

    /* rational */
    if (slash1)
#if !With_Gmp
      {
	s7_int d;
	const s7_int n = string_to_integer(q, radix, &overflow);
	if (overflow) return(make_undefined_bignum(sc, q));
	d = string_to_integer(slash1, radix, &overflow);

	if ((n == 0) && (d != 0))                        /* 0/100000000000000000000000000000000000000 */
	  return(int_zero);
	if (d == 0) return(real_NaN);                    /* nan.__LINE__ here seems less than optimal */
	if (overflow) return(make_undefined_bignum(sc, q));
	/* it would be neat to return 1 from 10000000000000000000000000000/10000000000000000000000000000
	 *   but q is the entire number ('/' included) and slash1 is the stuff after the '/', and every
	 *   big number comes through here, so there's no clean and safe way to check that q == slash1.
	 */
	return(make_ratio(sc, n, d));
      }
#else
    return(string_to_either_ratio(sc, q, slash1, radix));
#endif
    /* integer */
#if !With_Gmp
    {
      s7_int x = string_to_integer(q, radix, &overflow);
      if (overflow) return(make_undefined_bignum(sc, q));
      return(make_integer(sc, x));
    }
#else
    return(string_to_either_integer(sc, q, radix));
#endif
  }
}


/* -------------------------------- string->number -------------------------------- */
static s7_pointer string_to_number(s7_scheme *sc, char *str, int32_t radix)
{
  s7_pointer x = make_atom(sc, str, radix, No_Symbols, Without_Overflow_Error);
  return((is_number(x)) ? x : sc->F);  /* only needed because str might start with '#' and not be a number (#t for example) */
}

static s7_pointer string_to_number_p_p(s7_scheme *sc, s7_pointer str1)
{
  char *str;
  if (!is_string(str1))
    wrong_type_error_nr(sc, sc->string_to_number_symbol, 1, str1, sc->type_names[T_STRING]);
  str = (char *)string_value(str1);
  return(((!str) || (!*str)) ? sc->F : string_to_number(sc, str, 10));
}

static s7_pointer string_to_number_p_pp(s7_scheme *sc, s7_pointer str1, s7_pointer radix1)
{
  s7_int radix;
  char *str;
  if (!is_string(str1))
    wrong_type_error_nr(sc, sc->string_to_number_symbol, 1, str1, sc->type_names[T_STRING]);

  if (!is_t_integer(radix1))
    wrong_type_error_nr(sc, sc->string_to_number_symbol, 2, radix1, sc->type_names[T_INTEGER]);
  radix = integer(radix1);
  if ((radix < 2) || (radix > 16))
    out_of_range_error_nr(sc, sc->string_to_number_symbol, int_two, radix1, a_valid_radix_string);

  str = (char *)string_value(str1);
  if ((!str) || (!*str))
    return(sc->F);
  return(string_to_number(sc, str, radix));
}

static s7_pointer string_to_number_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  s7_int radix;
  char *str;
  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), caller, args, sc->type_names[T_STRING], 1));

  if (is_pair(cdr(args)))
    {
      const s7_pointer rad = cadr(args);
      if (!s7_is_integer(rad))
	return(method_or_bust(sc, rad, caller, args, sc->type_names[T_INTEGER], 2));
      radix = s7_integer_clamped_if_gmp(sc, rad);
      if ((radix < 2) || (radix > 16))
	out_of_range_error_nr(sc, caller, int_two, rad, a_valid_radix_string);
    }
  else radix = 10;
  str = (char *)string_value(car(args));
  if ((!str) || (!*str))
    return(sc->F);
  return(string_to_number(sc, str, radix));
}

static s7_pointer g_string_to_number(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_number "(string->number str (radix 10)) converts str into a number. \
If str does not represent a number, string->number returns #f.  If 'str' has an embedded radix, \
the optional 'radix' argument is ignored: (string->number \"#x11\" 2) -> 17 not 3."
  #define Q_string_to_number s7_make_signature(sc, 3, \
                               s7_make_signature(sc, 2, sc->is_number_symbol, sc->not_symbol), \
                               sc->is_string_symbol, sc->is_integer_symbol)
  return(string_to_number_1(sc, args, sc->string_to_number_symbol));
}


/* -------------------------------- abs -------------------------------- */
static inline s7_pointer abs_p_p(s7_scheme *sc, s7_pointer x)
{
#if !With_Gmp
  if (is_t_integer(x))
    {
      if (integer(x) >= 0) return(x);
      if (integer(x) > s7_Int64_Min) return(make_integer(sc, -integer(x)));
    }
  if (is_t_real(x))
    {
#if 0
      if (is_NaN(real(x)))
	return((nan_payload(real(x)) > 0) ? x : real_NaN);         /* (abs -nan.0) -> +nan.0?? */
#endif
      return((signbit(real(x))) ? make_real(sc, -real(x)) : x);
    }
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) >= 0) return(x);
#if With_Gmp
      if (integer(x) == s7_Int64_Min)
	{
	  x = s7_int_to_big_integer(sc, integer(x));
	  mpz_neg(big_integer(x), big_integer(x));
	  return(x);
	}
#else
      if (integer(x) == s7_Int64_Min)
	sole_arg_out_of_range_error_nr(sc, sc->abs_symbol, set_elist_1(sc, x), result_is_too_large_string);
#endif
      return(make_integer(sc, -integer(x)));

    case T_RATIO:
      if (numerator(x) >= 0) return(x);
#if With_Gmp && (!Pointer_32)
      if (numerator(x) == s7_Int64_Min)
	{
	  s7_pointer new_bgr;
	  mpz_set_si(sc->mpz_1, s7_Int64_Min);
	  mpz_neg(sc->mpz_1, sc->mpz_1);
	  mpz_set_si(sc->mpz_2, denominator(x));
	  new_cell(sc, new_bgr, T_BIG_RATIO);
	  big_ratio_bgr(new_bgr) = alloc_bigrat(sc);
	  add_big_ratio(sc, new_bgr);
	  mpq_set_num(big_ratio(new_bgr), sc->mpz_1);
	  mpq_set_den(big_ratio(new_bgr), sc->mpz_2);
	  return(new_bgr);
	}
#else
      if (numerator(x) == s7_Int64_Min)
	return(make_ratio(sc, s7_Int64_Max, denominator(x))); /* not rationalized, so can't call make_simpler_ratio */
#endif
      return(make_simpler_ratio(sc, -numerator(x), denominator(x)));

    case T_REAL:
      if (is_NaN(real(x)))                  /* (abs -nan.0) -> +nan.0, not -nan.0 */
	return((nan_payload(real(x)) > 0) ? x : real_NaN);
      return((signbit(real(x))) ? make_real(sc, -real(x)) : x); /* (abs -0.0) returns -0.0 -- Shiro Kawai */
#if With_Gmp
    case T_BIG_INTEGER:
      mpz_abs(sc->mpz_1, big_integer(x));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
      mpq_abs(sc->mpq_1, big_ratio(x));
      return(mpq_to_rational(sc, sc->mpq_1));
    case T_BIG_REAL:
      mpfr_abs(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->abs_symbol, sc->type_names[T_REAL]));
    }
}

static s7_pointer g_abs(s7_scheme *sc, s7_pointer args)
{
  #define H_abs "(abs x) returns the absolute value of the real number x"
  #define Q_abs s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_real_symbol)
  return(abs_p_p(sc, car(args)));
}

#ifdef __TINYC__
  static s7_double abs_d_d(s7_double x) {return((x < 0) ? (-x) : x);} /* signbit is very slow in tcc */
#else
  static s7_double abs_d_d(s7_double x) {return((signbit(x)) ? (-x) : x);}
#endif
static s7_int abs_i_i(s7_int x) {return((x < 0) ? (-x) : x);}
/* (abs|magnitude -9223372036854775808) won't work here */


/* -------------------------------- magnitude -------------------------------- */
static double my_hypot(double x, double y)
{
  /* useless: if (x == 0.0) return(fabs(y)); if (y == 0.0) return(fabs(x)); if (is_NaN(x)) return(x); if (is_NaN(y)) return(y); */
  if ((fabs(x) < 1.0e6) && (fabs(y) < 1.0e6)) /* max error is ca. e-14 */
    return(sqrt(x * x + y * y));              /* timing diffs: 62 for this form, 107 if just libm's hypot */
  return(hypot(x, y));                        /* libm's hypot protects against over/underflow */
}

static s7_pointer magnitude_p_p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_complex(x))
    return(make_real(sc, my_hypot(real_part(x), imag_part(x)))); /* was reversed? 8-Nov-22 */

  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) < 0)
	{
	  if (integer(x) == s7_Int64_Min) return(mostfix);
	  /* (magnitude -9223372036854775808) -> -9223372036854775808
	   *   same thing happens in abs, lcm and gcd: (gcd -9223372036854775808) -> -9223372036854775808
	   */
	  return(make_integer(sc, -integer(x)));
	}
      return(x);
    case T_RATIO:
      return((numerator(x) < 0) ? make_simpler_ratio(sc, -numerator(x), denominator(x)) : x);
    case T_REAL:
      if (is_NaN(real(x)))                 /* (magnitude -nan.0) -> +nan.0, not -nan.0 */
	return((nan_payload(real(x)) > 0) ? x : real_NaN);
      return((signbit(real(x))) ? make_real(sc, -real(x)) : x);
#if With_Gmp
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
      return(abs_p_p(sc, x));
    case T_BIG_COMPLEX:
      mpc_abs(sc->mpfr_1, big_complex(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->magnitude_symbol, a_number_string));
    }
}

static s7_pointer g_magnitude(s7_scheme *sc, s7_pointer args)
{
  #define H_magnitude "(magnitude z) returns the magnitude of z"
  #define Q_magnitude s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  return(magnitude_p_p(sc, car(args)));
}

static s7_int magnitude_i_i(s7_int x) {return((x < 0) ? (-x) : x);}
static s7_double magnitude_d_d(s7_double x) {return((signbit(x)) ? (-x) : x);}
static s7_pointer magnitude_p_z(s7_scheme *sc, s7_pointer z) {return(make_real(sc, my_hypot(real_part(z), imag_part(z))));}

#if 0
static s7_pointer magnitude_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
#if !With_Gmp
  s7_pointer arg = cadr(expr);
  if ((is_pair(arg)) && (has_fn(arg)) && (fn_proc(arg) == complex_vector_ref_p_pi))
    set_fn_direct(arg, complex_vector_ref_p_pi_wrapped);
#endif
  return(func);
}
#endif

/* -------------------------------- rationalize -------------------------------- */
#if With_Gmp

static rat_locals_t *init_rat_locals_t(s7_scheme *sc)
{
  rat_locals_t *r = (rat_locals_t *)Malloc(sizeof(rat_locals_t));
  sc->ratloc = r;
  mpz_inits(r->i, r->i0, r->i1, r->n, r->p0, r->q0, r->r, r->r1, r->p1, r->q1, r->old_p1, r->old_q1, NULL);
  mpq_init(r->q);
  mpfr_inits2(sc->bignum_precision, r->error, r->ux, r->x0, r->x1, r->val, r->e0, r->e1, r->e0p, r->e1p, r->old_e0, r->old_e1, r->old_e0p, NULL);
  return(r);
}

static void free_rat_locals(s7_scheme *sc)
{
  rat_locals_t *r = sc->ratloc;
  mpz_clears(r->i, r->i0, r->i1, r->n, r->p0, r->q0, r->r, r->r1, r->p1, r->q1, r->old_p1, r->old_q1, NULL);
  mpq_clear(r->q);
  mpfr_clears(r->error, r->ux, r->x0, r->x1, r->val, r->e0, r->e1, r->e0p, r->e1p, r->old_e0, r->old_e1, r->old_e0p, NULL);
  free(r);
}

static s7_pointer big_rationalize(s7_scheme *sc, s7_pointer args)
{
  /* can return be non-rational? */
  /* currently (rationalize 1/0 1e18) -> 0
   * remember to pad with many trailing zeros:
   *    (rationalize 0.1 0)                -> 3602879701896397/36028797018963968
   *    (rationalize 0.1000000000000000 0) -> 1/10
   * perhaps gmp number reader used if gmp -- could this be the trailing zeros problem?  (why is the non-gmp case ok?)
   *         also the bignum function is faking it.
   *         (rationalize (real-part (bignum "0.1+i")) 0) -> 3602879701896397/36028797018963968
   * a confusing case:
   *   (rationalize 5925563891587147521650777143.74135805596e05) should be 148139097289678688041269428593533951399/250000
   * but that requires more than 128 bits of bignum-precision.
   */

  const s7_pointer num = car(args);
  rat_locals_t *r = (sc->ratloc) ? sc->ratloc : init_rat_locals_t(sc);

  switch (type(num))
    {
    case T_INTEGER:
      mpfr_set_si(r->ux, integer(num), Mpfr_Rndn);
      break;
    case T_RATIO:
      mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
      mpfr_set_q(r->ux, sc->mpq_1, Mpfr_Rndn);
      break;
    case T_REAL:
      if (is_NaN(real(num)))
	out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, num, it_is_nan_string);
      if (is_inf(real(num)))
	out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, num, it_is_infinite_string);
      mpfr_set_d(r->ux, real(num), Mpfr_Rndn);
      break;
    case T_BIG_INTEGER:
      mpfr_set_z(r->ux, big_integer(num), Mpfr_Rndn);
      break;
    case T_BIG_RATIO:
      mpfr_set_q(r->ux, big_ratio(num), Mpfr_Rndn);
      break;
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(num)))
	out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, num, it_is_nan_string);
      if (mpfr_inf_p(big_real(num)))
	out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, num, it_is_infinite_string);
      mpfr_set(r->ux, big_real(num), Mpfr_Rndn);
      break;
    case T_COMPLEX: case T_BIG_COMPLEX:
      wrong_type_error_nr(sc, sc->rationalize_symbol, 1, num, sc->type_names[T_REAL]);
    default:
      return(method_or_bust(sc, num, sc->rationalize_symbol, args, sc->type_names[T_REAL], 1));
    }

  if (is_null(cdr(args)))
    mpfr_set_d(r->error, sc->default_rationalize_error, Mpfr_Rndn);
  else
    {
      const s7_pointer err = cadr(args);
      switch (type(err))
	{
	case T_INTEGER:
	  mpfr_set_si(r->error, integer(err), Mpfr_Rndn);
	  break;
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(err), denominator(err));
	  mpfr_set_q(r->error, sc->mpq_1, Mpfr_Rndn);
	  break;
	case T_REAL:
	  if (is_NaN(real(err)))
	    out_of_range_error_nr(sc, sc->rationalize_symbol, int_two, err, it_is_nan_string);
	  if (is_inf(real(err)))
	    return(int_zero);
	  mpfr_set_d(r->error, real(err), Mpfr_Rndn);
	  break;
	case T_BIG_INTEGER:
	  mpfr_set_z(r->error, big_integer(err), Mpfr_Rndn);
	  break;
	case T_BIG_RATIO:
	  mpfr_set_q(r->error, big_ratio(err), Mpfr_Rndn);
	  break;
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(err)))
	    out_of_range_error_nr(sc, sc->rationalize_symbol, int_two, err, it_is_nan_string);
	  if (mpfr_inf_p(big_real(err)))
	    return(int_zero);
	  mpfr_set(r->error, big_real(err), Mpfr_Rndn);
	  break;
	case T_COMPLEX:	case T_BIG_COMPLEX:
	  wrong_type_error_nr(sc, sc->rationalize_symbol, 2, err, sc->type_names[T_REAL]);
	default:
	  return(method_or_bust(sc, err, sc->rationalize_symbol, args, sc->type_names[T_REAL], 2));
	}
      mpfr_abs(r->error, r->error, Mpfr_Rndn);
    }

  mpfr_set(r->x0, r->ux, Mpfr_Rndn);            /* x0 = ux - error */
  mpfr_sub(r->x0, r->x0, r->error, Mpfr_Rndn);
  mpfr_set(r->x1, r->ux, Mpfr_Rndn);            /* x1 = ux + error */
  mpfr_add(r->x1, r->x1, r->error, Mpfr_Rndn);
  mpfr_get_z(r->i, r->x0, MPFR_RNDU);           /* i = ceil(x0) */

  if (mpfr_cmp_ui(r->error, 1) >= 0)            /* if (error >= 1.0) */
    {
      if (mpfr_cmp_ui(r->x0, 0) < 0)            /* if (x0 < 0) */
	{
	  if (mpfr_cmp_ui(r->x1, 0) < 0)        /*   if (x1 < 0) */
	    mpfr_get_z(r->n, r->x1, MPFR_RNDD); /*     num = floor(x1) */
	  else mpz_set_ui(r->n, 0);             /*   else num = 0 */
	}
      else mpz_set(r->n, r->i);                 /* else num = i */
      return(mpz_to_integer(sc, r->n));
    }

  if (mpfr_cmp_z(r->x1, r->i) >= 0)             /* if (x1 >= i) */
    {
      if (mpz_cmp_ui(r->i, 0) >= 0)             /* if (i >= 0) */
	mpz_set(r->n, r->i);                    /*   num = i */
      else mpfr_get_z(r->n, r->x1, MPFR_RNDD);  /* else num = floor(x1) */
      return(mpz_to_integer(sc, r->n));
    }

  mpfr_get_z(r->i0, r->x0, MPFR_RNDD);          /* i0 = floor(x0) */
  mpfr_get_z(r->i1, r->x1, MPFR_RNDU);          /* i1 = ceil(x1) */

  mpz_set(r->p0, r->i0);                        /* p0 = i0 */
  mpz_set_ui(r->q0, 1);                         /* q0 = 1 */
  mpz_set(r->p1, r->i1);                        /* p1 = i1 */
  mpz_set_ui(r->q1, 1);                         /* q1 = 1 */
  mpfr_sub_z(r->e0, r->x0, r->i1, Mpfr_Rndn);   /* e0 = i1 - x0 */
  mpfr_neg(r->e0, r->e0, Mpfr_Rndn);
  mpfr_sub_z(r->e1, r->x0, r->i0, Mpfr_Rndn);   /* e1 = x0 - i0 */
  mpfr_sub_z(r->e0p, r->x1, r->i1, Mpfr_Rndn);  /* e0p = i1 - x1 */
  mpfr_neg(r->e0p, r->e0p, Mpfr_Rndn);
  mpfr_sub_z(r->e1p, r->x1, r->i0, Mpfr_Rndn);  /* e1p = x1 - i0 */

  while (true)
    {
      mpfr_set_z(r->val, r->p0, Mpfr_Rndn);
      mpfr_div_z(r->val, r->val, r->q0, Mpfr_Rndn);  /* val = p0/q0 */

      if (((mpfr_lessequal_p(r->x0, r->val)) &&      /* if ((x0 <= val) && (val <= x1)) */
	   (mpfr_lessequal_p(r->val, r->x1))) ||
	  (mpfr_cmp_ui(r->e1, 0) == 0) ||
	  (mpfr_cmp_ui(r->e1p, 0) == 0))
	/* these last 2 are probably not needed -- they protect against running out of bits in the non-gmp case above */
	{
	  mpq_set_num(r->q, r->p0);                /* return(p0/q0) */
	  mpq_set_den(r->q, r->q0);
	  return(mpq_to_rational(sc, r->q));
	}
      mpfr_div(r->val, r->e0, r->e1, Mpfr_Rndn);
      mpfr_get_z(r->r, r->val, MPFR_RNDD);         /* r = floor(e0/e1) */
      mpfr_div(r->val, r->e0p, r->e1p, Mpfr_Rndn);
      mpfr_get_z(r->r1, r->val, MPFR_RNDU);        /* r1 = ceil(e0p/e1p) */
      if (mpz_cmp(r->r1, r->r) < 0)                /* if (r1 < r) */
	mpz_set(r->r, r->r1);                      /*   r = r1 */
      mpz_set(r->old_p1, r->p1);                   /* old_p1 = p1 */
      mpz_set(r->p1, r->p0);                       /* p1 = p0 */
      mpz_set(r->old_q1, r->q1);                   /* old_q1 = q1 */
      mpz_set(r->q1, r->q0);                       /* q1 = q0 */
      mpfr_set(r->old_e0, r->e0, Mpfr_Rndn);       /* old_e0 = e0 */
      mpfr_set(r->e0, r->e1p, Mpfr_Rndn);          /* e0 = e1p */
      mpfr_set(r->old_e0p, r->e0p, Mpfr_Rndn);     /* old_e0p = e0p */
      mpfr_set(r->e0p, r->e1, Mpfr_Rndn);          /* e0p = e1 */
      mpfr_set(r->old_e1, r->e1, Mpfr_Rndn);       /* old_e1 = e1 */
      mpz_mul(r->p0, r->p0, r->r);                 /* p0 = old_p1 + r * p0 */
      mpz_add(r->p0, r->p0, r->old_p1);
      mpz_mul(r->q0, r->q0, r->r);                 /* q0 = old_q1 + r * q0 */
      mpz_add(r->q0, r->q0, r->old_q1);
      mpfr_mul_z(r->e1, r->e1p, r->r, Mpfr_Rndn);  /* e1 = old_e0p - r * e1p */
      mpfr_sub(r->e1, r->old_e0p, r->e1, Mpfr_Rndn);
      mpfr_mul_z(r->e1p, r->old_e1, r->r, Mpfr_Rndn);/* e1p = old_e0 - r * old_e1 */
      mpfr_sub(r->e1p, r->old_e0, r->e1p, Mpfr_Rndn);
    }
}
#endif

static s7_pointer g_rationalize(s7_scheme *sc, s7_pointer args)
{
  #define H_rationalize "(rationalize x err) returns the ratio with smallest denominator within err of x"
  #define Q_rationalize s7_make_signature(sc, 3, sc->is_rational_symbol, sc->is_real_symbol, sc->is_real_symbol)
  /* I can't find a case where this returns a non-rational result */

  s7_double err;
  const s7_pointer x = car(args);

#if With_Gmp
  if (is_big_number(x))
    return(big_rationalize(sc, args));
#endif
  if (!is_real(x))
    return(method_or_bust(sc, x, sc->rationalize_symbol, args, sc->type_names[T_REAL], 1));
  if (is_null(cdr(args)))
    err = sc->default_rationalize_error;
  else
    {
      const s7_pointer ex = cadr(args);
#if With_Gmp
      if (is_big_number(ex))
	return(big_rationalize(sc, args));
#endif
      if (!is_real(ex))
	return(method_or_bust(sc, ex, sc->rationalize_symbol, args, sc->type_names[T_REAL], 2));
      err = real_to_double(sc, ex, "rationalize");
      if (is_NaN(err))
	out_of_range_error_nr(sc, sc->rationalize_symbol, int_two, ex, it_is_nan_string);
      if (err < 0.0) err = -err;
    }

  switch (type(x))
    {
    case T_INTEGER:
      {
	s7_int a, b, pa;
	if (err < 1.0) return(x);
	a = integer(x);
	pa = (a < 0) ? -a : a;
	if (err >= pa) return(int_zero);
	b = (s7_int)err;
	pa -= b;
	return(make_integer(sc, (a < 0) ? -pa : pa));
      }
    case T_RATIO:
      if (err == 0.0)
	return(x);
    case T_REAL:
      {
	const s7_double rat = s7_real(x); /* possible fall through from above */
	s7_int numer = 0, denom = 1;
	if ((is_NaN(rat)) || (is_inf(rat)))
	  out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, x, a_normal_real_string);
	if (err >= fabs(rat))
	  return(int_zero);
#if With_Gmp
	if (fabs(rat) > Rationalize_Limit)
	  return(big_rationalize(sc, set_plist_2(sc, x, wrap_real(sc, err))));
#else
	if (fabs(rat) > Rationalize_Limit)
	  out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, x, it_is_too_large_string);
#endif
	if ((fabs(rat) + fabs(err)) < 1.0e-18)
	  err = 1.0e-18;
	/* (/ 1.0 most-positive-fixnum) is 1.0842021e-19, so if we let err be less than that,
	 * (rationalize 1e-19 1e-20) hangs, but this only affects the initial ceiling, I believe.
	 */
	if (fabs(rat) < fabs(err))
	  return(int_zero);
	return((c_rationalize(rat, err, &numer, &denom)) ? make_simpler_ratio_or_integer(sc, numer, denom) : sc->F);
      }}
  if (s7_Debugging) fprintf(stderr, "%s[%d]: we should not be here\n", __func__, __LINE__);
  return(sc->F); /* make compiler happy */
}

static s7_int rationalize_i_i(s7_int x) {return(x);}
static s7_pointer rationalize_p_i(s7_scheme *sc, s7_int x) {return(make_integer(sc, x));}
static s7_pointer rationalize_p_d(s7_scheme *sc, s7_double x)
{
  if ((is_NaN(x)) || (is_inf(x)))
    out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, wrap_real(sc, x), a_normal_real_string); /* was make_real, also below */
  if (fabs(x) > Rationalize_Limit)
#if With_Gmp
    return(big_rationalize(sc, set_plist_1(sc, wrap_real(sc, x))));
#else
    out_of_range_error_nr(sc, sc->rationalize_symbol, int_one, wrap_real(sc, x), it_is_too_large_string);
#endif
  return(s7_rationalize(sc, x, sc->default_rationalize_error));
}


/* -------------------------------- angle -------------------------------- */
static s7_pointer g_angle(s7_scheme *sc, s7_pointer args)
{
  #define H_angle "(angle z) returns the angle of z"
  #define Q_angle s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)

  const s7_pointer x = car(args);  /* (angle inf+infi) -> 0.78539816339745 ? I think this should be -pi < ang <= pi */
  switch (type(x))
    {
    case T_INTEGER: return((integer(x) < 0) ? real_pi : int_zero);
    case T_RATIO:   return((numerator(x) < 0) ? real_pi : int_zero);
    case T_COMPLEX: return(make_real(sc, atan2(imag_part(x), real_part(x))));

    case T_REAL:
      if (is_NaN(real(x))) return(x);
      return((real(x) < 0.0) ? real_pi : real_zero);
#if With_Gmp
    case T_BIG_INTEGER: return((mpz_cmp_ui(big_integer(x), 0) >= 0) ? int_zero : big_pi(sc));
    case T_BIG_RATIO:   return((mpq_cmp_ui(big_ratio(x), 0, 1) >= 0) ? int_zero : big_pi(sc));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x))) return(x);
      return((mpfr_cmp_d(big_real(x), 0.0) >= 0) ? real_zero : big_pi(sc));
    case T_BIG_COMPLEX:
      {
	s7_pointer new_bgf;
	new_cell(sc, new_bgf, T_BIG_REAL);
	big_real_bgf(new_bgf) = alloc_bigflt(sc);
	add_big_real(sc, new_bgf);
	mpc_arg(big_real(new_bgf), big_complex(x), Mpfr_Rndn);
	return(new_bgf);
      }
#endif
    default:
      return(method_or_bust_p(sc, x, sc->angle_symbol, a_number_string));
    }
}

static s7_double angle_d_d(s7_double x) {return((is_NaN(x)) ? x : ((x < 0.0) ? M_PI : 0.0));}


/* -------------------------------- complex -------------------------------- */

static s7_pointer complex_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if With_Gmp
  if ((is_big_number(x)) || (is_big_number(y)))
    {
      s7_pointer new_num = NULL; /* placate the compiler */
      if (!is_real(x))
	return(method_or_bust(sc, x, sc->complex_symbol, set_plist_2(sc, x, y), sc->type_names[T_REAL], 1));
      if (!is_real(y))
	return(method_or_bust(sc, y, sc->complex_symbol, set_plist_2(sc, x, y), sc->type_names[T_REAL], 2));

      switch (type(y))
	{
	case T_INTEGER: case T_RATIO: case T_REAL:
	  {
	    s7_double iz = s7_real(y);
	    if (iz == 0.0)                      /* imag-part is 0.0 */
	      return(x);
	    new_cell(sc, new_num, T_BIG_COMPLEX);
	    big_complex_bgc(new_num) = alloc_bigcmp(sc);
	    mpfr_set_d(mpc_imagref(big_complex(new_num)), iz, Mpfr_Rndn);
	  }
	  break;
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y))) return(x);
	  new_cell(sc, new_num, T_BIG_COMPLEX);
	  big_complex_bgc(new_num) = alloc_bigcmp(sc);
	  mpfr_set(mpc_imagref(big_complex(new_num)), big_real(y), Mpfr_Rndn);
	  break;
	case T_BIG_RATIO:
	  new_cell(sc, new_num, T_BIG_COMPLEX);
	  big_complex_bgc(new_num) = alloc_bigcmp(sc);
	  mpfr_set_q(mpc_imagref(big_complex(new_num)), big_ratio(y), Mpfr_Rndn);
	  break;
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0) return(x);
	  new_cell(sc, new_num, T_BIG_COMPLEX);
	  big_complex_bgc(new_num) = alloc_bigcmp(sc);
	  mpfr_set_z(mpc_imagref(big_complex(new_num)), big_integer(y), Mpfr_Rndn);
	  break;
	}
      switch (type(x))
	{
	case T_INTEGER: case T_RATIO: case T_REAL:
	  mpfr_set_d(mpc_realref(big_complex(new_num)), s7_real(x), Mpfr_Rndn);
	  break;
	case T_BIG_REAL:
	  mpfr_set(mpc_realref(big_complex(new_num)), big_real(x), Mpfr_Rndn);
	  break;
	case T_BIG_RATIO:
	  mpfr_set_q(mpc_realref(big_complex(new_num)), big_ratio(x), Mpfr_Rndn);
	  break;
	case T_BIG_INTEGER:
	  mpfr_set_z(mpc_realref(big_complex(new_num)), big_integer(x), Mpfr_Rndn);
	  break;
	}
      add_big_complex(sc, new_num);
      return(new_num);
    }
#endif
  if ((is_t_real(x)) && (is_t_real(y))) return((real(y) == 0.0) ? x : make_complex_not_0i(sc, real(x), real(y)));
  switch (type(y))
    {
    case T_INTEGER:
      switch (type(x))
	{
	case T_INTEGER: return((integer(y) == 0) ? x : make_complex_not_0i(sc, (s7_double)integer(x), (s7_double)integer(y)));
	  /* these int->dbl's are problematic:
	   *   (complex 9223372036854775807 9007199254740995): 9223372036854776000.0+9007199254740996.0i
	   * should we raise an error?
	   */
	case T_RATIO:  return((integer(y) == 0) ? x : make_complex_not_0i(sc, (s7_double)fraction(x), (s7_double)integer(y)));
	case T_REAL:   return((integer(y) == 0) ? x : make_complex_not_0i(sc, real(x), (s7_double)integer(y)));
	default:       return(method_or_bust(sc, x, sc->complex_symbol, set_plist_2(sc, x, y), sc->type_names[T_REAL], 1));
	}
    case T_RATIO:
      switch (type(x))
	{
	case T_INTEGER: return(make_complex(sc, (s7_double)integer(x), (s7_double)fraction(y))); /* can fraction be 0.0? */
	case T_RATIO:   return(make_complex(sc, (s7_double)fraction(x), (s7_double)fraction(y)));
	case T_REAL:    return(make_complex(sc, real(x), (s7_double)fraction(y)));
	default:	return(method_or_bust(sc, x, sc->complex_symbol, set_plist_2(sc, x, y), sc->type_names[T_REAL], 1));
	}
    case T_REAL:
      switch (type(x))
	{
	case T_INTEGER: return((real(y) == 0.0) ? x : make_complex_not_0i(sc, (s7_double)integer(x), real(y)));
	case T_RATIO:	return((real(y) == 0.0) ? x : make_complex_not_0i(sc, (s7_double)fraction(x), real(y)));
	case T_REAL:    return((real(y) == 0.0) ? x : make_complex_not_0i(sc, real(x), real(y)));
	default:	return(method_or_bust(sc, x, sc->complex_symbol, set_plist_2(sc, x, y), sc->type_names[T_REAL], 1));
	}
    default:
      return(method_or_bust(sc, (is_let(x)) ? x : y, sc->complex_symbol, set_plist_2(sc, x, y), sc->type_names[T_REAL], 2));
    }
}

static s7_pointer complex_p_pp_wrapped(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (is_t_real(x))
    {
      if (is_t_real(y)) return(wrap_complex(sc, real(x), real(y)));
      if (is_t_integer(y)) return(wrap_complex(sc, real(x), (s7_double)integer(y)));
    }
  else
    if (is_t_integer(x))
      {
	if (is_t_integer(y)) return(wrap_complex(sc, (s7_double)integer(x), (s7_double)integer(y)));
	if (is_t_real(y)) return(wrap_complex(sc, (s7_double)integer(x), real(y)));
      }
  return(complex_p_pp(sc, x, y));
}

static s7_pointer g_complex(s7_scheme *sc, s7_pointer args)
{
  #define H_complex "(complex x1 x2) returns a complex number with real-part x1 and imaginary-part x2"
  #define Q_complex s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)
  return(complex_p_pp(sc, car(args), cadr(args)));
}

static s7_pointer complex_wrapped(s7_scheme *sc, s7_pointer args) {return(complex_p_pp_wrapped(sc, car(args), cadr(args)));}
static s7_pointer complex_p_ii_wrapped(s7_scheme *sc, s7_int x, s7_int y) {return(wrap_complex(sc, (s7_double)x, (s7_double)y));} /* tcomplex p_ii_ok */
static s7_pointer complex_p_dd_wrapped(s7_scheme *sc, s7_double x, s7_double y) {return(wrap_complex(sc, x, y));}

static s7_pointer complex_p_ii(s7_scheme *sc, s7_int x, s7_int y)
{
  return((y == 0.0) ? make_integer(sc, x) : make_complex_not_0i(sc, (s7_double)x, (s7_double)y));
}

static s7_pointer complex_p_dd(s7_scheme *sc, s7_double x, s7_double y)
{
  return((y == 0.0) ? make_real(sc, x) : make_complex_not_0i(sc, x, y));
}


/* -------------------------------- bignum -------------------------------- */
static s7_pointer g_bignum(s7_scheme *sc, s7_pointer args)
{
  #define H_bignum "(bignum val (radix 10)) returns a multiprecision version of the string 'val'. If the argument is a number \
bignum returns that number as a bignum"
#if With_Gmp
  #define Q_bignum s7_make_signature(sc, 3, sc->is_bignum_symbol, s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_string_symbol), sc->is_integer_symbol)
#else
  #define Q_bignum s7_make_signature(sc, 3, \
                     s7_make_signature(sc, 2, sc->is_number_symbol, sc->not_symbol), \
                     s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_string_symbol), \
                     sc->is_integer_symbol)
#endif

  s7_pointer num = car(args);
  if (is_number(num))
    {
      if (!is_null(cdr(args)))
	error_nr(sc, make_symbol(sc, "bignum-error", 12),
		 set_elist_2(sc, wrap_string(sc, "bignum of a number takes only one argument: ~S", 46), args));
#if With_Gmp
      switch (type(num))
	{
	case T_INTEGER: return(s7_int_to_big_integer(sc, integer(num)));
	case T_RATIO:   return(s7_int_to_big_ratio(sc, numerator(num), denominator(num)));
	case T_REAL:    return(s7_double_to_big_real(sc, real(num)));
	case T_COMPLEX: return(s7_double_to_big_complex(sc, real_part(num), imag_part(num)));
	default:        return(num);
	}
#else
      return(num);
#endif
    }
  num = string_to_number_1(sc, args, sc->bignum_symbol);
  if (is_false(sc, num))                                    /* (bignum "1/3.0") */
    error_nr(sc, make_symbol(sc, "bignum-error", 12),
	     set_elist_2(sc, wrap_string(sc, "bignum string argument does not represent a number: ~S", 54), car(args))); /* car(args) to get original */
#if With_Gmp
  switch (type(num))
    {
    case T_INTEGER:   return(s7_int_to_big_integer(sc, integer(num)));
    case T_RATIO:     return(s7_int_to_big_ratio(sc, numerator(num), denominator(num)));
    case T_COMPLEX:   return(s7_number_to_big_complex(sc, num));
    case T_REAL:
      if (is_NaN(real(num))) return(num);
      return(s7_double_to_big_real(sc, real(num)));
      /* 9Sep21: this was return(string_to_big_real(sc, string_value(car(args)), (is_pair(cdr(args))) ? s7_integer_clamped_if_gmp(sc, cadr(args)) : 10)); */
    default:
      return(num);
    }
#else
  return(num);
#endif
}


/* -------------------------------- exp -------------------------------- */
#if !Have_Complex_Numbers
  static s7_pointer no_complex_numbers_string;
#endif

#define Exp_Limit 100.0

#if With_Gmp
static s7_pointer exp_1(s7_scheme *sc, s7_double x)
{
  mpfr_set_d(sc->mpfr_1, x, Mpfr_Rndn);
  mpfr_exp(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
}

static s7_pointer exp_2(s7_scheme *sc, s7_double x, s7_double y)
{
  mpc_set_d_d(sc->mpc_1, x, y, MPC_RNDNN);
  mpc_exp(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
  return(mpc_to_number(sc, sc->mpc_1));
}
#endif

static s7_pointer exp_p_p(s7_scheme *sc, s7_pointer x)
{
  s7_double z;
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_one);                       /* (exp 0) -> 1 */
      z = (s7_double)integer(x);
#if With_Gmp
      if (fabs(z) > Exp_Limit)
	return(exp_1(sc, z));
#endif
      return(make_real(sc, exp(z)));

    case T_RATIO:
      z = (s7_double)fraction(x);
#if With_Gmp
      if (fabs(z) > Exp_Limit)
	return(exp_1(sc, z));
#endif
      return(make_real(sc, exp(z)));

    case T_REAL:
#if With_Gmp
      if (fabs(real(x)) > Exp_Limit)
	return(exp_1(sc, real(x)));
#endif
      return(make_real(sc, exp(real(x))));

    case T_COMPLEX:
#if Have_Complex_Numbers
#if With_Gmp
      if ((fabs(real_part(x)) > Exp_Limit) ||
	  (fabs(imag_part(x)) > Exp_Limit))
	return(exp_2(sc, real_part(x), imag_part(x)));
#endif
      return(c_complex_to_s7(sc, cexp(to_c_complex(x))));
      /* this is inaccurate for large arguments:
       *   (exp 0+1e20i) -> -0.66491178990701-0.74692189125949i, not 7.639704044417283004001468027378811228331E-1-6.45251285265780844205811711312523007406E-1i
       */
#else
      out_of_range_error_nr(sc, sc->exp_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_exp(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_exp(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_exp(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_exp(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->exp_symbol, a_number_string));
    }
}

static s7_pointer g_exp(s7_scheme *sc, s7_pointer args)
{
  #define H_exp "(exp z) returns e^z, (exp 1) is 2.718281828459"
  #define Q_exp sc->pl_nn
  return(exp_p_p(sc, car(args)));
}

static s7_double exp_d_d(s7_double x) {return(exp(x));}
static s7_pointer exp_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, exp(x)));}


/* -------------------------------- log -------------------------------- */
#if __cplusplus
  #define Log_2 1.4426950408889634074
#else
  #define Log_2 1.4426950408889634073599246810018921L /* (/ (log 2.0)) */
#endif

#if With_Gmp
static s7_pointer big_log(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer x = car(args);
  s7_pointer base = NULL;

  if (!is_number(x))
    return(method_or_bust(sc, x, sc->log_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      base = cadr(args);
      if (!is_number(base))
	return(method_or_bust(sc, base, sc->log_symbol, args, a_number_string, 2));
    }

  if (is_real(x))
    {
      s7_pointer result = any_real_to_mpfr(sc, x, sc->mpfr_1);
      if (result == real_NaN) return(result);
      if ((is_positive(sc, x)) &&
	  ((!base) ||
	   ((is_real(base)) && (is_positive(sc, base)))))
	{
	  if (result) return(result);
	  mpfr_log(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  if (base)
	    {
	      result = any_real_to_mpfr(sc, base, sc->mpfr_2);
	      if (result)
		return((result == real_infinity) ? real_zero : result);
	      if (mpfr_zero_p(sc->mpfr_2))
		out_of_range_error_nr(sc, sc->log_symbol, int_two, base, wrap_string(sc, "can't be zero", 13));
	      mpfr_log(sc->mpfr_2, sc->mpfr_2, Mpfr_Rndn);
	      mpfr_div(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	    }
	  if ((mpfr_integer_p(sc->mpfr_1)) && ((is_rational(x)) && ((!base) || (is_rational(base)))))
	    return(mpfr_to_integer(sc, sc->mpfr_1));
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}}
  if (base)
    {
      s7_pointer result = any_number_to_mpc(sc, base, sc->mpc_2);
      if (result)
	return((result == real_infinity) ? real_zero : complex_NaN);
      if (mpc_zero_p(sc->mpc_2))
	out_of_range_error_nr(sc, sc->log_symbol, int_two, base, wrap_string(sc, "can't be zero", 13));
    }
  {
    s7_pointer result = any_number_to_mpc(sc, x, sc->mpc_1);
    if (result)
      {
	if ((result == real_infinity) && (base) && ((is_negative(sc, x))))
	  return(make_complex_not_0i(sc, Infinity, -Nan));
	return((result == real_NaN) ? complex_NaN : result);
      }}
  mpc_log(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
  if (base)
    {
      mpc_log(sc->mpc_2, sc->mpc_2, MPC_RNDNN);
      mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
    }
  if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
    return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
  return(mpc_to_number(sc, sc->mpc_1));
}
#endif

static s7_pointer g_int_log2(s7_scheme *sc, s7_pointer args)
{
  s7_int ix = integer(car(args));
  s7_double fx = log2((double)ix);
  return(((ix & (ix - 1)) == 0) ? make_integer(sc, (s7_int)s7_round(fx)) : make_real(sc, fx));
}

static s7_pointer g_log(s7_scheme *sc, s7_pointer args)
{
  #define H_log "(log z1 (z2 e)) returns log(z1) / log(z2) where z2 (the base) defaults to e: (log 8 2) = 3"
  #define Q_log sc->pcl_n

  const s7_pointer x = car(args);

#if With_Gmp
  if (is_big_number(x)) return(big_log(sc, args));
#endif

  if (!is_number(x))
    return(method_or_bust(sc, x, sc->log_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      const s7_pointer y = cadr(args);
      if (!is_number(y))
	return(method_or_bust(sc, y, sc->log_symbol, args, a_number_string, 2));

#if With_Gmp
      if (is_big_number(y)) return(big_log(sc, args));
#endif
      if ((is_t_integer(y)) && (integer(y) == 2))
	{
	  /* (define (2^n? x) (and (not (zero? x)) (zero? (logand x (- x 1))))) */
	  if (is_t_integer(x))
	    {
	      s7_int ix = integer(x);
	      if (ix > 0)
		{
		  s7_double fx;
#if (__ANDROID__) || (MS_Windows)
		  /* just a guess -- log2 gets a warning in gcc 4.3.2, but not in 4.4.4 */
		  fx = log((double)ix) * Log_2;
#else
		  fx = log2((double)ix);
#endif
		  /* (s7_int)fx rounds (log 8 2) to 2 in FreeBSD! */
		  return(((ix & (ix - 1)) == 0) ? make_integer(sc, (s7_int)s7_round(fx)) : make_real(sc, fx));
		}}
	  if ((is_real(x)) &&
	      (is_positive(sc, x)))
	    return(make_real(sc, log(s7_real(x)) * Log_2));
	  return(c_complex_to_s7(sc, clog(s7_to_c_complex(x)) * Log_2));
	}

      if ((is_t_integer(x)) && (integer(x) == 1) && (is_t_integer(y)) && (integer(y) == 1))  /* (log 1 1) -> 0 (this is NaN in the bignum case) */
	return(int_zero);

      /* (log 1 0) must be 0 since everyone says (expt 0 0) is 1 */
      if (is_zero(y))
	{
	  if ((is_t_integer(y)) && (is_t_integer(x)) && (integer(x) == 1))
	    return(y);
	  out_of_range_error_nr(sc, sc->log_symbol, int_two, y, wrap_string(sc, "can't be zero", 13));
	}

      if ((is_t_real(x)) && (is_NaN(real(x))))
	return(x);
      if (is_one(y))                                     /* this used to raise an error, but the bignum case is simpler if we return inf */
	return((is_one(x)) ? real_zero : real_infinity); /* but (log 1.0 1.0) -> 0.0, currently (log 1/0 1) is inf? */

      if ((is_real(x)) && (is_real(y)) &&
	  (is_positive(sc, x)) && (is_positive(sc, y)))
	{
	  if ((is_rational(x)) && (is_rational(y)))
	    {
	      const s7_double result = log(rational_to_double(sc, x)) / log(rational_to_double(sc, y));
	      const s7_int ires = (s7_int)result;
	      if (result - ires == 0.0)
		return(make_integer(sc, ires));   /* (log 8 2) -> 3 or (log 1/8 2) -> -3 */
	      if (fabs(result) < Rationalize_Limit)
		{
		  s7_int num, den;
		  if (c_rationalize(result, sc->default_rationalize_error, &num, &den))
		      /* && (s7_int_abs(num) < 100) && (s7_int_abs(den) < 100)) *//* why this? */
		    return(make_simpler_ratio_or_integer(sc, num, den));
		}
	      return(make_real(sc, result));
	    }
	  return(make_real(sc, log(s7_real(x)) / log(s7_real(y))));
	}
      if ((is_t_real(x)) && (is_NaN(real(x))))
	return(x);
      if ((is_t_complex(y)) && ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))))
	return(y);
      return(c_complex_to_s7(sc, clog(s7_to_c_complex(x)) / clog(s7_to_c_complex(y))));
    }

  if (!is_real(x))
    return(c_complex_to_s7(sc, clog(s7_to_c_complex(x))));
  if (is_positive(sc, x))
    return(make_real(sc, log(s7_real(x))));
  return(make_complex_not_0i(sc, log(-s7_real(x)), M_PI));
}

static s7_pointer log_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
#if !With_Gmp
  if (args == 2)
    {
      s7_pointer x = cadr(expr), y = caddr(expr);
      if ((is_t_integer(y)) && (integer(y) == 2) && (is_t_integer(x)) && (integer(x) > 0))
	return(sc->int_log2);
    }
#endif
  return(func);
}

/* -------------------------------- sin -------------------------------- */
#define Sin_Limit 1.0e16
#define Sinh_Limit 20.0
/* (- (sinh (bignum 30.0)) (sinh 30.0)): -3.718172657214174140191915872003397016115E-4
 * (- (sinh (bignum 20.0)) (sinh 20.0)): -7.865629467297586346406367346575835463792E-10, slightly worse (e-8) if imag-part
 */

static s7_pointer sin_p_p(s7_scheme *sc, s7_pointer x)
{
#if !With_Gmp
  if (is_t_real(x)) return(make_real(sc, sin(real(x)))); /* range check in gmp case */
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);           /* (sin 0) -> 0 */
#if With_Gmp
      if (integer(x) > Sin_Limit)
	{
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpfr_set_z(sc->mpfr_1, sc->mpz_1, Mpfr_Rndn);
	  mpfr_sin(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, sin((s7_double)(integer(x))))); /* bogus for very large integers, but so is the equivalent real (see Sin_Limit) */

    case T_RATIO:
      return(make_real(sc, sin((s7_double)(fraction(x)))));
    case T_REAL:
      {
	s7_double y = real(x);
#if With_Gmp
	if (fabs(y) > Sin_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
	    mpfr_sin(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, sin(y)));
      }
    case T_COMPLEX:
#if With_Gmp
      if ((fabs(real_part(x)) > Sin_Limit) || (fabs(imag_part(x)) > Sinh_Limit))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sin(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if Have_Complex_Numbers
      return(c_complex_to_s7(sc, csin(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->sin_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_sin(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_sin(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_sin(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_sin(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->sin_symbol, a_number_string));
    }
  /* sin is inaccurate over about 1e30.  There's a way to get true results, but it involves fancy "range reduction" techniques.
   * (sin 1e32): 0.5852334864823946
   *   but it should be 3.901970254333630491697613212893425767786E-1
   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22 !! (it's now a range error)
   *   it should be 5.263007914620499494429139986095833592117E0
   * before comparing imag-part to 0, we need to look for NaN and inf, else:
   *    (sinh 0+0/0i) -> 0.0 and (sinh (log 0.0)) -> inf.0
   */
}

static s7_pointer g_sin(s7_scheme *sc, s7_pointer args)
{
  #define H_sin "(sin z) returns sin(z)"
  #define Q_sin sc->pl_nn
  return(sin_p_p(sc, car(args)));
}

#if With_Gmp
static s7_pointer sin_p_d(s7_scheme *sc, s7_double x)
{
  if (fabs(x) <= Sin_Limit)
    return(make_real(sc, sin(x)));
  mpfr_set_d(sc->mpfr_1, x, Mpfr_Rndn);
  mpfr_sin(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
}
#else
static s7_pointer sin_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, sin(x)));}
#endif

static s7_double sin_d_d(s7_double x) {return(sin(x));}


/* -------------------------------- cos -------------------------------- */
static s7_pointer cos_p_p(s7_scheme *sc, s7_pointer x)
{
#if !With_Gmp
  if (is_t_real(x)) return(make_real(sc, cos(real(x)))); /* range check in gmp case */
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_one);             /* (cos 0) -> 1 */
#if With_Gmp
      if (integer(x) > Sin_Limit)
	{
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpfr_set_z(sc->mpfr_1, sc->mpz_1, Mpfr_Rndn);
	  mpfr_cos(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, cos((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, cos((s7_double)(fraction(x)))));
    case T_REAL: /* if with_gmp */
      {
	s7_double y = real(x);
#if With_Gmp
	if (fabs(y) > Sin_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
	    mpfr_cos(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, cos(y)));
      }
    case T_COMPLEX:
#if With_Gmp
      if ((fabs(real_part(x)) > Sin_Limit) || (fabs(imag_part(x)) > Sinh_Limit))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_cos(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if Have_Complex_Numbers
      return(c_complex_to_s7(sc, ccos(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->cos_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_cos(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_cos(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_cos(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_cos(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->cos_symbol, a_number_string));
    }
}

static s7_pointer g_cos(s7_scheme *sc, s7_pointer args)
{
  #define H_cos "(cos z) returns cos(z)"
  #define Q_cos sc->pl_nn
  return(cos_p_p(sc, car(args)));
}

#if With_Gmp
static s7_pointer cos_p_d(s7_scheme *sc, s7_double x)
{
  if (fabs(x) <= Sin_Limit)
    return(make_real(sc, cos(x)));
  mpfr_set_d(sc->mpfr_1, x, Mpfr_Rndn);
  mpfr_cos(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
}
#else
static s7_pointer cos_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, cos(x)));}
/* the optimizer can replace (cos x) = cos_p_p(x) with cos_p_d(x) if x is real, but x might be 0 so (byte? (cos x)) will return different results */
#endif

static s7_double cos_d_d(s7_double x) {return(cos(x));}


#if !With_Pure_s7
static s7_pointer multiply_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y);

static s7_pointer g_make_polar(s7_scheme *sc, s7_pointer args)
{
  #define H_make_polar "(make-polar magnitude angle) returns (complex (* magnitude (cos angle)) (* magnitude (sin angle)))"
  #define Q_make_polar s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)
  s7_pointer mag = car(args), ang = cadr(args);
  if (!s7_is_real(mag))
    return(method_or_bust_pp(sc, mag, sc->make_polar_symbol, mag, ang, sc->type_names[T_REAL], 1));
  if (!s7_is_real(ang))
    return(method_or_bust_pp(sc, ang, sc->make_polar_symbol, mag, ang, sc->type_names[T_REAL], 2));
  return(complex_p_pp(sc, multiply_p_pp(sc, mag, cos_p_p(sc, ang)), multiply_p_pp(sc, mag, sin_p_p(sc, ang))));
}
#endif


/* -------------------------------- tan -------------------------------- */
#define Tan_Limit 1.0e18

static s7_pointer tan_p_p(s7_scheme *sc, s7_pointer x)
{
#if !With_Gmp
  if (is_t_real(x)) return(make_real(sc, tan(real(x))));
#endif
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                      /* (tan 0) -> 0 */
#if With_Gmp
      if (integer(x) > Tan_Limit)
	{
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpfr_set_z(sc->mpfr_1, sc->mpz_1, Mpfr_Rndn);
	  mpfr_tan(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, tan((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, tan((s7_double)(fraction(x)))));
#if With_Gmp
    case T_REAL:
      if (fabs(real(x)) > Tan_Limit)
	{
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_tan(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
#endif
      return(make_real(sc, tan(real(x))));
    case T_COMPLEX:
#if Have_Complex_Numbers
      if (imag_part(x) > 350.0)
	return(make_complex_not_0i(sc, 0.0, 1.0));
      return((imag_part(x) < -350.0) ? make_complex_not_0i(sc, 0.0, -1.0) : c_complex_to_s7(sc, ctan(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->tan_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_tan(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_tan(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_tan(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      if ((MPC_INEX_IM(mpc_cmp_si_si(big_complex(x), 1, 350))) > 0)
	return(make_complex_not_0i(sc, 0.0, 1.0));
      if ((MPC_INEX_IM(mpc_cmp_si_si(big_complex(x), 1, -350))) < 0)
	return(make_complex_not_0i(sc, 0.0, -1.0));
      mpc_tan(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->tan_symbol, a_number_string));
    }
}

static s7_pointer g_tan(s7_scheme *sc, s7_pointer args)
{
  #define H_tan "(tan z) returns tan(z)"
  #define Q_tan sc->pl_nn
  return(tan_p_p(sc, car(args)));
}

static s7_double tan_d_d(s7_double x) {return(tan(x));}


/* -------------------------------- asin -------------------------------- */
static s7_pointer c_asin(s7_scheme *sc, s7_double x)
{
  s7_double absx = fabs(x), recip;
  s7_complex result;

  if (absx <= 1.0) return(make_real(sc, asin(x)));

  /* otherwise use maxima code: */
  recip = 1.0 / absx;
  result = (M_PI / 2.0) - (s7_complex_i * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip)))));
  return((x < 0.0) ? c_complex_to_s7(sc, -result) : c_complex_to_s7(sc, result));
}

static s7_pointer asin_p_p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_real(x)) return(c_asin(sc, real(x)));
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                    /* (asin 0) -> 0 */
      /* in netBSD, (asin 2) returns 0.25383842987008+0.25383842987008i according to Peter Bex */
      return(c_asin(sc, (s7_double)integer(x)));
    case T_RATIO:
      return(c_asin(sc, (s7_double)fraction(x)));
    case T_COMPLEX:
#if Have_Complex_Numbers
      /* if either real or imag part is very large, use explicit formula, not casin */
      /*   this code taken from sbcl's src/code/irrat.lisp; break is around x+70000000i */
      if ((fabs(real_part(x)) > 1.0e7) ||
	  (fabs(imag_part(x)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z = to_c_complex(x);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);
	  return(make_complex(sc, atan(real_part(x) / creal(sq1mz * sq1pz)), asinh(cimag(sq1pz * conj(sq1mz)))));
	}
      return(c_complex_to_s7(sc, casin(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->asin_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      goto Asin_Big_Real;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      goto Asin_Big_Real;
    case T_BIG_REAL:
      if (mpfr_inf_p(big_real(x)))
	{
	  if (mpfr_cmp_ui(big_real(x), 0) < 0)
	    return(make_complex_not_0i(sc, Nan, Infinity)); /* match non-bignum choice */
	  return(make_complex_not_0i(sc, Nan, -Infinity));
	}
      mpfr_set(sc->mpfr_1, big_real(x), Mpfr_Rndn);
    Asin_Big_Real:
      mpfr_set_ui(sc->mpfr_2, 1, Mpfr_Rndn);
      if (mpfr_cmpabs(sc->mpfr_1, sc->mpfr_2) <= 0)
	{
	  mpfr_asin(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      mpc_set_fr(sc->mpc_1, sc->mpfr_1, MPC_RNDNN);
      mpc_asin(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_COMPLEX:
      mpc_asin(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->asin_symbol, a_number_string));
    }
}

static s7_pointer g_asin(s7_scheme *sc, s7_pointer args)
{
  #define H_asin "(asin z) returns asin(z); (sin (asin x)) = x"
  #define Q_asin sc->pl_nn
  return(asin_p_p(sc, car(args)));
}


/* -------------------------------- acos -------------------------------- */
static s7_pointer c_acos(s7_scheme *sc, s7_double x)
{
  s7_double absx = fabs(x), recip;
  s7_complex result;
  if (absx <= 1.0)
    return(make_real(sc, acos(x)));
  /* else follow maxima again: */
  recip = 1.0 / absx;
  if (x > 0.0)
    result = s7_complex_i * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  else result = M_PI - s7_complex_i * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  return(c_complex_to_s7(sc, result));
}

static s7_pointer acos_p_p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_real(x)) return(c_acos(sc, real(x)));
  switch (type(x))
    {
    case T_INTEGER:
      return((integer(x) == 1) ? int_zero : c_acos(sc, (s7_double)integer(x)));
    case T_RATIO:
      return(c_acos(sc, (s7_double)fraction(x)));
    case T_COMPLEX:
#if Have_Complex_Numbers
      /* if either real or imag part is very large, use explicit formula, not cacos */
      /*   this code taken from sbcl's src/code/irrat.lisp */

      if ((fabs(real_part(x)) > 1.0e7) ||
	  (fabs(imag_part(x)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z = to_c_complex(x);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);	  /* creal(sq1pz) can be 0.0 */
	  if (creal(sq1pz) == 0.0)        /* so the atan arg will be inf, so the real part will be pi/2(?) */
	    return(make_complex(sc, M_PI / 2.0, asinh(cimag(sq1mz * conj(sq1pz)))));
	  return(make_complex(sc, 2.0 * atan(creal(sq1mz) / creal(sq1pz)), asinh(cimag(sq1mz * conj(sq1pz)))));
	}
      return(c_complex_to_s7(sc, cacos(s7_to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->acos_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      goto Acos_Big_Real;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      goto Acos_Big_Real;
    case T_BIG_REAL:
      if (mpfr_inf_p(big_real(x)))
	{
	  if (mpfr_cmp_ui(big_real(x), 0) < 0)
	    return(make_complex_not_0i(sc, -Nan, -Infinity)); /* match non-bignum choice */
	  return(make_complex_not_0i(sc, -Nan, Infinity));
	}
      mpfr_set(sc->mpfr_1, big_real(x), Mpfr_Rndn);
    Acos_Big_Real:
      mpfr_set_ui(sc->mpfr_2, 1, Mpfr_Rndn);
      if (mpfr_cmpabs(sc->mpfr_1, sc->mpfr_2) <= 0)
	{
	  mpfr_acos(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      mpc_set_fr(sc->mpc_1, sc->mpfr_1, MPC_RNDNN);
      mpc_acos(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_COMPLEX:
      mpc_acos(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->acos_symbol, a_number_string));
    }
}

static s7_pointer g_acos(s7_scheme *sc, s7_pointer args)
{
  #define H_acos "(acos z) returns acos(z); (cos (acos 1)) = 1"
  #define Q_acos sc->pl_nn
  return(acos_p_p(sc, car(args)));
}


/* -------------------------------- atan -------------------------------- */
static s7_pointer g_atan(s7_scheme *sc, s7_pointer args)
{
  #define H_atan "(atan z) returns atan(z), (atan y x) returns atan(y/x)"
  #define Q_atan s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_real_symbol)
  /* actually if there are two args, both should be real, but how to express that in the signature? */

  const s7_pointer x = car(args);
  s7_pointer y;
  /* currently (atan inf.0 inf.0) -> 0.78539816339745, and (atan inf.0 -inf.0) -> 2.3561944901923 (etc) */

  if (!is_pair(cdr(args)))
    {
      switch (type(x))
	{
	case T_INTEGER:  return((integer(x) == 0) ? int_zero : make_real(sc, atan((double)integer(x))));
	case T_RATIO:    return(make_real(sc, atan((s7_double)fraction(x))));
	case T_REAL:     return(make_real(sc, atan(real(x))));
	case T_COMPLEX:
#if Have_Complex_Numbers
	  return(c_complex_to_s7(sc, catan(to_c_complex(x))));
#else
	  out_of_range_error_nr(sc, sc->atan_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
	case T_BIG_INTEGER:
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  mpfr_atan(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  mpfr_atan(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_atan(sc->mpfr_1, big_real(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_atan(sc->mpc_1, big_complex(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_p(sc, x, sc->atan_symbol, a_number_string));
	}}

  y = cadr(args);
  /* this is one place where s7 notices -0.0 != 0.0 -- this is apparently built into atan2, so I guess I'll leave it, but:
   *   (atan 0.0 0.0): 0.0, (atan 0.0 -0.0): pi, (atan 0 -0.0): pi, (atan 0 -0) 0.0, (atan 0 -0.0): pi.
   *   so you can sneak up on 0.0 from the left, but you can't fool 0??
   */
  switch (type(x))
    {
    case T_INTEGER: case T_RATIO: case T_REAL:
      if (is_small_real(y))
	return(make_real(sc, atan2(s7_real(x), s7_real(y))));
      if (!is_real(y))
	return(method_or_bust(sc, y, sc->atan_symbol, args, sc->type_names[T_REAL], 2));
#if !With_Gmp
      return(make_real(sc, atan2(s7_real(x), real(y))));
#else
      mpfr_set_d(sc->mpfr_1, s7_real(x), Mpfr_Rndn);
      goto Atan2_Big_Real;
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      goto Atan2_Big_Real;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      goto Atan2_Big_Real;
    case T_BIG_REAL:
      mpfr_set(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      goto Atan2_Big_Real;
#endif
    default:
      return(method_or_bust(sc, x, sc->atan_symbol, args, sc->type_names[T_REAL], 1));
    }
#if With_Gmp
 Atan2_Big_Real:
  if (is_small_real(y))
    mpfr_set_d(sc->mpfr_2, s7_real(y), Mpfr_Rndn);
  else
    if (is_t_big_real(y))
      mpfr_set(sc->mpfr_2, big_real(y), Mpfr_Rndn);
    else
      if (is_t_big_integer(y))
	mpfr_set_z(sc->mpfr_2, big_integer(y), Mpfr_Rndn);
      else
	if (is_t_big_ratio(y))
	  mpfr_set_q(sc->mpfr_2, big_ratio(y), Mpfr_Rndn);
	else return(method_or_bust(sc, y, sc->atan_symbol, args, sc->type_names[T_REAL], 2));
  mpfr_atan2(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
  return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
}

static s7_double atan_d_d(s7_double x) {return(atan(x));}
static s7_double atan_d_dd(s7_double x, s7_double y) {return(atan2(x, y));}


/* -------------------------------- sinh -------------------------------- */
static s7_pointer sinh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                    /* (sinh 0) -> 0 */
    case T_RATIO:
    case T_REAL:
      {
	s7_double y = s7_real(x);
#if With_Gmp
	if (fabs(y) > Sinh_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
	    mpfr_sinh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, sinh(y)));
      }
    case T_COMPLEX:
#if With_Gmp
      if ((fabs(real_part(x)) > Sinh_Limit) || (fabs(imag_part(x)) > Sinh_Limit))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sinh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if Have_Complex_Numbers
      return(c_complex_to_s7(sc, csinh(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->sinh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_sinh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_sinh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_sinh(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_sinh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->sinh_symbol, a_number_string));
    }
}

static s7_pointer g_sinh(s7_scheme *sc, s7_pointer args)
{
  #define H_sinh "(sinh z) returns sinh(z)"
  #define Q_sinh sc->pl_nn
  return(sinh_p_p(sc, car(args)));
}

static s7_double sinh_d_d(s7_double x) {return(sinh(x));}
static s7_pointer sinh_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, sinh(x)));}
  /* so sinh in a do-loop with 0 arg may return 0.0 because sinh_p_d does not check if x=0 */


/* -------------------------------- cosh -------------------------------- */
static s7_pointer cosh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_one);                   /* (cosh 0) -> 1 */
    case T_RATIO:
    case T_REAL:
      {
	s7_double y = s7_real(x);
#if With_Gmp
	if (fabs(y) > Sinh_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
	    mpfr_cosh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
	  }
#endif
	return(make_real(sc, cosh(y)));
      }
    case T_COMPLEX:
#if With_Gmp
      if ((fabs(real_part(x)) > Sinh_Limit) || (fabs(imag_part(x)) > Sinh_Limit))
	{
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_cosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
#endif
#if Have_Complex_Numbers
      return(c_complex_to_s7(sc, ccosh(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->cosh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_cosh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_cosh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_cosh(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_cosh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->cosh_symbol, a_number_string));
    }
}

static s7_pointer g_cosh(s7_scheme *sc, s7_pointer args)
{
  #define H_cosh "(cosh z) returns cosh(z)"
  #define Q_cosh sc->pl_nn
  return(cosh_p_p(sc, car(args)));
}

static s7_double cosh_d_d(s7_double x) {return(cosh(x));}
static s7_pointer cosh_p_d(s7_scheme *sc, s7_double x) {return(make_real(sc, cosh(x)));}


/* -------------------------------- tanh -------------------------------- */
#define Tanh_Limit 350.0
static s7_pointer tanh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: return((integer(x) == 0) ? int_zero : make_real(sc, tanh((s7_double)integer(x))));
    case T_RATIO:   return(make_real(sc, tanh((s7_double)fraction(x))));
    case T_REAL:    return(make_real(sc, tanh(real(x))));
    case T_COMPLEX:
#if Have_Complex_Numbers
      if (real_part(x) > Tanh_Limit)
	return(real_one);                         /* closer than 0.0 which is what ctanh is about to return! */
      if (real_part(x) < -Tanh_Limit)
	return(make_real(sc, -1.0));              /* closer than ctanh's -0.0 */
      return(c_complex_to_s7(sc, ctanh(to_c_complex(x))));
#else
      out_of_range_error_nr(sc, sc->tanh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      goto Big_Real_Tanh;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      goto Big_Real_Tanh;
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x))) return(x);
      mpfr_set(sc->mpfr_1, big_real(x), Mpfr_Rndn);
    Big_Real_Tanh:
      if (mpfr_cmp_d(sc->mpfr_1, Tanh_Limit) > 0) return(real_one);
      if (mpfr_cmp_d(sc->mpfr_1, -Tanh_Limit) < 0) return(make_real(sc, -1.0));
      mpfr_tanh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      if ((MPC_INEX_RE(mpc_cmp_si_si(big_complex(x), Tanh_Limit, 1))) > 0)
	return(real_one);
      if ((MPC_INEX_RE(mpc_cmp_si_si(big_complex(x), -Tanh_Limit, 1))) < 0)
	return(make_real(sc, -1.0));
      if ((mpfr_nan_p(mpc_imagref(big_complex(x)))) ||
	  (mpfr_inf_p(mpc_imagref(big_complex(x)))))
	{
	  if (mpfr_cmp_ui(mpc_realref(big_complex(x)), 0) == 0)
	    return(make_complex_not_0i(sc, 0.0, Nan)); /* match non-bignum choice */
	  return(complex_NaN);
	}
      mpc_tanh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      if (mpfr_zero_p(mpc_imagref(sc->mpc_1)))
	return(mpfr_to_big_real(sc, mpc_realref(sc->mpc_1)));
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->tanh_symbol, a_number_string));
    }
}

static s7_pointer g_tanh(s7_scheme *sc, s7_pointer args)
{
  #define H_tanh "(tanh z) returns tanh(z)"
  #define Q_tanh sc->pl_nn
  return(tanh_p_p(sc, car(args)));
}

static s7_double tanh_d_d(s7_double x) {return(tanh(x));}


/* -------------------------------- asinh -------------------------------- */
static s7_pointer asinh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: return((integer(x) == 0) ? int_zero : make_real(sc, asinh((s7_double)integer(x))));
    case T_RATIO:   return(make_real(sc, asinh((s7_double)fraction(x))));
    case T_REAL:    return(make_real(sc, asinh(real(x))));
    case T_COMPLEX:
#if Have_Complex_Numbers
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(c_complex_to_s7(sc, casinh_1(to_c_complex(x))));
  #else
      return(c_complex_to_s7(sc, casinh(to_c_complex(x))));
  #endif
#else
      out_of_range_error_nr(sc, sc->asinh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      mpfr_asinh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      mpfr_asinh(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_asinh(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_asinh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->asinh_symbol, a_number_string));
    }
}

static s7_pointer g_asinh(s7_scheme *sc, s7_pointer args)
{
  #define H_asinh "(asinh z) returns asinh(z)"
  #define Q_asinh sc->pl_nn
  return(asinh_p_p(sc, car(args)));
}


/* -------------------------------- acosh -------------------------------- */
static s7_pointer acosh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 1) return(int_zero);
    case T_REAL:
    case T_RATIO:
      {
	s7_double x1 = s7_real(x);
	if (x1 >= 1.0)
	  return(make_real(sc, acosh(x1)));
      }
    case T_COMPLEX:
#if Have_Complex_Numbers
  #ifdef __OpenBSD__
      return(c_complex_to_s7(sc, cacosh_1(s7_to_c_complex(x))));
  #else
      return(c_complex_to_s7(sc, cacosh(s7_to_c_complex(x)))); /* not to_c_complex because x might not be complex */
  #endif
#else
      /* since we can fall through to this branch, we need a better error message than "must be a number, not 0.0" */
      out_of_range_error_nr(sc, sc->acosh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
      mpc_acosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_RATIO:
      mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
      mpc_acosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_REAL:
      mpc_set_fr(sc->mpc_1, big_real(x), MPC_RNDNN);
      mpc_acosh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_COMPLEX:
      mpc_acosh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->acosh_symbol, a_number_string));
    }
}

static s7_pointer g_acosh(s7_scheme *sc, s7_pointer args)
{
  #define H_acosh "(acosh z) returns acosh(z)"
  #define Q_acosh sc->pl_nn
  return(acosh_p_p(sc, car(args)));
}


/* -------------------------------- atanh -------------------------------- */
static s7_pointer atanh_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(int_zero);                    /* (atanh 0) -> 0 */
    case T_REAL:
    case T_RATIO:
      {
	s7_double x1 = s7_real(x);
	if (fabs(x1) < 1.0)
	  return(make_real(sc, atanh(x1)));
      }
      /* if we can't distinguish x from 1.0 even with long_doubles, we'll get inf.0:
       *    (atanh 9223372036854775/9223372036854776) -> 18.714973875119
       *    (atanh 92233720368547758/92233720368547757) -> inf.0
       *    (atanh (bignum 92233720368547758/92233720368547757)) -> 1.987812468492420421418925013176932317086E1+1.570796326794896619231321691639751442098E0i
       *    but the imaginary part is unnecessary
       */
    case T_COMPLEX:
#if Have_Complex_Numbers
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(c_complex_to_s7(sc, catanh_1(s7_to_c_complex(x))));
  #else
      return(c_complex_to_s7(sc, catanh(s7_to_c_complex(x))));
  #endif
#else
      out_of_range_error_nr(sc, sc->atanh_symbol, int_one, x, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_z(sc->mpfr_2, big_integer(x), Mpfr_Rndn);
      goto Atanh_Big_Real;
    case T_BIG_RATIO:
      mpfr_set_q(sc->mpfr_2, big_ratio(x), Mpfr_Rndn);
      goto Atanh_Big_Real;
    case T_BIG_REAL:
      mpfr_set(sc->mpfr_2, big_real(x), Mpfr_Rndn);
    Atanh_Big_Real:
      mpfr_set_ui(sc->mpfr_1, 1, Mpfr_Rndn);
      if (mpfr_cmpabs(sc->mpfr_2, sc->mpfr_1) < 0)
	{
	  mpfr_atanh(sc->mpfr_2, sc->mpfr_2, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_2));
	}
      mpc_set_fr(sc->mpc_1, sc->mpfr_2, MPC_RNDNN);
      mpc_atanh(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
    case T_BIG_COMPLEX:
      mpc_atanh(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->atanh_symbol, a_number_string));
    }
}

static s7_pointer g_atanh(s7_scheme *sc, s7_pointer args)
{
  #define H_atanh "(atanh z) returns atanh(z)"
  #define Q_atanh sc->pl_nn
  return(atanh_p_p(sc, car(args)));
}


/* -------------------------------- sqrt -------------------------------- */
static s7_pointer sqrt_p_p(s7_scheme *sc, s7_pointer num)
{
  switch (type(num))
    {
    case T_INTEGER:
      {
	s7_double sqx;
	if (integer(num) >= 0)
	  {
	    s7_int ix;
#if With_Gmp
	    mpz_set_si(sc->mpz_1, integer(num));
	    mpz_sqrtrem(sc->mpz_1, sc->mpz_2, sc->mpz_1);
	    if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	      return(make_integer(sc, mpz_get_si(sc->mpz_1)));
	    mpfr_set_si(sc->mpfr_1, integer(num), Mpfr_Rndn);
	    mpfr_sqrt(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	    return(mpfr_to_big_real(sc, sc->mpfr_1));
#endif
	    sqx = sqrt((s7_double)integer(num));
	    ix = (s7_int)sqx;
	    return(((ix * ix) == integer(num)) ? make_integer(sc, ix) : make_real(sc, sqx));
	    /* Mark Weaver notes that (zero? (- (sqrt 9007199136250226) 94906265.0)) -> #t
	     * but (* 94906265 94906265) -> 9007199136250225 -- oops
	     * if we use bigfloats, we're ok:
	     *    (* (sqrt 9007199136250226.0) (sqrt 9007199136250226.0)) -> 9.007199136250226000000000000000000000026E15
	     * at least we return a real here, not an incorrect integer and (sqrt 9007199136250225) -> 94906265
	     */
	  }
#if Have_Complex_Numbers
#if With_Gmp
	mpc_set_si(sc->mpc_1, integer(num), MPC_RNDNN);
	mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	return(mpc_to_number(sc, sc->mpc_1));
#endif
	sqx = (s7_double)integer(num); /* we're trying to protect against (sqrt -9223372036854775808) where we can't negate the integer argument */
	return(make_complex_not_0i(sc, 0.0, sqrt((s7_double)(-sqx))));
#else
	out_of_range_error_nr(sc, sc->sqrt_symbol, int_one, num, no_complex_numbers_string);
#endif
      }

    case T_RATIO:
      if (numerator(num) > 0) /* else it's complex, so it can't be a ratio */
	{
	  s7_int nm = (s7_int)sqrt(numerator(num));
	  if (nm * nm == numerator(num))
	    {
	      s7_int dn = (s7_int)sqrt(denominator(num));
	      if (dn * dn == denominator(num))
		return(make_ratio(sc, nm, dn));
	    }
	  return(make_real(sc, sqrt((s7_double)fraction(num))));
	}
#if Have_Complex_Numbers
      return(make_complex(sc, 0.0, sqrt((s7_double)(-fraction(num)))));
#else
      out_of_range_error_nr(sc, sc->sqrt_symbol, int_one, num, no_complex_numbers_string);
#endif

    case T_REAL:
      if (is_NaN(real(num))) return(num);     /* needed because otherwise (sqrt +nan.0) -> 0.0-nan.0i ?? */
      if (real(num) >= 0.0)
	return(make_real(sc, sqrt(real(num))));
      return(make_complex_not_0i(sc, 0.0, sqrt(-real(num))));

    case T_COMPLEX:    /* (* inf.0 (sqrt -1)) -> -nan+infi, but (sqrt -inf.0) -> 0+infi */
#if Have_Complex_Numbers
      return(c_complex_to_s7(sc, csqrt(to_c_complex(num)))); /* sqrt(+inf.0+1.0i) -> +inf.0 */
#else
      out_of_range_error_nr(sc, sc->sqrt_symbol, int_one, num, no_complex_numbers_string);
#endif

#if With_Gmp
    case T_BIG_INTEGER:
      if (mpz_cmp_ui(big_integer(num), 0) >= 0)
	{
	  mpz_sqrtrem(sc->mpz_1, sc->mpz_2, big_integer(num));
	  if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	    return(mpz_to_integer(sc, sc->mpz_1));
	  mpfr_set_z(sc->mpfr_1, big_integer(num), Mpfr_Rndn);
	  mpfr_sqrt(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      mpc_set_z(sc->mpc_1, big_integer(num), MPC_RNDNN);
      mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));

    case T_BIG_RATIO: /* if big ratio, check both num and den for squares */
      if (mpq_cmp_ui(big_ratio(num), 0, 1) < 0)
	{
	  mpc_set_q(sc->mpc_1, big_ratio(num), MPC_RNDNN);
	  mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
      mpz_sqrtrem(sc->mpz_1, sc->mpz_2, mpq_numref(big_ratio(num)));
      if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	{
	  mpz_sqrtrem(sc->mpz_3, sc->mpz_2, mpq_denref(big_ratio(num)));
	  if (mpz_cmp_ui(sc->mpz_2, 0) == 0)
	    {
	      mpq_set_num(sc->mpq_1, sc->mpz_1);
	      mpq_set_den(sc->mpq_1, sc->mpz_3);
	      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	    }}
      mpfr_set_q(sc->mpfr_1, big_ratio(num), Mpfr_Rndn);
      mpfr_sqrt(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));

    case T_BIG_REAL:
      if (mpfr_cmp_ui(big_real(num), 0) < 0)
	{
	  mpc_set_fr(sc->mpc_1, big_real(num), MPC_RNDNN);
	  mpc_sqrt(sc->mpc_1, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	}
      mpfr_sqrt(sc->mpfr_1, big_real(num), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));

    case T_BIG_COMPLEX:
      mpc_sqrt(sc->mpc_1, big_complex(num), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, num, sc->sqrt_symbol, a_number_string));
    }
}

static s7_pointer g_sqrt(s7_scheme *sc, s7_pointer args)
{
  #define H_sqrt "(sqrt z) returns the square root of z"
  #define Q_sqrt sc->pl_nn
  return(sqrt_p_p(sc, car(args)));
}


/* -------------------------------- expt -------------------------------- */
static s7_int int_to_int(s7_int x, s7_int n)
{
  /* from GSL */
  s7_int value = 1;
  do {
    if (n & 1) value *= x;
    n >>= 1;
#if Have_Overflow_Checks
    if (multiply_overflow(x, x, &x))
      break;
#else
    x *= x;
#endif
  } while (n);
  return(value);
}

static const s7_int nth_roots[63] = {
  s7_Int64_Max, s7_Int64_Max, 3037000499LL, 2097151, 55108, 6208, 1448, 511, 234, 127, 78, 52, 38, 28, 22,
  18, 15, 13, 11, 9, 8, 7, 7, 6, 6, 5, 5, 5, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2};

static bool int_pow_ok(s7_int x, s7_int y) {return((y < s7_Int_Bits) && (nth_roots[y] >= s7_int_abs(x)));}

#if With_Gmp
static s7_pointer real_part_p_p(s7_scheme *sc, s7_pointer x);
static bool lt_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2);

static s7_pointer big_expt(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer x = car(args), y = cadr(args);
  s7_pointer result;
  if (!is_number(x))
    return(method_or_bust(sc, x, sc->expt_symbol, args, a_number_string, 1));
  if (!is_number(y))
    return(method_or_bust(sc, y, sc->expt_symbol, args, a_number_string, 2));

  if (is_zero(x))
    {
      if ((s7_is_integer(x)) && (s7_is_integer(y)) && (is_zero(y)))
	return(int_one);

      if (is_real(y))
	{
	  if (is_negative(sc, y))
	    division_by_zero_error_2_nr(sc, sc->expt_symbol, x, y);
	}
      else
	if (s7_real_part(y) < 0.0)
	  division_by_zero_error_2_nr(sc, sc->expt_symbol, x, y);

      if ((is_rational(x)) && (is_rational(y)))
	return(int_zero);
      return(real_zero);
    }

  if (s7_is_integer(y))
    {
      s7_int yval = s7_integer_clamped_if_gmp(sc, y);
      if (yval == 0)
	return((is_rational(x)) ? int_one : real_one);

      if (yval == 1)
	return(x);

      if ((!is_big_number(x)) &&
	  ((is_one(x)) || (is_zero(x))))
	return(x);

      if ((yval < s7_Int32_Max) &&
	  (yval > s7_Int32_Min))
	{
	  /* (protect against gmp exception if for example (expt 1/9223372036854775807 -9223372036854775807) */
	  if (s7_is_integer(x))
	    {
	      if (is_t_big_integer(x))
		mpz_set(sc->mpz_2, big_integer(x));
	      else mpz_set_si(sc->mpz_2, integer(x));
	      if (yval >= 0)
		{
		  mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)yval);
		  return(mpz_to_integer(sc, sc->mpz_2));
		}
	      mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)(-yval));
	      mpq_set_z(sc->mpq_1, sc->mpz_2);
	      mpq_inv(sc->mpq_1, sc->mpq_1);
	      if (mpz_cmp_ui(mpq_denref(sc->mpq_1), 1) == 0)
		return(mpz_to_integer(sc, mpq_numref(sc->mpq_1)));
	      return(mpq_to_big_ratio(sc, sc->mpq_1));
	    }

	  if (s7_is_ratio(x)) /* here y is an integer */
	    {
	      if (is_t_big_ratio(x))
		{
		  mpz_set(sc->mpz_1, mpq_numref(big_ratio(x)));
		  mpz_set(sc->mpz_2, mpq_denref(big_ratio(x)));
		}
	      else
		{
		  mpz_set_si(sc->mpz_1, numerator(x));
		  mpz_set_si(sc->mpz_2, denominator(x));
		}
	      if (yval >= 0)
		{
		  mpz_pow_ui(sc->mpz_1, sc->mpz_1, (uint32_t)yval);
		  mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)yval);
		  mpq_set_num(sc->mpq_1, sc->mpz_1);
		  mpq_set_den(sc->mpq_1, sc->mpz_2);
		}
	      else
		{
		  yval = -yval;
		  mpz_pow_ui(sc->mpz_1, sc->mpz_1, (uint32_t)yval);
		  mpz_pow_ui(sc->mpz_2, sc->mpz_2, (uint32_t)yval);
		  mpq_set_num(sc->mpq_1, sc->mpz_2);
		  mpq_set_den(sc->mpq_1, sc->mpz_1);
		  mpq_canonicalize(sc->mpq_1);
		}
	      if (mpz_cmp_ui(mpq_denref(sc->mpq_1), 1) == 0)
		return(mpz_to_integer(sc, mpq_numref(sc->mpq_1)));
	      return(mpq_to_big_ratio(sc, sc->mpq_1));
	    }

	  if (is_real(x))
	    {
	      if (is_t_big_real(x))
		mpfr_set(sc->mpfr_1, big_real(x), Mpfr_Rndn);
	      else mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	      mpfr_pow_si(sc->mpfr_1, sc->mpfr_1, yval, Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }}}

  if ((is_t_ratio(y)) &&              /* not s7_is_ratio which accepts bignums */
      (numerator(y) == 1))
    {
      if (denominator(y) == 2)
	return(sqrt_p_p(sc, x));

      if ((is_real(x)) &&
	  (denominator(y) == 3))
	{
	  any_real_to_mpfr(sc, x, sc->mpfr_1);
	  mpfr_cbrt(sc->mpfr_1, sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}}

  result = any_number_to_mpc(sc, y, sc->mpc_2);
  if (result == real_infinity)
    {
      if (is_one(x)) return(int_one);
      if (!is_real(x)) return((is_negative(sc, y)) ? real_zero : complex_NaN);
      if (is_zero(x))
	{
	  if (is_negative(sc, y)) division_by_zero_error_2_nr(sc, sc->expt_symbol, x, y);
	  return(real_zero);
	}
      if (lt_b_pi(sc, x, 0))
	{
	  if (lt_b_pi(sc, x, -1))
	    return((is_positive(sc, y)) ? real_infinity : real_zero);
	  return((is_positive(sc, y)) ? real_zero : real_infinity);
	}
      if (lt_b_pi(sc, x, 1))
	return((is_positive(sc, y)) ? real_zero : real_infinity);
      return((is_positive(sc, y)) ? real_infinity : real_zero);
    }
  if (result) return(complex_NaN);

  if ((is_real(x)) &&
      (is_real(y)) &&
      (is_positive(sc, x)))
    {
      result = any_real_to_mpfr(sc, x, sc->mpfr_1);
      if (result)
	{
	  if (result == real_infinity)
	    {
	      if (is_negative(sc, y)) return(real_zero);
	      return((is_zero(y)) ? real_one : real_infinity);
	    }
	  return(complex_NaN);
	}
      mpfr_pow(sc->mpfr_1, sc->mpfr_1, mpc_realref(sc->mpc_2), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    }

  result = any_number_to_mpc(sc, x, sc->mpc_1);
  if (result)
    {
      if ((result == real_infinity) && (is_real(y)))
	{
	  if (is_negative(sc, y)) return(real_zero);
	  return((is_zero(y)) ? real_one : real_infinity);
	}
      return(complex_NaN);
    }
  if (mpc_cmp_si_si(sc->mpc_1, 0, 0) == 0)
    return(int_zero);
  if (mpc_cmp_si_si(sc->mpc_1, 1, 0) == 0)
    return(int_one);

  mpc_pow(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);

  if ((!mpfr_nan_p(mpc_imagref(sc->mpc_1))) && (mpfr_cmp_ui(mpc_imagref(sc->mpc_1), 0) == 0)) /* (expt -inf.0 1/3) -> +inf.0+nan.0i in mpc */
    {
      if ((is_rational(car(args))) &&
	  (is_rational(cadr(args))) &&
	  (mpfr_integer_p(mpc_realref(sc->mpc_1)) != 0))
	{
	  /* mpfr_integer_p can be confused: (expt 2718/1000 (bignum 617/5)) returns an int32_t if precision=128, float if 512 */
	  /*   so first make sure we're within (say) 31 bits */
	  mpfr_set_ui(sc->mpfr_1, s7_Int32_Max, Mpfr_Rndn);
	  if (mpfr_cmpabs(mpc_realref(sc->mpc_1), sc->mpfr_1) < 0)
	    {
	      mpfr_get_z(sc->mpz_1, mpc_realref(sc->mpc_1), Mpfr_Rndn);
	      return(mpz_to_integer(sc, sc->mpz_1));
	    }}
      mpfr_set(sc->mpfr_1, mpc_realref(sc->mpc_1), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    }
  return(mpc_to_number(sc, sc->mpc_1));
}
#endif

static s7_pointer expt_p_pp(s7_scheme *sc, s7_pointer n, s7_pointer pw)
{
  if (!is_number(n))
    return(method_or_bust_pp(sc, n, sc->expt_symbol, n, pw, a_number_string, 1));
  if (!is_number(pw))
    return(method_or_bust_pp(sc, pw, sc->expt_symbol, n, pw, a_number_string, 2));

  if (is_zero(n))
    {
      if (is_zero(pw))
	{
	  if ((s7_is_integer(n)) && (s7_is_integer(pw)))       /* (expt 0 0) -> 1 */
	    return(int_one);
	  return(real_zero);                                   /* (expt 0.0 0) -> 0.0 */
	}
      if (is_real(pw))
	{
	  if (is_negative(sc, pw))                              /* (expt 0 -1) */
	    division_by_zero_error_2_nr(sc, sc->expt_symbol, n, pw);
	  /* (Clisp gives divide-by-zero error here, Guile returns inf.0) */

	  if (is_NaN(s7_real(pw)))                             /* (expt 0 +nan.0) */
	    return(pw);
	}
      else
	{                                                      /* (expt 0 a+bi) */
	  if (real_part(pw) < 0.0)                             /* (expt 0 -1+i) */
	    division_by_zero_error_2_nr(sc, sc->expt_symbol, n, pw);
	  if ((is_NaN(real_part(pw))) ||                       /* (expt 0 0+1/0i) */
	      (is_NaN(imag_part(pw))))
	    return(pw);
	}
      if ((s7_is_integer(n)) && (s7_is_integer(pw)))           /* pw != 0, (expt 0 2312) */
	return(int_zero);
      return(real_zero);                                       /* (expt 0.0 123123) */
    }

  if (is_one(pw))
    {
      if (s7_is_integer(pw))                                   /* (expt x 1) */
	return(n);
      if (is_rational(n))                                      /* (expt ratio 1.0) */
	return(make_real(sc, rational_to_double(sc, n)));
      return(n);
    }
  if (is_t_integer(pw))
    {
      const s7_int y = integer(pw);
      if (y == 0)
	{
	  if (is_rational(n))                                 /* (expt 3 0) */
	    return(int_one);
	  if ((is_NaN(s7_real_part(n))) ||                    /* (expt 1/0 0) -> NaN */
	      (is_NaN(s7_imag_part(n))))                      /* (expt (complex 0 1/0) 0) -> NaN */
	    return(n);
	  return(real_one);                                   /* (expt 3.0 0) */
	}
      switch (type(n))
	{
	case T_INTEGER:
	  {
	    const s7_int x = integer(n);
	    if (x == 1)                                       /* (expt 1 y) */
	      return(n);

	    if (x == -1)
	      {
		if (y == s7_Int64_Min)                        /* (expt -1 most-negative-fixnum) */
		  return(int_one);
		if (y & 1)                                    /* (expt -1 odd-int) */
		  return(n);                                  /*    n == -1 */
		return(int_one);                              /* (expt -1 even-int) */
	      }

	    if (y == s7_Int64_Min)                            /* (expt x most-negative-fixnum) */
	      return(int_zero);
	    if (x == s7_Int64_Min)                            /* (expt most-negative-fixnum y) */
	      return(make_real(sc, pow((double)x, (double)y)));

	    if (int_pow_ok(x, s7_int_abs(y)))
	      {
		if (y > 0)
		  return(make_integer(sc, int_to_int(x, y)));
		return(make_ratio(sc, 1, int_to_int(x, -y)));
	      }}
	  break;

	case T_RATIO:
	  {
	    const s7_int nm = numerator(n), dn = denominator(n);
	    if (y == s7_Int64_Min)
	      {
		if (s7_int_abs(nm) > dn)
		  return(int_zero);                  /* (expt 4/3 most-negative-fixnum) -> 0? */
		return(real_infinity);               /* (expt 3/4 most-negative-fixnum) -> inf? */
	      }
	    if ((int_pow_ok(nm, s7_int_abs(y))) &&
		(int_pow_ok(dn, s7_int_abs(y))))
	      {
		if (y > 0)
		  return(make_ratio(sc, int_to_int(nm, y), int_to_int(dn, y)));
		return(make_ratio_with_div_check(sc, sc->expt_symbol, int_to_int(dn, -y), int_to_int(nm, -y)));
	      }}
	  break;
	  /* occasionally int^rat can be int32_t but it happens so infrequently it's probably not worth checking
	   *  one possibly easy case: (expt 1 1/2) -> 1 (-1?) etc
	   */

	case T_REAL:
	  /* (expt -1.0 most-positive-fixnum) should be -1.0
	   * (expt -1.0 (+ (expt 2 53) 1)) -> -1.0
	   * (expt -1.0 (- 1 (expt 2 54))) -> -1.0
	   */
	  if (real(n) == -1.0)
	    {
	      if (y == s7_Int64_Min)
		return(real_one);
	      return((s7_int_abs(y) & 1) ? n : real_one);
	    }
	  break;

	case T_COMPLEX:
#if Have_Complex_Numbers
	  if ((s7_real_part(n) == 0.0) &&
	      ((s7_imag_part(n) == 1.0) ||
	       (s7_imag_part(n) == -1.0)))
	    {
	      bool yp = (y > 0), np = (s7_imag_part(n) > 0.0);
	      switch (s7_int_abs(y) % 4)
		{
		case 0: return(real_one);
		case 1: return(make_complex_not_0i(sc, 0.0, (yp == np) ? 1.0 : -1.0));
		case 2: return(make_real(sc, -1.0));
		case 3: return(make_complex_not_0i(sc, 0.0, (yp == np) ? -1.0 : 1.0));
		}}
#else
	  out_of_range_error_nr(sc, sc->expt_symbol, int_two, n, no_complex_numbers_string);
#endif
	  break;
	}}

  if ((is_real(n)) &&
      (is_real(pw)))
    {
      s7_double x, y;
      if ((is_t_ratio(pw)) && (numerator(pw) == 1))
	{
	  if (denominator(pw) == 2)
	    return(sqrt_p_p(sc, n));
	  if (denominator(pw) == 3)
	    return(make_real(sc, cbrt(s7_real(n)))); /* (expt 27 1/3) should be 3, not 3.0... */
	  /* but: (expt 512/729 1/3) -> 0.88888888888889, and 4 -> sqrt(sqrt...) etc? */
	}

      x = s7_real(n);
      y = s7_real(pw);
      if (is_NaN(x)) return(n);
      if (is_NaN(y)) return(pw);
      if (y == 0.0) return(real_one);
      /* I think pow(rl, inf) is ok */
      if (x > 0.0)
	return(make_real(sc, pow(x, y)));      /* tricky cases abound here: (expt -1 1/9223372036854775807) */
    }

  /* (expt 0+i 1e+16) = 0.98156860153485-0.19111012657867i ?
   * (expt 0+i 1+1/0i) = 0.0 ??
   */
  return(c_complex_to_s7(sc, cpow(s7_to_c_complex(n), s7_to_c_complex(pw))));
}

static s7_pointer g_expt(s7_scheme *sc, s7_pointer args)
{
  #define H_expt "(expt z1 z2) returns z1^z2"
  #define Q_expt sc->pcl_n
#if With_Gmp
  return(big_expt(sc, args));
  /* big_expt sometimes chooses a different value: g_expt (expt -1 1/3) is -1, but big_expt (expt -1 (bignum 1/3)) is (complex 1/2 (/ (sqrt 3) 2)) */
#endif
  return(expt_p_pp(sc, car(args), cadr(args)));
}


/* -------------------------------- lcm -------------------------------- */
#if With_Gmp
static s7_pointer big_lcm(s7_scheme *sc, s7_int num, s7_int den, s7_pointer args)
{
  mpz_set_si(sc->mpz_3, num);
  mpz_set_si(sc->mpz_4, den);

  for (s7_pointer x = args; is_pair(x); x = cdr(x))
    {
      const s7_pointer rat = car(x);
      switch (type(rat))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(rat));
	  mpz_lcm(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  mpz_set_si(sc->mpz_4, 1);
	  break;
	case T_RATIO:
	  mpz_set_si(sc->mpz_1, numerator(rat));
	  mpz_set_si(sc->mpz_2, denominator(rat));
	  mpz_lcm(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  mpz_gcd(sc->mpz_4, sc->mpz_4, sc->mpz_2);
	  break;
	case T_BIG_INTEGER:
	  mpz_lcm(sc->mpz_3, sc->mpz_3, big_integer(rat));
	  mpz_set_si(sc->mpz_4, 1);
	  break;
	case T_BIG_RATIO:
	  mpz_lcm(sc->mpz_3, sc->mpz_3, mpq_numref(big_ratio(rat)));
	  mpz_gcd(sc->mpz_4, sc->mpz_4, mpq_denref(big_ratio(rat)));
	  break;
	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_error_nr(sc, sc->lcm_symbol, position_of(x, args), rat, a_rational_string);
	default:
	  return(method_or_bust(sc, rat, sc->lcm_symbol,
				set_ulist_1(sc, mpz_to_rational(sc, sc->mpz_3, sc->mpz_4), x),
				a_rational_string, position_of(x, args)));
	}}
  return(mpz_to_rational(sc, sc->mpz_3, sc->mpz_4));
}
#endif

static s7_pointer g_lcm(s7_scheme *sc, s7_pointer args)
{
  /* (/ (* m n) (gcd m n)), (lcm a b c) -> (lcm a (lcm b c)) */
  #define H_lcm "(lcm ...) returns the least common multiple of its rational arguments"
  #define Q_lcm sc->pcl_f

  s7_int n = 1, d = 0;
  if (!is_pair(args))
    return(int_one);

  if (!is_pair(cdr(args)))
    {
      if (!is_rational(car(args)))
	return(method_or_bust(sc, car(args), sc->lcm_symbol, args, a_rational_string, 1));
      return(g_abs(sc, args));
    }

  for (s7_pointer nums = args; is_pair(nums); nums = cdr(nums))
    {
      const s7_pointer x = car(nums);
      s7_int b;
#if Have_Overflow_Checks
      s7_int n1;
#endif
      switch (type(x))
	{
	case T_INTEGER:
	  d = 1;
	  if (integer(x) == 0) /* return 0 unless there's a wrong-type-arg (geez what a mess) */
	    {
	      for (nums = cdr(nums); is_pair(nums); nums = cdr(nums))
		{
		  const s7_pointer x1 = car(nums);
		  if (is_number(x1))
		    {
		      if (!is_rational(x1))
			wrong_type_error_nr(sc, sc->lcm_symbol, position_of(nums, args), x1, a_rational_string);
		    }
		  else
		    if (has_active_methods(sc, x1))
		      {
			s7_pointer func = find_method_with_let(sc, x1, sc->is_rational_symbol);
			if ((func == sc->undefined) ||
			    (is_false(sc, s7_apply_function(sc, func, set_plist_1(sc, x1)))))
			  wrong_type_error_nr(sc, sc->lcm_symbol, position_of(nums, args), x1, a_rational_string);
		      }
		    else wrong_type_error_nr(sc, sc->lcm_symbol, position_of(nums, args), x1, a_rational_string);
		}
	      return(int_zero);
	    }
	  b = integer(x);
	  if (b < 0)
	    {
	      if (b == s7_Int64_Min)
#if With_Gmp
		return(big_lcm(sc, n, d, nums));
#else
		sole_arg_out_of_range_error_nr(sc, sc->lcm_symbol, args, it_is_too_large_string);
#endif
	      b = -b;
	    }
#if Have_Overflow_Checks
	  if (multiply_overflow(n / c_gcd(n, b), b, &n1))
#if With_Gmp
	    return(big_lcm(sc, n, d, nums));
#else
	    sole_arg_out_of_range_error_nr(sc, sc->lcm_symbol, args, result_is_too_large_string);
#endif
	  n = n1;
#else
	  n = (n / c_gcd(n, b)) * b;
#endif
	  break;

	case T_RATIO:
	  b = numerator(x);
	  if (b < 0)
	    {
	      if (b == s7_Int64_Min)
#if With_Gmp
		return(big_lcm(sc, n, d, nums));
#else
		sole_arg_out_of_range_error_nr(sc, sc->lcm_symbol, args, it_is_too_large_string);
#endif
	      b = -b;
	    }
#if Have_Overflow_Checks
	  if (multiply_overflow(n / c_gcd(n, b), b, &n1))  /* (lcm 92233720368547758/3 3005/2) */
#if With_Gmp
	    return(big_lcm(sc, n, d, nums));
#else
	    sole_arg_out_of_range_error_nr(sc, sc->lcm_symbol, args, intermediate_too_large_string);
#endif
          n = n1;
#else
	  n = (n / c_gcd(n, b)) * b;
#endif
	  if (d == 0)
	    d = (nums == args) ? denominator(x) : 1;
	  else d = c_gcd(d, denominator(x));
	  break;

#if With_Gmp
	case T_BIG_INTEGER:
	  d = 1;
	case T_BIG_RATIO:
	  return(big_lcm(sc, n, d, nums));
#endif
	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_error_nr(sc, sc->lcm_symbol, position_of(nums, args), x, a_rational_string);

	default:
	  return(method_or_bust(sc, x, sc->lcm_symbol,
				(nums == args) ? set_ulist_1(sc, x, cdr(nums)) :
				                 set_ulist_1(sc, (d <= 1) ? make_integer(sc, n) : make_ratio_with_div_check(sc, sc->lcm_symbol, n, d), nums),
				a_rational_string, position_of(nums, args)));
	}}
  return((d <= 1) ? make_integer(sc, n) : make_simple_ratio(sc, n, d));
}


/* -------------------------------- gcd -------------------------------- */
#if With_Gmp
static s7_pointer big_gcd(s7_scheme *sc, s7_int num, s7_int den, s7_pointer args)
{
  mpz_set_si(sc->mpz_3, num);
  mpz_set_si(sc->mpz_4, den);

  for (s7_pointer x = args; is_pair(x); x = cdr(x))
    {
      const s7_pointer rat = car(x);
      switch (type(rat))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(rat));
	  mpz_gcd(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  break;
	case T_RATIO:
	  mpz_set_si(sc->mpz_1, numerator(rat));
	  mpz_set_si(sc->mpz_2, denominator(rat));
	  mpz_gcd(sc->mpz_3, sc->mpz_3, sc->mpz_1);
	  mpz_lcm(sc->mpz_4, sc->mpz_4, sc->mpz_2);
	  break;
	case T_BIG_INTEGER:
	  mpz_gcd(sc->mpz_3, sc->mpz_3, big_integer(rat));
	  break;
	case T_BIG_RATIO:
	  mpz_gcd(sc->mpz_3, sc->mpz_3, mpq_numref(big_ratio(rat)));
	  mpz_lcm(sc->mpz_4, sc->mpz_4, mpq_denref(big_ratio(rat)));
	  break;
	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_error_nr(sc, sc->gcd_symbol, position_of(x, args), rat, a_rational_string);
	default:
	  return(method_or_bust(sc, rat, sc->gcd_symbol,
				set_ulist_1(sc, mpz_to_rational(sc, sc->mpz_3, sc->mpz_4), x),
				a_rational_string, position_of(x, args)));
	}}
  return(mpz_to_rational(sc, sc->mpz_3, sc->mpz_4));
}
#endif

static s7_pointer g_gcd(s7_scheme *sc, s7_pointer args)
{
  #define H_gcd "(gcd ...) returns the greatest common divisor of its rational arguments"
  #define Q_gcd sc->pcl_f

  s7_int n = 0, d = 1;
  s7_pointer n_args;
  if (!is_pair(args))       /* (gcd) */
    return(int_zero);

  if (!is_pair(cdr(args)))  /* (gcd 3/4) */
    {
      if (!is_rational(car(args)))
	return(method_or_bust(sc, car(args), sc->gcd_symbol, args, a_rational_string, 1));
      return(abs_p_p(sc, car(args)));
    }

  if (is_t_integer(car(args)))
    {
      n = integer(car(args));
      n_args = cdr(args);
    }
  else n_args = args;

  for (s7_pointer nums = n_args; is_pair(nums); nums = cdr(nums))
    {
      const s7_pointer x = car(nums);
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == s7_Int64_Min)
#if With_Gmp
	    return(big_gcd(sc, n, d, nums));
#else
	  {
	    if ((n == s7_Int64_Min) && (is_null(cdr(nums)))) /* gcd is supposed to return a positive integer, but we can't take abs(s7_Int64_Min) */
	      sole_arg_out_of_range_error_nr(sc, sc->gcd_symbol, args, it_is_too_large_string);
	  }
#endif
	  n = c_gcd(n, integer(x));
	  break;

	case T_RATIO:
	  {
#if Have_Overflow_Checks
	    s7_int dn;
#endif
	    n = c_gcd(n, numerator(x));
	    if (d == 1)
	      d = denominator(x);
	    else
	      {
		const s7_int b = denominator(x);
#if Have_Overflow_Checks
		if (multiply_overflow(d / c_gcd(d, b), b, &dn)) /* (gcd 1/92233720368547758 1/3005) */
#if With_Gmp
		  return(big_gcd(sc, n, d, x));
#else
		  sole_arg_out_of_range_error_nr(sc, sc->gcd_symbol, args, intermediate_too_large_string);
#endif
		d = dn;
#else
		d = (d / c_gcd(d, b)) * b;
#endif
	      }}
	  break;

#if With_Gmp
	case T_BIG_INTEGER: case T_BIG_RATIO:
	  return(big_gcd(sc, n, d, nums));
#endif

	case T_REAL: case T_BIG_REAL: case T_COMPLEX: case T_BIG_COMPLEX:
	  wrong_type_error_nr(sc, sc->gcd_symbol, position_of(nums, args), x, a_rational_string);

	default:
	  return(method_or_bust(sc, x, sc->gcd_symbol,
				(nums == args) ? set_ulist_1(sc, x, cdr(nums)) :
				                 set_ulist_1(sc, (d <= 1) ? make_integer(sc, n) : make_ratio_with_div_check(sc, sc->gcd_symbol, n, d), nums),
				a_rational_string, position_of(nums, args)));
	}}
  return((d <= 1) ? make_integer(sc, n) : make_simple_ratio(sc, n, d));
}


/* -------------------------------- floor -------------------------------- */
static s7_pointer floor_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      {
	s7_int val = numerator(x) / denominator(x);
	/* C "/" truncates? -- C spec says "truncation toward 0" */
	/* we're avoiding "floor" here because the int->double conversion introduces inaccuracies for big numbers
	 *   but it's used by opt_i_d_c (via s7_number_to_real) so floor_i_7d below can return different results:
	 *   (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (display (floor 3441313796169221281/1720656898084610641)) (newline))) (func)): 1
	 *   (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (display (/ (floor 3441313796169221281/1720656898084610641))) (newline))) (func)): 1/2
	 */
	return(make_integer(sc, (numerator(x) < 0) ? (val - 1) : val)); /* not "val" because it might be truncated to 0 */
      }
    case T_REAL:
      {
	const s7_double z = real(x);
	if (is_NaN(z))
	  sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, x, it_is_nan_string);
	if (is_inf(z))
	  sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, x, it_is_infinite_string);
#if With_Gmp
	if (fabs(z) > Double_To_Int64_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, z, Mpfr_Rndn);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_1, MPFR_RNDD);
	    return(mpz_to_integer(sc, sc->mpz_1));
	  }
#else
	if (fabs(z) > Double_To_Int64_Limit)
	  sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, x, it_is_too_large_string);
#endif
	return(make_integer(sc, (s7_int)floor(z)));
	/* floor here rounds down, whereas a straight int<=real coercion apparently rounds towards 0 */
      }
#if With_Gmp
    case T_BIG_INTEGER:
      return(x);
    case T_BIG_RATIO:
      mpz_fdiv_q(sc->mpz_1, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, x, it_is_nan_string);
      if (mpfr_inf_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, x, it_is_infinite_string);
      mpfr_get_z(sc->mpz_1, big_real(x), MPFR_RNDD);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      sole_arg_wrong_type_error_nr(sc, sc->floor_symbol, x, sc->type_names[T_REAL]);
    default:
      return(method_or_bust_p(sc, x, sc->floor_symbol, sc->type_names[T_REAL]));
    }
}

static s7_pointer g_floor(s7_scheme *sc, s7_pointer args)
{
  #define H_floor "(floor x) returns the integer closest to x toward -inf"
  #define Q_floor s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(floor_p_p(sc, car(args)));
}

static s7_int floor_i_i(s7_int i) {return(i);}
static s7_pointer floor_p_i(s7_scheme *sc, s7_int x) {return(make_integer(sc, x));}

#if !With_Gmp
static s7_int floor_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, real_NaN, it_is_nan_string);
  if (fabs(x) > Double_To_Int64_Limit)
    sole_arg_out_of_range_error_nr(sc, sc->floor_symbol, wrap_real(sc, x), it_is_too_large_string);
  return((s7_int)floor(x));
}

static s7_int floor_i_7p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_integer(x)) return(integer(x));
  if (is_t_real(x)) return(floor_i_7d(sc, real(x)));
  if (is_t_ratio(x)) /* for consistency with floor_p_p, don't use floor(fraction(x)) */
    {
      s7_int val = numerator(x) / denominator(x);
      return((numerator(x) < 0) ? val - 1 : val);
    }
  return(s7_integer(method_or_bust_p(sc, x, sc->floor_symbol, sc->type_names[T_REAL])));
}

static s7_pointer floor_p_d(s7_scheme *sc, s7_double x) {return(make_integer(sc, floor_i_7d(sc, x)));}
#endif


/* -------------------------------- ceiling -------------------------------- */
static s7_pointer ceiling_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      {
	s7_int val = numerator(x) / denominator(x);
	return(make_integer(sc, (numerator(x) < 0) ? val : (val + 1)));
      }
    case T_REAL:
      {
	const s7_double z = real(x);
	if (is_NaN(z))
	  sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, x, it_is_nan_string);
	if (is_inf(z))
	  sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, x, it_is_infinite_string);
#if With_Gmp
	if (fabs(z) > Double_To_Int64_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, z, Mpfr_Rndn);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_1, MPFR_RNDU);
	    return(mpz_to_integer(sc, sc->mpz_1));
	  }
#else
	if (fabs(z) > Double_To_Int64_Limit)
	  sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, x, it_is_too_large_string);
#endif
	return(make_integer(sc, (s7_int)ceil(real(x))));
      }
#if With_Gmp
    case T_BIG_INTEGER:
      return(x);
    case T_BIG_RATIO:
      mpz_cdiv_q(sc->mpz_1, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, x, it_is_nan_string);
      if (mpfr_inf_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, x, it_is_infinite_string);
      mpfr_get_z(sc->mpz_1, big_real(x), MPFR_RNDU);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      sole_arg_wrong_type_error_nr(sc, sc->ceiling_symbol, x, sc->type_names[T_REAL]);
    default:
      return(method_or_bust_p(sc, x, sc->ceiling_symbol, sc->type_names[T_REAL]));
    }
}

static s7_pointer g_ceiling(s7_scheme *sc, s7_pointer args)
{
  #define H_ceiling "(ceiling x) returns the integer closest to x toward inf"
  #define Q_ceiling s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(ceiling_p_p(sc, car(args)));
}

static s7_int ceiling_i_i(s7_int i) {return(i);}
static s7_pointer ceiling_p_i(s7_scheme *sc, s7_int x) {return(make_integer(sc, x));}

#if !With_Gmp
static s7_int ceiling_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, real_NaN, it_is_nan_string);
  if ((is_inf(x)) ||
      (x > Double_To_Int64_Limit) || (x < -Double_To_Int64_Limit))
    sole_arg_out_of_range_error_nr(sc, sc->ceiling_symbol, wrap_real(sc, x), it_is_too_large_string);
  return((s7_int)ceil(x));
}

static s7_int ceiling_i_7p(s7_scheme *sc, s7_pointer x)
{
  if (is_t_integer(x)) return(integer(x));
  if (is_t_real(x)) return(ceiling_i_7d(sc, real(x)));
  if (is_t_ratio(x)) return((s7_int)(ceil((s7_double)fraction(x))));
  return(s7_integer(method_or_bust_p(sc, x, sc->ceiling_symbol, sc->type_names[T_REAL])));
}

static s7_pointer ceiling_p_d(s7_scheme *sc, s7_double x) {return(make_integer(sc, ceiling_i_7d(sc, x)));}
#endif


/* -------------------------------- truncate -------------------------------- */
static s7_pointer truncate_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      return(make_integer(sc, (s7_int)(numerator(x) / denominator(x)))); /* C "/" already truncates (but this divide is not accurate over e13) */
    case T_REAL:
      {
	const s7_double z = real(x);
	if (is_NaN(z))
	  sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, x, it_is_nan_string);
	if (is_inf(z))
	  sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, x, it_is_infinite_string);
#if With_Gmp
	if (fabs(z) > Double_To_Int64_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_1, MPFR_RNDZ);
	    return(mpz_to_integer(sc, sc->mpz_1));
	  }
#else
	if (fabs(z) > Double_To_Int64_Limit)
	  sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, x, it_is_too_large_string);
#endif
	return(make_integer(sc, (z > 0.0) ? (s7_int)floor(z) : (s7_int)ceil(z)));
      }
#if With_Gmp
    case T_BIG_INTEGER:
      return(x);
    case T_BIG_RATIO:
      mpz_tdiv_q(sc->mpz_1, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, x, it_is_nan_string);
      if (mpfr_inf_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, x, it_is_infinite_string);
      mpfr_get_z(sc->mpz_1, big_real(x), MPFR_RNDZ);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      sole_arg_wrong_type_error_nr(sc, sc->truncate_symbol, x, sc->type_names[T_REAL]);
    default:
      return(method_or_bust_p(sc, x, sc->truncate_symbol, sc->type_names[T_REAL]));
    }
}

static s7_pointer g_truncate(s7_scheme *sc, s7_pointer args)
{
  #define H_truncate "(truncate x) returns the integer closest to x toward 0"
  #define Q_truncate s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(truncate_p_p(sc, car(args)));
}

static s7_int truncate_i_i(s7_int i) {return(i);}
static s7_pointer truncate_p_i(s7_scheme *sc, s7_int x) {return(make_integer(sc, x));}

#if !With_Gmp
static s7_int truncate_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, real_NaN, it_is_nan_string);
  if (is_inf(x))
    sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, wrap_real(sc, x), it_is_infinite_string);
  if (fabs(x) > Double_To_Int64_Limit)
    sole_arg_out_of_range_error_nr(sc, sc->truncate_symbol, wrap_real(sc, x), it_is_too_large_string);
  return((x > 0.0) ? (s7_int)floor(x) : (s7_int)ceil(x));
}

static s7_pointer truncate_p_d(s7_scheme *sc, s7_double x) {return(make_integer(sc, truncate_i_7d(sc, x)));}
#endif


/* -------------------------------- round -------------------------------- */
static s7_double r5rs_round(s7_double x)
{
  s7_double fl = floor(x), ce = ceil(x);
  s7_double dfl = x - fl;
  s7_double dce = ce - x;
  if (dfl > dce) return(ce);
  if (dfl < dce) return(fl);
  return((fmod(fl, 2.0) == 0.0) ? fl : ce);
}

static s7_pointer round_p_p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
      return(x);
    case T_RATIO:
      {
	s7_int truncated = numerator(x) / denominator(x), remains = numerator(x) % denominator(x);
	long_double frac = s7_fabsl((long_double)remains / (long_double)denominator(x));
	if ((frac > 0.5) ||
	    ((frac == 0.5) &&
	     (truncated % 2 != 0)))
	  return(make_integer(sc, (numerator(x) < 0) ? (truncated - 1) : (truncated + 1)));
	return(make_integer(sc, truncated));
      }
    case T_REAL:
      {
	const s7_double z = real(x);
	if (is_NaN(z))
	  sole_arg_out_of_range_error_nr(sc, sc->round_symbol, x, it_is_nan_string);
	if (is_inf(z))
	  sole_arg_out_of_range_error_nr(sc, sc->round_symbol, x, it_is_infinite_string);
#if With_Gmp
	if (fabs(z) > Double_To_Int64_Limit)
	  {
	    mpfr_set_d(sc->mpfr_1, z, Mpfr_Rndn);
	    mpfr_rint(sc->mpfr_2, sc->mpfr_1, Mpfr_Rndn); /* mpfr_roundeven in mpfr 4.0.0 */
	    mpfr_get_z(sc->mpz_3, sc->mpfr_2, Mpfr_Rndn);
	    return(mpz_to_integer(sc, sc->mpz_3));
	  }
#else
	if (fabs(z) > Double_To_Int64_Limit)
	  sole_arg_out_of_range_error_nr(sc, sc->round_symbol, x, it_is_too_large_string);
#endif
	return(make_integer(sc, (s7_int)r5rs_round(z)));
      }
#if With_Gmp
      case T_BIG_INTEGER:
	return(x);
    case T_BIG_RATIO:
      {
	int32_t rnd;
	mpz_fdiv_qr(sc->mpz_1, sc->mpz_2, mpq_numref(big_ratio(x)), mpq_denref(big_ratio(x)));
	mpz_mul_ui(sc->mpz_2, sc->mpz_2, 2);
	rnd = mpz_cmpabs(sc->mpz_2, mpq_denref(big_ratio(x)));
	mpz_fdiv_q(sc->mpz_2, sc->mpz_2, mpq_denref(big_ratio(x)));
	if (rnd > 0)
	  mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	else
	  if ((rnd == 0) &&
	      (mpz_odd_p(sc->mpz_1)))
	    mpz_add_ui(sc->mpz_1, sc->mpz_1, 1);
	return(mpz_to_integer(sc, sc->mpz_1));
      }
    case T_BIG_REAL:
      if (mpfr_nan_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->round_symbol, x, it_is_nan_string);
      if (mpfr_inf_p(big_real(x)))
	sole_arg_out_of_range_error_nr(sc, sc->round_symbol, x, it_is_infinite_string);
      mpfr_set(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      mpfr_rint(sc->mpfr_2, sc->mpfr_1, Mpfr_Rndn);
      mpfr_get_z(sc->mpz_3, sc->mpfr_2, Mpfr_Rndn);
      return(mpz_to_integer(sc, sc->mpz_3));
    case T_BIG_COMPLEX:
#endif
    case T_COMPLEX:
      sole_arg_wrong_type_error_nr(sc, sc->round_symbol, x, sc->type_names[T_REAL]);
    default:
      return(method_or_bust_p(sc, x, sc->round_symbol, sc->type_names[T_REAL]));
    }
}

static s7_pointer g_round(s7_scheme *sc, s7_pointer args)
{
  #define H_round "(round x) returns the integer closest to x"
  #define Q_round s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)
  return(round_p_p(sc, car(args)));
}
/* (round (/ ...)) -> real_divide etc (wrapped) -- round_p_p is called in tbit via fx_c_op_opssqq_s_direct */

static s7_int round_i_i(s7_int i) {return(i);}
static s7_pointer round_p_i(s7_scheme *sc, s7_int x) {return(make_integer(sc, x));}

#if !With_Gmp
static s7_int round_i_7d(s7_scheme *sc, s7_double z)
{
  if (is_NaN(z))
    sole_arg_out_of_range_error_nr(sc, sc->round_symbol, real_NaN, it_is_nan_string);
  if ((is_inf(z)) ||
      (z > Double_To_Int64_Limit) || (z < -Double_To_Int64_Limit))
    sole_arg_out_of_range_error_nr(sc, sc->round_symbol, wrap_real(sc, z), it_is_too_large_string);
  return((s7_int)r5rs_round(z));
}

static s7_pointer round_p_d(s7_scheme *sc, s7_double x) {return(make_integer(sc,round_i_7d(sc, x)));}
#endif


/* ---------------------------------------- add ---------------------------------------- */
static inline s7_pointer add_if_overflow_to_real_or_big_integer(s7_scheme *sc, s7_int x, s7_int y)
{
#if Have_Overflow_Checks
  s7_int val;
  if (add_overflow(x, y, &val))
#if With_Gmp
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_set_si(sc->mpz_2, y);
      mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      return(mpz_to_big_integer(sc, sc->mpz_1));
    }
#else
    {
      if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "integer add overflow: (+ %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (long_double)x + (long_double)y));
    }
#endif
  return(make_integer(sc, val));
#else
  return(make_integer(sc, x + y));
#endif
}

static s7_pointer integer_ratio_add_if_overflow_to_real_or_rational(s7_scheme *sc, s7_pointer x, s7_pointer y) /* x: int, y:ratio */
{
#if Have_Overflow_Checks
  s7_int z;
  if ((multiply_overflow(integer(x), denominator(y), &z)) ||
      (add_overflow(z, numerator(y), &z)))
#if With_Gmp
    {
      mpz_set_si(sc->mpz_1, integer(x));
      mpz_mul_si(sc->mpz_1, sc->mpz_1, denominator(y));
      mpz_set_si(sc->mpz_2, numerator(y));
      mpz_add(mpq_numref(sc->mpq_1), sc->mpz_2, sc->mpz_1);
      mpz_set_si(mpq_denref(sc->mpq_1), denominator(y));
      return(mpq_to_rational(sc, sc->mpq_1));
    }
#else
    {
      if (With_Warnings)
	s7_warn(sc, s7_Warn_Bufsize, "integer + ratio overflow: (+ %" ld64 " %" ld64 "/%" ld64 ")\n", integer(x), numerator(y), denominator(y));
      return(make_real(sc, (long_double)integer(x) + fraction(y)));
    }
#endif
    return(make_ratio(sc, z, denominator(y)));
#else
  return(make_ratio(sc, integer(x) * denominator(y) + numerator(y), denominator(y)));
#endif
}

#define parcel_out_fractions(X, Y) do {d1 = denominator(x); n1 = numerator(x); d2 = denominator(y); n2 = numerator(y);} while (0)
/* add_out_x|y here (as in lt_out_x|y) gives a small speed-up, say 3-7 callgrind units, about 2% */

static s7_pointer add_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* an experiment: try to avoid the switch statement */
  /* this wins in most s7 cases, not so much elsewhere? parallel subtract/multiply code is slower */
  if (is_t_integer(x))
    {
      if (is_t_integer(y))
	return(add_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));
#if !With_Gmp
      if (is_t_real(y))
	return(make_real(sc, (long_double)integer(x) + real(y)));
#endif
    }
  else
    if (is_t_real(x))
      {
	if (is_t_real(y))
	  return(make_real(sc, real(x) + real(y)));
      }
    else
      if ((is_t_complex(x)) && (is_t_complex(y)))
	return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(add_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));
	case T_RATIO:
	  return(integer_ratio_add_if_overflow_to_real_or_rational(sc, x, y));
	case T_REAL:
#if With_Gmp
	  if (s7_int_abs(integer(x)) >= Int64_To_Double_Limit) /* (+ 9223372036854775807 .1), >= needed for (+ 9007199254740992 1.0) */
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), Mpfr_Rndn);
	      mpfr_add_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (long_double)integer(x) + real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, (long_double)integer(x) + (long_double)real_part(y), imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpz_add(sc->mpz_1, sc->mpz_1, big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_add(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_add_si(sc->mpfr_1, big_real(y), integer(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_si(sc->mpc_1, integer(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer_ratio_add_if_overflow_to_real_or_rational(sc, y, x));
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
	    if (d1 == d2)
	      {
#if Have_Overflow_Checks
		s7_int q;
		if (add_overflow(n1, n2, &q))
#if With_Gmp
		  {
		    mpq_set_si(sc->mpq_1, n1, d1);
		    mpq_set_si(sc->mpq_2, n2, d2);
		    mpq_add(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		    return(mpq_to_rational(sc, sc->mpq_1));
		  }
#else
		  {
 		    if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "ratio + ratio overflow: (/ (+ %" ld64 " %" ld64 ") %" ld64 ")\n", n1, n2, d1);
		    return(make_real(sc, ((long_double)n1 + (long_double)n2) / (long_double)d1));
		  }
#endif
	        return(make_ratio_with_div_check(sc, sc->add_symbol, q, d1));
#else
		return(make_ratio_with_div_check(sc, sc->add_symbol, n1 + n2, d1));
#endif
	      }

#if Have_Overflow_Checks
	    {
	      s7_int n1d2, n2d1, d1d2, q;
	      if ((multiply_overflow(d1, d2, &d1d2)) ||
		  (multiply_overflow(n1, d2, &n1d2)) ||
		  (multiply_overflow(n2, d1, &n2d1)) ||
		  (add_overflow(n1d2, n2d1, &q)))
#if With_Gmp
		{
		  mpq_set_si(sc->mpq_1, n1, d1);
		  mpq_set_si(sc->mpq_2, n2, d2);
		  mpq_add(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		  return(mpq_to_rational(sc, sc->mpq_1));
		}
#else
	        {
 		  if (With_Warnings)
		    s7_warn(sc, s7_Warn_Bufsize, "ratio + ratio overflow: (+ %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
	          return(make_real(sc, ((long_double)n1 / (long_double)d1) + ((long_double)n2 / (long_double)d2)));
		}
#endif
	      return(make_ratio_with_div_check(sc, sc->add_symbol, q, d1d2));
	    }
#else
	    return(make_ratio_with_div_check(sc, sc->add_symbol, n1 * d2 + n2 * d1, d1 * d2));
#endif
	  }
	case T_REAL:
	  return(make_real(sc, fraction(x) + real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, (s7_double)fraction(x) + real_part(y), imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_set_z(sc->mpq_2, big_integer(y));
	  mpq_add(sc->mpq_1, sc->mpq_1, sc->mpq_2);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_add(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_add_q(sc->mpfr_1, big_real(y), sc->mpq_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
#if With_Gmp
	  if (s7_int_abs(integer(y)) >= Int64_To_Double_Limit) /* (+ .1 9223372036854775807) */
	    {
	      mpfr_set_si(sc->mpfr_1, integer(y), Mpfr_Rndn);
	      mpfr_add_d(sc->mpfr_1, sc->mpfr_1, real(x), Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, real(x) + (long_double)integer(y)));
	case T_RATIO:
	  return(make_real(sc, real(x) + (s7_double)fraction(y)));
	case T_REAL:
	  return(make_real(sc, real(x) + real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, real(x) + real_part(y), imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_add_z(sc->mpfr_1, sc->mpfr_1, big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_add_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_add_d(sc->mpfr_1, big_real(y), real(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_complex_not_0i(sc, real_part(x) + integer(y), imag_part(x)));
	case T_RATIO:
	  return(make_complex_not_0i(sc, real_part(x) + (s7_double)fraction(y), imag_part(x)));
	case T_REAL:
	  return(make_complex_not_0i(sc, real_part(x) + real(y), imag_part(x)));
	case T_COMPLEX:
	  return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_add_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(y));
	  mpz_add(sc->mpz_1, big_integer(x), sc->mpz_1);
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_add(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	  mpfr_add_z(sc->mpfr_1, sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpz_add(sc->mpz_1, big_integer(x), big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  mpq_add(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_add_z(sc->mpfr_1, big_real(y), big_integer(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_add(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_add(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	  mpfr_add_q(sc->mpfr_1, sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_add(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_add(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_add_q(sc->mpfr_1, big_real(y), big_ratio(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_add(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  mpfr_add_si(sc->mpfr_1, big_real(x), integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_add_q(sc->mpfr_1, big_real(x), sc->mpq_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_add_d(sc->mpfr_1, big_real(x), real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_add_fr(sc->mpc_1, sc->mpc_1, big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpfr_add_z(sc->mpfr_1, big_real(x), big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_add_q(sc->mpfr_1, big_real(x), big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_add(sc->mpfr_1, big_real(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_add_fr(sc->mpc_1, big_complex(y), big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  mpc_set_si(sc->mpc_1, integer(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(y); */
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_add(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_add_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_add(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->add_symbol, x, y, a_number_string, 2));
	}
#endif
      default:
	return(method_or_bust_pp(sc, x, sc->add_symbol, x, y, a_number_string, 1));
    }
}

#if !With_Gmp
static inline s7_pointer add_if_overflow_to_real_wrapped(s7_scheme *sc, s7_int x, s7_int y)
{
#if Have_Overflow_Checks
  s7_int val;
  if (add_overflow(x, y, &val))
    {
      if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "integer add overflow: (+ %" ld64 " %" ld64 ")\n", x, y);
      return(wrap_real(sc, (long_double)x + (long_double)y));
    }
  return(wrap_integer(sc, val));
#else
  return(wrap_integer(sc, x + y));
#endif
}

static s7_pointer add_p_pp_wrapped(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* an experiment -- wraps rather than boxes results */
#if 1
  if (is_t_integer(x))
    {
      if (is_t_integer(y))
	return(add_if_overflow_to_real_wrapped(sc, integer(x), integer(y)));
#if !With_Gmp
      if (is_t_real(y))
	return(wrap_real(sc, (long_double)integer(x) + real(y)));
#endif
    }
  else
    if (is_t_real(x))
      {
	if (is_t_real(y))
	  return(wrap_real(sc, real(x) + real(y)));
      }
    else
      if ((is_t_complex(x)) && (is_t_complex(y)))
	return(wrap_real_or_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
#endif
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(add_if_overflow_to_real_wrapped(sc, integer(x), integer(y)));
	case T_REAL:
	  return(wrap_real(sc, (long_double)integer(x) + real(y)));
	case T_COMPLEX:
	  return(wrap_complex(sc, (long_double)integer(x) + (long_double)real_part(y), imag_part(y)));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(wrap_real(sc, real(x) + (long_double)integer(y)));
	case T_REAL:
	  return(make_real(sc, real(x) + real(y)));
	case T_COMPLEX:
	  return(wrap_complex(sc, real(x) + real_part(y), imag_part(y)));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(wrap_complex(sc, real_part(x) + integer(y), imag_part(x)));
	case T_REAL:
	  return(wrap_complex(sc, real_part(x) + real(y), imag_part(x)));
	case T_COMPLEX:
	  return(wrap_real_or_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
	}}
  return(add_p_pp(sc, x, y));
}
#else
#define add_p_pp_wrapped add_p_pp
#endif

static s7_pointer add_p_ppp(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_pointer z)
{
  if ((is_t_integer(x)) && (is_t_integer(y)) && (is_t_integer(z)))
    {
#if Have_Overflow_Checks
      s7_int val;
      if ((!add_overflow(integer(x), integer(y), &val)) &&
	  (!add_overflow(val, integer(z), &val)))
	return(make_integer(sc, val));
#if With_Gmp
      mpz_set_si(sc->mpz_1, integer(x));
      mpz_set_si(sc->mpz_2, integer(y));
      mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      mpz_set_si(sc->mpz_2, integer(z));
      mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      return(mpz_to_integer(sc, sc->mpz_1));
#else
      if (With_Warnings)
	s7_warn(sc, s7_Warn_Bufsize, "integer add overflow: (+ %" ld64 " %" ld64 " %" ld64 ")\n", integer(x), integer(y), integer(z));
      return(make_real(sc, (long_double)integer(x) + (long_double)integer(y) + (long_double)integer(z)));
#endif
#else
      return(make_integer(sc, integer(x) + integer(y) + integer(z)));
#endif
    }
  if ((is_t_real(x)) && (is_t_real(y)) && (is_t_real(z)))
    return(make_real(sc, real(x) + real(y) + real(z)));
  {
    s7_pointer num = add_p_pp_wrapped(sc, x, y);
    sc->error_argnum = 1;
    num = add_p_pp(sc, num, z);
    sc->error_argnum = 0;
    return(num);
  }
}

#if !With_Gmp
static s7_pointer add_p_ppp_wrapped(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_pointer z)
{
  if ((is_t_integer(x)) && (is_t_integer(y)) && (is_t_integer(z)))
    {
#if Have_Overflow_Checks
      s7_int val;
      if ((!add_overflow(integer(x), integer(y), &val)) &&
	  (!add_overflow(val, integer(z), &val)))
	return(wrap_integer(sc, val));
      if (With_Warnings)
	s7_warn(sc, s7_Warn_Bufsize, "integer add overflow: (+ %" ld64 " %" ld64 " %" ld64 ")\n", integer(x), integer(y), integer(z));
      return(wrap_real(sc, (long_double)integer(x) + (long_double)integer(y) + (long_double)integer(z)));
#else
      return(wrap_integer(sc, integer(x) + integer(y) + integer(z)));
#endif
    }
  if ((is_t_real(x)) && (is_t_real(y)) && (is_t_real(z)))
    return(wrap_real(sc, real(x) + real(y) + real(z)));
  {
    s7_pointer num = add_p_pp_wrapped(sc, x, y);
    sc->error_argnum = 1;
    num = add_p_pp_wrapped(sc, num, z);
    sc->error_argnum = 0;
    return(num);
  }
}
#else
#define add_p_ppp_wrapped add_p_ppp
#endif


static s7_pointer g_add(s7_scheme *sc, s7_pointer args)
{
  #define H_add "(+ ...) adds its arguments"
  #define Q_add sc->pcl_n

  s7_pointer x, p;
  if (is_null(args))
    return(int_zero);
  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	return(method_or_bust_p(sc, x, sc->add_symbol, a_number_string));
      return(x);
    }
  if (is_null(cdr(p)))
    return(add_p_pp(sc, x, car(p)));
  for (sc->error_argnum = 0; is_pair(cdr(p)); p = cdr(p), sc->error_argnum++)
    x = add_p_pp_wrapped(sc, x, car(p));
  x = add_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_add_2(s7_scheme *sc, s7_pointer args) {return(add_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_add_2_wrapped(s7_scheme *sc, s7_pointer args) {return(add_p_pp_wrapped(sc, car(args), cadr(args)));}

static s7_pointer g_add_ii(s7_scheme *sc, s7_pointer args) /* this doesn't matter much, to my disappointment */
{
  return(add_if_overflow_to_real_or_big_integer(sc, integer(car(args)), integer(cadr(args))));
}

static s7_pointer g_add_ii_wrapped(s7_scheme *sc, s7_pointer args)
{
#if !With_Gmp
  return(add_if_overflow_to_real_wrapped(sc, integer(car(args)), integer(cadr(args))));
#else
  return(add_if_overflow_to_real_or_big_integer(sc, integer(car(args)), integer(cadr(args))));
#endif
}

static s7_pointer g_add_3(s7_scheme *sc, s7_pointer args) {return(add_p_ppp(sc, car(args), cadr(args), caddr(args)));}
static s7_pointer g_add_3_wrapped(s7_scheme *sc, s7_pointer args) {return(add_p_ppp_wrapped(sc, car(args), cadr(args), caddr(args)));}

static s7_pointer g_add_4(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = add_p_pp_wrapped(sc, car(args), cadr(args));
  s7_pointer p = cddr(args);
  sc->error_argnum = 2;
  p = add_p_pp(sc, x, add_p_pp_wrapped(sc, car(p), cadr(p)));
  sc->error_argnum = 0;
  return(p);
}

static s7_pointer g_add_x1_1(s7_scheme *sc, s7_pointer x, int32_t pos)
{
  if (is_t_integer(x))
    return(add_if_overflow_to_real_or_big_integer(sc, integer(x), 1));

  switch (type(x))
    {
    case T_RATIO:   return(integer_ratio_add_if_overflow_to_real_or_rational(sc, int_one, x)); /* return(add_p_pp(sc, x, int_one)) */
    case T_REAL:    return(make_real(sc, real(x) + 1.0));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) + 1.0, imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER:
      mpz_set_si(sc->mpz_1, 1);
      mpz_add(sc->mpz_1, big_integer(x), sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(add_p_pp(sc, x, int_one));
#endif
    default:
      return(method_or_bust(sc, x, sc->add_symbol,
			    (pos == 1) ? set_plist_2(sc, x, int_one) : set_plist_2(sc, int_one, x),
			    a_number_string, pos));
    }
  return(x);
}

#if With_Gmp
static s7_pointer g_add_x1(s7_scheme *sc, s7_pointer args) {return(g_add_x1_1(sc, car(args), 1));}
#else
static s7_pointer g_add_x1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args);
  if (is_t_integer(x)) return(add_if_overflow_to_real_or_big_integer(sc, integer(x), 1)); /* return(make_integer(sc, integer(x) + 1)); */
  if (is_t_real(x)) return(make_real(sc, real(x) + 1.0));
  if (is_t_complex(x)) return(make_complex_not_0i(sc, real_part(x) + 1.0, imag_part(x)));
  return(add_p_pp(sc, x, int_one));
}
#endif
static s7_pointer g_add_1x(s7_scheme *sc, s7_pointer args) {return(g_add_x1_1(sc, cadr(args), 2));}

static s7_pointer g_add_xi(s7_scheme *sc, s7_pointer x, s7_int y, int32_t loc)
{
  if (is_t_integer(x))
    return(add_if_overflow_to_real_or_big_integer(sc, integer(x), y));

  switch (type(x))
    {
    case T_RATIO:   return(add_p_pp(sc, x, wrap_integer(sc, y)));
    case T_REAL:    return(make_real(sc, real(x) + y));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) + y, imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER:
      mpz_set_si(sc->mpz_1, y);
      mpz_add(sc->mpz_1, big_integer(x), sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(add_p_pp(sc, x, wrap_integer(sc, y)));
#endif
    default: return(method_or_bust_pp(sc, x, sc->add_symbol, x, make_integer(sc, y), a_number_string, loc));
    }
  return(x);
}

static s7_pointer g_add_xf(s7_scheme *sc, s7_pointer x, s7_double y, int32_t loc)
{
  if (is_t_real(x)) return(make_real(sc, real(x) + y));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) + y));
    case T_RATIO:   return(make_real(sc, (s7_double)fraction(x) + y));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) + y, imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(add_p_pp(sc, x, wrap_real(sc, y)));
#endif
    default: return(method_or_bust_pp(sc, x, sc->add_symbol, x, make_real(sc, y), a_number_string, loc));
    }
  return(x);
}

static s7_pointer add_p_pi(s7_scheme *sc, s7_pointer p1, s7_int i1) {return(g_add_xi(sc, p1, i1, 1));}
static s7_pointer add_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 + x2));} /* very few calls */
static s7_pointer add_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_integer(sc, x1 + x2));}    /* no calls */
static s7_double add_d_d(s7_double x) {return(x);}
static s7_double add_d_dd(s7_double x1, s7_double x2) {return(x1 + x2);}
static s7_double add_d_id(s7_int x1, s7_double x2) {return(x1 + x2);}
static s7_double add_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 + x2 + x3);}
static s7_double add_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 + x2 + x3 + x4);}
static s7_int add_i_ii(s7_int i1, s7_int i2) {return(i1 + i2);}
static s7_int add_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 + i2 + i3);}

static bool is_int_arg(s7_scheme *sc, s7_pointer arg)
{
  if (is_t_integer(arg)) return(true);
  if (is_pair(arg))
    {
      s7_pointer sig = NULL;
      if (is_c_function(car(arg)))
	sig = c_function_signature(car(arg));
      else
	if ((is_symbol(car(arg))) && (is_defined_global(car(arg))) && (is_c_function(global_value(car(arg)))))
	  sig = c_function_signature(global_value(car(arg)));
      if ((sig) && (is_pair(sig)) && (car(sig) == sc->is_integer_symbol))
	return(true);
    }
  return(false);
}

static bool has_two_int_args(s7_scheme *sc, s7_pointer expr)
{
  /* maybe support (apply int-func...), also the global business is wrong if it is currently shadowed */
  return((is_int_arg(sc, cadr(expr))) && 
	 (is_int_arg(sc, caddr(expr))));
}

static s7_pointer g_random_i(s7_scheme *sc, s7_pointer args);
static s7_pointer g_multiply_2(s7_scheme *sc, s7_pointer args);
static s7_pointer g_multiply_2_wrapped(s7_scheme *sc, s7_pointer args);
static s7_pointer g_subtract_2(s7_scheme *sc, s7_pointer args);
static s7_pointer g_subtract_2_wrapped(s7_scheme *sc, s7_pointer args);
/* C should let forward declarations be local to the procedure they occur in */

static s7_pointer add_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  /* (+ s f) (+ (* s s) s) (+ s s) (+ s (* s s)) */
  if (args == 2)
    {
      const s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
      if (has_two_int_args(sc, expr))
	return(sc->add_2_ints);

      if ((is_pair(arg1)) && (has_fn(arg1)) && (fn_proc(arg1) == g_multiply_2)) set_fn_direct(arg1, g_multiply_2_wrapped);
      if ((is_pair(arg2)) && (has_fn(arg2)))
	{
	  if (fn_proc(arg2) == g_multiply_2) set_fn_direct(arg2, g_multiply_2_wrapped);
	  if (fn_proc(arg2) == g_subtract_2) set_fn_direct(arg2, g_subtract_2_wrapped);
	}
      if (arg2 == int_one)    /* (+ ... 1) */
	return(sc->add_x1);
      if ((is_t_integer(arg1)) && ((is_pair(arg2)) && (is_optimized(arg2)) && (is_h_safe_c_nc(arg2)) && (fn_proc(arg2) == g_random_i)))
	{
	  set_opt3_int(cdr(expr), integer(cadr(arg2)));
	  set_safe_optimize_op(expr, HOP_SAFE_C_NC); /* i.e. don't evaluate random call beforehand(?) */
	  return(sc->add_i_random);
	}
      if (arg1 == int_one) return(sc->add_1x);
      return(sc->add_2);
    }
  return((args == 3) ? sc->add_3 : ((args == 4) ? sc->add_4 : func));
}

/* ---------------------------------------- subtract ---------------------------------------- */
static s7_pointer negate_p_p(s7_scheme *sc, s7_pointer x)     /* can't use "negate" because it confuses C++! */
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == s7_Int64_Min)
#if With_Gmp
	{
	  mpz_set_si(sc->mpz_1, s7_Int64_Min);
	  mpz_neg(sc->mpz_1, sc->mpz_1);
	  return(mpz_to_big_integer(sc, sc->mpz_1));
	}
#else
	sole_arg_out_of_range_error_nr(sc, sc->subtract_symbol, x, wrap_string(sc, "most-negative-fixnum can't be negated", 37));
#endif
      return(make_integer(sc, -integer(x)));

    case T_RATIO:   return(make_simpler_ratio(sc, -numerator(x), denominator(x)));
    case T_REAL:    return(make_real(sc, -real(x)));
    case T_COMPLEX: return(make_complex_not_0i(sc, -real_part(x), -imag_part(x)));

#if With_Gmp
    case T_BIG_INTEGER:
      mpz_neg(sc->mpz_1, big_integer(x));
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO:
      mpq_neg(sc->mpq_1, big_ratio(x));
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    case T_BIG_REAL:
      mpfr_neg(sc->mpfr_1, big_real(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_neg(sc->mpc_1, big_complex(x), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_p(sc, x, sc->subtract_symbol, a_number_string));
    }
}

static inline s7_pointer subtract_if_overflow_to_real_or_big_integer(s7_scheme *sc, s7_int x, s7_int y)
{
#if Have_Overflow_Checks
  s7_int val;
  if (subtract_overflow(x, y, &val))
#if With_Gmp
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_set_si(sc->mpz_2, y);
      mpz_sub(sc->mpz_1, sc->mpz_1, sc->mpz_2);
      return(mpz_to_big_integer(sc, sc->mpz_1));
    }
#else
    {
      if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "integer subtract overflow: (- %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (long_double)x - (long_double)y));
    }
#endif
  return(make_integer(sc, val));
#else
  return(make_integer(sc, x - y));
#endif
}

static s7_pointer subtract_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0)
	return(negate_p_p(sc, y));
      switch (type(y))
	{
	case T_INTEGER:
	  return(subtract_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));

	case T_RATIO:
	  {
#if Have_Overflow_Checks
	    s7_int z;
	    if ((multiply_overflow(integer(x), denominator(y), &z)) ||
		(subtract_overflow(z, numerator(y), &z)))
#if With_Gmp
	      {
		mpz_set_si(sc->mpz_1, integer(x));
		mpz_mul_si(sc->mpz_1, sc->mpz_1, denominator(y));
		mpz_set_si(sc->mpz_2, numerator(y));
		mpz_sub(mpq_numref(sc->mpq_1), sc->mpz_1, sc->mpz_2);
		mpz_set_si(mpq_denref(sc->mpq_1), denominator(y));
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
	      {
		if (With_Warnings)
		  s7_warn(sc, s7_Warn_Bufsize, "integer - ratio overflow: (- %" ld64 " %" ld64 "/%" ld64 ")\n", integer(x), numerator(y), denominator(y));
		return(make_real(sc, (long_double)integer(x) - fraction(y)));
	      }
#endif
	      return(make_ratio(sc, z, denominator(y)));
#else
	    return(make_ratio(sc, integer(x) * denominator(y) - numerator(y), denominator(y)));
#endif
	  }
	case T_REAL:
#if With_Gmp
	  if (s7_int_abs(integer(x)) >= Int64_To_Double_Limit) /* (- 9223372036854775807 .1) */
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), Mpfr_Rndn);
	      mpfr_sub_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (long_double)integer(x) - real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, (long_double)integer(x) - real_part(y), -imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(x));
	  mpz_sub(sc->mpz_1, sc->mpz_1, big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_sub(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_si_sub(sc->mpfr_1, integer(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_si(sc->mpc_1, integer(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  {
#if Have_Overflow_Checks
	    s7_int z;
	    if ((multiply_overflow(integer(y), denominator(x), &z)) ||
		(subtract_overflow(numerator(x), z, &z)))
#if With_Gmp
	      {
		mpz_set_si(sc->mpz_1, integer(y));
		mpz_mul_si(sc->mpz_1, sc->mpz_1, denominator(x));
		mpz_set_si(sc->mpz_2, numerator(x));
		mpz_sub(mpq_numref(sc->mpq_1), sc->mpz_2, sc->mpz_1);
		mpz_set_si(mpq_denref(sc->mpq_1), denominator(x));
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
	      {
		if (With_Warnings)
		  s7_warn(sc, s7_Warn_Bufsize, "ratio - integer overflow: (- %" ld64 "/%" ld64 " %" ld64 ")\n", numerator(x), denominator(x), integer(y));
		return(make_real(sc, fraction(x) - (long_double)integer(y)));
	      }
#endif
	    return(make_ratio(sc, z, denominator(x)));
#else
	    return(make_ratio(sc, numerator(x) - (integer(y) * denominator(x)), denominator(x)));
#endif
	  }
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
	    if (d1 == d2)
	      {
#if Have_Overflow_Checks
		s7_int q;
		if (subtract_overflow(n1, n2, &q))
#if With_Gmp
		  {
		    mpq_set_si(sc->mpq_1, n1, d1);
		    mpq_set_si(sc->mpq_2, n2, d2);
		    mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		    return(mpq_to_rational(sc, sc->mpq_1));
		  }
#else
		  {
		    if (With_Warnings)
		      s7_warn(sc, s7_Warn_Bufsize, "ratio - ratio overflow: (- %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
		    return(make_real(sc, ((long_double)n1 - (long_double)n2) / (long_double)d1));
		  }
#endif
	        return(make_ratio_with_div_check(sc, sc->subtract_symbol, q, d1));
#else
		return(make_ratio(sc, numerator(x) - numerator(y), denominator(x)));
#endif
	      }

#if Have_Overflow_Checks
	    {
	      s7_int n1d2, n2d1, d1d2, q;
	      if ((multiply_overflow(d1, d2, &d1d2)) ||
		  (multiply_overflow(n1, d2, &n1d2)) ||
		  (multiply_overflow(n2, d1, &n2d1)) ||
		  (subtract_overflow(n1d2, n2d1, &q)))
#if With_Gmp
		{
		  mpq_set_si(sc->mpq_1, n1, d1);
		  mpq_set_si(sc->mpq_2, n2, d2);
		  mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		  return(mpq_to_rational(sc, sc->mpq_1));
		}
#else
	        {
		  if (With_Warnings)
		    s7_warn(sc, s7_Warn_Bufsize, "ratio - ratio overflow: (- %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
		  return(make_real(sc, ((long_double)n1 / (long_double)d1) - ((long_double)n2 / (long_double)d2)));
		}
#endif
	      return(make_ratio_with_div_check(sc, sc->subtract_symbol, q, d1d2));
	    }
#else
	    return(make_ratio_with_div_check(sc, sc->subtract_symbol, n1 * d2 - n2 * d1, d1 * d2));
#endif
	  }
	case T_REAL:
	  return(make_real(sc, (s7_double)fraction(x) - real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, (s7_double)fraction(x) - real_part(y), -imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_set_z(sc->mpq_2, big_integer(y));
	  mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_sub(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_set_q(sc->mpfr_1, sc->mpq_1, Mpfr_Rndn);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
#if With_Gmp
	  if (s7_int_abs(integer(y)) >= Int64_To_Double_Limit) /* (- .1 92233720368547758071) */
	    {
	      mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	      mpfr_sub_si(sc->mpfr_1, sc->mpfr_1, integer(y), Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, real(x) - (long_double)integer(y))); /* long_double saves (- 9007199254740996.0 9007199254740995): 1.0 */
	case T_RATIO:
	  return(make_real(sc, real(x) - (s7_double)fraction(y)));
	case T_REAL:
	  return(make_real(sc, real(x) - real(y)));
	case T_COMPLEX:
	  return(make_complex_not_0i(sc, real(x) - real_part(y), -imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_sub_z(sc->mpfr_1, sc->mpfr_1, big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_sub_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_d_sub(sc->mpfr_1, real(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_complex_not_0i(sc, real_part(x) - integer(y), imag_part(x)));
	case T_RATIO:
	  return(make_complex_not_0i(sc, real_part(x) - (s7_double)fraction(y), imag_part(x)));
	case T_REAL:
	  return(make_complex_not_0i(sc, real_part(x) - real(y), imag_part(x)));
	case T_COMPLEX:
	  return(make_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sub_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  mpz_set_si(sc->mpz_1, integer(y));
	  mpz_sub(sc->mpz_1, big_integer(x), sc->mpz_1);
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_sub(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  mpfr_sub_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpz_sub(sc->mpz_1, big_integer(x), big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  mpq_sub(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_sub(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_sub(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  mpfr_sub_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_sub(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_sub(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  mpfr_sub_si(sc->mpfr_1, big_real(x), integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_sub_q(sc->mpfr_1, big_real(x), sc->mpq_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_sub_d(sc->mpfr_1, big_real(x), real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_fr_sub(sc->mpc_1, big_real(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpfr_sub_z(sc->mpfr_1, big_real(x), big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_sub_q(sc->mpfr_1, big_real(x), big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_sub(sc->mpfr_1, big_real(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_fr_sub(sc->mpc_1, big_real(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  mpc_set_si(sc->mpc_2, integer(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(y); */
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_sub(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_sub_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_sub(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->subtract_symbol, x, y, a_number_string, 2));
	}
#endif
      default:
	return(method_or_bust_pp(sc, x, sc->subtract_symbol, x, y, a_number_string, 1));
    }
}

static s7_pointer negate_p_p_wrapped(s7_scheme *sc, s7_pointer x)     /* can't use "negate" because it confuses C++! */
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == s7_Int64_Min)
	sole_arg_out_of_range_error_nr(sc, sc->subtract_symbol, x, wrap_string(sc, "most-negative-fixnum can't be negated", 37));
      return(wrap_integer(sc, -integer(x)));
    case T_REAL:
      return(wrap_real(sc, -real(x)));
    case T_COMPLEX:
      return(wrap_complex(sc, -real_part(x), -imag_part(x)));
    }
  return(negate_p_p(sc, x));
}

#if !With_Gmp
static s7_pointer subtract_if_overflow_to_real_wrapped(s7_scheme *sc, s7_int x, s7_int y)
{
#if Have_Overflow_Checks
  s7_int val;
  if (subtract_overflow(x, y, &val))
    {
      if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "integer subtract overflow: (- %" ld64 " %" ld64 ")\n", x, y);
      return(wrap_real(sc, (long_double)x - (long_double)y));
    }
  return(wrap_integer(sc, val));
#else
  return(wrap_integer(sc, x - y));
#endif
}

static s7_pointer subtract_p_pp_wrapped(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(negate_p_p_wrapped(sc, y));
      switch (type(y))
	{
	case T_INTEGER:	  return(subtract_if_overflow_to_real_wrapped(sc, integer(x), integer(y)));
	case T_REAL:	  return(wrap_real(sc, (long_double)integer(x) - real(y)));
	case T_COMPLEX:	  return(wrap_complex(sc, (long_double)integer(x) - real_part(y), -imag_part(y)));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:	  return(wrap_real(sc, real(x) - (long_double)integer(y))); /* long_double saves (- 9007199254740996.0 9007199254740995): 1.0 */
	case T_REAL:	  return(wrap_real(sc, real(x) - real(y)));
	case T_COMPLEX:	  return(wrap_complex(sc, real(x) - real_part(y), -imag_part(y)));
	}
    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:	  return(wrap_complex(sc, real_part(x) - integer(y), imag_part(x)));
	case T_REAL:	  return(wrap_complex(sc, real_part(x) - real(y), imag_part(x)));
	case T_COMPLEX:	  return(wrap_real_or_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
	}}
  return(subtract_p_pp(sc, x, y));
}
#else
#define subtract_p_pp_wrapped subtract_p_pp
#endif

static s7_pointer g_subtract(s7_scheme *sc, s7_pointer args)
{
  #define H_subtract "(- x1 ...) subtracts its trailing arguments from the first, or negates the first if only one it is given"
  #define Q_subtract sc->pcl_n

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(p))
    return(negate_p_p(sc, x));
  for (sc->error_argnum = 0; is_pair(cdr(p)); p = cdr(p), sc->error_argnum++)
    x = subtract_p_pp_wrapped(sc, x, car(p));
  x = subtract_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_subtract_1(s7_scheme *sc, s7_pointer args) {return(negate_p_p(sc, car(args)));}
static s7_pointer g_subtract_1_wrapped(s7_scheme *sc, s7_pointer args) {return(negate_p_p_wrapped(sc, car(args)));}
static s7_pointer g_subtract_2(s7_scheme *sc, s7_pointer args) {return(subtract_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_subtract_2_wrapped(s7_scheme *sc, s7_pointer args) {return(subtract_p_pp_wrapped(sc, car(args), cadr(args)));}

static s7_pointer g_subtract_3(s7_scheme *sc, s7_pointer args) /* wrapped version gets no hits */
{
  s7_pointer x = car(args);
  x = subtract_p_pp_wrapped(sc, x, cadr(args));
  sc->error_argnum = 1;
  x = subtract_p_pp(sc, x, caddr(args));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer minus_c1(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: return(subtract_if_overflow_to_real_or_big_integer(sc, integer(x), 1));
    case T_RATIO:   return(subtract_p_pp(sc, x, int_one));
    case T_REAL:    return(make_real(sc, real(x) - 1.0));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) - 1.0, imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, x, int_one));
#endif
    default:
      return(method_or_bust_pp(sc, x, sc->subtract_symbol, x, int_one, a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_x1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer num = car(args);
#if With_Gmp
  return(subtract_p_pp(sc, num, int_one));
#endif
  /* return((is_t_integer(num)) ? make_integer(sc, integer(num) - 1) : minus_c1(sc, num)); */
  return((is_t_integer(num)) ? subtract_if_overflow_to_real_or_big_integer(sc, integer(num), 1) : minus_c1(sc, num));
}

static s7_pointer g_subtract_2f(s7_scheme *sc, s7_pointer args) /* (- x f) */
{
  const s7_pointer x = car(args);
  const s7_double n = real(cadr(args)); /* checked below is_t_real */
  if (is_t_real(x)) return(make_real(sc, real(x) - n));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) - n));
    case T_RATIO:   return(make_real(sc, (s7_double)fraction(x) - n));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) - n, imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, x, cadr(args)));
#endif
    default:
      return(method_or_bust(sc, x, sc->subtract_symbol, args, a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_f2(s7_scheme *sc, s7_pointer args) /* (- f x) */
{
  const s7_pointer x = cadr(args);
  const s7_double n = real(car(args)); /* checked below is_t_real */

  if (is_t_real(x)) return(make_real(sc, n - real(x)));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, n - integer(x)));
    case T_RATIO:   return(make_real(sc, n - (s7_double)fraction(x)));
    case T_COMPLEX: return(make_complex_not_0i(sc, n - real_part(x), -imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, car(args), x));
#endif
    default:
      return(method_or_bust(sc, x, sc->subtract_symbol, args, a_number_string, 1));
    }
  return(x);
}

static s7_int subtract_i_ii(s7_int i1, s7_int i2) {return(i1 - i2);}
static s7_int subtract_i_i(s7_int x) {return(-x);}
static s7_int subtract_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 - i2 - i3);}
static s7_double subtract_d_d(s7_double x) {return(-x);}
static s7_double subtract_d_dd(s7_double x1, s7_double x2) {return(x1 - x2);}
static s7_double subtract_d_id(s7_int x1, s7_double x2) {return(x1 - x2);}
static s7_double subtract_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 - x2 - x3);}
static s7_double subtract_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 - x2 - x3 - x4);}
static s7_pointer subtract_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 - x2));}
static s7_pointer subtract_p_ii(s7_scheme *sc, s7_int i1, s7_int i2) {return(make_integer(sc, i1 - i2));}

static s7_pointer g_sub_xi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x))
    return(subtract_if_overflow_to_real_or_big_integer(sc, integer(x), y));

  switch (type(x))
    {
    case T_RATIO:   return(make_ratio(sc, numerator(x) - (y * denominator(x)), denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) - y));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(x) - y, imag_part(x)));
#if With_Gmp
    case T_BIG_INTEGER:
      mpz_set_si(sc->mpz_1, y);
      mpz_sub(sc->mpz_1, big_integer(x), sc->mpz_1);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(subtract_p_pp(sc, x, wrap_integer(sc, y)));
#endif
    default: return(method_or_bust_pp(sc, x, sc->subtract_symbol, x, make_integer(sc, y), a_number_string, 1));
    }
  return(x);
}

static s7_pointer subtract_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  s7_pointer arg1, arg2;
  if (args == 1) return(sc->subtract_1);
  if (args != 2) return((args == 3) ? sc->subtract_3 : func);
  arg1 = cadr(expr);
  arg2 = caddr(expr);
  if ((is_pair(arg1)) && (has_fn(arg1)))
    {
      if (fn_proc(arg1) == g_multiply_2) set_fn_direct(arg1, g_multiply_2_wrapped);
      if (fn_proc(arg1) == g_add_2) set_fn_direct(arg1, g_add_2_wrapped);
      if (fn_proc(arg1) == g_add_ii) set_fn_direct(arg1, g_add_ii_wrapped);
    }
  if ((is_pair(arg2)) && (has_fn(arg2)) && (fn_proc(arg2) == g_multiply_2)) set_fn_direct(arg2, g_multiply_2_wrapped);
  /* sub_random_i (parallels add_i_random) only occurs in tmap.scm */
  if (arg2 == int_one) return(sc->subtract_x1);
  if (is_t_real(arg1)) return(sc->subtract_f2);
  if (is_t_real(arg2)) return(sc->subtract_2f);
  return(sc->subtract_2);
}


/* ---------------------------------------- multiply ---------------------------------------- */
#define Quotient_Float_Limit 1e13
#define Quotient_Int_Limit 10000000000000
/* fraction(x) is not accurate enough if it involves numbers over e18 even when done with long_doubles */

static inline s7_pointer multiply_if_overflow_to_real_or_big_integer(s7_scheme *sc, s7_int x, s7_int y)
{
#if Have_Overflow_Checks
  s7_int val;
  if (multiply_overflow(x, y, &val))
#if With_Gmp
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_mul_si(sc->mpz_1, sc->mpz_1, y);
      return(mpz_to_big_integer(sc, sc->mpz_1));
    }
#else
    {
      if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "integer multiply overflow: (* %" ld64 " %" ld64 ")\n", x, y);
      return(make_real(sc, (s7_double)x * (s7_double)y));
    }
#endif
    return(make_integer(sc, val));
#else
  return(make_integer(sc, x * y));
#endif
}

static s7_pointer integer_ratio_multiply_if_overflow_to_real_or_ratio(s7_scheme *sc, s7_int x, s7_pointer y)
{
#if Have_Overflow_Checks
  s7_int z;
  if (multiply_overflow(x, numerator(y), &z))
#if With_Gmp
    {
      mpz_set_si(sc->mpz_1, x);
      mpz_mul_si(sc->mpz_1, sc->mpz_1, numerator(y));
      mpq_set_si(sc->mpq_1, 1, denominator(y));
      mpq_set_num(sc->mpq_1, sc->mpz_1);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    }
#else
    {
      if (With_Warnings)
	s7_warn(sc, s7_Warn_Bufsize, "integer * ratio overflow: (* %" ld64 " %" ld64 "/%" ld64 ")\n", x, numerator(y), denominator(y));
      return(make_real(sc, (s7_double)x * (s7_double)fraction(y)));
    }
#endif
    return(make_ratio(sc, z, denominator(y)));
#else
  return(make_ratio(sc, x * numerator(y), denominator(y)));
#endif
}

static s7_pointer multiply_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(multiply_if_overflow_to_real_or_big_integer(sc, integer(x), integer(y)));
	case T_RATIO:
	  return(integer_ratio_multiply_if_overflow_to_real_or_ratio(sc, integer(x), y));
	case T_REAL:
#if With_Gmp
	  if (s7_int_abs(integer(x)) > Quotient_Int_Limit)
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), Mpfr_Rndn);
	      mpfr_set_d(sc->mpfr_2, real(y), Mpfr_Rndn);
	      mpfr_mul(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (long_double)integer(x) * real(y)));
	case T_COMPLEX:
	  return(make_complex(sc, (long_double)integer(x) * real_part(y), (long_double)integer(x) * imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpz_mul_si(sc->mpz_1, big_integer(y), integer(x));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_mul(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_mul_si(sc->mpfr_1, big_real(y), integer(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_mul_si(sc->mpc_1, big_complex(y), integer(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* x might be 0 */
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer_ratio_multiply_if_overflow_to_real_or_ratio(sc, integer(y), x));
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
#if Have_Overflow_Checks
	    {
	      s7_int n1n2, d1d2;
	      if ((multiply_overflow(d1, d2, &d1d2)) ||
		  (multiply_overflow(n1, n2, &n1n2)))
#if With_Gmp
		{
		  mpq_set_si(sc->mpq_1, n1, d1);
		  mpq_set_si(sc->mpq_2, n2, d2);
		  mpq_mul(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
		}
#else
	        {
		  if (With_Warnings)
		    s7_warn(sc, s7_Warn_Bufsize, "ratio * ratio overflow: (* %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", n1, d1, n2, d2);
		  return(make_real(sc, (s7_double)fraction(x) * (s7_double)fraction(y)));
		}
#endif
	      return(make_ratio_with_div_check(sc, sc->multiply_symbol, n1n2, d1d2));
	    }
#else
	    return(make_ratio_with_div_check(sc, sc->multiply_symbol, n1 * n2, d1 * d2));
#endif
	  }
	case T_REAL:
#if With_Gmp
	  if (numerator(x) > Quotient_Int_Limit)
	    {
	      mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	      mpfr_set_q(sc->mpfr_1, sc->mpq_1, Mpfr_Rndn);
	      mpfr_mul_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (s7_double)fraction(x) * real(y)));
	case T_COMPLEX:
	  return(make_complex(sc, (s7_double)fraction(x) * real_part(y), (s7_double)fraction(x) * imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_set_z(sc->mpq_2, big_integer(y));
	  mpq_mul(sc->mpq_1, sc->mpq_1, sc->mpq_2);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_mul(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_mul_q(sc->mpfr_1, big_real(y), sc->mpq_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
#if With_Gmp
	  if (s7_int_abs(integer(y)) > Quotient_Int_Limit)
	    {
	      mpfr_set_si(sc->mpfr_1, integer(y), Mpfr_Rndn);
	      mpfr_set_d(sc->mpfr_2, real(x), Mpfr_Rndn);
	      mpfr_mul(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, real(x) * (long_double)integer(y)));
	case T_RATIO:
#if With_Gmp
	  if (numerator(y) > Quotient_Int_Limit)
	    {
	      mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	      mpfr_set_q(sc->mpfr_1, sc->mpq_1, Mpfr_Rndn);
	      mpfr_mul_d(sc->mpfr_1, sc->mpfr_1, real(x), Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (s7_double)fraction(y) * real(x)));
	case T_REAL:
	  return(make_real(sc, real(x) * real(y)));
	case T_COMPLEX:
	  return(make_complex(sc, real(x) * real_part(y), real(x) * imag_part(y)));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_mul_z(sc->mpfr_1, sc->mpfr_1, big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_mul_d(sc->mpfr_1, big_real(y), real(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* x might = 0.0 */
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_complex(sc, real_part(x) * integer(y), imag_part(x) * integer(y)));
	case T_RATIO:
	  return(make_complex(sc, real_part(x) * (s7_double)fraction(y), imag_part(x) * (s7_double)fraction(y)));
	case T_REAL:
	  return(make_complex(sc, real_part(x) * real(y), imag_part(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double r1 = real_part(x), r2 = real_part(y), i1 = imag_part(x), i2 = imag_part(y);
	    return(make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	  }
#if With_Gmp
	case T_BIG_INTEGER:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_mul_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  mpz_mul_si(sc->mpz_1, big_integer(x), integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_mul(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	  mpfr_mul_z(sc->mpfr_1, sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpz_mul(sc->mpz_1, big_integer(x), big_integer(y));
	  return(mpz_to_integer(sc, sc->mpz_1));
	case T_BIG_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  mpq_mul(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_mul_z(sc->mpfr_1, big_real(y), big_integer(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_mul(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_mul(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	  mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_mul(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_mul(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  mpfr_mul_q(sc->mpfr_1, big_real(y), big_ratio(x), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  mpfr_mul_si(sc->mpfr_1, big_real(x), integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_mul_q(sc->mpfr_1, big_real(x), sc->mpq_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_mul_d(sc->mpfr_1, big_real(x), real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_mul_fr(sc->mpc_1, sc->mpc_1, big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpfr_mul_z(sc->mpfr_1, big_real(x), big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_mul_q(sc->mpfr_1, big_real(x), big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  mpfr_mul(sc->mpfr_1, big_real(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  mpc_mul_fr(sc->mpc_1, big_complex(y), big_real(x), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* 0.0? */
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  mpc_mul_si(sc->mpc_1, big_complex(x), integer(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(y); */
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_mul(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  mpc_mul_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  mpc_mul(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->multiply_symbol, x, y, a_number_string, 2));
	}
#endif
      default:
	return(method_or_bust_pp(sc, x, sc->multiply_symbol, x, y, a_number_string, 1));
    }
}

#if !With_Gmp
static inline s7_pointer multiply_if_overflow_to_real_wrapped(s7_scheme *sc, s7_int x, s7_int y)
{
#if Have_Overflow_Checks
  s7_int val;
  if (multiply_overflow(x, y, &val))
    {
      if (With_Warnings) s7_warn(sc, s7_Warn_Bufsize, "integer multiply overflow: (* %" ld64 " %" ld64 ")\n", x, y);
      return(wrap_real(sc, (s7_double)x * (s7_double)y));
    }
    return(wrap_integer(sc, val));
#else
  return(wrap_integer(sc, x * y));
#endif
}

static s7_pointer multiply_p_pp_wrapped(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:	  return(multiply_if_overflow_to_real_wrapped(sc, integer(x), integer(y)));
	case T_REAL:	  return(wrap_real(sc, (long_double)integer(x) * real(y)));
	case T_COMPLEX:	  return(wrap_real_or_complex(sc, (long_double)integer(x) * real_part(y), (long_double)integer(x) * imag_part(y)));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:	  return(wrap_real(sc, real(x) * (long_double)integer(y)));
	case T_REAL:	  return(wrap_real(sc, real(x) * real(y)));
	case T_COMPLEX:	  return(wrap_real_or_complex(sc, real(x) * real_part(y), real(x) * imag_part(y)));
	}
    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:	  return(wrap_real_or_complex(sc, real_part(x) * integer(y), imag_part(x) * integer(y)));
	case T_REAL:	  return(wrap_real_or_complex(sc, real_part(x) * real(y), imag_part(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double r1 = real_part(x), r2 = real_part(y), i1 = imag_part(x), i2 = imag_part(y);
	    return(wrap_real_or_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	  }}}
  return(multiply_p_pp(sc, x, y));
}
#else
#define multiply_p_pp_wrapped multiply_p_pp
#endif

static s7_pointer multiply_p_ppp(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_pointer z)
{
  /* no hits for reals in tnum */
  /* if ((is_t_real(x)) && (is_t_real(y)) && (is_t_real(z))) return(make_real(sc, real(x) * real(y) * real(z))); */
  x = multiply_p_pp_wrapped(sc, x, y);
  sc->error_argnum = 1;
  x = multiply_p_pp(sc, x, z);
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer multiply_p_ppp_wrapped(s7_scheme *sc, s7_pointer x, s7_pointer y, s7_pointer z)
{
  /* no hits for reals in tnum */
  /* if ((is_t_real(x)) && (is_t_real(y)) && (is_t_real(z))) return(make_real(sc, real(x) * real(y) * real(z))); */
  x = multiply_p_pp_wrapped(sc, x, y);
  sc->error_argnum = 1;
  x = multiply_p_pp_wrapped(sc, x, z);
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer multiply_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (has_active_methods(sc, obj))
    return(find_and_apply_method(sc, obj, sc->multiply_symbol, args));
  if (num == 0)
    sole_arg_wrong_type_error_nr(sc, sc->multiply_symbol, obj, typ);
  wrong_type_error_nr(sc, sc->multiply_symbol, num, obj, typ);
  return(NULL);
}

static s7_pointer g_multiply(s7_scheme *sc, s7_pointer args)
{
  #define H_multiply "(* ...) multiplies its arguments"
  #define Q_multiply sc->pcl_n

  s7_pointer x, p;
  if (is_null(args))
    return(int_one);
  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	return(multiply_method_or_bust(sc, x, args, a_number_string, 0));
      return(x);
    }
  for (sc->error_argnum = 0; is_pair(cdr(p)); p = cdr(p), sc->error_argnum++)
    x = multiply_p_pp_wrapped(sc, x, car(p));
  x = multiply_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_multiply_2(s7_scheme *sc, s7_pointer args) {return(multiply_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_multiply_2_wrapped(s7_scheme *sc, s7_pointer args) {return(multiply_p_pp_wrapped(sc, car(args), cadr(args)));}
static s7_pointer g_multiply_3(s7_scheme *sc, s7_pointer args) {return(multiply_p_ppp(sc, car(args), cadr(args), caddr(args)));}
static s7_pointer g_multiply_3_wrapped(s7_scheme *sc, s7_pointer args) {return(multiply_p_ppp_wrapped(sc, car(args), cadr(args), caddr(args)));}

static s7_pointer g_mul_xi(s7_scheme *sc, s7_pointer x, s7_int n, int32_t loc)
{
  switch (type(x))
    {
    case T_INTEGER: return(multiply_if_overflow_to_real_or_big_integer(sc, integer(x), n));
    case T_RATIO:   return(integer_ratio_multiply_if_overflow_to_real_or_ratio(sc, n, x));
    case T_REAL:    return(make_real(sc, real(x) * n));
    case T_COMPLEX: return(make_complex(sc, real_part(x) * n, imag_part(x) * n));
#if With_Gmp
    case T_BIG_INTEGER:
      mpz_mul_si(sc->mpz_1, big_integer(x), n);
      return(mpz_to_integer(sc, sc->mpz_1));
    case T_BIG_RATIO: case T_BIG_REAL: case T_BIG_COMPLEX:
      return(multiply_p_pp(sc, x, wrap_integer(sc, n)));
#endif
    default:
      /* we can get here from mul_2_xi for example so the non-integer argument might not be a symbol */
      return(method_or_bust_pp(sc, x, sc->multiply_symbol, x, make_integer(sc, n), a_number_string, loc));
    }
  return(x);
}

static s7_pointer multiply_p_pi(s7_scheme *sc, s7_pointer p1, s7_int i1) {return(g_mul_xi(sc, p1, i1, 1));}

static s7_pointer g_mul_xf(s7_scheme *sc, s7_pointer x, s7_double y, int32_t num)
{
  /* it's possible to return different argument NaNs depending on the expression or how it is wrapped:
   *   (* (bignum +nan.0) +nan.123) -> nan.123
   *   (let () (define (func) (* (bignum +nan.0) +nan.123)) (func) (func)) -> nan.0
   * latter call is fx_c_aaa->fx_c_ac->g_mul_xf (if +nan.122 instead of +nan.0, we get +nan.122 so we always get one of the NaNs)
   */
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) * y));
    case T_RATIO:   return(make_real(sc, numerator(x) * y / denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) * y));
    case T_COMPLEX: return(make_complex(sc, real_part(x) * y, imag_part(x) * y));
#if With_Gmp
    case T_BIG_INTEGER:
      mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
      mpfr_mul_z(sc->mpfr_1, sc->mpfr_1, big_integer(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_RATIO:
      mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
      mpfr_mul_q(sc->mpfr_1, sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_REAL:
      mpfr_mul_d(sc->mpfr_1, big_real(x), y, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
      mpc_mul_fr(sc->mpc_1, big_complex(x), sc->mpfr_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default: return(method_or_bust_pp(sc, x, sc->multiply_symbol, x, make_real(sc, y), a_number_string, num));
    }
  return(x);
}

static s7_int multiply_i_ii(s7_int i1, s7_int i2)
{
#if Have_Overflow_Checks
  s7_int val;
  if (multiply_overflow(i1, i2, &val))
    {
#if With_Warnings
      fprintf(stderr, "%s[%d]: integer multiply overflow: (* %" ld64 " %" ld64 ")\n", __func__, __LINE__, i1, i2);
#endif
      return(s7_Int64_Max); /* this is inconsistent with other unopt cases where an overflow -> double result */
    }
  /* (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (do ((j 0 (+ j 1))) ((= j 1)) (even? (* (ash 1 43) (ash 1 43)))))) (define (hi) (func)) (hi)) */
  return(val);
#else
  return(i1 * i2);
#endif
}

static s7_int multiply_i_iii(s7_int i1, s7_int i2, s7_int i3)
{
#if Have_Overflow_Checks
  s7_int val1, val2;
  if ((multiply_overflow(i1, i2, &val1)) ||
      (multiply_overflow(val1, i3, &val2)))
    {
#if With_Warnings
      fprintf(stderr, "%s[%d]: integer multiply overflow: (* %" ld64 " %" ld64 " %" ld64 ")\n", __func__, __LINE__, i1, i2, i3);
#endif
      return(s7_Int64_Max);
    }
  return(val2);
#else
  return(i1 * i2 * i3);
#endif
}

static s7_double multiply_d_d(s7_double x) {return(x);}
static s7_double multiply_d_dd(s7_double x1, s7_double x2) {return(x1 * x2);}
static s7_double multiply_d_id(s7_int x1, s7_double x2) {return(x1 * x2);}
static s7_double multiply_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 * x2 * x3);}
static s7_double multiply_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 * x2 * x3 * x4);}
static s7_pointer mul_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 * x2));}

static s7_pointer multiply_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  s7_pointer arg1, arg2;
  if (args < 2) return(func);
  arg1 = cadr(expr);
  if ((is_pair(arg1)) && (has_fn(arg1)))
    {
      if (fn_proc(arg1) == g_add_2) set_fn_direct(arg1, g_add_2_wrapped);
      if (fn_proc(arg1) == g_add_ii) set_fn_direct(arg1, g_add_ii_wrapped);
      if (fn_proc(arg1) == g_add_3) set_fn_direct(arg1, g_add_3_wrapped);
      if (fn_proc(arg1) == g_subtract_2) set_fn_direct(arg1, g_subtract_2_wrapped);
      if (fn_proc(arg1) == g_subtract_1) set_fn_direct(arg1, g_subtract_1_wrapped);
    }
  arg2 = caddr(expr);
  if ((is_pair(arg2)) && (has_fn(arg2)))
    {
      if (fn_proc(arg2) == g_add_2) set_fn_direct(arg2, g_add_2_wrapped);
      if (fn_proc(arg2) == g_add_ii) set_fn_direct(arg2, g_add_ii_wrapped);
      if (fn_proc(arg2) == g_add_3) set_fn_direct(arg2, g_add_3_wrapped);
      if (fn_proc(arg2) == g_subtract_2) set_fn_direct(arg2, g_subtract_2_wrapped);
      if (fn_proc(arg2) == g_subtract_1) set_fn_direct(arg2, g_subtract_1_wrapped);
    }
  if (args == 2) return(sc->multiply_2);
  if (args == 3) return(sc->multiply_3);
  return(func);
}


/* ---------------------------------------- divide ---------------------------------------- */
static s7_pointer complex_invert(s7_scheme *sc, s7_pointer x)
{
  s7_double r2 = real_part(x), i2 = imag_part(x);
  s7_double den = (r2 * r2 + i2 * i2);
  /* here if x is, for example, -inf.0+i, den is +inf.0 so -i2/den is -0.0 (in gcc anyway), so the imag part is 0.0 */
  return(make_complex(sc, r2 / den, -i2 / den));
}

static s7_pointer invert_p_p(s7_scheme *sc, s7_pointer num)
{
#if With_Gmp
  s7_pointer x;
#endif
  switch (type(num))
    {
    case T_INTEGER:
#if With_Gmp && (!Pointer_32)
      if (integer(num) == s7_Int64_Min) /* (/ 1 (*s7* 'most-negative-fixnum)) -> -1/9223372036854775808 */
	{
	  new_cell(sc, x, T_BIG_RATIO);
	  big_ratio_bgr(x) = alloc_bigrat(sc);
	  add_big_ratio(sc, x);
	  mpz_set_si(sc->mpz_1, s7_Int64_Max);
	  mpz_set_si(sc->mpz_2, 1);
	  mpz_add(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	  mpq_set_si(big_ratio(x), -1, 1);
	  mpq_set_den(big_ratio(x), sc->mpz_1); /* geez... */
	  return(x);
	}
#endif
      if (integer(num) == 0)
	division_by_zero_error_1_nr(sc, sc->divide_symbol, num);
      return(make_simple_ratio(sc, 1, integer(num)));  /* this checks for int */
    case T_RATIO:
      return(make_simple_ratio(sc, denominator(num), numerator(num)));
    case T_REAL:
      if (real(num) == 0.0)
	division_by_zero_error_1_nr(sc, sc->divide_symbol, num);
      return(make_real(sc, 1.0 / real(num)));
    case T_COMPLEX:
      return(complex_invert(sc, num));

#if With_Gmp
    case T_BIG_INTEGER:
      if (mpz_cmp_ui(big_integer(num), 0) == 0)
	division_by_zero_error_1_nr(sc, sc->divide_symbol, num);
      if ((mpz_cmp_ui(big_integer(num), 1) == 0) || (mpz_cmp_si(big_integer(num), -1) == 0))
	return(num);
      new_cell(sc, x, T_BIG_RATIO);
      big_ratio_bgr(x) = alloc_bigrat(sc);
      add_big_ratio(sc, x);
      mpq_set_si(big_ratio(x), 1, 1);
      mpq_set_den(big_ratio(x), big_integer(num));
      mpq_canonicalize(big_ratio(x));
      return(x);
    case T_BIG_RATIO:
      if (mpz_cmp_ui(mpq_numref(big_ratio(num)), 1) == 0)
	return(mpz_to_integer(sc, mpq_denref(big_ratio(num))));
      if (mpz_cmp_si(mpq_numref(big_ratio(num)), -1) == 0)
	{
	  mpz_neg(sc->mpz_1, mpq_denref(big_ratio(num)));
	  return(mpz_to_integer(sc, sc->mpz_1));
	}
      new_cell(sc, x, T_BIG_RATIO);
      big_ratio_bgr(x) = alloc_bigrat(sc);
      add_big_ratio(sc, x);
      mpq_inv(big_ratio(x), big_ratio(num));
      mpq_canonicalize(big_ratio(x));
      return(x);
    case T_BIG_REAL:
      if (mpfr_zero_p(big_real(num)))
	division_by_zero_error_1_nr(sc, sc->divide_symbol, num);
      x = mpfr_to_big_real(sc, big_real(num));
      mpfr_ui_div(big_real(x), 1, big_real(x), Mpfr_Rndn);
      return(x);
    case T_BIG_COMPLEX:
      if ((!mpfr_number_p(mpc_realref(big_complex(num)))) || (!mpfr_number_p(mpc_imagref(big_complex(num)))))
	return(complex_NaN);
      mpc_ui_div(sc->mpc_1, 1, big_complex(num), MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1)); /* x might be 0+0i if real-part is inf? */
#endif
    default:
      if_method_exists_return_value(sc, num, sc->divide_symbol, set_plist_1(sc, num));
      wrong_type_error_nr(sc, sc->divide_symbol, 1, num, a_number_string);
    }
  return(NULL);
}

static s7_pointer divide_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* splitting out real/real here saves very little */
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	  /* -------- integer x -------- */
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  if (integer(x) == 1)  /* mainly to handle (/ 1 -9223372036854775808) correctly! */
	    return(invert_p_p(sc, y));
	  return(make_ratio(sc, integer(x), integer(y))); /* make_ratio calls gcd */
	case T_RATIO:
#if Have_Overflow_Checks
	  {
	    s7_int dn;
	    if (multiply_overflow(integer(x), denominator(y), &dn))
#if With_Gmp
	      {
		mpq_set_si(sc->mpq_1, integer(x), 1);
		mpq_set_si(sc->mpq_2, numerator(y), denominator(y));
		mpq_div(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	      }
#else
              {
		if (With_Warnings)
		  s7_warn(sc, s7_Warn_Bufsize, "integer / ratio overflow: (/ %" ld64 " %" ld64 "/%" ld64 ")\n", integer(x), numerator(y), denominator(y));
  	        return(make_real(sc, integer(x) * inverted_fraction(y)));
	      }
#endif
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, dn, numerator(y)));
	  }
#else
	  return(make_ratio_with_div_check(sc, sc->divide_symbol, integer(x) * denominator(y), numerator(y)));
#endif
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  if (is_inf(real(y))) return(real_zero);
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
#if With_Gmp
	  if ((s7_int_abs(integer(x))) > Quotient_Int_Limit)
	    {
	      mpfr_set_si(sc->mpfr_1, integer(x), Mpfr_Rndn);
	      mpfr_set_d(sc->mpfr_2, real(y), Mpfr_Rndn);
	      mpfr_div(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	      return(mpfr_to_big_real(sc, sc->mpfr_1));
	    }
#endif
	  return(make_real(sc, (s7_double)(integer(x)) / real(y)));
	case T_COMPLEX:
	  {
	    s7_double r1 = (s7_double)integer(x), r2 = real_part(y), i2 = imag_part(y);
	    s7_double den = 1.0 / (r2 * r2 + i2 * i2);
	    /* we could avoid the squaring (see Knuth II p613 16), not a big deal: (/ 1.0e308+1.0e308i 2.0e308+2.0e308i) => nan, (gmp case is ok here) */
	    return(make_complex(sc, r1 * r2 * den, -(r1 * i2 * den)));
	  }

#if With_Gmp
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_set_den(sc->mpq_1, big_integer(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, integer(x), 1);
	  mpq_div(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_si_div(sc->mpfr_1, integer(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_si(sc->mpc_1, integer(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1)); /* x might be 0? */
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
      break;

      /* -------- ratio x -------- */
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
#if Have_Overflow_Checks
	  {
	    s7_int dn;
	    if (multiply_overflow(denominator(x), integer(y), &dn))
#if With_Gmp
	      {
		mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
		mpq_set_si(sc->mpq_2, integer(y), 1);
		mpq_div(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
              {
		if (With_Warnings)
		  s7_warn(sc, s7_Warn_Bufsize, "ratio / integer overflow: (/ %" ld64 "/%" ld64 " %" ld64 ")\n", numerator(x), denominator(x), integer(y));
	        return(make_real(sc, (long_double)numerator(x) / ((long_double)denominator(x) * (long_double)integer(y))));
	      }
#endif
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, numerator(x), dn));
	  }
#else
	  return(make_ratio_with_div_check(sc, sc->divide_symbol, numerator(x), denominator(x) * integer(y)));
#endif
	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    parcel_out_fractions(x, y);
	    if (d1 == d2)
	      return(make_ratio_with_div_check(sc, sc->divide_symbol, n1, n2));
#if Have_Overflow_Checks
	    if ((multiply_overflow(n1, d2, &n1)) ||
		(multiply_overflow(n2, d1, &d1)))
	      {
#if With_Gmp
		mpq_set_si(sc->mpq_1, numerator(x), denominator(x)); /* not n1 and d1! they are garbage here */
		mpq_set_si(sc->mpq_2, n2, d2);
		mpq_div(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_rational(sc, sc->mpq_1));
#else
		s7_double r1, r2;
		if (With_Warnings)
		  s7_warn(sc, s7_Warn_Bufsize, "ratio / ratio overflow: (/ %" ld64 "/%" ld64 " %" ld64 "/%" ld64 ")\n", numerator(x), denominator(x), numerator(y), denominator(y));
		r1 = fraction(x);
		r2 = inverted_fraction(y);
		return(make_real(sc, r1 * r2));
#endif
	      }
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, n1, d1));
#else
	    return(make_ratio_with_div_check(sc, sc->divide_symbol, n1 * d2, n2 * d1));
#endif
	  }
	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  return(make_real(sc, (s7_double)fraction(x) / real(y)));
	case T_COMPLEX:
	  {
	    s7_double rx = fraction(x), r2 = real_part(y), i2 = imag_part(y);
	    s7_double den = 1.0 / (r2 * r2 + i2 * i2);
	    return(make_complex(sc, rx * r2 * den, -rx * i2 * den)); /* not unchecked: (/ 3/4 -inf.0+i) */
	  }

#if With_Gmp
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_set_si(sc->mpq_2, numerator(x), denominator(x));
	  mpq_div(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpq_div(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpfr_set_q(sc->mpfr_1, sc->mpq_1, Mpfr_Rndn);
	  mpfr_div(sc->mpfr_1, sc->mpfr_1, big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}

      /* -------- real x -------- */
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  if (is_NaN(real(x))) return(x); /* what is (/ +nan.0 0)? */
	  if (is_inf(real(x)))
	    return((real(x) > 0.0) ? ((integer(y) > 0) ? real_infinity : real_minus_infinity) : ((integer(y) > 0) ? real_minus_infinity : real_infinity));
	  return(make_real(sc, (long_double)real(x) / (long_double)integer(y)));
	case T_RATIO:
	  if (is_NaN(real(x))) return(x);
	  if (is_inf(real(x)))
	    return((real(x) > 0) ? ((numerator(y) > 0) ? real_infinity : real_minus_infinity) : ((numerator(y) > 0) ? real_minus_infinity : real_infinity));
	  return(make_real(sc, real(x) * inverted_fraction(y)));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  if (is_NaN(real(x))) return(x);
	  if (is_inf(real(y)))
	    return((is_inf(real(x))) ? real_NaN : real_zero);
	  return(make_real(sc, real(x) / real(y)));
	case T_COMPLEX:
	  {
	    s7_double den, r2, i2;
	    if (is_NaN(real(x))) return(complex_NaN);
	    r2 = real_part(y);
	    i2 = imag_part(y);
	    if ((is_NaN(r2)) || (is_inf(r2))) return(complex_NaN);
	    if ((is_NaN(i2)) || (is_inf(i2))) return(complex_NaN);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(make_complex(sc, real(x) * r2 * den, -real(x) * i2 * den));
	  }

#if With_Gmp
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_set_z(sc->mpfr_1, big_integer(y), Mpfr_Rndn);
	  mpfr_d_div(sc->mpfr_1, real(x), sc->mpfr_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  mpfr_div_q(sc->mpfr_1, sc->mpfr_1, big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_d_div(sc->mpfr_1, real(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((is_NaN(real(x))) || (!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real(x), 0.0, MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}

      /* -------- complex x -------- */
    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  {
	    s7_double r1;
	    if (integer(y) == 0)
	      division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	    r1 = (long_double)1.0 / (long_double)integer(y);
	    return(make_complex(sc, real_part(x) * r1, imag_part(x) * r1));
	  }
	case T_RATIO:
	  {
	    s7_double frac = inverted_fraction(y);
	    return(make_complex(sc, real_part(x) * frac, imag_part(x) * frac));
	  }
	case T_REAL:
	  {
	    s7_double r1;
	    if (real(y) == 0.0)
	      division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	    r1 = 1.0 / real(y);
	    return(make_complex(sc, real_part(x) * r1, imag_part(x) * r1)); /* (/ 0.0+1.0i +inf.0) */
	  }
	case T_COMPLEX:
	  {
	    s7_double r1 = real_part(x), r2, i1, i2, den;
	    if (is_NaN(r1)) return(x);
	    i1 = imag_part(x);
	    if (is_NaN(i1)) return(x);
	    r2 = real_part(y);
	    if (is_NaN(r2)) return(y);
	    if (is_inf(r2)) return(complex_NaN);
	    i2 = imag_part(y);
	    if (is_NaN(i2)) return(y);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(make_complex(sc, (r1 * r2 + i1 * i2) * den, (r2 * i1 - r1 * i2) * den));
	  }

#if With_Gmp
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_set_q(sc->mpc_2, big_ratio(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_div_fr(sc->mpc_1, sc->mpc_1, big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
#endif
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpz_set_si(sc->mpz_1, integer(y));
	  mpq_set_num(sc->mpq_1, big_integer(x));
	  mpq_set_den(sc->mpq_1, sc->mpz_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_z(sc->mpq_2, big_integer(x));
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y)); /* can't invert here, mpq den=unsigned */
	  mpq_div(sc->mpq_1, sc->mpq_2, sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  mpfr_div_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))) return(y);
	  if ((is_inf(real_part(y))) || (is_inf(imag_part(y)))) return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_set_z(sc->mpc_2, big_integer(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_2, sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpq_set_num(sc->mpq_1, big_integer(x));
	  mpq_set_den(sc->mpq_1, big_integer(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, 0, 1);
	  mpq_set_num(sc->mpq_1, big_integer(x));
	  mpq_div(sc->mpq_1, sc->mpq_1, big_ratio(y));
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  mpfr_div(sc->mpfr_1, sc->mpfr_1, big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_z(sc->mpc_1, big_integer(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpq_set_si(sc->mpq_1, integer(y), 1);
	  mpq_div(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpq_div(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_rational(sc, sc->mpq_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  mpfr_div_d(sc->mpfr_1, sc->mpfr_1, real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))) return(y);
	  if ((is_inf(real_part(y))) || (is_inf(imag_part(y)))) return(complex_NaN);
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_set_d_d(sc->mpc_2, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, sc->mpc_2, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpq_set_z(sc->mpq_1, big_integer(y));
	  mpq_div(sc->mpq_1, big_ratio(x), sc->mpq_1);
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_RATIO:
	  mpq_div(sc->mpq_1, big_ratio(x), big_ratio(y));
	  return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  mpfr_div(sc->mpfr_1, sc->mpfr_1, big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_set_q(sc->mpc_1, big_ratio(x), MPC_RNDNN);
	  mpc_div(sc->mpc_1, sc->mpc_1, big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_div_si(sc->mpfr_1, big_real(x), integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpfr_div_q(sc->mpfr_1, big_real(x), sc->mpq_1, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_div_d(sc->mpfr_1, big_real(x), real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))) return(y);
	  if ((is_inf(real_part(y))) || (is_inf(imag_part(y)))) return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_fr_div(sc->mpc_1, big_real(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_div_z(sc->mpfr_1, big_real(x), big_integer(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_RATIO:
	  mpfr_div_q(sc->mpfr_1, big_real(x), big_ratio(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpfr_div(sc->mpfr_1, big_real(x), big_real(y), Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_fr_div(sc->mpc_1, big_real(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpc_set_si(sc->mpc_1, integer(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  mpc_set_q(sc->mpc_1, sc->mpq_1, MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_REAL:
	  /* if (is_NaN(real(y))) return(y); */
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpc_set_d_d(sc->mpc_1, real(y), 0.0, MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y)))) return(y);
	  if ((is_inf(real_part(y))) || (is_inf(imag_part(y)))) return(complex_NaN);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_INTEGER:
	  if (mpz_cmp_ui(big_integer(y), 0) == 0)
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpc_set_z(sc->mpc_1, big_integer(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_RATIO:
	  mpc_set_q(sc->mpc_1, big_ratio(y), MPC_RNDNN);
	  mpc_div(sc->mpc_1, big_complex(x), sc->mpc_1, MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_REAL:
	  if (mpfr_zero_p(big_real(y)))
	    division_by_zero_error_2_nr(sc, sc->divide_symbol, x, y);
	  mpc_div_fr(sc->mpc_1, big_complex(x), big_real(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	case T_BIG_COMPLEX:
	  if ((!mpfr_number_p(mpc_realref(big_complex(y)))) || (!mpfr_number_p(mpc_imagref(big_complex(y)))))
	    return(complex_NaN);
	  mpc_div(sc->mpc_1, big_complex(x), big_complex(y), MPC_RNDNN);
	  return(mpc_to_number(sc, sc->mpc_1));
	default:
	  return(method_or_bust_with_type_and_loc_pp(sc, y, sc->divide_symbol, x, y, a_number_string, 2));
	}
#endif

    default: /* x is not a built-in number */
      return(method_or_bust_pp(sc, x, sc->divide_symbol, x, y, a_number_string, 1)); /* not args here! y = apply * to cdr(args) */
    }
  return(NULL); /* make the compiler happy */
}

static s7_pointer g_divide(s7_scheme *sc, s7_pointer args)
{
  #define H_divide "(/ x1 ...) divides its first argument by the rest, or inverts the first if there is only one argument"
  #define Q_divide sc->pcl_n

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(p))            /* (/ x) */
    {
      if (!is_number(x))
	return(method_or_bust_p(sc, x, sc->divide_symbol, a_number_string));
      return(invert_p_p(sc, x));
    }
  for (sc->error_argnum = 0; is_pair(p); p = cdr(p), sc->error_argnum++)
    x = divide_p_pp(sc, x, car(p));
  sc->error_argnum = 0;
  return(x);
}

static s7_pointer g_invert_1(s7_scheme *sc, s7_pointer args) {return(invert_p_p(sc, car(args)));}
static s7_pointer g_divide_2(s7_scheme *sc, s7_pointer args) {return(divide_p_pp(sc, car(args), cadr(args)));}

static s7_pointer g_divide_by_2(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer num = car(args);
  if (is_t_integer(num))
    {
      s7_int i = integer(num);
      if (i & 1)
	{
	  s7_pointer x;
	  new_cell(sc, x, T_RATIO);
	  set_numerator(x, i);
	  set_denominator(x, 2);
	  return(x);
	}
      return(make_integer(sc, i >> 1));
    }
  switch (type(num))
    {
    case T_RATIO:
#if Have_Overflow_Checks
      {
	s7_int dn;
	if (multiply_overflow(denominator(num), 2, &dn))
	  {
	    if ((numerator(num) & 1) == 1)
#if With_Gmp
	      {
		mpq_set_si(sc->mpq_1, numerator(num), denominator(num));
		mpq_set_si(sc->mpq_2, 1, 2);
		mpq_mul(sc->mpq_1, sc->mpq_1, sc->mpq_2);
		return(mpq_to_rational(sc, sc->mpq_1));
	      }
#else
	      {
		if (With_Warnings)
		  s7_warn(sc, s7_Warn_Bufsize, "ratio / 2 overflow: (/ %" ld64 "/%" ld64 " 2)\n", numerator(num), denominator(num));
	        return(make_real(sc, ((long_double)numerator(num) * 0.5) / (long_double)denominator(num)));
	      }
#endif
	    return(make_ratio(sc, numerator(num) / 2, denominator(num)));
	  }
	return(make_ratio_with_div_check(sc, sc->divide_symbol, numerator(num), dn));
      }
#else
      return(make_ratio(sc, numerator(num), denominator(num) * 2));
#endif
    case T_REAL:    return(make_real(sc, real(num) * 0.5));
    case T_COMPLEX: return(make_complex_not_0i(sc, real_part(num) * 0.5, imag_part(num) * 0.5));

#if With_Gmp
    case T_BIG_INTEGER:
      mpq_set_z(sc->mpq_1, big_integer(num));
      mpz_mul_ui(mpq_denref(sc->mpq_1), mpq_denref(sc->mpq_1), 2);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    case T_BIG_RATIO:
      mpq_set_si(sc->mpq_1, 2, 1);
      mpq_div(sc->mpq_1, big_ratio(num), sc->mpq_1);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    case T_BIG_REAL:
      mpfr_div_si(sc->mpfr_1, big_real(num), 2, Mpfr_Rndn);
      return(mpfr_to_big_real(sc, sc->mpfr_1));
    case T_BIG_COMPLEX:
      mpc_set_si(sc->mpc_1, 2, MPC_RNDNN);
      mpc_div(sc->mpc_1, big_complex(num), sc->mpc_1, MPC_RNDNN);
      return(mpc_to_number(sc, sc->mpc_1));
#endif
    default:
      return(method_or_bust_pp(sc, num, sc->divide_symbol, num, int_two, a_number_string, 1));
    }
}

static s7_pointer g_invert_x(s7_scheme *sc, s7_pointer args)
{
  /* (/ 1.0 x) */
  const s7_pointer x = cadr(args);
  if (is_t_real(x))
    {
      s7_double rl = real(x);
      if (rl == 0.0)
	division_by_zero_error_2_nr(sc, sc->divide_symbol, car(args), x);
      return((is_NaN(rl)) ? x : make_real(sc, 1.0 / rl));
    }
  return(divide_p_pp(sc, car(args), x));
}

static s7_double divide_d_7d(s7_scheme *sc, s7_double x)
{
  if (x == 0.0) division_by_zero_error_1_nr(sc, sc->divide_symbol, real_zero);
  return(1.0 / x);
}

static s7_double divide_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  if (x2 == 0.0) division_by_zero_error_1_nr(sc, sc->divide_symbol, real_zero);
  return(x1 / x2);
}

static s7_pointer divide_p_ii(s7_scheme *sc, s7_int x, s7_int y) {return(make_ratio_with_div_check(sc, sc->divide_symbol, x, y));}
static s7_pointer divide_p_i(s7_scheme *sc, s7_int x) {return(make_ratio_with_div_check(sc, sc->divide_symbol, 1, x));}

static s7_pointer divide_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  if (args == 1) return(sc->invert_1);
  if (args == 2)
    {
      const s7_pointer arg1 = cadr(expr), arg2 = caddr(expr);
      if ((is_t_real(arg1)) && (real(arg1) == 1.0)) return(sc->invert_x);
      if ((is_pair(arg1)) && (has_fn(arg1)))
	{
	  if (fn_proc(arg1) == g_multiply_2) set_fn_direct(arg1, g_multiply_2_wrapped);
	  else if (fn_proc(arg1) == g_multiply_3) set_fn_direct(arg1, g_multiply_3_wrapped);
	}
      if ((is_pair(arg2)) && (has_fn(arg2)) && (fn_proc(arg2) == g_multiply_2)) set_fn_direct(arg2, g_multiply_2_wrapped);
      return(((is_t_integer(arg2)) && (integer(arg2) == 2)) ? sc->divide_by_2 : sc->divide_2);
    }
  return(func);
}


/* -------------------------------- quotient -------------------------------- */
static inline s7_int quotient_i_7ii(s7_scheme *sc, s7_int x, s7_int y)
{
  if ((y > 0) || (y < -1)) return(x / y);
  if (y == 0)
    division_by_zero_error_2_nr(sc, sc->quotient_symbol, wrap_integer(sc, x), int_zero);
  if (x == s7_Int64_Min)   /* (quotient most-negative-fixnum -1) */
    sole_arg_out_of_range_error_nr(sc, sc->quotient_symbol, set_elist_2(sc, leastfix, minus_one), it_is_too_large_string);
  return(-x); /* (quotient x -1) */
}

#if !With_Gmp
static s7_pointer s7_truncate(s7_scheme *sc, s7_pointer caller, s7_double xf)   /* can't use "truncate" -- it's in unistd.h */
{
  if (fabs(xf) > Quotient_Float_Limit)
    sole_arg_out_of_range_error_nr(sc, caller, wrap_real(sc, xf), it_is_too_large_string);
  return(make_integer(sc, (xf > 0.0) ? (s7_int)floor(xf) : (s7_int)ceil(xf)));
}

static s7_int c_quo_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_double xf;
  if (y == 0.0)
    division_by_zero_error_2_nr(sc, sc->quotient_symbol, wrap_real(sc, x), real_zero);
  if ((is_inf(y)) || (is_NaN(y))) /* here we can't return Nan so I guess we should signal an error */
    wrong_type_error_nr(sc, sc->quotient_symbol, 2, wrap_real(sc, y), a_normal_real_string);
  xf = x / y;
  if (fabs(xf) > Quotient_Float_Limit)
    sole_arg_out_of_range_error_nr(sc, sc->quotient_symbol, wrap_real(sc, xf), it_is_too_large_string);
  return((xf > 0.0) ? (s7_int)floor(xf) : (s7_int)ceil(xf));
}
#endif

static s7_int quotient_i_ii_unchecked(s7_int i1, s7_int i2) {return(i1 / i2);} /* i2 > 0 */

static s7_pointer quotient_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if With_Gmp
  if ((is_real(x)) && (is_real(y)))
    {
      if (is_zero(y))
	division_by_zero_error_2_nr(sc, sc->quotient_symbol, x, y);
      if ((s7_is_integer(x)) && (s7_is_integer(y)))
	{
	  if (is_t_integer(x)) mpz_set_si(sc->mpz_1, integer(x)); else mpz_set(sc->mpz_1, big_integer(x));
	  if (is_t_integer(y)) mpz_set_si(sc->mpz_2, integer(y)); else mpz_set(sc->mpz_2, big_integer(y));
	  mpz_tdiv_q(sc->mpz_1, sc->mpz_1, sc->mpz_2);
	}
      else
	if ((!is_rational(x)) || (!is_rational(y)))
	  {
	    if (any_real_to_mpfr(sc, x, sc->mpfr_1)) return(make_nan_with_payload(sc, __LINE__));
	    if (any_real_to_mpfr(sc, y, sc->mpfr_2)) return(make_nan_with_payload(sc, __LINE__));
	    mpfr_div(sc->mpfr_3, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	    mpfr_get_z(sc->mpz_1, sc->mpfr_3, MPFR_RNDZ);
	  }
	else
	  {
	    any_rational_to_mpq(sc, x, sc->mpq_1);
	    any_rational_to_mpq(sc, y, sc->mpq_2);
	    mpq_div(sc->mpq_3, sc->mpq_1, sc->mpq_2);
	    mpz_tdiv_q(sc->mpz_1, mpq_numref(sc->mpq_3), mpq_denref(sc->mpq_3));
	  }
      return(mpz_to_integer(sc, sc->mpz_1));
    }
  return(method_or_bust_pp(sc, (is_real(x)) ? y : x, sc->quotient_symbol, x, y, sc->type_names[T_REAL], (is_real(x)) ? 2 : 1));
#else

  s7_int d1, d2, n1, n2;
  if ((is_t_integer(x)) && (is_t_integer(y)))
    return(make_integer(sc, quotient_i_7ii(sc, integer(x), integer(y))));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, quotient_i_7ii(sc, integer(x), integer(y))));
	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  /* (quotient -9223372036854775808 -1/9223372036854775807): arithmetic exception in the no-overflow-checks case */
	  goto Ratio_Quo_Ratio;
	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->quotient_symbol, x, y);
	  if (is_inf(real(y))) return(make_nan_with_payload(sc, __LINE__));
	  if (is_NaN(real(y))) return(y);
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)integer(x) / real(y))); /* s7_truncate returns an integer */
	default:
	  return(method_or_bust_pp(sc, y, sc->quotient_symbol, x, y, sc->type_names[T_REAL], 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->quotient_symbol, x, y);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);
	  d2 = 1;
	  goto Ratio_Quo_Ratio;
	  /* this can lose:
	   *   (quotient 1 2305843009213693952/4611686018427387903) -> 2, not 1
	   *   (quotient 21053343141/6701487259 3587785776203/1142027682075) -> 1, not 0
	   */
	case T_RATIO:
	  parcel_out_fractions(x, y);
	Ratio_Quo_Ratio:
	  if (d1 == d2)
	    return(make_integer(sc, n1 / n2));              /* (quotient 3/9223372036854775807 1/9223372036854775807) */
	  if (n1 == n2)
	    return(make_integer(sc, d2 / d1));              /* (quotient 9223372036854775807/2 9223372036854775807/8) */
#if Have_Overflow_Checks
	  {
	    s7_int n1d2, n2d1;
	    if ((multiply_overflow(n1, d2, &n1d2)) ||
		(multiply_overflow(n2, d1, &n2d1)))
	      return(s7_truncate(sc, sc->quotient_symbol, ((long_double)n1 / (long_double)n2) * ((long_double)d2 / (long_double)d1)));
	    return(make_integer(sc, n1d2 / n2d1));
	  }
#else
	  return(make_integer(sc, (n1 * d2) / (n2 * d1)));
#endif
	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->quotient_symbol, x, y);
	  if (is_inf(real(y))) return(make_nan_with_payload(sc, __LINE__));
	  if (is_NaN(real(y))) return(y);
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)fraction(x) / real(y)));
	default:
	  return(method_or_bust_pp(sc, y, sc->quotient_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    case T_REAL:
      if (((is_inf(real(x))) || (is_NaN(real(x)))) && (is_real(y)))
	return(make_nan_with_payload(sc, __LINE__));
      /* if infs allowed we need to return infs/nans, else:
       *    (quotient inf.0 1e-309) -> -9223372036854775808
       *    (quotient inf.0 inf.0) -> -9223372036854775808
       */
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->quotient_symbol, x, y);
	  return(s7_truncate(sc, sc->quotient_symbol, (long_double)real(x) / (long_double)integer(y)));

	case T_RATIO: return(s7_truncate(sc, sc->quotient_symbol, real(x) / (s7_double)fraction(y)));
	case T_REAL:  return(make_integer(sc, c_quo_dbl(sc, real(x), real(y)))); /* c_quo_dbl returns an integer */
	default:      return(method_or_bust_pp(sc, y, sc->quotient_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    default:
      return(method_or_bust_pp(sc, x, sc->quotient_symbol, x, y, sc->type_names[T_REAL], 2));
    }
#endif
}

static s7_pointer quotient_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if ((is_t_integer(x)) && ((y > 0) || (y < -1))) return(make_integer(sc, integer(x) / y));
  return(quotient_p_pp(sc, x, wrap_integer(sc, y)));
}

static s7_pointer g_quotient(s7_scheme *sc, s7_pointer args)
{
  #define H_quotient "(quotient x1 x2) returns the integer quotient of x1 and x2; (quotient 4 3) = 1"
  #define Q_quotient sc->pcl_r
  /* sig was '(integer? ...) but quotient can return NaN */
  /* (define (quo x1 x2) (truncate (/ x1 x2))) ; slib */
  return(quotient_p_pp(sc, car(args), cadr(args)));
}


/* -------------------------------- remainder -------------------------------- */
#if With_Gmp
static s7_pointer big_mod_or_rem(s7_scheme *sc, s7_pointer x, s7_pointer y, bool use_floor)
{
  if ((is_real(x)) && (is_real(y)))
    {
      if ((s7_is_integer(x)) && (s7_is_integer(y)))
	{
	  if (is_t_integer(x)) mpz_set_si(sc->mpz_1, integer(x)); else mpz_set(sc->mpz_1, big_integer(x));
	  if (is_t_integer(y)) mpz_set_si(sc->mpz_2, integer(y)); else mpz_set(sc->mpz_2, big_integer(y));
	  if (use_floor)
	    mpz_fdiv_q(sc->mpz_3, sc->mpz_1, sc->mpz_2);
	  else mpz_tdiv_q(sc->mpz_3, sc->mpz_1, sc->mpz_2);
	  mpz_mul(sc->mpz_3, sc->mpz_3, sc->mpz_2);
	  mpz_sub(sc->mpz_1, sc->mpz_1, sc->mpz_3);
	  return(mpz_to_integer(sc, sc->mpz_1));
	}
      if ((!is_rational(x)) || (!is_rational(y)))
	{
	  any_real_to_mpfr(sc, x, sc->mpfr_1);
	  any_real_to_mpfr(sc, y, sc->mpfr_2);
	  mpfr_div(sc->mpfr_3, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	  if (use_floor)
	    mpfr_get_z(sc->mpz_1, sc->mpfr_3, MPFR_RNDD);
	  else mpfr_get_z(sc->mpz_1, sc->mpfr_3, MPFR_RNDZ);
	  mpfr_mul_z(sc->mpfr_2, sc->mpfr_2, sc->mpz_1, Mpfr_Rndn);
	  mpfr_sub(sc->mpfr_1, sc->mpfr_1, sc->mpfr_2, Mpfr_Rndn);
	  return(mpfr_to_big_real(sc, sc->mpfr_1));
	}
      any_rational_to_mpq(sc, x, sc->mpq_1);
      any_rational_to_mpq(sc, y, sc->mpq_2);
      mpq_div(sc->mpq_3, sc->mpq_1, sc->mpq_2);
      if (use_floor)
	mpz_fdiv_q(sc->mpz_1, mpq_numref(sc->mpq_3), mpq_denref(sc->mpq_3));
      else mpz_tdiv_q(sc->mpz_1, mpq_numref(sc->mpq_3), mpq_denref(sc->mpq_3));
      mpz_mul(mpq_numref(sc->mpq_2), sc->mpz_1, mpq_numref(sc->mpq_2));
      mpq_sub(sc->mpq_1, sc->mpq_1, sc->mpq_2);
      return(mpq_to_canonicalized_rational(sc, sc->mpq_1));
    }
  return(method_or_bust_pp(sc, (is_real(x)) ? y : x, (use_floor) ? sc->modulo_symbol : sc->remainder_symbol, x, y, sc->type_names[T_REAL], (is_real(x)) ? 2 : 1));
}
#endif

#define Remainder_Float_Limit 1e13

static inline s7_int remainder_i_7ii(s7_scheme *sc, s7_int x, s7_int y)
{
  if ((y > 1) || (y < -1)) return(x % y); /* avoid floating exception if (remainder -9223372036854775808 -1)! */
  if (y == 0)
    division_by_zero_error_2_nr(sc, sc->remainder_symbol, wrap_integer(sc, x), int_zero);
  return(0);
}

static s7_double c_rem_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_int quo;
  s7_double pre_quo;
  if (is_NaN(y)) return(y);
  if (is_inf(y)) return(Nan);
  pre_quo = x / y;
  if (fabs(pre_quo) > Remainder_Float_Limit)
    sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, wrap_real(sc, x), wrap_real(sc, y)), it_is_too_large_string);
  quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
  return(x - (y * quo));
}

static s7_int remainder_i_ii_unchecked(s7_int i1, s7_int i2) {return(i1 % i2);} /* i2 > 1 */
static s7_double remainder_d_7dd(s7_scheme *sc, s7_double x, s7_double y)
{
  if (y == 0.0)
    division_by_zero_error_2_nr(sc, sc->remainder_symbol, wrap_real(sc, x), real_zero);
  if (is_NaN(x)) return(x);
  if (is_inf(x)) return(Nan); /* match remainder_p_pp */
  return(c_rem_dbl(sc, x, y));
}

static s7_pointer remainder_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if With_Gmp
  if (is_zero(y))
    division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
  return(big_mod_or_rem(sc, x, y, false));
#else
  s7_int quo, d1, d2, n1, n2;
  s7_double pre_quo;

  if ((is_t_integer(x)) && (is_t_integer(y)))
    return(make_integer(sc, remainder_i_7ii(sc, integer(x), integer(y))));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, remainder_i_7ii(sc, integer(x), integer(y))));
	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto Ratio_Rem_Ratio;
	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
	  if (is_inf(real(y))) return(make_nan_with_payload(sc, __LINE__));
	  if (is_NaN(real(y))) return(y);
	  pre_quo = (long_double)integer(x) / (long_double)real(y);
	  if (fabs(pre_quo) > Remainder_Float_Limit)
	    sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, x, y), it_is_too_large_string);
	  quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	  return(make_real(sc, integer(x) - real(y) * quo));
	default:
	  return(method_or_bust_pp(sc, y, sc->remainder_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  n2 = integer(y);
 	  if (n2 == 0)
 	    division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  d2 = 1;
	  goto Ratio_Rem_Ratio;
	case T_RATIO:
	  parcel_out_fractions(x, y);
	Ratio_Rem_Ratio:
	  if (d1 == d2)
	    quo = (s7_int)(n1 / n2);
	  else
	    {
	      if (n1 == n2)
		quo = (s7_int)(d2 / d1);
	      else
		{
#if Have_Overflow_Checks
		  s7_int n1d2, n2d1;
		  if ((multiply_overflow(n1, d2, &n1d2)) ||
		      (multiply_overflow(n2, d1, &n2d1)))
		    {
		      pre_quo = ((long_double)n1 / (long_double)n2) * ((long_double)d2 / (long_double)d1);
		      if (fabs(pre_quo) > Remainder_Float_Limit)
			sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, x, y), it_is_too_large_string);
		      quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
		    }
		  else quo = n1d2 / n2d1;
#else
		  quo = (n1 * d2) / (n2 * d1);
#endif
		}}
	  if (quo == 0)
	    return(x);
#if Have_Overflow_Checks
	  {
	    s7_int dn, nq;
	    if (!multiply_overflow(n2, quo, &nq))
	      {
		if ((d1 == d2) &&
		    (!subtract_overflow(n1, nq, &dn)))
		  return(make_ratio_with_div_check(sc, sc->remainder_symbol, dn, d1));

		if ((!multiply_overflow(n1, d2, &dn)) &&
		    (!multiply_overflow(nq, d1, &nq)) &&
		    (!subtract_overflow(dn, nq, &nq)) &&
		    (!multiply_overflow(d1, d2, &d1)))
		  return(make_ratio_with_div_check(sc, sc->remainder_symbol, nq, d1));
	      }}
#else
	  if (d1 == d2)
	    return(make_ratio_with_div_check(sc, sc->remainder_symbol, n1 - n2 * quo, d1));

	  return(make_ratio_with_div_check(sc, sc->remainder_symbol, n1 * d2 - n2 * d1 * quo, d1 * d2));
#endif
	  sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, x, y), intermediate_too_large_string);
	case T_REAL:
	  {
	    s7_double frac;
	    if (real(y) == 0.0)
	      division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
	    if (is_inf(real(y))) return(make_nan_with_payload(sc, __LINE__));
	    if (is_NaN(real(y))) return(y);
	    if (s7_int_abs(numerator(x)) > Quotient_Int_Limit)
	      return(subtract_p_pp(sc, x, multiply_p_pp_wrapped(sc, y, quotient_p_pp(sc, x, y))));
	    frac = (s7_double)fraction(x);
	    pre_quo = frac / real(y);
	    if (fabs(pre_quo) > Remainder_Float_Limit)
	      sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, x, y), it_is_too_large_string);
	    quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	    return(make_real(sc, frac - real(y) * quo));
	  }
	default:
	  return(method_or_bust_pp(sc, y, sc->remainder_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    case T_REAL:
      if (((is_inf(real(x))) || (is_NaN(real(x)))) && (is_real(y)))
	{
	  if (is_zero(y))
	    division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
	  return(make_nan_with_payload(sc, __LINE__));
	}
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
	  /* actually here (and elsewhere) if y > Int64_To_Double_Limit, the result is probably wrong */
	  pre_quo = (long_double)real(x) / (long_double)integer(y);
	  if (fabs(pre_quo) > Remainder_Float_Limit)
	    sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, x, y), it_is_too_large_string);
	  quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	  return(make_real(sc, real(x) - integer(y) * quo));
	  /* but... (remainder 1e+18 9223372036854775807) -> 1e+18 */
	case T_RATIO:
	  if (s7_int_abs(numerator(y)) > Quotient_Int_Limit)
	    return(subtract_p_pp(sc, x, multiply_p_pp_wrapped(sc, y, quotient_p_pp(sc, x, y))));
	  {
	    s7_double frac = (s7_double)fraction(y);
	    pre_quo = real(x) / frac;
	    if (fabs(pre_quo) > Remainder_Float_Limit)
	      sole_arg_out_of_range_error_nr(sc, sc->remainder_symbol, set_elist_2(sc, x, y), it_is_too_large_string);
	    quo = (pre_quo > 0.0) ? (s7_int)floor(pre_quo) : (s7_int)ceil(pre_quo);
	    return(make_real(sc, real(x) - frac * quo));
	  }
	case T_REAL:
	  if (real(y) == 0.0)
	    division_by_zero_error_2_nr(sc, sc->remainder_symbol, x, y);
	  return(make_real(sc, c_rem_dbl(sc, real(x), real(y))));
	  /* see under sin -- this calculation is completely bogus if "a" is large
	   * (quotient 1e22 (* 2 pi)) -> -9223372036854775808 but it should be 1591549430918953357688,
	   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22 -- the "remainder" is greater than the original argument!
	   * Clisp gives 0.0 here, as does sbcl, currently s7 throws an error (out-of-range).
	   */
	default:
	  return(method_or_bust_pp(sc, y, sc->remainder_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    default:
      return(method_or_bust_pp(sc, x, sc->remainder_symbol, x, y, sc->type_names[T_REAL], 1));
    }
#endif
}

static s7_pointer remainder_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if ((is_t_integer(x)) && ((y > 1) || (y < -1))) return(make_integer(sc, integer(x) % y));
  return(remainder_p_pp(sc, x, wrap_integer(sc, y)));
}

static s7_pointer g_remainder(s7_scheme *sc, s7_pointer args)
{
  #define H_remainder "(remainder x y) returns the remainder of x/y; (remainder 10 3) = 1"
  #define Q_remainder sc->pcl_r
  /* (define (rem x y) (- x (* y (quo x y)))) ; slib, if y is an integer (- x (truncate x y)), fractional part: (remainder x 1) */

  s7_pointer x = car(args), y = cadr(args);
  if ((is_t_integer(x)) && (is_t_integer(y)))
    return(make_integer(sc, remainder_i_7ii(sc, integer(x), integer(y))));
  return(remainder_p_pp(sc, x, y));
}


/* -------------------------------- modulo -------------------------------- */
static s7_int modulo_i_ii(s7_int x, s7_int y)
{
  s7_int z;
  if (y > 1)
    {
      z = x % y;
      return((z >= 0) ? z : z + y);
    }
  if (y < -1)
    {
      z = x % y;
      return((z > 0) ? z + y : z);
    }
  if (y == 0) return(x);     /* else arithmetic exception */
  return(0);
}

static s7_int modulo_i_ii_unchecked(s7_int i1, s7_int i2) /* here we know i2 > 1 */
{
  s7_int z = i1 % i2;
  return((z < 0) ? (z + i2) : z);
}

static s7_double modulo_d_7dd(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_double z;
  if (is_NaN(x)) return(x);
  if (is_NaN(y)) return(y);
  if ((is_inf(x)) || (is_inf(y))) return(Nan);
  if (y == 0.0) return(x);
  if (fabs(x) > 1e17)
    out_of_range_error_nr(sc, sc->modulo_symbol, int_one, wrap_real(sc, x), it_is_too_large_string);
  z = x / y;
  if ((z > 1e19) || (z < -1e19))
    sole_arg_out_of_range_error_nr(sc, sc->modulo_symbol,
			set_elist_3(sc, sc->divide_symbol, wrap_real(sc, x), wrap_real(sc, y)),
			intermediate_too_large_string);
  return(x - y * (s7_int)floor(z));
}

static s7_pointer modulo_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if With_Gmp
  /* as tricky as expt, so just use bignums; mpz_mod|_ui = mpz_fdiv_r_ui, but sign ignored -- probably not worth the code
   *   originally   subtract_p_pp(sc, x, multiply_p_pp(sc, y, floor_p_p(sc, divide_p_pp(sc, x, y))))
   *   quotient is                                            truncate_p_p(sc, divide_p_pp(sc, x, y))
   *   remainder is subtract_p_pp(sc, x, multiply_p_pp(sc, y, quotient_p_pp(sc, x, y)))
   */
  if (!is_zero(y)) return(big_mod_or_rem(sc, x, y, true));
  if (is_real(x)) return(x);
  return(method_or_bust_pp(sc, x, sc->modulo_symbol, x, y, sc->type_names[T_REAL], 1));
#else
  s7_double a, b;
  s7_int n1, n2, d1, d2;
  if ((is_t_integer(x)) && (is_t_integer(y))) /* this is nearly always the case */
    return(make_integer(sc, modulo_i_ii(integer(x), integer(y))));

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, modulo_i_ii(integer(x), integer(y))));
	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  if ((n1 == n2) && (d1 > d2)) return(x); /* signs match so this should be ok */
	  goto Ratio_Mod_Ratio;
	case T_REAL:
	  if ((integer(x) == s7_Int64_Min) || (s7_int_abs(integer(x)) > Quotient_Int_Limit))
	    out_of_range_error_nr(sc, sc->modulo_symbol, int_one, x, it_is_too_large_string);
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(make_nan_with_payload(sc, __LINE__));
	  a = (s7_double)integer(x);
	  goto Real_Mod;
	default:
	  return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);
	  if ((n2 > 0) && (n1 > 0) && (n2 > n1)) return(x);
	  if ((n2 < 0) && (n1 < 0) && (n2 < n1)) return(x);
	  if (n2 == s7_Int64_Min)
	    sole_arg_out_of_range_error_nr(sc, sc->modulo_symbol, set_elist_3(sc, sc->divide_symbol, x, y), intermediate_too_large_string);
	  /* the problem here is that (modulo 3/2 most-negative-fixnum)
	   * will segfault with signal SIGFPE, Arithmetic exception, so try to trap it.
	   */
	  if ((n1 == n2) && (d1 > 1)) return(x);
	  d2 = 1;
	  goto Ratio_Mod_Ratio;
	case T_RATIO:
	  parcel_out_fractions(x, y);
	  if (d1 == d2)
	    return(make_ratio_with_div_check(sc, sc->modulo_symbol, modulo_i_ii(n1, n2), d1));
	  if ((n1 == n2) && (d1 > d2)) return(x);
	Ratio_Mod_Ratio:
#if Have_Overflow_Checks
	  {
	    s7_int n2d1, n1d2, d1d2, fl;
	    if (!multiply_overflow(n2, d1, &n2d1))
	      {
		if ((n2d1 == 1) || (n2d1 == -1)) /* (modulo 100 -1/2) */
		  return(int_zero);
		if (!multiply_overflow(n1, d2, &n1d2))
		  {
		    fl = (s7_int)(n1d2 / n2d1);
		    if (((n1 < 0) && (n2 > 0)) ||
			((n1 > 0) && (n2 < 0)))
		      fl -= 1;
		    if (fl == 0)
		      return(x);
		    if ((!multiply_overflow(d1, d2, &d1d2)) &&
			(!multiply_overflow(fl, n2d1, &fl)) &&
			(!subtract_overflow(n1d2, fl, &fl)))
		      return(make_ratio_with_div_check(sc, sc->modulo_symbol, fl, d1d2));
		  }}}
#else
	  {
	    s7_int fl;
	    s7_int n1d2 = n1 * d2;
	    s7_int n2d1 = n2 * d1;
	    if ((n2d1 == 1) || (n2d1 == -1)) /* (modulo 100 -1/2) as above) */
	      return(int_zero);
	    /* can't use "floor" here (float->int ruins everything) */
	    fl = (s7_int)(n1d2 / n2d1);
	    if (((n1 < 0) && (n2 > 0)) ||
		((n1 > 0) && (n2 < 0)))
	      fl -= 1;
	    if (fl == 0)
	      return(x);
	    return(make_ratio_with_div_check(sc, sc->modulo_symbol, n1d2 - (n2d1 * fl), d1 * d2));
	  }
#endif
	  sole_arg_out_of_range_error_nr(sc, sc->modulo_symbol,
				     set_elist_3(sc, sc->divide_symbol, x, y),
				     intermediate_too_large_string);
	case T_REAL:
	  b = real(y);
	  if (is_inf(b)) return(make_nan_with_payload(sc, __LINE__));
	  if (fabs(b) > 1e17)
	    out_of_range_error_nr(sc, sc->modulo_symbol, int_two, y, it_is_too_large_string);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  a = fraction(x);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));
	default:
	  return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    case T_REAL:
      a = real(x);
      if (!is_real(y))
	return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, sc->type_names[T_REAL], 2));
      if (is_NaN(a)) return(x);
      if (is_inf(a)) return(make_nan_with_payload(sc, __LINE__)); /* not b */
      if (fabs(a) > 1e17)
	out_of_range_error_nr(sc, sc->modulo_symbol, int_one, x, it_is_too_large_string);

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  if ((integer(y) == s7_Int64_Min) || (s7_int_abs(integer(y)) > Quotient_Int_Limit))
	    out_of_range_error_nr(sc, sc->modulo_symbol, int_two, y, it_is_too_large_string);
	  b = (s7_double)integer(y);
	  goto Real_Mod;
	case T_RATIO:
	  b = fraction(y);
	  goto Real_Mod;
	case T_REAL:
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(make_nan_with_payload(sc, __LINE__));
	Real_Mod:
	  {
	    s7_double c = a / b;
	    if (fabs(c) > 1e19)
	      sole_arg_out_of_range_error_nr(sc, sc->modulo_symbol, set_elist_3(sc, sc->divide_symbol, x, y), intermediate_too_large_string);
	    return(make_real(sc, a - b * (s7_int)floor(c)));
	  }
	default:
	  return(method_or_bust_pp(sc, y, sc->modulo_symbol, x, y, sc->type_names[T_REAL], 2));
	}
    default:
      return(method_or_bust_pp(sc, x, sc->modulo_symbol, x, y, sc->type_names[T_REAL], 1));
    }
#endif
}

static s7_pointer modulo_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x)) return(make_integer(sc, modulo_i_ii(integer(x), y)));
  return(modulo_p_pp(sc, x, wrap_integer(sc, y)));
}

static s7_pointer g_modulo(s7_scheme *sc, s7_pointer args)
{
  #define H_modulo "(modulo x y) returns x mod y; (modulo 4 3) = 1.  The arguments can be real numbers."
  #define Q_modulo sc->pcl_r
  /* (define (mod x y) (- x (* y (floor (/ x y))))) from slib
   * (mod x 0) = x according to "Concrete Mathematics"
   */
  return(modulo_p_pp(sc, car(args), cadr(args)));
}


/* ---------------------------------------- max ---------------------------------------- */
static bool is_real_via_method_1(s7_scheme *sc, s7_pointer p)
{
  s7_pointer func = find_method_with_let(sc, p, sc->is_real_symbol);
  if (func != sc->undefined)
    return(is_true(sc, s7_apply_function(sc, func, set_plist_1(sc, p))));
  return(false);
}

#define is_real_via_method(sc, p) ((is_real(p)) || ((has_active_methods(sc, p)) && (is_real_via_method_1(sc, p))))

#define max_out_x(Sc, X, Y) method_or_bust_pp(Sc, X, Sc->max_symbol, X, Y, Sc->type_names[T_REAL], 1)
#define max_out_y(Sc, X, Y) method_or_bust_pp(Sc, Y, Sc->max_symbol, X, Y, Sc->type_names[T_REAL], 2)

static s7_pointer max_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* same basic code as lt_b_7_pp (or any relop) but max returns NaN if NaN encountered, and methods for < and max return
   *    different results, so it seems simpler to repeat the other code.
   */
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return((integer(x) < integer(y)) ? y : x);
      if (is_t_real(x))
	/* return(((is_NaN(real(x))) || (real(x) >= real(y))) ? x : y); */
	return(((real(x) >= real(y)) || (is_NaN(real(x)))) ? x : y);
      if (is_t_ratio(x))
	return((fraction(x) < fraction(y)) ? y : x);
#if With_Gmp
      if (is_t_big_integer(x))
	return((mpz_cmp(big_integer(x), big_integer(y)) < 0) ? y : x);
      if (is_t_big_ratio(x))
	return((mpq_cmp(big_ratio(x), big_ratio(y)) < 0) ? y : x);
      if (is_t_big_real(x))
	return(((mpfr_nan_p(big_real(x)) != 0) || (mpfr_greaterequal_p(big_real(x), big_real(y)))) ? x : y); /* ?? */
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:
	  return((integer(x) < fraction(y)) ? y : x);
	case T_REAL:
	  return(((integer(x) < real(y)) || (is_NaN(real(y)))) ? y : x);
#if With_Gmp
	case T_BIG_INTEGER:
	  return((mpz_cmp_si(big_integer(y), integer(x)) < 0) ? x : y);
	case T_BIG_RATIO:
	  return((mpq_cmp_si(big_ratio(y), integer(x), 1) < 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_si(big_real(y), integer(x)) < 0) ? x : y);
#endif
	default:
	  return(max_out_y(sc, x, y));
	}
      break;
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((fraction(x) < integer(y)) ? y : x);
	case T_REAL:
	  return(((fraction(x) < real(y)) || (is_NaN(real(y)))) ? y : x);
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpq_cmp_z(sc->mpq_1, big_integer(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) < 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpfr_cmp_q(big_real(y), sc->mpq_1) < 0) ? x : y);
#endif
	default:
	  return(max_out_y(sc, x, y));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(((real(x) >= integer(y)) || (is_NaN(real(x)))) ? x : y);
	case T_RATIO:
	  return((real(x) < fraction(y)) ? y : x);
#if With_Gmp
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return((mpfr_cmp_z(sc->mpfr_1, big_integer(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return((mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) < 0) ? y : x);
	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(x);
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_d(big_real(y), real(x)) < 0) ? x : y);
#endif
	default:
	  return(max_out_y(sc, x, y));
	}
      break;

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpz_cmp_si(big_integer(x), integer(y)) < 0) ? y : x);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return((mpq_cmp_z(sc->mpq_1, big_integer(x)) < 0) ? x : y);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_z(big_ratio(y), big_integer(x)) < 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_z(big_real(y), big_integer(x)) < 0) ? x : y);
	default:
	  return(max_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpq_cmp_si(big_ratio(x), integer(y), 1) < 0) ? y : x);
	case T_RATIO:
	  return((mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) < 0) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) < 0) ? y : x);
	case T_BIG_INTEGER:
	  return((mpq_cmp_z(big_ratio(x), big_integer(y)) < 0) ? y : x);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_q(big_real(y), big_ratio(x)) < 0) ? x : y);
	default:
	  return(max_out_y(sc, x, y));
	}
    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_si(big_real(x), integer(y)) < 0) ? y : x);
	case T_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return((mpfr_cmp_q(big_real(x), sc->mpq_1) < 0) ? y : x);
	case T_REAL:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  if (is_NaN(real(y))) return(y);
	  return((mpfr_cmp_d(big_real(x), real(y)) < 0) ? y : x);
	case T_BIG_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_z(big_real(x), big_integer(y)) < 0) ? y : x);
	case T_BIG_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_q(big_real(x), big_ratio(y)) < 0) ? y : x);
	default:
	  return(max_out_y(sc, x, y));
	}
#endif
    default:
      return(max_out_x(sc, x, y));
    }
  return(x);
}

static s7_pointer g_max(s7_scheme *sc, s7_pointer args)
{
  #define H_max "(max ...) returns the maximum of its arguments"
  #define Q_max sc->pcl_r

  s7_pointer x = car(args);
  if (is_null(cdr(args)))
    {
      if (is_real(x)) return(x);
      return(method_or_bust_p(sc, x, sc->max_symbol, sc->type_names[T_REAL]));
    }
  for (s7_pointer nums = cdr(args); is_pair(nums); nums = cdr(nums))
    x = max_p_pp(sc, x, car(nums));
  return(x);
}

static s7_pointer g_max_2(s7_scheme *sc, s7_pointer args) {return(max_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_max_3(s7_scheme *sc, s7_pointer args) {return(max_p_pp(sc, max_p_pp(sc, car(args), cadr(args)), caddr(args)));}

static s7_pointer max_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer unused_expr)
{
  return((args == 2) ? sc->max_2 : ((args == 3) ? sc->max_3 : func));
}

static s7_int max_i_ii(s7_int i1, s7_int i2) {return((i1 > i2) ? i1 : i2);}
static s7_int max_i_iii(s7_int i1, s7_int i2, s7_int i3) {return((i1 > i2) ? ((i1 > i3) ? i1 : i3) : ((i2 > i3) ? i2 : i3));}
static s7_double max_d_dd(s7_double x1, s7_double x2) {return(((x1 > x2) || (is_NaN(x1))) ? x1 : x2);}
static s7_double max_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(max_d_dd(x1, max_d_dd(x2, x3)));}
static s7_double max_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(max_d_dd(x1, max_d_ddd(x2, x3, x4)));}


/* ---------------------------------------- min ---------------------------------------- */
#define min_out_x(Sc, X, Y) method_or_bust_pp(Sc, X, Sc->min_symbol, X, Y, Sc->type_names[T_REAL], 1)
#define min_out_y(Sc, X, Y) method_or_bust_pp(Sc, Y, Sc->min_symbol, X, Y, Sc->type_names[T_REAL], 2)

static s7_pointer min_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return((integer(x) > integer(y)) ? y : x);
      if (is_t_real(x))
	return(((real(x) <= real(y)) || (is_NaN(real(x)))) ? x : y);
      if (is_t_ratio(x))
	return((fraction(x) > fraction(y)) ? y : x);
#if With_Gmp
      if (is_t_big_integer(x))
	return((mpz_cmp(big_integer(x), big_integer(y)) > 0) ? y : x);
      if (is_t_big_ratio(x))
	return((mpq_cmp(big_ratio(x), big_ratio(y)) > 0) ? y : x);
      if (is_t_big_real(x))
	return(((mpfr_nan_p(big_real(x)) != 0) || (mpfr_lessequal_p(big_real(x), big_real(y)))) ? x : y); /* ?? */
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:       return((integer(x) > fraction(y)) ? y : x);
	case T_REAL:
	  return(((integer(x) > real(y)) || (is_NaN(real(y)))) ? y : x);
#if With_Gmp
	case T_BIG_INTEGER: return((mpz_cmp_si(big_integer(y), integer(x)) > 0) ? x : y);
	case T_BIG_RATIO:   return((mpq_cmp_si(big_ratio(y), integer(x), 1) > 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_si(big_real(y), integer(x)) > 0) ? x : y);
#endif
	default:
	  return(min_out_y(sc, x, y));
	}
      break;
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((fraction(x) > integer(y)) ? y : x);
	case T_REAL:
	  return(((fraction(x) > real(y)) || (is_NaN(real(y)))) ? y : x);
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpq_cmp_z(sc->mpq_1, big_integer(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) > 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return((mpfr_cmp_q(big_real(y), sc->mpq_1) > 0) ? x : y);
#endif
	default:
	  return(min_out_y(sc, x, y));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(((real(x) <= integer(y)) || (is_NaN(real(x)))) ? x : y);
	case T_RATIO:
	  return((real(x) > fraction(y)) ? y : x);
#if With_Gmp
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return((mpfr_cmp_z(sc->mpfr_1, big_integer(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(x);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return((mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) > 0) ? y : x);
	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(x);
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_d(big_real(y), real(x)) > 0) ? x : y);
#endif
	default:
	  return(min_out_y(sc, x, y));
	}
      break;

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpz_cmp_si(big_integer(x), integer(y)) > 0) ? y : x);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return((mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) > 0) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  return((mpq_cmp_z(big_ratio(y), big_integer(x)) > 0) ? x : y);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_z(big_real(y), big_integer(x)) > 0) ? x : y);
	default:
	  return(min_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpq_cmp_si(big_ratio(x), integer(y), 1) > 0) ? y : x);
	case T_RATIO:
	  return((mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) > 0) ? y : x);
	case T_REAL:
	  if (is_NaN(real(y))) return(y);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  return((mpfr_cmp_d(sc->mpfr_1, real(y)) > 0) ? y : x);
	case T_BIG_INTEGER:
	  return((mpq_cmp_z(big_ratio(x), big_integer(y)) > 0) ? y : x);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(y);
	  return((mpfr_cmp_q(big_real(y), big_ratio(x)) > 0) ? x : y);
	default:
	  return(min_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_si(big_real(x), integer(y)) > 0) ? y : x);
	case T_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return((mpfr_cmp_q(big_real(x), sc->mpq_1) > 0) ? y : x);
	case T_REAL:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  if (is_NaN(real(y))) return(y);
	  return((mpfr_cmp_d(big_real(x), real(y)) > 0) ? y : x);
	case T_BIG_INTEGER:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_z(big_real(x), big_integer(y)) > 0) ? y : x);
	case T_BIG_RATIO:
	  if (mpfr_nan_p(big_real(x))) return(x);
	  return((mpfr_cmp_q(big_real(x), big_ratio(y)) > 0) ? y : x);
	default:
	  return(min_out_y(sc, x, y));
	}
#endif
    default:
      return(min_out_x(sc, x, y));
    }
  return(x);
}

static s7_pointer g_min(s7_scheme *sc, s7_pointer args)
{
  #define H_min "(min ...) returns the minimum of its arguments"
  #define Q_min sc->pcl_r

  s7_pointer x = car(args);
  if (is_null(cdr(args)))
    {
      if (is_real(x)) return(x);
      return(method_or_bust_p(sc, x, sc->min_symbol, sc->type_names[T_REAL]));
    }
  for (s7_pointer nums = cdr(args); is_pair(nums); nums = cdr(nums))
    x = min_p_pp(sc, x, car(nums));
  return(x);
}

static s7_pointer g_min_2(s7_scheme *sc, s7_pointer args) {return(min_p_pp(sc, car(args), cadr(args)));}
static s7_pointer g_min_3(s7_scheme *sc, s7_pointer args) {return(min_p_pp(sc, min_p_pp(sc, car(args), cadr(args)), caddr(args)));}

static s7_pointer min_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer unused_expr)
{
  return((args == 2) ? sc->min_2 : ((args == 3) ? sc->min_3 : func));
}

static s7_int min_i_ii(s7_int i1, s7_int i2) {return((i1 < i2) ? i1 : i2);}
static s7_int min_i_iii(s7_int i1, s7_int i2, s7_int i3) {return((i1 < i2) ? ((i1 < i3) ? i1 : i3) : ((i2 < i3) ? i2 : i3));}
static s7_double min_d_dd(s7_double x1, s7_double x2) {return(((x1 < x2) || (is_NaN(x1))) ? x1 : x2);}
static s7_double min_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(min_d_dd(x1, min_d_dd(x2, x3)));}
static s7_double min_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(min_d_dd(x1, min_d_ddd(x2, x3, x4)));}


/* ---------------------------------------- = ---------------------------------------- */
static bool eq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->num_eq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_error_nr(sc, sc->num_eq_symbol, 1, x, a_number_string);
  return(false);
}

static bool eq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->num_eq_symbol, set_plist_2(sc, x, y)) != sc->F);
  wrong_type_error_nr(sc, sc->num_eq_symbol, 2, y, a_number_string);
  return(false);
}

static bool num_eq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  /* (= float int) here can be confusing if the float is the result of (say) (* 4478554083/3166815962 4478554083/3166815962) -- sometimes
   *   the extra low order bits are lost somewhere, so it looks like (= 2.0 2) returning #t.  Maybe the caller should have used eqv?
   */
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) == integer(y));
      if (is_t_real(x))
	return(real(x) == real(y));
      if (is_t_complex(x))
	return((real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y)));
      if (is_t_ratio(x))
	return((numerator(x) == numerator(y)) && (denominator(x) == denominator(y)));
#if With_Gmp
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) == 0);
      if (is_t_big_ratio(x))
	return(mpq_equal(big_ratio(x), big_ratio(y)));
      if (is_t_big_real(x))
	return(mpfr_equal_p(big_real(x), big_real(y)));
      if (is_t_big_complex(x)) /* mpc_cmp can't handle NaN */
	{
	  if ((mpfr_nan_p(mpc_realref(big_complex(x)))) || (mpfr_nan_p(mpc_imagref(big_complex(x)))) ||
	      (mpfr_nan_p(mpc_realref(big_complex(y)))) || (mpfr_nan_p(mpc_imagref(big_complex(y)))))
	    return(false);
	  return(mpc_cmp(big_complex(x), big_complex(y)) == 0);
	}
#endif
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:
	  return(false);
	case T_REAL:
#if With_Gmp
	  if (s7_int_abs(integer(x)) >= Int64_To_Double_Limit)
	    {
	      if (is_NaN(real(y))) return(false);
	      mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	      return(mpfr_cmp_si(sc->mpfr_1, integer(x)) == 0);
	    }
#endif
	  return(integer(x) == real(y));
	case T_COMPLEX:
	  return(false);
#if With_Gmp
	case T_BIG_INTEGER:
	  return((mpz_fits_slong_p(big_integer(y))) && (integer(x) == mpz_get_si(big_integer(y))));
	case T_BIG_RATIO:
	  return(false);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_si(big_real(y), integer(x)) == 0));
	case T_BIG_COMPLEX:
	  return(false);
#endif
	default: return(eq_out_y(sc, x, y));
	}
      break;
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(false);
	case T_REAL:    return(fraction(x) == real(y));
	case T_COMPLEX: return(false);
#if With_Gmp
	case T_BIG_INTEGER:
	  return(false);
	case T_BIG_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_equal(sc->mpq_1, big_ratio(y)));
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(false);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) == 0);
	case T_BIG_COMPLEX:
	  return(false);
#endif
	default: return(eq_out_y(sc, x, y));
	}
      break;
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(real(x) == integer(y));
	case T_RATIO:
	  return(real(x) == fraction(y));
	case T_COMPLEX:
	  return(false);
#if With_Gmp
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) == 0);
	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) == 0);
	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(false);
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_d(big_real(y), real(x)) == 0));
	case T_BIG_COMPLEX:
	  return(false);
#endif
	default: return(eq_out_y(sc, x, y));
	}
      break;
    case T_COMPLEX:
      if (is_real(y)) return(false);
#if With_Gmp
      if (is_t_big_complex(y))
	{
	  if ((is_NaN(real_part(x))) || (is_NaN(imag_part(x))) ||
	      (mpfr_nan_p(mpc_realref(big_complex(y)))) || (mpfr_nan_p(mpc_imagref(big_complex(y)))))
	    return(false);
	  mpc_set_d_d(sc->mpc_1, real_part(x), imag_part(x), MPC_RNDNN);
	  return(mpc_cmp(big_complex(y), sc->mpc_1) == 0);
	}
#endif
      return(eq_out_y(sc, x, y));

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return((mpz_fits_slong_p(big_integer(x))) && (integer(y) == mpz_get_si(big_integer(x))));
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(x)) == 0);
	case T_RATIO: case T_COMPLEX: case T_BIG_RATIO: case T_BIG_COMPLEX:
	  return(false);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_z(big_real(y), big_integer(x)) == 0));
	default: return(eq_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpq_equal(sc->mpq_1, big_ratio(x)));
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(y), Mpfr_Rndn);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(x)) == 0);
	case T_INTEGER: case T_BIG_INTEGER: case T_COMPLEX: case T_BIG_COMPLEX:
	  return(false);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_q(big_real(y), big_ratio(x)) == 0));
	default: return(eq_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      if ((is_number(y)) && (mpfr_nan_p(big_real(x)))) return(false);
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) == 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) == 0);
	case T_REAL:
	  return((!is_NaN(real(y))) && (mpfr_cmp_d(big_real(x), real(y)) == 0));
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) == 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) == 0);
	case T_COMPLEX: case T_BIG_COMPLEX:
	  return(false);
	default: return(eq_out_y(sc, x, y));
	}

    case T_BIG_COMPLEX:
      switch (type(y))
	{
	case T_RATIO: case T_REAL: case T_INTEGER: case T_BIG_INTEGER: case T_BIG_RATIO: case T_BIG_REAL:
	  return(false);
	case T_COMPLEX:
	  if ((is_NaN(real_part(y))) || (is_NaN(imag_part(y))) ||
	      (mpfr_nan_p(mpc_realref(big_complex(x)))) || (mpfr_nan_p(mpc_imagref(big_complex(x)))))
	    return(false);
	  mpc_set_d_d(sc->mpc_1, real_part(y), imag_part(y), MPC_RNDNN);
	  return(mpc_cmp(big_complex(x), sc->mpc_1) == 0); /* NaN's not allowed! */
	default: return(eq_out_y(sc, x, y));
	}
#endif
    default: return(eq_out_x(sc, x, y));
    }
  return(false);
}

static bool is_number_via_method(s7_scheme *sc, s7_pointer p)
{
  if (is_number(p))
    return(true);
  if (has_active_methods(sc, p))
    {
      s7_pointer func = find_method_with_let(sc, p, sc->is_number_symbol);
      if (func != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, func, set_plist_1(sc, p))));
    }
  return(false);
}

static s7_pointer g_num_eq(s7_scheme *sc, s7_pointer args)
{
  #define H_num_eq "(= z1 ...) returns #t if all its arguments are equal"
  #define Q_num_eq s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_number_symbol)

  const s7_pointer x = car(args);
  s7_pointer nums = cdr(args);
  if (is_null(cdr(nums)))
    return(make_boolean(sc, num_eq_b_7pp(sc, x, car(nums))));

  for (; is_pair(nums); nums = cdr(nums))
    if (!num_eq_b_7pp(sc, x, car(nums)))
      {
	for (nums = cdr(nums); is_pair(nums); nums = cdr(nums))
	  if (!is_number_via_method(sc, car(nums)))
	    wrong_type_error_nr(sc, sc->num_eq_symbol, position_of(nums, args), car(nums), a_number_string);
	return(sc->F);
      }
  return(sc->T);
}

static bool num_eq_b_ii(s7_int i1, s7_int i2) {return(i1 == i2);}
static bool num_eq_b_dd(s7_double i1, s7_double i2) {return(i1 == i2);}
static s7_pointer num_eq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 == x2));}
static s7_pointer num_eq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2)       {return(make_boolean(sc, x1 == x2));}
static s7_pointer num_eq_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, num_eq_b_7pp(sc, x, y)));}

static s7_pointer num_eq_p_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) == y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) == y));
#if With_Gmp
  if (is_t_big_integer(x))
    return(make_boolean(sc, (mpz_fits_slong_p(big_integer(x))) && (y == mpz_get_si(big_integer(x)))));
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_si(big_real(x), y) == 0));
#endif
  if (is_number(x))
    return(sc->F); /* complex/ratio can't == int */
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->num_eq_symbol, set_plist_2(sc, x, make_integer(sc, y))));
  wrong_type_error_nr(sc, sc->num_eq_symbol, 1, x, a_number_string);
#ifdef __TINYC__
  return(sc->F);
#endif
}

static bool num_eq_b_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x))
    return(integer(x) == y);
  if (is_t_real(x))
    return(real(x) == y);
#if With_Gmp
  if (is_t_big_integer(x))
    return((mpz_fits_slong_p(big_integer(x))) && (y == mpz_get_si(big_integer(x))));
  if (is_t_big_real(x))
    return(mpfr_cmp_si(big_real(x), y) == 0);
#endif
  if (!is_number(x)) /* complex/ratio can't == int */
    wrong_type_error_nr(sc, sc->num_eq_symbol, 1, x, a_number_string);
  return(false);
}

static s7_pointer g_num_eq_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x = car(args), y = cadr(args);
  if ((is_t_integer(x)) && (is_t_integer(y))) /* this is by far the most common case (ratios aren't used much, and = with floats is frowned upon) */
    return(make_boolean(sc, integer(x) == integer(y)));
  return(make_boolean(sc, num_eq_b_7pp(sc, x, y)));
}

static inline s7_pointer num_eq_xx(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) == integer(y)));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) == integer(y)));
  if (!is_number(x))
    return(make_boolean(sc, eq_out_x(sc, x, y)));
#if With_Gmp
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), integer(y)) == 0));
  if (is_t_big_real(x))
    {
      if (mpfr_nan_p(big_real(x))) return(sc->F);
      return(make_boolean(sc, mpfr_cmp_si(big_real(x), integer(y)) == 0));
    }
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), integer(y), 1) == 0));
#endif
  return(sc->F);
}

static s7_pointer g_num_eq_xi(s7_scheme *sc, s7_pointer args) {return(num_eq_xx(sc, car(args), cadr(args)));}
static s7_pointer g_num_eq_ix(s7_scheme *sc, s7_pointer args) {return(num_eq_xx(sc, cadr(args), car(args)));}

static s7_pointer num_eq_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  s7_pointer arg1, arg2;
  if (args != 2) return(func);
  arg1 = cadr(expr);
  arg2 = caddr(expr);
  if ((is_pair(arg1)) && (has_fn(arg1)) && (fn_proc(arg1) == g_add_3)) set_fn_direct(arg1, g_add_3_wrapped);
  if (is_t_integer(arg2)) return(sc->num_eq_xi);
  return((is_t_integer(arg1)) ? sc->num_eq_ix : sc->num_eq_2);
}


/* ---------------------------------------- < ---------------------------------------- */
static bool lt_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->lt_symbol, set_mlist_2(sc, x, y)) != sc->F); /* not plist */
  wrong_type_error_nr(sc, sc->lt_symbol, 1, x, sc->type_names[T_REAL]);
  return(false);
}

static bool lt_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->lt_symbol, set_mlist_2(sc, x, y)) != sc->F);
  wrong_type_error_nr(sc, sc->lt_symbol, 2, y, sc->type_names[T_REAL]);
  return(false);
}

static bool lt_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) < integer(y));
      if (is_t_real(x))
	return(real(x) < real(y));
      if (is_t_ratio(x))
	return(fraction(x) < fraction(y));
#if With_Gmp
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) < 0);
      if (is_t_big_ratio(x))
	return(mpq_cmp(big_ratio(x), big_ratio(y)) < 0);
      if (is_t_big_real(x))
	return(mpfr_less_p(big_real(x), big_real(y)));
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:	return(integer(x) < fraction(y)); /* ?? */
	case T_REAL:    return(integer(x) < real(y));
#if With_Gmp
	case T_BIG_INTEGER: return(mpz_cmp_si(big_integer(y), integer(x)) > 0);
	case T_BIG_RATIO:   return(mpq_cmp_si(big_ratio(y), integer(x), 1) > 0);
	case T_BIG_REAL:    return(mpfr_cmp_si(big_real(y), integer(x)) > 0);
#endif
	default: return(lt_out_y(sc, x, y));
	}
      break;
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(fraction(x) < integer(y));
	case T_REAL:    return(fraction(x) < real(y));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_cmp_z(sc->mpq_1, big_integer(y)) < 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) > 0);
	case T_BIG_REAL:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) > 0);
#endif
	default: return(lt_out_y(sc, x, y));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(real(x) < integer(y));
	case T_RATIO:	return(real(x) < fraction(y));
#if With_Gmp
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) < 0);
	case T_BIG_RATIO:
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) < 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_d(big_real(y), real(x)) > 0);
#endif
	default: return(lt_out_y(sc, x, y));
	}
      break;

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpz_cmp_si(big_integer(x), integer(y)) < 0);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return(mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) < 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) < 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_z(big_ratio(y), big_integer(x)) > 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_z(big_real(y), big_integer(x)) > 0);
	default: return(lt_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpq_cmp_si(big_ratio(x), integer(y), 1) < 0);
	case T_RATIO:
	  return(mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) < 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) < 0);
	case T_BIG_INTEGER:
	  return(mpq_cmp_z(big_ratio(x), big_integer(y)) < 0);
	case T_BIG_REAL:
	  return(mpfr_cmp_q(big_real(y), big_ratio(x)) > 0);
	default: return(lt_out_y(sc, x, y));
	}
    case T_BIG_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) < 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) < 0);
	case T_REAL:
	  return(mpfr_cmp_d(big_real(x), real(y)) < 0);
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) < 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) < 0);
	default: return(lt_out_y(sc, x, y));
	}
#endif
    default: return(lt_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer g_less(s7_scheme *sc, s7_pointer args)
{
  #define H_less "(< x1 ...) returns #t if its arguments are in increasing order"
  #define Q_less s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), p = cdr(args);
  if (is_null(cdr(p)))
    return(make_boolean(sc, lt_b_7pp(sc, x, car(p))));

  for (; is_pair(p); p = cdr(p))
    {
      if (!lt_b_7pp(sc, x, car(p)))
	{
	  for (p = cdr(p); is_pair(p); p = cdr(p))
	    if (!is_real_via_method(sc, car(p)))
	      wrong_type_error_nr(sc, sc->lt_symbol, position_of(p, args), car(p), sc->type_names[T_REAL]);
	  return(sc->F);
	}
      x = car(p);
    }
  return(sc->T);
}

static bool ratio_lt_pi(s7_pointer x, s7_int y)
{
  if ((y >= 0) && (numerator(x) < 0))
    return(true);
  if ((y <= 0) && (numerator(x) > 0))
    return(false);
  if (denominator(x) < s7_Int32_Max)
    return(numerator(x) < (y * denominator(x)));
  return(fraction(x) < y);
}

static s7_pointer g_less_x0(s7_scheme *sc, s7_pointer args)
{
  const s7_pointer x = car(args);
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) < 0));
  if (is_small_real(x))
    return(make_boolean(sc, is_negative(sc, x)));
#if With_Gmp
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), 0) < 0));
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_si(big_real(x), 0) < 0));
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), 0, 1) < 0));
#endif
  return(method_or_bust(sc, x, sc->lt_symbol, args, sc->type_names[T_REAL], 1));
}

static s7_pointer g_less_xi(s7_scheme *sc, s7_pointer args)
{
  const s7_int y = integer(cadr(args));
  const s7_pointer x = car(args);

  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) < y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) < y));
  if (is_t_ratio(x))
    return(make_boolean(sc, ratio_lt_pi(x, y)));
#if With_Gmp
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), y) < 0));
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_si(big_real(x), y) < 0));
  if (is_t_big_ratio(x))
    return(make_boolean(sc, mpq_cmp_si(big_ratio(x), y, 1) < 0));
#endif
  return(method_or_bust(sc, x, sc->lt_symbol, args, sc->type_names[T_REAL], 1));
}

static s7_pointer g_less_xf(s7_scheme *sc, s7_pointer args)
{
  const s7_double y = real(cadr(args)); /* chooser below checks is_t_real(y) */
  const s7_pointer x = car(args);

  if (is_t_real(x))
    return(make_boolean(sc, real(x) < y));
  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) < y));
  if (is_t_ratio(x))
    return(make_boolean(sc, fraction(x) < y));
#if With_Gmp
  if (is_t_big_real(x))
    return(make_boolean(sc, mpfr_cmp_d(big_real(x), y) < 0));
  if (is_t_big_integer(x))
    {
      mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
      return(make_boolean(sc, mpfr_cmp_z(sc->mpfr_1, big_integer(x)) > 0));
    }
  if (is_t_big_ratio(x))
    {
      mpfr_set_d(sc->mpfr_1, y, Mpfr_Rndn);
      return(make_boolean(sc, mpfr_cmp_q(sc->mpfr_1, big_ratio(x)) > 0));
    }
#endif
  return(method_or_bust(sc, x, sc->lt_symbol, args, sc->type_names[T_REAL], 1));
}

static inline s7_pointer lt_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, lt_b_7pp(sc, x, y)));}
static bool lt_b_ii(s7_int i1, s7_int i2) {return(i1 < i2);}
static bool lt_b_dd(s7_double i1, s7_double i2) {return(i1 < i2);}
static s7_pointer lt_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 < x2));}
static s7_pointer lt_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 < x2));}

static bool lt_b_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_t_integer(x)) return(integer(x) < y);
  if (is_t_real(x))  return(real(x) < y);
  if (is_t_ratio(x)) return(ratio_lt_pi(x, y));
#if With_Gmp
  if (is_t_big_integer(x))
    return(mpz_cmp_si(big_integer(x), y) < 0);
  if (is_t_big_real(x))
    return(mpfr_cmp_si(big_real(x), y) < 0);
  if (is_t_big_ratio(x))
    return(mpq_cmp_si(big_ratio(x), y, 1) < 0);
#endif
  return(lt_out_x(sc, x, make_integer(sc, y)));
}

static s7_pointer g_less_2(s7_scheme *sc, s7_pointer args) {return(lt_p_pp(sc, car(args), cadr(args)));}
static s7_pointer lt_p_pi(s7_scheme *sc, s7_pointer x, s7_int y) {return(make_boolean(sc, lt_b_pi(sc, x, y)));}

static s7_pointer less_chooser(s7_scheme *sc, s7_pointer func, int32_t args, s7_pointer expr)
{
  s7_pointer arg2;
  if (args != 2) return(func);
  arg2 = caddr(expr);
  if (is_t_integer(arg2))
    {
      if (integer(arg2) == 0)
	return(sc->less_x0);
      if ((integer(arg2) < s7_Int32_Max) && (integer(arg2) > s7_Int32_Min))
	return(sc->less_xi);
    }
  if (is_t_real(arg2))
    return(sc->less_xf);
  return(sc->less_2);
}


/* ---------------------------------------- <= ---------------------------------------- */
static bool leq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, x))
    return(find_and_apply_method(sc, x, sc->leq_symbol, set_mlist_2(sc, x, y)) != sc->F); /* not plist */
  wrong_type_error_nr(sc, sc->leq_symbol, 1, x, sc->type_names[T_REAL]);
  return(false);
}

static bool leq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_active_methods(sc, y))
    return(find_and_apply_method(sc, y, sc->leq_symbol, set_mlist_2(sc, x, y)) != sc->F);
  wrong_type_error_nr(sc, sc->leq_symbol, 2, y, sc->type_names[T_REAL]);
  return(false);
}

static bool leq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_integer(x))
	return(integer(x) <= integer(y));
      if (is_t_real(x))
	return(real(x) <= real(y));
      if (is_t_ratio(x))
	return(fraction(x) <= fraction(y));
#if With_Gmp
      if (is_t_big_integer(x))
	return(mpz_cmp(big_integer(x), big_integer(y)) <= 0);
      if (is_t_big_ratio(x))
	return(mpq_cmp(big_ratio(x), big_ratio(y)) <= 0);
      if (is_t_big_real(x))
	return(mpfr_lessequal_p(big_real(x), big_real(y)));
#endif
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_RATIO:	return(integer(x) <= fraction(y)); /* ?? */
	case T_REAL:	return(integer(x) <= real(y));
#if With_Gmp
	case T_BIG_INTEGER: return(mpz_cmp_si(big_integer(y), integer(x)) >= 0);
	case T_BIG_RATIO:   return(mpq_cmp_si(big_ratio(y), integer(x), 1) >= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_si(big_real(y), integer(x)) >= 0));
#endif
	default: return(leq_out_y(sc, x, y));
	}
      break;
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(fraction(x) <= integer(y));
	case T_REAL:    return(fraction(x) <= real(y));
#if With_Gmp
	case T_BIG_INTEGER:
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpq_cmp_z(sc->mpq_1, big_integer(y)) <= 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_si(big_ratio(y), numerator(x), denominator(x)) >= 0);
	case T_BIG_REAL:
	  if (mpfr_nan_p(big_real(y))) return(false);
	  mpq_set_si(sc->mpq_1, numerator(x), denominator(x));
	  return(mpfr_cmp_q(big_real(y), sc->mpq_1) >= 0);
#endif
	default: return(leq_out_y(sc, x, y));
	}
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(real(x) <= integer(y));
	case T_RATIO:	return(real(x) <= fraction(y));
#if With_Gmp
	case T_BIG_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return(mpfr_cmp_z(sc->mpfr_1, big_integer(y)) <= 0);
	case T_BIG_RATIO:
	  if (is_NaN(real(x))) return(false);
	  mpfr_set_d(sc->mpfr_1, real(x), Mpfr_Rndn);
	  return(mpfr_cmp_q(sc->mpfr_1, big_ratio(y)) <= 0);
	case T_BIG_REAL:
	  if (is_NaN(real(x))) return(false);
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_d(big_real(y), real(x)) >= 0));
#endif
	default: return(leq_out_y(sc, x, y));
	}
      break;

#if With_Gmp
    case T_BIG_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpz_cmp_si(big_integer(x), integer(y)) <= 0);
	case T_RATIO:
	  mpq_set_z(sc->mpq_1, big_integer(x));
	  return(mpq_cmp_si(sc->mpq_1, numerator(y), denominator(y)) <= 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_z(sc->mpfr_1, big_integer(x), Mpfr_Rndn);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) <= 0);
	case T_BIG_RATIO:
	  return(mpq_cmp_z(big_ratio(y), big_integer(x)) >= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_z(big_real(y), big_integer(x)) >= 0));
	default: return(leq_out_y(sc, x, y));
	}
    case T_BIG_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpq_cmp_si(big_ratio(x), integer(y), 1) <= 0);
	case T_RATIO:
	  return(mpq_cmp_si(big_ratio(x), numerator(y), denominator(y)) <= 0);
	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  mpfr_set_q(sc->mpfr_1, big_ratio(x), Mpfr_Rndn);
	  return(mpfr_cmp_d(sc->mpfr_1, real(y)) <= 0);
	case T_BIG_INTEGER:
	  return(mpq_cmp_z(big_ratio(x), big_integer(y)) <= 0);
	case T_BIG_REAL:
	  return((!mpfr_nan_p(big_real(y))) && (mpfr_cmp_q(big_real(y), big_ratio(x)) >= 0));
	default: return(leq_out_y(sc, x, y));
	}

    case T_BIG_REAL:
      if ((is_real(y)) && (mpfr_nan_p(big_real(x)))) return(false);
      switch (type(y))
	{
	case T_INTEGER:
	  return(mpfr_cmp_si(big_real(x), integer(y)) <= 0);
	case T_RATIO:
	  mpq_set_si(sc->mpq_1, numerator(y), denominator(y));
	  return(mpfr_cmp_q(big_real(x), sc->mpq_1) <= 0);
	case T_REAL:
	  return((!is_NaN(real(y))) && (mpfr_cmp_d(big_real(x), real(y)) <= 0));
	case T_BIG_INTEGER:
	  return(mpfr_cmp_z(big_real(x), big_integer(y)) <= 0);
	case T_BIG_RATIO:
	  return(mpfr_cmp_q(big_real(x), big_ratio(y)) <= 0);
	default: return(leq_out_y(sc, x, y));
	}
#endif
    default: return(leq_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer g_less_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_less_or_equal "(<= x1 ...) returns #t if its arguments are in non-decreasing order"
  #define Q_less_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x = car(args), p = cdr(args);

  if (is_null(cdr(p)))
    return(make_boolean(sc, leq_b_7pp(sc, x, car(p))));
  for (; is_pair(p); x = car(p), p = cdr(p))
    if (!leq_b_7pp(sc, x, car(p)))
      {
	for (p = cdr(p); is_pair(p); p = cdr(p))
	  if (!is_real_via_method(sc, car(p)))
	    wrong_type_error_nr(sc, sc->leq_symbol, position_of(p, args), car(p), sc->type_names[T_REAL]);
	return(sc->F);
      }
  return(sc->T);
}

static inline s7_pointer leq_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(make_boolean(sc, leq_b_7pp(sc, x, y)));}
static bool leq_b_ii(s7_int i1, s7_int i2) {return(i1 <= i2);}
static bool leq_b_dd(s7_double i1, s7_double i2) {return(i1 <= i2);}
static s7_pointer leq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 <= x2));}
static s7_pointer leq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 <= x2));}

static bool ratio_leq_pi(s7_pointer x, s7_int y)
{
  if ((y >= 0) && (numerator(x) <= 0))
    return(true);
  if ((y <= 0) && (numerator(x) > 0))
    return(false);
  if (denominator(x) < s7_Int32_Max)
    return(numerator(x) <= (y * denominator(x)));
  return(fraction(x) <= y);
}

static s7_pointer g_leq_xi(s7_scheme *sc, s7_pointer args)
{
  const s7_int y = integer(cadr(args));
  const s7_pointer x = car(args);

  if (is_t_integer(x))
    return(make_boolean(sc, integer(x) <= y));
  if (is_t_real(x))
    return(make_boolean(sc, real(x) <= y));
  if (is_t_ratio(x))
    return(make_boolean(sc, ratio_leq_pi(x, y)));
#if With_Gmp
  if (is_t_big_integer(x))
    return(make_boolean(sc, mpz_cmp_si(big_integer(x), y) <= 0));
  if (is_t_big_real(x))
    {
      if (mpfr_nan_p(big_real(x))) return(sc->F);
     