// Generated by astgen // -*- mode: C++; c-file-style: "cc-mode" -*-
#line 1 "../V3Const.cpp"
// -*- mode: C++; c-file-style: "cc-mode" -*-
//*************************************************************************
// DESCRIPTION: Verilator: Constant folding
//
// Code available from: https://verilator.org
//
//*************************************************************************
//
// Copyright 2003-2025 by Wilson Snyder. This program is free software; you
// can redistribute it and/or modify it under the terms of either the GNU
// Lesser General Public License Version 3 or the Perl Artistic License
// Version 2.0.
// SPDX-License-Identifier: LGPL-3.0-only OR Artistic-2.0
//
//*************************************************************************
// CONST TRANSFORMATIONS:
//   Call on one node for PARAM values, or netlist for overall constant folding:
//      Bottom up traversal:
//          Attempt to convert operands to constants
//          If operands are constant, replace this node with constant.
//*************************************************************************

#include "V3PchAstNoMT.h"  // VL_MT_DISABLED_CODE_UNIT

#include "config_build.h"
#include "verilatedos.h"

#include "V3Const.h"

#include "V3Ast.h"
#include "V3Global.h"
#include "V3Simulate.h"
#include "V3Stats.h"
#include "V3String.h"
#include "V3UniqueNames.h"
#include "V3Width.h"

#include <algorithm>
#include <memory>
#include <type_traits>
#include <unordered_set>

VL_DEFINE_DEBUG_FUNCTIONS;

#define TREE_SKIP_VISIT(...)
#define TREEOP1(...)
#define TREEOPA(...)
#define TREEOP(...)
#define TREEOPS(...)
#define TREEOPC(...)
#define TREEOPV(...)

//######################################################################
// Utilities

static bool isConst(const AstNode* nodep, uint64_t v) {
    const AstConst* const constp = VN_CAST(nodep, Const);
    return constp && constp->toUQuad() == v;
}

template <typename T>
static typename std::enable_if<std::is_integral<T>::value, bool>::type isPow2(T val) {
    return (val & (val - 1)) == 0;
}

static int countTrailingZeroes(uint64_t val) {
    UASSERT(val, "countTrailingZeroes argument must be non-zero");
#if defined(__GNUC__) && !defined(VL_NO_BUILTINS)
    return __builtin_ctzll(val);
#else
    int bit = 0;
    val = ~val;
    while (val & 1) {
        ++bit;
        val >>= 1;
    }
    return bit;
#endif
}

// This visitor can be used in the post-expanded Ast from V3Expand, where the Ast satisfies:
// - Constants are 64 bit at most (because words are accessed via AstWordSel)
// - Variables are scoped.
class ConstBitOpTreeVisitor final : public VNVisitorConst {
    // NODE STATE
    // AstVarRef::user4u      -> Base index of m_varInfos that points VarInfo
    // AstVarScope::user4u    -> Same as AstVarRef::user4
    const VNUser4InUse m_inuser4;

    // TYPES

    // Holds a node to be added as a term in the reduction tree, it's equivalent op count, and a
    // bool indicating if the term is clean (0/1 value, or if the top bits might be dirty)
    using ResultTerm = std::tuple<AstNodeExpr*, unsigned, bool>;

    class LeafInfo final {  // Leaf node (either AstConst or AstVarRef)
        // MEMBERS
        bool m_polarity = true;
        int m_lsb = 0;  // LSB of actually used bit of m_refp->varp()
        int m_msb = 0;  // MSB of actually used bit of m_refp->varp()
        int m_wordIdx = -1;  // -1 means AstWordSel is not used.
        AstVarRef* m_refp = nullptr;
        const AstConst* m_constp = nullptr;

    public:
        // CONSTRUCTORS
        LeafInfo() = default;
        LeafInfo(const LeafInfo& other) = default;
        explicit LeafInfo(int lsb)
            : m_lsb{lsb} {}

        // METHODS
        void setLeaf(AstVarRef* refp) {
            UASSERT_OBJ(!m_refp && !m_constp, refp, "Must be called just once");
            m_refp = refp;
            m_msb = refp->varp()->widthMin() - 1;
        }
        void setLeaf(const AstConst* constp) {
            UASSERT_OBJ(!m_refp && !m_constp, constp, "Must be called just once");
            m_constp = constp;
            m_msb = constp->widthMin() - 1;
        }
        // updateBitRange(), limitBitRangeToLsb(), and polarity() must be called during ascending
        // back to the root.
        void updateBitRange(int newLsb, int newMsb) {
            if ((m_lsb <= m_msb && newLsb > newMsb) || (m_lsb > m_msb && m_lsb < newLsb)) {
                // When the new bit range is out of m_refp, clear polarity because nodes below is
                // shifted out to zero.
                // This kind of clear may happen several times. e.g. (!(1'b1 >> 1)) >> 1
                polarity(true);
            }
            m_lsb = newLsb;
            m_msb = newMsb;
        }
        void updateBitRange(const AstCCast* castp) {
            updateBitRange(m_lsb, std::min(m_msb, m_lsb + castp->width() - 1));
        }
        void updateBitRange(const AstShiftR* shiftp) {
            updateBitRange(m_lsb + VN_AS(shiftp->rhsp(), Const)->toUInt(), m_msb);
        }
        void limitBitRangeToLsb() { updateBitRange(m_lsb, std::min(m_msb, m_lsb)); }
        int wordIdx() const { return m_wordIdx; }
        void wordIdx(int i) { m_wordIdx = i; }
        bool polarity() const { return m_polarity; }
        void polarity(bool p) { m_polarity = p; }

        AstVarRef* refp() const { return m_refp; }
        const AstConst* constp() const { return m_constp; }
        bool missingWordSel() const {
            // When V3Expand is skipped, WordSel is not inserted.
            return m_refp->isWide() && m_wordIdx == -1;
        }
        int lsb() const { return m_lsb; }

        int msb() const { return std::min(m_msb, varWidth() - 1); }
        int varWidth() const {
            UASSERT(m_refp, "m_refp should be set");
            const int width = m_refp->varp()->widthMin();
            if (!m_refp->isWide()) {
                UASSERT_OBJ(m_wordIdx == -1, m_refp, "Bad word index into non-wide");
                return width;
            } else {
                if (missingWordSel()) return width;
                UASSERT_OBJ(m_wordIdx >= 0, m_refp, "Bad word index into wide");
                const int bitsInMSW = VL_BITBIT_E(width) ? VL_BITBIT_E(width) : VL_EDATASIZE;
                return m_wordIdx == m_refp->widthWords() - 1 ? bitsInMSW : VL_EDATASIZE;
            }
        }
    };

    struct BitPolarityEntry final {  // Found bit polarity during iterate()
        LeafInfo m_info;
        bool m_polarity = false;
        int m_bit = 0;
        BitPolarityEntry(const LeafInfo& info, bool pol, int bit)
            : m_info{info}
            , m_polarity{pol}
            , m_bit{bit} {}
        BitPolarityEntry() = default;
    };

    struct FrozenNodeInfo final {  // Context when a frozen node is found
        bool m_polarity;
        int m_lsb;
        bool operator<(const FrozenNodeInfo& other) const {
            if (m_lsb != other.m_lsb) return m_lsb < other.m_lsb;
            return m_polarity < other.m_polarity;
        }
    };

    class Restorer final {  // Restore the original state unless disableRestore() is called
        ConstBitOpTreeVisitor& m_visitor;
        const size_t m_polaritiesSize;
        const size_t m_frozenSize;
        const unsigned m_ops;
        const bool m_polarity;
        bool m_restore;

    public:
        explicit Restorer(ConstBitOpTreeVisitor& visitor)
            : m_visitor{visitor}
            , m_polaritiesSize{visitor.m_bitPolarities.size()}
            , m_frozenSize{visitor.m_frozenNodes.size()}
            , m_ops{visitor.m_ops}
            , m_polarity{visitor.m_polarity}
            , m_restore{true} {}
        ~Restorer() {
            UASSERT(m_visitor.m_bitPolarities.size() >= m_polaritiesSize,
                    "m_bitPolarities must grow monotonically");
            UASSERT(m_visitor.m_frozenNodes.size() >= m_frozenSize,
                    "m_frozenNodes must grow monotonically");
            if (m_restore) restoreNow();
        }
        void disableRestore() { m_restore = false; }
        void restoreNow() {
            UASSERT(m_restore, "Can be called just once");
            m_visitor.m_bitPolarities.resize(m_polaritiesSize);
            m_visitor.m_frozenNodes.resize(m_frozenSize);
            m_visitor.m_ops = m_ops;
            m_visitor.m_polarity = m_polarity;
            m_restore = false;
        }
    };
    // Collect information for each Variable to transform as below
    class VarInfo final {
        // MEMBERS
        int m_knownResult = -1;  // -1: result is not known, 0 or 1: result of this tree
        const ConstBitOpTreeVisitor* const
            m_parentp;  // ConstBitOpTreeVisitor holding this VarInfo
        AstVarRef* const m_refp;  // Points the variable that this VarInfo covers
        const int m_width;  // Width of term this VarInfo refers to
        V3Number m_bitPolarity;  // Coefficient of each bit

    public:
        // METHODS
        bool hasConstResult() const { return m_knownResult >= 0 || m_bitPolarity.isAllX(); }
        // The constant result. Only valid if hasConstResult() returned true.
        bool getConstResult() const {
            // Note that this condition covers m_knownResult == -1 but m_bitPolarity.isAllX(),
            // in which case the result is 0
            return m_knownResult == 1;
        }
        const AstVarRef* refp() const { return m_refp; }
        bool sameVarAs(const AstNodeVarRef* otherp) const { return m_refp->sameNode(otherp); }
        void setPolarity(bool compBit, int bit) {
            // Ignore if already determined a known reduction
            if (m_knownResult >= 0) return;
            UASSERT_OBJ(bit < m_width, m_refp,
                        "Bit index out of range: " << bit << " width: " << m_width);
            if (m_bitPolarity.bitIsX(bit)) {  // The bit is not yet marked with either polarity
                m_bitPolarity.setBit(bit, compBit);
            } else {  // The bit has already been marked with some polarity
                const bool sameFlag = m_bitPolarity.bitIs1(bit) == compBit;
                if (m_parentp->isXorTree()) {
                    UASSERT_OBJ(compBit && sameFlag, m_refp, "Only true is set in Xor tree");
                    // a ^ a ^ b == b so we can ignore a
                    m_bitPolarity.setBit(bit, 'x');
                } else {  // And, Or
                    // Can ignore this nodep as the bit is already marked with the same polarity
                    if (sameFlag) return;  // a & a == a, b | b == b
                    // Otherwise result is constant (a & ~a == 0) or (a | ~a == 1)
                    m_knownResult = m_parentp->isAndTree() ? 0 : 1;
                    m_bitPolarity.setAllBitsX();  // The variable is not referred anymore
                }
            }
        }

        // Return reduction term for this VarInfo, together with the number of ops in the term,
        // and a boolean indicating if the term is clean (1-bit vs multi-bit value)
        ResultTerm getResultTerm() const {
            UASSERT_OBJ(!hasConstResult(), m_refp, "getTerm on reduction that yields constant");
            FileLine* const fl = m_refp->fileline();

            // Get the term we are referencing (the WordSel, if wide, otherwise just the VarRef)
            AstNodeExpr* srcp = VN_CAST(m_refp->backp(), WordSel);
            if (!srcp) srcp = m_refp;
            srcp = srcp->cloneTree(false);

            // Signed variables might have redundant sign bits that need masking.
            const bool hasRedundantSignBits
                = m_refp->varp()->dtypep()->isSigned()
                  && (m_refp->isWide() ? (m_width != VL_EDATASIZE)
                                       : (m_width < 8 || !isPow2(m_width)));

            // Get the mask that selects the bits that are relevant in this term
            V3Number maskNum{srcp, m_width, 0};
            maskNum.opBitsNonX(m_bitPolarity);  // 'x' -> 0, 0->1, 1->1
            const uint64_t maskVal = maskNum.toUQuad();
            UASSERT_OBJ(maskVal != 0, m_refp,
                        "Should have been recognized as having const 0 result");

            // Parts of the return value
            AstNodeExpr* resultp = srcp;  // The tree for this term
            unsigned ops = 0;  // Number of ops in this term
            bool clean = false;  // Whether the term is clean (has value 0 or 1)

            if (isPow2(maskVal)) {
                // If we only want a single bit, shift it out instead of a masked compare. Shifts
                // don't go through the flags register on x86 and are hence faster. This is also
                // always fewer or same ops as mask and compare, but with shorter instructions on
                // x86.

                // Find the index of the bit we want.
                const int bit = countTrailingZeroes(maskVal);
                // If we want something other than the bottom bit, shift it out
                if (bit != 0) {
                    resultp = new AstShiftR{fl, resultp,
                                            new AstConst{fl, static_cast<uint32_t>(bit)}, m_width};
                    ++ops;
                }
                // Negate it if necessary
                const bool negate = m_bitPolarity.bitIs0(bit);
                if (negate) {
                    resultp = new AstNot{fl, resultp};
                    ++ops;
                }
                // Clean if MSB of unsigned value, and not negated
                clean = (bit == m_width - 1) && !hasRedundantSignBits && !negate;
            } else {
                // We want multiple bits. Go ahead and extract them.

                // Check if masking is required, and if so apply it
                const bool needsMasking = maskVal != VL_MASK_Q(m_width) || hasRedundantSignBits;
                if (needsMasking) {
                    resultp = new AstAnd{fl, new AstConst{fl, maskNum}, resultp};
                    ++ops;
                }

                // Create the sub-expression for this term
                if (m_parentp->isXorTree()) {
                    if (needsMasking) {
                        // Reduce the masked term to the minimum known width,
                        // to use the smallest RedXor formula
                        const int widthMin = maskNum.widthToFit();
                        resultp->dtypeChgWidth(widthMin, widthMin);
                    }
                    resultp = new AstRedXor{fl, resultp};
                    ++ops;
                    clean = false;
                    // VL_REDXOR_* returns IData, set width accordingly to avoid unnecessary casts
                    resultp->dtypeChgWidth(VL_IDATASIZE, 1);
                } else if (m_parentp->isAndTree()) {
                    V3Number compNum{srcp, m_width, 0};
                    compNum.opBitsOne(m_bitPolarity);  // 'x'->0, 0->0, 1->1
                    resultp = new AstEq{fl, new AstConst{fl, compNum}, resultp};
                    ++ops;
                    clean = true;
                } else {  // Or
                    V3Number compNum{srcp, m_width, 0};
                    compNum.opBitsOne(m_bitPolarity);  // 'x'->0, 0->0, 1->1
                    compNum.opXor(V3Number{compNum}, maskNum);
                    resultp = new AstNeq{fl, new AstConst{fl, compNum}, resultp};
                    ++ops;
                    clean = true;
                }
            }

            return ResultTerm{resultp, ops, clean};
        }

        // CONSTRUCTORS
        VarInfo(ConstBitOpTreeVisitor* parent, AstVarRef* refp, int width)
            : m_parentp{parent}
            , m_refp{refp}
            , m_width{width}
            , m_bitPolarity{refp, m_width} {
            m_bitPolarity.setAllBitsX();
        }
    };

    // MEMBERS
    bool m_failed = false;
    bool m_polarity = true;  // Flip when AstNot comes
    unsigned m_ops;  // Number of operations such as And, Or, Xor, Sel...
    int m_lsb = 0;  // Current LSB
    LeafInfo* m_leafp = nullptr;  // AstConst or AstVarRef that currently looking for
    const AstNodeExpr* const m_rootp;  // Root of this AST subtree

    std::vector<std::pair<AstNodeExpr*, FrozenNodeInfo>>
        m_frozenNodes;  // Nodes that cannot be optimized
    std::vector<BitPolarityEntry> m_bitPolarities;  // Polarity of bits found during iterate()
    std::vector<std::unique_ptr<VarInfo>> m_varInfos;  // VarInfo for each variable, [0] is nullptr

    // METHODS

    bool isAndTree() const { return VN_IS(m_rootp, And); }
    bool isOrTree() const { return VN_IS(m_rootp, Or); }
    bool isXorTree() const { return VN_IS(m_rootp, Xor) || VN_IS(m_rootp, RedXor); }

#define CONST_BITOP_RETURN_IF(cond, nodep) \
    if (setFailed(cond, #cond, nodep, __LINE__)) return

#define CONST_BITOP_SET_FAILED(reason, nodep) setFailed(true, reason, nodep, __LINE__)

    bool setFailed(bool fail, const char* reason, const AstNode* nodep, int line) {
        if (fail && !m_failed) {
            UINFO(9, "cannot optimize " << m_rootp << " reason:" << reason << " called from line:"
                                        << line << " when checking:" << nodep);
            // UINFOTREE(9, m_rootp, "", "root");
            m_failed = true;
        }
        return m_failed;
    }
    void incrOps(const AstNode* nodep, int line) {
        ++m_ops;
        UINFO(9, "Increment to " << m_ops << " " << nodep << " called from line " << line);
    }
    VarInfo& getVarInfo(const LeafInfo& ref) {
        UASSERT_OBJ(ref.refp(), m_rootp, "null varref in And/Or/Xor optimization");
        AstNode* nodep = ref.refp()->varScopep();
        if (!nodep) nodep = ref.refp()->varp();  // Not scoped
        int baseIdx = nodep->user4();
        if (baseIdx == 0) {  // Not set yet
            baseIdx = m_varInfos.size();
            const int numWords
                = ref.refp()->dtypep()->isWide() ? ref.refp()->dtypep()->widthWords() : 1;
            m_varInfos.resize(m_varInfos.size() + numWords);
            nodep->user4(baseIdx);
        }
        const size_t idx = baseIdx + std::max(0, ref.wordIdx());
        VarInfo* varInfop = m_varInfos[idx].get();
        if (!varInfop) {
            varInfop = new VarInfo{this, ref.refp(), ref.varWidth()};
            m_varInfos[idx].reset(varInfop);
            if (ref.missingWordSel()) {
                // ConstBitOpTreeVisitor makes some constants for masks and its type is uint64_t.
                // That's why V3Expand, that inserts WordSel, is needed.
                CONST_BITOP_SET_FAILED("V3Expand is skipped", ref.refp());
            }
        } else {
            if (!varInfop->sameVarAs(ref.refp()))
                CONST_BITOP_SET_FAILED("different var (scope?)", ref.refp());
        }
        return *varInfop;
    }

    // Traverse down to see AstConst or AstVarRef
    LeafInfo findLeaf(AstNode* nodep, bool expectConst) {
        LeafInfo info{m_lsb};
        {
            VL_RESTORER(m_leafp);
            m_leafp = &info;
            iterateConst(nodep);
        }

        bool ok = !m_failed;
        if (expectConst) {
            ok &= !info.refp() && info.constp();
        } else {
            ok &= info.refp() && !info.constp();
        }
        return ok ? info : LeafInfo{};
    }

    // VISITORS
    void visit(AstNode* nodep) override { CONST_BITOP_SET_FAILED("Hit unexpected op", nodep); }
    void visit(AstCCast* nodep) override {
        iterateChildrenConst(nodep);
        if (m_leafp) m_leafp->updateBitRange(nodep);
    }
    void visit(AstShiftR* nodep) override {
        CONST_BITOP_RETURN_IF(!m_leafp, nodep);
        const AstConst* const constp = VN_CAST(nodep->rhsp(), Const);
        CONST_BITOP_RETURN_IF(!constp, nodep->rhsp());
        m_lsb += constp->toUInt();
        incrOps(nodep, __LINE__);
        iterateConst(nodep->lhsp());
        m_leafp->updateBitRange(nodep);
        m_lsb -= constp->toUInt();
    }
    void visit(AstNot* nodep) override {
        CONST_BITOP_RETURN_IF(nodep->widthMin() != 1, nodep);
        AstNode* lhsp = nodep->lhsp();
        AstCCast* const castp = VN_CAST(lhsp, CCast);
        if (castp) lhsp = castp->lhsp();
        CONST_BITOP_RETURN_IF(!isXorTree() && !VN_IS(lhsp, VarRef) && !VN_IS(lhsp, ShiftR), lhsp);
        incrOps(nodep, __LINE__);
        m_polarity = !m_polarity;
        iterateChildrenConst(nodep);
        // Don't restore m_polarity for Xor as it counts parity of the entire tree
        if (!isXorTree()) m_polarity = !m_polarity;
        if (m_leafp && castp) m_leafp->updateBitRange(castp);
        if (m_leafp) m_leafp->polarity(!m_leafp->polarity());
    }
    void visit(AstWordSel* nodep) override {
        CONST_BITOP_RETURN_IF(!m_leafp, nodep);
        const AstConst* const constp = VN_CAST(nodep->bitp(), Const);
        CONST_BITOP_RETURN_IF(!constp, nodep->bitp());
        UASSERT_OBJ(m_leafp->wordIdx() == -1, nodep, "Unexpected nested WordSel");
        m_leafp->wordIdx(constp->toSInt());
        iterateConst(nodep->fromp());
    }
    void visit(AstVarRef* nodep) override {
        CONST_BITOP_RETURN_IF(!m_leafp, nodep);
        m_leafp->setLeaf(nodep);
    }
    void visit(AstConst* nodep) override {
        CONST_BITOP_RETURN_IF(!m_leafp, nodep);
        m_leafp->setLeaf(nodep);
    }

    void visit(AstRedXor* nodep) override {
        Restorer restorer{*this};
        CONST_BITOP_RETURN_IF(!VN_IS(m_rootp, Xor), nodep);
        AstNode* lhsp = nodep->lhsp();
        const AstCCast* const castp = VN_CAST(lhsp, CCast);
        if (castp) lhsp = castp->lhsp();
        if (const AstAnd* const andp = VN_CAST(lhsp, And)) {  // '^(mask & leaf)'
            CONST_BITOP_RETURN_IF(!andp, lhsp);

            const LeafInfo& mask = findLeaf(andp->lhsp(), true);
            CONST_BITOP_RETURN_IF(!mask.constp() || mask.lsb() != 0, andp->lhsp());

            LeafInfo ref = findLeaf(andp->rhsp(), false);
            CONST_BITOP_RETURN_IF(!ref.refp(), andp->rhsp());
            if (castp) ref.updateBitRange(castp);

            restorer.disableRestore();  // Now all subtree succeeded

            const V3Number& maskNum = mask.constp()->num();

            incrOps(nodep, __LINE__);
            incrOps(andp, __LINE__);

            // Mark all bits checked in this reduction
            const int maxBitIdx = std::min(ref.lsb() + maskNum.width(), ref.msb() + 1);
            for (int bitIdx = ref.lsb(); bitIdx < maxBitIdx; ++bitIdx) {
                const int maskIdx = bitIdx - ref.lsb();
                if (maskNum.bitIs0(maskIdx)) continue;
                // Set true, m_polarity takes care of the entire parity
                m_bitPolarities.emplace_back(ref, true, bitIdx);
            }
        } else {  // '^leaf'
            LeafInfo ref = findLeaf(lhsp, false);
            CONST_BITOP_RETURN_IF(!ref.refp(), lhsp);
            if (castp) ref.updateBitRange(castp);

            restorer.disableRestore();  // Now all checks passed

            incrOps(nodep, __LINE__);

            // Mark all bits checked by this comparison
            for (int bitIdx = ref.lsb(); bitIdx <= ref.msb(); ++bitIdx) {
                m_bitPolarities.emplace_back(ref, true, bitIdx);
            }
        }
    }

    void visit(AstNodeBiop* nodep) override {
        if (VN_IS(nodep, And) && isConst(nodep->lhsp(), 1)) {  // 1 & _
            // Always reach past a plain making AND
            Restorer restorer{*this};
            incrOps(nodep, __LINE__);
            iterateConst(nodep->rhsp());
            if (m_leafp) m_leafp->limitBitRangeToLsb();
            CONST_BITOP_RETURN_IF(m_failed, nodep->rhsp());
            restorer.disableRestore();  // Now all checks passed
        } else if (nodep->type() == m_rootp->type()) {  // And, Or, Xor
            // subtree under NOT can be optimized only in XOR tree.
            CONST_BITOP_RETURN_IF(!m_polarity && !isXorTree(), nodep);
            incrOps(nodep, __LINE__);
            for (const bool right : {false, true}) {
                VL_RESTORER(m_leafp);
                Restorer restorer{*this};
                LeafInfo leafInfo{m_lsb};
                // cppcheck-suppress danglingLifetime
                m_leafp = &leafInfo;
                AstNodeExpr* opp = right ? nodep->rhsp() : nodep->lhsp();
                const bool origFailed = m_failed;
                iterateConst(opp);
                if (leafInfo.constp() || m_failed) {
                    // Revert changes in leaf
                    restorer.restoreNow();
                    // Reach past a cast then add to frozen nodes to be added to final reduction
                    if (const AstCCast* const castp = VN_CAST(opp, CCast)) opp = castp->lhsp();
                    const bool pol = isXorTree() || m_polarity;  // Only AND/OR tree needs polarity
                    UASSERT_OBJ(pol, nodep, "AND/OR tree expects m_polarity==true");
                    m_frozenNodes.emplace_back(opp, FrozenNodeInfo{pol, m_lsb});
                    m_failed = origFailed;
                    continue;
                }
                restorer.disableRestore();  // Now all checks passed
                if (leafInfo.refp()) {
                    // The conditional on the lsb being in range is necessary for some degenerate
                    // case, e.g.: (IData)((QData)wide[0] >> 32), or <1-bit-var> >> 1, which is
                    // just zero
                    if (leafInfo.lsb() <= leafInfo.msb()) {
                        m_bitPolarities.emplace_back(leafInfo, isXorTree() || leafInfo.polarity(),
                                                     leafInfo.lsb());
                    } else if ((isAndTree() && leafInfo.polarity())
                               || (isOrTree() && !leafInfo.polarity())) {
                        // If there is a constant 0 term in an And tree or 1 term in an Or tree, we
                        // must include it. Fudge this by adding a bit with both polarities, which
                        // will simplify to zero or one respectively.
                        // Note that Xor tree does not need this kind of care, polarity of Xor tree
                        // is already cared when visitin AstNot. Taking xor with 1'b0 is nop.
                        m_bitPolarities.emplace_back(leafInfo, true, 0);
                        m_bitPolarities.emplace_back(leafInfo, false, 0);
                    }
                }
            }
        } else if ((isAndTree() && VN_IS(nodep, Eq)) || (isOrTree() && VN_IS(nodep, Neq))) {
            Restorer restorer{*this};
            CONST_BITOP_RETURN_IF(!m_polarity, nodep);
            CONST_BITOP_RETURN_IF(m_lsb, nodep);  // the result of EQ/NE is 1 bit width
            const AstNode* lhsp = nodep->lhsp();
            if (const AstCCast* const castp = VN_CAST(lhsp, CCast)) lhsp = castp->lhsp();
            const AstConst* const constp = VN_CAST(lhsp, Const);
            CONST_BITOP_RETURN_IF(!constp, nodep->lhsp());

            const V3Number& compNum = constp->num();

            auto setPolarities = [this, &compNum](const LeafInfo& ref, const V3Number* maskp) {
                const bool maskFlip = isAndTree() ^ ref.polarity();
                int constantWidth = compNum.width();
                if (maskp) constantWidth = std::max(constantWidth, maskp->width());
                const int maxBitIdx = std::max(ref.lsb() + constantWidth, ref.msb() + 1);
                // Mark all bits checked by this comparison
                for (int bitIdx = ref.lsb(); bitIdx < maxBitIdx; ++bitIdx) {
                    const int maskIdx = bitIdx - ref.lsb();
                    const bool mask0 = maskp && maskp->bitIs0(maskIdx);
                    const bool outOfRange = bitIdx > ref.msb();
                    if (mask0 || outOfRange) {  // RHS is 0
                        if (compNum.bitIs1(maskIdx)) {
                            // LHS is 1
                            // And tree: 1 == 0 => always false, set v && !v
                            // Or tree : 1 != 0 => always true, set v || !v
                            m_bitPolarities.emplace_back(ref, true, 0);
                            m_bitPolarities.emplace_back(ref, false, 0);
                            break;
                        } else {  // This bitIdx is irrelevant
                            continue;
                        }
                    }
                    const bool polarity = compNum.bitIs1(maskIdx) != maskFlip;
                    m_bitPolarities.emplace_back(ref, polarity, bitIdx);
                }
            };

            if (const AstAnd* const andp = VN_CAST(nodep->rhsp(), And)) {  // comp == (mask & v)
                const LeafInfo& mask = findLeaf(andp->lhsp(), true);
                CONST_BITOP_RETURN_IF(!mask.constp() || mask.lsb() != 0, andp->lhsp());

                const LeafInfo& ref = findLeaf(andp->rhsp(), false);
                CONST_BITOP_RETURN_IF(!ref.refp(), andp->rhsp());

                restorer.disableRestore();  // Now all checks passed

                const V3Number& maskNum = mask.constp()->num();

                incrOps(nodep, __LINE__);
                incrOps(andp, __LINE__);

                setPolarities(ref, &maskNum);
            } else {  // comp == v
                const LeafInfo& ref = findLeaf(nodep->rhsp(), false);
                CONST_BITOP_RETURN_IF(!ref.refp(), nodep->rhsp());

                restorer.disableRestore();  // Now all checks passed

                incrOps(nodep, __LINE__);

                setPolarities(ref, nullptr);
            }
        } else {
            CONST_BITOP_SET_FAILED("Mixture of different ops cannot be optimized", nodep);
        }
    }

    // CONSTRUCTORS
    ConstBitOpTreeVisitor(AstNodeExpr* nodep, unsigned externalOps)
        : m_ops{externalOps}
        , m_rootp{nodep} {
        // Fill nullptr at [0] because AstVarScope::user4 is 0 by default
        m_varInfos.push_back(nullptr);
        CONST_BITOP_RETURN_IF(!isAndTree() && !isOrTree() && !isXorTree(), nodep);
        if (AstNodeBiop* const biopp = VN_CAST(nodep, NodeBiop)) {
            iterateConst(biopp);
        } else {
            UASSERT_OBJ(VN_IS(nodep, RedXor), nodep, "Must be RedXor");
            incrOps(nodep, __LINE__);
            iterateChildrenConst(nodep);
        }
        for (auto&& entry : m_bitPolarities) {
            getVarInfo(entry.m_info).setPolarity(entry.m_polarity, entry.m_bit);
        }
        UASSERT_OBJ(isXorTree() || m_polarity, nodep, "must be the original polarity");
    }
    virtual ~ConstBitOpTreeVisitor() = default;
#undef CONST_BITOP_RETURN_IF
#undef CONST_BITOP_SET_FAILED

public:
    // Transform as below.
    // v[0] & v[1] => 2'b11 == (2'b11 & v)
    // v[0] | v[1] => 2'b00 != (2'b11 & v)
    // v[0] ^ v[1] => ^{2'b11 & v}
    // (3'b011 == (3'b011 & v)) & v[2]  => 3'b111 == (3'b111 & v)
    // (3'b000 != (3'b011 & v)) | v[2]  => 3'b000 != (3'b111 & v)
    // Reduction ops are transformed in the same way.
    // &{v[0], v[1]} => 2'b11 == (2'b11 & v)
    static AstNodeExpr* simplify(AstNodeExpr* nodep, int resultWidth, unsigned externalOps,
                                 VDouble0& reduction) {
        UASSERT_OBJ(1 <= resultWidth && resultWidth <= 64, nodep, "resultWidth out of range");

        // Walk tree, gathering all terms referenced in expression
        const ConstBitOpTreeVisitor visitor{nodep, externalOps};

        // If failed on root node is not optimizable, or there are no variable terms, then done
        if (visitor.m_failed || visitor.m_varInfos.size() == 1) return nullptr;

        // FileLine used for constructing all new nodes in this function
        FileLine* const fl = nodep->fileline();

        // Get partial result each term referenced, count total number of ops and keep track of
        // whether we have clean/dirty terms. visitor.m_varInfos appears in deterministic order,
        // so the optimized tree is deterministic as well.

        std::vector<AstNodeExpr*> termps;
        termps.reserve(visitor.m_varInfos.size() - 1);
        unsigned resultOps = 0;
        bool hasCleanTerm = false;
        bool hasDirtyTerm = false;

        for (auto&& v : visitor.m_varInfos) {
            if (!v) continue;  // Skip nullptr at m_varInfos[0]
            if (v->hasConstResult()) {
                // If a constant term is known, we can either drop it or the whole tree is constant
                AstNodeExpr* resultp = nullptr;
                if (v->getConstResult()) {
                    UASSERT_OBJ(visitor.isOrTree(), nodep,
                                "Only OR tree can yield known 1 result");
                    UINFO(9, "OR tree with const 1 term: " << v->refp());
                    // Known 1 bit in OR tree, whole result is 1
                    resultp = new AstConst{fl, AstConst::BitTrue{}};
                } else if (visitor.isAndTree()) {
                    UINFO(9, "AND tree with const 0 term: " << v->refp());
                    // Known 0 bit in AND tree, whole result is 0
                    resultp = new AstConst{fl, AstConst::BitFalse{}};
                } else {
                    // Known 0 bit in OR or XOR tree. Ignore it.
                    continue;
                }
                // Set width and widthMin precisely
                resultp->dtypeChgWidth(resultWidth, 1);
                for (AstNode* const termp : termps) VL_DO_DANGLING(termp->deleteTree(), termp);
                return resultp;
            }
            const ResultTerm result = v->getResultTerm();
            termps.push_back(std::get<0>(result));
            resultOps += std::get<1>(result);
            if (std::get<2>(result)) {
                hasCleanTerm = true;
                UINFO(9, "Clean term: " << termps.back());
            } else {
                hasDirtyTerm = true;
                UINFO(9, "Dirty term: " << termps.back());
            }
        }

        // Group by FrozenNodeInfo
        std::map<FrozenNodeInfo, std::vector<AstNodeExpr*>> frozenNodes;
        // Check if frozen terms are clean or not
        for (const auto& frozenInfo : visitor.m_frozenNodes) {
            AstNodeExpr* const termp = frozenInfo.first;
            // Comparison operators are clean
            if ((VN_IS(termp, Eq) || VN_IS(termp, Neq) || VN_IS(termp, Lt) || VN_IS(termp, Lte)
                 || VN_IS(termp, Gt) || VN_IS(termp, Gte))
                && frozenInfo.second.m_lsb == 0) {
                hasCleanTerm = true;
            } else {
                // Otherwise, conservatively assume the frozen term is dirty
                hasDirtyTerm = true;
                UINFO(9, "Dirty frozen term: " << termp);
            }
            frozenNodes[frozenInfo.second].push_back(termp);
        }

        // Figure out if a final negation is required
        const bool needsFlip = visitor.isXorTree() && !visitor.m_polarity;

        // Figure out if the final tree needs cleaning
        const bool needsCleaning = visitor.isAndTree() ? !hasCleanTerm : hasDirtyTerm;

        // Add size of reduction tree to op count
        resultOps += termps.size() - 1;
        for (const auto& lsbAndNodes : frozenNodes) {
            if (lsbAndNodes.first.m_lsb > 0) ++resultOps;  // Needs AstShiftR
            if (!lsbAndNodes.first.m_polarity) ++resultOps;  // Needs AstNot
            resultOps += lsbAndNodes.second.size();
        }
        // Add final polarity flip in Xor tree
        if (needsFlip) ++resultOps;
        // Add final cleaning AND
        if (needsCleaning) ++resultOps;

        if (debug() >= 9) {  // LCOV_EXCL_START
            cout << "-  Bitop tree considered:\n";
            for (const AstNodeExpr* const termp : termps) termp->dumpTree("-  Reduced term: ");
            for (const std::pair<AstNodeExpr*, FrozenNodeInfo>& termp : visitor.m_frozenNodes) {
                termp.first->dumpTree("-  Frozen term with lsb "
                                      + std::to_string(termp.second.m_lsb) + " polarity "
                                      + std::to_string(termp.second.m_polarity) + ": ");
            }
            cout << "-  Needs flipping: " << needsFlip << "\n";
            cout << "-  Needs cleaning: " << needsCleaning << "\n";
            cout << "-  Size: " << resultOps << " input size: " << visitor.m_ops << "\n";
        }  // LCOV_EXCL_STOP

        // Sometimes we have no terms left after ignoring redundant terms
        // (all of which were zeroes)
        if (termps.empty() && visitor.m_frozenNodes.empty()) {
            reduction += visitor.m_ops;
            AstNodeExpr* const resultp = needsFlip ? new AstConst{fl, AstConst::BitTrue{}}
                                                   : new AstConst{fl, AstConst::BitFalse{}};
            resultp->dtypeChgWidth(resultWidth, 1);
            return resultp;
        }

        // Only substitute the result if beneficial as determined by operation count
        if (visitor.m_ops <= resultOps) {
            for (AstNode* const termp : termps) VL_DO_DANGLING(termp->deleteTree(), termp);
            return nullptr;
        }

        // Update statistics
        reduction += visitor.m_ops - resultOps;

        // Reduction op to combine terms
        const auto reduce = [&visitor, fl](AstNodeExpr* lhsp, AstNodeExpr* rhsp) -> AstNodeExpr* {
            if (!lhsp) return rhsp;
            if (visitor.isAndTree()) {
                return new AstAnd{fl, lhsp, rhsp};
            } else if (visitor.isOrTree()) {
                return new AstOr{fl, lhsp, rhsp};
            } else {
                return new AstXor{fl, lhsp, rhsp};
            }
        };

        // Compute result by reducing all terms
        AstNodeExpr* resultp = nullptr;
        for (AstNodeExpr* const termp : termps) {  //
            resultp = reduce(resultp, termp);
        }
        // Add any frozen terms to the reduction
        for (auto&& nodes : frozenNodes) {
            // nodes.second has same lsb and polarity
            AstNodeExpr* termp = nullptr;
            for (AstNodeExpr* const itemp : nodes.second) {
                termp = reduce(termp, itemp->unlinkFrBack());
            }
            if (nodes.first.m_lsb > 0) {  // LSB is not 0, so shiftR
                AstNodeDType* const dtypep = termp->dtypep();
                termp = new AstShiftR{termp->fileline(), termp,
                                      new AstConst(termp->fileline(), AstConst::WidthedValue{},
                                                   termp->width(), nodes.first.m_lsb)};
                termp->dtypep(dtypep);
            }
            if (!nodes.first.m_polarity) {  // Polarity is inverted, so append Not
                AstNodeDType* const dtypep = termp->dtypep();
                termp = new AstNot{termp->fileline(), termp};
                termp->dtypep(dtypep);
            }
            resultp = reduce(resultp, termp);
        }

        // Set width of masks to expected result width. This is required to prevent later removal
        // of the masking node e.g. by the "AND with all ones" rule. If the result width happens
        // to be 1, we still need to ensure the AstAnd is not dropped, so use a wider mask in this
        // special case.
        const int maskWidth
            = std::max(resultp->width(), resultWidth == 1 ? VL_IDATASIZE : resultWidth);

        // Apply final polarity flip
        if (needsFlip) {
            if (needsCleaning) {
                // Cleaning will be added below. Use a NOT which is a byte shorter on x86
                resultp = new AstNot{fl, resultp};
            } else {
                // Keep result clean by using XOR(1, _)
                AstConst* const maskp = new AstConst{fl, AstConst::WidthedValue{}, maskWidth, 1};
                resultp = new AstXor{fl, maskp, resultp};
            }
        }

        // Apply final cleaning
        if (needsCleaning) {
            AstConst* const maskp = new AstConst{fl, AstConst::WidthedValue{}, maskWidth, 1};
            resultp = new AstAnd{fl, maskp, resultp};
        }

        // Cast back to original size if required
        if (resultp->width() != resultWidth) {
            resultp = new AstCCast{fl, resultp, resultWidth, 1};
        }

        // Set width and widthMin precisely
        resultp->dtypeChgWidth(resultWidth, 1);

        return resultp;
    }
};

//######################################################################
// Const state, as a visitor of each AstNode

class ConstVisitor final : public VNVisitor {
    // CONSTANTS
    static constexpr unsigned CONCAT_MERGABLE_MAX_DEPTH = 10;  // Limit alg recursion

    // NODE STATE
    // ** only when m_warn/m_doExpensive is set.  If state is needed other times,
    // ** must track down everywhere V3Const is called and make sure no overlaps.
    // AstVar::user4p           -> Used by variable marking/finding
    // AstEnum::user4           -> bool.  Recursing.

    // STATE
    bool m_params = false;  // If true, propagate parameterized and true numbers only
    bool m_required = false;  // If true, must become a constant
    bool m_wremove = true;  // Inside scope, no assignw removal
    bool m_warn = false;  // Output warnings
    bool m_doExpensive = false;  // Enable computationally expensive optimizations
    bool m_doCpp = false;  // Enable late-stage C++ optimizations
    bool m_doNConst = false;  // Enable non-constant-child simplifications
    bool m_doV = false;  // Verilog, not C++ conversion
    bool m_doGenerate = false;  // Postpone width checking inside generate
    bool m_convertLogicToBit = false;  // Convert logical operators to bitwise
    bool m_hasJumpDelay = false;  // JumpGo or Delay under this loop
    bool m_hasLoopTest = false;  // Contains AstLoopTest
    bool m_underRecFunc = false;  // Under a recursive function
    AstNodeModule* m_modp = nullptr;  // Current module
    const AstArraySel* m_selp = nullptr;  // Current select
    const AstNode* m_scopep = nullptr;  // Current scope
    const AstAttrOf* m_attrp = nullptr;  // Current attribute
    VDouble0 m_statBitOpReduction;  // Ops reduced in ConstBitOpTreeVisitor
    VDouble0 m_statConcatMerge;  // Concat merges
    VDouble0 m_statCondExprRedundant;  // Conditional repeated expressions
    VDouble0 m_statIfCondExprRedundant;  // Conditional repeated expressions
    const bool m_globalPass;  // ConstVisitor invoked as a global pass
    static uint32_t s_globalPassNum;  // Counts number of times ConstVisitor invoked as global pass
    V3UniqueNames m_concswapNames;  // For generating unique temporary variable names
    std::map<const AstNode*, bool> m_containsMemberAccess;  // Caches results of matchBiopToBitwise
    std::unordered_set<AstJumpBlock*> m_usedJumpBlocks;  // JumpBlocks used by some JumpGo

    // METHODS

    V3Number constNumV(AstNode* nodep) {
        // Contract C width to V width (if needed, else just direct copy)
        // The upper zeros in the C representation can otherwise cause
        // wrong results in some operations, e.g. MulS
        const V3Number& numc = VN_AS(nodep, Const)->num();
        return !numc.isNumber() ? numc : V3Number{nodep, nodep->widthMinV(), numc};
    }
    V3Number toNumC(AstNode* nodep, const V3Number& numv) {
        // Extend V width back to C width for given node
        return !numv.isNumber() ? numv : V3Number{nodep, nodep->width(), numv};
    }

    bool operandConst(const AstNode* nodep) { return VN_IS(nodep, Const); }
    bool operandAsvConst(const AstNode* nodep) {
        // BIASV(CONST, BIASV(CONST,...)) -> BIASV( BIASV_CONSTED(a,b), ...)
        const AstNodeBiComAsv* const bnodep = VN_CAST(nodep, NodeBiComAsv);
        if (!bnodep) return false;
        if (!VN_IS(bnodep->lhsp(), Const)) return false;
        const AstNodeBiComAsv* const rnodep = VN_CAST(bnodep->rhsp(), NodeBiComAsv);
        if (!rnodep) return false;
        if (rnodep->type() != bnodep->type()) return false;
        if (rnodep->width() != bnodep->width()) return false;
        if (rnodep->lhsp()->width() != bnodep->lhsp()->width()) return false;
        if (!VN_IS(rnodep->lhsp(), Const)) return false;
        return true;
    }
    bool operandAsvSame(const AstNode* nodep) {
        // BIASV(SAMEa, BIASV(SAMEb,...)) -> BIASV( BIASV(SAMEa,SAMEb), ...)
        const AstNodeBiComAsv* const bnodep = VN_CAST(nodep, NodeBiComAsv);
        if (!bnodep) return false;
        const AstNodeBiComAsv* const rnodep = VN_CAST(bnodep->rhsp(), NodeBiComAsv);
        if (!rnodep) return false;
        if (rnodep->type() != bnodep->type()) return false;
        if (rnodep->width() != bnodep->width()) return false;
        return operandsSame(bnodep->lhsp(), rnodep->lhsp());
    }
    bool operandAsvLUp(const AstNode* nodep) {
        // BIASV(BIASV(CONSTll,lr),r) -> BIASV(CONSTll,BIASV(lr,r)) ?
        //
        // Example of how this is useful:
        // BIASV(BIASV(CONSTa,b...),BIASV(CONSTc,d...))  // hits operandAsvUp
        // BIASV(CONSTa,BIASV(b...,BIASV(CONSTc,d...)))  // hits operandAsvUp
        // BIASV(CONSTa,BIASV(CONSTc,BIASV(c...,d...)))  // hits operandAsvConst
        // BIASV(BIASV(CONSTa,CONSTc),BIASV(c...,d...))) // hits normal constant propagation
        // BIASV(CONST_a_c,BIASV(c...,d...)))
        //
        // Idea for the future: All BiComAsvs could be lists, sorted by if they're constant
        const AstNodeBiComAsv* const bnodep = VN_CAST(nodep, NodeBiComAsv);
        if (!bnodep) return false;
        const AstNodeBiComAsv* const lnodep = VN_CAST(bnodep->lhsp(), NodeBiComAsv);
        if (!lnodep) return false;
        if (lnodep->type() != bnodep->type()) return false;
        if (lnodep->width() != bnodep->width()) return false;
        return VN_IS(lnodep->lhsp(), Const);
    }
    bool operandAsvRUp(const AstNode* nodep) {
        // BIASV(l,BIASV(CONSTrl,rr)) -> BIASV(CONSTrl,BIASV(l,rr)) ?
        const AstNodeBiComAsv* const bnodep = VN_CAST(nodep, NodeBiComAsv);
        if (!bnodep) return false;
        const AstNodeBiComAsv* const rnodep = VN_CAST(bnodep->rhsp(), NodeBiComAsv);
        if (!rnodep) return false;
        if (rnodep->type() != bnodep->type()) return false;
        if (rnodep->width() != bnodep->width()) return false;
        return VN_IS(rnodep->lhsp(), Const);
    }
    static bool operandSubAdd(const AstNode* nodep) {
        // SUB( ADD(CONSTx,y), CONSTz) -> ADD(SUB(CONSTx,CONSTz), y)
        const AstNodeBiop* const np = VN_CAST(nodep, NodeBiop);
        const AstNodeBiop* const lp = VN_CAST(np->lhsp(), NodeBiop);
        return (lp && VN_IS(lp->lhsp(), Const) && VN_IS(np->rhsp(), Const)
                && lp->width() == np->width());
    }
    bool matchRedundantClean(AstAnd* andp) {
        // Remove And with constant one inserted by V3Clean
        // 1 & (a == b)  -> (IData)(a == b)
        // When bool is casted to int, the value is either 0 or 1
        AstConst* const constp = VN_AS(andp->lhsp(), Const);
        UASSERT_OBJ(constp && constp->isOne(), andp->lhsp(), "TRREEOPC must meet this condition");
        AstNodeExpr* const rhsp = andp->rhsp();
        AstCCast* ccastp = nullptr;
        const auto isEqOrNeq = [](const AstNode* nodep) -> bool {  //
            return VN_IS(nodep, Eq) || VN_IS(nodep, Neq);
        };
        if (isEqOrNeq(rhsp)) {
            ccastp = new AstCCast{andp->fileline(), rhsp->unlinkFrBack(), andp};
        } else if (AstCCast* const tmpp = VN_CAST(rhsp, CCast)) {
            if (isEqOrNeq(tmpp->lhsp())) {
                if (tmpp->width() == andp->width()) {
                    tmpp->unlinkFrBack();
                    ccastp = tmpp;
                } else {
                    ccastp = new AstCCast{andp->fileline(), tmpp->lhsp()->unlinkFrBack(), andp};
                }
            }
        }
        if (ccastp) {
            andp->replaceWithKeepDType(ccastp);
            VL_DO_DANGLING(pushDeletep(andp), andp);
            return true;
        }
        return false;
    }

    static bool operandAndOrSame(const AstNode* nodep) {
        // OR( AND(VAL,x), AND(VAL,y)) -> AND(VAL,OR(x,y))
        // OR( AND(x,VAL), AND(y,VAL)) -> AND(OR(x,y),VAL)
        const AstNodeBiop* const np = VN_CAST(nodep, NodeBiop);
        const AstNodeBiop* const lp = VN_CAST(np->lhsp(), NodeBiop);
        const AstNodeBiop* const rp = VN_CAST(np->rhsp(), NodeBiop);
        return (lp && rp && lp->width() == rp->width() && lp->type() == rp->type()
                && (operandsSame(lp->lhsp(), rp->lhsp()) || operandsSame(lp->rhsp(), rp->rhsp())));
    }
    bool matchOrAndNot(AstNodeBiop* nodep) {
        // AstOr{$a, AstAnd{AstNot{$b}, $c}} if $a.width1, $a==$b => AstOr{$a,$c}
        // Someday we'll sort the biops completely and this can be simplified
        // This often results from our simplified clock generation:
        // if (rst) ... else if (enable)... -> OR(rst,AND(!rst,enable))
        const AstNodeExpr* ap;
        AstNodeBiop* andp;
        if (VN_IS(nodep->lhsp(), And)) {
            andp = VN_AS(nodep->lhsp(), And);
            ap = nodep->rhsp();
        } else if (VN_IS(nodep->rhsp(), And)) {
            andp = VN_AS(nodep->rhsp(), And);
            ap = nodep->lhsp();
        } else {
            return false;
        }
        const AstNodeUniop* notp;
        AstNodeExpr* cp;
        if (VN_IS(andp->lhsp(), Not)) {
            notp = VN_AS(andp->lhsp(), Not);
            cp = andp->rhsp();
        } else if (VN_IS(andp->rhsp(), Not)) {
            notp = VN_AS(andp->rhsp(), Not);
            cp = andp->lhsp();
        } else {
            return false;
        }
        const AstNodeExpr* const bp = notp->lhsp();
        if (!operandsSame(ap, bp)) return false;
        // Do it
        cp->unlinkFrBack();
        VL_DO_DANGLING(pushDeletep(andp->unlinkFrBack()), andp);
        VL_DANGLING(notp);
        // Replace whichever branch is now dangling
        if (nodep->rhsp()) {
            nodep->lhsp(cp);
        } else {
            nodep->rhsp(cp);
        }
        return true;
    }
    bool matchAndCond(AstAnd* nodep) {
        // Push down a AND into conditional, when one side of conditional is constant
        // (otherwise we'd be trading one operation for two operations)
        // V3Clean often makes this pattern, as it postpones the AND until
        // as high as possible, which is usually the right choice, except for this.
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstCond* const condp = VN_CAST(nodep->rhsp(), Cond);
        if (!condp) return false;
        if (!VN_IS(condp->thenp(), Const) && !VN_IS(condp->elsep(), Const)) return false;
        AstConst* const maskp = VN_CAST(nodep->lhsp(), Const);
        if (!maskp) return false;
        UINFO(4, "AND(CONSTm, CONDcond(c, i, e))->CONDcond(c, AND(m,i), AND(m, e)) " << nodep);
        AstCond* const newp = new AstCond{
            condp->fileline(), condp->condp()->unlinkFrBack(),
            new AstAnd{nodep->fileline(), maskp->cloneTree(false), condp->thenp()->unlinkFrBack()},
            new AstAnd{nodep->fileline(), maskp->cloneTree(false),
                       condp->elsep()->unlinkFrBack()}};
        newp->thenp()->dtypeFrom(nodep);  // As And might have been to change widths
        newp->elsep()->dtypeFrom(nodep);
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }

    bool matchCondCond(AstCond* nodep) {
        // Same condition on either leg of a condition means that
        // expression is either always true or always false
        if (VN_IS(nodep->backp(), Cond)) return false;  // Was checked when visited parent
        if (!VN_IS(nodep->thenp(), Cond) && !VN_IS(nodep->elsep(), Cond))
            return false;  // Short circuit
        std::vector<AstNodeExpr*> truesp;
        std::vector<AstNodeExpr*> falsesp;
        matchCondCondRecurse(nodep, truesp /*ref*/, falsesp /*ref*/);
        return false;  // Can optimize further
    }
    void matchCondCondRecurse(AstCond* nodep, std::vector<AstNodeExpr*>& truesp,
                              std::vector<AstNodeExpr*>& falsesp) {
        // Avoid O(n^2) compares
        // Could reduce cost with hash table, but seems unlikely to be worth cost
        if (truesp.size() > 4 || falsesp.size() > 4) return;
        if (!nodep->condp()->isPure()) return;
        bool replaced = false;
        for (AstNodeExpr* condp : truesp) {
            if (replaced) break;
            if (!operandsSame(nodep->condp(), condp)) continue;
            UINFO(9, "COND(c, CONDb(c, tt, tf), f) -> CONDb(1, tt, tf) " << nodep);
            replaceNum(nodep->condp(), 1);
            replaced = true;
            ++m_statCondExprRedundant;
        }
        for (AstNodeExpr* condp : falsesp) {
            if (replaced) break;
            if (!operandsSame(nodep->condp(), condp)) continue;
            UINFO(9, "COND(c, t, CONDb(c, ft, ff)) -> CONDb(0, ft, ff) " << nodep);
            replaceZero(nodep->condp());
            replaced = true;
            ++m_statCondExprRedundant;
        }
        if (AstCond* subCondp = VN_CAST(nodep->thenp(), Cond)) {
            if (!replaced) truesp.emplace_back(nodep->condp());
            matchCondCondRecurse(subCondp, truesp /*ref*/, falsesp /*ref*/);
            if (!replaced) truesp.pop_back();
        }
        if (AstCond* subCondp = VN_CAST(nodep->elsep(), Cond)) {
            if (!replaced) falsesp.emplace_back(nodep->condp());
            matchCondCondRecurse(subCondp, truesp /*ref*/, falsesp /*ref*/);
            if (!replaced) falsesp.pop_back();
        }
    }
    void matchIfCondCond(AstNodeIf* nodep) {
        // Same condition on either leg of a condition means that
        // expression is either always true or always false
        if (VN_IS(nodep->backp(), If)) return;  // Was checked when visited parent
        if (!VN_IS(nodep->thensp(), If) && !VN_IS(nodep->elsesp(), If)) return;  // Short circuit
        std::vector<AstNodeExpr*> truesp;
        std::vector<AstNodeExpr*> falsesp;
        matchIfCondCondRecurse(nodep, truesp /*ref*/, falsesp /*ref*/);
    }
    void matchIfCondCondRecurse(AstNodeIf* nodep, std::vector<AstNodeExpr*>& truesp,
                                std::vector<AstNodeExpr*>& falsesp) {
        // Avoid O(n^2) compares
        // Could reduce cost with hash table, but seems unlikely to be worth cost
        if (truesp.size() > 4 || falsesp.size() > 4) return;
        if (!nodep->condp()->isPure()) return;
        bool replaced = false;
        for (AstNodeExpr* condp : truesp) {
            if (replaced) break;
            if (!operandsSame(nodep->condp(), condp)) continue;
            UINFO(9, "COND(c, CONDb(c, tt, tf), f) -> CONDb(1, tt, tf) " << nodep);
            replaceNum(nodep->condp(), 1);
            replaced = true;
            ++m_statIfCondExprRedundant;
        }
        for (AstNodeExpr* condp : falsesp) {
            if (replaced) break;
            if (!operandsSame(nodep->condp(), condp)) continue;
            UINFO(9, "COND(c, t, CONDb(c, ft, ff)) -> CONDb(0, ft, ff) " << nodep);
            replaceZero(nodep->condp());
            replaced = true;
            ++m_statIfCondExprRedundant;
        }
        // We only check the first statement of parent IF is an If
        // So we don't need to check for effects in the executing thensp/elsesp
        // altering the child't condition.  e.g. 'if (x) begin x=1; if (x) end'
        if (AstNodeIf* subIfp = VN_CAST(nodep->thensp(), NodeIf)) {
            if (!replaced) truesp.emplace_back(nodep->condp());
            matchIfCondCondRecurse(subIfp, truesp /*ref*/, falsesp /*ref*/);
            if (!replaced) truesp.pop_back();
        }
        if (AstNodeIf* subIfp = VN_CAST(nodep->elsesp(), NodeIf)) {
            if (!replaced) falsesp.emplace_back(nodep->condp());
            matchIfCondCondRecurse(subIfp, truesp /*ref*/, falsesp /*ref*/);
            if (!replaced) falsesp.pop_back();
        }
    }

    bool matchMaskedOr(AstAnd* nodep) {
        // Masking an OR with terms that have no bits set under the mask is replaced with masking
        // only the remaining terms. Canonical example as generated by V3Expand is:
        // 0xff & (a << 8 | b >> 24) --> 0xff & (b >> 24)

        // Compute how many significant bits are in the mask
        const AstConst* const constp = VN_AS(nodep->lhsp(), Const);
        const uint32_t significantBits = constp->num().widthToFit();

        AstOr* const orp = VN_AS(nodep->rhsp(), Or);

        // Predicate for checking whether the bottom 'significantBits' bits of the given expression
        // are all zeroes.
        const auto checkBottomClear = [=](const AstNode* nodep) -> bool {
            if (const AstShiftL* const shiftp = VN_CAST(nodep, ShiftL)) {
                if (const AstConst* const scp = VN_CAST(shiftp->rhsp(), Const)) {
                    return scp->num().toUInt() >= significantBits;
                }
            }
            return false;
        };

        const bool orLIsRedundant = checkBottomClear(orp->lhsp());
        const bool orRIsRedundant = checkBottomClear(orp->rhsp());

        if (orLIsRedundant && orRIsRedundant) {
            nodep->replaceWithKeepDType(
                new AstConst{nodep->fileline(), AstConst::DTyped{}, nodep->dtypep()});
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
            return true;
        } else if (orLIsRedundant) {
            orp->replaceWithKeepDType(orp->rhsp()->unlinkFrBack());
            VL_DO_DANGLING(pushDeletep(orp), orp);
            return false;  // input node is still valid, keep going
        } else if (orRIsRedundant) {
            orp->replaceWithKeepDType(orp->lhsp()->unlinkFrBack());
            VL_DO_DANGLING(pushDeletep(orp), orp);
            return false;  // input node is still valid, keep going
        } else {
            return false;
        }
    }
    bool matchMaskedShift(AstAnd* nodep) {
        // Drop redundant masking of right shift result. E.g: 0xff & ((uint32_t)a >> 24). This
        // commonly appears after V3Expand and the simplification in matchMaskedOr. Similarly,
        // drop redundant masking of left shift result. E.g.: 0xff000000 & ((uint32_t)a << 24).

        const auto checkMask = [nodep, this](const V3Number& mask) -> bool {
            const AstConst* const constp = VN_AS(nodep->lhsp(), Const);
            if (constp->num().isCaseEq(mask)) {
                AstNode* const rhsp = nodep->rhsp();
                rhsp->unlinkFrBack();
                nodep->replaceWithKeepDType(rhsp);
                VL_DO_DANGLING(pushDeletep(nodep), nodep);
                return true;
            }
            return false;
        };

        // Check if masking is redundant
        if (const AstShiftR* const shiftp = VN_CAST(nodep->rhsp(), ShiftR)) {
            if (const AstConst* const scp = VN_CAST(shiftp->rhsp(), Const)) {
                // Check if mask is full over the non-zero bits
                V3Number maskLo{nodep, nodep->width()};
                maskLo.setMask(nodep->width() - scp->num().toUInt());
                return checkMask(maskLo);
            }
        } else if (const AstShiftL* const shiftp = VN_CAST(nodep->rhsp(), ShiftL)) {
            if (const AstConst* const scp = VN_CAST(shiftp->rhsp(), Const)) {
                // Check if mask is full over the non-zero bits
                V3Number mask{nodep, nodep->width()};
                const uint32_t shiftAmount = scp->num().toUInt();
                mask.setMask(nodep->width() - shiftAmount, shiftAmount);
                return checkMask(mask);
            }
        }
        return false;
    }

    bool matchBitOpTree(AstNodeExpr* nodep) {
        if (nodep->widthMin() != 1) return false;
        if (!v3Global.opt.fConstBitOpTree()) return false;

        string debugPrefix;
        if (debug() >= 9) {  // LCOV_EXCL_START
            static int s_c = 0;
            debugPrefix = "-  matchBitOpTree[";
            debugPrefix += cvtToStr(++s_c);
            debugPrefix += "] ";
            nodep->dumpTree(debugPrefix + "INPUT: ");
        }  // LCOV_EXCL_STOP

        AstNode* newp = nullptr;
        const AstAnd* const andp = VN_CAST(nodep, And);
        const int width = nodep->width();
        if (andp && isConst(andp->lhsp(), 1)) {  // 1 & BitOpTree
            newp = ConstBitOpTreeVisitor::simplify(andp->rhsp(), width, 1, m_statBitOpReduction);
        } else {  // BitOpTree
            newp = ConstBitOpTreeVisitor::simplify(nodep, width, 0, m_statBitOpReduction);
        }

        if (newp) {
            nodep->replaceWithKeepDType(newp);
            UINFO(4, "Transformed leaf of bit tree to " << newp);
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
        }

        if (debug() >= 9) {  // LCOV_EXCL_START
            if (newp) {
                newp->dumpTree(debugPrefix + "RESULT: ");
            } else {
                cout << debugPrefix << "not replaced" << endl;
            }
        }  // LCOV_EXCL_STOP

        return newp;
    }
    static bool operandShiftSame(const AstNode* nodep) {
        const AstNodeBiop* const np = VN_AS(nodep, NodeBiop);
        {
            const AstShiftL* const lp = VN_CAST(np->lhsp(), ShiftL);
            const AstShiftL* const rp = VN_CAST(np->rhsp(), ShiftL);
            if (lp && rp) {
                return (lp->width() == rp->width() && lp->lhsp()->width() == rp->lhsp()->width()
                        && operandsSame(lp->rhsp(), rp->rhsp()));
            }
        }
        {
            const AstShiftR* const lp = VN_CAST(np->lhsp(), ShiftR);
            const AstShiftR* const rp = VN_CAST(np->rhsp(), ShiftR);
            if (lp && rp) {
                return (lp->width() == rp->width() && lp->lhsp()->width() == rp->lhsp()->width()
                        && operandsSame(lp->rhsp(), rp->rhsp()));
            }
        }
        return false;
    }
    bool operandHugeShiftL(const AstNodeBiop* nodep) {
        return (VN_IS(nodep->rhsp(), Const) && !VN_AS(nodep->rhsp(), Const)->num().isFourState()
                && (!VN_AS(nodep->rhsp(), Const)->num().fitsInUInt()  // > 2^32 shift
                    || (VN_AS(nodep->rhsp(), Const)->toUInt()
                        >= static_cast<uint32_t>(nodep->width())))
                && nodep->lhsp()->isPure());
    }
    bool operandHugeShiftR(const AstNodeBiop* nodep) {
        return (VN_IS(nodep->rhsp(), Const) && !VN_AS(nodep->rhsp(), Const)->num().isFourState()
                && (!VN_AS(nodep->rhsp(), Const)->num().fitsInUInt()  // > 2^32 shift
                    || (VN_AS(nodep->rhsp(), Const)->toUInt()
                        >= static_cast<uint32_t>(nodep->lhsp()->width())))
                && nodep->lhsp()->isPure());
    }
    bool operandIsTwo(const AstNode* nodep) {
        const AstConst* const constp = VN_CAST(nodep, Const);
        if (!constp) return false;  // not constant
        if (constp->num().isFourState()) return false;  // four-state
        if (nodep->width() > VL_QUADSIZE) return false;  // too wide
        if (nodep->isSigned() && constp->num().isNegative()) return false;  // signed and negative
        return constp->toUQuad() == 2;
    }
    bool operandIsTwostate(const AstNode* nodep) {
        return (VN_IS(nodep, Const) && !VN_AS(nodep, Const)->num().isFourState());
    }
    bool operandIsPowTwo(const AstNode* nodep) {
        if (!operandIsTwostate(nodep)) return false;
        return (1 == VN_AS(nodep, Const)->num().countOnes());
    }
    bool operandShiftOp(const AstNodeBiop* nodep) {
        if (!VN_IS(nodep->rhsp(), Const)) return false;
        const AstNodeBiop* const lhsp = VN_CAST(nodep->lhsp(), NodeBiop);
        if (!lhsp || !(VN_IS(lhsp, And) || VN_IS(lhsp, Or) || VN_IS(lhsp, Xor))) return false;
        if (nodep->width() != lhsp->width()) return false;
        if (nodep->width() != lhsp->lhsp()->width()) return false;
        if (nodep->width() != lhsp->rhsp()->width()) return false;

        // Only do it if we can fruther reduce one of the sides if pulling the shift through:
        // - Either operands are constants
        if (VN_IS(lhsp->lhsp(), Const)) return true;
        if (VN_IS(lhsp->rhsp(), Const)) return true;
        // - Const shift on either side
        if (AstNodeBiop* const llp = VN_CAST(lhsp->lhsp(), NodeBiop)) {
            return (VN_IS(llp, ShiftL) || VN_IS(llp, ShiftR)) && VN_IS(llp->rhsp(), Const);
        }
        if (AstNodeBiop* const lrp = VN_CAST(lhsp->rhsp(), NodeBiop)) {
            return (VN_IS(lrp, ShiftL) || VN_IS(lrp, ShiftR)) && VN_IS(lrp->rhsp(), Const);
        }
        // Otherwise we would increase logic size
        return false;
    }
    bool operandShiftShift(const AstNodeBiop* nodep) {
        // We could add a AND though.
        const AstNodeBiop* const lhsp = VN_CAST(nodep->lhsp(), NodeBiop);
        if (!lhsp || !(VN_IS(lhsp, ShiftL) || VN_IS(lhsp, ShiftR))) return false;
        // We can only get rid of a<<b>>c or a<<b<<c, with constant b & c
        // because bits may be masked in that process, or (b+c) may exceed the word width.
        if (!(VN_IS(nodep->rhsp(), Const) && VN_IS(lhsp->rhsp(), Const))) return false;
        if (VN_AS(nodep->rhsp(), Const)->num().isFourState()
            || VN_AS(lhsp->rhsp(), Const)->num().isFourState())
            return false;
        if (nodep->width() != lhsp->width()) return false;
        if (nodep->width() != lhsp->lhsp()->width()) return false;
        return true;
    }
    bool operandWordOOB(const AstWordSel* nodep) {
        // V3Expand may make a arraysel that exceeds the bounds of the array
        // It was an expression, then got constified.  In reality, the WordSel
        // must be wrapped in a Cond, that will be false.
        return (VN_IS(nodep->bitp(), Const) && VN_IS(nodep->fromp(), NodeVarRef)
                && VN_AS(nodep->fromp(), NodeVarRef)->access().isReadOnly()
                && (static_cast<int>(VN_AS(nodep->bitp(), Const)->toUInt())
                    >= VN_AS(nodep->fromp(), NodeVarRef)->varp()->widthWords()));
    }
    bool operandSelFull(const AstSel* nodep) {
        return (VN_IS(nodep->lsbp(), Const) && nodep->lsbConst() == 0
                && static_cast<int>(nodep->widthConst()) == nodep->fromp()->width());
    }
    bool operandSelExtend(AstSel* nodep) {
        // A pattern created by []'s after offsets have been removed
        // SEL(EXTEND(any,width,...),(width-1),0) -> ...
        // Since select's return unsigned, this is always an extend
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstExtend* const extendp = VN_CAST(nodep->fromp(), Extend);
        if (!(m_doV && extendp && VN_IS(nodep->lsbp(), Const) && nodep->lsbConst() == 0
              && static_cast<int>(nodep->widthConst()) == extendp->lhsp()->width()))
            return false;
        VL_DO_DANGLING(replaceWChild(nodep, extendp->lhsp()), nodep);
        return true;
    }
    bool operandSelBiLower(AstSel* nodep) {
        // SEL(ADD(a,b),(width-1),0) -> ADD(SEL(a),SEL(b))
        // Add or any operation which doesn't care if we discard top bits
        AstNodeBiop* const bip = VN_CAST(nodep->fromp(), NodeBiop);
        if (!(m_doV && bip && VN_IS(nodep->lsbp(), Const) && nodep->lsbConst() == 0)) return false;
        UINFOTREE(9, nodep, "", "SEL(BI)-in");
        AstNodeExpr* const bilhsp = bip->lhsp()->unlinkFrBack();
        AstNodeExpr* const birhsp = bip->rhsp()->unlinkFrBack();
        bip->lhsp(new AstSel{nodep->fileline(), bilhsp, 0, nodep->widthConst()});
        bip->rhsp(new AstSel{nodep->fileline(), birhsp, 0, nodep->widthConst()});
        UINFOTREE(9, bip, "", "SEL(BI)-ou");
        VL_DO_DANGLING(replaceWChild(nodep, bip), nodep);
        return true;
    }
    bool operandSelShiftLower(AstSel* nodep) {
        // AND({a}, SHIFTR({b}, {c})) is often shorthand in C for Verilog {b}[{c} :+ {a}]
        // becomes thought other optimizations
        // SEL(SHIFTR({a},{b}),{lsb},{width}) -> SEL({a},{lsb+b},{width})
        const AstShiftR* const shiftp = VN_CAST(nodep->fromp(), ShiftR);
        if (!(m_doV && shiftp && VN_IS(shiftp->rhsp(), Const) && VN_IS(nodep->lsbp(), Const))) {
            return false;
        }
        AstNodeExpr* const ap = shiftp->lhsp();
        const AstConst* const bp = VN_AS(shiftp->rhsp(), Const);
        const AstConst* const lp = VN_AS(nodep->lsbp(), Const);
        if (bp->isWide() || bp->num().isFourState() || bp->num().isNegative() || lp->isWide()
            || lp->num().isFourState() || lp->num().isNegative()) {
            return false;
        }
        const int newLsb = lp->toSInt() + bp->toSInt();
        if (newLsb + nodep->widthConst() > ap->width()) return false;
        //
        UINFO(9, "SEL(SHIFTR(a,b),l,w) -> SEL(a,l+b,w)");
        UINFOTREE(9, nodep, "", "SEL(SH)-in");
        AstSel* const newp
            = new AstSel{nodep->fileline(), ap->unlinkFrBack(), newLsb, nodep->widthConst()};
        nodep->replaceWithKeepDType(newp);
        UINFOTREE(9, newp, "", "SEL(SH)-ou");
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }

    bool operandBiExtendConstShrink(AstNodeBiop* nodep) {
        // Loop unrolling favors standalone compares
        // EQ(const{width32}, EXTEND(xx{width3})) -> EQ(const{3}, xx{3})
        // The constant must have zero bits (+ 1 if signed) or compare
        // would be incorrect. See also operandBiExtendConst
        AstExtend* const extendp = VN_CAST(nodep->rhsp(), Extend);
        if (!extendp) return false;
        AstNodeExpr* const smallerp = extendp->lhsp();
        const int subsize = smallerp->width();
        AstConst* const constp = VN_CAST(nodep->lhsp(), Const);
        if (!constp) return false;
        if (!constp->num().isBitsZero(constp->width() - 1, subsize)) return false;
        //
        UINFOTREE(9, nodep, "", "BI(EXTEND)-in");
        smallerp->unlinkFrBack();
        VL_DO_DANGLING(pushDeletep(extendp->unlinkFrBack()), extendp);  // aka nodep->lhsp.
        nodep->rhsp(smallerp);

        constp->unlinkFrBack();
        V3Number num{constp, subsize, constp->num()};
        nodep->lhsp(new AstConst{constp->fileline(), num});
        VL_DO_DANGLING(pushDeletep(constp), constp);
        UINFOTREE(9, nodep, "", "BI(EXTEND)-ou");
        return true;
    }
    bool operandBiExtendConstOver(const AstNodeBiop* nodep) {
        // EQ(const{width32}, EXTEND(xx{width3})) -> constant
        // When the constant has non-zero bits above the extend it's a constant.
        // Avoids compiler warning
        const AstExtend* const extendp = VN_CAST(nodep->rhsp(), Extend);
        if (!extendp) return false;
        const AstNode* const smallerp = extendp->lhsp();
        const int subsize = smallerp->width();
        const AstConst* const constp = VN_CAST(nodep->lhsp(), Const);
        if (!constp) return false;
        if (constp->num().isBitsZero(constp->width() - 1, subsize)) return false;
        return true;
    }

    // Extraction checks
    bool warnSelect(AstSel* nodep) {
        if (m_doGenerate) {
            // Never checked yet
            V3Width::widthParamsEdit(nodep);
            iterateChildren(nodep);  // May need "constifying"
        }
        // Find range of dtype we are selecting from
        // Similar code in V3Unknown::AstSel
        const bool doit = true;
        if (m_warn && VN_IS(nodep->lsbp(), Const) && doit) {
            const int maxDeclBit = nodep->declRange().hiMaxSelect() * nodep->declElWidth()
                                   + (nodep->declElWidth() - 1);
            if (VN_AS(nodep->lsbp(), Const)->num().isFourState()) {
                nodep->v3error("Selection index is constantly unknown or tristated: "
                               "lsb="
                               << nodep->lsbp()->name() << " width=" << nodep->widthConst());
                // Replacing nodep will make a mess above, so we replace the offender
                replaceZero(nodep->lsbp());
            } else if (nodep->declRange().ranged()
                       && (nodep->msbConst() > maxDeclBit || nodep->lsbConst() > maxDeclBit)) {
                // See also warning in V3Width
                // Must adjust by element width as declRange() is in number of elements
                string msbLsbProtected;
                if (nodep->declElWidth() == 0) {
                    msbLsbProtected = "(nodep->declElWidth() == 0) "
                                      + std::to_string(nodep->msbConst()) + ":"
                                      + std::to_string(nodep->lsbConst());
                } else {
                    msbLsbProtected = std::to_string(nodep->msbConst() / nodep->declElWidth())
                                      + ":"
                                      + std::to_string(nodep->lsbConst() / nodep->declElWidth());
                }
                nodep->v3warn(SELRANGE,
                              "Selection index out of range: "
                                  << msbLsbProtected << " outside "
                                  << nodep->declRange().hiMaxSelect() << ":0"
                                  << (nodep->declRange().lo() >= 0
                                          ? ""
                                          : (" (adjusted +" + cvtToStr(-nodep->declRange().lo())
                                             + " to account for negative lsb)")));
                UINFO(1, "    Related Raw index is " << nodep->msbConst() << ":"
                                                     << nodep->lsbConst());
                // Don't replace with zero, we'll do it later
            }
        }
        return false;  // Not a transform, so NOP
    }

    static bool operandsSame(const AstNode* node1p, const AstNode* node2p) {
        // For now we just detect constants & simple vars, though it could be more generic
        if (VN_IS(node1p, Const) && VN_IS(node2p, Const)) return node1p->sameGateTree(node2p);
        if (VN_IS(node1p, VarRef) && VN_IS(node2p, VarRef)) {
            // Avoid comparing widthMin's, which results in lost optimization attempts
            // If cleanup sameGateTree to be smarter, this can be restored.
            // return node1p->sameGateTree(node2p);
            return node1p->isSame(node2p);
        }
        // Pattern created by coverage-line; avoid compiler tautological-compare warning
        if (const AstAnd* const and1p = VN_CAST(node1p, And)) {
            if (const AstAnd* const and2p = VN_CAST(node2p, And)) {
                if (VN_IS(and1p->lhsp(), Const) && VN_IS(and1p->rhsp(), NodeVarRef)
                    && VN_IS(and2p->lhsp(), Const) && VN_IS(and2p->rhsp(), NodeVarRef))
                    return node1p->sameGateTree(node2p);
            }
        }
        return false;
    }
    bool ifSameAssign(const AstNodeIf* nodep) {
        const AstNodeAssign* const thensp = VN_CAST(nodep->thensp(), NodeAssign);
        const AstNodeAssign* const elsesp = VN_CAST(nodep->elsesp(), NodeAssign);
        if (!thensp || thensp->nextp()) return false;  // Must be SINGLE statement
        if (!elsesp || elsesp->nextp()) return false;
        if (thensp->type() != elsesp->type()) return false;  // Can't mix an assigndly with assign
        if (!thensp->lhsp()->sameGateTree(elsesp->lhsp())) return false;
        if (!thensp->rhsp()->gateTree()) return false;
        if (!elsesp->rhsp()->gateTree()) return false;
        if (m_underRecFunc) return false;  // This optimization may lead to infinite recursion
        return true;
    }
    bool operandIfIf(const AstNodeIf* nodep) {
        if (nodep->elsesp()) return false;
        const AstNodeIf* const lowerIfp = VN_CAST(nodep->thensp(), NodeIf);
        if (!lowerIfp || lowerIfp->nextp()) return false;
        if (nodep->type() != lowerIfp->type()) return false;
        if (AstNode::afterCommentp(lowerIfp->elsesp())) return false;
        return true;
    }
    bool ifConcatMergeableBiop(const AstNode* nodep) {
        return (VN_IS(nodep, And) || VN_IS(nodep, Or) || VN_IS(nodep, Xor));
    }
    bool ifAdjacentSel(const AstSel* lhsp, const AstSel* rhsp) {
        if (!v3Global.opt.fAssemble()) return false;  // opt disabled
        if (!lhsp || !rhsp) return false;
        const AstNode* const lfromp = lhsp->fromp();
        const AstNode* const rfromp = rhsp->fromp();
        if (!lfromp || !rfromp || !lfromp->sameGateTree(rfromp)) return false;
        const AstConst* const lstart = VN_CAST(lhsp->lsbp(), Const);
        const AstConst* const rstart = VN_CAST(rhsp->lsbp(), Const);
        if (!lstart || !rstart) return false;  // too complicated
        const int rend = (rstart->toSInt() + rhsp->widthConst());
        return (rend == lstart->toSInt());
    }
    bool ifMergeAdjacent(const AstNodeExpr* lhsp, const AstNodeExpr* rhsp) {
        // Called by concatmergeable to determine if {lhsp, rhsp} make sense
        if (!v3Global.opt.fAssemble()) return false;  // opt disabled
        // Two same varref
        if (operandsSame(lhsp, rhsp)) return true;
        const AstSel* const lselp = VN_CAST(lhsp, Sel);
        const AstSel* const rselp = VN_CAST(rhsp, Sel);
        if (!lselp && !rselp) return false;
        if (lselp && !VN_IS(lselp->lsbp(), Const)) return false;
        if (rselp && !VN_IS(rselp->lsbp(), Const)) return false;
        // a[i:j] {a[j-1:k], b}
        if (lselp && !rselp && VN_IS(rhsp, Concat))
            return ifMergeAdjacent(lhsp, VN_AS(rhsp, Concat)->lhsp());
        // {b, a[j:k]} a[k-1:i]
        if (rselp && !lselp && VN_IS(lhsp, Concat))
            return ifMergeAdjacent(VN_AS(lhsp, Concat)->rhsp(), rhsp);
        // a a[msb:j]
        const AstNodeExpr* const lfromp = lselp                                ? lselp->fromp()
                                          : lhsp->sameGateTree(rselp->fromp()) ? lhsp
                                                                               : nullptr;
        // a[i:0] a
        const AstNodeExpr* const rfromp = rselp                                ? rselp->fromp()
                                          : rhsp->sameGateTree(lselp->fromp()) ? rhsp
                                                                               : nullptr;
        if (!lfromp || !rfromp || !lfromp->sameGateTree(rfromp)) return false;

        const int32_t lstart = lselp ? lselp->lsbConst() : 0;
        const int32_t rstart = rselp ? rselp->lsbConst() : 0;
        const int32_t rend = rstart + (rselp ? rselp->widthConst() : rhsp->width());
        // a[i:j] a[j-1:k]
        if (rend == lstart) return true;
        // a[i:0] a[msb:j]
        if (rend == rfromp->width() && lstart == 0) return true;
        return false;
    }
    bool concatMergeable(const AstNodeExpr* lhsp, const AstNodeExpr* rhsp, unsigned depth) {
        // Determine if {a OP b, c OP d} => {a, c} OP {b, d} is advantageous
        if (!v3Global.opt.fAssemble()) return false;  // opt disabled
        if (lhsp->type() != rhsp->type()) return false;
        if (!ifConcatMergeableBiop(lhsp)) return false;
        if (depth > CONCAT_MERGABLE_MAX_DEPTH) return false;  // As worse case O(n^2) algorithm

        const AstNodeBiop* const lp = VN_CAST(lhsp, NodeBiop);
        const AstNodeBiop* const rp = VN_CAST(rhsp, NodeBiop);
        if (!lp || !rp) return false;
        // {a[]&b[], a[]&b[]}
        const bool lad = ifMergeAdjacent(lp->lhsp(), rp->lhsp());
        const bool rad = ifMergeAdjacent(lp->rhsp(), rp->rhsp());
        if (lad && rad) return true;
        // {a[] & b[]&c[], a[] & b[]&c[]}
        if (lad && concatMergeable(lp->rhsp(), rp->rhsp(), depth + 1)) return true;
        // {a[]&b[] & c[], a[]&b[] & c[]}
        if (rad && concatMergeable(lp->lhsp(), rp->lhsp(), depth + 1)) return true;
        // {(a[]&b[])&(c[]&d[]), (a[]&b[])&(c[]&d[])}
        if (concatMergeable(lp->lhsp(), rp->lhsp(), depth + 1)
            && concatMergeable(lp->rhsp(), rp->rhsp(), depth + 1)) {
            return true;
        }
        return false;
    }
    static bool operandsSameWidth(const AstNode* lhsp, const AstNode* rhsp) {
        return lhsp->width() == rhsp->width();
    }

    //----------------------------------------
    // Constant Replacement functions.
    // These all take a node, delete its tree, and replaces it with a constant

    void replaceNum(AstNode* oldp, const V3Number& num) {
        // Replace oldp node with a constant set to specified value
        UASSERT(oldp, "Null old");
        UASSERT_OBJ(!(VN_IS(oldp, Const) && !VN_AS(oldp, Const)->num().isFourState()), oldp,
                    "Already constant??");
        AstNode* const newp = new AstConst{oldp->fileline(), num};
        oldp->replaceWithKeepDType(newp);
        UINFOTREE(6, oldp, "", "const_old");
        UINFOTREE(6, newp, "", "_new");
        VL_DO_DANGLING(pushDeletep(oldp), oldp);
    }
    void replaceNum(AstNode* nodep, uint32_t val) {
        V3Number num{nodep, nodep->width(), val};
        VL_DO_DANGLING(replaceNum(nodep, num), nodep);
    }
    void replaceNumSigned(AstNodeBiop* nodep, uint32_t val) {
        // We allow both sides to be constant, as one may have come from
        // parameter propagation, etc.
        if (m_warn && !(VN_IS(nodep->lhsp(), Const) && VN_IS(nodep->rhsp(), Const))) {
            nodep->v3warn(UNSIGNED, "Comparison is constant due to unsigned arithmetic");
        }
        VL_DO_DANGLING(replaceNum(nodep, val), nodep);
    }
    void replaceNumLimited(AstNodeBiop* nodep, uint32_t val) {
        // Avoids gcc warning about same
        if (m_warn) nodep->v3warn(CMPCONST, "Comparison is constant due to limited range");
        VL_DO_DANGLING(replaceNum(nodep, val), nodep);
    }
    void replaceZero(AstNode* nodep) { VL_DO_DANGLING(replaceNum(nodep, 0), nodep); }
    void replaceZeroChkPure(AstNode* nodep, AstNodeExpr* checkp) {
        // For example, "0 * n" -> 0 if n has no side effects
        // Else strength reduce it to 0 & n.
        // If ever change the operation note AstAnd rule specially ignores this created pattern
        if (checkp->isPure()) {
            VL_DO_DANGLING(replaceNum(nodep, 0), nodep);
        } else {
            AstNode* const newp = new AstAnd{nodep->fileline(), new AstConst{nodep->fileline(), 0},
                                             checkp->unlinkFrBack()};
            nodep->replaceWithKeepDType(newp);
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
        }
    }
    void replaceAllOnes(AstNode* nodep) {
        V3Number ones{nodep, nodep->width(), 0};
        ones.setMask(nodep->width());
        VL_DO_DANGLING(replaceNum(nodep, ones), nodep);
    }
    void replaceConst(AstNodeUniop* nodep) {
        V3Number numv{nodep, nodep->widthMinV()};
        nodep->numberOperate(numv, constNumV(nodep->lhsp()));
        const V3Number& num = toNumC(nodep, numv);
        UINFO(4, "UNICONST -> " << num);
        VL_DO_DANGLING(replaceNum(nodep, num), nodep);
    }
    void replaceConst(AstNodeBiop* nodep) {
        V3Number numv{nodep, nodep->widthMinV()};
        nodep->numberOperate(numv, constNumV(nodep->lhsp()), constNumV(nodep->rhsp()));
        const V3Number& num = toNumC(nodep, numv);
        UINFO(4, "BICONST -> " << num);
        VL_DO_DANGLING(replaceNum(nodep, num), nodep);
    }
    void replaceConst(AstNodeTriop* nodep) {
        V3Number numv{nodep, nodep->widthMinV()};
        nodep->numberOperate(numv, constNumV(nodep->lhsp()), constNumV(nodep->rhsp()),
                             constNumV(nodep->thsp()));
        const V3Number& num = toNumC(nodep, numv);
        UINFO(4, "TRICONST -> " << num);
        VL_DO_DANGLING(replaceNum(nodep, num), nodep);
    }
    void replaceConst(AstNodeQuadop* nodep) {
        V3Number numv{nodep, nodep->widthMinV()};
        nodep->numberOperate(numv, constNumV(nodep->lhsp()), constNumV(nodep->rhsp()),
                             constNumV(nodep->thsp()), constNumV(nodep->fhsp()));
        const V3Number& num = toNumC(nodep, numv);
        UINFO(4, "QUADCONST -> " << num);
        VL_DO_DANGLING(replaceNum(nodep, num), nodep);
    }

    void replaceConstString(AstNode* oldp, const string& num) {
        // Replace oldp node with a constant set to specified value
        UASSERT(oldp, "Null old");
        AstNode* const newp = new AstConst{oldp->fileline(), AstConst::String{}, num};
        UINFOTREE(6, oldp, "", "const_old");
        UINFOTREE(6, newp, "", "_new");
        oldp->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(oldp), oldp);
    }
    //----------------------------------------
    // Replacement functions.
    // These all take a node and replace it with something else

    void replaceWChild(AstNode* nodep, AstNodeExpr* childp) {
        // NODE(..., CHILD(...)) -> CHILD(...)
        childp->unlinkFrBackWithNext();
        // If replacing a SEL for example, the data type comes from the parent (is less wide).
        // This may adversely affect the operation of the node being replaced.
        nodep->replaceWithKeepDType(childp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceWChildBool(AstNode* nodep, AstNodeExpr* childp) {
        // NODE(..., CHILD(...)) -> REDOR(CHILD(...))
        childp->unlinkFrBack();
        if (childp->width1()) {
            nodep->replaceWithKeepDType(childp);
        } else {
            nodep->replaceWithKeepDType(new AstRedOr{childp->fileline(), childp});
        }
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }

    //! Replace a ternary node with its RHS after iterating
    //! Used with short-circuiting, where the RHS has not yet been iterated.
    void replaceWIteratedRhs(AstNodeTriop* nodep) {
        if (AstNode* const rhsp = nodep->rhsp()) iterateAndNextNull(rhsp);
        replaceWChild(nodep, nodep->rhsp());  // May have changed
    }

    //! Replace a ternary node with its THS after iterating
    //! Used with short-circuiting, where the THS has not yet been iterated.
    void replaceWIteratedThs(AstNodeTriop* nodep) {
        if (AstNode* const thsp = nodep->thsp()) iterateAndNextNull(thsp);
        replaceWChild(nodep, nodep->thsp());  // May have changed
    }
    void replaceWLhs(AstNodeUniop* nodep) {
        // Keep LHS, remove RHS
        replaceWChild(nodep, nodep->lhsp());
    }
    void replaceWLhs(AstNodeBiop* nodep) {
        // Keep LHS, remove RHS
        replaceWChild(nodep, nodep->lhsp());
    }
    void replaceWRhs(AstNodeBiop* nodep) {
        // Keep RHS, remove LHS
        replaceWChild(nodep, nodep->rhsp());
    }
    void replaceWLhsBool(AstNodeBiop* nodep) { replaceWChildBool(nodep, nodep->lhsp()); }
    void replaceWRhsBool(AstNodeBiop* nodep) { replaceWChildBool(nodep, nodep->rhsp()); }
    void replaceAsv(AstNodeBiop* nodep) {
        // BIASV(CONSTa, BIASV(CONSTb, c)) -> BIASV( BIASV_CONSTED(a,b), c)
        // BIASV(SAMEa,  BIASV(SAMEb, c))  -> BIASV( BIASV(SAMEa,SAMEb), c)
        // UINFOTREE(1, nodep, "", "repAsvConst_old");
        AstNodeExpr* const ap = nodep->lhsp();
        AstNodeBiop* const rp = VN_AS(nodep->rhsp(), NodeBiop);
        AstNodeExpr* const bp = rp->lhsp();
        AstNodeExpr* const cp = rp->rhsp();
        ap->unlinkFrBack();
        bp->unlinkFrBack();
        cp->unlinkFrBack();
        rp->unlinkFrBack();
        nodep->lhsp(rp);
        nodep->rhsp(cp);
        rp->lhsp(ap);
        rp->rhsp(bp);
        rp->dtypeFrom(nodep);  // Upper widthMin more likely correct
        if (VN_IS(rp->lhsp(), Const) && VN_IS(rp->rhsp(), Const)) replaceConst(rp);
        // UINFOTREE(1, nodep, "", "repAsvConst_new");
    }
    void replaceAsvLUp(AstNodeBiop* nodep) {
        // BIASV(BIASV(CONSTll,lr),r) -> BIASV(CONSTll,BIASV(lr,r))
        AstNodeBiop* const lp = VN_AS(nodep->lhsp()->unlinkFrBack(), NodeBiop);
        AstNodeExpr* const llp = lp->lhsp()->unlinkFrBack();
        AstNodeExpr* const lrp = lp->rhsp()->unlinkFrBack();
        AstNodeExpr* const rp = nodep->rhsp()->unlinkFrBack();
        nodep->lhsp(llp);
        nodep->rhsp(lp);
        lp->lhsp(lrp);
        lp->rhsp(rp);
        lp->dtypeFrom(nodep);  // Upper widthMin more likely correct
        // UINFOTREE(1, nodep, "", "repAsvLUp_new");
    }
    void replaceAsvRUp(AstNodeBiop* nodep) {
        // BIASV(l,BIASV(CONSTrl,rr)) -> BIASV(CONSTrl,BIASV(l,rr))
        AstNodeExpr* const lp = nodep->lhsp()->unlinkFrBack();
        AstNodeBiop* const rp = VN_AS(nodep->rhsp()->unlinkFrBack(), NodeBiop);
        AstNodeExpr* const rlp = rp->lhsp()->unlinkFrBack();
        AstNodeExpr* const rrp = rp->rhsp()->unlinkFrBack();
        nodep->lhsp(rlp);
        nodep->rhsp(rp);
        rp->lhsp(lp);
        rp->rhsp(rrp);
        rp->dtypeFrom(nodep);  // Upper widthMin more likely correct
        // UINFOTREE(1, nodep, "", "repAsvRUp_new");
    }
    void replaceAndOr(AstNodeBiop* nodep) {
        //  OR  (AND (CONSTll,lr), AND(CONSTrl==ll,rr))    -> AND (CONSTll, OR(lr,rr))
        //  OR  (AND (CONSTll,lr), AND(CONSTrl,    rr=lr)) -> AND (OR(ll,rl), rr)
        // nodep ^lp  ^llp   ^lrp  ^rp  ^rlp       ^rrp
        // (Or/And may also be reversed)
        AstNodeBiop* const lp = VN_AS(nodep->lhsp()->unlinkFrBack(), NodeBiop);
        AstNodeExpr* const llp = lp->lhsp()->unlinkFrBack();
        AstNodeExpr* const lrp = lp->rhsp()->unlinkFrBack();
        AstNodeBiop* const rp = VN_AS(nodep->rhsp()->unlinkFrBack(), NodeBiop);
        AstNodeExpr* const rlp = rp->lhsp()->unlinkFrBack();
        AstNodeExpr* const rrp = rp->rhsp()->unlinkFrBack();
        nodep->replaceWithKeepDType(lp);
        if (operandsSame(llp, rlp)) {
            lp->lhsp(llp);
            lp->rhsp(nodep);
            lp->dtypeFrom(nodep);
            nodep->lhsp(lrp);
            nodep->rhsp(rrp);
            VL_DO_DANGLING(pushDeletep(rp), rp);
            VL_DO_DANGLING(pushDeletep(rlp), rlp);
        } else if (operandsSame(lrp, rrp)) {
            lp->lhsp(nodep);
            lp->rhsp(rrp);
            lp->dtypeFrom(nodep);
            nodep->lhsp(llp);
            nodep->rhsp(rlp);
            VL_DO_DANGLING(pushDeletep(rp), rp);
            VL_DO_DANGLING(pushDeletep(lrp), lrp);
        } else {
            nodep->v3fatalSrc("replaceAndOr on something operandAndOrSame shouldn't have matched");
        }
        // UINFOTREE(1, nodep, "", "repAndOr_new");
    }
    void replaceShiftSame(AstNodeBiop* nodep) {
        // Or(Shift(ll,CONSTlr),Shift(rl,CONSTrr==lr)) -> Shift(Or(ll,rl),CONSTlr)
        // (Or/And may also be reversed)
        AstNodeBiop* const lp = VN_AS(nodep->lhsp()->unlinkFrBack(), NodeBiop);
        AstNodeExpr* const llp = lp->lhsp()->unlinkFrBack();
        AstNodeExpr* const lrp = lp->rhsp()->unlinkFrBack();
        AstNodeBiop* const rp = VN_AS(nodep->rhsp()->unlinkFrBack(), NodeBiop);
        AstNodeExpr* const rlp = rp->lhsp()->unlinkFrBack();
        AstNodeExpr* const rrp = rp->rhsp()->unlinkFrBack();
        nodep->replaceWithKeepDType(lp);
        lp->lhsp(nodep);
        lp->rhsp(lrp);
        nodep->lhsp(llp);
        nodep->rhsp(rlp);
        nodep->dtypep(llp->dtypep());  // dtype of Biop is before shift.
        VL_DO_DANGLING(pushDeletep(rp), rp);
        VL_DO_DANGLING(pushDeletep(rrp), rrp);
        // UINFOTREE(1, nodep, "", "repShiftSame_new");
    }
    void replaceConcatSel(AstConcat* nodep) {
        // {a[1], a[0]} -> a[1:0]
        AstSel* const lselp = VN_AS(nodep->lhsp()->unlinkFrBack(), Sel);
        AstSel* const rselp = VN_AS(nodep->rhsp()->unlinkFrBack(), Sel);
        const int lstart = lselp->lsbConst();
        const int lwidth = lselp->widthConst();
        const int rstart = rselp->lsbConst();
        const int rwidth = rselp->widthConst();

        UASSERT_OBJ((rstart + rwidth) == lstart, nodep,
                    "tried to merge two selects which are not adjacent");
        AstSel* const newselp = new AstSel{
            lselp->fromp()->fileline(), rselp->fromp()->unlinkFrBack(), rstart, lwidth + rwidth};
        UINFO(5, "merged two adjacent sel " << lselp << " and " << rselp << " to one " << newselp);

        nodep->replaceWithKeepDType(newselp);
        VL_DO_DANGLING(pushDeletep(lselp), lselp);
        VL_DO_DANGLING(pushDeletep(rselp), rselp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceConcatMerge(AstConcat* nodep) {
        // {llp OP lrp, rlp OP rrp} => {llp, rlp} OP {lrp, rrp}, where OP = AND/OR/XOR
        AstNodeBiop* const lp = VN_AS(nodep->lhsp(), NodeBiop);
        AstNodeBiop* const rp = VN_AS(nodep->rhsp(), NodeBiop);
        if (concatMergeable(lp, rp, 0)) {
            AstNodeExpr* const llp = lp->lhsp();
            AstNodeExpr* const lrp = lp->rhsp();
            AstNodeExpr* const rlp = rp->lhsp();
            AstNodeExpr* const rrp = rp->rhsp();
            AstConcat* const newlp = new AstConcat{rlp->fileline(), llp->cloneTreePure(false),
                                                   rlp->cloneTreePure(false)};
            AstConcat* const newrp = new AstConcat{rrp->fileline(), lrp->cloneTreePure(false),
                                                   rrp->cloneTreePure(false)};
            // use the lhs to replace the parent concat
            llp->replaceWith(newlp);
            VL_DO_DANGLING(pushDeletep(llp), llp);
            lrp->replaceWith(newrp);
            VL_DO_DANGLING(pushDeletep(lrp), lrp);
            lp->dtypeChgWidthSigned(newlp->width(), newlp->width(), VSigning::UNSIGNED);
            UINFO(5, "merged " << nodep);
            ++m_statConcatMerge;
            VL_DO_DANGLING(pushDeletep(rp->unlinkFrBack()), rp);
            nodep->replaceWithKeepDType(lp->unlinkFrBack());
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
            iterate(lp->lhsp());
            iterate(lp->rhsp());
        } else {
            nodep->v3fatalSrc("tried to merge two Concat which are not adjacent");
        }
    }
    void replaceExtend(AstNode* nodep, AstNodeExpr* arg0p) {
        // -> EXTEND(nodep)
        // like a AstExtend{$rhsp}, but we need to set the width correctly from base node
        arg0p->unlinkFrBack();
        AstNodeExpr* const newp
            = (VN_IS(nodep, ExtendS)
                   ? static_cast<AstNodeExpr*>(new AstExtendS{nodep->fileline(), arg0p})
                   : static_cast<AstNodeExpr*>(new AstExtend{nodep->fileline(), arg0p}));
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replacePowShift(AstNodeBiop* nodep) {  // Pow or PowS
        UINFO(5, "POW(2,b)->SHIFTL(1,b) " << nodep);
        AstNodeExpr* const rhsp = nodep->rhsp()->unlinkFrBack();
        AstShiftL* const newp
            = new AstShiftL{nodep->fileline(), new AstConst{nodep->fileline(), 1}, rhsp};
        newp->lhsp()->dtypeFrom(nodep);
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceMulShift(AstMul* nodep) {  // Mul, but not MulS as not simple shift
        UINFO(5, "MUL(2^n,b)->SHIFTL(b,n) " << nodep);
        const int amount = VN_AS(nodep->lhsp(), Const)->num().mostSetBitP1() - 1;  // 2^n->n+1
        AstNodeExpr* const opp = nodep->rhsp()->unlinkFrBack();
        AstShiftL* const newp
            = new AstShiftL{nodep->fileline(), opp, new AstConst(nodep->fileline(), amount)};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceDivShift(AstDiv* nodep) {  // Mul, but not MulS as not simple shift
        UINFO(5, "DIV(b,2^n)->SHIFTR(b,n) " << nodep);
        const int amount = VN_AS(nodep->rhsp(), Const)->num().mostSetBitP1() - 1;  // 2^n->n+1
        AstNodeExpr* const opp = nodep->lhsp()->unlinkFrBack();
        AstShiftR* const newp
            = new AstShiftR{nodep->fileline(), opp, new AstConst(nodep->fileline(), amount)};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceModAnd(AstModDiv* nodep) {  // Mod, but not ModS as not simple shift
        UINFO(5, "MOD(b,2^n)->AND(b,2^n-1) " << nodep);
        const int amount = VN_AS(nodep->rhsp(), Const)->num().mostSetBitP1() - 1;  // 2^n->n+1
        V3Number mask{nodep, nodep->width()};
        mask.setMask(amount);
        AstNodeExpr* const opp = nodep->lhsp()->unlinkFrBack();
        AstAnd* const newp
            = new AstAnd{nodep->fileline(), opp, new AstConst{nodep->fileline(), mask}};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceShiftOp(AstNodeBiop* nodep) {
        UINFO(5, "SHIFT(AND(a,b),CONST) with a/b special ->AND(SHIFT(a,CONST),SHIFT(b,CONST)) "
                     << nodep);
        const int width = nodep->width();
        const int widthMin = nodep->widthMin();
        VNRelinker handle;
        nodep->unlinkFrBack(&handle);
        AstNodeBiop* const lhsp = VN_AS(nodep->lhsp(), NodeBiop);
        lhsp->unlinkFrBack();
        AstNodeExpr* const shiftp = nodep->rhsp()->unlinkFrBack();
        AstNodeExpr* const ap = lhsp->lhsp()->unlinkFrBack();
        AstNodeExpr* const bp = lhsp->rhsp()->unlinkFrBack();
        AstNodeBiop* const shift1p = nodep;
        AstNodeBiop* const shift2p = nodep->cloneTree(true);
        shift1p->lhsp(ap);
        shift1p->rhsp(shiftp->cloneTreePure(true));
        shift2p->lhsp(bp);
        shift2p->rhsp(shiftp);
        AstNodeBiop* const newp = lhsp;
        newp->lhsp(shift1p);
        newp->rhsp(shift2p);
        newp->dtypeChgWidth(width, widthMin);  // The new AND must have width of the original SHIFT
        handle.relink(newp);
        iterate(newp);  // Further reduce, either node may have more reductions.
    }
    void replaceShiftShift(AstNodeBiop* nodep) {
        UINFO(4, "SHIFT(SHIFT(a,s1),s2)->SHIFT(a,ADD(s1,s2)) " << nodep);
        UINFOTREE(9, nodep, "", "repShiftShift_old");
        AstNodeBiop* const lhsp = VN_AS(nodep->lhsp(), NodeBiop);
        lhsp->unlinkFrBack();
        AstNodeExpr* const ap = lhsp->lhsp()->unlinkFrBack();
        AstNodeExpr* const shift1p = lhsp->rhsp()->unlinkFrBack();
        AstNodeExpr* const shift2p = nodep->rhsp()->unlinkFrBack();
        // Shift1p and shift2p may have different sizes, both are
        // self-determined so sum with infinite width
        if (nodep->type() == lhsp->type()) {
            const int shift1 = VN_AS(shift1p, Const)->toUInt();
            const int shift2 = VN_AS(shift2p, Const)->toUInt();
            const int newshift = shift1 + shift2;
            VL_DO_DANGLING(pushDeletep(shift1p), shift1p);
            VL_DO_DANGLING(pushDeletep(shift2p), shift2p);
            nodep->lhsp(ap);
            nodep->rhsp(new AstConst(nodep->fileline(), newshift));
            iterate(nodep);  // Further reduce, either node may have more reductions.
        } else {
            // We know shift amounts are constant, but might be a mixed left/right shift
            int shift1 = VN_AS(shift1p, Const)->toUInt();
            if (VN_IS(lhsp, ShiftR)) shift1 = -shift1;
            int shift2 = VN_AS(shift2p, Const)->toUInt();
            if (VN_IS(nodep, ShiftR)) shift2 = -shift2;
            const int newshift = shift1 + shift2;
            VL_DO_DANGLING(pushDeletep(shift1p), shift1p);
            VL_DO_DANGLING(pushDeletep(shift2p), shift2p);
            AstNodeExpr* newp;
            V3Number mask1{nodep, nodep->width()};
            V3Number ones{nodep, nodep->width()};
            ones.setMask(nodep->width());
            if (shift1 < 0) {
                mask1.opShiftR(ones, V3Number(nodep, VL_IDATASIZE, -shift1));
            } else {
                mask1.opShiftL(ones, V3Number(nodep, VL_IDATASIZE, shift1));
            }
            V3Number mask{nodep, nodep->width()};
            if (shift2 < 0) {
                mask.opShiftR(mask1, V3Number(nodep, VL_IDATASIZE, -shift2));
            } else {
                mask.opShiftL(mask1, V3Number(nodep, VL_IDATASIZE, shift2));
            }
            if (newshift < 0) {
                newp = new AstShiftR{nodep->fileline(), ap,
                                     new AstConst(nodep->fileline(), -newshift)};
            } else {
                newp = new AstShiftL{nodep->fileline(), ap,
                                     new AstConst(nodep->fileline(), newshift)};
            }
            newp->dtypeFrom(nodep);
            newp = new AstAnd{nodep->fileline(), newp, new AstConst{nodep->fileline(), mask}};
            nodep->replaceWithKeepDType(newp);
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
            // UINFOTREE(1, newp, "", "repShiftShift_new");
            iterate(newp);  // Further reduce, either node may have more reductions.
        }
        VL_DO_DANGLING(pushDeletep(lhsp), lhsp);
    }

    bool replaceAssignMultiSel(AstNodeAssign* nodep) {
        // Multiple assignments to sequential bits can be concated
        // ASSIGN(SEL(a),aq), ASSIGN(SEL(a+1),bq) -> ASSIGN(SEL(a:b),CONCAT(aq,bq)
        // ie. assign var[2]=a, assign var[3]=b -> assign var[3:2]={b,a}

        // Skip if we're not const'ing an entire module (IE doing only one assign, etc)
        if (!m_modp) return false;
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstSel* const sel1p = VN_CAST(nodep->lhsp(), Sel);
        if (!sel1p) return false;
        AstNodeAssign* const nextp = VN_CAST(nodep->nextp(), NodeAssign);
        if (!nextp) return false;
        if (nodep->type() != nextp->type()) return false;
        const AstSel* const sel2p = VN_CAST(nextp->lhsp(), Sel);
        if (!sel2p) return false;
        AstVarRef* const varref1p = VN_CAST(sel1p->fromp(), VarRef);
        if (!varref1p) return false;
        const AstVarRef* const varref2p = VN_CAST(sel2p->fromp(), VarRef);
        if (!varref2p) return false;
        if (!varref1p->sameGateTree(varref2p)) return false;
        const AstConst* const con1p = VN_CAST(sel1p->lsbp(), Const);
        if (!con1p) return false;
        const AstConst* const con2p = VN_CAST(sel2p->lsbp(), Const);
        if (!con2p) return false;
        // We need to make sure there's no self-references involved in either
        // assignment.  For speed, we only look 3 deep, then give up.
        if (!varNotReferenced(nodep->rhsp(), varref1p->varp())) return false;
        if (!varNotReferenced(nextp->rhsp(), varref2p->varp())) return false;
        // If a variable is marked split_var, access to the variable should not be merged.
        if (varref1p->varp()->attrSplitVar() || varref2p->varp()->attrSplitVar()) return false;
        // Swap?
        if ((con1p->toSInt() != con2p->toSInt() + sel2p->width())
            && (con2p->toSInt() != con1p->toSInt() + sel1p->width())) {
            return false;
        }
        const bool lsbFirstAssign = (con1p->toUInt() < con2p->toUInt());
        UINFO(4, "replaceAssignMultiSel " << nodep);
        UINFO(4, "                   && " << nextp);
        // UINFOTREE(1, nodep, "", "comb1");
        // UINFOTREE(1, nextp, "", "comb2");
        AstNodeExpr* const rhs1p = nodep->rhsp()->unlinkFrBack();
        AstNodeExpr* const rhs2p = nextp->rhsp()->unlinkFrBack();
        AstNodeAssign* newp;
        if (lsbFirstAssign) {
            newp = nodep->cloneType(new AstSel{sel1p->fileline(), varref1p->unlinkFrBack(),
                                               sel1p->lsbConst(), sel1p->width() + sel2p->width()},
                                    new AstConcat{rhs1p->fileline(), rhs2p, rhs1p});
        } else {
            newp = nodep->cloneType(new AstSel{sel1p->fileline(), varref1p->unlinkFrBack(),
                                               sel2p->lsbConst(), sel1p->width() + sel2p->width()},
                                    new AstConcat{rhs1p->fileline(), rhs1p, rhs2p});
        }
        // UINFOTREE(1, pnewp, "", "conew");
        nodep->replaceWith(newp);  // dypep intentionally changing
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        VL_DO_DANGLING(pushDeletep(nextp->unlinkFrBack()), nextp);
        return true;
    }

    bool varNotReferenced(AstNode* nodep, AstVar* varp, int level = 0) {
        // Return true if varp never referenced under node.
        // Return false if referenced, or tree too deep to be worth it, or side effects
        if (!nodep) return true;
        if (level > 2) return false;
        if (!nodep->isPure()) return false;  // For example a $fgetc can't be reordered
        if (VN_IS(nodep, NodeVarRef) && VN_AS(nodep, NodeVarRef)->varp() == varp) return false;
        return (varNotReferenced(nodep->nextp(), varp, level + 1)
                && varNotReferenced(nodep->op1p(), varp, level + 1)
                && varNotReferenced(nodep->op2p(), varp, level + 1)
                && varNotReferenced(nodep->op3p(), varp, level + 1)
                && varNotReferenced(nodep->op4p(), varp, level + 1));
    }

    bool replaceNodeAssign(AstNodeAssign* nodep) {
        if (VN_IS(nodep->lhsp(), VarRef) && VN_IS(nodep->rhsp(), VarRef)
            && VN_AS(nodep->lhsp(), VarRef)->sameNoLvalue(VN_AS(nodep->rhsp(), VarRef))
            && !VN_IS(nodep, AssignDly)) {
            // X = X.  Quite pointless, though X <= X may override another earlier assignment
            if (VN_IS(nodep, AssignW)) {
                nodep->v3error("Wire inputs its own output, creating circular logic (wire x=x)");
                return false;  // Don't delete the assign, or V3Gate will freak out
            } else {
                VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
                return true;
            }
        } else if (m_doV && VN_IS(nodep->lhsp(), Concat)) {
            bool need_temp = false;
            bool need_temp_pure = !nodep->rhsp()->isPure();
            if (m_warn && !VN_IS(nodep, AssignDly)
                && !need_temp_pure) {  // Is same var on LHS and RHS?
                // Note only do this (need user4) when m_warn, which is
                // done as unique visitor
                // If the rhs is not pure, we need a temporary variable anyway
                const VNUser4InUse m_inuser4;
                nodep->lhsp()->foreach([](const AstVarRef* nodep) {
                    UASSERT_OBJ(nodep->varp(), nodep, "Unlinked VarRef");
                    nodep->varp()->user4(1);
                });
                nodep->rhsp()->foreach([&need_temp](const AstVarRef* nodep) {
                    UASSERT_OBJ(nodep->varp(), nodep, "Unlinked VarRef");
                    if (nodep->varp()->user4()) need_temp = true;
                });
            }
            if (need_temp_pure) {
                // if the RHS is impure we need to create a temporary variable for it, because
                // further handling involves copying of the RHS.
                UINFO(4, "  ASSITEMPPURE " << nodep);
                // ASSIGN(CONCAT(lc1,lc2),rhs) -> ASSIGN(temp,rhs),
                //                                ASSIGN(lc1,SEL(temp,{size1})),
                //                                ASSIGN(lc2,SEL(temp,{size2}))

                AstNodeExpr* const rhsp = nodep->rhsp()->unlinkFrBack();
                AstVar* const tempPurep = new AstVar{rhsp->fileline(), VVarType::BLOCKTEMP,
                                                     m_concswapNames.get(rhsp), rhsp->dtypep()};
                m_modp->addStmtsp(tempPurep);
                AstVarRef* const tempPureRefp
                    = new AstVarRef{rhsp->fileline(), tempPurep, VAccess::WRITE};
                AstNodeAssign* const asnp
                    = VN_IS(nodep, AssignDly)
                          ? new AstAssign{nodep->fileline(), tempPureRefp, rhsp}
                          : nodep->cloneType(tempPureRefp, rhsp);
                nodep->addHereThisAsNext(asnp);
                nodep->rhsp(new AstVarRef{rhsp->fileline(), tempPurep, VAccess::READ});
            } else if (need_temp) {
                // The first time we constify, there may be the same variable on the LHS
                // and RHS.  In that case, we must use temporaries, or {a,b}={b,a} will break.
                UINFO(4, "  ASSITEMP " << nodep);
                // ASSIGN(CONCAT(lc1,lc2),rhs) -> ASSIGN(temp1,SEL(rhs,{size})),
                //                                ASSIGN(temp2,SEL(newrhs,{size}))
                //                                ASSIGN(lc1,temp1),
                //                                ASSIGN(lc2,temp2)
            } else {
                UINFO(4, "  ASSI " << nodep);
                // ASSIGN(CONCAT(lc1,lc2),rhs) -> ASSIGN(lc1,SEL(rhs,{size})),
                //                                ASSIGN(lc2,SEL(newrhs,{size}))
            }
            UINFOTREE(9, nodep, "", "Ass_old");
            // Unlink the stuff
            AstNodeExpr* const lc1p = VN_AS(nodep->lhsp(), Concat)->lhsp()->unlinkFrBack();
            AstNodeExpr* const lc2p = VN_AS(nodep->lhsp(), Concat)->rhsp()->unlinkFrBack();
            AstNodeExpr* const conp = VN_AS(nodep->lhsp(), Concat)->unlinkFrBack();
            AstNodeExpr* const rhsp = nodep->rhsp()->unlinkFrBack();
            AstNodeExpr* const rhs2p = rhsp->cloneTreePure(false);
            // Calc widths
            const int lsb2 = 0;
            const int msb2 = lsb2 + lc2p->width() - 1;
            const int lsb1 = msb2 + 1;
            const int msb1 = lsb1 + lc1p->width() - 1;
            UASSERT_OBJ(msb1 == (conp->width() - 1), nodep, "Width calc mismatch");
            // Form ranges
            AstSel* const sel1p = new AstSel{conp->fileline(), rhsp, lsb1, msb1 - lsb1 + 1};
            AstSel* const sel2p = new AstSel{conp->fileline(), rhs2p, lsb2, msb2 - lsb2 + 1};
            // Make new assigns of same flavor as old one
            //*** Not cloneTree; just one node.
            AstNodeAssign* newp = nullptr;
            if (!need_temp) {
                AstNodeAssign* const asn1ap = nodep->cloneType(lc1p, sel1p);
                AstNodeAssign* const asn2ap = nodep->cloneType(lc2p, sel2p);
                asn1ap->dtypeFrom(sel1p);
                asn2ap->dtypeFrom(sel2p);
                newp = AstNode::addNext(newp, asn1ap);
                newp = AstNode::addNext(newp, asn2ap);
            } else {
                UASSERT_OBJ(m_modp, nodep, "Not under module");
                UASSERT_OBJ(m_globalPass, nodep,
                            "Should not reach here when not invoked on whole AstNetlist");
                // We could create just one temp variable, but we'll get better optimization
                // if we make one per term.
                AstVar* const temp1p
                    = new AstVar{sel1p->fileline(), VVarType::BLOCKTEMP,
                                 m_concswapNames.get(sel1p), VFlagLogicPacked{}, msb1 - lsb1 + 1};
                AstVar* const temp2p
                    = new AstVar{sel2p->fileline(), VVarType::BLOCKTEMP,
                                 m_concswapNames.get(sel2p), VFlagLogicPacked{}, msb2 - lsb2 + 1};
                m_modp->addStmtsp(temp1p);
                m_modp->addStmtsp(temp2p);
                AstNodeAssign* const asn1ap = nodep->cloneType(
                    new AstVarRef{sel1p->fileline(), temp1p, VAccess::WRITE}, sel1p);
                AstNodeAssign* const asn2ap = nodep->cloneType(
                    new AstVarRef{sel2p->fileline(), temp2p, VAccess::WRITE}, sel2p);
                AstNodeAssign* const asn1bp = nodep->cloneType(
                    lc1p, new AstVarRef{sel1p->fileline(), temp1p, VAccess::READ});
                AstNodeAssign* const asn2bp = nodep->cloneType(
                    lc2p, new AstVarRef{sel2p->fileline(), temp2p, VAccess::READ});
                asn1ap->dtypeFrom(temp1p);
                asn1bp->dtypeFrom(temp1p);
                asn2ap->dtypeFrom(temp2p);
                asn2bp->dtypeFrom(temp2p);
                // This order matters
                newp = AstNode::addNext(newp, asn1ap);
                newp = AstNode::addNext(newp, asn2ap);
                newp = AstNode::addNext(newp, asn1bp);
                newp = AstNode::addNext(newp, asn2bp);
            }
            if (debug() >= 9 && newp) newp->dumpTreeAndNext(cout, "-     _new: ");
            nodep->addNextHere(newp);
            // Cleanup
            VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
            VL_DO_DANGLING(pushDeletep(conp), conp);
            // Further reduce, either node may have more reductions.
            return true;
        } else if (m_doV && VN_IS(nodep->rhsp(), StreamR)) {
            // The right-streaming operator on rhs of assignment does not
            // change the order of bits. Eliminate stream but keep its lhsp.
            // Add a cast if needed.
            AstStreamR* const streamp = VN_AS(nodep->rhsp(), StreamR)->unlinkFrBack();
            AstNodeExpr* srcp = streamp->lhsp()->unlinkFrBack();
            AstNodeDType* const srcDTypep = srcp->dtypep()->skipRefp();
            const AstNodeDType* const dstDTypep = nodep->lhsp()->dtypep()->skipRefp();
            if (VN_IS(srcDTypep, QueueDType) || VN_IS(srcDTypep, DynArrayDType)) {
                if (VN_IS(dstDTypep, QueueDType) || VN_IS(dstDTypep, DynArrayDType)) {
                    int srcElementBits = 0;
                    if (const AstNodeDType* const elemDtp = srcDTypep->subDTypep()) {
                        srcElementBits = elemDtp->width();
                    }
                    int dstElementBits = 0;
                    if (const AstNodeDType* const elemDtp = dstDTypep->subDTypep()) {
                        dstElementBits = elemDtp->width();
                    }
                    srcp = new AstCvtArrayToArray{
                        srcp->fileline(), srcp,          nodep->dtypep(), false, 1,
                        dstElementBits,   srcElementBits};
                } else {
                    srcp = new AstCvtArrayToPacked{srcp->fileline(), srcp, nodep->dtypep()};
                }
            } else if (VN_IS(srcDTypep, UnpackArrayDType)) {
                srcp = new AstCvtArrayToPacked{srcp->fileline(), srcp, srcDTypep};
                // Handling the case where lhs is wider than rhs by inserting zeros. StreamL does
                // not require this, since the left streaming operator implicitly handles this.
                const int packedBits = nodep->lhsp()->widthMin();
                const int unpackBits
                    = srcDTypep->arrayUnpackedElements() * srcDTypep->subDTypep()->widthMin();
                const uint32_t offset = packedBits > unpackBits ? packedBits - unpackBits : 0;
                srcp = new AstShiftL{srcp->fileline(), srcp,
                                     new AstConst{srcp->fileline(), offset}, packedBits};
            }
            nodep->rhsp(srcp);
            VL_DO_DANGLING(pushDeletep(streamp), streamp);
            // Further reduce, any of the nodes may have more reductions.
            return true;
        } else if (m_doV && VN_IS(nodep->lhsp(), StreamL)) {
            // Push the stream operator to the rhs of the assignment statement
            AstNodeExpr* streamp = nodep->lhsp()->unlinkFrBack();
            AstNodeExpr* const dstp = VN_AS(streamp, StreamL)->lhsp()->unlinkFrBack();
            AstNodeDType* const dstDTypep = dstp->dtypep()->skipRefp();
            AstNodeExpr* const srcp = nodep->rhsp()->unlinkFrBack();
            const AstNodeDType* const srcDTypep = srcp->dtypep()->skipRefp();
            const int sWidth = srcp->width();
            const int dWidth = dstp->width();
            // Connect the rhs to the stream operator and update its width
            VN_AS(streamp, StreamL)->lhsp(srcp);
            if (VN_IS(srcDTypep, DynArrayDType) || VN_IS(srcDTypep, QueueDType)
                || VN_IS(srcDTypep, UnpackArrayDType)) {
                streamp->dtypeSetStream();
            } else {
                streamp->dtypeSetLogicUnsized(srcp->width(), srcp->widthMin(), VSigning::UNSIGNED);
            }
            if (VN_IS(dstDTypep, UnpackArrayDType)) {
                streamp = new AstCvtPackedToArray{nodep->fileline(), streamp, dstDTypep};
            } else {
                if (dWidth == 0) {
                    streamp = new AstCvtPackedToArray{nodep->fileline(), streamp, dstDTypep};
                } else if (sWidth >= dWidth) {
                    streamp = new AstSel{streamp->fileline(), streamp, sWidth - dWidth, dWidth};
                }
            }
            nodep->lhsp(dstp);
            nodep->rhsp(streamp);
            nodep->dtypep(dstDTypep);
            return true;
        } else if (m_doV && VN_IS(nodep->lhsp(), StreamR)) {
            // The right stream operator on lhs of assignment statement does
            // not reorder bits. However, if the rhs is wider than the lhs,
            // then we select bits from the left-most, not the right-most.
            AstNodeExpr* const streamp = nodep->lhsp()->unlinkFrBack();
            AstNodeExpr* const dstp = VN_AS(streamp, StreamR)->lhsp()->unlinkFrBack();
            AstNodeDType* const dstDTypep = dstp->dtypep()->skipRefp();
            AstNodeExpr* srcp = nodep->rhsp()->unlinkFrBack();
            const int sWidth = srcp->width();
            const int dWidth = dstp->width();
            if (VN_IS(dstDTypep, UnpackArrayDType)) {
                const int dstBitWidth
                    = dWidth * VN_AS(dstDTypep, UnpackArrayDType)->arrayUnpackedElements();
                // Handling the case where rhs is wider than lhs. StreamL does not require this
                // since the combination of the left streaming operation and the implicit
                // truncation in VL_ASSIGN_UNPACK automatically selects the left-most bits.
                if (sWidth > dstBitWidth) {
                    srcp
                        = new AstSel{streamp->fileline(), srcp, sWidth - dstBitWidth, dstBitWidth};
                }
                srcp = new AstCvtPackedToArray{nodep->fileline(), srcp, dstDTypep};
            } else {
                UASSERT_OBJ(sWidth >= dWidth, nodep,
                            "sWidth >= dWidth should have caused an error earlier");
                if (dWidth == 0) {
                    srcp = new AstCvtPackedToArray{nodep->fileline(), srcp, dstDTypep};
                } else if (sWidth >= dWidth) {
                    srcp = new AstSel{streamp->fileline(), srcp, sWidth - dWidth, dWidth};
                }
            }
            nodep->lhsp(dstp);
            nodep->rhsp(srcp);
            nodep->dtypep(dstDTypep);
            VL_DO_DANGLING(pushDeletep(streamp), streamp);
            // Further reduce, any of the nodes may have more reductions.
            return true;
        } else if (m_doV && VN_IS(nodep->rhsp(), StreamL)) {
            AstStreamL* streamp = VN_AS(nodep->rhsp(), StreamL);
            AstNodeExpr* srcp = streamp->lhsp();
            const AstNodeDType* const srcDTypep = srcp->dtypep()->skipRefp();
            AstNodeDType* const dstDTypep = nodep->lhsp()->dtypep()->skipRefp();
            if ((VN_IS(srcDTypep, QueueDType) || VN_IS(srcDTypep, DynArrayDType)
                 || VN_IS(srcDTypep, UnpackArrayDType))) {
                if (VN_IS(dstDTypep, QueueDType) || VN_IS(dstDTypep, DynArrayDType)) {
                    int blockSize = 1;
                    if (const AstConst* const constp = VN_CAST(streamp->rhsp(), Const)) {
                        blockSize = constp->toSInt();
                        if (VL_UNLIKELY(blockSize <= 0)) {
                            // Not reachable due to higher level checks when parsing stream
                            // operators commented out to not fail v3error-coverage-checks.
                            // nodep->v3error("Stream block size must be positive, got " <<
                            // blockSize); nodep->v3error("Stream block size must be positive, got
                            // " << blockSize);
                            blockSize = 1;
                        }
                    }
                    // Not reachable due to higher level checks when parsing stream operators
                    // commented out to not fail v3error-coverage-checks.
                    // else {
                    // nodep->v3error("Stream block size must be constant (got " <<
                    // streamp->rhsp()->prettyTypeName() << ")");
                    // }
                    int srcElementBits = 0;
                    if (const AstNodeDType* const elemDtp = srcDTypep->subDTypep()) {
                        srcElementBits = elemDtp->width();
                    }
                    int dstElementBits = 0;
                    if (const AstNodeDType* const elemDtp = dstDTypep->subDTypep()) {
                        dstElementBits = elemDtp->width();
                    }
                    streamp->unlinkFrBack();
                    srcp = new AstCvtArrayToArray{
                        srcp->fileline(), srcp->unlinkFrBack(), dstDTypep,     true,
                        blockSize,        dstElementBits,       srcElementBits};
                    nodep->rhsp(srcp);
                    VL_DO_DANGLING(pushDeletep(streamp), streamp);
                } else {
                    streamp->lhsp(new AstCvtArrayToPacked{srcp->fileline(), srcp->unlinkFrBack(),
                                                          dstDTypep});
                    streamp->dtypeFrom(dstDTypep);
                }
            }
        } else if (m_doV && replaceAssignMultiSel(nodep)) {
            return true;
        }
        return false;
    }

    // Boolean replacements
    bool operandBoolShift(const AstNode* nodep) {
        // boolean test of AND(const,SHIFTR(x,const)) -> test of AND(SHIFTL(x,const), x)
        if (!VN_IS(nodep, And)) return false;
        if (!VN_IS(VN_AS(nodep, And)->lhsp(), Const)) return false;
        if (!VN_IS(VN_AS(nodep, And)->rhsp(), ShiftR)) return false;
        const AstShiftR* const shiftp = VN_AS(VN_AS(nodep, And)->rhsp(), ShiftR);
        if (!VN_IS(shiftp->rhsp(), Const)) return false;
        if (static_cast<uint32_t>(nodep->width()) <= VN_AS(shiftp->rhsp(), Const)->toUInt()) {
            return false;
        }
        return true;
    }
    void replaceBoolShift(AstNode* nodep) {
        UINFOTREE(9, nodep, "", "bshft_old");
        AstConst* const andConstp = VN_AS(VN_AS(nodep, And)->lhsp(), Const);
        AstNodeExpr* const fromp
            = VN_AS(VN_AS(nodep, And)->rhsp(), ShiftR)->lhsp()->unlinkFrBack();
        AstConst* const shiftConstp
            = VN_AS(VN_AS(VN_AS(nodep, And)->rhsp(), ShiftR)->rhsp(), Const);
        V3Number val{andConstp, andConstp->width()};
        val.opShiftL(andConstp->num(), shiftConstp->num());
        AstAnd* const newp
            = new AstAnd{nodep->fileline(), new AstConst{nodep->fileline(), val}, fromp};
        // widthMin no longer applicable if different C-expanded width
        newp->dtypeSetLogicSized(nodep->width(), VSigning::UNSIGNED);
        nodep->replaceWith(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        UINFOTREE(9, newp, "", "_new");
    }

    void replaceWithSimulation(AstNode* nodep) {
        SimulateVisitor simvis;
        // Run it - may be unoptimizable due to large for loop, etc
        simvis.mainParamEmulate(nodep);
        if (!simvis.optimizable()) {
            const AstNode* errorp = simvis.whyNotNodep();
            if (!errorp) errorp = nodep;
            nodep->v3error("Expecting expression to be constant, but can't determine constant for "
                           << nodep->prettyTypeName() << '\n'
                           << errorp->warnOther() << "... Location of non-constant "
                           << errorp->prettyTypeName() << ": " << simvis.whyNotMessage());
            VL_DO_DANGLING(replaceZero(nodep), nodep);
        } else {
            // Fetch the result
            AstNode* const valuep = simvis.fetchValueNull(nodep);  // valuep is owned by Simulate
            UASSERT_OBJ(valuep, nodep, "No value returned from simulation");
            // Replace it
            AstNode* const newp = valuep->cloneTree(false);
            newp->fileline(nodep->fileline());
            nodep->replaceWithKeepDType(newp);
            UINFO(4, "Simulate->" << newp);
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
        }
    }

    //----------------------------------------

    // VISITORS
    void visit(AstNetlist* nodep) override {
        // Iterate modules backwards, in bottom-up order.  That's faster
        iterateChildrenBackwardsConst(nodep);
    }
    void visit(AstNodeModule* nodep) override {
        VL_RESTORER(m_modp);
        m_modp = nodep;
        m_concswapNames.reset();
        iterateChildren(nodep);
    }
    void visit(AstCFunc* nodep) override {
        // No ASSIGNW removals under funcs, we've long eliminated INITIALs
        // (We should perhaps rename the assignw's to just assigns)
        VL_RESTORER(m_wremove);
        m_wremove = false;
        iterateChildren(nodep);
    }
    void visit(AstCLocalScope* nodep) override {
        iterateChildren(nodep);
        if (!nodep->stmtsp()) {
            nodep->unlinkFrBack();
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
        }
    }
    void visit(AstScope* nodep) override {
        // No ASSIGNW removals under scope, we've long eliminated INITIALs
        VL_RESTORER(m_wremove);
        VL_RESTORER(m_scopep);
        m_wremove = false;
        m_scopep = nodep;
        iterateChildren(nodep);
    }

    void swapSides(AstNodeBiCom* nodep) {
        // COMMUTATIVE({a},CONST) -> COMMUTATIVE(CONST,{a})
        // This simplifies later optimizations
        AstNodeExpr* const lhsp = nodep->lhsp()->unlinkFrBackWithNext();
        AstNodeExpr* const rhsp = nodep->rhsp()->unlinkFrBackWithNext();
        nodep->lhsp(rhsp);
        nodep->rhsp(lhsp);
        iterate(nodep);  // Again?
    }

    bool containsMemberAccessRecurse(const AstNode* const nodep) {
        if (!nodep) return false;
        const auto it = m_containsMemberAccess.lower_bound(nodep);
        if (it != m_containsMemberAccess.end() && it->first == nodep) return it->second;
        bool result = false;
        if (VN_IS(nodep, MemberSel) || VN_IS(nodep, MethodCall) || VN_IS(nodep, CMethodCall)) {
            result = true;
        } else if (const AstNodeFTaskRef* const funcRefp = VN_CAST(nodep, NodeFTaskRef)) {
            if (containsMemberAccessRecurse(funcRefp->taskp())) result = true;
        } else if (const AstNodeCCall* const funcRefp = VN_CAST(nodep, NodeCCall)) {
            if (containsMemberAccessRecurse(funcRefp->funcp())) result = true;
        } else if (const AstNodeFTask* const funcp = VN_CAST(nodep, NodeFTask)) {
            // Assume that it has a member access
            if (funcp->recursive()) result = true;
        } else if (const AstCFunc* const funcp = VN_CAST(nodep, CFunc)) {
            if (funcp->recursive()) result = true;
        }
        if (!result) {
            result = containsMemberAccessRecurse(nodep->op1p())
                     || containsMemberAccessRecurse(nodep->op2p())
                     || containsMemberAccessRecurse(nodep->op3p())
                     || containsMemberAccessRecurse(nodep->op4p());
        }
        if (!result && !VN_IS(nodep, NodeFTask)
            && !VN_IS(nodep, CFunc)  // don't enter into next function
            && containsMemberAccessRecurse(nodep->nextp())) {
            result = true;
        }
        m_containsMemberAccess.insert(it, std::make_pair(nodep, result));
        return result;
    }

    bool matchBiopToBitwise(AstNodeBiop* const nodep) {
        if (!m_convertLogicToBit) return false;
        if (!nodep->lhsp()->width1()) return false;
        if (!nodep->rhsp()->width1()) return false;
        if (!nodep->isPure()) return false;
        if (containsMemberAccessRecurse(nodep)) return false;
        return true;
    }
    bool matchConcatRand(AstConcat* nodep) {
        //    CONCAT(RAND, RAND) - created by Chisel code
        AstRand* const aRandp = VN_CAST(nodep->lhsp(), Rand);
        const AstRand* const bRandp = VN_CAST(nodep->rhsp(), Rand);
        if (!aRandp || !bRandp) return false;
        if (!aRandp->combinable(bRandp)) return false;
        UINFO(4, "Concat(Rand,Rand) => Rand: " << nodep);
        nodep->replaceWithKeepDType(aRandp->unlinkFrBack());
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }
    bool matchSelRand(AstSel* nodep) {
        //    SEL(RAND) - created by Chisel code
        AstRand* const aRandp = VN_CAST(nodep->fromp(), Rand);
        if (!aRandp) return false;
        if (aRandp->seedp()) return false;
        UINFO(4, "Sel(Rand) => Rand: " << nodep);
        nodep->replaceWithKeepDType(aRandp->unlinkFrBack());
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }
    bool matchToStringNConst(AstToStringN* nodep) {
        iterateChildren(nodep);
        if (AstInitArray* const initp = VN_CAST(nodep->lhsp(), InitArray)) {
            if (!(m_doExpensive || m_params)) return false;
            // At present only support 1D unpacked arrays
            const auto initOfConst = [](const AstNode* const nodep) -> bool {  //
                return VN_IS(nodep, Const) || VN_IS(nodep, InitItem);
            };
            if (initp->initsp() && !initp->initsp()->forall(initOfConst)) return false;
            if (initp->defaultp() && !initp->defaultp()->forall(initOfConst)) return false;
        } else if (!VN_IS(nodep->lhsp(), Const)) {
            return false;
        }
        replaceWithSimulation(nodep);
        return true;
    }
    int operandConcatMove(const AstConcat* nodep) {
        //    CONCAT under concat  (See moveConcat)
        // Return value: true indicates to do it; 2 means move to LHS
        const AstConcat* const abConcp = VN_CAST(nodep->lhsp(), Concat);
        const AstConcat* const bcConcp = VN_CAST(nodep->rhsp(), Concat);
        if (!abConcp && !bcConcp) return 0;
        if (bcConcp) {
            const AstNodeExpr* const ap = nodep->lhsp();
            const AstNodeExpr* const bp = bcConcp->lhsp();
            // If a+b == 32,64,96 etc, then we want to have a+b together on LHS
            if (VL_BITBIT_I(ap->width() + bp->width()) == 0) return 2;  // Transform 2: to abConc
        } else {  // abConcp
            // Unless lhs is already 32 bits due to above, reorder it
            if (VL_BITBIT_I(nodep->lhsp()->width()) != 0) return 1;  // Transform 1: to bcConc
        }
        return 0;  // ok
    }
    void moveConcat(AstConcat* nodep) {
        //    1: CONCAT(CONCAT({a},{b}),{c})  -> CONCAT({a},CONCAT({b}, {c}))
        // or 2: CONCAT({a}, CONCAT({b},{c})) -> CONCAT(CONCAT({a},{b}),{c})
        // Because the lhs of a concat needs a shift but the rhs doesn't,
        // putting additional CONCATs on the RHS leads to fewer assembler operations.
        // However, we'll end up with lots of wide moves if we make huge trees
        // like that, so on 32 bit boundaries, we'll do the opposite form.
        UINFO(4, "Move concat: " << nodep);
        if (operandConcatMove(nodep) > 1) {
            AstNodeExpr* const ap = nodep->lhsp()->unlinkFrBack();
            AstConcat* const bcConcp = VN_AS(nodep->rhsp(), Concat);
            bcConcp->unlinkFrBack();
            AstNodeExpr* const bp = bcConcp->lhsp()->unlinkFrBack();
            AstNodeExpr* const cp = bcConcp->rhsp()->unlinkFrBack();
            AstConcat* const abConcp = new AstConcat{bcConcp->fileline(), ap, bp};
            nodep->lhsp(abConcp);
            nodep->rhsp(cp);
            // If bp was a concat, then we have this exact same form again!
            // Recurse rather then calling node->iterate to prevent 2^n recursion!
            if (operandConcatMove(abConcp)) moveConcat(abConcp);
            VL_DO_DANGLING(pushDeletep(bcConcp), bcConcp);
        } else {
            AstConcat* const abConcp = VN_AS(nodep->lhsp(), Concat);
            abConcp->unlinkFrBack();
            AstNodeExpr* const ap = abConcp->lhsp()->unlinkFrBack();
            AstNodeExpr* const bp = abConcp->rhsp()->unlinkFrBack();
            AstNodeExpr* const cp = nodep->rhsp()->unlinkFrBack();
            AstConcat* const bcConcp = new AstConcat{abConcp->fileline(), bp, cp};
            nodep->lhsp(ap);
            nodep->rhsp(bcConcp);
            if (operandConcatMove(bcConcp)) moveConcat(bcConcp);
            VL_DO_DANGLING(pushDeletep(abConcp), abConcp);
        }
    }

    // Special cases
    void visit(AstConst*) override {}  // Already constant

    void visit(AstCell* nodep) override {
        if (m_params) {
            iterateAndNextNull(nodep->paramsp());
        } else {
            iterateChildren(nodep);
        }
    }
    void visit(AstClassOrPackageRef* nodep) override { iterateChildren(nodep); }
    void visit(AstPin* nodep) override { iterateChildren(nodep); }

    void replaceLogEq(AstLogEq* nodep) {
        // LOGEQ(a,b) => AstLogAnd{AstLogOr{AstLogNot{a},b},AstLogOr{AstLogNot{b},a}}
        AstNodeExpr* const lhsp = nodep->lhsp()->unlinkFrBack();
        AstNodeExpr* const rhsp = nodep->rhsp()->unlinkFrBack();
        // Do exactly as IEEE says, might result in extra terms, so in future may do differently
        AstLogAnd* const newp = new AstLogAnd{
            nodep->fileline(),
            new AstLogOr{nodep->fileline(), new AstLogNot{nodep->fileline(), lhsp}, rhsp},
            new AstLogOr{nodep->fileline(),
                         new AstLogNot{nodep->fileline(), rhsp->cloneTreePure(false)},
                         lhsp->cloneTreePure(false)}};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }

    void replaceSelSel(AstSel* nodep) {
        // SEL(SEL({x},a,b),c,d) => SEL({x},a+c,d)
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstSel* const belowp = VN_AS(nodep->fromp(), Sel);
        AstNodeExpr* const fromp = belowp->fromp()->unlinkFrBack();
        AstNodeExpr* const lsb1p = nodep->lsbp()->unlinkFrBack();
        AstNodeExpr* const lsb2p = belowp->lsbp()->unlinkFrBack();
        // Eliminate lower range
        UINFO(4, "Elim Lower range: " << nodep);
        AstNodeExpr* newlsbp;
        if (VN_IS(lsb1p, Const) && VN_IS(lsb2p, Const)) {
            newlsbp = new AstConst{lsb1p->fileline(),
                                   VN_AS(lsb1p, Const)->toUInt() + VN_AS(lsb2p, Const)->toUInt()};
            VL_DO_DANGLING(pushDeletep(lsb1p), lsb1p);
            VL_DO_DANGLING(pushDeletep(lsb2p), lsb2p);
        } else {
            // Width is important, we need the width of the fromp's expression, not the
            // potentially smaller lsb1p's width, but don't insert a redundant AstExtend.
            // Note that due to some sloppiness in earlier passes, lsb1p might actually be wider,
            // so extend to the wider type.
            const AstNodeExpr* const widep = lsb1p->width() > lsb2p->width() ? lsb1p : lsb2p;
            AstNodeExpr* const lhsp = widep->width() > lsb2p->width()
                                          ? new AstExtend{lsb2p->fileline(), lsb2p}
                                          : lsb2p;
            AstNodeExpr* const rhsp = widep->width() > lsb1p->width()
                                          ? new AstExtend{lsb1p->fileline(), lsb1p}
                                          : lsb1p;
            lhsp->dtypeFrom(widep);
            rhsp->dtypeFrom(widep);
            newlsbp = new AstAdd{lsb1p->fileline(), lhsp, rhsp};
            newlsbp->dtypeFrom(widep);
        }
        AstSel* const newp = new AstSel{nodep->fileline(), fromp, newlsbp, nodep->widthConst()};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }

    void replaceSelConcat(AstSel* nodep) {
        // SEL(CONCAT(a,b),c,d) => SEL(a or b, . .)
        AstConcat* const conp = VN_AS(nodep->fromp(), Concat);
        AstNodeExpr* const conLhsp = conp->lhsp();
        AstNodeExpr* const conRhsp = conp->rhsp();
        if (static_cast<int>(nodep->lsbConst()) >= conRhsp->width()) {
            conLhsp->unlinkFrBack();
            AstSel* const newp
                = new AstSel{nodep->fileline(), conLhsp, nodep->lsbConst() - conRhsp->width(),
                             nodep->widthConst()};
            nodep->replaceWithKeepDType(newp);
        } else if (static_cast<int>(nodep->msbConst()) < conRhsp->width()) {
            conRhsp->unlinkFrBack();
            AstSel* const newp
                = new AstSel{nodep->fileline(), conRhsp, nodep->lsbConst(), nodep->widthConst()};
            nodep->replaceWithKeepDType(newp);
        } else {
            // Yuk, split between the two
            conRhsp->unlinkFrBack();
            conLhsp->unlinkFrBack();
            AstConcat* const newp
                = new AstConcat{nodep->fileline(),
                                new AstSel{nodep->fileline(), conLhsp, 0,
                                           nodep->msbConst() - conRhsp->width() + 1},
                                new AstSel{nodep->fileline(), conRhsp, nodep->lsbConst(),
                                           conRhsp->width() - nodep->lsbConst()}};
            nodep->replaceWithKeepDType(newp);
        }
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    bool operandSelReplicate(AstSel* nodep) {
        // SEL(REPLICATE(from,rep),lsb,width) => SEL(from,0,width) as long
        // as SEL's width <= b's width
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstReplicate* const repp = VN_AS(nodep->fromp(), Replicate);
        AstNodeExpr* const fromp = repp->srcp();
        const AstConst* const lsbp = VN_CAST(nodep->lsbp(), Const);
        if (!lsbp) return false;
        UASSERT_OBJ(fromp->width(), nodep, "Not widthed");
        if ((lsbp->toUInt() / fromp->width())
            != ((lsbp->toUInt() + nodep->width() - 1) / fromp->width())) {
            return false;
        }
        //
        fromp->unlinkFrBack();
        AstSel* const newp = new AstSel{
            nodep->fileline(), fromp,
            new AstConst{lsbp->fileline(), lsbp->toUInt() % fromp->width()}, nodep->widthConst()};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }
    bool operandRepRep(AstReplicate* nodep) {
        // REPLICATE(REPLICATE2(from2,cnt2),cnt1) => REPLICATE(from2,(cnt1+cnt2))
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstReplicate* const rep2p = VN_AS(nodep->srcp(), Replicate);
        AstNodeExpr* const from2p = rep2p->srcp();
        AstConst* const cnt1p = VN_CAST(nodep->countp(), Const);
        if (!cnt1p) return false;
        AstConst* const cnt2p = VN_CAST(rep2p->countp(), Const);
        if (!cnt2p) return false;
        //
        from2p->unlinkFrBack();
        AstReplicate* const newp
            = new AstReplicate{nodep->fileline(), from2p, cnt1p->toUInt() * cnt2p->toUInt()};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }
    bool operandConcatSame(AstConcat* nodep) {
        // CONCAT(fromp,fromp) -> REPLICATE(fromp,1+1)
        // CONCAT(REP(fromp,cnt1),fromp) -> REPLICATE(fromp,cnt1+1)
        // CONCAT(fromp,REP(fromp,cnt1)) -> REPLICATE(fromp,1+cnt1)
        // CONCAT(REP(fromp,cnt1),REP(fromp,cnt2)) -> REPLICATE(fromp,cnt1+cnt2)
        AstNodeExpr* from1p = nodep->lhsp();
        uint32_t cnt1 = 1;
        AstNodeExpr* from2p = nodep->rhsp();
        uint32_t cnt2 = 1;
        if (VN_IS(from1p, Replicate)) {
            const AstConst* const cnt1p = VN_CAST(VN_CAST(from1p, Replicate)->countp(), Const);
            if (!cnt1p) return false;
            from1p = VN_AS(from1p, Replicate)->srcp();
            cnt1 = cnt1p->toUInt();
        }
        if (VN_IS(from2p, Replicate)) {
            const AstConst* const cnt2p = VN_CAST(VN_CAST(from2p, Replicate)->countp(), Const);
            if (!cnt2p) return false;
            from2p = VN_AS(from2p, Replicate)->srcp();
            cnt2 = cnt2p->toUInt();
        }
        if (!operandsSame(from1p, from2p)) return false;
        //
        from1p->unlinkFrBack();
        AstReplicate* const newp = new AstReplicate{nodep->fileline(), from1p, cnt1 + cnt2};
        nodep->replaceWithKeepDType(newp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
        return true;
    }
    void replaceSelIntoBiop(AstSel* nodep) {
        // SEL(BUFIF1(a,b),1,bit) => BUFIF1(SEL(a,1,bit),SEL(b,1,bit))
        AstNodeBiop* const fromp = VN_AS(nodep->fromp()->unlinkFrBack(), NodeBiop);
        UASSERT_OBJ(fromp, nodep, "Called on non biop");
        AstNodeExpr* const lsbp = nodep->lsbp()->unlinkFrBack();
        //
        AstNodeExpr* const bilhsp = fromp->lhsp()->unlinkFrBack();
        AstNodeExpr* const birhsp = fromp->rhsp()->unlinkFrBack();
        //
        fromp->lhsp(
            new AstSel{nodep->fileline(), bilhsp, lsbp->cloneTreePure(true), nodep->widthConst()});
        fromp->rhsp(new AstSel{nodep->fileline(), birhsp, lsbp, nodep->widthConst()});
        nodep->replaceWithKeepDType(fromp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }
    void replaceSelIntoUniop(AstSel* nodep) {
        // SEL(NOT(a),1,bit) => NOT(SEL(a,bit))
        AstNodeUniop* const fromp = VN_AS(nodep->fromp()->unlinkFrBack(), NodeUniop);
        UASSERT_OBJ(fromp, nodep, "Called on non biop");
        AstNodeExpr* const lsbp = nodep->lsbp()->unlinkFrBack();
        //
        AstNodeExpr* const bilhsp = fromp->lhsp()->unlinkFrBack();
        //
        fromp->lhsp(new AstSel{nodep->fileline(), bilhsp, lsbp, nodep->widthConst()});
        nodep->replaceWithKeepDType(fromp);
        VL_DO_DANGLING(pushDeletep(nodep), nodep);
    }

    void visit(AstAttrOf* nodep) override {
        VL_RESTORER(m_attrp);
        m_attrp = nodep;
        iterateChildren(nodep);
    }

    void visit(AstArraySel* nodep) override {
        iterateAndNextNull(nodep->bitp());
        if (VN_IS(nodep->bitp(), Const)
            && VN_IS(nodep->fromp(), VarRef)
            // Need to make sure it's an array object so don't mis-allow a constant (bug509.)
            && VN_AS(nodep->fromp(), VarRef)->varp()
            && VN_IS(VN_AS(nodep->fromp(), VarRef)->varp()->valuep(), InitArray)) {
            m_selp = nodep;  // Ask visit(AstVarRef) to replace varref with const
        }
        iterateAndNextNull(nodep->fromp());
        if (VN_IS(nodep->fromp(), Const)) {  // It did.
            if (!m_selp) {
                nodep->v3error("Illegal assignment of constant to unpacked array");
            } else {
                AstNode* const fromp = nodep->fromp()->unlinkFrBack();
                nodep->replaceWithKeepDType(fromp);
                VL_DO_DANGLING(pushDeletep(nodep), nodep);
            }
        }
        m_selp = nullptr;
    }

    // Evaluate a slice of an unpacked array.  If constantification is
    // required (m_required=true), call replaceWithSimulation() to compute
    // the slice via simulation.  Otherwise just iterate the children.
    void visit(AstSliceSel* nodep) override {
        // First constify or width any child nodes
        iterateChildren(nodep);
        if (!m_required) return;  // Do nothing unless we are in parameter mode
        // Fallback to simulation: this will invoke SimulateVisitor::visit(AstSliceSel*)
        replaceWithSimulation(nodep);
    }

    void visit(AstCAwait* nodep) override {
        m_hasJumpDelay = true;
        iterateChildren(nodep);
    }
    void visit(AstNodeVarRef* nodep) override {
        iterateChildren(nodep);
        UASSERT_OBJ(nodep->varp(), nodep, "Not linked");
        bool did = false;
        if (m_doV && nodep->varp()->valuep() && !m_attrp) {
            // UINFOTREE(1, valuep, "", "visitvaref");
            iterateAndNextNull(nodep->varp()->valuep());  // May change nodep->varp()->valuep()
            AstNode* const valuep = nodep->varp()->valuep();
            if (nodep->access().isReadOnly()
                && ((!m_params  // Can reduce constant wires into equations
                     && m_doNConst
                     && v3Global.opt.fConst()
                     // Default value, not a "known" constant for this usage
                     && !nodep->varp()->isClassMember() && !nodep->varp()->sensIfacep()
                     && !(nodep->varp()->isFuncLocal() && nodep->varp()->isNonOutput())
                     && !nodep->varp()->noSubst() && !nodep->varp()->isSigPublic())
                    || nodep->varp()->isParam())) {
                if (operandConst(valuep)) {
                    const V3Number& num = VN_AS(valuep, Const)->num();
                    // UINFO(2, "constVisit " << cvtToHex(valuep) << " " << num);
                    VL_DO_DANGLING(replaceNum(nodep, num), nodep);
                    did = true;
                } else if (m_selp && VN_IS(valuep, InitArray)) {
                    const AstInitArray* const initarp = VN_AS(valuep, InitArray);
                    const uint32_t bit = m_selp->bitConst();
                    const AstNode* const itemp = initarp->getIndexDefaultedValuep(bit);
                    if (VN_IS(itemp, Const)) {
                        const V3Number& num = VN_AS(itemp, Const)->num();
                        // UINFO(2, "constVisit " << cvtToHex(valuep) << " " << num);
                        VL_DO_DANGLING(replaceNum(nodep, num), nodep);
                        did = true;
                    }
                } else if (m_params && VN_IS(valuep, InitArray)) {
                    // Allow parameters to pass arrays
                    // Earlier recursion of InitArray made sure each array value is constant
                    // This exception is fairly fragile, i.e. doesn't
                    // support arrays of arrays or other stuff
                    AstNode* const newp = valuep->cloneTree(false);
                    nodep->replaceWithKeepDType(newp);
                    VL_DO_DANGLING(pushDeletep(nodep), nodep);
                    did = true;
                } else if (nodep->varp()->isParam() && VN_IS(valuep, Unbounded)) {
                    AstNode* const newp = valuep->cloneTree(false);
                    nodep->replaceWithKeepDType(newp);
                    VL_DO_DANGLING(pushDeletep(nodep), nodep);
                    did = true;
                }
            }
        }
        if (!did && m_required) {
            nodep->v3error("Expecting expression to be constant, but variable isn't const: "
                           << nodep->varp()->prettyNameQ());
        }
    }
    void visit(AstExprStmt* nodep) override {
        iterateChildren(nodep);
        if (!AstNode::afterCommentp(nodep->stmtsp())) {
            UINFO(8, "ExprStmt(...) " << nodep << " " << nodep->resultp());
            nodep->replaceWith(nodep->resultp()->unlinkFrBack());
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
            // Removing the ExprStmt might have made something impure above now pure
        }
    }
    void visit(AstEnumItemRef* nodep) override {
        iterateChildren(nodep);
        UASSERT_OBJ(nodep->itemp(), nodep, "Not linked");
        bool did = false;
        if (nodep->itemp()->valuep()) {
            // UINFOTREE(1, nodep->itemp()->valuep(), "", "visitvaref");
            if (nodep->itemp()->user4()) {
                nodep->v3error("Recursive enum value: " << nodep->itemp()->prettyNameQ());
            } else {
                nodep->itemp()->user4(true);
                iterateAndNextNull(nodep->itemp()->valuep());
                nodep->itemp()->user4(false);
            }
            if (AstConst* const valuep = VN_CAST(nodep->itemp()->valuep(), Const)) {
                const V3Number& num = valuep->num();
                VL_DO_DANGLING(replaceNum(nodep, num), nodep);
                did = true;
            }
        }
        if (!did && m_required) {
            nodep->v3error("Expecting expression to be constant, but enum value isn't const: "
                           << nodep->itemp()->prettyNameQ());
        }
    }

    //  void visit(AstCvtPackString* nodep) override {
    // Not constant propagated (for today) because AstNodeExpr::isOpaque is set
    // Someday if lower is constant, convert to quoted "string".

    bool onlySenItemInSenTree(const AstSenItem* nodep) {
        // Only one if it's not in a list
        return (!nodep->nextp() && nodep->backp()->nextp() != nodep);
    }
    void visit(AstSenItem* nodep) override {
        iterateChildren(nodep);
        if (m_doNConst
            && !v3Global.opt.timing().isSetTrue()  // If --timing, V3Sched would turn this into an
                                                   // infinite loop. See #5080
            && (VN_IS(nodep->sensp(), Const) || VN_IS(nodep->sensp(), EnumItemRef)
                || (nodep->varrefp() && nodep->varrefp()->varp()->isParam()))) {
            // Constants in sensitivity lists may be removed (we'll simplify later)
            if (nodep->isClocked()) {  // A constant can never get a pos/negedge
                if (onlySenItemInSenTree(nodep)) {
                    if (nodep->edgeType() == VEdgeType::ET_CHANGED) {
                        // TODO: This really is dodgy, as strictly compliant simulators will not
                        //       execute this block, but but t_func_check relies on it
                        nodep->replaceWith(
                            new AstSenItem{nodep->fileline(), AstSenItem::Initial{}});
                    } else {
                        nodep->replaceWith(new AstSenItem{nodep->fileline(), AstSenItem::Never{}});
                    }
                    VL_DO_DANGLING(pushDeletep(nodep), nodep);
                } else {
                    VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
                }
            } else {  // Otherwise it may compute a result that needs to settle out
                nodep->replaceWith(new AstSenItem{nodep->fileline(), AstSenItem::Combo{}});
                VL_DO_DANGLING(pushDeletep(nodep), nodep);
            }
        } else if (m_doNConst && VN_IS(nodep->sensp(), Not)) {
            // V3Gate may propagate NOTs into clocks... Just deal with it
            AstNode* const sensp = nodep->sensp();
            AstNode* lastSensp = sensp;
            bool invert = false;
            while (VN_IS(lastSensp, Not)) {
                lastSensp = VN_AS(lastSensp, Not)->lhsp();
                invert = !invert;
            }
            UINFO(8, "senItem(NOT...) " << nodep << " " << invert);
            if (invert) nodep->edgeType(nodep->edgeType().invert());
            sensp->replaceWith(lastSensp->unlinkFrBack());
            VL_DO_DANGLING(pushDeletep(sensp), sensp);
        }
    }

    class SenItemCmp final {
        static int cmp(const AstNodeExpr* ap, const AstNodeExpr* bp) {
            const VNType aType = ap->type();
            const VNType bType = bp->type();
            if (aType != bType) return static_cast<int>(bType) - static_cast<int>(aType);

            if (const AstVarRef* const aRefp = VN_CAST(ap, VarRef)) {
                const AstVarRef* const bRefp = VN_AS(bp, VarRef);
                // Looks visually better if we keep sorted by name
                if (aRefp->name() < bRefp->name()) return -1;
                if (aRefp->name() > bRefp->name()) return 1;
                // But might be same name with different scopes
                if (aRefp->varScopep() < bRefp->varScopep()) return -1;
                if (aRefp->varScopep() > bRefp->varScopep()) return 1;
                // Or rarely, different data types
                if (aRefp->dtypep() < bRefp->dtypep()) return -1;
                if (aRefp->dtypep() > bRefp->dtypep()) return 1;
                return 0;
            }

            if (const AstConst* const aConstp = VN_CAST(ap, Const)) {
                const AstConst* const bConstp = VN_AS(bp, Const);
                if (aConstp->num().isLtXZ(bConstp->num())) return -1;
                if (bConstp->num().isLtXZ(aConstp->num())) return 1;
                return 0;
            }

            if (const AstNodeBiop* const aBiOpp = VN_CAST(ap, NodeBiop)) {
                const AstNodeBiop* const bBiOpp = VN_AS(bp, NodeBiop);
                // Compare RHSs first as LHS might be const, but the variable term should become
                // adjacent for optimization if identical.
                if (const int c = cmp(aBiOpp->rhsp(), bBiOpp->rhsp())) return c;
                return cmp(aBiOpp->lhsp(), bBiOpp->lhsp());
            }

            if (const AstCMethodHard* const aCallp = VN_CAST(ap, CMethodHard)) {
                const AstCMethodHard* const bCallp = VN_AS(bp, CMethodHard);
                if (aCallp->method() < bCallp->method()) return -1;
                if (aCallp->method() > bCallp->method()) return 1;
                if (const int c = cmp(aCallp->fromp(), bCallp->fromp())) return c;
                const AstNodeExpr* aPinsp = aCallp->pinsp();
                const AstNodeExpr* bPinsp = bCallp->pinsp();
                while (aPinsp && bPinsp) {
                    if (const int c = cmp(aPinsp, bPinsp)) return c;
                    aPinsp = VN_AS(aPinsp->nextp(), NodeExpr);
                    bPinsp = VN_AS(bPinsp->nextp(), NodeExpr);
                }
                return aPinsp ? -1 : bPinsp ? 1 : 0;
            }

            return 0;
        }

    public:
        bool operator()(const AstSenItem* lhsp, const AstSenItem* rhsp) const {
            const AstNodeExpr* const lSensp = lhsp->sensp();
            const AstNodeExpr* const rSensp = rhsp->sensp();
            if (lSensp && rSensp) {
                // If both terms have sensitivity expressions, recursively compare them
                if (const int c = cmp(lSensp, rSensp)) return c < 0;
            } else if (lSensp || rSensp) {
                // Terms with sensitivity expressions come after those without
                return rSensp;
            }
            // Finally sort by edge, AFTER variable, as we want multiple edges for same var
            // adjacent. note the SenTree optimizer requires this order (more general first,
            // less general last)
            return lhsp->edgeType() < rhsp->edgeType();
        }
    };

    void visit(AstSenTree* nodep) override {
        iterateChildren(nodep);
        if (m_doExpensive) {
            // UINFOTREE(1, nodep, "", "ssin");
            // Optimize ideas for the future:
            //   SENTREE(... SENGATE(x,a), SENGATE(SENITEM(x),b) ...)  => SENGATE(x,OR(a,b))

            //   SENTREE(... SENITEM(x),   SENGATE(SENITEM(x),*) ...)  => SENITEM(x)
            // Do we need the SENITEM's to be identical?  No because we're
            // ORing between them; we just need to ensure that the result is at
            // least as frequently activating.  So we
            // SENGATE(SENITEM(x)) -> SENITEM(x), then let it collapse with the
            // other SENITEM(x).

            // Mark x in SENITEM(x)
            for (AstSenItem* senp = nodep->sensesp(); senp; senp = VN_AS(senp->nextp(), SenItem)) {
                if (senp->varrefp() && senp->varrefp()->varScopep()) {
                    senp->varrefp()->varScopep()->user4(1);
                }
            }

            // Pass 1: Sort the sensitivity items so "posedge a or b" and "posedge b or a" and
            // similar, optimizable expressions end up next to each other.
            for (AstSenItem *nextp, *senp = nodep->sensesp(); senp; senp = nextp) {
                nextp = VN_AS(senp->nextp(), SenItem);
                // cppcheck-suppress unassignedVariable  // cppcheck bug
                const SenItemCmp cmp;
                if (nextp && !cmp(senp, nextp)) {
                    // Something's out of order, sort it
                    senp = nullptr;
                    std::vector<AstSenItem*> vec;
                    for (AstSenItem* senp = nodep->sensesp(); senp;
                         senp = VN_AS(senp->nextp(), SenItem)) {
                        vec.push_back(senp);
                    }
                    stable_sort(vec.begin(), vec.end(), SenItemCmp());
                    for (const auto& ip : vec) ip->unlinkFrBack();
                    for (const auto& ip : vec) nodep->addSensesp(ip);
                    break;
                }
            }

            // Pass 2, remove duplicates and simplify adjacent terms if possible
            for (AstSenItem *senp = nodep->sensesp(), *nextp; senp; senp = nextp) {
                nextp = VN_AS(senp->nextp(), SenItem);
                if (!nextp) break;
                AstSenItem* const lItemp = senp;
                AstSenItem* const rItemp = nextp;
                const AstNodeExpr* const lSenp = lItemp->sensp();
                const AstNodeExpr* const rSenp = rItemp->sensp();
                if (!lSenp || !rSenp) continue;

                if (lSenp->sameGateTree(rSenp)) {
                    // POSEDGE or NEGEDGE -> BOTHEDGE. (We've sorted POSEDGE, before NEGEDGE, so we
                    // do not need to test for the opposite orders.)
                    if (lItemp->edgeType() == VEdgeType::ET_POSEDGE
                        && rItemp->edgeType() == VEdgeType::ET_NEGEDGE) {
                        // Make both terms BOTHEDGE, the second will be removed below
                        lItemp->edgeType(VEdgeType::ET_BOTHEDGE);
                        rItemp->edgeType(VEdgeType::ET_BOTHEDGE);
                    }

                    // Remove identical expressions
                    if (lItemp->edgeType() == rItemp->edgeType()) {
                        VL_DO_DANGLING(pushDeletep(rItemp->unlinkFrBack()), rItemp);
                        nextp = lItemp;
                    }

                    continue;
                }

                // Not identical terms, check if they can be combined
                if (lSenp->width() != rSenp->width()) continue;
                if (const AstAnd* const lAndp = VN_CAST(lSenp, And)) {
                    if (const AstAnd* const rAndp = VN_CAST(rSenp, And)) {
                        if (AstConst* const lConstp = VN_CAST(lAndp->lhsp(), Const)) {
                            if (AstConst* const rConstp = VN_CAST(rAndp->lhsp(), Const)) {
                                if (lAndp->rhsp()->sameTree(rAndp->rhsp())) {
                                    const V3Number lNum{lConstp->num()};
                                    lConstp->num().opOr(lNum, rConstp->num());
                                    // Remove redundant term
                                    VL_DO_DANGLING(pushDeletep(rItemp->unlinkFrBack()), rItemp);
                                    nextp = lItemp;
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    //-----
    // Zero elimination
    void visit(AstNodeAssign* nodep) override {
        iterateChildren(nodep);
        if (nodep->timingControlp()) m_hasJumpDelay = true;
        if (m_doNConst && replaceNodeAssign(nodep)) return;
    }
    void visit(AstAlias* nodep) override {
        // Don't perform any optimizations, keep the alias around
    }
    void visit(AstAliasScope* nodep) override {
        // Don't perform any optimizations, keep the alias around
    }
    void visit(AstAssignW* nodep) override {
        iterateChildren(nodep);
        if (m_doNConst && replaceNodeAssign(nodep)) return;
        // Process containing this AssignW as single body statement
        AstAlways* const procp = VN_CAST(nodep->backp(), Always);
        if (!procp || procp->stmtsp() != nodep || nodep->nextp()) return;
        // Not VarXRef, as different refs may set different values to each hierarchy
        AstNodeVarRef* const varrefp = VN_CAST(nodep->lhsp(), VarRef);
        if (m_wremove && !m_params && m_doNConst && m_modp && operandConst(nodep->rhsp())
            && !VN_AS(nodep->rhsp(), Const)->num().isFourState()
            && varrefp  // Don't do messes with BITREFs/ARRAYREFs
            && !varrefp->varp()->hasStrengthAssignment()  // Strengths are resolved in V3Tristate
            && !varrefp->varp()->valuep()  // Not already constified
            && !varrefp->varScopep()  // Not scoped (or each scope may have different initial val.)
            && !varrefp->varp()->isForced()  // Not forced (not really a constant)
        ) {
            // ASSIGNW (VARREF, const) -> INITIAL ( ASSIGN (VARREF, const) )
            UINFO(4, "constAssignW " << nodep);
            // Make a initial assignment
            AstNodeExpr* const exprp = nodep->rhsp()->unlinkFrBack();
            varrefp->unlinkFrBack();
            FileLine* const flp = nodep->fileline();
            AstInitial* const newinitp = new AstInitial{flp, new AstAssign{flp, varrefp, exprp}};
            procp->replaceWith(newinitp);
            VL_DO_DANGLING(pushDeletep(procp), procp);
            // Set the initial value right in the variable so we can constant propagate
            AstNode* const initvaluep = exprp->cloneTree(false);
            varrefp->varp()->valuep(initvaluep);
        }
    }
    void visit(AstCvtArrayToArray* nodep) override {
        iterateChildren(nodep);
        // Handle the case where we have a stream operation inside a cast conversion
        // To avoid infinite recursion, mark the node as processed by setting user1.
        if (!nodep->user1()) {
            nodep->user1(true);
            // Check for both StreamL and StreamR operations
            AstNodeStream* streamp = nullptr;
            bool isReverse = false;
            if (AstStreamL* const streamLp = VN_CAST(nodep->fromp(), StreamL)) {
                streamp = streamLp;
                isReverse = true;  // StreamL reverses the operation
            } else if (AstStreamR* const streamRp = VN_CAST(nodep->fromp(), StreamR)) {
                streamp = streamRp;
                isReverse = false;  // StreamR doesn't reverse the operation
            }
            if (streamp) {
                AstNodeExpr* srcp = streamp->lhsp();
                const AstNodeDType* const srcDTypep = srcp->dtypep()->skipRefp();
                AstNodeDType* const dstDTypep = nodep->dtypep()->skipRefp();
                if (VN_IS(srcDTypep, QueueDType) && VN_IS(dstDTypep, QueueDType)) {
                    int blockSize = 1;
                    if (const AstConst* const constp = VN_CAST(streamp->rhsp(), Const)) {
                        blockSize = constp->toSInt();
                        if (VL_UNLIKELY(blockSize <= 0)) {
                            // Not reachable due to higher level checks when parsing stream
                            // operators commented out to not fail v3error-coverage-checks.
                            // nodep->v3error("Stream block size must be positive, got " <<
                            // blockSize);
                            blockSize = 1;
                        }
                    }
                    // Not reachable due to higher level checks when parsing stream operators
                    // commented out to not fail v3error-coverage-checks.
                    // else {
                    //    nodep->v3error("Stream block size must be constant (got " <<
                    //    streamp->rhsp()->prettyTypeName() << ")");
                    // }
                    int srcElementBits = 0;
                    if (const AstNodeDType* const elemDtp = srcDTypep->subDTypep()) {
                        srcElementBits = elemDtp->width();
                    }
                    int dstElementBits = 0;
                    if (const AstNodeDType* const elemDtp = dstDTypep->subDTypep()) {
                        dstElementBits = elemDtp->width();
                    }
                    streamp->unlinkFrBack();
                    AstNodeExpr* newp = new AstCvtArrayToArray{
                        srcp->fileline(), srcp->unlinkFrBack(), dstDTypep,     isReverse,
                        blockSize,        dstElementBits,       srcElementBits};
                    nodep->replaceWith(newp);
                    VL_DO_DANGLING(pushDeletep(streamp), streamp);
                    VL_DO_DANGLING(pushDeletep(nodep), nodep);
                    return;
                }
            }
        }
    }
    void visit(AstRelease* nodep) override {
        // cppcheck-suppress constVariablePointer // children unlinked below
        if (AstConcat* const concatp = VN_CAST(nodep->lhsp(), Concat)) {
            FileLine* const flp = nodep->fileline();
            AstRelease* const newLp = new AstRelease{flp, concatp->lhsp()->unlinkFrBack()};
            AstRelease* const newRp = new AstRelease{flp, concatp->rhsp()->unlinkFrBack()};
            nodep->replaceWith(newLp);
            newLp->addNextHere(newRp);
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
            visit(newLp);
            visit(newRp);
        }
    }

    void visit(AstNodeIf* nodep) override {
        iterateChildren(nodep);
        if (m_doNConst) {
            if (const AstConst* const constp = VN_CAST(nodep->condp(), Const)) {
                AstNode* keepp = nullptr;
                if (constp->isZero()) {
                    UINFO(4, "IF(0,{any},{x}) => {x}: " << nodep);
                    keepp = nodep->elsesp();
                } else if (!m_doV || constp->isNeqZero()) {  // Might be X in Verilog
                    UINFO(4, "IF(!0,{x},{any}) => {x}: " << nodep);
                    keepp = nodep->thensp();
                } else {
                    UINFO(4, "IF condition is X, retaining: " << nodep);
                    return;
                }
                if (keepp) {
                    keepp->unlinkFrBackWithNext();
                    nodep->replaceWith(keepp);
                } else {
                    nodep->unlinkFrBack();
                }
                VL_DO_DANGLING(pushDeletep(nodep), nodep);

                // Removed branch could contain only impurity within a function and the
                // function could be purified
                VIsCached::clearCacheTree();
            } else if (!AstNode::afterCommentp(nodep->thensp())
                       && !AstNode::afterCommentp(nodep->elsesp())) {
                if (!nodep->condp()->isPure()) {
                    // Condition has side effect - leave, but don't need return value
                    UINFO(4, "IF({x}) ; => STMTEXPR({x}): " << nodep);
                    nodep->fileline()->warnOff(V3ErrorCode::IGNOREDRETURN, true);
                    AstNodeStmt* const newp
                        = new AstStmtExpr{nodep->fileline(), nodep->condp()->unlinkFrBack()};
                    nodep->replaceWith(newp);
                    VL_DO_DANGLING(pushDeletep(nodep), nodep);
                } else {
                    // Empty block, remove it
                    UINFO(4, "IF({x}) ; => ; : " << nodep);
                    VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
                }
            } else if (!AstNode::afterCommentp(nodep->thensp())) {
                UINFO(4, "IF({x}) nullptr {...} => IF(NOT{x}}: " << nodep);
                AstNodeExpr* const condp = nodep->condp();
                AstNode* const elsesp = nodep->elsesp();
                condp->unlinkFrBackWithNext();
                elsesp->unlinkFrBackWithNext();
                if (nodep->thensp()) {  // Must have been comment
                    pushDeletep(nodep->thensp()->unlinkFrBackWithNext());
                }
                nodep->condp(new AstLogNot{condp->fileline(),
                                           condp});  // LogNot, as C++ optimization also possible
                nodep->addThensp(elsesp);
            } else if (((VN_IS(nodep->condp(), Not) && nodep->condp()->width() == 1)
                        || VN_IS(nodep->condp(), LogNot))
                       && nodep->thensp() && nodep->elsesp()) {
                UINFO(4, "IF(NOT {x})  => IF(x) swapped if/else" << nodep);
                AstNodeExpr* const condp
                    = VN_AS(nodep->condp(), NodeUniop)->lhsp()->unlinkFrBackWithNext();
                AstNode* const thensp = nodep->thensp()->unlinkFrBackWithNext();
                AstNode* const elsesp = nodep->elsesp()->unlinkFrBackWithNext();
                AstIf* const ifp = new AstIf{nodep->fileline(), condp, elsesp, thensp};
                ifp->isBoundsCheck(nodep->isBoundsCheck());  // Copy bounds check info
                ifp->branchPred(nodep->branchPred().invert());
                nodep->replaceWith(ifp);
                VL_DO_DANGLING(pushDeletep(nodep), nodep);
            } else if (ifSameAssign(nodep)) {
                UINFO(4,
                      "IF({a}) ASSIGN({b},{c}) else ASSIGN({b},{d}) => ASSIGN({b}, {a}?{c}:{d})");
                AstNodeAssign* const thensp = VN_AS(nodep->thensp(), NodeAssign);
                // cppcheck-suppress constVariablePointer // children unlinked below
                AstNodeAssign* const elsesp = VN_AS(nodep->elsesp(), NodeAssign);
                thensp->unlinkFrBack();
                AstNodeExpr* const condp = nodep->condp()->unlinkFrBack();
                AstNodeExpr* const truep = thensp->rhsp()->unlinkFrBack();
                AstNodeExpr* const falsep = elsesp->rhsp()->unlinkFrBack();
                thensp->rhsp(new AstCond{truep->fileline(), condp, truep, falsep});
                nodep->replaceWith(thensp);
                VL_DO_DANGLING(pushDeletep(nodep), nodep);
            } else if (false  // Disabled, as vpm assertions are faster
                              // without due to short-circuiting
                       && operandIfIf(nodep)) {
                UINFO(9, "IF({a}) IF({b}) => IF({a} && {b})");
                AstNodeIf* const lowerIfp = VN_AS(nodep->thensp(), NodeIf);
                AstNodeExpr* const condp = nodep->condp()->unlinkFrBack();
                AstNode* const lowerThensp = lowerIfp->thensp()->unlinkFrBackWithNext();
                AstNodeExpr* const lowerCondp = lowerIfp->condp()->unlinkFrBackWithNext();
                nodep->condp(new AstLogAnd{lowerIfp->fileline(), condp, lowerCondp});
                lowerIfp->replaceWith(lowerThensp);
                VL_DO_DANGLING(pushDeletep(lowerIfp), lowerIfp);
            } else {
                // Optimizations that don't reform the IF itself
                if (operandBoolShift(nodep->condp())) replaceBoolShift(nodep->condp());
                matchIfCondCond(nodep);
            }
        }
    }

    void visit(AstDisplay* nodep) override {
        // DISPLAY(SFORMAT(text1)),DISPLAY(SFORMAT(text2)) -> DISPLAY(SFORMAT(text1+text2))
        iterateChildren(nodep);
        if (stmtDisplayDisplay(nodep)) return;
    }
    bool stmtDisplayDisplay(AstDisplay* nodep) {
        // DISPLAY(SFORMAT(text1)),DISPLAY(SFORMAT(text2)) -> DISPLAY(SFORMAT(text1+text2))
        if (!m_modp) return false;  // Don't optimize under single statement
        AstDisplay* const prevp = VN_CAST(nodep->backp(), Display);
        if (!prevp) return false;
        if (!((prevp->displayType() == nodep->displayType())
              || (prevp->displayType() == VDisplayType::DT_WRITE
                  && nodep->displayType() == VDisplayType::DT_DISPLAY)
              || (prevp->displayType() == VDisplayType::DT_DISPLAY
                  && nodep->displayType() == VDisplayType::DT_WRITE)))
            return false;
        if ((prevp->filep() && !nodep->filep()) || (!prevp->filep() && nodep->filep())
            || (prevp->filep() && nodep->filep() && !prevp->filep()->sameTree(nodep->filep())))
            return false;
        if (!prevp->fmtp() || prevp->fmtp()->nextp() || !nodep->fmtp() || nodep->fmtp()->nextp())
            return false;
        AstSFormatF* const pformatp = prevp->fmtp();
        // cppcheck-suppress constVariablePointer // children unlinked below
        AstSFormatF* const nformatp = nodep->fmtp();
        // We don't merge scopeNames as can have only one and might be different scopes (late in
        // process) Also rare for real code to print %m multiple times in same message
        if (nformatp->scopeNamep() && pformatp->scopeNamep()) return false;
        // We don't early merge arguments as might need to later print warnings with
        // right line numbers, nor scopeNames as might be different scopes (late in process)
        if (!m_doCpp && pformatp->exprsp()) return false;
        if (!m_doCpp && nformatp->exprsp()) return false;
        if (pformatp->exprsp() && !pformatp->exprsp()->isPureAndNext()) return false;
        if (nformatp->exprsp() && !nformatp->exprsp()->isPureAndNext()) return false;
        // Avoid huge merges
        static constexpr int DISPLAY_MAX_MERGE_LENGTH = 500;
        if (pformatp->text().length() + nformatp->text().length() > DISPLAY_MAX_MERGE_LENGTH)
            return false;
        //
        UINFO(9, "DISPLAY(SF({a})) DISPLAY(SF({b})) -> DISPLAY(SF({a}+{b}))");
        // Convert DT_DISPLAY to DT_WRITE as may allow later optimizations
        if (prevp->displayType() == VDisplayType::DT_DISPLAY) {
            prevp->displayType(VDisplayType::DT_WRITE);
            pformatp->text(pformatp->text() + "\n");
        }
        // We can't replace prev() as the edit tracking iterators will get confused.
        // So instead we edit the prev note itself.
        if (prevp->addNewline()) pformatp->text(pformatp->text() + "\n");
        pformatp->text(pformatp->text() + nformatp->text());
        if (!prevp->addNewline() && nodep->addNewline()) pformatp->text(pformatp->text() + "\n");
        if (nformatp->exprsp()) pformatp->addExprsp(nformatp->exprsp()->unlinkFrBackWithNext());
        if (AstScopeName* const scopeNamep = nformatp->scopeNamep()) {
            scopeNamep->unlinkFrBackWithNext();
            pformatp->scopeNamep(scopeNamep);
        }
        VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
        return true;
    }
    void visit(AstSFormatF* nodep) override {
        // Substitute constants into displays.  The main point of this is to
        // simplify assertion methodologies which call functions with display's.
        // This eliminates a pile of wide temps, and makes the C a whole lot more readable.
        iterateChildren(nodep);
        bool anyconst = false;
        for (AstNode* argp = nodep->exprsp(); argp; argp = argp->nextp()) {
            if (VN_IS(argp, Const)) {
                anyconst = true;
                break;
            }
        }
        if (m_doNConst && anyconst) {
            // UINFO(9, "  Display in  " << nodep->text());
            string newFormat;
            string fmt;
            bool inPct = false;
            AstNode* argp = nodep->exprsp();
            const string text = nodep->text();
            for (const char ch : text) {
                if (!inPct && ch == '%') {
                    inPct = true;
                    fmt = ch;
                } else if (inPct && (std::isdigit(ch) || ch == '.' || ch == '-')) {
                    fmt += ch;
                } else if (inPct) {
                    inPct = false;
                    fmt += ch;
                    switch (std::tolower(ch)) {
                    case '%': break;  // %% - still %%
                    case 'm': break;  // %m - still %m - auto insert "name"
                    case 'l': break;  // %l - still %l - auto insert "library"
                    case 't':  // FALLTHRU
                    case '^':  // %t/%^ - don't know $timeformat so can't constify
                        if (argp) argp = argp->nextp();
                        break;
                    default:  // Most operators, just move to next argument
                        if (argp) {
                            AstNode* const nextp = argp->nextp();
                            if (VN_IS(argp, Const)) {  // Convert it
                                const string out = constNumV(argp).displayed(nodep, fmt);
                                UINFO(9, "     DispConst: " << fmt << " -> " << out << "  for "
                                                            << argp);
                                // fmt = out w/ replace % with %% as it must be literal.
                                fmt = VString::quotePercent(out);
                                VL_DO_DANGLING(pushDeletep(argp->unlinkFrBack()), argp);
                            }
                            argp = nextp;
                        }
                        break;
                    }  // switch
                    newFormat += fmt;
                } else {
                    newFormat += ch;
                }
            }
            if (newFormat != nodep->text()) {
                nodep->text(newFormat);
                UINFO(9, "  Display out " << nodep);
            }
        }
        if (!nodep->exprsp() && nodep->name().find('%') == string::npos && !nodep->hidden()) {
            // Just a simple constant string - the formatting is pointless
            VL_DO_DANGLING(replaceConstString(nodep, nodep->name()), nodep);
        }
    }
    void visit(AstNodeFTask* nodep) override {
        VL_RESTORER(m_underRecFunc);
        if (nodep->recursive()) m_underRecFunc = true;
        iterateChildren(nodep);
    }

    void visit(AstNodeCCall* nodep) override {
        iterateChildren(nodep);
        m_hasJumpDelay = true;  // As don't analyze inside tasks for timing controls
    }
    void visit(AstNodeFTaskRef* nodep) override {
        // Note excludes AstFuncRef as other visitor below
        iterateChildren(nodep);
        m_hasJumpDelay = true;  // As don't analyze inside tasks for timing controls
    }
    void visit(AstFuncRef* nodep) override {
        visit(static_cast<AstNodeFTaskRef*>(nodep));
        if (m_params) {  // Only parameters force us to do constant function call propagation
            replaceWithSimulation(nodep);
        }
    }
    void visit(AstArg* nodep) override {
        // replaceWithSimulation on the Arg's parent FuncRef replaces these
        iterateChildren(nodep);
    }
    void visit(AstLoop* nodep) override {
        VL_RESTORER(m_hasLoopTest);
        const bool oldHasJumpDelay = m_hasJumpDelay;
        m_hasJumpDelay = false;
        m_hasLoopTest = false;
        iterateChildren(nodep);
        bool thisLoopHasJumpDelay = m_hasJumpDelay;
        m_hasJumpDelay = thisLoopHasJumpDelay || oldHasJumpDelay;
        // If the first statement always break, the loop is useless
        if (AstLoopTest* const testp = VN_CAST(nodep->stmtsp(), LoopTest)) {
            if (testp->condp()->isZero()) {
                nodep->v3warn(UNUSEDLOOP, "Loop condition is always false");
                VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
                return;
            }
        }
        // If last statement always breaks, repalce loop with body
        if (AstNode* lastp = nodep->stmtsp()) {
            while (AstNode* const nextp = lastp->nextp()) lastp = nextp;
            if (AstLoopTest* const testp = VN_CAST(lastp, LoopTest)) {
                if (testp->condp()->isZero()) {
                    VL_DO_DANGLING(pushDeletep(testp->unlinkFrBack()), testp);
                    nodep->replaceWith(nodep->stmtsp()->unlinkFrBackWithNext());
                    VL_DO_DANGLING(pushDeletep(nodep), nodep);
                    return;
                }
            }
        }
        // Warn on infinite loop
        if (!m_hasLoopTest && !thisLoopHasJumpDelay) {
            nodep->v3warn(INFINITELOOP, "Infinite loop (condition always true)");
            nodep->fileline()->modifyWarnOff(V3ErrorCode::INFINITELOOP, true);  // Complain once
        }
    }
    void visit(AstLoopTest* nodep) override {
        iterateChildren(nodep);

        // If never breaks, remove
        if (nodep->condp()->isNeqZero()) {
            VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
            return;
        }

        m_hasLoopTest = true;

        // If always breaks, subsequent statements are dead code, delete them
        if (nodep->condp()->isZero()) {
            if (AstNode* const nextp = nodep->nextp()) {
                VL_DO_DANGLING(pushDeletep(nextp->unlinkFrBackWithNext()), nodep);
            }
            return;
        }

        if (operandBoolShift(nodep->condp())) replaceBoolShift(nodep->condp());
    }

    void visit(AstInitArray* nodep) override { iterateChildren(nodep); }
    void visit(AstInitItem* nodep) override { iterateChildren(nodep); }
    void visit(AstUnbounded* nodep) override { iterateChildren(nodep); }
    // These are converted by V3Param.  Don't constify as we don't want the
    // from() VARREF to disappear, if any.
    // If output of a presel didn't get consted, chances are V3Param didn't visit properly
    void visit(AstNodePreSel*) override {}

    // Ignored, can eliminate early
    void visit(AstSysIgnore* nodep) override {
        iterateChildren(nodep);
        if (m_doNConst) VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
    }

    void visit(AstStmtExpr* nodep) override {
        iterateChildren(nodep);
        if (!nodep->exprp() || VN_IS(nodep->exprp(), Const)) {
            VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
            return;
        }
        // TODO if there's an ExprStmt underneath just keep lower statements
        // (No current test case needs this)
        // TODO if non-pure, can remove. First need to clean up that many expressions used
        // under this node do not yet properly indicate non-pure.
        // Also, under here, a NodeUniOp/NodeBiOp that is pure (but child of such is not
        // pure) can be peeled off until get to the non-pure child-node expression,
        // because all that is needed is to execute what causes the side effect.
    }

    // Simplify
    void visit(AstBasicDType* nodep) override {
        iterateChildren(nodep);
        nodep->cvtRangeConst();
    }

    //-----
    // Jump elimination

    void visit(AstJumpGo* nodep) override {
        // Any statements following the JumpGo (at this statement level) never execute, delete
        if (nodep->nextp()) pushDeletep(nodep->nextp()->unlinkFrBackWithNext());

        // JumpGo as last statement in target JumpBlock (including last in a last sub-list),
        // is a no-op, remove it.
        for (AstNode* abovep = nodep->abovep(); abovep; abovep = abovep->abovep()) {
            if (abovep == nodep->blockp()) {
                VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
                return;
            }
            // Stop if not doing expensive, or if the above node is not the last in its list,
            // ... or if it's not an 'if' TODO: it would be enough if it was not a branch.
            if (!m_doExpensive || abovep->nextp() || !VN_IS(abovep, If)) break;
        }
        // Mark JumpBlock as used
        m_usedJumpBlocks.emplace(nodep->blockp());
        m_hasJumpDelay = true;
    }

    void visit(AstJumpBlock* nodep) override {
        iterateChildren(nodep);

        // If first statement is an AstLoopTest, pull it before the jump block
        if (AstLoopTest* const testp = VN_CAST(nodep->stmtsp(), LoopTest)) {
            nodep->addHereThisAsNext(testp->unlinkFrBack());
        }

        // Remove if empty
        if (!nodep->stmtsp()) {
            UINFO(4, "JUMPLABEL => empty " << nodep);
            VL_DO_DANGLING(pushDeletep(nodep->unlinkFrBack()), nodep);
            return;
        }

        // If no JumpGo points to this node, replace it with its body
        if (!m_usedJumpBlocks.count(nodep)) {
            UINFO(4, "JUMPLABEL => unused " << nodep);
            nodep->replaceWith(nodep->stmtsp()->unlinkFrBackWithNext());
            VL_DO_DANGLING(pushDeletep(nodep), nodep);
        }
    }

    //-----
    // Below lines are magic expressions processed by astgen
    //  TREE_SKIP_VISIT("AstNODETYPE")    # Rename normal visit to visitGen and don't iterate
    //-----
    // clang-format off
//    TREE_SKIP_VISIT("ArraySel");
#line 3746 "V3Const__gen.cpp"
#line 3743 "../V3Const.cpp"
//    TREE_SKIP_VISIT("CAwait");
#line 3749 "V3Const__gen.cpp"
#line 3744 "../V3Const.cpp"

    //-----
    //  "AstNODETYPE {             # bracket not paren
    //                $accessor_name, ...
    //                             # .castFoo is the test VN_IS(object,Foo)
    //                             # ,, gets replaced with a , rather than &&
    //                DISABLE_BASE # Turnes off checking Ast's base class treeops
    //               }"            # bracket not paren
    //    ,"what to call"
    //
    // Where "what_to_call" is:
    //          "function to call"
    //          "AstREPLACEMENT_TYPE{ $accessor }"
    //          "!              # Print line number when matches, so can see operations
    //          "NEVER"         # Print error message
    //          "DONE"          # Process of matching did the transform already

    // In the future maybe support more complicated match & replace:
    //   ("AstOr  {%a, AstAnd{AstNot{%b}, %c}} if %a.width1 if %a==%b", "AstOr{%a,%c}; %b.delete");
    // Lhs/rhs would be implied; for non math operations you'd need $lhsp etc.

    //    v--- * * This op done on Verilog or C+++ mode, in all non-m_doConst stages
    //    v--- *1* These ops are always first, as we warn before replacing
    //    v--- *C* This op is a (C)++ op, only in m_doCpp mode
    //    v--- *V* This op is a (V)erilog op, only in m_doV mode
    //    v--- *A* This op works on (A)ll constant children, allowed in m_doConst mode
    //    v--- *S* This op specifies a type should use (S)hort-circuiting of its lhs op

//    TREEOP1("AstSel{warnSelect(nodep)}",        "NEVER");
#line 3780 "V3Const__gen.cpp"
    // TREEOP functions, each return true if they matched & transformed
    // Generated by astgen
    bool match_Add_0(AstAdd* nodep) {
	// TREEOP ("AstAdd   {$lhsp.isZero, $rhsp}",   "replaceWRhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAdd $lhsp.isZero, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Add_1(AstAdd* nodep) {
	// TREEOP ("AstAdd   {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAdd $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_0(AstAnd* nodep) {
	// TREEOP ("AstAnd   {$lhsp.isZero, $rhsp, $rhsp.isPure}",   "replaceZero(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero() && nodep->rhsp()->isPure()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp.isZero, $rhsp, $rhsp.isPure , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_1(AstAnd* nodep) {
	// TREEOP ("AstAnd   {$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp, $rhsp.isZero , replaceZeroChkPure(nodep,$lhsp) )");
	    replaceZeroChkPure(nodep,nodep->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_2(AstAnd* nodep) {
	// TREEOP ("AstAnd   {$lhsp.isAllOnes, $rhsp}",        "replaceWRhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp.isAllOnes, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_3(AstAnd* nodep) {
	// TREEOP ("AstAnd   {$lhsp, $rhsp.isAllOnes}",        "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp, $rhsp.isAllOnes , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_4(AstAnd* nodep) {
	// TREEOPC("AstAnd   {$lhsp.isOne, matchRedundantClean(nodep)}", "DONE")
	if (m_doCpp && nodep->lhsp()->isOne() && matchRedundantClean(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPC( AstAnd $lhsp.isOne, matchRedundantClean(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_5(AstAnd* nodep) {
	// TREEOP ("AstAnd    {operandsSame($lhsp,,$rhsp)}",   "replaceWLhs(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd operandsSame($lhsp,,$rhsp) , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_6(AstAnd* nodep) {
	// TREEOPC("AstAnd {$lhsp.castConst, $rhsp.castRedXor, matchBitOpTree(nodep)}", "DONE")
	if (m_doCpp && VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),RedXor) && matchBitOpTree(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPC( AstAnd $lhsp.castConst, $rhsp.castRedXor, matchBitOpTree(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_7(AstAnd* nodep) {
	// TREEOP ("AstAnd {$lhsp.castConst,matchAndCond(nodep)}", "DONE")
	if (m_doNConst && VN_IS(nodep->lhsp(),Const) && matchAndCond(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp.castConst,matchAndCond(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_8(AstAnd* nodep) {
	// TREEOP ("AstAnd {$lhsp.castConst, $rhsp.castOr, matchMaskedOr(nodep)}", "DONE")
	if (m_doNConst && VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Or) && matchMaskedOr(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp.castConst, $rhsp.castOr, matchMaskedOr(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_9(AstAnd* nodep) {
	// TREEOPC("AstAnd {$lhsp.castConst, matchMaskedShift(nodep)}", "DONE")
	if (m_doCpp && VN_IS(nodep->lhsp(),Const) && matchMaskedShift(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPC( AstAnd $lhsp.castConst, matchMaskedShift(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_10(AstAnd* nodep) {
	// TREEOP ("AstAnd {$lhsp.castOr, $rhsp.castOr, operandAndOrSame(nodep)}", "replaceAndOr(nodep)")
	if (m_doNConst && VN_IS(nodep->lhsp(),Or) && VN_IS(nodep->rhsp(),Or) && operandAndOrSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd $lhsp.castOr, $rhsp.castOr, operandAndOrSame(nodep) , replaceAndOr(nodep) )");
	    replaceAndOr(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_11(AstAnd* nodep) {
	// TREEOP ("AstAnd {operandShiftSame(nodep)}",         "replaceShiftSame(nodep)")
	if (m_doNConst && operandShiftSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstAnd operandShiftSame(nodep) , replaceShiftSame(nodep) )");
	    replaceShiftSame(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_And_12(AstAnd* nodep) {
	// TREEOPC("AstAnd {matchBitOpTree(nodep)}", "DONE")
	if (m_doCpp && matchBitOpTree(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPC( AstAnd matchBitOpTree(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Concat_0(AstConcat* nodep) {
	// TREEOPV("AstConcat{matchConcatRand(nodep)}",      "DONE")
	if (m_doV && matchConcatRand(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstConcat matchConcatRand(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Concat_1(AstConcat* nodep) {
	// TREEOPV("AstConcat{operandConcatMove(nodep)}",      "moveConcat(nodep)")
	if (m_doV && operandConcatMove(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstConcat operandConcatMove(nodep) , moveConcat(nodep) )");
	    moveConcat(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Concat_2(AstConcat* nodep) {
	// TREEOPV("AstConcat{$lhsp.isZero, $rhsp}",           "replaceExtend(nodep, nodep->rhsp())")
	if (m_doV && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstConcat $lhsp.isZero, $rhsp , replaceExtend(nodep, nodep->rhsp()) )");
	    replaceExtend(nodep, nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Concat_3(AstConcat* nodep) {
	// TREEOPV("AstConcat{$lhsp.castSel, $rhsp.castSel, ifAdjacentSel(VN_AS($lhsp,,Sel),,VN_AS($rhsp,,Sel))}",  "replaceConcatSel(nodep)")
	if (m_doV && VN_IS(nodep->lhsp(),Sel) && VN_IS(nodep->rhsp(),Sel) && ifAdjacentSel(VN_AS(nodep->lhsp(),Sel),VN_AS(nodep->rhsp(),Sel))) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstConcat $lhsp.castSel, $rhsp.castSel, ifAdjacentSel(VN_AS($lhsp,,Sel),,VN_AS($rhsp,,Sel)) , replaceConcatSel(nodep) )");
	    replaceConcatSel(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Concat_4(AstConcat* nodep) {
	// TREEOPV("AstConcat{ifConcatMergeableBiop($lhsp), concatMergeable($lhsp,,$rhsp,,0)}", "replaceConcatMerge(nodep)")
	if (m_doV && ifConcatMergeableBiop(nodep->lhsp()) && concatMergeable(nodep->lhsp(),nodep->rhsp(),0)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstConcat ifConcatMergeableBiop($lhsp), concatMergeable($lhsp,,$rhsp,,0) , replaceConcatMerge(nodep) )");
	    replaceConcatMerge(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Concat_5(AstConcat* nodep) {
	// TREEOPV("AstConcat{operandConcatSame(nodep)}", "DONE")
	if (m_doV && operandConcatSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstConcat operandConcatSame(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_0(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.isZero,       $thenp, $elsep}", "replaceWChild(nodep,$elsep)")
	if (m_doNConst && nodep->condp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.isZero, $thenp, $elsep , replaceWChild(nodep,$elsep) )");
	    replaceWChild(nodep,nodep->elsep());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_1(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.isNeqZero,    $thenp, $elsep}", "replaceWChild(nodep,$thenp)")
	if (m_doNConst && nodep->condp()->isNeqZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.isNeqZero, $thenp, $elsep , replaceWChild(nodep,$thenp) )");
	    replaceWChild(nodep,nodep->thenp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_2(AstCond* nodep) {
	// TREEOPA("AstCond{$condp.isZero,       $thenp.castConst, $elsep.castConst}", "replaceWChild(nodep,$elsep)")
	if (nodep->condp()->isZero() && VN_IS(nodep->thenp(),Const) && VN_IS(nodep->elsep(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstCond $condp.isZero, $thenp.castConst, $elsep.castConst , replaceWChild(nodep,$elsep) )");
	    replaceWChild(nodep,nodep->elsep());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_3(AstCond* nodep) {
	// TREEOPA("AstCond{$condp.isNeqZero,    $thenp.castConst, $elsep.castConst}", "replaceWChild(nodep,$thenp)")
	if (nodep->condp()->isNeqZero() && VN_IS(nodep->thenp(),Const) && VN_IS(nodep->elsep(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstCond $condp.isNeqZero, $thenp.castConst, $elsep.castConst , replaceWChild(nodep,$thenp) )");
	    replaceWChild(nodep,nodep->thenp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_4(AstCond* nodep) {
	// TREEOP ("AstCond{$condp, operandsSame($thenp,,$elsep)}","replaceWChild(nodep,$thenp)")
	if (m_doNConst && operandsSame(nodep->thenp(),nodep->elsep())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp, operandsSame($thenp,,$elsep) , replaceWChild(nodep,$thenp) )");
	    replaceWChild(nodep,nodep->thenp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_5(AstCond* nodep) {
	// TREEOP ("AstCond{$condp, matchCondCond(nodep)}", "DONE")
	if (m_doNConst && matchCondCond(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp, matchCondCond(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_6(AstCond* nodep) {
	// TREEOPS("AstCond{$condp.isZero}",           "replaceWIteratedThs(nodep)")
	if (m_doNConst && nodep->condp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPS( AstCond $condp.isZero , replaceWIteratedThs(nodep) )");
	    replaceWIteratedThs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_7(AstCond* nodep) {
	// TREEOPS("AstCond{$condp.isNeqZero}",        "replaceWIteratedRhs(nodep)")
	if (m_doNConst && nodep->condp()->isNeqZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPS( AstCond $condp.isNeqZero , replaceWIteratedRhs(nodep) )");
	    replaceWIteratedRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_8(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.castNot, $thenp, $elsep}", "AstCond{$condp->castNot()->lhsp(), $elsep, $thenp}")
	if (m_doNConst && VN_IS(nodep->condp(),Not)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.castNot, $thenp, $elsep , AstCond $condp->castNot()->lhsp(), $elsep, $thenp )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->condp(),Not)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->elsep()->unlinkFrBack();
AstNodeExpr* arg3p = nodep->thenp()->unlinkFrBack();
AstNodeExpr* newp = new AstCond(nodep->fileline(), arg1p, arg2p, arg3p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_9(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp.isAllOnes, $elsep}", "AstLogOr {$condp, $elsep}")
	if (m_doNConst && nodep->condp()->width1() && nodep->thenp()->width1() && nodep->thenp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.width1, $thenp.width1, $thenp.isAllOnes, $elsep , AstLogOr $condp, $elsep )");
	    AstNodeExpr* arg1p = nodep->condp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->elsep()->unlinkFrBack();
AstNodeExpr* newp = new AstLogOr(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_10(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp, $elsep.isZero, !$elsep.isClassHandleValue}", "AstLogAnd{$condp, $thenp}")
	if (m_doNConst && nodep->condp()->width1() && nodep->thenp()->width1() && nodep->elsep()->isZero() && !nodep->elsep()->isClassHandleValue()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.width1, $thenp.width1, $thenp, $elsep.isZero, !$elsep.isClassHandleValue , AstLogAnd $condp, $thenp )");
	    AstNodeExpr* arg1p = nodep->condp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->thenp()->unlinkFrBack();
AstNodeExpr* newp = new AstLogAnd(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_11(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp, $elsep.isAllOnes}", "AstLogOr {AstNot{$condp}, $thenp}")
	if (m_doNConst && nodep->condp()->width1() && nodep->thenp()->width1() && nodep->elsep()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.width1, $thenp.width1, $thenp, $elsep.isAllOnes , AstLogOr AstNot $condp , $thenp )");
	    AstNodeExpr* arg1p = nodep->condp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->thenp()->unlinkFrBack();
AstNodeExpr* newp = new AstLogOr(nodep->fileline(), new AstNot(nodep->fileline(), arg1p), arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_12(AstCond* nodep) {
	// TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp.isZero, !$thenp.isClassHandleValue, $elsep}", "AstLogAnd{AstNot{$condp}, $elsep}")
	if (m_doNConst && nodep->condp()->width1() && nodep->thenp()->width1() && nodep->thenp()->isZero() && !nodep->thenp()->isClassHandleValue()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond $condp.width1, $thenp.width1, $thenp.isZero, !$thenp.isClassHandleValue, $elsep , AstLogAnd AstNot $condp , $elsep )");
	    AstNodeExpr* arg1p = nodep->condp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->elsep()->unlinkFrBack();
AstNodeExpr* newp = new AstLogAnd(nodep->fileline(), new AstNot(nodep->fileline(), arg1p), arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Cond_13(AstCond* nodep) {
	// TREEOP ("AstCond{!$condp.width1, operandBoolShift(nodep->condp())}", "replaceBoolShift(nodep->condp())")
	if (m_doNConst && !nodep->condp()->width1() && operandBoolShift(nodep->condp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstCond !$condp.width1, operandBoolShift(nodep->condp()) , replaceBoolShift(nodep->condp()) )");
	    replaceBoolShift(nodep->condp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_CvtPackString_0(AstCvtPackString* nodep) {
	// TREEOPA("AstCvtPackString{$lhsp.castConst}", "replaceConstString(nodep, VN_AS(nodep->lhsp(), Const)->num().toString())")
	if (VN_IS(nodep->lhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstCvtPackString $lhsp.castConst , replaceConstString(nodep, VN_AS(nodep->lhsp(), Const)->num().toString()) )");
	    replaceConstString(nodep, VN_AS(nodep->lhsp(), Const)->num().toString());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Div_0(AstDiv* nodep) {
	// TREEOP ("AstDiv   {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDiv $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Div_1(AstDiv* nodep) {
	// TREEOP ("AstDiv   {$lhsp, $rhsp.isOne}",    "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isOne()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDiv $lhsp, $rhsp.isOne , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Div_2(AstDiv* nodep) {
	// TREEOP ("AstDiv   {$lhsp, operandIsPowTwo($rhsp)}", "replaceDivShift(nodep)")
	if (m_doNConst && operandIsPowTwo(nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDiv $lhsp, operandIsPowTwo($rhsp) , replaceDivShift(nodep) )");
	    replaceDivShift(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Div_3(AstDiv* nodep) {
	// TREEOP ("AstDiv    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDiv operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_DivS_0(AstDivS* nodep) {
	// TREEOP ("AstDivS  {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDivS $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_DivS_1(AstDivS* nodep) {
	// TREEOP ("AstDivS  {$lhsp, $rhsp.isOne}",    "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isOne()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDivS $lhsp, $rhsp.isOne , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_DivS_2(AstDivS* nodep) {
	// TREEOP ("AstDivS   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstDivS operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_0(AstEq* nodep) {
	// TREEOPV("AstEq    {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstShrink(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstEq $rhsp.castExtend,operandBiExtendConstShrink(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_1(AstEq* nodep) {
	// TREEOPV("AstEq    {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceZero(nodep)")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstOver(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstEq $rhsp.castExtend,operandBiExtendConstOver(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_2(AstEq* nodep) {
	// TREEOP ("AstEq     {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstEq operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_3(AstEq* nodep) {
	// TREEOPV("AstEq    {$rhsp.width1, $lhsp.isZero,    $rhsp}",  "AstNot{$rhsp}")
	if (m_doV && nodep->rhsp()->width1() && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstEq $rhsp.width1, $lhsp.isZero, $rhsp , AstNot $rhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNot(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_4(AstEq* nodep) {
	// TREEOPV("AstEq    {$lhsp.width1, $lhsp, $rhsp.isZero}",     "AstNot{$lhsp}")
	if (m_doV && nodep->lhsp()->width1() && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstEq $lhsp.width1, $lhsp, $rhsp.isZero , AstNot $lhsp )");
	    AstNodeExpr* arg1p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNot(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_5(AstEq* nodep) {
	// TREEOPV("AstEq    {$rhsp.width1, $lhsp.isAllOnes, $rhsp}",  "replaceWRhs(nodep)")
	if (m_doV && nodep->rhsp()->width1() && nodep->lhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstEq $rhsp.width1, $lhsp.isAllOnes, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Eq_6(AstEq* nodep) {
	// TREEOPV("AstEq    {$lhsp.width1, $lhsp, $rhsp.isAllOnes}",  "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1() && nodep->rhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstEq $lhsp.width1, $lhsp, $rhsp.isAllOnes , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_EqCase_0(AstEqCase* nodep) {
	// TREEOP ("AstEqCase {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstEqCase operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_EqD_0(AstEqD* nodep) {
	// TREEOP ("AstEqD    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstEqD operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_EqN_0(AstEqN* nodep) {
	// TREEOP ("AstEqN    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstEqN operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_EqWild_0(AstEqWild* nodep) {
	// TREEOP ("AstEqWild {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstEqWild operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Extend_0(AstExtend* nodep) {
	// TREEOPV("AstExtend{operandsSameWidth(nodep,,$lhsp)}",  "replaceWLhs(nodep)")
	if (m_doV && operandsSameWidth(nodep,nodep->lhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstExtend operandsSameWidth(nodep,,$lhsp) , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Extend_1(AstExtend* nodep) {
	// TREEOPV("AstExtend{$lhsp.castExtend}",  "replaceExtend(nodep, VN_AS(nodep->lhsp(), Extend)->lhsp())")
	if (m_doV && VN_IS(nodep->lhsp(),Extend)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstExtend $lhsp.castExtend , replaceExtend(nodep, VN_AS(nodep->lhsp(), Extend)->lhsp()) )");
	    replaceExtend(nodep, VN_AS(nodep->lhsp(), Extend)->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ExtendS_0(AstExtendS* nodep) {
	// TREEOPV("AstExtendS{$lhsp.castExtendS}", "replaceExtend(nodep, VN_AS(nodep->lhsp(), ExtendS)->lhsp())")
	if (m_doV && VN_IS(nodep->lhsp(),ExtendS)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstExtendS $lhsp.castExtendS , replaceExtend(nodep, VN_AS(nodep->lhsp(), ExtendS)->lhsp()) )");
	    replaceExtend(nodep, VN_AS(nodep->lhsp(), ExtendS)->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_0(AstGt* nodep) {
	// TREEOP ("AstGt   {!$lhsp.castConst,$rhsp.castConst}",       "AstLt  {$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGt !$lhsp.castConst,$rhsp.castConst , AstLt $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLt(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_1(AstGt* nodep) {
	// TREEOP1("AstGt   {$lhsp.isZero, $rhsp}",            "replaceNumSigned(nodep,0)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstGt $lhsp.isZero, $rhsp , replaceNumSigned(nodep,0) )");
	    replaceNumSigned(nodep,0);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_2(AstGt* nodep) {
	// TREEOP1("AstGt   {$lhsp, $rhsp.isAllOnes, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,0)")
	if (m_doNConst && nodep->rhsp()->isAllOnes() && nodep->lhsp()->width()==nodep->rhsp()->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstGt $lhsp, $rhsp.isAllOnes, $lhsp->width()==$rhsp->width() , replaceNumLimited(nodep,0) )");
	    replaceNumLimited(nodep,0);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_3(AstGt* nodep) {
	// TREEOPV("AstGt    {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstShrink(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstGt $rhsp.castExtend,operandBiExtendConstShrink(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_4(AstGt* nodep) {
	// TREEOPV("AstGt    {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceNum(nodep,1)")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstOver(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstGt $rhsp.castExtend,operandBiExtendConstOver(nodep) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_5(AstGt* nodep) {
	// TREEOP ("AstGt     {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGt operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gt_6(AstGt* nodep) {
	// TREEOPV("AstGt    {$lhsp.width1, $lhsp, $rhsp.isZero}",     "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1() && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstGt $lhsp.width1, $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GtD_0(AstGtD* nodep) {
	// TREEOP ("AstGtD    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGtD operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GtN_0(AstGtN* nodep) {
	// TREEOP ("AstGtN    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGtN operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GtS_0(AstGtS* nodep) {
	// TREEOP ("AstGtS  {!$lhsp.castConst,$rhsp.castConst}",       "AstLtS {$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGtS !$lhsp.castConst,$rhsp.castConst , AstLtS $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLtS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GtS_1(AstGtS* nodep) {
	// TREEOP ("AstGtS    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGtS operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gte_0(AstGte* nodep) {
	// TREEOP ("AstGte  {!$lhsp.castConst,$rhsp.castConst}",       "AstLte {$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGte !$lhsp.castConst,$rhsp.castConst , AstLte $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLte(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gte_1(AstGte* nodep) {
	// TREEOP1("AstGte  {$lhsp, $rhsp.isZero}",            "replaceNumSigned(nodep,1)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstGte $lhsp, $rhsp.isZero , replaceNumSigned(nodep,1) )");
	    replaceNumSigned(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gte_2(AstGte* nodep) {
	// TREEOP1("AstGte  {$lhsp.isAllOnes, $rhsp, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,1)")
	if (m_doNConst && nodep->lhsp()->isAllOnes() && nodep->lhsp()->width()==nodep->rhsp()->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstGte $lhsp.isAllOnes, $rhsp, $lhsp->width()==$rhsp->width() , replaceNumLimited(nodep,1) )");
	    replaceNumLimited(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gte_3(AstGte* nodep) {
	// TREEOPV("AstGte   {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstShrink(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstGte $rhsp.castExtend,operandBiExtendConstShrink(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gte_4(AstGte* nodep) {
	// TREEOPV("AstGte   {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceNum(nodep,1)")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstOver(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstGte $rhsp.castExtend,operandBiExtendConstOver(nodep) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Gte_5(AstGte* nodep) {
	// TREEOP ("AstGte    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGte operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GteD_0(AstGteD* nodep) {
	// TREEOP ("AstGteD   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGteD operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GteN_0(AstGteN* nodep) {
	// TREEOP ("AstGteN   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGteN operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GteS_0(AstGteS* nodep) {
	// TREEOP ("AstGteS {!$lhsp.castConst,$rhsp.castConst}",       "AstLteS{$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGteS !$lhsp.castConst,$rhsp.castConst , AstLteS $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLteS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_GteS_1(AstGteS* nodep) {
	// TREEOP ("AstGteS   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstGteS operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_IsUnbounded_0(AstIsUnbounded* nodep) {
	// TREEOPV("AstIsUnbounded{$lhsp.castUnbounded}", "replaceNum(nodep, 1)")
	if (m_doV && VN_IS(nodep->lhsp(),Unbounded)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstIsUnbounded $lhsp.castUnbounded , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_0(AstLogAnd* nodep) {
	// TREEOPS("AstLogAnd   {$lhsp.isZero}",       "replaceZero(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPS( AstLogAnd $lhsp.isZero , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_1(AstLogAnd* nodep) {
	// TREEOP ("AstLogAnd{$lhsp.isZero, $rhsp}",   "replaceZero(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogAnd $lhsp.isZero, $rhsp , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_2(AstLogAnd* nodep) {
	// TREEOP ("AstLogAnd{$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogAnd $lhsp, $rhsp.isZero , replaceZeroChkPure(nodep,$lhsp) )");
	    replaceZeroChkPure(nodep,nodep->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_3(AstLogAnd* nodep) {
	// TREEOP ("AstLogAnd{$lhsp.isNeqZero, $rhsp}",        "replaceWRhsBool(nodep)")
	if (m_doNConst && nodep->lhsp()->isNeqZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogAnd $lhsp.isNeqZero, $rhsp , replaceWRhsBool(nodep) )");
	    replaceWRhsBool(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_4(AstLogAnd* nodep) {
	// TREEOP ("AstLogAnd{$lhsp, $rhsp.isNeqZero}",        "replaceWLhsBool(nodep)")
	if (m_doNConst && nodep->rhsp()->isNeqZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogAnd $lhsp, $rhsp.isNeqZero , replaceWLhsBool(nodep) )");
	    replaceWLhsBool(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_5(AstLogAnd* nodep) {
	// TREEOP ("AstLogAnd {operandsSame($lhsp,,$rhsp)}",   "replaceWLhsBool(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogAnd operandsSame($lhsp,,$rhsp) , replaceWLhsBool(nodep) )");
	    replaceWLhsBool(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogAnd_6(AstLogAnd* nodep) {
	// TREEOPV("AstLogAnd{matchBiopToBitwise(nodep)}", "AstAnd{$lhsp,$rhsp}")
	if (m_doV && matchBiopToBitwise(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLogAnd matchBiopToBitwise(nodep) , AstAnd $lhsp,$rhsp )");
	    AstNodeExpr* arg1p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstAnd(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogEq_0(AstLogEq* nodep) {
	// TREEOPV("AstLogEq{$lhsp, $rhsp}",  "replaceLogEq(nodep)")
	if (m_doV) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLogEq $lhsp, $rhsp , replaceLogEq(nodep) )");
	    replaceLogEq(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogIf_0(AstLogIf* nodep) {
	// TREEOPS("AstLogIf{$lhsp.isZero}",  "replaceNum(nodep, 1)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPS( AstLogIf $lhsp.isZero , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogIf_1(AstLogIf* nodep) {
	// TREEOPV("AstLogIf{$lhsp, $rhsp}",  "AstLogOr{AstLogNot{$lhsp},$rhsp}")
	if (m_doV) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLogIf $lhsp, $rhsp , AstLogOr AstLogNot $lhsp ,$rhsp )");
	    AstNodeExpr* arg1p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLogOr(nodep->fileline(), new AstLogNot(nodep->fileline(), arg1p), arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_0(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castLogNot}",              "replaceWChild(nodep, $lhsp->castLogNot()->lhsp())")
	if (m_doNConst && VN_IS(nodep->lhsp(),LogNot)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castLogNot , replaceWChild(nodep, $lhsp->castLogNot()->lhsp()) )");
	    replaceWChild(nodep, VN_CAST(nodep->lhsp(),LogNot)->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_1(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castEqCase}",              "AstNeqCase{$lhsp->castEqCase()->lhsp(),$lhsp->castEqCase()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),EqCase)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castEqCase , AstNeqCase $lhsp->castEqCase()->lhsp(),$lhsp->castEqCase()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),EqCase)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),EqCase)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNeqCase(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_2(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castNeqCase}",             "AstEqCase {$lhsp->castNeqCase()->lhsp(),$lhsp->castNeqCase()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),NeqCase)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castNeqCase , AstEqCase $lhsp->castNeqCase()->lhsp(),$lhsp->castNeqCase()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),NeqCase)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),NeqCase)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstEqCase(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_3(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castEqWild}",              "AstNeqWild{$lhsp->castEqWild()->lhsp(),$lhsp->castEqWild()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),EqWild)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castEqWild , AstNeqWild $lhsp->castEqWild()->lhsp(),$lhsp->castEqWild()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),EqWild)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),EqWild)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNeqWild(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_4(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castNeqWild}",             "AstEqWild {$lhsp->castNeqWild()->lhsp(),$lhsp->castNeqWild()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),NeqWild)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castNeqWild , AstEqWild $lhsp->castNeqWild()->lhsp(),$lhsp->castNeqWild()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),NeqWild)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),NeqWild)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstEqWild(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_5(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castEq}",                  "AstNeq {$lhsp->castEq()->lhsp(),$lhsp->castEq()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Eq)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castEq , AstNeq $lhsp->castEq()->lhsp(),$lhsp->castEq()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Eq)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Eq)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNeq(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_6(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castNeq}",                 "AstEq  {$lhsp->castNeq()->lhsp(),$lhsp->castNeq()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Neq)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castNeq , AstEq $lhsp->castNeq()->lhsp(),$lhsp->castNeq()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Neq)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Neq)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstEq(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_7(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castLt}",                  "AstGte {$lhsp->castLt()->lhsp(),$lhsp->castLt()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Lt)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castLt , AstGte $lhsp->castLt()->lhsp(),$lhsp->castLt()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Lt)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Lt)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGte(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_8(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castLtS}",                 "AstGteS{$lhsp->castLtS()->lhsp(),$lhsp->castLtS()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),LtS)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castLtS , AstGteS $lhsp->castLtS()->lhsp(),$lhsp->castLtS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),LtS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),LtS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGteS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_9(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castLte}",                 "AstGt  {$lhsp->castLte()->lhsp(),$lhsp->castLte()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Lte)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castLte , AstGt $lhsp->castLte()->lhsp(),$lhsp->castLte()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Lte)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Lte)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGt(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_10(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castLteS}",                "AstGtS {$lhsp->castLteS()->lhsp(),$lhsp->castLteS()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),LteS)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castLteS , AstGtS $lhsp->castLteS()->lhsp(),$lhsp->castLteS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),LteS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),LteS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGtS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_11(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castGt}",                  "AstLte {$lhsp->castGt()->lhsp(),$lhsp->castGt()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Gt)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castGt , AstLte $lhsp->castGt()->lhsp(),$lhsp->castGt()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Gt)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Gt)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLte(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_12(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castGtS}",                 "AstLteS{$lhsp->castGtS()->lhsp(),$lhsp->castGtS()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),GtS)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castGtS , AstLteS $lhsp->castGtS()->lhsp(),$lhsp->castGtS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),GtS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),GtS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLteS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_13(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castGte}",                 "AstLt  {$lhsp->castGte()->lhsp(),$lhsp->castGte()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Gte)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castGte , AstLt $lhsp->castGte()->lhsp(),$lhsp->castGte()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Gte)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Gte)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLt(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_14(AstLogNot* nodep) {
	// TREEOP ("AstLogNot{$lhsp.castGteS}",                "AstLtS {$lhsp->castGteS()->lhsp(),$lhsp->castGteS()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),GteS)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogNot $lhsp.castGteS , AstLtS $lhsp->castGteS()->lhsp(),$lhsp->castGteS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),GteS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),GteS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLtS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogNot_15(AstLogNot* nodep) {
	// TREEOPV("AstLogNot{$lhsp.width1}",  "AstNot{$lhsp}")
	if (m_doV && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLogNot $lhsp.width1 , AstNot $lhsp )");
	    AstNodeExpr* arg1p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNot(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_0(AstLogOr* nodep) {
	// TREEOPS("AstLogOr   {$lhsp.isOne}",         "replaceNum(nodep, 1)")
	if (m_doNConst && nodep->lhsp()->isOne()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPS( AstLogOr $lhsp.isOne , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_1(AstLogOr* nodep) {
	// TREEOP ("AstLogOr {$lhsp.isZero, $rhsp}",   "replaceWRhsBool(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogOr $lhsp.isZero, $rhsp , replaceWRhsBool(nodep) )");
	    replaceWRhsBool(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_2(AstLogOr* nodep) {
	// TREEOP ("AstLogOr {$lhsp, $rhsp.isZero}",   "replaceWLhsBool(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogOr $lhsp, $rhsp.isZero , replaceWLhsBool(nodep) )");
	    replaceWLhsBool(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_3(AstLogOr* nodep) {
	// TREEOP ("AstLogOr {$lhsp.isNeqZero, $rhsp}",        "replaceNum(nodep,1)")
	if (m_doNConst && nodep->lhsp()->isNeqZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogOr $lhsp.isNeqZero, $rhsp , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_4(AstLogOr* nodep) {
	// TREEOP ("AstLogOr {$lhsp, $rhsp.isNeqZero, $lhsp.isPure, nodep->isPure()}",        "replaceNum(nodep,1)")
	if (m_doNConst && nodep->rhsp()->isNeqZero() && nodep->lhsp()->isPure() && nodep->isPure()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogOr $lhsp, $rhsp.isNeqZero, $lhsp.isPure, nodep->isPure() , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_5(AstLogOr* nodep) {
	// TREEOP ("AstLogOr  {operandsSame($lhsp,,$rhsp)}",   "replaceWLhsBool(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLogOr operandsSame($lhsp,,$rhsp) , replaceWLhsBool(nodep) )");
	    replaceWLhsBool(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LogOr_6(AstLogOr* nodep) {
	// TREEOPV("AstLogOr {matchBiopToBitwise(nodep)}", "AstOr{$lhsp,$rhsp}")
	if (m_doV && matchBiopToBitwise(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLogOr matchBiopToBitwise(nodep) , AstOr $lhsp,$rhsp )");
	    AstNodeExpr* arg1p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstOr(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_0(AstLt* nodep) {
	// TREEOP ("AstLt   {!$lhsp.castConst,$rhsp.castConst}",       "AstGt  {$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLt !$lhsp.castConst,$rhsp.castConst , AstGt $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGt(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_1(AstLt* nodep) {
	// TREEOP1("AstLt   {$lhsp, $rhsp.isZero}",            "replaceNumSigned(nodep,0)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstLt $lhsp, $rhsp.isZero , replaceNumSigned(nodep,0) )");
	    replaceNumSigned(nodep,0);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_2(AstLt* nodep) {
	// TREEOP1("AstLt   {$lhsp.isAllOnes, $rhsp, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,0)")
	if (m_doNConst && nodep->lhsp()->isAllOnes() && nodep->lhsp()->width()==nodep->rhsp()->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstLt $lhsp.isAllOnes, $rhsp, $lhsp->width()==$rhsp->width() , replaceNumLimited(nodep,0) )");
	    replaceNumLimited(nodep,0);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_3(AstLt* nodep) {
	// TREEOPV("AstLt    {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstShrink(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLt $rhsp.castExtend,operandBiExtendConstShrink(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_4(AstLt* nodep) {
	// TREEOPV("AstLt    {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceZero(nodep)")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstOver(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLt $rhsp.castExtend,operandBiExtendConstOver(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_5(AstLt* nodep) {
	// TREEOP ("AstLt     {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLt operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lt_6(AstLt* nodep) {
	// TREEOPV("AstLt    {$rhsp.width1, $lhsp.isZero,    $rhsp}",  "replaceWRhs(nodep)")
	if (m_doV && nodep->rhsp()->width1() && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLt $rhsp.width1, $lhsp.isZero, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LtD_0(AstLtD* nodep) {
	// TREEOP ("AstLtD    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLtD operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LtN_0(AstLtN* nodep) {
	// TREEOP ("AstLtN    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLtN operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LtS_0(AstLtS* nodep) {
	// TREEOP ("AstLtS  {!$lhsp.castConst,$rhsp.castConst}",       "AstGtS {$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLtS !$lhsp.castConst,$rhsp.castConst , AstGtS $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGtS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LtS_1(AstLtS* nodep) {
	// TREEOP ("AstLtS    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLtS operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_0(AstLte* nodep) {
	// TREEOP ("AstLte  {!$lhsp.castConst,$rhsp.castConst}",       "AstGte {$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLte !$lhsp.castConst,$rhsp.castConst , AstGte $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGte(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_1(AstLte* nodep) {
	// TREEOP1("AstLte  {$lhsp.isZero, $rhsp}",            "replaceNumSigned(nodep,1)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstLte $lhsp.isZero, $rhsp , replaceNumSigned(nodep,1) )");
	    replaceNumSigned(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_2(AstLte* nodep) {
	// TREEOP1("AstLte  {$lhsp, $rhsp.isAllOnes, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,1)")
	if (m_doNConst && nodep->rhsp()->isAllOnes() && nodep->lhsp()->width()==nodep->rhsp()->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstLte $lhsp, $rhsp.isAllOnes, $lhsp->width()==$rhsp->width() , replaceNumLimited(nodep,1) )");
	    replaceNumLimited(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_3(AstLte* nodep) {
	// TREEOPV("AstLte   {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstShrink(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLte $rhsp.castExtend,operandBiExtendConstShrink(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_4(AstLte* nodep) {
	// TREEOPV("AstLte   {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceZero(nodep)")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstOver(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLte $rhsp.castExtend,operandBiExtendConstOver(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_5(AstLte* nodep) {
	// TREEOP ("AstLte    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLte operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Lte_6(AstLte* nodep) {
	// TREEOPV("AstLte   {$lhsp->width()==$rhsp->width(), $rhsp.isAllOnes}", "replaceNum(nodep,1)")
	if (m_doV && nodep->lhsp()->width()==nodep->rhsp()->width() && nodep->rhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstLte $lhsp->width()==$rhsp->width(), $rhsp.isAllOnes , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LteD_0(AstLteD* nodep) {
	// TREEOP ("AstLteD   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLteD operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LteN_0(AstLteN* nodep) {
	// TREEOP ("AstLteN   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLteN operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LteS_0(AstLteS* nodep) {
	// TREEOP ("AstLteS {!$lhsp.castConst,$rhsp.castConst}",       "AstGteS{$rhsp,$lhsp}")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLteS !$lhsp.castConst,$rhsp.castConst , AstGteS $rhsp,$lhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGteS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_LteS_1(AstLteS* nodep) {
	// TREEOP ("AstLteS   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstLteS operandsSame($lhsp,,$rhsp) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ModDiv_0(AstModDiv* nodep) {
	// TREEOP ("AstModDiv{$lhsp, operandIsPowTwo($rhsp)}", "replaceModAnd(nodep)")
	if (m_doNConst && operandIsPowTwo(nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstModDiv $lhsp, operandIsPowTwo($rhsp) , replaceModAnd(nodep) )");
	    replaceModAnd(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Mul_0(AstMul* nodep) {
	// TREEOP ("AstMul   {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMul $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Mul_1(AstMul* nodep) {
	// TREEOP ("AstMul   {$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMul $lhsp, $rhsp.isZero , replaceZeroChkPure(nodep,$lhsp) )");
	    replaceZeroChkPure(nodep,nodep->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Mul_2(AstMul* nodep) {
	// TREEOP ("AstMul   {$lhsp.isOne, $rhsp}",    "replaceWRhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isOne()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMul $lhsp.isOne, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Mul_3(AstMul* nodep) {
	// TREEOP ("AstMul   {operandIsPowTwo($lhsp), operandsSameWidth($lhsp,,$rhsp)}", "replaceMulShift(nodep)")
	if (m_doNConst && operandIsPowTwo(nodep->lhsp()) && operandsSameWidth(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMul operandIsPowTwo($lhsp), operandsSameWidth($lhsp,,$rhsp) , replaceMulShift(nodep) )");
	    replaceMulShift(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_MulS_0(AstMulS* nodep) {
	// TREEOP ("AstMulS  {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMulS $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_MulS_1(AstMulS* nodep) {
	// TREEOP ("AstMulS  {$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMulS $lhsp, $rhsp.isZero , replaceZeroChkPure(nodep,$lhsp) )");
	    replaceZeroChkPure(nodep,nodep->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_MulS_2(AstMulS* nodep) {
	// TREEOP ("AstMulS  {$lhsp.isOne, $rhsp}",    "replaceWRhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isOne()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstMulS $lhsp.isOne, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_0(AstNeq* nodep) {
	// TREEOPV("AstNeq   {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstShrink(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNeq $rhsp.castExtend,operandBiExtendConstShrink(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_1(AstNeq* nodep) {
	// TREEOPV("AstNeq   {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceNum(nodep,1)")
	if (m_doV && VN_IS(nodep->rhsp(),Extend) && operandBiExtendConstOver(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNeq $rhsp.castExtend,operandBiExtendConstOver(nodep) , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_2(AstNeq* nodep) {
	// TREEOP ("AstNeq    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNeq operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_3(AstNeq* nodep) {
	// TREEOPV("AstNeq   {$rhsp.width1, $lhsp.isZero,    $rhsp}",  "replaceWRhs(nodep)")
	if (m_doV && nodep->rhsp()->width1() && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNeq $rhsp.width1, $lhsp.isZero, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_4(AstNeq* nodep) {
	// TREEOPV("AstNeq   {$lhsp.width1, $lhsp, $rhsp.isZero}",     "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1() && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNeq $lhsp.width1, $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_5(AstNeq* nodep) {
	// TREEOPV("AstNeq   {$rhsp.width1, $lhsp.isAllOnes, $rhsp}",  "AstNot{$rhsp}")
	if (m_doV && nodep->rhsp()->width1() && nodep->lhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNeq $rhsp.width1, $lhsp.isAllOnes, $rhsp , AstNot $rhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNot(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Neq_6(AstNeq* nodep) {
	// TREEOPV("AstNeq   {$lhsp.width1, $lhsp, $rhsp.isAllOnes}",  "AstNot{$lhsp}")
	if (m_doV && nodep->lhsp()->width1() && nodep->rhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNeq $lhsp.width1, $lhsp, $rhsp.isAllOnes , AstNot $lhsp )");
	    AstNodeExpr* arg1p = nodep->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNot(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NeqCase_0(AstNeqCase* nodep) {
	// TREEOP ("AstNeqCase{operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNeqCase operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NeqD_0(AstNeqD* nodep) {
	// TREEOP ("AstNeqD   {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNeqD operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NeqN_0(AstNeqN* nodep) {
	// TREEOP ("AstNeqN   {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNeqN operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NeqWild_0(AstNeqWild* nodep) {
	// TREEOP ("AstNeqWild{operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNeqWild operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeBiCom_0(AstNodeBiCom* nodep) {
	// TREEOP ("AstNodeBiCom{!$lhsp.castConst, $rhsp.castConst}",  "swapSides(nodep)")
	if (m_doNConst && !VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNodeBiCom !$lhsp.castConst, $rhsp.castConst , swapSides(nodep) )");
	    swapSides(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeBiComAsv_0(AstNodeBiComAsv* nodep) {
	// TREEOP ("AstNodeBiComAsv{operandAsvConst(nodep)}",  "replaceAsv(nodep)")
	if (m_doNConst && operandAsvConst(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNodeBiComAsv operandAsvConst(nodep) , replaceAsv(nodep) )");
	    replaceAsv(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeBiComAsv_1(AstNodeBiComAsv* nodep) {
	// TREEOP ("AstNodeBiComAsv{operandAsvSame(nodep)}",   "replaceAsv(nodep)")
	if (m_doNConst && operandAsvSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNodeBiComAsv operandAsvSame(nodep) , replaceAsv(nodep) )");
	    replaceAsv(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeBiComAsv_2(AstNodeBiComAsv* nodep) {
	// TREEOP ("AstNodeBiComAsv{operandAsvLUp(nodep)}",    "replaceAsvLUp(nodep)")
	if (m_doNConst && operandAsvLUp(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNodeBiComAsv operandAsvLUp(nodep) , replaceAsvLUp(nodep) )");
	    replaceAsvLUp(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeBiComAsv_3(AstNodeBiComAsv* nodep) {
	// TREEOP ("AstNodeBiComAsv{operandAsvRUp(nodep)}",    "replaceAsvRUp(nodep)")
	if (m_doNConst && operandAsvRUp(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNodeBiComAsv operandAsvRUp(nodep) , replaceAsvRUp(nodep) )");
	    replaceAsvRUp(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeBiop_0(AstNodeBiop* nodep) {
	// TREEOPA("AstNodeBiop {$lhsp.castConst, $rhsp.castConst, nodep->isPredictOptimizable()}",  "replaceConst(nodep)")
	if (VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const) && nodep->isPredictOptimizable()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstNodeBiop $lhsp.castConst, $rhsp.castConst, nodep->isPredictOptimizable() , replaceConst(nodep) )");
	    replaceConst(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeQuadop_0(AstNodeQuadop* nodep) {
	// TREEOPA("AstNodeQuadop{$lhsp.castConst, $rhsp.castConst, $thsp.castConst, $fhsp.castConst}",  "replaceConst(nodep)")
	if (VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const) && VN_IS(nodep->thsp(),Const) && VN_IS(nodep->fhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstNodeQuadop $lhsp.castConst, $rhsp.castConst, $thsp.castConst, $fhsp.castConst , replaceConst(nodep) )");
	    replaceConst(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_NodeUniop_0(AstNodeUniop* nodep) {
	// TREEOPA("AstNodeUniop{$lhsp.castConst, !nodep->isOpaque(), nodep->isPredictOptimizable()}",  "replaceConst(nodep)")
	if (VN_IS(nodep->lhsp(),Const) && !nodep->isOpaque() && nodep->isPredictOptimizable()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstNodeUniop $lhsp.castConst, !nodep->isOpaque(), nodep->isPredictOptimizable() , replaceConst(nodep) )");
	    replaceConst(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_0(AstNot* nodep) {
	// TREEOP ("AstNot   {$lhsp.castNot,  $lhsp->width()==VN_AS($lhsp,,Not)->lhsp()->width()}", "replaceWChild(nodep, $lhsp->castNot()->lhsp())")
	if (m_doNConst && VN_IS(nodep->lhsp(),Not) && nodep->lhsp()->width()==VN_AS(nodep->lhsp(),Not)->lhsp()->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstNot $lhsp.castNot, $lhsp->width()==VN_AS($lhsp,,Not)->lhsp()->width() , replaceWChild(nodep, $lhsp->castNot()->lhsp()) )");
	    replaceWChild(nodep, VN_CAST(nodep->lhsp(),Not)->lhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_1(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castEqCase, $lhsp.width1}","AstNeqCase{$lhsp->castEqCase()->lhsp(),$lhsp->castEqCase()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),EqCase) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castEqCase, $lhsp.width1 , AstNeqCase $lhsp->castEqCase()->lhsp(),$lhsp->castEqCase()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),EqCase)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),EqCase)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNeqCase(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_2(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castNeqCase, $lhsp.width1}","AstEqCase{$lhsp->castNeqCase()->lhsp(),$lhsp->castNeqCase()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),NeqCase) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castNeqCase, $lhsp.width1 , AstEqCase $lhsp->castNeqCase()->lhsp(),$lhsp->castNeqCase()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),NeqCase)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),NeqCase)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstEqCase(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_3(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castEqWild, $lhsp.width1}","AstNeqWild{$lhsp->castEqWild()->lhsp(),$lhsp->castEqWild()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),EqWild) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castEqWild, $lhsp.width1 , AstNeqWild $lhsp->castEqWild()->lhsp(),$lhsp->castEqWild()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),EqWild)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),EqWild)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNeqWild(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_4(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castNeqWild, $lhsp.width1}","AstEqWild{$lhsp->castNeqWild()->lhsp(),$lhsp->castNeqWild()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),NeqWild) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castNeqWild, $lhsp.width1 , AstEqWild $lhsp->castNeqWild()->lhsp(),$lhsp->castNeqWild()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),NeqWild)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),NeqWild)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstEqWild(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_5(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castEq, $lhsp.width1}",    "AstNeq {$lhsp->castEq()->lhsp(),$lhsp->castEq()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Eq) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castEq, $lhsp.width1 , AstNeq $lhsp->castEq()->lhsp(),$lhsp->castEq()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Eq)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Eq)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNeq(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_6(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castNeq, $lhsp.width1}",   "AstEq  {$lhsp->castNeq()->lhsp(),$lhsp->castNeq()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Neq) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castNeq, $lhsp.width1 , AstEq $lhsp->castNeq()->lhsp(),$lhsp->castNeq()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Neq)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Neq)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstEq(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_7(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castLt, $lhsp.width1}",    "AstGte {$lhsp->castLt()->lhsp(),$lhsp->castLt()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Lt) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castLt, $lhsp.width1 , AstGte $lhsp->castLt()->lhsp(),$lhsp->castLt()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Lt)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Lt)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGte(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_8(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castLtS, $lhsp.width1}",   "AstGteS{$lhsp->castLtS()->lhsp(),$lhsp->castLtS()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),LtS) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castLtS, $lhsp.width1 , AstGteS $lhsp->castLtS()->lhsp(),$lhsp->castLtS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),LtS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),LtS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGteS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_9(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castLte, $lhsp.width1}",   "AstGt  {$lhsp->castLte()->lhsp(),$lhsp->castLte()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Lte) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castLte, $lhsp.width1 , AstGt $lhsp->castLte()->lhsp(),$lhsp->castLte()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Lte)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Lte)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGt(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_10(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castLteS, $lhsp.width1}",  "AstGtS {$lhsp->castLteS()->lhsp(),$lhsp->castLteS()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),LteS) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castLteS, $lhsp.width1 , AstGtS $lhsp->castLteS()->lhsp(),$lhsp->castLteS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),LteS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),LteS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstGtS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_11(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castGt, $lhsp.width1}",    "AstLte {$lhsp->castGt()->lhsp(),$lhsp->castGt()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Gt) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castGt, $lhsp.width1 , AstLte $lhsp->castGt()->lhsp(),$lhsp->castGt()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Gt)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Gt)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLte(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_12(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castGtS, $lhsp.width1}",   "AstLteS{$lhsp->castGtS()->lhsp(),$lhsp->castGtS()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),GtS) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castGtS, $lhsp.width1 , AstLteS $lhsp->castGtS()->lhsp(),$lhsp->castGtS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),GtS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),GtS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLteS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_13(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castGte, $lhsp.width1}",   "AstLt  {$lhsp->castGte()->lhsp(),$lhsp->castGte()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Gte) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castGte, $lhsp.width1 , AstLt $lhsp->castGte()->lhsp(),$lhsp->castGte()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Gte)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Gte)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLt(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Not_14(AstNot* nodep) {
	// TREEOPV("AstNot   {$lhsp.castGteS, $lhsp.width1}",  "AstLtS {$lhsp->castGteS()->lhsp(),$lhsp->castGteS()->rhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),GteS) && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstNot $lhsp.castGteS, $lhsp.width1 , AstLtS $lhsp->castGteS()->lhsp(),$lhsp->castGteS()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),GteS)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),GteS)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstLtS(nodep->fileline(), arg1p, arg2p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_OneHot_0(AstOneHot* nodep) {
	// TREEOPV("AstOneHot{$lhsp.width1}",          "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstOneHot $lhsp.width1 , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_OneHot0_0(AstOneHot0* nodep) {
	// TREEOPV("AstOneHot0{$lhsp.width1}",         "replaceNum(nodep,1)")
	if (m_doV && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstOneHot0 $lhsp.width1 , replaceNum(nodep,1) )");
	    replaceNum(nodep,1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_0(AstOr* nodep) {
	// TREEOP ("AstOr    {$lhsp.isZero, $rhsp}",   "replaceWRhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr $lhsp.isZero, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_1(AstOr* nodep) {
	// TREEOP ("AstOr    {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_2(AstOr* nodep) {
	// TREEOP ("AstOr    {$lhsp.isAllOnes, $rhsp, $rhsp.isPure}",        "replaceWLhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isAllOnes() && nodep->rhsp()->isPure()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr $lhsp.isAllOnes, $rhsp, $rhsp.isPure , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_3(AstOr* nodep) {
	// TREEOP ("AstOr    {$lhsp, $rhsp.isAllOnes, $lhsp.isPure}",        "replaceWRhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isAllOnes() && nodep->lhsp()->isPure()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr $lhsp, $rhsp.isAllOnes, $lhsp.isPure , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_4(AstOr* nodep) {
	// TREEOP ("AstOr     {operandsSame($lhsp,,$rhsp)}",   "replaceWLhs(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr operandsSame($lhsp,,$rhsp) , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_5(AstOr* nodep) {
	// TREEOP ("AstOr  {$lhsp.castAnd,$rhsp.castAnd,operandAndOrSame(nodep)}", "replaceAndOr(nodep)")
	if (m_doNConst && VN_IS(nodep->lhsp(),And) && VN_IS(nodep->rhsp(),And) && operandAndOrSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr $lhsp.castAnd,$rhsp.castAnd,operandAndOrSame(nodep) , replaceAndOr(nodep) )");
	    replaceAndOr(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_6(AstOr* nodep) {
	// TREEOP ("AstOr  {matchOrAndNot(nodep)}",            "DONE")
	if (m_doNConst && matchOrAndNot(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr matchOrAndNot(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_7(AstOr* nodep) {
	// TREEOP ("AstOr  {operandShiftSame(nodep)}",         "replaceShiftSame(nodep)")
	if (m_doNConst && operandShiftSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstOr operandShiftSame(nodep) , replaceShiftSame(nodep) )");
	    replaceShiftSame(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Or_8(AstOr* nodep) {
	// TREEOPC("AstOr  {matchBitOpTree(nodep)}", "DONE")
	if (m_doCpp && matchBitOpTree(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPC( AstOr matchBitOpTree(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Pow_0(AstPow* nodep) {
	// TREEOP ("AstPow   {$rhsp.isZero}",          "replaceNum(nodep, 1)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstPow $rhsp.isZero , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Pow_1(AstPow* nodep) {
	// TREEOP ("AstPow   {operandIsTwo($lhsp), !$rhsp.isZero}",    "replacePowShift(nodep)")
	if (m_doNConst && operandIsTwo(nodep->lhsp()) && !nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstPow operandIsTwo($lhsp), !$rhsp.isZero , replacePowShift(nodep) )");
	    replacePowShift(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_PowSS_0(AstPowSS* nodep) {
	// TREEOP ("AstPowSS {$rhsp.isZero}",          "replaceNum(nodep, 1)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstPowSS $rhsp.isZero , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_PowSU_0(AstPowSU* nodep) {
	// TREEOP ("AstPowSU {$rhsp.isZero}",          "replaceNum(nodep, 1)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstPowSU $rhsp.isZero , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_PowSU_1(AstPowSU* nodep) {
	// TREEOP ("AstPowSU {operandIsTwo($lhsp), !$rhsp.isZero}",    "replacePowShift(nodep)")
	if (m_doNConst && operandIsTwo(nodep->lhsp()) && !nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstPowSU operandIsTwo($lhsp), !$rhsp.isZero , replacePowShift(nodep) )");
	    replacePowShift(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_PowUS_0(AstPowUS* nodep) {
	// TREEOP ("AstPowUS {$rhsp.isZero}",          "replaceNum(nodep, 1)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstPowUS $rhsp.isZero , replaceNum(nodep, 1) )");
	    replaceNum(nodep, 1);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_PutcN_0(AstPutcN* nodep) {
	// TREEOPA("AstPutcN{$lhsp.castConst, $rhsp.castConst, $thsp.castConst}",  "replaceConst(nodep)")
	if (VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const) && VN_IS(nodep->thsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstPutcN $lhsp.castConst, $rhsp.castConst, $thsp.castConst , replaceConst(nodep) )");
	    replaceConst(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedAnd_0(AstRedAnd* nodep) {
	// TREEOPV("AstRedAnd{$lhsp, $lhsp.width1}",   "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedAnd $lhsp, $lhsp.width1 , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedAnd_1(AstRedAnd* nodep) {
	// TREEOPV("AstRedAnd{$lhsp.castConcat}",      "AstAnd{AstRedAnd{$lhsp->castConcat()->lhsp()}, AstRedAnd{$lhsp->castConcat()->rhsp()}}")
	if (m_doV && VN_IS(nodep->lhsp(),Concat)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedAnd $lhsp.castConcat , AstAnd AstRedAnd $lhsp->castConcat()->lhsp() , AstRedAnd $lhsp->castConcat()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Concat)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Concat)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstAnd(nodep->fileline(), new AstRedAnd(nodep->fileline(), arg1p), new AstRedAnd(nodep->fileline(), arg2p));
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedAnd_2(AstRedAnd* nodep) {
	// TREEOPV("AstRedAnd{$lhsp.castExtend, $lhsp->width() > VN_AS($lhsp,,Extend)->lhsp()->width()}", "replaceZero(nodep)")
	if (m_doV && VN_IS(nodep->lhsp(),Extend) && nodep->lhsp()->width() > VN_AS(nodep->lhsp(),Extend)->lhsp()->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedAnd $lhsp.castExtend, $lhsp->width() > VN_AS($lhsp,,Extend)->lhsp()->width() , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedOr_0(AstRedOr* nodep) {
	// TREEOPV("AstRedOr {$lhsp, $lhsp.width1}",   "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedOr $lhsp, $lhsp.width1 , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedOr_1(AstRedOr* nodep) {
	// TREEOPV("AstRedOr {$lhsp.castConcat}",      "AstOr {AstRedOr {$lhsp->castConcat()->lhsp()}, AstRedOr {$lhsp->castConcat()->rhsp()}}")
	if (m_doV && VN_IS(nodep->lhsp(),Concat)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedOr $lhsp.castConcat , AstOr AstRedOr $lhsp->castConcat()->lhsp() , AstRedOr $lhsp->castConcat()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Concat)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Concat)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstOr(nodep->fileline(), new AstRedOr(nodep->fileline(), arg1p), new AstRedOr(nodep->fileline(), arg2p));
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedOr_2(AstRedOr* nodep) {
	// TREEOPV("AstRedOr {$lhsp.castExtend}",      "AstRedOr {$lhsp->castExtend()->lhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Extend)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedOr $lhsp.castExtend , AstRedOr $lhsp->castExtend()->lhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Extend)->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstRedOr(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedXor_0(AstRedXor* nodep) {
	// TREEOPV("AstRedXor{$lhsp, $lhsp.width1}",   "replaceWLhs(nodep)")
	if (m_doV && nodep->lhsp()->width1()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedXor $lhsp, $lhsp.width1 , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedXor_1(AstRedXor* nodep) {
	// TREEOPV("AstRedXor{$lhsp.castConcat}",      "AstXor{AstRedXor{$lhsp->castConcat()->lhsp()}, AstRedXor{$lhsp->castConcat()->rhsp()}}")
	if (m_doV && VN_IS(nodep->lhsp(),Concat)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedXor $lhsp.castConcat , AstXor AstRedXor $lhsp->castConcat()->lhsp() , AstRedXor $lhsp->castConcat()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Concat)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Concat)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstXor(nodep->fileline(), new AstRedXor(nodep->fileline(), arg1p), new AstRedXor(nodep->fileline(), arg2p));
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedXor_2(AstRedXor* nodep) {
	// TREEOPV("AstRedXor{$lhsp.castExtend}",      "AstRedXor{$lhsp->castExtend()->lhsp()}")
	if (m_doV && VN_IS(nodep->lhsp(),Extend)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstRedXor $lhsp.castExtend , AstRedXor $lhsp->castExtend()->lhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Extend)->lhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstRedXor(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_RedXor_3(AstRedXor* nodep) {
	// TREEOP ("AstRedXor{$lhsp.castXor, VN_IS(VN_AS($lhsp,,Xor)->lhsp(),,Const)}", "AstXor{AstRedXor{$lhsp->castXor()->lhsp()}, AstRedXor{$lhsp->castXor()->rhsp()}}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Xor) && VN_IS(VN_AS(nodep->lhsp(),Xor)->lhsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstRedXor $lhsp.castXor, VN_IS(VN_AS($lhsp,,Xor)->lhsp(),,Const) , AstXor AstRedXor $lhsp->castXor()->lhsp() , AstRedXor $lhsp->castXor()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Xor)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = VN_CAST(nodep->lhsp(),Xor)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstXor(nodep->fileline(), new AstRedXor(nodep->fileline(), arg1p), new AstRedXor(nodep->fileline(), arg2p));
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Replicate_0(AstReplicate* nodep) {
	// TREEOPV("AstReplicate{$srcp, $countp.isOne, $srcp->width()==nodep->width()}", "replaceWLhs(nodep)")
	if (m_doV && nodep->countp()->isOne() && nodep->srcp()->width()==nodep->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstReplicate $srcp, $countp.isOne, $srcp->width()==nodep->width() , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Replicate_1(AstReplicate* nodep) {
	// TREEOPV("AstReplicate{$srcp.castReplicate, operandRepRep(nodep)}", "DONE")
	if (m_doV && VN_IS(nodep->srcp(),Replicate) && operandRepRep(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstReplicate $srcp.castReplicate, operandRepRep(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ReplicateN_0(AstReplicateN* nodep) {
	// TREEOPV("AstReplicateN{$lhsp, $rhsp.isOne, $lhsp->width()==nodep->width()}", "replaceWLhs(nodep)")
	if (m_doV && nodep->rhsp()->isOne() && nodep->lhsp()->width()==nodep->width()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstReplicateN $lhsp, $rhsp.isOne, $lhsp->width()==nodep->width() , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_0(AstSel* nodep) {
	// TREEOP1("AstSel{warnSelect(nodep)}",        "NEVER")
	if (m_doNConst && warnSelect(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP1( AstSel warnSelect(nodep) , NEVER )");
	    nodep->v3fatalSrc("Executing transform that was NEVERed");
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_1(AstSel* nodep) {
	// TREEOPV("AstSel{matchSelRand(nodep)}",      "DONE")
	if (m_doV && matchSelRand(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel matchSelRand(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_2(AstSel* nodep) {
	// TREEOPV("AstSel{operandSelExtend(nodep)}",  "DONE")
	if (m_doV && operandSelExtend(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel operandSelExtend(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_3(AstSel* nodep) {
	// TREEOPV("AstSel{operandSelFull(nodep)}",    "replaceWChild(nodep, nodep->fromp())")
	if (m_doV && operandSelFull(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel operandSelFull(nodep) , replaceWChild(nodep, nodep->fromp()) )");
	    replaceWChild(nodep, nodep->fromp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_4(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castSel}",           "replaceSelSel(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),Sel)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castSel , replaceSelSel(nodep) )");
	    replaceSelSel(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_5(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castAdd, operandSelBiLower(nodep)}", "DONE")
	if (m_doV && VN_IS(nodep->fromp(),Add) && operandSelBiLower(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castAdd, operandSelBiLower(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_6(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castAnd, operandSelBiLower(nodep)}", "DONE")
	if (m_doV && VN_IS(nodep->fromp(),And) && operandSelBiLower(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castAnd, operandSelBiLower(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_7(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castOr,  operandSelBiLower(nodep)}", "DONE")
	if (m_doV && VN_IS(nodep->fromp(),Or) && operandSelBiLower(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castOr, operandSelBiLower(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_8(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castSub, operandSelBiLower(nodep)}", "DONE")
	if (m_doV && VN_IS(nodep->fromp(),Sub) && operandSelBiLower(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castSub, operandSelBiLower(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_9(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castXor, operandSelBiLower(nodep)}", "DONE")
	if (m_doV && VN_IS(nodep->fromp(),Xor) && operandSelBiLower(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castXor, operandSelBiLower(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_10(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castShiftR, operandSelShiftLower(nodep)}",   "DONE")
	if (m_doV && VN_IS(nodep->fromp(),ShiftR) && operandSelShiftLower(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castShiftR, operandSelShiftLower(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_11(AstSel* nodep) {
	// TREEOPA("AstSel{$fromp.castConst, $lsbp.castConst, }",   "replaceConst(nodep)")
	if (VN_IS(nodep->fromp(),Const) && VN_IS(nodep->lsbp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstSel $fromp.castConst, $lsbp.castConst, , replaceConst(nodep) )");
	    replaceConst(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_12(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castConcat, $lsbp.castConst, }",  "replaceSelConcat(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),Concat) && VN_IS(nodep->lsbp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castConcat, $lsbp.castConst, , replaceSelConcat(nodep) )");
	    replaceSelConcat(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_13(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castReplicate, $lsbp.castConst, operandSelReplicate(nodep) }",    "DONE")
	if (m_doV && VN_IS(nodep->fromp(),Replicate) && VN_IS(nodep->lsbp(),Const) && operandSelReplicate(nodep) ) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castReplicate, $lsbp.castConst, operandSelReplicate(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_14(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castBufIf1}",                "replaceSelIntoBiop(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),BufIf1)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castBufIf1 , replaceSelIntoBiop(nodep) )");
	    replaceSelIntoBiop(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_15(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castNot}",                   "replaceSelIntoUniop(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),Not)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castNot , replaceSelIntoUniop(nodep) )");
	    replaceSelIntoUniop(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_16(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castAnd,$lsbp.castConst}",   "replaceSelIntoBiop(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),And) && VN_IS(nodep->lsbp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castAnd,$lsbp.castConst , replaceSelIntoBiop(nodep) )");
	    replaceSelIntoBiop(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_17(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castOr,$lsbp.castConst}",    "replaceSelIntoBiop(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),Or) && VN_IS(nodep->lsbp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castOr,$lsbp.castConst , replaceSelIntoBiop(nodep) )");
	    replaceSelIntoBiop(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sel_18(AstSel* nodep) {
	// TREEOPV("AstSel{$fromp.castXor,$lsbp.castConst}",   "replaceSelIntoBiop(nodep)")
	if (m_doV && VN_IS(nodep->fromp(),Xor) && VN_IS(nodep->lsbp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPV( AstSel $fromp.castXor,$lsbp.castConst , replaceSelIntoBiop(nodep) )");
	    replaceSelIntoBiop(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftL_0(AstShiftL* nodep) {
	// TREEOP ("AstShiftL    {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftL $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftL_1(AstShiftL* nodep) {
	// TREEOP ("AstShiftL    {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftL $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftL_2(AstShiftL* nodep) {
	// TREEOP ("AstShiftL   {operandHugeShiftL(nodep)}",   "replaceZero(nodep)")
	if (m_doNConst && operandHugeShiftL(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftL operandHugeShiftL(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftL_3(AstShiftL* nodep) {
	// TREEOP ("AstShiftL{operandShiftOp(nodep)}",         "replaceShiftOp(nodep)")
	if (m_doNConst && operandShiftOp(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftL operandShiftOp(nodep) , replaceShiftOp(nodep) )");
	    replaceShiftOp(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftL_4(AstShiftL* nodep) {
	// TREEOP ("AstShiftL{operandShiftShift(nodep)}",      "replaceShiftShift(nodep)")
	if (m_doNConst && operandShiftShift(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftL operandShiftShift(nodep) , replaceShiftShift(nodep) )");
	    replaceShiftShift(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftLOvr_0(AstShiftLOvr* nodep) {
	// TREEOP ("AstShiftLOvr {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftLOvr $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftLOvr_1(AstShiftLOvr* nodep) {
	// TREEOP ("AstShiftLOvr {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftLOvr $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftLOvr_2(AstShiftLOvr* nodep) {
	// TREEOP ("AstShiftLOvr{operandHugeShiftL(nodep)}",   "replaceZero(nodep)")
	if (m_doNConst && operandHugeShiftL(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftLOvr operandHugeShiftL(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftR_0(AstShiftR* nodep) {
	// TREEOP ("AstShiftR    {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftR $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftR_1(AstShiftR* nodep) {
	// TREEOP ("AstShiftR    {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftR $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftR_2(AstShiftR* nodep) {
	// TREEOP ("AstShiftR   {operandHugeShiftR(nodep)}",   "replaceZero(nodep)")
	if (m_doNConst && operandHugeShiftR(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftR operandHugeShiftR(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftR_3(AstShiftR* nodep) {
	// TREEOP ("AstShiftR{operandShiftOp(nodep)}",         "replaceShiftOp(nodep)")
	if (m_doNConst && operandShiftOp(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftR operandShiftOp(nodep) , replaceShiftOp(nodep) )");
	    replaceShiftOp(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftR_4(AstShiftR* nodep) {
	// TREEOP ("AstShiftR{operandShiftShift(nodep)}",      "replaceShiftShift(nodep)")
	if (m_doNConst && operandShiftShift(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftR operandShiftShift(nodep) , replaceShiftShift(nodep) )");
	    replaceShiftShift(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftROvr_0(AstShiftROvr* nodep) {
	// TREEOP ("AstShiftROvr {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftROvr $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftROvr_1(AstShiftROvr* nodep) {
	// TREEOP ("AstShiftROvr {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftROvr $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftROvr_2(AstShiftROvr* nodep) {
	// TREEOP ("AstShiftROvr{operandHugeShiftR(nodep)}",   "replaceZero(nodep)")
	if (m_doNConst && operandHugeShiftR(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftROvr operandHugeShiftR(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftRS_0(AstShiftRS* nodep) {
	// TREEOP ("AstShiftRS   {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftRS $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftRS_1(AstShiftRS* nodep) {
	// TREEOP ("AstShiftRS   {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftRS $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftRSOvr_0(AstShiftRSOvr* nodep) {
	// TREEOP ("AstShiftRSOvr{$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftRSOvr $lhsp.isZero, $rhsp , replaceZeroChkPure(nodep,$rhsp) )");
	    replaceZeroChkPure(nodep,nodep->rhsp());
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ShiftRSOvr_1(AstShiftRSOvr* nodep) {
	// TREEOP ("AstShiftRSOvr{$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstShiftRSOvr $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sub_0(AstSub* nodep) {
	// TREEOP ("AstSub   {$lhsp.isZero, $rhsp}",   "AstNegate{$rhsp}")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstSub $lhsp.isZero, $rhsp , AstNegate $rhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNegate(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sub_1(AstSub* nodep) {
	// TREEOP ("AstSub   {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstSub $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sub_2(AstSub* nodep) {
	// TREEOP ("AstSub   {$lhsp.castAdd, operandSubAdd(nodep)}", "AstAdd{AstSub{$lhsp->castAdd()->lhsp(),$rhsp}, $lhsp->castAdd()->rhsp()}")
	if (m_doNConst && VN_IS(nodep->lhsp(),Add) && operandSubAdd(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstSub $lhsp.castAdd, operandSubAdd(nodep) , AstAdd AstSub $lhsp->castAdd()->lhsp(),$rhsp , $lhsp->castAdd()->rhsp() )");
	    AstNodeExpr* arg1p = VN_CAST(nodep->lhsp(),Add)->lhsp()->unlinkFrBack();
AstNodeExpr* arg2p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* arg3p = VN_CAST(nodep->lhsp(),Add)->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstAdd(nodep->fileline(), new AstSub(nodep->fileline(), arg1p, arg2p), arg3p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Sub_3(AstSub* nodep) {
	// TREEOP ("AstSub    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstSub operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_SubstrN_0(AstSubstrN* nodep) {
	// TREEOPA("AstSubstrN{$lhsp.castConst, $rhsp.castConst, $thsp.castConst}",  "replaceConst(nodep)")
	if (VN_IS(nodep->lhsp(),Const) && VN_IS(nodep->rhsp(),Const) && VN_IS(nodep->thsp(),Const)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPA( AstSubstrN $lhsp.castConst, $rhsp.castConst, $thsp.castConst , replaceConst(nodep) )");
	    replaceConst(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_ToStringN_0(AstToStringN* nodep) {
	// TREEOP ("AstToStringN{matchToStringNConst(nodep)}", "DONE")
	if (m_doNConst && matchToStringNConst(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstToStringN matchToStringNConst(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_WordSel_0(AstWordSel* nodep) {
	// TREEOP ("AstWordSel{operandWordOOB(nodep)}",        "replaceZero(nodep)")
	if (m_doNConst && operandWordOOB(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstWordSel operandWordOOB(nodep) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Xor_0(AstXor* nodep) {
	// TREEOP ("AstXor   {$lhsp.isZero, $rhsp}",   "replaceWRhs(nodep)")
	if (m_doNConst && nodep->lhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstXor $lhsp.isZero, $rhsp , replaceWRhs(nodep) )");
	    replaceWRhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Xor_1(AstXor* nodep) {
	// TREEOP ("AstXor   {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)")
	if (m_doNConst && nodep->rhsp()->isZero()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstXor $lhsp, $rhsp.isZero , replaceWLhs(nodep) )");
	    replaceWLhs(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Xor_2(AstXor* nodep) {
	// TREEOP ("AstXor   {$lhsp.isAllOnes, $rhsp}",        "AstNot{$rhsp}")
	if (m_doNConst && nodep->lhsp()->isAllOnes()) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstXor $lhsp.isAllOnes, $rhsp , AstNot $rhsp )");
	    AstNodeExpr* arg1p = nodep->rhsp()->unlinkFrBack();
AstNodeExpr* newp = new AstNot(nodep->fileline(), arg1p);
nodep->replaceWith(newp);VL_DO_DANGLING(nodep->deleteTree(), nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Xor_3(AstXor* nodep) {
	// TREEOP ("AstXor    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)")
	if (m_doNConst && operandsSame(nodep->lhsp(),nodep->rhsp())) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstXor operandsSame($lhsp,,$rhsp) , replaceZero(nodep) )");
	    replaceZero(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Xor_4(AstXor* nodep) {
	// TREEOP ("AstXor {operandShiftSame(nodep)}",         "replaceShiftSame(nodep)")
	if (m_doNConst && operandShiftSame(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOP ( AstXor operandShiftSame(nodep) , replaceShiftSame(nodep) )");
	    replaceShiftSame(nodep);
	    return true;
	}
	return false;
    }
    // Generated by astgen
    bool match_Xor_5(AstXor* nodep) {
	// TREEOPC("AstXor {matchBitOpTree(nodep)}", "DONE")
	if (m_doCpp && matchBitOpTree(nodep)) {
	    UINFO(7, cvtToHex(nodep) << " TREEOPC( AstXor matchBitOpTree(nodep) , DONE )");
	    
	    return true;
	}
	return false;
    }
    // TREEOP visitors, call each base type's match
    // Bottom class up, as more simple transforms are generally better
    // Generated by astgen
    void visit(AstAcosD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAcoshD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAdd* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
        if (match_Add_0(nodep)) return;
        if (match_Add_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAddD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAnd* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
        if (match_And_0(nodep)) return;
        if (match_And_1(nodep)) return;
        if (match_And_2(nodep)) return;
        if (match_And_3(nodep)) return;
        if (match_And_4(nodep)) return;
        if (match_And_5(nodep)) return;
        if (match_And_6(nodep)) return;
        if (match_And_7(nodep)) return;
        if (match_And_8(nodep)) return;
        if (match_And_9(nodep)) return;
        if (match_And_10(nodep)) return;
        if (match_And_11(nodep)) return;
        if (match_And_12(nodep)) return;
    }
    // Generated by astgen
    virtual void visitGen(AstArraySel* nodep) {
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAsinD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAsinhD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAssocSel* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAtan2D* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAtanD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAtanhD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstAtoN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstBitsToRealD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstBufIf1* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    virtual void visitGen(AstCAwait* nodep) {
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCCast* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCLog2* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCastDynamic* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCastWrap* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCeilD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCompareNN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstConcat* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_Concat_0(nodep)) return;
        if (match_Concat_1(nodep)) return;
        if (match_Concat_2(nodep)) return;
        if (match_Concat_3(nodep)) return;
        if (match_Concat_4(nodep)) return;
        if (match_Concat_5(nodep)) return;
    }
    // Generated by astgen
    void visit(AstConcatN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen with short-circuiting
    void visit(AstCond* nodep) override {
      iterateAndNextNull(nodep->condp());
        if (match_Cond_6(nodep)) return;
        if (match_Cond_7(nodep)) return;
      iterateAndNextNull(nodep->thenp());
      iterateAndNextNull(nodep->elsep());
        if (match_Cond_0(nodep)) return;
        if (match_Cond_1(nodep)) return;
        if (match_Cond_2(nodep)) return;
        if (match_Cond_3(nodep)) return;
        if (match_Cond_4(nodep)) return;
        if (match_Cond_5(nodep)) return;
        if (match_Cond_8(nodep)) return;
        if (match_Cond_9(nodep)) return;
        if (match_Cond_10(nodep)) return;
        if (match_Cond_11(nodep)) return;
        if (match_Cond_12(nodep)) return;
        if (match_Cond_13(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCosD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCoshD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCountBits* nodep) override {
        iterateChildren(nodep);
        if (match_NodeQuadop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCountOnes* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstCvtPackString* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_CvtPackString_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDistChiSquare* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDistExponential* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDistPoisson* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDistT* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDiv* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_Div_0(nodep)) return;
        if (match_Div_1(nodep)) return;
        if (match_Div_2(nodep)) return;
        if (match_Div_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDivD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstDivS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_DivS_0(nodep)) return;
        if (match_DivS_1(nodep)) return;
        if (match_DivS_2(nodep)) return;
    }
    // Generated by astgen
    void visit(AstEq* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_Eq_0(nodep)) return;
        if (match_Eq_1(nodep)) return;
        if (match_Eq_2(nodep)) return;
        if (match_Eq_3(nodep)) return;
        if (match_Eq_4(nodep)) return;
        if (match_Eq_5(nodep)) return;
        if (match_Eq_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstEqCase* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_EqCase_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstEqD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_EqD_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstEqN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_EqN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstEqT* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstEqWild* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_EqWild_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstExpD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstExtend* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_Extend_0(nodep)) return;
        if (match_Extend_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstExtendS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_ExtendS_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstFEof* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstFGetC* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstFGetS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstFUngetC* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstFloorD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGetcN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGetcRefN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGt* nodep) override {
        iterateChildren(nodep);
        if (match_Gt_2(nodep)) return;
        if (match_Gt_1(nodep)) return;
        if (match_NodeBiop_0(nodep)) return;
        if (match_Gt_0(nodep)) return;
        if (match_Gt_3(nodep)) return;
        if (match_Gt_4(nodep)) return;
        if (match_Gt_5(nodep)) return;
        if (match_Gt_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGtD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_GtD_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGtN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_GtN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGtS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_GtS_0(nodep)) return;
        if (match_GtS_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGte* nodep) override {
        iterateChildren(nodep);
        if (match_Gte_2(nodep)) return;
        if (match_Gte_1(nodep)) return;
        if (match_NodeBiop_0(nodep)) return;
        if (match_Gte_0(nodep)) return;
        if (match_Gte_3(nodep)) return;
        if (match_Gte_4(nodep)) return;
        if (match_Gte_5(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGteD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_GteD_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGteN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_GteN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstGteS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_GteS_0(nodep)) return;
        if (match_GteS_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstHypotD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstISToRD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstIToRD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstIsUnbounded* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_IsUnbounded_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstIsUnknown* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLenN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLog10D* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen with short-circuiting
    void visit(AstLogAnd* nodep) override {
      iterateAndNextNull(nodep->lhsp());
        if (match_LogAnd_0(nodep)) return;
      iterateAndNextNull(nodep->rhsp());
        if (match_NodeBiop_0(nodep)) return;
        if (match_LogAnd_1(nodep)) return;
        if (match_LogAnd_2(nodep)) return;
        if (match_LogAnd_3(nodep)) return;
        if (match_LogAnd_4(nodep)) return;
        if (match_LogAnd_5(nodep)) return;
        if (match_LogAnd_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLogD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLogEq* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_LogEq_0(nodep)) return;
    }
    // Generated by astgen with short-circuiting
    void visit(AstLogIf* nodep) override {
      iterateAndNextNull(nodep->lhsp());
        if (match_LogIf_0(nodep)) return;
      iterateAndNextNull(nodep->rhsp());
        if (match_NodeBiop_0(nodep)) return;
        if (match_LogIf_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLogNot* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_LogNot_0(nodep)) return;
        if (match_LogNot_1(nodep)) return;
        if (match_LogNot_2(nodep)) return;
        if (match_LogNot_3(nodep)) return;
        if (match_LogNot_4(nodep)) return;
        if (match_LogNot_5(nodep)) return;
        if (match_LogNot_6(nodep)) return;
        if (match_LogNot_7(nodep)) return;
        if (match_LogNot_8(nodep)) return;
        if (match_LogNot_9(nodep)) return;
        if (match_LogNot_10(nodep)) return;
        if (match_LogNot_11(nodep)) return;
        if (match_LogNot_12(nodep)) return;
        if (match_LogNot_13(nodep)) return;
        if (match_LogNot_14(nodep)) return;
        if (match_LogNot_15(nodep)) return;
    }
    // Generated by astgen with short-circuiting
    void visit(AstLogOr* nodep) override {
      iterateAndNextNull(nodep->lhsp());
        if (match_LogOr_0(nodep)) return;
      iterateAndNextNull(nodep->rhsp());
        if (match_NodeBiop_0(nodep)) return;
        if (match_LogOr_1(nodep)) return;
        if (match_LogOr_2(nodep)) return;
        if (match_LogOr_3(nodep)) return;
        if (match_LogOr_4(nodep)) return;
        if (match_LogOr_5(nodep)) return;
        if (match_LogOr_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLt* nodep) override {
        iterateChildren(nodep);
        if (match_Lt_2(nodep)) return;
        if (match_Lt_1(nodep)) return;
        if (match_NodeBiop_0(nodep)) return;
        if (match_Lt_0(nodep)) return;
        if (match_Lt_3(nodep)) return;
        if (match_Lt_4(nodep)) return;
        if (match_Lt_5(nodep)) return;
        if (match_Lt_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLtD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_LtD_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLtN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_LtN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLtS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_LtS_0(nodep)) return;
        if (match_LtS_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLte* nodep) override {
        iterateChildren(nodep);
        if (match_Lte_2(nodep)) return;
        if (match_Lte_1(nodep)) return;
        if (match_NodeBiop_0(nodep)) return;
        if (match_Lte_0(nodep)) return;
        if (match_Lte_3(nodep)) return;
        if (match_Lte_4(nodep)) return;
        if (match_Lte_5(nodep)) return;
        if (match_Lte_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLteD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_LteD_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLteN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_LteN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstLteS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_LteS_0(nodep)) return;
        if (match_LteS_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstModDiv* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ModDiv_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstModDivS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstMul* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
        if (match_Mul_0(nodep)) return;
        if (match_Mul_1(nodep)) return;
        if (match_Mul_2(nodep)) return;
        if (match_Mul_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstMulD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstMulS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
        if (match_MulS_0(nodep)) return;
        if (match_MulS_1(nodep)) return;
        if (match_MulS_2(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNToI* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNegate* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNegateD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNeq* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_Neq_0(nodep)) return;
        if (match_Neq_1(nodep)) return;
        if (match_Neq_2(nodep)) return;
        if (match_Neq_3(nodep)) return;
        if (match_Neq_4(nodep)) return;
        if (match_Neq_5(nodep)) return;
        if (match_Neq_6(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNeqCase* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NeqCase_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNeqD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NeqD_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNeqN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NeqN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNeqT* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNeqWild* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NeqWild_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeBiCom* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeBiComAsv* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeBiop* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeDistBiop* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeQuadop* nodep) override {
        iterateChildren(nodep);
        if (match_NodeQuadop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeSel* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeStream* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeSystemBiopD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeSystemUniopD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNodeUniop* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNot* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_Not_0(nodep)) return;
        if (match_Not_1(nodep)) return;
        if (match_Not_2(nodep)) return;
        if (match_Not_3(nodep)) return;
        if (match_Not_4(nodep)) return;
        if (match_Not_5(nodep)) return;
        if (match_Not_6(nodep)) return;
        if (match_Not_7(nodep)) return;
        if (match_Not_8(nodep)) return;
        if (match_Not_9(nodep)) return;
        if (match_Not_10(nodep)) return;
        if (match_Not_11(nodep)) return;
        if (match_Not_12(nodep)) return;
        if (match_Not_13(nodep)) return;
        if (match_Not_14(nodep)) return;
    }
    // Generated by astgen
    void visit(AstNullCheck* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstOneHot* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_OneHot_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstOneHot0* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_OneHot0_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstOr* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
        if (match_Or_0(nodep)) return;
        if (match_Or_1(nodep)) return;
        if (match_Or_2(nodep)) return;
        if (match_Or_3(nodep)) return;
        if (match_Or_4(nodep)) return;
        if (match_Or_5(nodep)) return;
        if (match_Or_6(nodep)) return;
        if (match_Or_7(nodep)) return;
        if (match_Or_8(nodep)) return;
    }
    // Generated by astgen
    void visit(AstPow* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_Pow_0(nodep)) return;
        if (match_Pow_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstPowD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstPowSS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_PowSS_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstPowSU* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_PowSU_0(nodep)) return;
        if (match_PowSU_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstPowUS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_PowUS_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstPutcN* nodep) override {
        iterateChildren(nodep);
        if (match_PutcN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstRToIRoundS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstRToIS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstRealToBits* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstRedAnd* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_RedAnd_0(nodep)) return;
        if (match_RedAnd_1(nodep)) return;
        if (match_RedAnd_2(nodep)) return;
    }
    // Generated by astgen
    void visit(AstRedOr* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_RedOr_0(nodep)) return;
        if (match_RedOr_1(nodep)) return;
        if (match_RedOr_2(nodep)) return;
    }
    // Generated by astgen
    void visit(AstRedXor* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
        if (match_RedXor_0(nodep)) return;
        if (match_RedXor_1(nodep)) return;
        if (match_RedXor_2(nodep)) return;
        if (match_RedXor_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstReplicate* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_Replicate_0(nodep)) return;
        if (match_Replicate_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstReplicateN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ReplicateN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstResizeLValue* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSel* nodep) override {
        iterateChildren(nodep);
        if (match_Sel_0(nodep)) return;
        if (match_NodeBiop_0(nodep)) return;
        if (match_Sel_1(nodep)) return;
        if (match_Sel_2(nodep)) return;
        if (match_Sel_3(nodep)) return;
        if (match_Sel_4(nodep)) return;
        if (match_Sel_5(nodep)) return;
        if (match_Sel_6(nodep)) return;
        if (match_Sel_7(nodep)) return;
        if (match_Sel_8(nodep)) return;
        if (match_Sel_9(nodep)) return;
        if (match_Sel_10(nodep)) return;
        if (match_Sel_11(nodep)) return;
        if (match_Sel_12(nodep)) return;
        if (match_Sel_13(nodep)) return;
        if (match_Sel_14(nodep)) return;
        if (match_Sel_15(nodep)) return;
        if (match_Sel_16(nodep)) return;
        if (match_Sel_17(nodep)) return;
        if (match_Sel_18(nodep)) return;
    }
    // Generated by astgen
    void visit(AstShiftL* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ShiftL_0(nodep)) return;
        if (match_ShiftL_1(nodep)) return;
        if (match_ShiftL_2(nodep)) return;
        if (match_ShiftL_3(nodep)) return;
        if (match_ShiftL_4(nodep)) return;
    }
    // Generated by astgen
    void visit(AstShiftLOvr* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ShiftLOvr_0(nodep)) return;
        if (match_ShiftLOvr_1(nodep)) return;
        if (match_ShiftLOvr_2(nodep)) return;
    }
    // Generated by astgen
    void visit(AstShiftR* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ShiftR_0(nodep)) return;
        if (match_ShiftR_1(nodep)) return;
        if (match_ShiftR_2(nodep)) return;
        if (match_ShiftR_3(nodep)) return;
        if (match_ShiftR_4(nodep)) return;
    }
    // Generated by astgen
    void visit(AstShiftROvr* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ShiftROvr_0(nodep)) return;
        if (match_ShiftROvr_1(nodep)) return;
        if (match_ShiftROvr_2(nodep)) return;
    }
    // Generated by astgen
    void visit(AstShiftRS* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ShiftRS_0(nodep)) return;
        if (match_ShiftRS_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstShiftRSOvr* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_ShiftRSOvr_0(nodep)) return;
        if (match_ShiftRSOvr_1(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSigned* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSinD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSinhD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSqrtD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstStreamL* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstStreamR* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSub* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_Sub_0(nodep)) return;
        if (match_Sub_1(nodep)) return;
        if (match_Sub_2(nodep)) return;
        if (match_Sub_3(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSubD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstSubstrN* nodep) override {
        iterateChildren(nodep);
        if (match_SubstrN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstTanD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstTanhD* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstTimeImport* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstToLowerN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstToStringN* nodep) override {
        iterateChildren(nodep);
        // DISABLE_BASE
        if (match_ToStringN_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstToUpperN* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstURandomRange* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstUnsigned* nodep) override {
        iterateChildren(nodep);
        if (match_NodeUniop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstWildcardSel* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstWordSel* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_WordSel_0(nodep)) return;
    }
    // Generated by astgen
    void visit(AstXor* nodep) override {
        iterateChildren(nodep);
        if (match_NodeBiop_0(nodep)) return;
        if (match_NodeBiCom_0(nodep)) return;
        if (match_NodeBiComAsv_0(nodep)) return;
        if (match_NodeBiComAsv_1(nodep)) return;
        if (match_NodeBiComAsv_2(nodep)) return;
        if (match_NodeBiComAsv_3(nodep)) return;
        if (match_Xor_0(nodep)) return;
        if (match_Xor_1(nodep)) return;
        if (match_Xor_2(nodep)) return;
        if (match_Xor_3(nodep)) return;
        if (match_Xor_4(nodep)) return;
        if (match_Xor_5(nodep)) return;
    }
#line 3773 "../V3Const.cpp"
    // Generic constants on both side.  Do this first to avoid other replacements
//    TREEOPA("AstNodeBiop {$lhsp.castConst, $rhsp.castConst, nodep->isPredictOptimizable()}",  "replaceConst(nodep)");
#line 7617 "V3Const__gen.cpp"
#line 3775 "../V3Const.cpp"
//    TREEOPA("AstNodeUniop{$lhsp.castConst, !nodep->isOpaque(), nodep->isPredictOptimizable()}",  "replaceConst(nodep)");
#line 7620 "V3Const__gen.cpp"
#line 3776 "../V3Const.cpp"
//    TREEOPA("AstNodeQuadop{$lhsp.castConst, $rhsp.castConst, $thsp.castConst, $fhsp.castConst}",  "replaceConst(nodep)");
#line 7623 "V3Const__gen.cpp"
#line 3777 "../V3Const.cpp"
    // Zero on one side or the other
//    TREEOP ("AstAdd   {$lhsp.isZero, $rhsp}",   "replaceWRhs(nodep)");
#line 7627 "V3Const__gen.cpp"
#line 3779 "../V3Const.cpp"
//    TREEOP ("AstAnd   {$lhsp.isZero, $rhsp, $rhsp.isPure}",   "replaceZero(nodep)");  // Can't use replaceZeroChkPure as we make this pattern in ChkPure
#line 7630 "V3Const__gen.cpp"
#line 3780 "../V3Const.cpp"
    // This visit function here must allow for short-circuiting.
//    TREEOPS("AstLogAnd   {$lhsp.isZero}",       "replaceZero(nodep)");
#line 7634 "V3Const__gen.cpp"
#line 3782 "../V3Const.cpp"
//    TREEOP ("AstLogAnd{$lhsp.isZero, $rhsp}",   "replaceZero(nodep)");
#line 7637 "V3Const__gen.cpp"
#line 3783 "../V3Const.cpp"
    // This visit function here must allow for short-circuiting.
//    TREEOPS("AstLogOr   {$lhsp.isOne}",         "replaceNum(nodep, 1)");
#line 7641 "V3Const__gen.cpp"
#line 3785 "../V3Const.cpp"
//    TREEOP ("AstLogOr {$lhsp.isZero, $rhsp}",   "replaceWRhsBool(nodep)");
#line 7644 "V3Const__gen.cpp"
#line 3786 "../V3Const.cpp"
//    TREEOP ("AstDiv   {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)");
#line 7647 "V3Const__gen.cpp"
#line 3787 "../V3Const.cpp"
//    TREEOP ("AstDivS  {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)");
#line 7650 "V3Const__gen.cpp"
#line 3788 "../V3Const.cpp"
//    TREEOP ("AstMul   {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)");
#line 7653 "V3Const__gen.cpp"
#line 3789 "../V3Const.cpp"
//    TREEOP ("AstMulS  {$lhsp.isZero, $rhsp}",   "replaceZeroChkPure(nodep,$rhsp)");
#line 7656 "V3Const__gen.cpp"
#line 3790 "../V3Const.cpp"
//    TREEOP ("AstPow   {$rhsp.isZero}",          "replaceNum(nodep, 1)");  // Overrides lhs zero rule
#line 7659 "V3Const__gen.cpp"
#line 3791 "../V3Const.cpp"
//    TREEOP ("AstPowSS {$rhsp.isZero}",          "replaceNum(nodep, 1)");  // Overrides lhs zero rule
#line 7662 "V3Const__gen.cpp"
#line 3792 "../V3Const.cpp"
//    TREEOP ("AstPowSU {$rhsp.isZero}",          "replaceNum(nodep, 1)");  // Overrides lhs zero rule
#line 7665 "V3Const__gen.cpp"
#line 3793 "../V3Const.cpp"
//    TREEOP ("AstPowUS {$rhsp.isZero}",          "replaceNum(nodep, 1)");  // Overrides lhs zero rule
#line 7668 "V3Const__gen.cpp"
#line 3794 "../V3Const.cpp"
//    TREEOP ("AstOr    {$lhsp.isZero, $rhsp}",   "replaceWRhs(nodep)");
#line 7671 "V3Const__gen.cpp"
#line 3795 "../V3Const.cpp"
//    TREEOP ("AstShiftL    {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)");
#line 7674 "V3Const__gen.cpp"
#line 3796 "../V3Const.cpp"
//    TREEOP ("AstShiftLOvr {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)");
#line 7677 "V3Const__gen.cpp"
#line 3797 "../V3Const.cpp"
//    TREEOP ("AstShiftR    {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)");
#line 7680 "V3Const__gen.cpp"
#line 3798 "../V3Const.cpp"
//    TREEOP ("AstShiftROvr {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)");
#line 7683 "V3Const__gen.cpp"
#line 3799 "../V3Const.cpp"
//    TREEOP ("AstShiftRS   {$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)");
#line 7686 "V3Const__gen.cpp"
#line 3800 "../V3Const.cpp"
//    TREEOP ("AstShiftRSOvr{$lhsp.isZero, $rhsp}",  "replaceZeroChkPure(nodep,$rhsp)");
#line 7689 "V3Const__gen.cpp"
#line 3801 "../V3Const.cpp"
//    TREEOP ("AstXor   {$lhsp.isZero, $rhsp}",   "replaceWRhs(nodep)");
#line 7692 "V3Const__gen.cpp"
#line 3802 "../V3Const.cpp"
//    TREEOP ("AstSub   {$lhsp.isZero, $rhsp}",   "AstNegate{$rhsp}");
#line 7695 "V3Const__gen.cpp"
#line 3803 "../V3Const.cpp"
//    TREEOP ("AstAdd   {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)");
#line 7698 "V3Const__gen.cpp"
#line 3804 "../V3Const.cpp"
//    TREEOP ("AstAnd   {$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)");
#line 7701 "V3Const__gen.cpp"
#line 3805 "../V3Const.cpp"
//    TREEOP ("AstLogAnd{$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)");
#line 7704 "V3Const__gen.cpp"
#line 3806 "../V3Const.cpp"
//    TREEOP ("AstLogOr {$lhsp, $rhsp.isZero}",   "replaceWLhsBool(nodep)");
#line 7707 "V3Const__gen.cpp"
#line 3807 "../V3Const.cpp"
//    TREEOP ("AstMul   {$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)");
#line 7710 "V3Const__gen.cpp"
#line 3808 "../V3Const.cpp"
//    TREEOP ("AstMulS  {$lhsp, $rhsp.isZero}",   "replaceZeroChkPure(nodep,$lhsp)");
#line 7713 "V3Const__gen.cpp"
#line 3809 "../V3Const.cpp"
//    TREEOP ("AstOr    {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)");
#line 7716 "V3Const__gen.cpp"
#line 3810 "../V3Const.cpp"
//    TREEOP ("AstShiftL    {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)");
#line 7719 "V3Const__gen.cpp"
#line 3811 "../V3Const.cpp"
//    TREEOP ("AstShiftLOvr {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)");
#line 7722 "V3Const__gen.cpp"
#line 3812 "../V3Const.cpp"
//    TREEOP ("AstShiftR    {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)");
#line 7725 "V3Const__gen.cpp"
#line 3813 "../V3Const.cpp"
//    TREEOP ("AstShiftROvr {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)");
#line 7728 "V3Const__gen.cpp"
#line 3814 "../V3Const.cpp"
//    TREEOP ("AstShiftRS   {$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)");
#line 7731 "V3Const__gen.cpp"
#line 3815 "../V3Const.cpp"
//    TREEOP ("AstShiftRSOvr{$lhsp, $rhsp.isZero}",  "replaceWLhs(nodep)");
#line 7734 "V3Const__gen.cpp"
#line 3816 "../V3Const.cpp"
//    TREEOP ("AstSub   {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)");
#line 7737 "V3Const__gen.cpp"
#line 3817 "../V3Const.cpp"
//    TREEOP ("AstXor   {$lhsp, $rhsp.isZero}",   "replaceWLhs(nodep)");
#line 7740 "V3Const__gen.cpp"
#line 3818 "../V3Const.cpp"
    // Non-zero on one side or the other
//    TREEOP ("AstAnd   {$lhsp.isAllOnes, $rhsp}",        "replaceWRhs(nodep)");
#line 7744 "V3Const__gen.cpp"
#line 3820 "../V3Const.cpp"
//    TREEOP ("AstLogAnd{$lhsp.isNeqZero, $rhsp}",        "replaceWRhsBool(nodep)");
#line 7747 "V3Const__gen.cpp"
#line 3821 "../V3Const.cpp"
//    TREEOP ("AstOr    {$lhsp.isAllOnes, $rhsp, $rhsp.isPure}",        "replaceWLhs(nodep)");  // ->allOnes
#line 7750 "V3Const__gen.cpp"
#line 3822 "../V3Const.cpp"
//    TREEOP ("AstLogOr {$lhsp.isNeqZero, $rhsp}",        "replaceNum(nodep,1)");
#line 7753 "V3Const__gen.cpp"
#line 3823 "../V3Const.cpp"
//    TREEOP ("AstAnd   {$lhsp, $rhsp.isAllOnes}",        "replaceWLhs(nodep)");
#line 7756 "V3Const__gen.cpp"
#line 3824 "../V3Const.cpp"
//    TREEOP ("AstLogAnd{$lhsp, $rhsp.isNeqZero}",        "replaceWLhsBool(nodep)");
#line 7759 "V3Const__gen.cpp"
#line 3825 "../V3Const.cpp"
//    TREEOP ("AstOr    {$lhsp, $rhsp.isAllOnes, $lhsp.isPure}",        "replaceWRhs(nodep)");  // ->allOnes
#line 7762 "V3Const__gen.cpp"
#line 3826 "../V3Const.cpp"
//    TREEOP ("AstLogOr {$lhsp, $rhsp.isNeqZero, $lhsp.isPure, nodep->isPure()}",        "replaceNum(nodep,1)");
#line 7765 "V3Const__gen.cpp"
#line 3827 "../V3Const.cpp"
//    TREEOP ("AstXor   {$lhsp.isAllOnes, $rhsp}",        "AstNot{$rhsp}");
#line 7768 "V3Const__gen.cpp"
#line 3828 "../V3Const.cpp"
//    TREEOP ("AstMul   {$lhsp.isOne, $rhsp}",    "replaceWRhs(nodep)");
#line 7771 "V3Const__gen.cpp"
#line 3829 "../V3Const.cpp"
//    TREEOP ("AstMulS  {$lhsp.isOne, $rhsp}",    "replaceWRhs(nodep)");
#line 7774 "V3Const__gen.cpp"
#line 3830 "../V3Const.cpp"
//    TREEOP ("AstDiv   {$lhsp, $rhsp.isOne}",    "replaceWLhs(nodep)");
#line 7777 "V3Const__gen.cpp"
#line 3831 "../V3Const.cpp"
//    TREEOP ("AstDivS  {$lhsp, $rhsp.isOne}",    "replaceWLhs(nodep)");
#line 7780 "V3Const__gen.cpp"
#line 3832 "../V3Const.cpp"
//    TREEOP ("AstMul   {operandIsPowTwo($lhsp), operandsSameWidth($lhsp,,$rhsp)}", "replaceMulShift(nodep)");  // a*2^n -> a<<n
#line 7783 "V3Const__gen.cpp"
#line 3833 "../V3Const.cpp"
//    TREEOP ("AstDiv   {$lhsp, operandIsPowTwo($rhsp)}", "replaceDivShift(nodep)");  // a/2^n -> a>>n
#line 7786 "V3Const__gen.cpp"
#line 3834 "../V3Const.cpp"
//    TREEOP ("AstModDiv{$lhsp, operandIsPowTwo($rhsp)}", "replaceModAnd(nodep)");  // a % 2^n -> a&(2^n-1)
#line 7789 "V3Const__gen.cpp"
#line 3835 "../V3Const.cpp"
//    TREEOP ("AstPow   {operandIsTwo($lhsp), !$rhsp.isZero}",    "replacePowShift(nodep)");  // 2**a == 1<<a
#line 7792 "V3Const__gen.cpp"
#line 3836 "../V3Const.cpp"
//    TREEOP ("AstPowSU {operandIsTwo($lhsp), !$rhsp.isZero}",    "replacePowShift(nodep)");  // 2**a == 1<<a
#line 7795 "V3Const__gen.cpp"
#line 3837 "../V3Const.cpp"
//    TREEOP ("AstSub   {$lhsp.castAdd, operandSubAdd(nodep)}", "AstAdd{AstSub{$lhsp->castAdd()->lhsp(),$rhsp}, $lhsp->castAdd()->rhsp()}");  // ((a+x)-y) -> (a+(x-y))
#line 7798 "V3Const__gen.cpp"
#line 3838 "../V3Const.cpp"
//    TREEOPC("AstAnd   {$lhsp.isOne, matchRedundantClean(nodep)}", "DONE")  // 1 & (a == b) -> (IData)(a == b)
#line 7801 "V3Const__gen.cpp"
#line 3839 "../V3Const.cpp"
    // Trinary ops
    // Note V3Case::Sel requires Cond to always be conditionally executed in C to prevent core dump!
//    TREEOP ("AstCond{$condp.isZero,       $thenp, $elsep}", "replaceWChild(nodep,$elsep)");
#line 7806 "V3Const__gen.cpp"
#line 3842 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp.isNeqZero,    $thenp, $elsep}", "replaceWChild(nodep,$thenp)");
#line 7809 "V3Const__gen.cpp"
#line 3843 "../V3Const.cpp"
//    TREEOPA("AstCond{$condp.isZero,       $thenp.castConst, $elsep.castConst}", "replaceWChild(nodep,$elsep)");
#line 7812 "V3Const__gen.cpp"
#line 3844 "../V3Const.cpp"
//    TREEOPA("AstCond{$condp.isNeqZero,    $thenp.castConst, $elsep.castConst}", "replaceWChild(nodep,$thenp)");
#line 7815 "V3Const__gen.cpp"
#line 3845 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp, operandsSame($thenp,,$elsep)}","replaceWChild(nodep,$thenp)");
#line 7818 "V3Const__gen.cpp"
#line 3846 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp, matchCondCond(nodep)}", "DONE")  // Same condition then skip
#line 7821 "V3Const__gen.cpp"
#line 3847 "../V3Const.cpp"
    // This visit function here must allow for short-circuiting.
//    TREEOPS("AstCond{$condp.isZero}",           "replaceWIteratedThs(nodep)");
#line 7825 "V3Const__gen.cpp"
#line 3849 "../V3Const.cpp"
//    TREEOPS("AstCond{$condp.isNeqZero}",        "replaceWIteratedRhs(nodep)");
#line 7828 "V3Const__gen.cpp"
#line 3850 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp.castNot, $thenp, $elsep}", "AstCond{$condp->castNot()->lhsp(), $elsep, $thenp}");
#line 7831 "V3Const__gen.cpp"
#line 3851 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp.isAllOnes, $elsep}", "AstLogOr {$condp, $elsep}");  // a?1:b == a||b
#line 7834 "V3Const__gen.cpp"
#line 3852 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp, $elsep.isZero, !$elsep.isClassHandleValue}", "AstLogAnd{$condp, $thenp}");  // a?b:0 == a&&b
#line 7837 "V3Const__gen.cpp"
#line 3853 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp, $elsep.isAllOnes}", "AstLogOr {AstNot{$condp}, $thenp}");  // a?b:1 == ~a||b
#line 7840 "V3Const__gen.cpp"
#line 3854 "../V3Const.cpp"
//    TREEOP ("AstCond{$condp.width1, $thenp.width1, $thenp.isZero, !$thenp.isClassHandleValue, $elsep}", "AstLogAnd{AstNot{$condp}, $elsep}");  // a?0:b == ~a&&b
#line 7843 "V3Const__gen.cpp"
#line 3855 "../V3Const.cpp"
//    TREEOP ("AstCond{!$condp.width1, operandBoolShift(nodep->condp())}", "replaceBoolShift(nodep->condp())");
#line 7846 "V3Const__gen.cpp"
#line 3856 "../V3Const.cpp"
    // Prefer constants on left, since that often needs a shift, it lets
    // constant red remove the shift
//    TREEOP ("AstNodeBiCom{!$lhsp.castConst, $rhsp.castConst}",  "swapSides(nodep)");
#line 7851 "V3Const__gen.cpp"
#line 3859 "../V3Const.cpp"
//    TREEOP ("AstNodeBiComAsv{operandAsvConst(nodep)}",  "replaceAsv(nodep)");
#line 7854 "V3Const__gen.cpp"
#line 3860 "../V3Const.cpp"
//    TREEOP ("AstNodeBiComAsv{operandAsvSame(nodep)}",   "replaceAsv(nodep)");
#line 7857 "V3Const__gen.cpp"
#line 3861 "../V3Const.cpp"
//    TREEOP ("AstNodeBiComAsv{operandAsvLUp(nodep)}",    "replaceAsvLUp(nodep)");
#line 7860 "V3Const__gen.cpp"
#line 3862 "../V3Const.cpp"
//    TREEOP ("AstNodeBiComAsv{operandAsvRUp(nodep)}",    "replaceAsvRUp(nodep)");
#line 7863 "V3Const__gen.cpp"
#line 3863 "../V3Const.cpp"
//    TREEOP ("AstLt   {!$lhsp.castConst,$rhsp.castConst}",       "AstGt  {$rhsp,$lhsp}");
#line 7866 "V3Const__gen.cpp"
#line 3864 "../V3Const.cpp"
//    TREEOP ("AstLtS  {!$lhsp.castConst,$rhsp.castConst}",       "AstGtS {$rhsp,$lhsp}");
#line 7869 "V3Const__gen.cpp"
#line 3865 "../V3Const.cpp"
//    TREEOP ("AstLte  {!$lhsp.castConst,$rhsp.castConst}",       "AstGte {$rhsp,$lhsp}");
#line 7872 "V3Const__gen.cpp"
#line 3866 "../V3Const.cpp"
//    TREEOP ("AstLteS {!$lhsp.castConst,$rhsp.castConst}",       "AstGteS{$rhsp,$lhsp}");
#line 7875 "V3Const__gen.cpp"
#line 3867 "../V3Const.cpp"
//    TREEOP ("AstGt   {!$lhsp.castConst,$rhsp.castConst}",       "AstLt  {$rhsp,$lhsp}");
#line 7878 "V3Const__gen.cpp"
#line 3868 "../V3Const.cpp"
//    TREEOP ("AstGtS  {!$lhsp.castConst,$rhsp.castConst}",       "AstLtS {$rhsp,$lhsp}");
#line 7881 "V3Const__gen.cpp"
#line 3869 "../V3Const.cpp"
//    TREEOP ("AstGte  {!$lhsp.castConst,$rhsp.castConst}",       "AstLte {$rhsp,$lhsp}");
#line 7884 "V3Const__gen.cpp"
#line 3870 "../V3Const.cpp"
//    TREEOP ("AstGteS {!$lhsp.castConst,$rhsp.castConst}",       "AstLteS{$rhsp,$lhsp}");
#line 7887 "V3Const__gen.cpp"
#line 3871 "../V3Const.cpp"
    //    v--- *1* as These ops are always first, as we warn before replacing
//    TREEOP1("AstLt   {$lhsp, $rhsp.isZero}",            "replaceNumSigned(nodep,0)");
#line 7891 "V3Const__gen.cpp"
#line 3873 "../V3Const.cpp"
//    TREEOP1("AstGte  {$lhsp, $rhsp.isZero}",            "replaceNumSigned(nodep,1)");
#line 7894 "V3Const__gen.cpp"
#line 3874 "../V3Const.cpp"
//    TREEOP1("AstGt   {$lhsp.isZero, $rhsp}",            "replaceNumSigned(nodep,0)");
#line 7897 "V3Const__gen.cpp"
#line 3875 "../V3Const.cpp"
//    TREEOP1("AstLte  {$lhsp.isZero, $rhsp}",            "replaceNumSigned(nodep,1)");
#line 7900 "V3Const__gen.cpp"
#line 3876 "../V3Const.cpp"
//    TREEOP1("AstGt   {$lhsp, $rhsp.isAllOnes, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,0)");
#line 7903 "V3Const__gen.cpp"
#line 3877 "../V3Const.cpp"
//    TREEOP1("AstLte  {$lhsp, $rhsp.isAllOnes, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,1)");
#line 7906 "V3Const__gen.cpp"
#line 3878 "../V3Const.cpp"
//    TREEOP1("AstLt   {$lhsp.isAllOnes, $rhsp, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,0)");
#line 7909 "V3Const__gen.cpp"
#line 3879 "../V3Const.cpp"
//    TREEOP1("AstGte  {$lhsp.isAllOnes, $rhsp, $lhsp->width()==$rhsp->width()}",  "replaceNumLimited(nodep,1)");
#line 7912 "V3Const__gen.cpp"
#line 3880 "../V3Const.cpp"
    // Two level bubble pushing
//    TREEOP ("AstNot   {$lhsp.castNot,  $lhsp->width()==VN_AS($lhsp,,Not)->lhsp()->width()}", "replaceWChild(nodep, $lhsp->castNot()->lhsp())");  // NOT(NOT(x))->x
#line 7916 "V3Const__gen.cpp"
#line 3882 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castLogNot}",              "replaceWChild(nodep, $lhsp->castLogNot()->lhsp())");  // LOGNOT(LOGNOT(x))->x
#line 7919 "V3Const__gen.cpp"
#line 3883 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castEqCase, $lhsp.width1}","AstNeqCase{$lhsp->castEqCase()->lhsp(),$lhsp->castEqCase()->rhsp()}");
#line 7922 "V3Const__gen.cpp"
#line 3884 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castEqCase}",              "AstNeqCase{$lhsp->castEqCase()->lhsp(),$lhsp->castEqCase()->rhsp()}");
#line 7925 "V3Const__gen.cpp"
#line 3885 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castNeqCase, $lhsp.width1}","AstEqCase{$lhsp->castNeqCase()->lhsp(),$lhsp->castNeqCase()->rhsp()}");
#line 7928 "V3Const__gen.cpp"
#line 3886 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castNeqCase}",             "AstEqCase {$lhsp->castNeqCase()->lhsp(),$lhsp->castNeqCase()->rhsp()}");
#line 7931 "V3Const__gen.cpp"
#line 3887 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castEqWild, $lhsp.width1}","AstNeqWild{$lhsp->castEqWild()->lhsp(),$lhsp->castEqWild()->rhsp()}");
#line 7934 "V3Const__gen.cpp"
#line 3888 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castEqWild}",              "AstNeqWild{$lhsp->castEqWild()->lhsp(),$lhsp->castEqWild()->rhsp()}");
#line 7937 "V3Const__gen.cpp"
#line 3889 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castNeqWild, $lhsp.width1}","AstEqWild{$lhsp->castNeqWild()->lhsp(),$lhsp->castNeqWild()->rhsp()}");
#line 7940 "V3Const__gen.cpp"
#line 3890 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castNeqWild}",             "AstEqWild {$lhsp->castNeqWild()->lhsp(),$lhsp->castNeqWild()->rhsp()}");
#line 7943 "V3Const__gen.cpp"
#line 3891 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castEq, $lhsp.width1}",    "AstNeq {$lhsp->castEq()->lhsp(),$lhsp->castEq()->rhsp()}");
#line 7946 "V3Const__gen.cpp"
#line 3892 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castEq}",                  "AstNeq {$lhsp->castEq()->lhsp(),$lhsp->castEq()->rhsp()}");
#line 7949 "V3Const__gen.cpp"
#line 3893 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castNeq, $lhsp.width1}",   "AstEq  {$lhsp->castNeq()->lhsp(),$lhsp->castNeq()->rhsp()}");
#line 7952 "V3Const__gen.cpp"
#line 3894 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castNeq}",                 "AstEq  {$lhsp->castNeq()->lhsp(),$lhsp->castNeq()->rhsp()}");
#line 7955 "V3Const__gen.cpp"
#line 3895 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castLt, $lhsp.width1}",    "AstGte {$lhsp->castLt()->lhsp(),$lhsp->castLt()->rhsp()}");
#line 7958 "V3Const__gen.cpp"
#line 3896 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castLt}",                  "AstGte {$lhsp->castLt()->lhsp(),$lhsp->castLt()->rhsp()}");
#line 7961 "V3Const__gen.cpp"
#line 3897 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castLtS, $lhsp.width1}",   "AstGteS{$lhsp->castLtS()->lhsp(),$lhsp->castLtS()->rhsp()}");
#line 7964 "V3Const__gen.cpp"
#line 3898 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castLtS}",                 "AstGteS{$lhsp->castLtS()->lhsp(),$lhsp->castLtS()->rhsp()}");
#line 7967 "V3Const__gen.cpp"
#line 3899 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castLte, $lhsp.width1}",   "AstGt  {$lhsp->castLte()->lhsp(),$lhsp->castLte()->rhsp()}");
#line 7970 "V3Const__gen.cpp"
#line 3900 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castLte}",                 "AstGt  {$lhsp->castLte()->lhsp(),$lhsp->castLte()->rhsp()}");
#line 7973 "V3Const__gen.cpp"
#line 3901 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castLteS, $lhsp.width1}",  "AstGtS {$lhsp->castLteS()->lhsp(),$lhsp->castLteS()->rhsp()}");
#line 7976 "V3Const__gen.cpp"
#line 3902 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castLteS}",                "AstGtS {$lhsp->castLteS()->lhsp(),$lhsp->castLteS()->rhsp()}");
#line 7979 "V3Const__gen.cpp"
#line 3903 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castGt, $lhsp.width1}",    "AstLte {$lhsp->castGt()->lhsp(),$lhsp->castGt()->rhsp()}");
#line 7982 "V3Const__gen.cpp"
#line 3904 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castGt}",                  "AstLte {$lhsp->castGt()->lhsp(),$lhsp->castGt()->rhsp()}");
#line 7985 "V3Const__gen.cpp"
#line 3905 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castGtS, $lhsp.width1}",   "AstLteS{$lhsp->castGtS()->lhsp(),$lhsp->castGtS()->rhsp()}");
#line 7988 "V3Const__gen.cpp"
#line 3906 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castGtS}",                 "AstLteS{$lhsp->castGtS()->lhsp(),$lhsp->castGtS()->rhsp()}");
#line 7991 "V3Const__gen.cpp"
#line 3907 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castGte, $lhsp.width1}",   "AstLt  {$lhsp->castGte()->lhsp(),$lhsp->castGte()->rhsp()}");
#line 7994 "V3Const__gen.cpp"
#line 3908 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castGte}",                 "AstLt  {$lhsp->castGte()->lhsp(),$lhsp->castGte()->rhsp()}");
#line 7997 "V3Const__gen.cpp"
#line 3909 "../V3Const.cpp"
//    TREEOPV("AstNot   {$lhsp.castGteS, $lhsp.width1}",  "AstLtS {$lhsp->castGteS()->lhsp(),$lhsp->castGteS()->rhsp()}");
#line 8000 "V3Const__gen.cpp"
#line 3910 "../V3Const.cpp"
//    TREEOP ("AstLogNot{$lhsp.castGteS}",                "AstLtS {$lhsp->castGteS()->lhsp(),$lhsp->castGteS()->rhsp()}");
#line 8003 "V3Const__gen.cpp"
#line 3911 "../V3Const.cpp"
    // Not common, but avoids compiler warnings about over shifting
//    TREEOP ("AstShiftL   {operandHugeShiftL(nodep)}",   "replaceZero(nodep)");
#line 8007 "V3Const__gen.cpp"
#line 3913 "../V3Const.cpp"
//    TREEOP ("AstShiftLOvr{operandHugeShiftL(nodep)}",   "replaceZero(nodep)");
#line 8010 "V3Const__gen.cpp"
#line 3914 "../V3Const.cpp"
//    TREEOP ("AstShiftR   {operandHugeShiftR(nodep)}",   "replaceZero(nodep)");
#line 8013 "V3Const__gen.cpp"
#line 3915 "../V3Const.cpp"
//    TREEOP ("AstShiftROvr{operandHugeShiftR(nodep)}",   "replaceZero(nodep)");
#line 8016 "V3Const__gen.cpp"
#line 3916 "../V3Const.cpp"
//    TREEOP ("AstShiftL{operandShiftOp(nodep)}",         "replaceShiftOp(nodep)");
#line 8019 "V3Const__gen.cpp"
#line 3917 "../V3Const.cpp"
//    TREEOP ("AstShiftR{operandShiftOp(nodep)}",         "replaceShiftOp(nodep)");
#line 8022 "V3Const__gen.cpp"
#line 3918 "../V3Const.cpp"
//    TREEOP ("AstShiftL{operandShiftShift(nodep)}",      "replaceShiftShift(nodep)");
#line 8025 "V3Const__gen.cpp"
#line 3919 "../V3Const.cpp"
//    TREEOP ("AstShiftR{operandShiftShift(nodep)}",      "replaceShiftShift(nodep)");
#line 8028 "V3Const__gen.cpp"
#line 3920 "../V3Const.cpp"
//    TREEOP ("AstWordSel{operandWordOOB(nodep)}",        "replaceZero(nodep)");
#line 8031 "V3Const__gen.cpp"
#line 3921 "../V3Const.cpp"
    // Compress out EXTENDs to appease loop unroller
//    TREEOPV("AstEq    {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE");
#line 8035 "V3Const__gen.cpp"
#line 3923 "../V3Const.cpp"
//    TREEOPV("AstNeq   {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE");
#line 8038 "V3Const__gen.cpp"
#line 3924 "../V3Const.cpp"
//    TREEOPV("AstGt    {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE");
#line 8041 "V3Const__gen.cpp"
#line 3925 "../V3Const.cpp"
//    TREEOPV("AstGte   {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE");
#line 8044 "V3Const__gen.cpp"
#line 3926 "../V3Const.cpp"
//    TREEOPV("AstLt    {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE");
#line 8047 "V3Const__gen.cpp"
#line 3927 "../V3Const.cpp"
//    TREEOPV("AstLte   {$rhsp.castExtend,operandBiExtendConstShrink(nodep)}",    "DONE");
#line 8050 "V3Const__gen.cpp"
#line 3928 "../V3Const.cpp"
//    TREEOPV("AstEq    {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceZero(nodep)");
#line 8053 "V3Const__gen.cpp"
#line 3929 "../V3Const.cpp"
//    TREEOPV("AstNeq   {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceNum(nodep,1)");
#line 8056 "V3Const__gen.cpp"
#line 3930 "../V3Const.cpp"
//    TREEOPV("AstGt    {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceNum(nodep,1)");
#line 8059 "V3Const__gen.cpp"
#line 3931 "../V3Const.cpp"
//    TREEOPV("AstGte   {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceNum(nodep,1)");
#line 8062 "V3Const__gen.cpp"
#line 3932 "../V3Const.cpp"
//    TREEOPV("AstLt    {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceZero(nodep)");
#line 8065 "V3Const__gen.cpp"
#line 3933 "../V3Const.cpp"
//    TREEOPV("AstLte   {$rhsp.castExtend,operandBiExtendConstOver(nodep)}",      "replaceZero(nodep)");
#line 8068 "V3Const__gen.cpp"
#line 3934 "../V3Const.cpp"
    // Identical operands on both sides
    // AstLogAnd/AstLogOr already converted to AstAnd/AstOr for these rules
    // AstAdd->ShiftL(#,1) but uncommon
//    TREEOP ("AstAnd    {operandsSame($lhsp,,$rhsp)}",   "replaceWLhs(nodep)");
#line 8074 "V3Const__gen.cpp"
#line 3938 "../V3Const.cpp"
//    TREEOP ("AstDiv    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8077 "V3Const__gen.cpp"
#line 3939 "../V3Const.cpp"
//    TREEOP ("AstDivS   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8080 "V3Const__gen.cpp"
#line 3940 "../V3Const.cpp"
//    TREEOP ("AstOr     {operandsSame($lhsp,,$rhsp)}",   "replaceWLhs(nodep)");
#line 8083 "V3Const__gen.cpp"
#line 3941 "../V3Const.cpp"
//    TREEOP ("AstSub    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8086 "V3Const__gen.cpp"
#line 3942 "../V3Const.cpp"
//    TREEOP ("AstXor    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8089 "V3Const__gen.cpp"
#line 3943 "../V3Const.cpp"
//    TREEOP ("AstEq     {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");  // We let X==X -> 1, although in a true 4-state sim it's X.
#line 8092 "V3Const__gen.cpp"
#line 3944 "../V3Const.cpp"
//    TREEOP ("AstEqD    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");  // We let X==X -> 1, although in a true 4-state sim it's X.
#line 8095 "V3Const__gen.cpp"
#line 3945 "../V3Const.cpp"
//    TREEOP ("AstEqN    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");  // We let X==X -> 1, although in a true 4-state sim it's X.
#line 8098 "V3Const__gen.cpp"
#line 3946 "../V3Const.cpp"
//    TREEOP ("AstEqCase {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8101 "V3Const__gen.cpp"
#line 3947 "../V3Const.cpp"
//    TREEOP ("AstEqWild {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8104 "V3Const__gen.cpp"
#line 3948 "../V3Const.cpp"
//    TREEOP ("AstGt     {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8107 "V3Const__gen.cpp"
#line 3949 "../V3Const.cpp"
//    TREEOP ("AstGtD    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8110 "V3Const__gen.cpp"
#line 3950 "../V3Const.cpp"
//    TREEOP ("AstGtN    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8113 "V3Const__gen.cpp"
#line 3951 "../V3Const.cpp"
//    TREEOP ("AstGtS    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8116 "V3Const__gen.cpp"
#line 3952 "../V3Const.cpp"
//    TREEOP ("AstGte    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8119 "V3Const__gen.cpp"
#line 3953 "../V3Const.cpp"
//    TREEOP ("AstGteD   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8122 "V3Const__gen.cpp"
#line 3954 "../V3Const.cpp"
//    TREEOP ("AstGteN   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8125 "V3Const__gen.cpp"
#line 3955 "../V3Const.cpp"
//    TREEOP ("AstGteS   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8128 "V3Const__gen.cpp"
#line 3956 "../V3Const.cpp"
//    TREEOP ("AstLt     {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8131 "V3Const__gen.cpp"
#line 3957 "../V3Const.cpp"
//    TREEOP ("AstLtD    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8134 "V3Const__gen.cpp"
#line 3958 "../V3Const.cpp"
//    TREEOP ("AstLtN    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8137 "V3Const__gen.cpp"
#line 3959 "../V3Const.cpp"
//    TREEOP ("AstLtS    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8140 "V3Const__gen.cpp"
#line 3960 "../V3Const.cpp"
//    TREEOP ("AstLte    {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8143 "V3Const__gen.cpp"
#line 3961 "../V3Const.cpp"
//    TREEOP ("AstLteD   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8146 "V3Const__gen.cpp"
#line 3962 "../V3Const.cpp"
//    TREEOP ("AstLteN   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8149 "V3Const__gen.cpp"
#line 3963 "../V3Const.cpp"
//    TREEOP ("AstLteS   {operandsSame($lhsp,,$rhsp)}",   "replaceNum(nodep,1)");
#line 8152 "V3Const__gen.cpp"
#line 3964 "../V3Const.cpp"
//    TREEOP ("AstNeq    {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8155 "V3Const__gen.cpp"
#line 3965 "../V3Const.cpp"
//    TREEOP ("AstNeqD   {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8158 "V3Const__gen.cpp"
#line 3966 "../V3Const.cpp"
//    TREEOP ("AstNeqN   {operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8161 "V3Const__gen.cpp"
#line 3967 "../V3Const.cpp"
//    TREEOP ("AstNeqCase{operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8164 "V3Const__gen.cpp"
#line 3968 "../V3Const.cpp"
//    TREEOP ("AstNeqWild{operandsSame($lhsp,,$rhsp)}",   "replaceZero(nodep)");
#line 8167 "V3Const__gen.cpp"
#line 3969 "../V3Const.cpp"
//    TREEOP ("AstLogAnd {operandsSame($lhsp,,$rhsp)}",   "replaceWLhsBool(nodep)");
#line 8170 "V3Const__gen.cpp"
#line 3970 "../V3Const.cpp"
//    TREEOP ("AstLogOr  {operandsSame($lhsp,,$rhsp)}",   "replaceWLhsBool(nodep)");
#line 8173 "V3Const__gen.cpp"
#line 3971 "../V3Const.cpp"
    ///=== Verilog operators
    // Comparison against 1'b0/1'b1; must be careful about widths.
    // These use Not, so must be Verilog only
//    TREEOPV("AstEq    {$rhsp.width1, $lhsp.isZero,    $rhsp}",  "AstNot{$rhsp}");
#line 8179 "V3Const__gen.cpp"
#line 3975 "../V3Const.cpp"
//    TREEOPV("AstEq    {$lhsp.width1, $lhsp, $rhsp.isZero}",     "AstNot{$lhsp}");
#line 8182 "V3Const__gen.cpp"
#line 3976 "../V3Const.cpp"
//    TREEOPV("AstEq    {$rhsp.width1, $lhsp.isAllOnes, $rhsp}",  "replaceWRhs(nodep)");
#line 8185 "V3Const__gen.cpp"
#line 3977 "../V3Const.cpp"
//    TREEOPV("AstEq    {$lhsp.width1, $lhsp, $rhsp.isAllOnes}",  "replaceWLhs(nodep)");
#line 8188 "V3Const__gen.cpp"
#line 3978 "../V3Const.cpp"
//    TREEOPV("AstNeq   {$rhsp.width1, $lhsp.isZero,    $rhsp}",  "replaceWRhs(nodep)");
#line 8191 "V3Const__gen.cpp"
#line 3979 "../V3Const.cpp"
//    TREEOPV("AstNeq   {$lhsp.width1, $lhsp, $rhsp.isZero}",     "replaceWLhs(nodep)");
#line 8194 "V3Const__gen.cpp"
#line 3980 "../V3Const.cpp"
//    TREEOPV("AstNeq   {$rhsp.width1, $lhsp.isAllOnes, $rhsp}",  "AstNot{$rhsp}");
#line 8197 "V3Const__gen.cpp"
#line 3981 "../V3Const.cpp"
//    TREEOPV("AstNeq   {$lhsp.width1, $lhsp, $rhsp.isAllOnes}",  "AstNot{$lhsp}");
#line 8200 "V3Const__gen.cpp"
#line 3982 "../V3Const.cpp"
//    TREEOPV("AstLt    {$rhsp.width1, $lhsp.isZero,    $rhsp}",  "replaceWRhs(nodep)");  // Because not signed #s
#line 8203 "V3Const__gen.cpp"
#line 3983 "../V3Const.cpp"
//    TREEOPV("AstGt    {$lhsp.width1, $lhsp, $rhsp.isZero}",     "replaceWLhs(nodep)");  // Because not signed #s
#line 8206 "V3Const__gen.cpp"
#line 3984 "../V3Const.cpp"
    // Useful for CONDs added around ARRAYSEL's in V3Case step
//    TREEOPV("AstLte   {$lhsp->width()==$rhsp->width(), $rhsp.isAllOnes}", "replaceNum(nodep,1)");
#line 8210 "V3Const__gen.cpp"
#line 3986 "../V3Const.cpp"
    // Simplify reduction operators
    // This also gets &{...,0,....} => const 0  (Common for unused_ok signals)
//    TREEOPV("AstRedAnd{$lhsp, $lhsp.width1}",   "replaceWLhs(nodep)");
#line 8215 "V3Const__gen.cpp"
#line 3989 "../V3Const.cpp"
//    TREEOPV("AstRedOr {$lhsp, $lhsp.width1}",   "replaceWLhs(nodep)");
#line 8218 "V3Const__gen.cpp"
#line 3990 "../V3Const.cpp"
//    TREEOPV("AstRedXor{$lhsp, $lhsp.width1}",   "replaceWLhs(nodep)");
#line 8221 "V3Const__gen.cpp"
#line 3991 "../V3Const.cpp"
//    TREEOPV("AstRedAnd{$lhsp.castConcat}",      "AstAnd{AstRedAnd{$lhsp->castConcat()->lhsp()}, AstRedAnd{$lhsp->castConcat()->rhsp()}}");  // &{a,b} => {&a}&{&b}
#line 8224 "V3Const__gen.cpp"
#line 3992 "../V3Const.cpp"
//    TREEOPV("AstRedOr {$lhsp.castConcat}",      "AstOr {AstRedOr {$lhsp->castConcat()->lhsp()}, AstRedOr {$lhsp->castConcat()->rhsp()}}");  // |{a,b} => {|a}|{|b}
#line 8227 "V3Const__gen.cpp"
#line 3993 "../V3Const.cpp"
//    TREEOPV("AstRedXor{$lhsp.castConcat}",      "AstXor{AstRedXor{$lhsp->castConcat()->lhsp()}, AstRedXor{$lhsp->castConcat()->rhsp()}}");  // ^{a,b} => {^a}^{^b}
#line 8230 "V3Const__gen.cpp"
#line 3994 "../V3Const.cpp"
//    TREEOPV("AstRedAnd{$lhsp.castExtend, $lhsp->width() > VN_AS($lhsp,,Extend)->lhsp()->width()}", "replaceZero(nodep)");  // &{0,...} => 0  Prevents compiler limited range error
#line 8233 "V3Const__gen.cpp"
#line 3995 "../V3Const.cpp"
//    TREEOPV("AstRedOr {$lhsp.castExtend}",      "AstRedOr {$lhsp->castExtend()->lhsp()}");
#line 8236 "V3Const__gen.cpp"
#line 3996 "../V3Const.cpp"
//    TREEOPV("AstRedXor{$lhsp.castExtend}",      "AstRedXor{$lhsp->castExtend()->lhsp()}");
#line 8239 "V3Const__gen.cpp"
#line 3997 "../V3Const.cpp"
//    TREEOP ("AstRedXor{$lhsp.castXor, VN_IS(VN_AS($lhsp,,Xor)->lhsp(),,Const)}", "AstXor{AstRedXor{$lhsp->castXor()->lhsp()}, AstRedXor{$lhsp->castXor()->rhsp()}}");  // ^(const ^ a) => (^const)^(^a)
#line 8242 "V3Const__gen.cpp"
#line 3998 "../V3Const.cpp"
//    TREEOPC("AstAnd {$lhsp.castConst, $rhsp.castRedXor, matchBitOpTree(nodep)}", "DONE");
#line 8245 "V3Const__gen.cpp"
#line 3999 "../V3Const.cpp"
//    TREEOPV("AstOneHot{$lhsp.width1}",          "replaceWLhs(nodep)");
#line 8248 "V3Const__gen.cpp"
#line 4000 "../V3Const.cpp"
//    TREEOPV("AstOneHot0{$lhsp.width1}",         "replaceNum(nodep,1)");
#line 8251 "V3Const__gen.cpp"
#line 4001 "../V3Const.cpp"
    // Binary AND/OR is faster than logical and/or (usually)
//    TREEOPV("AstLogAnd{matchBiopToBitwise(nodep)}", "AstAnd{$lhsp,$rhsp}");
#line 8255 "V3Const__gen.cpp"
#line 4003 "../V3Const.cpp"
//    TREEOPV("AstLogOr {matchBiopToBitwise(nodep)}", "AstOr{$lhsp,$rhsp}");
#line 8258 "V3Const__gen.cpp"
#line 4004 "../V3Const.cpp"
//    TREEOPV("AstLogNot{$lhsp.width1}",  "AstNot{$lhsp}");
#line 8261 "V3Const__gen.cpp"
#line 4005 "../V3Const.cpp"
    // CONCAT(CONCAT({a},{b}),{c}) -> CONCAT({a},CONCAT({b},{c}))
    // CONCAT({const},CONCAT({const},{c})) -> CONCAT((constifiedCONC{const|const},{c}))
//    TREEOPV("AstConcat{matchConcatRand(nodep)}",      "DONE");
#line 8266 "V3Const__gen.cpp"
#line 4008 "../V3Const.cpp"
//    TREEOPV("AstConcat{operandConcatMove(nodep)}",      "moveConcat(nodep)");
#line 8269 "V3Const__gen.cpp"
#line 4009 "../V3Const.cpp"
//    TREEOPV("AstConcat{$lhsp.isZero, $rhsp}",           "replaceExtend(nodep, nodep->rhsp())");
#line 8272 "V3Const__gen.cpp"
#line 4010 "../V3Const.cpp"
    // CONCAT(a[1],a[0]) -> a[1:0]
//    TREEOPV("AstConcat{$lhsp.castSel, $rhsp.castSel, ifAdjacentSel(VN_AS($lhsp,,Sel),,VN_AS($rhsp,,Sel))}",  "replaceConcatSel(nodep)");
#line 8276 "V3Const__gen.cpp"
#line 4012 "../V3Const.cpp"
//    TREEOPV("AstConcat{ifConcatMergeableBiop($lhsp), concatMergeable($lhsp,,$rhsp,,0)}", "replaceConcatMerge(nodep)");
#line 8279 "V3Const__gen.cpp"
#line 4013 "../V3Const.cpp"
    // Common two-level operations that can be simplified
//    TREEOP ("AstAnd {$lhsp.castConst,matchAndCond(nodep)}", "DONE");
#line 8283 "V3Const__gen.cpp"
#line 4015 "../V3Const.cpp"
//    TREEOP ("AstAnd {$lhsp.castConst, $rhsp.castOr, matchMaskedOr(nodep)}", "DONE");
#line 8286 "V3Const__gen.cpp"
#line 4016 "../V3Const.cpp"
//    TREEOPC("AstAnd {$lhsp.castConst, matchMaskedShift(nodep)}", "DONE");
#line 8289 "V3Const__gen.cpp"
#line 4017 "../V3Const.cpp"
//    TREEOP ("AstAnd {$lhsp.castOr, $rhsp.castOr, operandAndOrSame(nodep)}", "replaceAndOr(nodep)");
#line 8292 "V3Const__gen.cpp"
#line 4018 "../V3Const.cpp"
//    TREEOP ("AstOr  {$lhsp.castAnd,$rhsp.castAnd,operandAndOrSame(nodep)}", "replaceAndOr(nodep)");
#line 8295 "V3Const__gen.cpp"
#line 4019 "../V3Const.cpp"
//    TREEOP ("AstOr  {matchOrAndNot(nodep)}",            "DONE");
#line 8298 "V3Const__gen.cpp"
#line 4020 "../V3Const.cpp"
//    TREEOP ("AstAnd {operandShiftSame(nodep)}",         "replaceShiftSame(nodep)");
#line 8301 "V3Const__gen.cpp"
#line 4021 "../V3Const.cpp"
//    TREEOP ("AstOr  {operandShiftSame(nodep)}",         "replaceShiftSame(nodep)");
#line 8304 "V3Const__gen.cpp"
#line 4022 "../V3Const.cpp"
//    TREEOP ("AstXor {operandShiftSame(nodep)}",         "replaceShiftSame(nodep)");
#line 8307 "V3Const__gen.cpp"
#line 4023 "../V3Const.cpp"
//    TREEOPC("AstAnd {matchBitOpTree(nodep)}", "DONE");
#line 8310 "V3Const__gen.cpp"
#line 4024 "../V3Const.cpp"
//    TREEOPC("AstOr  {matchBitOpTree(nodep)}", "DONE");
#line 8313 "V3Const__gen.cpp"
#line 4025 "../V3Const.cpp"
//    TREEOPC("AstXor {matchBitOpTree(nodep)}", "DONE");
#line 8316 "V3Const__gen.cpp"
#line 4026 "../V3Const.cpp"
    // Note can't simplify a extend{extends}, extends{extend}, as the sign
    // bits end up in the wrong places
//    TREEOPV("AstExtend{operandsSameWidth(nodep,,$lhsp)}",  "replaceWLhs(nodep)");
#line 8321 "V3Const__gen.cpp"
#line 4029 "../V3Const.cpp"
//    TREEOPV("AstExtend{$lhsp.castExtend}",  "replaceExtend(nodep, VN_AS(nodep->lhsp(), Extend)->lhsp())");
#line 8324 "V3Const__gen.cpp"
#line 4030 "../V3Const.cpp"
//    TREEOPV("AstExtendS{$lhsp.castExtendS}", "replaceExtend(nodep, VN_AS(nodep->lhsp(), ExtendS)->lhsp())");
#line 8327 "V3Const__gen.cpp"
#line 4031 "../V3Const.cpp"
//    TREEOPV("AstReplicate{$srcp, $countp.isOne, $srcp->width()==nodep->width()}", "replaceWLhs(nodep)");  // {1{lhs}}->lhs
#line 8330 "V3Const__gen.cpp"
#line 4032 "../V3Const.cpp"
//    TREEOPV("AstReplicateN{$lhsp, $rhsp.isOne, $lhsp->width()==nodep->width()}", "replaceWLhs(nodep)");  // {1{lhs}}->lhs
#line 8333 "V3Const__gen.cpp"
#line 4033 "../V3Const.cpp"
//    TREEOPV("AstReplicate{$srcp.castReplicate, operandRepRep(nodep)}", "DONE");  // {2{3{lhs}}}->{6{lhs}}
#line 8336 "V3Const__gen.cpp"
#line 4034 "../V3Const.cpp"
//    TREEOPV("AstConcat{operandConcatSame(nodep)}", "DONE");  // {a,a}->{2{a}}, {a,2{a}}->{3{a}, etc
#line 8339 "V3Const__gen.cpp"
#line 4035 "../V3Const.cpp"
    // Next rule because AUTOINST puts the width of bits in
    // to pins, even when the widths are exactly the same across the hierarchy.
//    TREEOPV("AstSel{matchSelRand(nodep)}",      "DONE");
#line 8344 "V3Const__gen.cpp"
#line 4038 "../V3Const.cpp"
//    TREEOPV("AstSel{operandSelExtend(nodep)}",  "DONE");
#line 8347 "V3Const__gen.cpp"
#line 4039 "../V3Const.cpp"
//    TREEOPV("AstSel{operandSelFull(nodep)}",    "replaceWChild(nodep, nodep->fromp())");
#line 8350 "V3Const__gen.cpp"
#line 4040 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castSel}",           "replaceSelSel(nodep)");
#line 8353 "V3Const__gen.cpp"
#line 4041 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castAdd, operandSelBiLower(nodep)}", "DONE");
#line 8356 "V3Const__gen.cpp"
#line 4042 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castAnd, operandSelBiLower(nodep)}", "DONE");
#line 8359 "V3Const__gen.cpp"
#line 4043 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castOr,  operandSelBiLower(nodep)}", "DONE");
#line 8362 "V3Const__gen.cpp"
#line 4044 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castSub, operandSelBiLower(nodep)}", "DONE");
#line 8365 "V3Const__gen.cpp"
#line 4045 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castXor, operandSelBiLower(nodep)}", "DONE");
#line 8368 "V3Const__gen.cpp"
#line 4046 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castShiftR, operandSelShiftLower(nodep)}",   "DONE");
#line 8371 "V3Const__gen.cpp"
#line 4047 "../V3Const.cpp"
//    TREEOPA("AstSel{$fromp.castConst, $lsbp.castConst, }",   "replaceConst(nodep)");
#line 8374 "V3Const__gen.cpp"
#line 4048 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castConcat, $lsbp.castConst, }",  "replaceSelConcat(nodep)");
#line 8377 "V3Const__gen.cpp"
#line 4049 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castReplicate, $lsbp.castConst, operandSelReplicate(nodep) }",    "DONE");
#line 8380 "V3Const__gen.cpp"
#line 4050 "../V3Const.cpp"
    // V3Tristate requires selects below BufIf1.
    // Also do additional operators that are bit-independent, but only definite
    // win if bit select is a constant (otherwise we may need to compute bit index several times)
//    TREEOPV("AstSel{$fromp.castBufIf1}",                "replaceSelIntoBiop(nodep)");
#line 8386 "V3Const__gen.cpp"
#line 4054 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castNot}",                   "replaceSelIntoUniop(nodep)");
#line 8389 "V3Const__gen.cpp"
#line 4055 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castAnd,$lsbp.castConst}",   "replaceSelIntoBiop(nodep)");
#line 8392 "V3Const__gen.cpp"
#line 4056 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castOr,$lsbp.castConst}",    "replaceSelIntoBiop(nodep)");
#line 8395 "V3Const__gen.cpp"
#line 4057 "../V3Const.cpp"
//    TREEOPV("AstSel{$fromp.castXor,$lsbp.castConst}",   "replaceSelIntoBiop(nodep)");
#line 8398 "V3Const__gen.cpp"
#line 4058 "../V3Const.cpp"
    // This visit function here must allow for short-circuiting.
//    TREEOPS("AstLogIf{$lhsp.isZero}",  "replaceNum(nodep, 1)");
#line 8402 "V3Const__gen.cpp"
#line 4060 "../V3Const.cpp"
//    TREEOPV("AstLogIf{$lhsp, $rhsp}",  "AstLogOr{AstLogNot{$lhsp},$rhsp}");
#line 8405 "V3Const__gen.cpp"
#line 4061 "../V3Const.cpp"
//    TREEOPV("AstLogEq{$lhsp, $rhsp}",  "replaceLogEq(nodep)");
#line 8408 "V3Const__gen.cpp"
#line 4062 "../V3Const.cpp"
    // Strings
//    TREEOPA("AstPutcN{$lhsp.castConst, $rhsp.castConst, $thsp.castConst}",  "replaceConst(nodep)");
#line 8412 "V3Const__gen.cpp"
#line 4064 "../V3Const.cpp"
//    TREEOPA("AstSubstrN{$lhsp.castConst, $rhsp.castConst, $thsp.castConst}",  "replaceConst(nodep)");
#line 8415 "V3Const__gen.cpp"
#line 4065 "../V3Const.cpp"
//    TREEOPA("AstCvtPackString{$lhsp.castConst}", "replaceConstString(nodep, VN_AS(nodep->lhsp(), Const)->num().toString())");
#line 8418 "V3Const__gen.cpp"
#line 4066 "../V3Const.cpp"
//    TREEOP ("AstToStringN{DISABLE_BASE}", "DONE");  // Avoid uniop(const); use matchToStringNConst
#line 8421 "V3Const__gen.cpp"
#line 4067 "../V3Const.cpp"
//    TREEOP ("AstToStringN{matchToStringNConst(nodep)}", "DONE");
#line 8424 "V3Const__gen.cpp"
#line 4068 "../V3Const.cpp"
    // Custom
    // Implied by AstIsUnbounded::numberOperate: V("AstIsUnbounded{$lhsp.castConst}", "replaceNum(nodep, 0)");
//    TREEOPV("AstIsUnbounded{$lhsp.castUnbounded}", "replaceNum(nodep, 1)");
#line 8429 "V3Const__gen.cpp"
#line 4071 "../V3Const.cpp"
    // clang-format on

    // Possible futures:
    // (a?(b?y:x):y) -> (a&&!b)?x:y
    // (a?(b?x:y):y) -> (a&&b)?x:y
    // (a?x:(b?x:y)) -> (a||b)?x:y
    // (a?x:(b?y:x)) -> (a||!b)?x:y

    // Note we can't convert EqCase/NeqCase to Eq/Neq here because that would break 3'b1x1==3'b101

    //-----
    void visit(AstNode* nodep) override {
        // Default: Just iterate
        if (m_required) {
            if (VN_IS(nodep, NodeDType) || VN_IS(nodep, Range) || VN_IS(nodep, SliceSel)) {
                // Ignore dtypes for parameter type pins
            } else {
                nodep->v3error("Expecting expression to be constant, but can't convert a "
                               << nodep->prettyTypeName() << " to constant.");
            }
        } else {
            if (nodep->isTimingControl()) m_hasJumpDelay = true;
            // Calculate the width of this operation
            if (m_params && !nodep->width()) nodep = V3Width::widthParamsEdit(nodep);
            iterateChildren(nodep);
        }
    }

public:
    // Processing Mode Enum
    enum ProcMode : uint8_t {
        PROC_PARAMS_NOWARN,
        PROC_PARAMS,
        PROC_GENERATE,
        PROC_LIVE,
        PROC_V_WARN,
        PROC_V_NOWARN,
        PROC_V_EXPENSIVE,
        PROC_CPP
    };

    // CONSTRUCTORS
    ConstVisitor(ProcMode pmode, bool globalPass)
        : m_globalPass{globalPass}
        , m_concswapNames{globalPass ? ("__Vconcswap_" + cvtToStr(s_globalPassNum++)) : ""} {
        // clang-format off
        switch (pmode) {
        case PROC_PARAMS_NOWARN:  m_doV = true;  m_doNConst = true; m_params = true;
                                  m_required = false; break;
        case PROC_PARAMS:         m_doV = true;  m_doNConst = true; m_params = true;
                                  m_required = true; break;
        case PROC_GENERATE:       m_doV = true;  m_doNConst = true; m_params = true;
                                  m_required = true; m_doGenerate = true; break;
        case PROC_LIVE:           break;
        case PROC_V_WARN:         m_doV = true;  m_doNConst = true; m_warn = true; m_convertLogicToBit = true; break;
        case PROC_V_NOWARN:       m_doV = true;  m_doNConst = true; break;
        case PROC_V_EXPENSIVE:    m_doV = true;  m_doNConst = true; m_doExpensive = true; break;
        case PROC_CPP:            m_doV = false; m_doNConst = true; m_doCpp = true; break;
        default:                  v3fatalSrc("Bad case"); break;
        }
        // clang-format on
    }
    ~ConstVisitor() override {
        if (m_doCpp) {
            if (m_globalPass) {
                V3Stats::addStat("Optimizations, Const bit op reduction", m_statBitOpReduction);
            } else {
                V3Stats::addStatSum("Optimizations, Const bit op reduction", m_statBitOpReduction);
            }
        }
        V3Stats::addStatSum("Optimizations, Cond redundant expressions", m_statCondExprRedundant);
        V3Stats::addStatSum("Optimizations, If cond redundant expressions",
                            m_statIfCondExprRedundant);
        V3Stats::addStatSum("Optimizations, Concat merges", m_statConcatMerge);
    }

    AstNode* mainAcceptEdit(AstNode* nodep) {
        VIsCached::clearCacheTree();  // Avoid using any stale isPure
        // Operate starting at a random place
        return iterateSubtreeReturnEdits(nodep);
    }
};

uint32_t ConstVisitor::s_globalPassNum = 0;

//######################################################################
// Const class functions

//! Force this cell node's parameter list to become a constant
void V3Const::constifyParamsEdit(AstNode* nodep) {
    // UINFOTREE(1, nodep, "", "forceConPRE");
    // Resize even if the node already has a width, because buried in the tree
    // we may have a node we just created with signing, etc, that isn't sized yet.

    // Make sure we've sized everything first
    nodep = V3Width::widthParamsEdit(nodep);
    ConstVisitor visitor{ConstVisitor::PROC_PARAMS, /* globalPass: */ false};
    // cppcheck-suppress constVariablePointer // edited below
    if (AstVar* const varp = VN_CAST(nodep, Var)) {
        // If a var wants to be constified, it's really a param, and
        // we want the value to be constant.  We aren't passed just the
        // init value because we need widthing above to handle the var's type.
        if (varp->valuep()) visitor.mainAcceptEdit(varp->valuep());
    } else {
        (void)visitor.mainAcceptEdit(nodep);
    }
    // Because we do edits, nodep links may get trashed and core dump if have next line
    // UINFOTREE(1, nodep, "", "forceConDONE");
}

//! Constify this cell node's parameter list if possible
void V3Const::constifyParamsNoWarnEdit(AstNode* nodep) {
    // UINFOTREE(1, nodep, "", "forceConPRE");
    // Resize even if the node already has a width, because buried in the tree
    // we may have a node we just created with signing, etc, that isn't sized yet.

    // Make sure we've sized everything first
    nodep = V3Width::widthParamsEdit(nodep);
    ConstVisitor visitor{ConstVisitor::PROC_PARAMS_NOWARN, /* globalPass: */ false};
    // cppcheck-suppress constVariablePointer // edited below
    if (AstVar* const varp = VN_CAST(nodep, Var)) {
        // If a var wants to be constified, it's really a param, and
        // we want the value to be constant.  We aren't passed just the
        // init value because we need widthing above to handle the var's type.
        if (varp->valuep()) visitor.mainAcceptEdit(varp->valuep());
    } else {
        (void)visitor.mainAcceptEdit(nodep);
    }
    // Because we do edits, nodep links may get trashed and core dump this.
    // UINFOTREE(1, nodep, "", "forceConDONE");
}

//! Force this cell node's parameter list to become a constant inside generate.
//! If we are inside a generated "if", "case" or "for", we don't want to
//! trigger warnings when we deal with the width. It is possible that these
//! are spurious, existing within sub-expressions that will not actually be
//! generated. Since such occurrences, must be constant, in order to be
//! something a generate block can depend on, we can wait until later to do the
//! width check.
void V3Const::constifyGenerateParamsEdit(AstNode* nodep) {
    // UINFOTREE(1, nodep, "", "forceConPRE:");
    // Resize even if the node already has a width, because buried in the tree
    // we may have a node we just created with signing, etc, that isn't sized
    // yet.

    // Make sure we've sized everything first
    nodep = V3Width::widthGenerateParamsEdit(nodep);
    ConstVisitor visitor{ConstVisitor::PROC_GENERATE, /* globalPass: */ false};
    // cppcheck-suppress constVariablePointer // edited below
    if (AstVar* const varp = VN_CAST(nodep, Var)) {
        // If a var wants to be constified, it's really a param, and
        // we want the value to be constant.  We aren't passed just the
        // init value because we need widthing above to handle the var's type.
        if (varp->valuep()) visitor.mainAcceptEdit(varp->valuep());
    } else {
        VL_DO_DANGLING(visitor.mainAcceptEdit(nodep), nodep);
    }
    // Because we do edits, nodep links may get trashed and core dump this.
    // UINFOTREE(1, nodep, "", "forceConDONE");
}

void V3Const::constifyAllLint(AstNetlist* nodep) {
    // Only call from Verilator.cpp, as it uses user#'s
    UINFO(2, __FUNCTION__ << ":");
    {
        ConstVisitor visitor{ConstVisitor::PROC_V_WARN, /* globalPass: */ true};
        (void)visitor.mainAcceptEdit(nodep);
    }  // Destruct before checking
    V3Global::dumpCheckGlobalTree("const", 0, dumpTreeEitherLevel() >= 3);
}

void V3Const::constifyCpp(AstNetlist* nodep) {
    UINFO(2, __FUNCTION__ << ":");
    {
        ConstVisitor visitor{ConstVisitor::PROC_CPP, /* globalPass: */ true};
        (void)visitor.mainAcceptEdit(nodep);
    }  // Destruct before checking
    V3Global::dumpCheckGlobalTree("const_cpp", 0, dumpTreeEitherLevel() >= 3);
}

AstNode* V3Const::constifyEdit(AstNode* nodep) {
    ConstVisitor visitor{ConstVisitor::PROC_V_NOWARN, /* globalPass: */ false};
    nodep = visitor.mainAcceptEdit(nodep);
    return nodep;
}

AstNode* V3Const::constifyEditCpp(AstNode* nodep) {
    ConstVisitor visitor{ConstVisitor::PROC_CPP, /* globalPass: */ false};
    nodep = visitor.mainAcceptEdit(nodep);
    return nodep;
}

void V3Const::constifyAllLive(AstNetlist* nodep) {
    // Only call from Verilator.cpp, as it uses user#'s
    // This only pushes constants up, doesn't make any other edits
    // IE doesn't prune dead statements, as we need to do some usability checks after this
    UINFO(2, __FUNCTION__ << ":");
    {
        ConstVisitor visitor{ConstVisitor::PROC_LIVE, /* globalPass: */ true};
        (void)visitor.mainAcceptEdit(nodep);
    }  // Destruct before checking
    V3Global::dumpCheckGlobalTree("const", 0, dumpTreeEitherLevel() >= 3);
}

void V3Const::constifyAll(AstNetlist* nodep) {
    // Only call from Verilator.cpp, as it uses user#'s
    UINFO(2, __FUNCTION__ << ":");
    {
        ConstVisitor visitor{ConstVisitor::PROC_V_EXPENSIVE, /* globalPass: */ true};
        (void)visitor.mainAcceptEdit(nodep);
    }  // Destruct before checking
    V3Global::dumpCheckGlobalTree("const", 0, dumpTreeEitherLevel() >= 3);
}

AstNode* V3Const::constifyExpensiveEdit(AstNode* nodep) {
    ConstVisitor visitor{ConstVisitor::PROC_V_EXPENSIVE, /* globalPass: */ false};
    nodep = visitor.mainAcceptEdit(nodep);
    return nodep;
}
