import events from "node:events";
import { type TsType as AdapterTsType } from "../adapter";
import type { ZclPayload } from "../adapter/events";
import type { CustomClusters } from "../zspec/zcl/definition/tstype";
import type * as ZdoTypes from "../zspec/zdo/definition/tstypes";
import type * as Events from "./events";
import GreenPower from "./greenPower";
import { Device } from "./model";
import Group from "./model/group";
import Touchlink from "./touchlink";
import type { DeviceType, RawPayload } from "./tstype";
interface Options {
    network: AdapterTsType.NetworkOptions;
    serialPort: AdapterTsType.SerialPortOptions;
    databasePath: string;
    databaseBackupPath: string;
    backupPath: string;
    adapter: AdapterTsType.AdapterOptions;
    /**
     * This lambda can be used by an application to explictly reject or accept an incoming device.
     * When false is returned zigbee-herdsman will not start the interview process and immidiately
     * try to remove the device from the network.
     */
    acceptJoiningDeviceHandler: (ieeeAddr: string) => Promise<boolean>;
}
export interface ControllerEventMap {
    message: [data: Events.MessagePayload];
    adapterDisconnected: [];
    deviceJoined: [data: Events.DeviceJoinedPayload];
    deviceInterview: [data: Events.DeviceInterviewPayload];
    deviceAnnounce: [data: Events.DeviceAnnouncePayload];
    deviceNetworkAddressChanged: [data: Events.DeviceNetworkAddressChangedPayload];
    deviceLeave: [data: Events.DeviceLeavePayload];
    permitJoinChanged: [data: Events.PermitJoinChangedPayload];
    lastSeenChanged: [data: Events.LastSeenChangedPayload];
}
/**
 * @noInheritDoc
 */
export declare class Controller extends events.EventEmitter<ControllerEventMap> {
    #private;
    private options;
    private database;
    private adapter;
    private permitJoinTimer;
    private permitJoinEnd?;
    private backupTimer;
    private databaseSaveTimer;
    private stopping;
    private adapterDisconnected;
    private networkParametersCached;
    /** List of unknown devices detected during a single runtime session. Serves as de-dupe and anti-spam. */
    private unknownDevices;
    /**
     * Create a controller
     *
     * To auto detect the port provide `null` for `options.serialPort.path`
     */
    constructor(options: Options);
    get greenPower(): GreenPower;
    get touchlink(): Touchlink;
    /**
     * Start the Herdsman controller
     */
    start(abortSignal?: AbortSignal): Promise<AdapterTsType.StartResult>;
    /**
     * Send a request according to given payload.
     * @param rawPayload Payload used to determine and build the request
     * @param customClusters Manually passed custom clusters used in ZCL serialization (if any, if matching)
     * @returns A response may or may not be returned depending on given payload (up to caller to verify)
     */
    sendRaw(rawPayload: RawPayload, customClusters?: CustomClusters): Promise<ZdoTypes.GenericZdoResponse | ZclPayload | undefined>;
    addInstallCode(installCode: string): Promise<void>;
    permitJoin(time: number, device?: Device): Promise<void>;
    getPermitJoin(): boolean;
    getPermitJoinEnd(): number | undefined;
    isStopping(): boolean;
    isAdapterDisconnected(): boolean;
    stop(): Promise<void>;
    private databaseSave;
    backup(): Promise<void>;
    coordinatorCheck(): Promise<{
        missingRouters: Device[];
    }>;
    reset(type: "soft" | "hard"): Promise<void>;
    getCoordinatorVersion(): Promise<AdapterTsType.CoordinatorVersion>;
    getNetworkParameters(): Promise<AdapterTsType.NetworkParameters>;
    /**
     * Get all devices
     * @deprecated use getDevicesIterator()
     */
    getDevices(): Device[];
    /**
     * Get iterator for all devices
     */
    getDevicesIterator(predicate?: (value: Device) => boolean): Generator<Device>;
    /**
     * Get all devices with a specific type
     */
    getDevicesByType(type: DeviceType): Device[];
    /**
     * Get device by ieeeAddr
     */
    getDeviceByIeeeAddr(ieeeAddr: string): Device | undefined;
    /**
     * Get device by networkAddress
     */
    getDeviceByNetworkAddress(networkAddress: number): Device | undefined;
    /**
     * Get IEEE address for all devices
     */
    getDeviceIeeeAddresses(): string[];
    /**
     * Get group by ID
     */
    getGroupByID(groupID: number): Group | undefined;
    /**
     * Get all groups
     * @deprecated use getGroupsIterator()
     */
    getGroups(): Group[];
    /**
     * Get iterator for all groups
     */
    getGroupsIterator(predicate?: (value: Group) => boolean): Generator<Group>;
    /**
     * Create a Group
     */
    createGroup(groupID: number): Group;
    /**
     * Broadcast a network-wide channel change.
     */
    private changeChannel;
    identifyUnknownDevice(nwkAddress: number): Promise<Device | undefined>;
    private checkDeviceNetworkAddress;
    private onNetworkAddress;
    private onIEEEAddress;
    private onDeviceAnnounce;
    private onDeviceLeave;
    private onAdapterDisconnected;
    private onDeviceJoinedGreenPower;
    private onDeviceLeaveGreenPower;
    private selfAndDeviceEmit;
    private onDeviceJoined;
    private onZdoResponse;
    private onZclPayload;
}
export default Controller;
//# sourceMappingURL=controller.d.ts.map