/*
Copyright The Helm Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package action

import (
	"bytes"

	"helm.sh/helm/v4/pkg/kube"
	ri "helm.sh/helm/v4/pkg/release"
)

// Status is the action for checking the deployment status of releases.
//
// It provides the implementation of 'helm status'.
type Status struct {
	cfg *Configuration

	Version int

	// ShowResourcesTable is used with ShowResources. When true this will cause
	// the resulting objects to be retrieved as a kind=table.
	ShowResourcesTable bool
}

// NewStatus creates a new Status object with the given configuration.
func NewStatus(cfg *Configuration) *Status {
	return &Status{
		cfg: cfg,
	}
}

// Run executes 'helm status' against the given release.
func (s *Status) Run(name string) (ri.Releaser, error) {
	if err := s.cfg.KubeClient.IsReachable(); err != nil {
		return nil, err
	}

	reli, err := s.cfg.releaseContent(name, s.Version)
	if err != nil {
		return nil, err
	}

	rel, err := releaserToV1Release(reli)
	if err != nil {
		return nil, err
	}

	var resources kube.ResourceList
	if s.ShowResourcesTable {
		resources, err = s.cfg.KubeClient.BuildTable(bytes.NewBufferString(rel.Manifest), false)
		if err != nil {
			return nil, err
		}
	} else {
		resources, err = s.cfg.KubeClient.Build(bytes.NewBufferString(rel.Manifest), false)
		if err != nil {
			return nil, err
		}
	}

	resp, err := s.cfg.KubeClient.Get(resources, true)
	if err != nil {
		return nil, err
	}

	rel.Info.Resources = resp

	return rel, nil
}
