// Copyright 2025 The go-github AUTHORS. All rights reserved.
//
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

package github

import (
	"context"
	"fmt"
)

// SCIMSchemasURINamespacesGroups is the SCIM schema URI namespace for group resources.
// This constant represents the standard SCIM core schema for group objects as defined by RFC 7643.
const SCIMSchemasURINamespacesGroups = "urn:ietf:params:scim:schemas:core:2.0:Group"

// SCIMSchemasURINamespacesListResponse is the SCIM schema URI namespace for list response resources.
// This constant represents the standard SCIM namespace for list responses used in paginated queries, as defined by RFC 7644.
const SCIMSchemasURINamespacesListResponse = "urn:ietf:params:scim:api:messages:2.0:ListResponse"

// SCIMEnterpriseGroupAttributes represents supported SCIM Enterprise group attributes.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#supported-scim-group-attributes
type SCIMEnterpriseGroupAttributes struct {
	DisplayName *string                           `json:"displayName,omitempty"` // Human-readable name for a group.
	Members     []*SCIMEnterpriseDisplayReference `json:"members,omitempty"`     // List of members who are assigned to the group in SCIM provider
	ExternalID  *string                           `json:"externalId,omitempty"`  // This identifier is generated by a SCIM provider. Must be unique per user.
	// Bellow: Only populated as a result of calling SetSCIMInformationForProvisionedGroup:
	Schemas []string            `json:"schemas,omitempty"` // The URIs that are used to indicate the namespaces of the SCIM schemas.
	ID      *string             `json:"id,omitempty"`      // The internally generated id for the group object.
	Meta    *SCIMEnterpriseMeta `json:"meta,omitempty"`    // The metadata associated with the creation/updates to the group.
}

// SCIMEnterpriseDisplayReference represents a JSON SCIM (System for Cross-domain Identity Management) resource reference.
type SCIMEnterpriseDisplayReference struct {
	Value   string  `json:"value"`             // The local unique identifier for the member (e.g., user ID or group ID).
	Ref     string  `json:"$ref"`              // The URI reference to the member resource (e.g., https://api.github.com/scim/v2/Users/{id}).
	Display *string `json:"display,omitempty"` // The display name associated with the member (e.g., user name or group name).
}

// SCIMEnterpriseMeta represents metadata about the SCIM resource.
type SCIMEnterpriseMeta struct {
	ResourceType string     `json:"resourceType"`           // A type of a resource (`User` or `Group`).
	Created      *Timestamp `json:"created,omitempty"`      // A date and time when the user was created.
	LastModified *Timestamp `json:"lastModified,omitempty"` // A date and time when the user was last modified.
	Location     *string    `json:"location,omitempty"`     // A URL location of an object
}

// SCIMEnterpriseGroups represents the result of calling ListProvisionedSCIMGroups.
type SCIMEnterpriseGroups struct {
	Schemas      []string                         `json:"schemas,omitempty"`
	TotalResults *int                             `json:"totalResults,omitempty"`
	Resources    []*SCIMEnterpriseGroupAttributes `json:"Resources,omitempty"`
	StartIndex   *int                             `json:"startIndex,omitempty"`
	ItemsPerPage *int                             `json:"itemsPerPage,omitempty"`
}

// ListProvisionedSCIMGroupsEnterpriseOptions represents query parameters for ListProvisionedSCIMGroups.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#list-provisioned-scim-groups-for-an-enterprise--parameters
type ListProvisionedSCIMGroupsEnterpriseOptions struct {
	// If specified, only results that match the specified filter will be returned.
	// Possible filters are `externalId`, `id`, and `displayName`. For example, `externalId eq "a123"`.
	Filter string `url:"filter,omitempty"`
	// Excludes the specified attribute from being returned in the results.
	ExcludedAttributes string `url:"excludedAttributes,omitempty"`
	// Used for pagination: the starting index of the first result to return when paginating through values.
	// Default: 1.
	StartIndex int `url:"startIndex,omitempty"`
	// Used for pagination: the number of results to return per page.
	// Default: 30.
	Count int `url:"count,omitempty"`
}

// ListProvisionedSCIMGroups lists provisioned SCIM groups in an enterprise.
//
// GitHub API docs: https://docs.github.com/enterprise-cloud@latest/rest/enterprise-admin/scim#list-provisioned-scim-groups-for-an-enterprise
//
//meta:operation GET /scim/v2/enterprises/{enterprise}/Groups
func (s *EnterpriseService) ListProvisionedSCIMGroups(ctx context.Context, enterprise string, opts *ListProvisionedSCIMGroupsEnterpriseOptions) (*SCIMEnterpriseGroups, *Response, error) {
	u := fmt.Sprintf("scim/v2/enterprises/%v/Groups", enterprise)
	u, err := addOptions(u, opts)
	if err != nil {
		return nil, nil, err
	}

	req, err := s.client.NewRequest("GET", u, nil)
	if err != nil {
		return nil, nil, err
	}

	groups := new(SCIMEnterpriseGroups)
	resp, err := s.client.Do(ctx, req, groups)
	if err != nil {
		return nil, resp, err
	}

	return groups, resp, nil
}
