﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/DocumentFormat.h>
#include <aws/ssm/model/DocumentStatus.h>
#include <aws/ssm/model/ReviewStatus.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SSM {
namespace Model {

/**
 * <p>Version information about the document.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ssm-2014-11-06/DocumentVersionInfo">AWS
 * API Reference</a></p>
 */
class DocumentVersionInfo {
 public:
  AWS_SSM_API DocumentVersionInfo() = default;
  AWS_SSM_API DocumentVersionInfo(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API DocumentVersionInfo& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The document name.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  DocumentVersionInfo& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The friendly name of the SSM document. This value can differ for each version
   * of the document. If you want to update this value, see
   * <a>UpdateDocument</a>.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  DocumentVersionInfo& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document version.</p>
   */
  inline const Aws::String& GetDocumentVersion() const { return m_documentVersion; }
  inline bool DocumentVersionHasBeenSet() const { return m_documentVersionHasBeenSet; }
  template <typename DocumentVersionT = Aws::String>
  void SetDocumentVersion(DocumentVersionT&& value) {
    m_documentVersionHasBeenSet = true;
    m_documentVersion = std::forward<DocumentVersionT>(value);
  }
  template <typename DocumentVersionT = Aws::String>
  DocumentVersionInfo& WithDocumentVersion(DocumentVersionT&& value) {
    SetDocumentVersion(std::forward<DocumentVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the artifact associated with the document. For example, 12.6.
   * This value is unique across all versions of a document, and can't be
   * changed.</p>
   */
  inline const Aws::String& GetVersionName() const { return m_versionName; }
  inline bool VersionNameHasBeenSet() const { return m_versionNameHasBeenSet; }
  template <typename VersionNameT = Aws::String>
  void SetVersionName(VersionNameT&& value) {
    m_versionNameHasBeenSet = true;
    m_versionName = std::forward<VersionNameT>(value);
  }
  template <typename VersionNameT = Aws::String>
  DocumentVersionInfo& WithVersionName(VersionNameT&& value) {
    SetVersionName(std::forward<VersionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The date the document was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedDate() const { return m_createdDate; }
  inline bool CreatedDateHasBeenSet() const { return m_createdDateHasBeenSet; }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  void SetCreatedDate(CreatedDateT&& value) {
    m_createdDateHasBeenSet = true;
    m_createdDate = std::forward<CreatedDateT>(value);
  }
  template <typename CreatedDateT = Aws::Utils::DateTime>
  DocumentVersionInfo& WithCreatedDate(CreatedDateT&& value) {
    SetCreatedDate(std::forward<CreatedDateT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An identifier for the default version of the document.</p>
   */
  inline bool GetIsDefaultVersion() const { return m_isDefaultVersion; }
  inline bool IsDefaultVersionHasBeenSet() const { return m_isDefaultVersionHasBeenSet; }
  inline void SetIsDefaultVersion(bool value) {
    m_isDefaultVersionHasBeenSet = true;
    m_isDefaultVersion = value;
  }
  inline DocumentVersionInfo& WithIsDefaultVersion(bool value) {
    SetIsDefaultVersion(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The document format, either JSON or YAML.</p>
   */
  inline DocumentFormat GetDocumentFormat() const { return m_documentFormat; }
  inline bool DocumentFormatHasBeenSet() const { return m_documentFormatHasBeenSet; }
  inline void SetDocumentFormat(DocumentFormat value) {
    m_documentFormatHasBeenSet = true;
    m_documentFormat = value;
  }
  inline DocumentVersionInfo& WithDocumentFormat(DocumentFormat value) {
    SetDocumentFormat(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the SSM document, such as <code>Creating</code>,
   * <code>Active</code>, <code>Failed</code>, and <code>Deleting</code>.</p>
   */
  inline DocumentStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(DocumentStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline DocumentVersionInfo& WithStatus(DocumentStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A message returned by Amazon Web Services Systems Manager that explains the
   * <code>Status</code> value. For example, a <code>Failed</code> status might be
   * explained by the <code>StatusInformation</code> message, "The specified S3
   * bucket doesn't exist. Verify that the URL of the S3 bucket is correct."</p>
   */
  inline const Aws::String& GetStatusInformation() const { return m_statusInformation; }
  inline bool StatusInformationHasBeenSet() const { return m_statusInformationHasBeenSet; }
  template <typename StatusInformationT = Aws::String>
  void SetStatusInformation(StatusInformationT&& value) {
    m_statusInformationHasBeenSet = true;
    m_statusInformation = std::forward<StatusInformationT>(value);
  }
  template <typename StatusInformationT = Aws::String>
  DocumentVersionInfo& WithStatusInformation(StatusInformationT&& value) {
    SetStatusInformation(std::forward<StatusInformationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The current status of the approval review for the latest version of the
   * document.</p>
   */
  inline ReviewStatus GetReviewStatus() const { return m_reviewStatus; }
  inline bool ReviewStatusHasBeenSet() const { return m_reviewStatusHasBeenSet; }
  inline void SetReviewStatus(ReviewStatus value) {
    m_reviewStatusHasBeenSet = true;
    m_reviewStatus = value;
  }
  inline DocumentVersionInfo& WithReviewStatus(ReviewStatus value) {
    SetReviewStatus(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_displayName;

  Aws::String m_documentVersion;

  Aws::String m_versionName;

  Aws::Utils::DateTime m_createdDate{};

  bool m_isDefaultVersion{false};

  DocumentFormat m_documentFormat{DocumentFormat::NOT_SET};

  DocumentStatus m_status{DocumentStatus::NOT_SET};

  Aws::String m_statusInformation;

  ReviewStatus m_reviewStatus{ReviewStatus::NOT_SET};
  bool m_nameHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_documentVersionHasBeenSet = false;
  bool m_versionNameHasBeenSet = false;
  bool m_createdDateHasBeenSet = false;
  bool m_isDefaultVersionHasBeenSet = false;
  bool m_documentFormatHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_statusInformationHasBeenSet = false;
  bool m_reviewStatusHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
