// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A0__F103

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_128_0_T_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xaf1fda64u, 0x26623780u, 0, 6 }, // af1fda6426623780 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__16_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x3677f1f8u, 0x5127bfb3u, 0, 23 }, // 3677f1f85127bfb3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xf2dc967cu, 0x099dec17u, 74, 80 }, // f2dc967c099dec17 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xc00a714cu, 0xc71fb094u, 74, 23 }, // c00a714cc71fb094 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x73064ec6u, 0x056bc256u, 74, 126 }, // 73064ec6056bc256 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__32_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x4ba8d900u, 0xc08bb5d6u, 97, 6 }, // 4ba8d900c08bb5d6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xba9b6ab2u, 0x7d7017e3u, 120, 6 }, // ba9b6ab27d7017e3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__64_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x3bbb9261u, 0x97085514u, 120, 103 }, // 3bbb926197085514 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0xc4274e6bu, 0xdb81694au, 120, 126 }, // c4274e6bdb81694a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xefd623e0u, 0xb1ef3780u, 160, 80 }, // efd623e0b1ef3780 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_128_0_T_T_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{5,0,3,3,1,1,1,1,1,1},
 {6,4,6,2,1,1,1,1,1,1},
 {6,6,7,2,1,1,1,1,1,1},
 {6,6,6,2,1,2,2,2,2,2},
 {6,6,6,8,1,9,2,2,2,2},
 {6,6,6,8,8,9,9,9,9,9},
 {6,6,6,8,8,9,9,9,9,9},
 {6,6,6,8,8,9,9,9,9,9},
 {6,6,6,8,9,9,9,9,9,9},
 {6,6,6,9,9,9,9,6,9,9}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A0__F103 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 3;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 3;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 128
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 1

// vim: set fileencoding=utf-8

